(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[   2422647,      33522]
NotebookOptionsPosition[   2402630,      32961]
NotebookOutlinePosition[   2404983,      33022]
CellTagsIndexPosition[   2404940,      33019]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell[TextData[{
 "Gibbs estimation of microstructure models: \nTeaching notes\n\nJoel \
Hasbrouck\n",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"Copyright", " ", "2007"}], ",", " ", 
    RowBox[{"Joel", " ", 
     RowBox[{"Hasbrouck", ".", " ", "All"}], " ", "rights", " ", 
     RowBox[{"reserved", "."}]}]}], TraditionalForm]],
  FontSize->14,
  FontSlant->"Italic"],
 "\n",
 StyleBox["(Also see bibliography in EMM Gibbs v0100.doc)",
  FontSize->14]
}], "Title",
 CellChangeTimes->{{3.3896255910914*^9, 3.3896255932033997`*^9}, {
   3.3897167078988237`*^9, 3.389716768500824*^9}, 3.3897219191158237`*^9, {
   3.3897220399168243`*^9, 3.3897220640418243`*^9}, {3.3900309756364307`*^9, 
   3.39003097613043*^9}, {3.3900334141814303`*^9, 3.39003347526243*^9}, 
   3.3900410824694304`*^9}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Text", "  ", "@", " ", 
  RowBox[{"Style", "[", 
   RowBox[{
    RowBox[{"\"\<Notebook evaluated \>\"", "<>", 
     RowBox[{"DateString", "[", "\"\<DateTime\>\"", "]"}]}], ",", 
    "\"\<Subtitle\>\""}], "]"}]}]], "Input",
 CellLabelAutoDelete->True,
 CellChangeTimes->{{3.3896127180972*^9, 3.3896127205282*^9}, 
   3.389716999411824*^9, {3.3897170351038237`*^9, 3.389717047196824*^9}}],

Cell[BoxData[
 InterpretationBox[Cell[BoxData[
   StyleBox["\<\"Notebook evaluated Tuesday 5 June 2007 10:05:07\"\>", 
    "Subtitle",
    StripOnInput->False]], "Text", "TR"],
  Text[
   Style["Notebook evaluated Tuesday 5 June 2007 10:05:07", 
    "Subtitle"]]]], "Output",
 CellMargins->{{18, Inherited}, {Inherited, 0}},
 CellLabelAutoDelete->True,
 CellChangeTimes->{
  3.3896127211071997`*^9, {3.3896127536242*^9, 3.3896127555652*^9}, 
   3.3896128616452*^9, 3.389623031447*^9, 3.3896232934164*^9, 
   3.389716955274824*^9, 3.3897170008718243`*^9, {3.3897170377738237`*^9, 
   3.3897170482808237`*^9}, 3.389735230721224*^9, 3.3900335783634305`*^9, 
   3.3900411072124305`*^9},
 Background->RGBColor[0.866667, 0.733333, 1]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 
 CounterBox["Section"],
 ". ",
 "Introduction"
}], "SectionFirst"],

Cell["\<\
This note discusses Gibbs estimation of the Roll model and various \
modifications. The goal is a more discursive and heuristic treatment of \
material covered in Hasbrouck (2004, 2006). Other applications of Gibbs \
samplers in market microstructure include Hasbrouck (1999) and Ball and \
Chordia (2001).\
\>", "Text"],

Cell["\<\
The techniques discussed here follow an approach that relies on simulation to \
characterize model parameters. Applied to microstructure models, there are \
three key elements:\
\>", "Text"],

Cell[TextData[StyleBox["Bayesian analysis",
 FontSlant->"Italic"]], "BulletedList"],

Cell[TextData[StyleBox["Simulation",
 FontSlant->"Italic"]], "BulletedList"],

Cell[TextData[StyleBox["Characterization of microstructure data generating \
processes by their conditional probabilities.",
 FontSlant->"Italic"]], "BulletedList"],

Cell["In more detail:", "Text"],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Bayesian analysis",
 FontSlant->"Italic"]], "Subsubsection"],

Cell["\<\
The models are estimated in a Bayesian framework. The differences between \
Bayesian and classical analysis are continually debated and discussed. The \
framework here is definitely Bayesian, but it should be noted that even if \
one doesn't buy the full Bayesian philosophy, the techniques discussed here \
can be motivated on grounds of estimation simplicity and computational \
efficiency.
This is an unusual claim. Bayesian analyses are usually more complex (both \
conceptually and computationally) than their classical counterparts. This is \
sometimes cited by Bayesian adherents as the prime barrier to their \
widespread adoption. Most microstructure models, though, are dynamic (over \
time) and they have latent (hidden, unobservable) quantities. The classic \
Roll model is dynamic, and the trade direction indicator (\"buy or sell\") \
variables are not observed.\
\>", "Text"],

Cell["\<\
Dynamic latent variable models can be formulated in state-space form and \
estimated via maximum likelihood. The latent variables are often non-Gaussian \
(e.g., again, the trade direction indicator variables), and if one wants to \
go beyond the techniques of multivariate linear models (like VARs), \
estimation involves nonlinear filtering. The Gibbs estimates are usually \
quicker and simpler.\
\>", "Text"],

Cell["\<\
There are presently a number of Bayesian statistics textbooks available. In \
my opinion the most useful for financial econometricians, are those that \
discuss econometrics from a Bayesian perspective. Lancaster (2004) and Geweke \
(2005) are both excellent. Lancaster's treatment is particularly accessible; \
Geweke presents more results. Nelson and Kim (2000) is a good introduction to \
the techniques in the context of a specific problem (regime switching \
models). In financial econometrics, the heaviest use of Bayesian simulation \
has been in modeling stochastic volatility. Shephard (2005) is a good survery \
of this area. Tanner (1996) and Carlin and Louis (2004) consider a broader \
range of Bayesian statistical tools and applications.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Simulation.",
 FontSlant->"Italic"]], "Subsubsection"],

Cell[TextData[{
 "The output of a classical procedure (e.g., OLS) is usually a statement \
about the distribution of a parameter. E.g., \"\[Theta] is asympototically \
normally distributed with mean ",
 Cell[BoxData[
  OverscriptBox["\[Theta]", "_"]]],
 " and variance ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "\[Theta]", "2"]]],
 ",\" where the mean and variance quantities are computed directly. But we \
could also characterize a distribution by a sample of draws from that \
distribution. This is what most of modern Bayesian analysis does. The output \
of the estimation procedures discussed here is a stream of random draws of \
the parameters of interest (conditional on the model and the data). From this \
stream we can construct an estimate of the full distribution (via kernel \
smoothing) or simply a summary measure (like the mean or median)."
}], "Text"],

Cell[TextData[{
 "Among other things, simulation facilitates characterization of \
distributions for functions of random variables.  For example, suppose that \
",
 Cell[BoxData[
  RowBox[{
   RowBox[{"x", "~", "N"}], 
   RowBox[{"(", 
    RowBox[{"\[Mu]", ",", 
     SuperscriptBox["\[Sigma]", "2"]}], ")"}]}]]],
 " and we'd like to characterize the distribution of ",
 Cell[BoxData[
  RowBox[{"y", "=", 
   RowBox[{"f", 
    RowBox[{"(", "x", ")"}]}]}]]],
 " where ",
 Cell[BoxData["f"]],
 " is sufficiently complicated that we can't get closed-form results.  We \
simply generate random values ",
 Cell[BoxData[
  SubscriptBox["x", "i"]]],
 " and empirically examine the distribution of ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["y", "i"], "=", 
   RowBox[{"f", 
    RowBox[{"(", 
     SubscriptBox["x", "i"], ")"}]}]}]]],
 "."
}], "Text"],

Cell["\<\
The link between simulation and Bayesian analysis is strong for the following \
reason. The distributions that arise in Bayesian analysis often describe many \
random variables (i.e., they are of high dimension). It also often happens \
that they have no closed form representation. Instead, they are characterized \
by simulation. The Gibbs procedure belongs to a class of random number \
generators called Markov Chain Monte Carlo (MCMC) techniques. They work by \
setting up rules for moving from one realization (draw) of the random \
variables to a subsequent realization. These draws are viewed as \"states\" \
in a Markov chain, and the rules define the transition probabilities. The \
limiting distribution of the states is identical to the distribution of the \
variables of interest, and is approximated by repeated application of the \
transition function.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[StyleBox["Conditional probabilities",
 FontSlant->"Italic"]], "Subsubsection"],

Cell[TextData[{
 "To set up a Gibbs estimate, we need to compute conditional densities for \
all of the unknowns (parameters ",
 StyleBox["and",
  FontSlant->"Italic"],
 " latent data). The conditional distributions for the parameters are usually \
identical to those found in many other applications (e.g., the normal \
Bayesian linear regression model). This note merely summarizes these \
distributions, refering the reader elsewhere for a fuller treatment. The \
conditional distributions for the latent data, though, are specific to the \
problem at hand. Although not particularly complicated, they are \
non-standard, and this note covers them in greater detail."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Programs", "Subsubsection"],

Cell[TextData[{
 "This note is written in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ". Some ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " code and results are embedded. Most of the results, though, are computed \
using SAS code that is available on my web site. To obtain these programs, go \
to the link for ",
 StyleBox["Empirical Market Microstructure",
  FontSlant->"Italic"],
 " (2006, Oxford University Press), off of my home page at \
www.stern.nyu.edu/~jhasbrou. Follow the link for SAS programs and data. The \
unzipped files contain the programs used in this note (and other programs \
related to the book)."
}], "Text"],

Cell["\<\
The programs make heavy use of SAS/IML (\"Interactive Matrix Language\"). \
This is not the language I've used for most of my papers, but it is widely \
available. Anyone who has a copy of SAS should be able to run the programs. \
These programs are not \"industrial strength\". I've played around with them \
in generating the results for this note, but they haven't been tested against \
all the things that might come up in, say, the CRSP daily file. I haven't \
done any performance benchmarks, but I suspect that they run slower than \
comparable code in OX or Matlab.\
\>", "Text"],

Cell["The main programs used here are:", "Text"],

Cell[TextData[StyleBox["RollGibbs 2-trade case.sas\nRollGibbs Analyze q.sas\n\
RollGibbs 01.sas\nRollGibbsBeta 01.sas",
 FontSlant->"Italic"]], "Quote"],

Cell[TextData[{
 "These programs call two macros: ",
 StyleBox["RollGibbs.sas",
  FontSlant->"Italic"],
 " and ",
 StyleBox["RollGibbsBeta.sas. ",
  FontSlant->"Italic"],
 "These macros, in turn, make use of IML subroutines contained in a library \
called \"imlstor\". To set up this library, run the program ",
 StyleBox["RollGibbs Library 01.sas",
  FontSlant->"Italic"],
 " (which contains the code for the subroutines)."
}], "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 
 CounterBox["Section"],
 ". Overview"
}], "Section"],

Cell[TextData[{
 "This note illustrates the estimation approach for the Roll (1984) model of \
transaction prices. In this model, the \"efficient price\" ",
 Cell[BoxData[
  RowBox[{"(", 
   SubscriptBox["m", "t"], ")"}]]],
 " is assumed to follow a Gaussian random walk:"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   SubscriptBox["m", "t"], "=", 
   RowBox[{
    SubscriptBox["m", 
     RowBox[{"t", "-", "1"}]], "+", 
    SubscriptBox["u", "t"]}]}]]],
 " where ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    SubscriptBox["u", "t"], "~", "N"}], 
   RowBox[{"(", 
    RowBox[{"0", ",", 
     SubsuperscriptBox["\[Sigma]", "u", "2"]}], ")"}]}]]]
}], "NumberedEquation"],

Cell[TextData[{
 "The transaction price ",
 Cell[BoxData[
  RowBox[{"(", 
   SubscriptBox["p", "t"], ")"}]]],
 " is the efficient price, plus or minus a cost component that depends on \
whether the customer is buying or selling:"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   SubscriptBox["p", "t"], "=", 
   RowBox[{
    SubscriptBox["m", "t"], "+", 
    RowBox[{"c", " ", 
     SubscriptBox["q", "t"]}]}]}]]],
 " "
}], "NumberedEquation"],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  RowBox[{"c", Cell[""]}]]],
 " is the cost parameter and ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", 
   RowBox[{"\[PlusMinus]", "1"}]}]]],
 ". (If the customer is buying, ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", 
   RowBox[{"+", "1"}]}]]],
 "; if selling, ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", 
   RowBox[{"-", "1"}]}]]],
 "). The trade prices are observed. The ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " and the ",
 Cell[BoxData[
  SubscriptBox["m", "t"]]],
 " are not.\nBy taking first differences:"
}], "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  SubscriptBox["\[CapitalDelta]p", "t"], "=", 
  RowBox[{
   RowBox[{"c", " ", 
    SubscriptBox["\[CapitalDelta]q", "t"]}], "+", 
   SubscriptBox["u", "t"]}]}]]]], "NumberedEquation"],

Cell[TextData[{
 "This specification is important because if the ",
 Cell[BoxData[
  SubscriptBox["\[CapitalDelta]q", "t"]]],
 " were known, this would be a simple regression. "
}], "Text"],

Cell["\<\
Bayesian estimation of normal linear regressions is well understood. The \
discussion (in the next section) starts with a review of these procedures. \
\>", "BulletedList"],

Cell[TextData[{
 "There are two parameters in this \"regression\": ",
 Cell[BoxData["c"]],
 " (the coefficient) and ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", "2"]]],
 ". It is fairly easy to compute (in closed form) the posterior distributions \
",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"c", "|", 
      SubsuperscriptBox["\[Sigma]", "u", "2"]}], ",", " ", 
     SubscriptBox["p", "1"], ",", "...", ",", 
     SubscriptBox["p", "T"]}], ")"}]}]]],
 " and ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubsuperscriptBox["\[Sigma]", "u", "2"], "|", "c"}], ",", " ", 
     SubscriptBox["p", "1"], ",", "...", ",", 
     SubscriptBox["p", "T"]}], ")"}]}]]],
 ". "
}], "BulletedList"],

Cell[TextData[{
 "It is not possible to compute in closed form the joint posterior ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{"c", ",", 
     RowBox[{
      SubsuperscriptBox["\[Sigma]", "u", "2"], "|", " ", 
      SubscriptBox["p", "1"]}], ",", "...", ",", 
     SubscriptBox["p", "T"]}], ")"}]}]]],
 ". This motivates the next section, which summarizes the Gibbs sampler. "
}], "BulletedList"],

Cell[TextData[{
 "The Gibbs procedure is illustrated by applying it to a special case of the \
Roll model, one in which ",
 Cell[BoxData["c"]],
 " and ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", "2"]]],
 " are known, but the ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " are not. "
}], "BulletedList"],

Cell["\<\
The note then turns to a full estimation of the Roll model ...\
\>", "BulletedList"],

Cell["... and extensions.", "BulletedList"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 
 CounterBox["Section"],
 ". ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " initializations"
}], "Section"],

Cell[BoxData[
 RowBox[{"<<", "Notation`"}]], "Input",
 CellChangeTimes->{{3.3900336369184303`*^9, 3.3900336390534306`*^9}},
 CellLabel->"In[2]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{"Symbolize", "[", 
   TagBox[
    SubscriptBox["_Anything", "Rule"],
    NotationBoxTag,
    BaseStyle->"NotationTemplateStyle"], "]"}], ";"}]], "Input",
 CellLabel->"In[3]:="]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 
 CounterBox["Section"],
 ". Bayesian analysis of the normal linear regression"
}], "Section"],

Cell[CellGroupData[{

Cell["The basic Bayesian approach", "Subsection"],

Cell[TextData[{
 "Bayesian analysis consists of using a model and data to update prior \
beliefs. The revised beliefs are usually called posterior beliefs, or simply \
\"the posterior\". Let ",
 Cell[BoxData["y"]],
 " denote the observed data, and let the model be specified up to a parameter \
",
 Cell[BoxData["\[Theta]"]],
 " (possibly a vector).\nBayes theorem says:"
}], "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{"\[Theta]", "|", "y"}], ")"}]}], "=", 
  RowBox[{
   FractionBox[
    RowBox[{"f", 
     RowBox[{"(", 
      RowBox[{"\[Theta]", ",", "y"}], ")"}]}], 
    RowBox[{"f", 
     RowBox[{"(", "y", ")"}]}]], "=", 
   RowBox[{
    FractionBox[
     RowBox[{"f", 
      RowBox[{"(", 
       RowBox[{"y", "|", "\[Theta]"}], ")"}], "f", 
      RowBox[{"(", "\[Theta]", ")"}]}], 
     RowBox[{"f", 
      RowBox[{"(", "y", ")"}]}]], "\[Proportional]", 
    RowBox[{"f", 
     RowBox[{"(", 
      RowBox[{"y", "|", "\[Theta]"}], ")"}], "f", 
     RowBox[{"(", "\[Theta]", ")"}]}]}]}]}]]]], "NumberedEquation"],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", "\[Theta]", ")"}]}]]],
 " is an assumed prior distribution for the parameter.\n",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{"y", "|", "\[Theta]"}], ")"}]}]]],
 " is the likelihood function for the observations, given a particular value \
of ",
 Cell[BoxData["\[Theta]"]],
 ".\nThe use of ",
 Cell[BoxData["\[Proportional]"]],
 " (\"is proportional to\") reflects the fact that it is usually not \
necessary to compute ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", "y", ")"}]}]]],
 ", at least not by computing the marginal ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"f", 
    RowBox[{"(", "y", ")"}]}], "=", 
   RowBox[{"\[Integral]", 
    RowBox[{"f", 
     RowBox[{"(", 
      RowBox[{"\[Theta]", ",", "y"}], ")"}], "d\[Theta]"}]}]}]]],
 ". Instead, ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", "y", ")"}]}]]],
 "can treated as a normalization constant, set so that the posterior \
integrates to unity."
}], "Text"],

Cell[TextData[{
 "Often a distribution of interest, say ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", "x", ")"}]}]]],
 ",  can be written as ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"f", 
    RowBox[{"(", "x", ")"}]}], "=", 
   RowBox[{"k", " ", "g", 
    RowBox[{"(", "x", ")"}]}]}]]],
 ", where ",
 Cell[BoxData[
  RowBox[{"g", 
   RowBox[{"(", "x", ")"}]}]]],
 " is a parsimonious function of ",
 Cell[BoxData["x"]],
 " and ",
 Cell[BoxData["k"]],
 " is some scaling factor. ",
 Cell[BoxData["k"]],
 " might in fact be very complicated, possibly depending on other random \
variables and implicitly incorporating other distribution functions, but for \
purposes of characterizing the distribution of ",
 Cell[BoxData["x"]],
 ", it is constant. In this case, ",
 Cell[BoxData[
  RowBox[{"g", 
   RowBox[{"(", "x", ")"}]}]]],
 " is said to be the ",
 StyleBox["kernel",
  FontSlant->"Italic"],
 " of ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", "x", ")"}]}]]],
 ". "
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Bayesian estimation of the normal linear regression model", "Subsection"],

Cell["The normal regression model is:", "Text"],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   UnderscriptBox["y", 
    RowBox[{"N", "\[Times]", "1"}]], "=", 
   RowBox[{
    RowBox[{
     UnderscriptBox["X", 
      RowBox[{"N", "\[Times]", "K"}]], " ", 
     UnderscriptBox["\[Beta]", 
      RowBox[{"K", "\[Times]", "1"}]]}], " ", "+", " ", 
    UnderscriptBox["u", 
     RowBox[{"N", "\[Times]", "1"}]]}]}]]],
 " where ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"u", "~", "N"}], 
   RowBox[{"(", 
    RowBox[{"0", ",", 
     SubscriptBox["\[CapitalOmega]", "u"]}], ")"}]}]]]
}], "NumberedEquation"],

Cell[TextData[{
 Cell[BoxData["X"]],
 " is a matrix of covariates (explanatory variables) possibly including a \
constant; ",
 Cell[BoxData["\[Beta]"]],
 " is the coefficient vector."
}], "Text"],

Cell[CellGroupData[{

Cell["Estimation of coefficients (given the error variance)", "Subsubsection"],

Cell[TextData[{
 "Assume for the moment that ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", "2"]]],
 " is known. It is particularly convenient to assume a multivariate normal \
prior distribution for the coefficients:"
}], "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  RowBox[{"\[Beta]", "~", "N"}], 
  RowBox[{"(", 
   RowBox[{
    SubsuperscriptBox["\[Mu]", "\[Beta]", "Prior"], ",", 
    SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Prior"]}], 
   ")"}]}]]]], "NumberedEquation"],

Cell[TextData[{
 "The posterior distribution, ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{"\[Beta]", "|", "y"}], ")"}]}]]],
 "is"
}], "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{"N", 
  RowBox[{"(", 
   RowBox[{
    SubsuperscriptBox["\[Mu]", "\[Beta]", "Post"], ",", 
    SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Post"]}], 
   ")"}]}]]]], "NumberedEquation"],

Cell["where", "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  SubsuperscriptBox["\[Mu]", "\[Beta]", "Post"], "=", 
  RowBox[{"D", " ", "d"}]}]]]], "NumberedEquation"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Post"], "=", 
  SuperscriptBox["D", 
   RowBox[{"-", "1"}]]}]]]], "NumberedEquation"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  SuperscriptBox["D", 
   RowBox[{"-", "1"}]], "=", 
  RowBox[{
   RowBox[{
    SuperscriptBox["X", "\[Prime]"], 
    SubsuperscriptBox["\[CapitalOmega]", "u", 
     RowBox[{"-", "1"}]], "X"}], "+", 
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Prior"], ")"}], 
     RowBox[{"-", "1"}]], Cell[""]}]}]}]]]], "NumberedEquation"],

Cell[TextData[Cell[BoxData[
 RowBox[{"d", "=", 
  RowBox[{
   RowBox[{
    SuperscriptBox["X", "\[Prime]"], 
    SubsuperscriptBox["\[CapitalOmega]", "u", 
     RowBox[{"-", "1"}]], "y"}], "+", 
   RowBox[{
    SuperscriptBox[
     RowBox[{"(", 
      SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Prior"], ")"}], 
     RowBox[{"-", "1"}]], 
    SubsuperscriptBox["\[Mu]", "\[Beta]", 
     "Prior"]}]}]}]]]], "NumberedEquation"],

Cell[TextData[{
 "As ",
 Cell[BoxData[
  SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Prior"]]],
 " increases in magnitude, the posterior mean and variance converge toward \
the usual OLS values.\nIn this case, both the prior and posterior have the \
same form (multivariate normal). Such a prior is said to be conjugate."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simulating the coefficients", "Subsubsection"],

Cell[TextData[{
 "We'll often have to make a random draw from the coefficient distribution. \
To make a random draw from ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    UnderscriptBox["x", 
     RowBox[{"n", "\[Times]", "1"}]], "~", "MVN"}], 
   RowBox[{"(", 
    RowBox[{"\[Mu]", ",", "\[CapitalOmega]"}], ")"}]}]]],
 ":"
}], "Text"],

Cell[TextData[{
 "Compute the Cholesky factorization ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"F", ":", " ", "\[CapitalOmega]"}], "=", 
   RowBox[{
    SuperscriptBox["F", "\[Prime]"], "F"}]}]]],
 ", where ",
 Cell[BoxData["F"]],
 " is an upper triangular matrix."
}], "BulletedList"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  UnderscriptBox["z", 
   RowBox[{"n", "\[Times]", "1"}]]]],
 " where the ",
 Cell[BoxData[
  SubscriptBox["z", "i"]]],
 " are i.i.d. ",
 Cell[BoxData[
  RowBox[{"N", 
   RowBox[{"(", 
    RowBox[{"0", ",", "1"}], ")"}]}]]],
 "."
}], "BulletedList"],

Cell[TextData[{
 "Set the random draw as ",
 Cell[BoxData[
  RowBox[{"x", "=", 
   RowBox[{"\[Mu]", "+", 
    RowBox[{
     SuperscriptBox["F", "\[Prime]"], "z"}]}]}]]]
}], "BulletedList"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Restrictions on the prior", "Subsubsection"],

Cell[TextData[{
 "The economic model sometimes imposes bounds on the coefficients. For \
example, in the Roll model, we'll often want to require ",
 Cell[BoxData[
  RowBox[{"c", ">", "0"}]]],
 ". Suppose that the coefficient prior is"
}], "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"\[Beta]", "~", "N"}], 
   RowBox[{"(", 
    RowBox[{
     SubsuperscriptBox["\[Mu]", "\[Beta]", "Prior"], ",", 
     SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Prior"]}], ")"}]}], ",",
   " ", 
  RowBox[{
   UnderscriptBox["\[Beta]", "_"], "<", "\[Beta]", "<", 
   OverscriptBox["\[Beta]", "_"]}]}]]]], "NumberedEquation"],

Cell[TextData[{
 "Note that when we write this, ",
 Cell[BoxData[
  RowBox[{
   SubsuperscriptBox["\[Mu]", "\[Beta]", "Prior"], ",", 
   SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Prior"]}]]],
 " denote the formal parameters of the normal density. But since the \
distribution is truncated, they no longer denote the mean and covariance of \
the density."
}], "Text"],

Cell[TextData[{
 "With this prior, the posterior is simply ",
 Cell[BoxData[
  RowBox[{"N", 
   RowBox[{"(", 
    RowBox[{
     SubsuperscriptBox["\[Mu]", "\[Beta]", "Post"], ",", 
     SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Post"]}], ")"}]}]]],
 ", with ",
 Cell[BoxData[
  RowBox[{
   SubsuperscriptBox["\[Mu]", "\[Beta]", "Post"], Cell[""]}]]],
 " and ",
 Cell[BoxData[
  SubsuperscriptBox["\[CapitalOmega]", "\[Beta]", "Post"]]],
 "computed as described above, restricted to the space ",
 Cell[BoxData[
  RowBox[{
   UnderscriptBox["\[Beta]", "_"], "<", "\[Beta]", "<", 
   OverscriptBox["\[Beta]", "_"]}]]],
 "."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simulation from restricted normals.", "Subsubsection"],

Cell[TextData[{
 "First suppose that we want to make a random draw ",
 Cell[BoxData["z"]],
 " from a standard normal density, restricted to the interval ",
 Cell[BoxData[
  RowBox[{
   UnderscriptBox["z", "_"], "<", "z", "<", 
   OverscriptBox["z", "_"]}]]],
 ". The procedure is:"
}], "Text"],

Cell[TextData[{
 "Compute ",
 Cell[BoxData[
  RowBox[{
   UnderscriptBox["\[Pi]", "_"], "=", 
   RowBox[{"\[CapitalPhi]", 
    RowBox[{"(", 
     UnderscriptBox["z", "_"], ")"}]}]}]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   OverscriptBox["\[Pi]", "_"], "=", 
   RowBox[{"\[CapitalPhi]", 
    RowBox[{"(", 
     OverscriptBox["z", "_"], ")"}]}]}]]],
 ", where ",
 Cell[BoxData["\[CapitalPhi]"]],
 " is the c.d.f. of the standard normal."
}], "BulletedList"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData["u"]],
 " from the uniform distribution over ",
 Cell[BoxData[
  RowBox[{"(", 
   RowBox[{
    UnderscriptBox["\[Pi]", "_"], ",", 
    OverscriptBox["\[Pi]", "_"]}], ")"}]]]
}], "BulletedList"],

Cell[TextData[{
 "Set ",
 Cell[BoxData[
  RowBox[{"z", "=", 
   RowBox[{
    SuperscriptBox["\[CapitalPhi]", 
     RowBox[{"-", "1"}]], 
    RowBox[{"(", "u", ")"}]}]}]]],
 "."
}], "BulletedList"],

Cell[TextData[{
 "Now suppose that we want to make a bivariate random draw from ",
 Cell[BoxData[
  RowBox[{"x", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"(", GridBox[{
        {
         SubscriptBox["x", "1"]},
        {
         SubscriptBox["x", "2"]}
       }], ")"}], "~", "N"}], 
    RowBox[{"(", 
     RowBox[{"\[Mu]", ",", "\[CapitalOmega]"}], ")"}]}]}]]],
 ", ",
 Cell[BoxData[
  RowBox[{
   UnderscriptBox["x", "_"], "<", "x", "<", 
   OverscriptBox["x", "_"]}]]],
 "."
}], "Text"],

Cell[TextData[{
 "Compute the Cholesky factorization ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"F", ":", " ", "\[CapitalOmega]"}], "=", 
   RowBox[{
    SuperscriptBox["F", "\[Prime]"], "F"}]}]]],
 ", where ",
 Cell[BoxData["F"]],
 " is an upper triangular matrix."
}], "BulletedList"],

Cell[TextData[{
 "Set ",
 Cell[BoxData[
  RowBox[{
   UnderscriptBox["z", "_"], "=", 
   FractionBox[
    RowBox[{
     UnderscriptBox[
      SubscriptBox["x", "1"], "_"], "-", 
     SubscriptBox["\[Mu]", "1"]}], 
    SubscriptBox["F", "11"]]}]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   OverscriptBox["z", "_"], "=", 
   FractionBox[
    RowBox[{
     OverscriptBox[
      SubscriptBox["x", "1"], "_"], "-", 
     SubscriptBox["\[Mu]", "1"]}], 
    SubscriptBox["F", "11"]]}]]]
}], "BulletedList"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  SubscriptBox["z", "1"]]],
 " from the standard normal density, restricted to ",
 Cell[BoxData[
  RowBox[{"(", 
   RowBox[{
    UnderscriptBox["z", "_"], ",", 
    OverscriptBox["z", "_"]}], ")"}]]],
 ". Then ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["x", "1"], "=", 
   RowBox[{
    SubscriptBox["\[Mu]", "1"], "+", 
    RowBox[{
     SubscriptBox["F", "11"], 
     SubscriptBox["z", "1"]}]}]}]]],
 " will have the properties required of ",
 Cell[BoxData[
  SubscriptBox["x", "1"]]],
 "."
}], "BulletedList"],

Cell[TextData[{
 "Set ",
 Cell[BoxData[
  RowBox[{"\[Eta]", "=", 
   RowBox[{
    SubscriptBox["F", "11"], 
    SubscriptBox["z", "1"]}]}]]],
 ". "
}], "BulletedList"],

Cell[TextData[{
 "Set ",
 Cell[BoxData[
  RowBox[{
   UnderscriptBox["z", "_"], "=", 
   FractionBox[
    RowBox[{
     UnderscriptBox[
      SubscriptBox["x", "2"], "_"], "-", 
     SubscriptBox["\[Mu]", "2"], "-", "\[Eta]"}], 
    SubscriptBox["F", "22"]]}]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   OverscriptBox["z", "_"], "=", 
   FractionBox[
    RowBox[{
     OverscriptBox[
      SubscriptBox["x", "2"], "_"], "-", 
     SubscriptBox["\[Mu]", "2"], "-", "\[Eta]"}], 
    SubscriptBox["F", "22"]]}]]]
}], "BulletedList"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  SubscriptBox["z", "2"]]],
 " from the standard normal density, restricted to ",
 Cell[BoxData[
  RowBox[{"(", 
   RowBox[{
    UnderscriptBox["z", "_"], ",", 
    OverscriptBox["z", "_"]}], ")"}]]],
 ". Then ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["x", "2"], "=", 
   RowBox[{
    SubscriptBox["\[Mu]", "2"], "+", 
    RowBox[{
     SubscriptBox["F", "22"], 
     SubscriptBox["z", "2"]}]}]}]]],
 " will have the properties required of ",
 Cell[BoxData[
  SubscriptBox["x", "2"]]],
 "."
}], "BulletedList"],

Cell[TextData[{
 "The random draw as ",
 Cell[BoxData[
  RowBox[{"x", "=", 
   RowBox[{"\[Mu]", "+", 
    RowBox[{
     SuperscriptBox["F", "\[Prime]"], "z"}]}]}]]],
 " will have the required joint properties"
}], "BulletedList"],

Cell["This method may be generalized to higher dimensions.", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Estimation of error variance (given the coefficients)", "Subsubsection"],

Cell[TextData[{
 "Assuming that ",
 Cell[BoxData["\[Beta]"]],
 " is known, it is convenient to specify a inverted gamma prior for ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", "2"]]],
 ". One way of writing this is:"
}], "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   SubsuperscriptBox["\[Sigma]", "u", "2"]], "~", 
  RowBox[{"\[CapitalGamma]", "[", 
   RowBox[{
    SuperscriptBox["a", "Prior"], ",", 
    SuperscriptBox["b", "Prior"]}], "]"}]}]]]], "NumberedEquation"],

Cell["Then the posterior is", "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   SubsuperscriptBox["\[Sigma]", "u", "2"]], "|", 
  RowBox[{"y", "~", 
   RowBox[{"\[CapitalGamma]", "[", 
    RowBox[{
     SuperscriptBox["a", "Post"], ",", 
     SuperscriptBox["b", "Post"]}], "]"}]}]}]]]], "NumberedEquation"],

Cell[" where", "Text"],

Cell[TextData[{
 " ",
 Cell[BoxData[
  RowBox[{
   SuperscriptBox["a", "Post"], "=", 
   RowBox[{
    SuperscriptBox["a", "Prior"], "+", 
    FractionBox["N", "2"]}]}]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   SuperscriptBox["b", "Post"], "=", 
   RowBox[{
    SuperscriptBox["b", "Prior"], "+", 
    FractionBox[
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], "N"], 
      SubsuperscriptBox["u", "i", "2"]}], "2"]}]}]]]
}], "NumberedEquation"],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  SubscriptBox["u", "i"]]],
 " are the regression residuals ",
 Cell[BoxData[
  RowBox[{"u", "=", 
   RowBox[{"y", "-", 
    RowBox[{"X", " ", "\[Beta]"}]}]}]]],
 "."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Further notes", "Subsubsection"],

Cell[TextData[{
 "The density of a gamma variate ",
 Cell[BoxData["x"]],
 " with parameters ",
 Cell[BoxData["a"]],
 " and ",
 Cell[BoxData["b"]],
 " is:"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"GammaDistribution", "[", 
     RowBox[{"a", ",", "\[Lambda]"}], "]"}], ",", "x"}], "]"}], "/.", 
  RowBox[{"\[Lambda]", "\[Rule]", 
   RowBox[{"1", "/", "b"}]}]}]], "Input",
 CellLabel->"In[4]:="],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["1", "b"], ")"}], 
    RowBox[{"-", "a"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "b"}], " ", "x"}]], " ", 
   SuperscriptBox["x", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "a"}]]}], 
  RowBox[{"Gamma", "[", "a", "]"}]]], "Output",
 GeneratedCell->False,
 CellChangeTimes->{3.3900335784554305`*^9, 3.3900338395684304`*^9, 
  3.3900411088394303`*^9},
 CellLabel->"Out[4]="]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"GammaDistribution", "[", 
    RowBox[{"a", ",", "\[Lambda]"}], "]"}], "]"}], "/.", 
  RowBox[{"\[Lambda]", "\[Rule]", 
   RowBox[{"1", "/", "b"}]}]}]], "Input",
 CellLabel->"In[5]:="],

Cell[BoxData[
 FractionBox["a", "b"]], "Output",
 GeneratedCell->False,
 CellChangeTimes->{3.3900335785154305`*^9, 3.39003383973143*^9, 
  3.3900411088664303`*^9},
 CellLabel->"Out[5]="]
}, Open  ]],

Cell[TextData[{
 "Note: In the statistics literature, the Gamma distribution with parameters \
",
 Cell[BoxData[
  RowBox[{"a", Cell[""]}]]],
 " and ",
 Cell[BoxData["b"]],
 " is usually expressed as immediately above. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " parameterizes the distribution with the second parameter expressed as an \
inverse."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"PDF", "[", 
    RowBox[{
     RowBox[{"GammaDistribution", "[", 
      RowBox[{"a", ",", "\[Lambda]"}], "]"}], ",", "x"}], "]"}], "/.", 
   RowBox[{"\[Lambda]", "\[Rule]", 
    RowBox[{"1", "/", "b"}]}]}], "/.", 
  RowBox[{"x", "\[Rule]", 
   RowBox[{"1", "/", "z"}]}]}]], "Input",
 CellLabel->"In[6]:="],

Cell[BoxData[
 FractionBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["1", "b"], ")"}], 
    RowBox[{"-", "a"}]], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox["b", "z"]}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["1", "z"], ")"}], 
    RowBox[{
     RowBox[{"-", "1"}], "+", "a"}]]}], 
  RowBox[{"Gamma", "[", "a", "]"}]]], "Output",
 GeneratedCell->False,
 CellChangeTimes->{3.3900335785724306`*^9, 3.3900338397894306`*^9, 
  3.39004110889443*^9},
 CellLabel->"Out[6]="]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 
 CounterBox["Section"],
 ". The Gibbs recipe"
}], "Section"],

Cell[TextData[{
 "The Gibbs procedure is a vehicle for simulating from a complicated joint \
distribution ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     SubscriptBox["x", "1"], ",", "...", ",", 
     SubscriptBox["x", "n"]}], ")"}], Cell[""]}]]],
 ", possibly one that possesses no closed form representation."
}], "Text"],

Cell[TextData[{
 "The draws are constructed by iterating over the ",
 StyleBox["full conditional distributions",
  FontSlant->"Italic"],
 ":"
}], "Text"],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "1"], "|", 
      SubscriptBox["x", "2"]}], ",", "...", ",", 
     SubscriptBox["x", "n"]}], ")"}], Cell[""]}]]],
 "\n",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "2"], "|", 
      SubscriptBox["x", "1"]}], ",", 
     SubscriptBox["x", "3"], ",", "...", ",", 
     SubscriptBox["x", "n"]}], ")"}], Cell[""]}]]],
 "\n...\n",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "n"], "|", 
      SubscriptBox["x", "1"]}], ",", "...", ",", 
     SubscriptBox["x", 
      RowBox[{"n", "-", "1"}]]}], ")"}], Cell[""]}]]]
}], "NumberedEquation"],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  RowBox[{"x", "=", 
   RowBox[{"(", GridBox[{
      {
       SubscriptBox["x", "1"]},
      {
       SubscriptBox["x", "2"]},
      {"\[VerticalEllipsis]"},
      {
       SubscriptBox["x", "n"]}
     }], ")"}]}]]],
 "\nEach iteration of the Gibbs sampler is called a sweep. \nLet ",
 Cell[BoxData[
  SuperscriptBox["x", 
   RowBox[{"[", "i", "]"}]]]],
 " denote the value of ",
 Cell[BoxData["x"]],
 " at the conclusion of the ith sweep.\nThe procedure is:"
}], "Text"],

Cell[TextData[{
 "Initialization. Set ",
 Cell[BoxData[
  SuperscriptBox["x", 
   RowBox[{"[", "0", "]"}]]]],
 " to any feasible value."
}], "BulletedList"],

Cell[TextData[{
 "Sweep ",
 Cell[BoxData["i"]],
 ":\nGiven ",
 Cell[BoxData[
  SuperscriptBox["x", 
   RowBox[{"[", 
    RowBox[{"i", "-", "1"}], "]"}]]]],
 ":\nDraw ",
 Cell[BoxData[
  SubsuperscriptBox["x", "1", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "1"], "|", 
      SubsuperscriptBox["x", "2", 
       RowBox[{"[", 
        RowBox[{"i", "-", "1"}], "]"}]]}], ",", "...", ",", 
     SubsuperscriptBox["x", "n", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 "\nDraw ",
 Cell[BoxData[
  SubsuperscriptBox["x", "2", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "2"], "|", 
      SubsuperscriptBox["x", "1", 
       RowBox[{"[", 
        RowBox[{"i", "-", "1"}], "]"}]]}], ",", 
     SubsuperscriptBox["x", "3", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]], ",", "...", ",", 
     SubsuperscriptBox["x", "n", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 "\n...\nDraw ",
 Cell[BoxData[
  SubsuperscriptBox["x", "n", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "n"], "|", 
      SubsuperscriptBox["x", "1", 
       RowBox[{"[", 
        RowBox[{"i", "-", "1"}], "]"}]]}], ",", "...", ",", 
     SubsuperscriptBox["x", 
      RowBox[{"n", "-", "1"}], 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]]
}], "BulletedList"],

Cell["Repeat", "BulletedList"],

Cell[TextData[{
 "In the limit, as ",
 Cell[BoxData[
  RowBox[{"i", "\[Rule]", "\[Infinity]"}]]],
 ",  ",
 Cell[BoxData[
  SuperscriptBox["x", 
   RowBox[{"[", "i", "]"}]]]],
 "is distributed as ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", "x", ")"}]}]]],
 "."
}], "Text"],

Cell["Notes", "Text"],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  SuperscriptBox["x", 
   RowBox[{"[", "i", "]"}]]]],
 " are ",
 StyleBox["not",
  FontSlant->"Italic"],
 " independently distributed: ",
 Cell[BoxData[
  SuperscriptBox["x", 
   RowBox[{"[", "i", "]"}]]]],
 " takes ",
 Cell[BoxData[
  SuperscriptBox["x", 
   RowBox[{"[", 
    RowBox[{"i", "-", "1"}], "]"}]]]],
 " as its starting point. If the degree of dependence is high, a large number \
of sweeps may be needed to ensure proper mixing."
}], "BulletedList"],

Cell[TextData[{
 "The dependence may affect the calculation of some summary statistics. Think \
of the analogy to standard time series analysis. If ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["z", "1"], ",", "...", ",", 
   SubscriptBox["z", "T"]}]]],
 " are a sample of stationary time series data, ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["\[CapitalSigma]z", "i"], "/", "T"}]]],
 " is a consistent estimate of ",
 Cell[BoxData[
  RowBox[{"E", "[", 
   SubscriptBox["z", "t"], "]"}]]],
 ". The ",
 "standard error",
 " of this estimate, however, must be corrected for dependence."
}], "BulletedList"],

Cell["\<\
Convergence may be an issue. It is useful to graph the full sequence of \
draws.\
\>", "BulletedList"],

Cell[TextData[{
 "In analyzing the sequence of draws, it is common to throw out a few initial \
draws, so as to reduce the dependence on starting values. These discarded \
draws are sometimes called ",
 StyleBox["burn in",
  FontSlant->"Italic"],
 " draws."
}], "BulletedList"],

Cell[TextData[{
 "The Gibbs sampler also works when multiple variables are drawn at once. We \
might, for example, draw ",
 Cell[BoxData[
  SubsuperscriptBox["x", "1", 
   RowBox[{"[", "i", "]"}]]]],
 " and ",
 Cell[BoxData[
  SubsuperscriptBox["x", "2", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     SubscriptBox["x", "1"], ",", 
     RowBox[{
      SubscriptBox["x", "2"], "|", 
      SubsuperscriptBox["x", "3", 
       RowBox[{"[", 
        RowBox[{"i", "-", "1"}], "]"}]]}], ",", "...", ",", 
     SubsuperscriptBox["x", "n", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 ". This ",
 StyleBox["block sampling",
  FontSlant->"Italic"],
 " is often more computationally efficient."
}], "BulletedList"],

Cell[CellGroupData[{

Cell["Application to the normal regression model", "Subsubsection"],

Cell[TextData[{
 "From earlier results, we have ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"\[Beta]", "|", "y"}], ",", 
     SubsuperscriptBox["\[Sigma]", "u", "2"]}], ")"}]}]]],
 " and ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubsuperscriptBox["\[Sigma]", "u", "2"], "|", "\[Beta]"}], ",", "y"}], 
    ")"}]}]]],
 ". To obtain the full posterior ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{"\[Beta]", ",", 
     RowBox[{
      SubsuperscriptBox["\[Sigma]", "u", "2"], "|", "y"}]}], ")"}]}]]],
 " via the Gibbs procedure:\nInitialize ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", 
   RowBox[{"2", "[", "0", "]"}]]]],
 " to any feasible value. The ",
 Cell[BoxData["i"]],
 "th sweep of the sampler is:"
}], "Text"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  SuperscriptBox["\[Beta]", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"\[Beta]", "|", "y"}], ",", 
     SubsuperscriptBox["\[Sigma]", "u", 
      RowBox[{"2", "[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 ". (This will be a draw from a multivariate normal posterior.)"
}], "BulletedList"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", 
   RowBox[{"2", "[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{
     RowBox[{
      SubsuperscriptBox["\[Sigma]", "u", "2"], "|", "y"}], ",", 
     SuperscriptBox["\[Beta]", 
      RowBox[{"[", "i", "]"}]]}]}], ")"}]]],
 ". (That is, draw ",
 Cell[BoxData[
  RowBox[{"1", "/", 
   SubsuperscriptBox["\[Sigma]", "u", 
    RowBox[{"2", "[", "i", "]"}]]}]]],
 " from the gamma posterior.)"
}], "BulletedList"],

Cell[TextData[{
 "Proceed, iteratively drawing ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", 
   RowBox[{"2", "[", "i", "]"}]]]],
 " and ",
 Cell[BoxData[
  SuperscriptBox["\[Beta]", 
   RowBox[{"[", "i", "]"}]]]],
 "."
}], "Text"],

Cell["Notes", "Text"],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{"\[Beta]", ",", 
     RowBox[{
      SubsuperscriptBox["\[Sigma]", "u", "2"], "|", "y"}]}], ")"}]}]]],
 " is an exact small-sample distribution. "
}], "BulletedList"],

Cell["We now return to ...", "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 
 CounterBox["Section"],
 ". The Roll model"
}], "Section"],

Cell["\<\
Recall that we're working with the price change specification:\
\>", "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  SubscriptBox["\[CapitalDelta]p", "t"], "=", 
  RowBox[{
   RowBox[{"c", " ", 
    SubscriptBox["\[CapitalDelta]q", "t"]}], "+", 
   SubscriptBox["u", "t"]}]}]]]], "NumberedEquation"],

Cell[TextData[{
 "The sample is ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["p", "1"], ",", 
   SubscriptBox["p", "2"], ",", "...", ",", 
   SubscriptBox["p", "T"]}]]],
 ", and there are ",
 Cell[BoxData[
  RowBox[{"T", "-", "1"}]]],
 " price changes."
}], "Text"],

Cell[TextData[{
 "The unknowns are the parameters ",
 Cell[BoxData["c"]],
 " and ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", "2"]]],
 ", and the latent data: ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "1"], ",", "...", ",", 
   SubscriptBox["q", "T"]}]]],
 ".\nIn the Bayesian perspective, parameters and latent data are treated \
identically, and \"estimated\" in a similar fashion.\nWe don't need to \
construct priors for the ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 ". We can use the ones we assumed in the data generating process: ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", 
   RowBox[{"\[PlusMinus]", "1"}]}]]],
 " with equal probabilities.\nThe prior on ",
 Cell[BoxData["c"]],
 " is ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"c", "~", "N"}], 
   RowBox[{"(", 
    RowBox[{
     SuperscriptBox["\[Mu]", "Prior"], ",", 
     SuperscriptBox["\[CapitalOmega]", "Prior"]}], ")"}]}]]],
 " restricted to ",
 Cell[BoxData[
  RowBox[{"c", ">", "0"}]]],
 ". (I often take ",
 Cell[BoxData[
  RowBox[{
   SuperscriptBox["\[Mu]", "Prior"], "=", "0"}]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   SuperscriptBox["\[CapitalOmega]", "Prior"], "=", "1"}]]],
 ". Remember that these are parameters of the truncated distribution, not the \
true mean and variance.)\nThe prior on ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", "2"]]],
 " is inverted gamma with parameters ",
 Cell[BoxData["a"]],
 " and ",
 Cell[BoxData["b"]],
 ". (I often take ",
 Cell[BoxData[
  RowBox[{"a", "=", 
   RowBox[{"b", "=", 
    SuperscriptBox["10", 
     RowBox[{"-", "6"}]]}]}]]],
 ".)\nThe Gibbs sampler will look like this:"
}], "Text"],

Cell[TextData[{
 "Initialize ",
 Cell[BoxData[
  SuperscriptBox["c", 
   RowBox[{"[", "0", "]"}]]]],
 ", ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", 
   RowBox[{"2", "[", "0", "]"}]]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   SubsuperscriptBox["q", "1", 
    RowBox[{"[", "0", "]"}]], ",", "...", ",", " ", 
   SubsuperscriptBox["q", "T", 
    RowBox[{"[", "0", "]"}]]}]]],
 " to any feasible values. (I usually take ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "1"], "=", "1"}]]],
 "; ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", 
   RowBox[{"Sign", 
    RowBox[{"(", 
     SubscriptBox["\[CapitalDelta]p", "t"], ")"}]}]}]]],
 " if ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["\[CapitalDelta]p", "t"], "\[NotEqual]", "0"}]]],
 ", ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", 
   SubscriptBox["q", 
    RowBox[{"t", "-", "1"}]]}]]],
 " if ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["\[CapitalDelta]p", "t"], "=", "0"}]]],
 ". For US equities, ",
 Cell[BoxData[
  RowBox[{"c", "=", "0.01"}]]],
 " (\"1%\") is a good ballpark figure, if we're working in logs, and ",
 Cell[BoxData[
  RowBox[{
   SubsuperscriptBox["\[Sigma]", "u", "2"], "=", 
   SuperscriptBox["0.01", "2"]}]]],
 ")."
}], "BulletedList"],

Cell["For the ith sweep of the sampler:", "Text"],

Cell[TextData[{
 "Estimate the price change specification as a regression, assuming that ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", 
   SubsuperscriptBox["q", "t", 
    RowBox[{"[", 
     RowBox[{"i", "-", "1"}], "]"}]]}]]],
 " and that ",
 Cell[BoxData[
  RowBox[{
   SubsuperscriptBox["\[Sigma]", "u", "2"], "=", 
   SubsuperscriptBox["\[Sigma]", "u", 
    RowBox[{"2", "[", 
     RowBox[{"i", "-", "1"}], "]"}]]}]]],
 ". Construct the posterior for ",
 Cell[BoxData["c"]],
 ", and draw ",
 Cell[BoxData[
  SuperscriptBox["c", 
   RowBox[{"[", "i", "]"}]]]],
 " from this posterior."
}], "BulletedList"],

Cell[TextData[{
 "Using ",
 Cell[BoxData[
  SuperscriptBox["c", 
   RowBox[{"[", "i", "]"}]]]],
 ", compute the residuals from the regression. Construct the posterior for ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", "2"]]],
 " and draw ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", 
   RowBox[{"2", "[", "i", "]"}]]]],
 " from this posterior."
}], "BulletedList"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "1", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["q", "1"], "|", 
      SuperscriptBox["c", 
       RowBox[{"[", "i", "]"}]]}], ",", 
     SubsuperscriptBox["\[Sigma]", "u", 
      RowBox[{"2", "[", "i", "]"}]], ",", 
     SubsuperscriptBox["q", "2", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]], ",", 
     SubsuperscriptBox["q", "3", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]], ",", "...", ",", " ", 
     SubsuperscriptBox["q", "T", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 "\nDraw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "2", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["q", "2"], "|", 
      SuperscriptBox["c", 
       RowBox[{"[", "i", "]"}]]}], ",", 
     SubsuperscriptBox["\[Sigma]", "u", 
      RowBox[{"2", "[", "i", "]"}]], ",", 
     SubsuperscriptBox["q", "1", 
      RowBox[{"[", "i", "]"}]], ",", 
     SubsuperscriptBox["q", "3", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]], ",", "...", ",", " ", 
     SubsuperscriptBox["q", "T", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 "\n...\nDraw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "T", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"f", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["q", "T"], "|", 
      SuperscriptBox["c", 
       RowBox[{"[", "i", "]"}]]}], ",", 
     SubsuperscriptBox["\[Sigma]", "u", 
      RowBox[{"2", "[", "i", "]"}]], ",", 
     SubsuperscriptBox["q", "1", 
      RowBox[{"[", "i", "]"}]], ",", 
     SubsuperscriptBox["q", "2", 
      RowBox[{"[", "i", "]"}]], ",", "...", ",", " ", 
     SubsuperscriptBox["q", 
      RowBox[{"T", "-", "1"}], 
      RowBox[{"[", "i", "]"}]]}], ")"}]}]]]
}], "BulletedList"],

Cell["\<\
The first two steps of the sweep are, as discussed, standard Bayesian \
procedures. We now turn to the third step: simulating the trade direction \
indicators \
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 
 CounterBox["Section"],
 ". ",
 "Estimating the ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " using a Gibbs sampler."
}], "Section"],

Cell[TextData[{
 "In this section, we'll be taking ",
 Cell[BoxData["c"]],
 " and ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", "2"]]],
 " as known. We'll first look at the simple case where ",
 Cell[BoxData[
  RowBox[{"T", "=", "2"}]]],
 ". We can get closed-form results here, so we don't really need a Gibbs \
procedure, but it's a good starting point."
}], "Text"],

Cell[CellGroupData[{

Cell[TextData[{
 "The distribution of ",
 Cell[BoxData[
  SubscriptBox["q", "1"]]],
 " and ",
 Cell[BoxData[
  SubscriptBox["q", "2"]]],
 " when T=2"
}], "Subsection"],

Cell[BoxData[{
 RowBox[{
  RowBox[{"Symbolize", "[", 
   TagBox[
    SubscriptBox["p", "_"],
    NotationBoxTag,
    BaseStyle->"NotationTemplateStyle"], "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Symbolize", "[", 
   TagBox[
    SubscriptBox["q", "1"],
    NotationBoxTag,
    BaseStyle->"NotationTemplateStyle"], "]"}], ";", 
  RowBox[{"Symbolize", "[", 
   TagBox[
    SubscriptBox["q", "2"],
    NotationBoxTag,
    BaseStyle->"NotationTemplateStyle"], "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Symbolize", "[", 
   TagBox[
    SubscriptBox["\[Sigma]", "_"],
    NotationBoxTag,
    BaseStyle->"NotationTemplateStyle"], "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Symbolize", "[", 
   TagBox[
    SubscriptBox["u", "2"],
    NotationBoxTag,
    BaseStyle->"NotationTemplateStyle"], "]"}], ";"}]}], "Input",
 CellChangeTimes->{3.3900339351814303`*^9},
 CellLabel->"In[7]:="],

Cell[CellGroupData[{

Cell["Graph ", "Subsubsection"],

Cell[BoxData[
 RowBox[{
  RowBox[{"s", "=", "0.05"}], ";", " ", 
  RowBox[{"hc", "=", "0"}], ";", 
  RowBox[{"hl", "=", "0.02"}], ";"}]], "Input",
 CellLabel->"In[11]:="],

Cell["Suppose:", "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["p", "1"], "=", "1"}], ";", " ", 
  RowBox[{
   SubscriptBox["p", "2"], "=", "1.8"}], ";", " ", 
  RowBox[{"c", "=", ".5"}], ";"}]], "Input",
 CellLabel->"In[12]:="],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{
        SubscriptBox["p", "1"], "+", "c"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       RowBox[{
        SubscriptBox["p", "1"], "-", "c"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", 
       RowBox[{
        SubscriptBox["p", "2"], "+", "c"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", 
       RowBox[{
        SubscriptBox["p", "2"], "-", "c"}]}], "}"}]}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0.5", ",", "2.5"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "2.5"}], "}"}]}], "}"}]}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"AxesOrigin", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"0.5", ",", "0"}], "}"}]}], ",", 
   RowBox[{"Prolog", "\[Rule]", 
    RowBox[{"AbsolutePointSize", "[", "5", "]"}]}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"Frame", "\[Rule]", "True"}], ",", 
   RowBox[{"FrameLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Time", ",", "Price", ",", "None", ",", "None"}], "}"}]}], ",", 
   RowBox[{"FrameTicks", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "1", ",", "2"}], "}"}], ",", "Automatic", ",", 
      "None", ",", "None"}], "}"}]}], ",", "\[IndentingNewLine]", 
   RowBox[{"Epilog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Text", "[", 
       RowBox[{"\"\<\!\(\*SubscriptBox[\(p\), \(1\)]\)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"1", "-", "s"}], ",", 
          SubscriptBox["p", "1"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "0"}], "}"}]}], "]"}], ",", "\[IndentingNewLine]", 
      
      RowBox[{"Text", "[", 
       RowBox[{
       "\"\<\!\(\*SubscriptBox[\(m\), \(1\)]\)(\!\(\*SubscriptBox[\(q\), \
\(1\)]\)=-1)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"1", "-", "s"}], ",", 
          RowBox[{
           SubscriptBox["p", "1"], "+", "c"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "0"}], "}"}]}], "]"}], ",", "\[IndentingNewLine]", 
      
      RowBox[{"Text", "[", 
       RowBox[{
       "\"\<\!\(\*SubscriptBox[\(m\), \(1\)]\)(\!\(\*SubscriptBox[\(q\), \
\(1\)]\)=+1)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"1", "-", "s"}], ",", 
          RowBox[{
           SubscriptBox["p", "1"], "-", "c"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "0"}], "}"}]}], "]"}], ",", "\[IndentingNewLine]", 
      
      RowBox[{"Text", "[", 
       RowBox[{"\"\<\!\(\*SubscriptBox[\(p\), \(2\)]\)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"2", "+", "s"}], ",", 
          SubscriptBox["p", "2"]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"Text", "[", 
       RowBox[{
       "\"\<\!\(\*SubscriptBox[\(m\), \(2\)]\)(\!\(\*SubscriptBox[\(q\), \
\(2\)]\)=-1)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"2", "+", "s"}], ",", 
          RowBox[{
           SubscriptBox["p", "2"], "+", "c"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"Text", "[", 
       RowBox[{
       "\"\<\!\(\*SubscriptBox[\(m\), \(2\)]\)(\!\(\*SubscriptBox[\(q\), \
\(2\)]\)=+1)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"2", "+", "s"}], ",", 
          RowBox[{
           SubscriptBox["p", "2"], "-", "c"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"Arrow", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", 
            RowBox[{
             SubscriptBox["p", "1"], "+", "c"}]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"2", ",", 
            RowBox[{
             SubscriptBox["p", "2"], "+", "c"}]}], "}"}]}], "}"}], ",", 
        ".05"}], "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{"Arrow", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", 
            RowBox[{
             SubscriptBox["p", "1"], "+", "c"}]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"2", ",", 
            RowBox[{
             SubscriptBox["p", "2"], "-", "c"}]}], "}"}]}], "}"}], ",", 
        ".05"}], "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{"Arrow", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", 
            RowBox[{
             SubscriptBox["p", "1"], "-", "c"}]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"2", ",", 
            RowBox[{
             SubscriptBox["p", "2"], "+", "c"}]}], "}"}]}], "}"}], ",", 
        ".05"}], "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{"Arrow", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", 
            RowBox[{
             SubscriptBox["p", "1"], "-", "c"}]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"2", ",", 
            RowBox[{
             SubscriptBox["p", "2"], "-", "c"}]}], "}"}]}], "}"}], ",", 
        ".05"}], "]"}], ",", "\[IndentingNewLine]", 
      RowBox[{"Text", "[", 
       RowBox[{
       "\"\<A. \!\(\*SubscriptBox[\(u\), \(2\)]\)(\!\(\*SubscriptBox[\(q\), \
\(1\)]\)=-1,\!\(\*SubscriptBox[\(q\), \(2\)]\)=-1)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{"1.5", ",", "2"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "0"}], "}"}]}], "]"}], ",", "\[IndentingNewLine]", 
      
      RowBox[{"Text", "[", 
       RowBox[{
       "\"\<B. \!\(\*SubscriptBox[\(u\), \(2\)]\)(\!\(\*SubscriptBox[\(q\), \
\(1\)]\)=-1,\!\(\*SubscriptBox[\(q\), \(2\)]\)=+1)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{"1.7", ",", "1.45"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"Text", "[", 
       RowBox[{
       "\"\<C. \!\(\*SubscriptBox[\(u\), \(2\)]\)(\!\(\*SubscriptBox[\(q\), \
\(1\)]\)=+1,\!\(\*SubscriptBox[\(q\), \(2\)]\)=-1)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{"1.9", ",", "2"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"Text", "[", 
       RowBox[{
       "\"\<D. \!\(\*SubscriptBox[\(u\), \(2\)]\)(\!\(\*SubscriptBox[\(q\), \
\(1\)]\)=+1,\!\(\*SubscriptBox[\(q\), \(2\)]\)=+1)\>\"", ",", 
        RowBox[{"{", 
         RowBox[{"1.5", ",", ".8"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "1"}], ",", "0"}], "}"}]}], "]"}], ",", 
      "\[IndentingNewLine]", 
      RowBox[{"AbsolutePointSize", "[", "8", "]"}], ",", 
      RowBox[{"Point", "[", 
       RowBox[{"{", 
        RowBox[{"1", ",", 
         SubscriptBox["p", "1"]}], "}"}], "]"}], ",", 
      RowBox[{"Point", "[", 
       RowBox[{"{", 
        RowBox[{"2", ",", 
         SubscriptBox["p", "2"]}], "}"}], "]"}]}], "\[IndentingNewLine]", 
     "}"}]}], ",", "\[IndentingNewLine]", 
   RowBox[{"BaseStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"FontFamily", "\[Rule]", "\"\<Times\>\""}], "}"}]}]}], 
  "\[IndentingNewLine]", "]"}]], "Input",
 CellChangeTimes->{{3.39004039793443*^9, 3.3900405452424307`*^9}, {
  3.3900406417274303`*^9, 3.3900406441644306`*^9}, {3.3900407209224305`*^9, 
  3.3900409707384305`*^9}},
 CellLabel->"In[13]:="],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], 
    PointBox[{{1., 1.5}, {1., 0.5}, {2., 2.3}, {2., 1.3}}]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0.5, 0},
  BaseStyle->{FontFamily -> "Times"},
  Epilog->{
    InsetBox[
     FormBox[
     "\"\\!\\(\\*SubscriptBox[\\(p\\), \\(1\\)]\\)\"", TraditionalForm], {
     0.95, 1}, {1, 0}], 
    InsetBox[
     FormBox[
     "\"\\!\\(\\*SubscriptBox[\\(m\\), \\(1\\)]\\)(\\!\\(\\*SubscriptBox[\\(q\
\\), \\(1\\)]\\)=-1)\"", TraditionalForm], {0.95, 1.5}, {1, 0}], 
    InsetBox[
     FormBox[
     "\"\\!\\(\\*SubscriptBox[\\(m\\), \\(1\\)]\\)(\\!\\(\\*SubscriptBox[\\(q\
\\), \\(1\\)]\\)=+1)\"", TraditionalForm], {0.95, 0.5}, {1, 0}], 
    InsetBox[
     FormBox[
     "\"\\!\\(\\*SubscriptBox[\\(p\\), \\(2\\)]\\)\"", TraditionalForm], {
     2.05, 1.8}, {-1, 0}], 
    InsetBox[
     FormBox[
     "\"\\!\\(\\*SubscriptBox[\\(m\\), \\(2\\)]\\)(\\!\\(\\*SubscriptBox[\\(q\
\\), \\(2\\)]\\)=-1)\"", TraditionalForm], {2.05, 2.3}, {-1, 0}], 
    InsetBox[
     FormBox[
     "\"\\!\\(\\*SubscriptBox[\\(m\\), \\(2\\)]\\)(\\!\\(\\*SubscriptBox[\\(q\
\\), \\(2\\)]\\)=+1)\"", TraditionalForm], {2.05, 1.3}, {-1, 0}], 
    ArrowBox[{{1, 1.5}, {2, 2.3}}, 0.05], 
    ArrowBox[{{1, 1.5}, {2, 1.3}}, 0.05], 
    ArrowBox[{{1, 0.5}, {2, 2.3}}, 0.05], 
    ArrowBox[{{1, 0.5}, {2, 1.3}}, 0.05], 
    InsetBox[
     FormBox[
     "\"A. \\!\\(\\*SubscriptBox[\\(u\\), \
\\(2\\)]\\)(\\!\\(\\*SubscriptBox[\\(q\\), \
\\(1\\)]\\)=-1,\\!\\(\\*SubscriptBox[\\(q\\), \\(2\\)]\\)=-1)\"", 
      TraditionalForm], {1.5, 2}, {1, 0}], 
    InsetBox[
     FormBox[
     "\"B. \\!\\(\\*SubscriptBox[\\(u\\), \
\\(2\\)]\\)(\\!\\(\\*SubscriptBox[\\(q\\), \
\\(1\\)]\\)=-1,\\!\\(\\*SubscriptBox[\\(q\\), \\(2\\)]\\)=+1)\"", 
      TraditionalForm], {1.7, 1.45}, {-1, 0}], 
    InsetBox[
     FormBox[
     "\"C. \\!\\(\\*SubscriptBox[\\(u\\), \
\\(2\\)]\\)(\\!\\(\\*SubscriptBox[\\(q\\), \
\\(1\\)]\\)=+1,\\!\\(\\*SubscriptBox[\\(q\\), \\(2\\)]\\)=-1)\"", 
      TraditionalForm], {1.9, 2}, {-1, 0}], 
    InsetBox[
     FormBox[
     "\"D. \\!\\(\\*SubscriptBox[\\(u\\), \
\\(2\\)]\\)(\\!\\(\\*SubscriptBox[\\(q\\), \
\\(1\\)]\\)=+1,\\!\\(\\*SubscriptBox[\\(q\\), \\(2\\)]\\)=+1)\"", 
      TraditionalForm], {1.5, 0.8}, {-1, 0}], 
    AbsolutePointSize[8], 
    PointBox[{1, 1}], 
    PointBox[{2, 1.8}]},
  Frame->True,
  FrameLabel->{
    FormBox["Time", TraditionalForm], 
    FormBox["Price", TraditionalForm], None, None},
  FrameTicks->{{{0, 
      FormBox["0", TraditionalForm]}, {1, 
      FormBox["1", TraditionalForm]}, {2, 
      FormBox["2", TraditionalForm]}}, Automatic, None, None},
  PlotRange->{{0.5, 2.5}, {0, 2.5}},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic},
  Prolog->AbsolutePointSize[5]]], "Output",
 CellChangeTimes->{
  3.3900409123414307`*^9, {3.3900409456404305`*^9, 3.3900409717874303`*^9}, 
   3.3900410369694304`*^9, 3.3900411092164307`*^9},
 CellLabel->"Out[13]="]
}, Open  ]],

Cell[TextData[{
 "Given ",
 Cell[BoxData[
  SubscriptBox["p", "1"]]],
 " and ",
 Cell[BoxData["c"]],
 ", a choice of ",
 Cell[BoxData[
  SubscriptBox["q", "1"]]],
 " determines ",
 Cell[BoxData["m"]],
 ", and similarly for ",
 Cell[BoxData[
  SubscriptBox["p", "2"]]],
 ".\nTherefore, setting ",
 Cell[BoxData[
  SubscriptBox["q", "1"]]],
 " and ",
 Cell[BoxData[
  SubscriptBox["q", "2"]]],
 " fixes ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["u", "2"], "=", 
   RowBox[{
    SubscriptBox["m", "2"], "-", 
    SubscriptBox["m", "1"]}]}]]],
 ".\nSince ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "1"], ",", 
   RowBox[{
    SubscriptBox["q", "2"], "\[Element]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", 
      RowBox[{"+", "1"}]}], "}"}]}]}]]],
 ", there are four possible paths (values) for ",
 Cell[BoxData[
  SubscriptBox["u", "2"]]],
 ". These are labeled A, B, C and D in the figure. Since low values of ",
 Cell[BoxData[
  RowBox[{"|", 
   SubscriptBox["u", "2"], "|"}]]],
 " are more likely than high values, the relative lengths of these paths \
indicate the relative likelihood of the ",
 Cell[BoxData[
  RowBox[{"(", 
   RowBox[{
    SubscriptBox["q", "1"], ",", 
    SubscriptBox["q", "2"]}], ")"}]]],
 " realizations that determined them:"
}], "Text"],

Cell[TextData[{
 "Intuitively, path B is the shortest, so it is most likely that ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    SubscriptBox["q", "1"], "=", 
    RowBox[{"-", "1"}]}], ",", " ", 
   RowBox[{
    SubscriptBox["q", "2"], "=", 
    RowBox[{"+", "1"}]}]}]]],
 "."
}], "BulletedList"],

Cell[TextData[{
 "Path C is the longest, so it is least likely that ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    SubscriptBox["q", "1"], "=", 
    RowBox[{"+", "1"}]}], ",", " ", 
   RowBox[{
    SubscriptBox["q", "2"], "=", 
    RowBox[{"-", "1"}]}]}]]],
 "."
}], "BulletedList"],

Cell[TextData[{
 "Paths A and D are of equal length, corresponding to the realizations ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "1"], "=", 
   RowBox[{
    SubscriptBox["q", "2"], "=", 
    RowBox[{"-", "1"}]}]}]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "1"], "=", 
   RowBox[{
    SubscriptBox["q", "2"], "=", 
    RowBox[{"+", "1"}]}]}]]],
 "."
}], "BulletedList"],

Cell["We now turn to a more exact treatment.", "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{"Clear", "[", 
   RowBox[{"s", ",", 
    SubscriptBox["p", "1"], ",", 
    SubscriptBox["p", "2"], ",", "c"}], "]"}], ";"}]], "Input",
 CellLabel->"In[14]:="]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 "The joint density of ",
 Cell[BoxData[
  SubscriptBox["q", "1"]]],
 " and ",
 Cell[BoxData[
  SubscriptBox["q", "2"]]]
}], "Subsubsection"],

Cell[TextData[{
 "The density function for ",
 Cell[BoxData["u"]],
 " is ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"f", 
    RowBox[{"(", 
     SubscriptBox["u", "2"], ")"}]}], "="}]]]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"NormalDistribution", "[", 
    RowBox[{"0", ",", 
     SubscriptBox["\[Sigma]", "u"]}], "]"}], ",", "u"}], "]"}]], "Input",
 CellLabel->"In[15]:="],

Cell[BoxData[
 FractionBox[
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox["u", "2"], 
     RowBox[{"2", " ", 
      SubsuperscriptBox["\[Sigma]", "u", "2"]}]]}]], 
  RowBox[{
   SqrtBox[
    RowBox[{"2", " ", "\[Pi]"}]], " ", 
   SubscriptBox["\[Sigma]", "u"]}]]], "Output",
 CellChangeTimes->{3.39004100084743*^9, 3.3900410370104303`*^9, 
  3.3900411093404303`*^9},
 CellLabel->"Out[15]="]
}, Open  ]],

Cell[TextData[{
 "By rearranging the price change specification, ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["u", "2"], "="}]]]
}], "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["u", "2"], "[", 
   RowBox[{
    RowBox[{
     SubscriptBox["q", "1"], ":", "_"}], ",", 
    RowBox[{
     SubscriptBox["q", "2"], ":", "_"}]}], "]"}], ":=", 
  RowBox[{
   SubscriptBox["\[CapitalDelta]p", "2"], "-", 
   RowBox[{"c", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["q", "2"], "-", 
      SubscriptBox["q", "1"]}], ")"}]}]}]}]], "Input",
 CellLabel->"In[16]:="],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  SubscriptBox["u", "2"], "[", 
  RowBox[{
   SubscriptBox["q", "1"], ",", 
   SubscriptBox["q", "2"]}], "]"}]], "Input",
 CellLabel->"In[17]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"-", "c"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      SubscriptBox["q", "1"]}], "+", 
     SubscriptBox["q", "2"]}], ")"}]}], "+", 
  SubscriptBox["\[CapitalDelta]p", "2"]}]], "Output",
 CellChangeTimes->{3.39003357863943*^9, 3.3900410009834304`*^9, 
  3.3900410370864305`*^9, 3.3900411094104304`*^9},
 CellLabel->"Out[17]="]
}, Open  ]],

Cell[TextData[{
 "The probability ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"Pr", "[", 
    RowBox[{
     SubscriptBox["q", "1"], ",", 
     SubscriptBox["q", "2"]}], "]"}], "\[Proportional]", 
   RowBox[{"f", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["u", "2"], "[", 
      RowBox[{
       SubscriptBox["q", "1"], ",", 
       SubscriptBox["q", "2"]}], "]"}], ")"}]}]}]]]
}], "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["Pr", "Rule"], "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Pr", "[", 
      RowBox[{
       RowBox[{
        SubscriptBox["q", "1"], ":", "_"}], ",", 
       RowBox[{
        SubscriptBox["q", "2"], ":", "_"}]}], "]"}], ":>", 
     RowBox[{"PDF", "[", 
      RowBox[{
       RowBox[{"NormalDistribution", "[", 
        RowBox[{"0", ",", 
         SubscriptBox["\[Sigma]", "u"]}], "]"}], ",", 
       RowBox[{
        SubscriptBox["u", "2"], "[", 
        RowBox[{
         SubscriptBox["q", "1"], ",", 
         SubscriptBox["q", "2"]}], "]"}]}], "]"}]}], "//", "Simplify"}]}], 
  ";"}]], "Input",
 CellLabel->"In[18]:="],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Pr", "[", 
   RowBox[{
    SubscriptBox["q", "1"], ",", 
    SubscriptBox["q", "2"]}], "]"}], "/.", 
  SubscriptBox["Pr", "Rule"]}]], "Input",
 CellLabel->"In[19]:="],

Cell[BoxData[
 FractionBox[
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "c"}], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            SubscriptBox["q", "1"]}], "+", 
           SubscriptBox["q", "2"]}], ")"}]}], "+", 
        SubscriptBox["\[CapitalDelta]p", "2"]}], ")"}], "2"], 
     RowBox[{"2", " ", 
      SubsuperscriptBox["\[Sigma]", "u", "2"]}]]}]], 
  RowBox[{
   SqrtBox[
    RowBox[{"2", " ", "\[Pi]"}]], " ", 
   SubscriptBox["\[Sigma]", "u"]}]]], "Output",
 CellChangeTimes->{
  3.3900335788604307`*^9, {3.39004100126443*^9, 3.3900410371614304`*^9}, 
   3.3900411094904304`*^9},
 CellLabel->"Out[19]="]
}, Open  ]],

Cell["The possible outcomes are:", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"Outcomes", "=", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1"}], ",", 
        RowBox[{"-", "1"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1"}], ",", 
        RowBox[{"+", "1"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"+", "1"}], ",", 
        RowBox[{"-", "1"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"+", "1"}], ",", 
        RowBox[{"+", "1"}]}], "}"}]}], "}"}]}], ";"}], 
  " "}], "\[IndentingNewLine]", 
 RowBox[{"TableForm", "[", 
  RowBox[{"Outcomes", ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"None", ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["q", "1"], ",", 
        SubscriptBox["q", "2"]}], "}"}]}], "}"}]}]}], "]"}]}], "Input",
 CellLabel->"In[20]:="],

Cell[BoxData[
 TagBox[
  TagBox[GridBox[{
     {
      TagBox[
       SubscriptBox["q", "1"],
       HoldForm], 
      TagBox[
       SubscriptBox["q", "2"],
       HoldForm]},
     {
      RowBox[{"-", "1"}], 
      RowBox[{"-", "1"}]},
     {
      RowBox[{"-", "1"}], "1"},
     {"1", 
      RowBox[{"-", "1"}]},
     {"1", "1"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxDividers->{
     "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}],
   {None, OutputFormsDump`HeadedColumns}],
  Function[BoxForm`e$, 
   TableForm[
   BoxForm`e$, 
    TableHeadings -> {
     None, {$CellContext`q\[UnderBracket]Subscript\[UnderBracket]1, \
$CellContext`q\[UnderBracket]Subscript\[UnderBracket]2}}]]]], "Output",
 CellChangeTimes->{3.3900335789234304`*^9, 3.39004100137543*^9, 
  3.3900410372304306`*^9, 3.3900411095584307`*^9},
 CellLabel->"Out[21]//TableForm="]
}, Open  ]],

Cell["We'll normalize by the sum of the probabilities:", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrSum", "=", 
  RowBox[{"Plus", " ", "@@", " ", 
   RowBox[{"Apply", "[", 
    RowBox[{"Pr", ",", "Outcomes", ",", 
     RowBox[{"{", "1", "}"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[22]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{"Pr", "[", 
   RowBox[{
    RowBox[{"-", "1"}], ",", 
    RowBox[{"-", "1"}]}], "]"}], "+", 
  RowBox[{"Pr", "[", 
   RowBox[{
    RowBox[{"-", "1"}], ",", "1"}], "]"}], "+", 
  RowBox[{"Pr", "[", 
   RowBox[{"1", ",", 
    RowBox[{"-", "1"}]}], "]"}], "+", 
  RowBox[{"Pr", "[", 
   RowBox[{"1", ",", "1"}], "]"}]}]], "Output",
 CellChangeTimes->{3.39003357897943*^9, 3.3900410014914303`*^9, 
  3.3900410372984304`*^9, 3.3900411096154304`*^9},
 CellLabel->"Out[22]="]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["PrN", "Rule"], "=", 
   RowBox[{
    RowBox[{"PrN", "[", 
     RowBox[{
      RowBox[{
       SubscriptBox["q", "1"], ":", "_"}], ",", 
      RowBox[{
       SubscriptBox["q", "2"], ":", "_"}]}], "]"}], "\[RuleDelayed]", 
    RowBox[{
     RowBox[{"Pr", "[", 
      RowBox[{
       SubscriptBox["q", "1"], ",", 
       SubscriptBox["q", "2"]}], "]"}], "/", "PrSum"}]}]}], ";"}]], "Input",
 CellLabel->"In[23]:="],

Cell["The normalized probability is:", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"PrN", "[", 
   RowBox[{
    SubscriptBox["q", "1"], ",", 
    SubscriptBox["q", "2"]}], "]"}], "/.", 
  SubscriptBox["PrN", "Rule"]}]], "Input",
 CellLabel->"In[24]:="],

Cell[BoxData[
 FractionBox[
  RowBox[{"Pr", "[", 
   RowBox[{
    SubscriptBox["q", "1"], ",", 
    SubscriptBox["q", "2"]}], "]"}], 
  RowBox[{
   RowBox[{"Pr", "[", 
    RowBox[{
     RowBox[{"-", "1"}], ",", 
     RowBox[{"-", "1"}]}], "]"}], "+", 
   RowBox[{"Pr", "[", 
    RowBox[{
     RowBox[{"-", "1"}], ",", "1"}], "]"}], "+", 
   RowBox[{"Pr", "[", 
    RowBox[{"1", ",", 
     RowBox[{"-", "1"}]}], "]"}], "+", 
   RowBox[{"Pr", "[", 
    RowBox[{"1", ",", "1"}], "]"}]}]]], "Output",
 CellChangeTimes->{3.3900335793664303`*^9, 3.3900410015834303`*^9, 
  3.3900410373734303`*^9, 3.3900411096824303`*^9},
 CellLabel->"Out[24]="]
}, Open  ]],

Cell["For demonstration purposes, here are some values:", "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{"nValues", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["\[CapitalDelta]p", "2"], "\[Rule]", ".8"}], ",", 
     RowBox[{
      SubscriptBox["\[Sigma]", "u"], "\[Rule]", "1"}], ",", 
     RowBox[{"c", "\[Rule]", "0.5"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[25]:="],

Cell["With thes values, the normalized probabilities are:", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{
   RowBox[{"Transpose", "[", 
    RowBox[{"Append", "[", 
     RowBox[{
      RowBox[{"Transpose", "[", "Outcomes", "]"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"Apply", "[", 
          RowBox[{"PrN", ",", "Outcomes", ",", 
           RowBox[{"{", "1", "}"}]}], "]"}], "/.", 
         SubscriptBox["PrN", "Rule"]}], "/.", 
        SubscriptBox["Pr", "Rule"]}], "/.", "nValues"}]}], "]"}], "]"}], ",", 
   "\[IndentingNewLine]", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"None", ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["q", "1"], ",", 
        SubscriptBox["q", "2"], ",", "\"\<Probability\>\""}], "}"}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[26]:="],

Cell[BoxData[
 TagBox[
  TagBox[GridBox[{
     {
      TagBox[
       SubscriptBox["q", "1"],
       HoldForm], 
      TagBox[
       SubscriptBox["q", "2"],
       HoldForm], 
      TagBox["\<\"Probability\"\>",
       HoldForm]},
     {
      RowBox[{"-", "1"}], 
      RowBox[{"-", "1"}], "0.2760607869928766`"},
     {
      RowBox[{"-", "1"}], "1", "0.37264308474869745`"},
     {"1", 
      RowBox[{"-", "1"}], "0.07523534126554927`"},
     {"1", "1", "0.2760607869928766`"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxDividers->{
     "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}],
   {None, OutputFormsDump`HeadedColumns}],
  Function[BoxForm`e$, 
   TableForm[
   BoxForm`e$, 
    TableHeadings -> {
     None, {$CellContext`q\[UnderBracket]Subscript\[UnderBracket]1, \
$CellContext`q\[UnderBracket]Subscript\[UnderBracket]2, 
       "Probability"}}]]]], "Output",
 CellChangeTimes->{3.3900411097604303`*^9},
 CellLabel->"Out[26]//TableForm="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Gibbs sampler", "Subsubsection"],

Cell["\<\
We can draw the trade direction indicator variables jointly in this case. \
There's no need to use a Gibbs sampler. But for illustration purposes, let's \
build it.\
\>", "Text"],

Cell[TextData[{
 "The required conditional probabilities are ",
 Cell[BoxData[
  RowBox[{"Pr", "[", 
   RowBox[{
    SubscriptBox["q", "1"], "|", 
    SubscriptBox["q", "2"]}], "]"}]]],
 " and ",
 Cell[BoxData[
  RowBox[{"Pr", "[", 
   RowBox[{
    SubscriptBox["q", "2"], "|", 
    SubscriptBox["q", "1"]}], "]"}]]],
 ".  These may be computed directly from joint distribution given above, but \
it is usually computationally easier to work with the odds ratio. \nFor \
example, ",
 Cell[BoxData[
  RowBox[{
   RowBox[{
    SubscriptBox["u", "2"], "[", 
    RowBox[{
     SubscriptBox["q", "1"], ",", 
     SubscriptBox["q", "2"]}], "]"}], "="}]]]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  SubscriptBox["u", "2"], "[", 
  RowBox[{
   SubscriptBox["q", "1"], ",", 
   SubscriptBox["q", "2"]}], "]"}]], "Input",
 CellLabel->"In[27]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"-", "c"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      SubscriptBox["q", "1"]}], "+", 
     SubscriptBox["q", "2"]}], ")"}]}], "+", 
  SubscriptBox["\[CapitalDelta]p", "2"]}]], "Output",
 CellChangeTimes->{3.3900411098414307`*^9},
 CellLabel->"Out[27]="]
}, Open  ]],

Cell["Its density is:", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"NormalDistribution", "[", 
     RowBox[{"0", ",", 
      SubscriptBox["\[Sigma]", "u"]}], "]"}], ",", 
    RowBox[{
     SubscriptBox["u", "2"], "[", 
     RowBox[{
      SubscriptBox["q", "1"], ",", 
      SubscriptBox["q", "2"]}], "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[28]:="],

Cell[BoxData[
 FractionBox[
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "c"}], " ", 
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            SubscriptBox["q", "1"]}], "+", 
           SubscriptBox["q", "2"]}], ")"}]}], "+", 
        SubscriptBox["\[CapitalDelta]p", "2"]}], ")"}], "2"], 
     RowBox[{"2", " ", 
      SubsuperscriptBox["\[Sigma]", "u", "2"]}]]}]], 
  RowBox[{
   SqrtBox[
    RowBox[{"2", " ", "\[Pi]"}]], " ", 
   SubscriptBox["\[Sigma]", "u"]}]]], "Output",
 CellChangeTimes->{3.3900410374514303`*^9, 3.3900411098994303`*^9},
 CellLabel->"Out[28]="]
}, Open  ]],

Cell[TextData[{
 "So given ",
 Cell[BoxData[
  SubscriptBox["q", "2"]]],
 ", the odds in favor of a buy at time 1 are ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"Odds", 
    RowBox[{"(", "Buy", ")"}]}], "=", 
   FractionBox[
    RowBox[{"Pr", "[", 
     RowBox[{
      SubscriptBox["q", "1"], "=", 
      RowBox[{
       RowBox[{"+", "1"}], "|", "..."}]}], "]"}], 
    RowBox[{"Pr", "[", 
     RowBox[{
      SubscriptBox["q", "1"], "=", 
      RowBox[{
       RowBox[{"-", "1"}], "|", "..."}]}], "]"}]]}]]],
 "="
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"OddsBuy1", "=", 
  RowBox[{
   FractionBox[
    RowBox[{"f", "/.", 
     RowBox[{
      SubscriptBox["q", "1"], "\[Rule]", 
      RowBox[{"+", "1"}]}]}], 
    RowBox[{"f", "/.", 
     RowBox[{
      SubscriptBox["q", "1"], "\[Rule]", 
      RowBox[{"-", "1"}]}]}]], "//", "Simplify"}]}]], "Input",
 CellLabel->"In[29]:="],

Cell[BoxData[
 SuperscriptBox["\[ExponentialE]", 
  FractionBox[
   RowBox[{"2", " ", "c", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"c", " ", 
       SubscriptBox["q", "2"]}], "-", 
      SubscriptBox["\[CapitalDelta]p", "2"]}], ")"}]}], 
   SubsuperscriptBox["\[Sigma]", "u", "2"]]]], "Output",
 CellChangeTimes->{3.3900335794844303`*^9, 3.3900410375274305`*^9, 
  3.3900411100564303`*^9},
 CellLabel->"Out[29]="]
}, Open  ]],

Cell[TextData[{
 "Then ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"Pr", "[", "Buy", "]"}], "=", 
   FractionBox[
    RowBox[{"Odds", 
     RowBox[{"(", "Buy", ")"}]}], 
    RowBox[{"1", "+", 
     RowBox[{"Odds", 
      RowBox[{"(", "Buy", ")"}]}]}]]}]]],
 ". We compute this probability and make a draw for ",
 Cell[BoxData[
  SubscriptBox["q", "1"]]],
 "."
}], "Text"],

Cell[TextData[{
 "For the particular numeric values we worked with above ",
 Cell[BoxData[
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     SubscriptBox["\[CapitalDelta]p", "2"], "=", "0.8"}], ",", " ", 
    RowBox[{"c", "=", "0.5"}], ",", " ", 
    RowBox[{
     SubscriptBox["\[Sigma]", "u"], "=", "1"}]}], ")"}]]],
 ", these odds, for ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "2"], "=", 
   RowBox[{"+", "1"}]}]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "2"], "=", 
   RowBox[{"-", "1"}]}]]],
 " are:"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"OddsBuy1", "/.", 
     RowBox[{
      SubscriptBox["\[CapitalDelta]p", "2"], "\[Rule]", "0.8"}]}], "/.", 
    RowBox[{"c", "\[Rule]", "0.5"}]}], "/.", 
   RowBox[{
    SubscriptBox["\[Sigma]", "u"], "\[Rule]", "1"}]}], "/.", 
  RowBox[{
   SubscriptBox["q", "2"], "\[Rule]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"+", "1"}], ",", 
     RowBox[{"-", "1"}]}], "}"}]}]}]], "Input",
 CellLabel->"In[30]:="],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.7408182206817179`", ",", "0.2725317930340126`"}], 
  "}"}]], "Output",
 CellChangeTimes->{3.3900411101314306`*^9},
 CellLabel->"Out[30]="]
}, Open  ]],

Cell[TextData[{
 "So, for example, if ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "2"], "=", 
   RowBox[{"+", "1"}]}]]],
 ", ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"Pr", "[", 
    RowBox[{
     SubscriptBox["q", "1"], "=", 
     RowBox[{
      RowBox[{"+", "1"}], "|", "..."}]}], "]"}], "="}]]]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"%", "[", 
   RowBox[{"[", "1", "]"}], "]"}], "/", 
  RowBox[{"(", 
   RowBox[{"1", "+", 
    RowBox[{"%", "[", 
     RowBox[{"[", "1", "]"}], "]"}]}], ")"}]}]], "Input",
 CellLabel->"In[31]:="],

Cell[BoxData["0.4255574831883411`"], "Output",
 CellChangeTimes->{3.39004103759943*^9, 3.3900411102004304`*^9},
 CellLabel->"Out[31]="]
}, Open  ]],

Cell[TextData[{
 "Similarly, given ",
 Cell[BoxData[
  SubscriptBox["q", "1"]]],
 ", the odds in favor of a buy at time 2 are ",
 Cell[BoxData[
  FractionBox[
   RowBox[{"Pr", "[", 
    RowBox[{
     SubscriptBox["q", "2"], "=", 
     RowBox[{
      RowBox[{"+", "1"}], "|", "..."}]}], "]"}], 
   RowBox[{"Pr", "[", 
    RowBox[{
     SubscriptBox["q", "2"], "=", 
     RowBox[{
      RowBox[{"-", "1"}], "|", "..."}]}], "]"}]]]],
 "="
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"OddsBuy2", "=", 
  RowBox[{
   FractionBox[
    RowBox[{"f", "/.", 
     RowBox[{
      SubscriptBox["q", "2"], "\[Rule]", 
      RowBox[{"+", "1"}]}]}], 
    RowBox[{"f", "/.", 
     RowBox[{
      SubscriptBox["q", "2"], "\[Rule]", 
      RowBox[{"-", "1"}]}]}]], "//", "Simplify"}]}]], "Input",
 CellLabel->"In[32]:="],

Cell[BoxData[
 SuperscriptBox["\[ExponentialE]", 
  FractionBox[
   RowBox[{"2", " ", "c", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"c", " ", 
       SubscriptBox["q", "1"]}], "+", 
      SubscriptBox["\[CapitalDelta]p", "2"]}], ")"}]}], 
   SubsuperscriptBox["\[Sigma]", "u", "2"]]]], "Output",
 CellChangeTimes->{3.3900335795394306`*^9, 3.3900410376794305`*^9, 
  3.3900411102784305`*^9},
 CellLabel->"Out[32]="]
}, Open  ]],

Cell[TextData[{
 "For the numeric values, these are (for ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "1"], "=", 
   RowBox[{"+", "1"}]}]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "1"], "=", 
   RowBox[{"-", "1"}]}]]],
 "):"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"OddsBuy2", "/.", 
     RowBox[{
      SubscriptBox["\[CapitalDelta]p", "2"], "\[Rule]", "0.8"}]}], "/.", 
    RowBox[{"c", "\[Rule]", "0.5"}]}], "/.", 
   RowBox[{
    SubscriptBox["\[Sigma]", "u"], "\[Rule]", "1"}]}], "/.", 
  RowBox[{
   SubscriptBox["q", "1"], "\[Rule]", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"+", "1"}], ",", 
     RowBox[{"-", "1"}]}], "}"}]}]}]], "Input",
 CellLabel->"In[33]:="],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3.6692966676192444`", ",", "1.3498588075760032`"}], 
  "}"}]], "Output",
 CellChangeTimes->{3.3900411103454304`*^9},
 CellLabel->"Out[33]="]
}, Open  ]],

Cell["The Gibbs sampler involves the following steps:", "Text"],

Cell["\<\
We construct a series of realizations in the following fashion:\
\>", "Text"],

Cell[TextData[{
 "Initialize ",
 Cell[BoxData[
  RowBox[{
   SubsuperscriptBox["q", "1", 
    RowBox[{"[", "0", "]"}]], ",", 
   RowBox[{
    SubsuperscriptBox["q", "2", 
     RowBox[{"[", "0", "]"}]], "=", 
    RowBox[{"\[PlusMinus]", "1"}]}]}]]],
 " (it doesn't matter which)."
}], "Text"],

Cell["The ith sweep involves the following steps:", "Text"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "1", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"Pr", "[", 
   RowBox[{
    SubscriptBox["q", "1"], "|", 
    SubsuperscriptBox["q", "2", 
     RowBox[{"[", 
      RowBox[{"i", "-", "1"}], "]"}]]}], "]"}]]]
}], "BulletedList"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "2", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"Pr", "[", 
   RowBox[{
    SubscriptBox["q", "2"], "|", 
    SubsuperscriptBox["q", "1", 
     RowBox[{"[", "i", "]"}]]}], "]"}]]]
}], "BulletedList"],

Cell[TextData[{
 "After ",
 StyleBox["N",
  FontSlant->"Italic"],
 " sweeps we'll ",
 " have a series of ",
 StyleBox["N",
  FontSlant->"Italic"],
 " simulated realizations: ",
 Cell[BoxData[
  RowBox[{
   SuperscriptBox["q", 
    RowBox[{"[", "0", "]"}]], ",", 
   SuperscriptBox["q", 
    RowBox[{"[", "1", "]"}]], ",", "...", ",", 
   SuperscriptBox["q", 
    RowBox[{"[", "N", "]"}]]}]]],
 " where ",
 Cell[BoxData[
  RowBox[{
   SuperscriptBox["q", 
    RowBox[{"[", "i", "]"}]], "=", 
   RowBox[{"(", 
    RowBox[{
     SubsuperscriptBox["q", "1", 
      RowBox[{"[", "i", "]"}]], ",", 
     SubsuperscriptBox["q", "2", 
      RowBox[{"[", "i", "]"}]]}], ")"}]}]]],
 ".\nIn the limit, as ",
 Cell[BoxData[
  RowBox[{"N", "\[Rule]", "\[Infinity]"}]]],
 ", the distribution of ",
 Cell[BoxData[
  SuperscriptBox["q", 
   RowBox[{"[", "N", "]"}]]]],
 " is the joint distribution ",
 Cell[BoxData[
  RowBox[{"Pr", "[", 
   RowBox[{
    SubscriptBox["q", "1"], ",", 
    SubscriptBox["q", "2"]}], "]"}]]]
}], "Text"],

Cell[TextData[{
 "The sampler was implemented in a SAS program (",
 StyleBox["RollGibbs 2-trade case.sas",
  FontSlant->"Italic"],
 "), which was run for 10,000 sweeps. The tabulated frequencies of the \
simulated draws were:"
}], "Text"],

Cell["\<\
-----------------------------------------------
|                            |   N    |  PctN  |
|----------------------------+--------+--------|
|outcome                     |        |        |
|q1= 1 q2= 1                 |   2,785|    27.9|
|q1= 1 q2=-1                 |     752|     7.5|
|q1=-1 q2= 1                 |   3,751|    37.5|
|q1=-1 q2=-1                 |   2,712|    27.1|
|All                         |  10,000|   100.0|
------------------------------------------------\
\>", "Text",
 FontFamily->"Courier New",
 FontSize->10],

Cell["Compare these to the computed probabilities above.", "Text"],

Cell["\<\
Why do we ever need to use Gibbs sampler when we can compute path \
probabilities directly?\
\>", "Text"],

Cell[TextData[{
 "We need to compute path probabilities for the entire sample. With two price \
observations, there are ",
 Cell[BoxData[
  RowBox[{
   SuperscriptBox["2", "2"], "=", "4"}]]],
 " buy/sell paths.\nA year contains about 250 trading days. The number of \
buy/sell paths is:"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 SuperscriptBox["2", "250"]], "Input",
 CellLabel->"In[34]:="],

Cell[BoxData[\
"1809251394333065553493296640760748560207343510400633813116524750123642650624"\
], "Output",
 CellChangeTimes->{3.3900335796014304`*^9, 3.39004103775543*^9, 
  3.3900411104114304`*^9},
 CellLabel->"Out[34]="]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["The T-trade case", "Subsection"],

Cell["A sweep of the sampler will involve:", "Text"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "1", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"Pr", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["q", "1"], "|", 
      SubsuperscriptBox["q", "2", 
       RowBox[{"[", 
        RowBox[{"i", "-", "1"}], "]"}]]}], ",", 
     SubsuperscriptBox["q", "3", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]], ",", "...", ",", " ", 
     SubsuperscriptBox["q", "T", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 "\nDraw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "2", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"Pr", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["q", "2"], "|", 
      SubsuperscriptBox["q", "1", 
       RowBox[{"[", "i", "]"}]]}], ",", 
     SubsuperscriptBox["q", "3", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]], ",", "...", ",", " ", 
     SubsuperscriptBox["q", "T", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 "\n...\nDraw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "T", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"Pr", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["q", "T"], "|", 
      SubsuperscriptBox["q", "1", 
       RowBox[{"[", "i", "]"}]]}], ",", 
     SubsuperscriptBox["q", "2", 
      RowBox[{"[", "i", "]"}]], ",", "...", ",", " ", 
     SubsuperscriptBox["q", 
      RowBox[{"T", "-", "1"}], 
      RowBox[{"[", "i", "]"}]]}], ")"}]}]]]
}], "BulletedList"],

Cell[TextData[{
 "In general, ",
 Cell[BoxData[
  RowBox[{"Pr", "[", 
   RowBox[{
    SubscriptBox["q", "t"], "|", "..."}], "]"}]]],
 "depends only on the adjacent trades - those at times ",
 Cell[BoxData[
  RowBox[{"t", "-", "1"}]]],
 " and ",
 Cell[BoxData[
  RowBox[{"t", "+", "1"}]]],
 ". So the sampler becomes:"
}], "Text"],

Cell[TextData[{
 "Draw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "1", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"Pr", 
   RowBox[{"(", 
    RowBox[{
     SubscriptBox["q", "1"], "|", 
     SubsuperscriptBox["q", "2", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 "\nDraw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "2", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"Pr", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["q", "2"], "|", 
      SubsuperscriptBox["q", "1", 
       RowBox[{"[", "i", "]"}]]}], ",", 
     SubsuperscriptBox["q", "3", 
      RowBox[{"[", 
       RowBox[{"i", "-", "1"}], "]"}]]}], ")"}]}]]],
 "\n...\nDraw ",
 Cell[BoxData[
  SubsuperscriptBox["q", "T", 
   RowBox[{"[", "i", "]"}]]]],
 " from ",
 Cell[BoxData[
  RowBox[{"Pr", 
   RowBox[{"(", 
    RowBox[{
     SubscriptBox["q", "T"], "|", " ", 
     SubsuperscriptBox["q", 
      RowBox[{"T", "-", "1"}], 
      RowBox[{"[", "i", "]"}]]}], ")"}]}]]]
}], "BulletedList"],

Cell[TextData[{
 "The first draw, for ",
 Cell[BoxData[
  SubscriptBox["q", "1"]]],
 ", is the same as the draw for ",
 Cell[BoxData[
  SubscriptBox["q", "1"]]],
 " in the ",
 Cell[BoxData[
  RowBox[{"T", "=", "2"}]]],
 " case."
}], "Text"],

Cell[TextData[{
 "The last draw, for ",
 Cell[BoxData[
  SubscriptBox["q", "T"]]],
 ", is the same as the draw for ",
 Cell[BoxData[
  SubscriptBox["q", "2"]]],
 " in the ",
 Cell[BoxData[
  RowBox[{"T", "=", "2"}]]],
 " case."
}], "Text"],

Cell["We now turn to the middle draws.", "Text"],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["u", "Rule"], " ", "=", " ", 
   RowBox[{
    SubscriptBox["u", "t_"], "\[RuleDelayed]", 
    RowBox[{
     SubscriptBox["\[CapitalDelta]p", "t"], "+", 
     RowBox[{"c", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["q", "t"], "-", 
        SubscriptBox["q", 
         RowBox[{"t", "-", "1"}]]}], ")"}]}]}]}]}], ";"}]], "Input",
 CellLabel->"In[35]:="],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  SubscriptBox["u", 
   RowBox[{"t", "+", "1"}]], "/.", 
  SubscriptBox["u", "Rule"]}]], "Input",
 CellLabel->"In[36]:="],

Cell[BoxData[
 RowBox[{
  RowBox[{"c", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      SubscriptBox["q", "t"]}], "+", 
     SubscriptBox["q", 
      RowBox[{"1", "+", "t"}]]}], ")"}]}], "+", 
  SubscriptBox["\[CapitalDelta]p", 
   RowBox[{"1", "+", "t"}]]}]], "Output",
 CellChangeTimes->{3.3900335796674304`*^9, 3.3900410378814306`*^9, 
  3.3900411104884305`*^9},
 CellLabel->"Out[36]="]
}, Open  ]],

Cell[TextData[{
 "Since the ",
 Cell[BoxData[
  SubscriptBox["u", "t"]]],
 " are assumed to be independent, the joint density ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"f", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["u", "t"], ",", 
      SubscriptBox["u", 
       RowBox[{"t", "+", "1"}]]}], ")"}]}], "\[Proportional]"}]]]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"NormalDistribution", "[", 
     RowBox[{"0", ",", 
      SubscriptBox["\[Sigma]", "u"]}], "]"}], ",", 
    SubscriptBox["u", "t"]}], "]"}], 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"NormalDistribution", "[", 
     RowBox[{"0", ",", 
      SubscriptBox["\[Sigma]", "u"]}], "]"}], ",", 
    SubscriptBox["u", 
     RowBox[{"t", "+", "1"}]]}], "]"}]}]], "Input",
 CellLabel->"In[37]:="],

Cell[BoxData[
 FractionBox[
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{
    RowBox[{"-", 
     FractionBox[
      SubsuperscriptBox["u", "t", "2"], 
      RowBox[{"2", " ", 
       SubsuperscriptBox["\[Sigma]", "u", "2"]}]]}], "-", 
    FractionBox[
     SubsuperscriptBox["u", 
      RowBox[{"1", "+", "t"}], "2"], 
     RowBox[{"2", " ", 
      SubsuperscriptBox["\[Sigma]", "u", "2"]}]]}]], 
  RowBox[{"2", " ", "\[Pi]", " ", 
   SubsuperscriptBox["\[Sigma]", "u", "2"]}]]], "Output",
 CellChangeTimes->{3.39003357972443*^9, 3.3900410379174304`*^9, 
  3.3900411106744304`*^9},
 CellLabel->"Out[37]="]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f", "=", 
  RowBox[{
   RowBox[{"%", "/.", 
    SubscriptBox["u", "Rule"]}], "//", "Simplify"}]}]], "Input",
 CellLabel->"In[38]:="],

Cell[BoxData[
 FractionBox[
  SuperscriptBox["\[ExponentialE]", 
   RowBox[{"-", 
    FractionBox[
     RowBox[{
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "c"}], " ", 
          SubscriptBox["q", 
           RowBox[{
            RowBox[{"-", "1"}], "+", "t"}]]}], "+", 
         RowBox[{"c", " ", 
          SubscriptBox["q", "t"]}], "+", 
         SubscriptBox["\[CapitalDelta]p", "t"]}], ")"}], "2"], "+", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          RowBox[{"-", "c"}], " ", 
          SubscriptBox["q", "t"]}], "+", 
         RowBox[{"c", " ", 
          SubscriptBox["q", 
           RowBox[{"1", "+", "t"}]]}], "+", 
         SubscriptBox["\[CapitalDelta]p", 
          RowBox[{"1", "+", "t"}]]}], ")"}], "2"]}], 
     RowBox[{"2", " ", 
      SubsuperscriptBox["\[Sigma]", "u", "2"]}]]}]], 
  RowBox[{"2", " ", "\[Pi]", " ", 
   SubsuperscriptBox["\[Sigma]", "u", "2"]}]]], "Output",
 CellChangeTimes->{3.3900335798294306`*^9, 3.3900410380574303`*^9, 
  3.3900411109074306`*^9},
 CellLabel->"Out[38]="]
}, Open  ]],

Cell["The odds ratio is:", "Text"],

Cell[TextData[{
 " ",
 Cell[BoxData[
  RowBox[{"Odds", "=", 
   FractionBox[
    RowBox[{"Pr", "[", 
     RowBox[{
      SubscriptBox["q", "t"], "=", 
      RowBox[{
       RowBox[{"+", "1"}], "|", "..."}]}], "]"}], 
    RowBox[{"Pr", "[", 
     RowBox[{
      SubscriptBox["q", "t"], "=", 
      RowBox[{
       RowBox[{"-", "1"}], "|", "..."}]}], "]"}]]}]]]
}], "NumberedEquation"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Odds", "=", 
  RowBox[{
   FractionBox[
    RowBox[{"f", "/.", 
     RowBox[{
      SubscriptBox["q", "t"], "\[Rule]", 
      RowBox[{"+", "1"}]}]}], 
    RowBox[{"f", "/.", 
     RowBox[{
      SubscriptBox["q", "t"], "\[Rule]", 
      RowBox[{"-", "1"}]}]}]], "//", "Simplify"}]}]], "Input",
 CellLabel->"In[39]:="],

Cell[BoxData[
 SuperscriptBox["\[ExponentialE]", 
  FractionBox[
   RowBox[{"2", " ", "c", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"c", " ", 
       SubscriptBox["q", 
        RowBox[{
         RowBox[{"-", "1"}], "+", "t"}]]}], "+", 
      RowBox[{"c", " ", 
       SubscriptBox["q", 
        RowBox[{"1", "+", "t"}]]}], "-", 
      SubscriptBox["\[CapitalDelta]p", "t"], "+", 
      SubscriptBox["\[CapitalDelta]p", 
       RowBox[{"1", "+", "t"}]]}], ")"}]}], 
   SubsuperscriptBox["\[Sigma]", "u", "2"]]]], "Output",
 CellChangeTimes->{3.39003357988943*^9, 3.39004103811143*^9, 
  3.3900411110214305`*^9},
 CellLabel->"Out[39]="]
}, Open  ]],

Cell["from which we may compute", "Text"],

Cell[TextData[Cell[BoxData[
 RowBox[{
  RowBox[{"Pr", "[", 
   RowBox[{
    SubscriptBox["q", "t"], "=", 
    RowBox[{
     RowBox[{"+", "1"}], "|", "..."}]}], "]"}], "=", 
  FractionBox["Odds", 
   RowBox[{"1", "+", "Odds"}]]}]]]], "NumberedEquation"],

Cell["and make the desired draw.", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 "Examples ",
 "(SAS program ",
 StyleBox["RollGibbs Analyze q.sas",
  FontSlant->"Italic"],
 ")"
}], "Subsection"],

Cell[TextData[{
 "I simulated twenty trades for a price process with ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["\[Sigma]", "u"], "=", "0.01"}]]],
 " and ",
 Cell[BoxData[
  RowBox[{"c", "=", "0.01"}]]],
 ", and then ran the Gibbs sampler for 2,000 sweeps to estimate the trade \
directions. Here is a plot of the transaction prices:"
}], "Text"],

Cell[GraphicsData["Metafile", "\<\
CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@000F9P0@0006`00000000000000?D4003/0`00
000000000037FP00idL00215CDH00040H6810;49000400000000000000000000@0H00;04000U0@00
g000000000000000000008Qh101PF`<02@000100003f1000k@<000X0000@000000000000000D0000
300000d0000U0000300000P0080W00006000004000000000oooo0000000U000030000040000[0000
6000000000000000mP@00>d3000[00006000000000000000mP@00>d3000V00007000008000000000
0@000000000000009@0000`000020000E`0002@0002B0@002P0009<1000=00000P0009813@2C0@X0
E`0002@0002C0@002P0009<1000?00000P0009<12P2C0@l0E`0002@0002B0@003@0009<1000?0000
0P0009<13`2B0@d0E`0002@0002@0@002`000981000=00000P0009813@2@0@/0E`0002@0002=0@00
2P000901000;00000P0009012`2=0@X0E`0002@0002;0@002P0008d1000:00000P0008d12P2;0@X0
E`0002@000280@002P0008/1000;00000P0008/12P280@/0E`0002@000260@002`0008P1000=0000
0P0008P12`260@d0E`0002@000260@003@0008H1000?00000P0008H13@260@l0E`0002@000260@00
3`0008L1000@00000P0008H13`270A00E`0002@000270@00400008P1000A00000P0008L140280A40
E`0002@000280@004@0008X1000B00000P0008P14@2:0A80E`0002@0002:0@004P0008l1000D0000
0P0008X14P2?0A@0E`0002@0002?0@0050000941000E00000P0008l1502A0AD0E`0002@0002A0@00
5@0009<1000G00000P0009415@2C0AL0E`0002@000260@003`0008P1000@00000P0008H13`280A00
E`0002@000280@00400008X1000A00000P0008P1402:0A40E`0002@0002:0@004@0008l1000C0000
0P0008X14@2?0A<0E`0002@0002?0@004`000941000D00000P0008l14`2A0A@0E`0002@0002A0@00
50000981000E00000P000941502B0AD0E`0002@0002B0@005@0009<1000G00000P0009815@2C0AL0
E`0002@0002C0@005`0009<1000K00000P0009<15`2C0A/0E`0002@0002A0@006`0009<1000M0000
0P0009<16`2A0Ad0E`0002@0002>0@007@000941000M00000P0009417@2>0Ad0E`0002@0002<0@00
7@0008h1000M00000P0008h17@2<0Ad0E`0002@000290@007@0008`1000M00000P0008`17@290Ad0
E`0002@000270@006`0008T1000M00000P0008T17@270A/0E`0002@000260@00600008L1000K0000
0P0008L16`260AP0E`0002@000260@00600008H1000M00000P0008H160260Ad0E`0002@000260@00
6`0008L1000M00000P0008H17@270A/0E`0002@0002I0@002P0009X1000;00000P0009X12P2I0@/0
E`0002@0002I0@002`0009X1000<00000P0009T12`2J0@`0E`0002@0002J0@002`0009/1000<0000
0P0009X1302K0@/0E`0002@0002J0@002P0009/1000;00000P0009/12`2J0@X0E`0002@0002J0@00
400009X1000M00000P0009X1402J0Ad0E`0002@0002K0@00400009/1000M00000P0009/1402K0Ad0
E`0002@0002G0@00400009/1000@00000P0009L1402K0A00E`0002@0002G0@007@0009h1000M0000
0P0009L17@2N0Ad0E`0002@0002T0@0040000:@1000M00000P000:@1402T0Ad0E`0002@0002U0@00
40000:D1000M00000P000:D1402U0Ad0E`0002@0002U0@004@000:L1000C00000P000:D14`2W0A40
E`0002@0002W0@0040000:T1000A00000P000:L14@2Y0A00E`0002@0002Y0@0040000:/1000@0000
0P000:T1402[0A00E`0002@0002[0@0040000:h1000A00000P000:/1402^0A40E`0002@0002^0@00
4@000:l1000C00000P000:h14@2_0A<0E`0002@0002_0@004`000:l1000M00000P000:l14`2_0Ad0
E`0002@0002[0@0040000:d1000A00000P000:/1402]0A40E`0002@0002]0@004@000:h1000C0000
0P000:d14@2^0A<0E`0002@0002^0@004`000:h1000M00000P000:h14`2^0Ad0E`0002@0002_0@00
4@000;01000C00000P000:l14`2`0A40E`0002@0002`0@0040000;<1000A00000P000;014@2c0A00
E`0002@0002c0@0040000;D1000@00000P000;<1402e0A00E`0002@0002e0@0040000;P1000A0000
0P000;D1402h0A40E`0002@0002h0@004@000;T1000C00000P000;P14@2i0A<0E`0002@0002i0@00
4`000;T1000M00000P000;T14`2i0Ad0E`0002@0002e0@0040000;L1000A00000P000;D1402g0A40
E`0002@0002g0@004@000;P1000C00000P000;L14@2h0A<0E`0002@0002h0@004`000;P1000M0000
0P000;P14`2h0Ad0E`0002@0002Q0@0040000:D1000@00000P000:41402U0A00E`0002@0002Q0@00
7@000:P1000M00000P000:417@2X0Ad0E`0002@0002[0@007@000;41000M00000P000:/17@2a0Ad0
E`0002@0002e0@007@000;/1000M00000P000;D17@2k0Ad0E`0002@000310@0040000<41000K0000
0P000<4140310A/0E`0002@000310@006`000<81000M00000P000<416`320Ad0E`0002@000320@00
7@000<D1000M00000P000<817@350Ad0E`0002@000350@007@000<H1000M00000P000<D17@360Ad0
E`0002@000360@007@000<T1000M00000P000<H17@390Ad0E`0002@000390@006`000</1000M0000
0P000<T17@3;0A/0E`0002@000320@0040000<81000K00000P000<8140320A/0E`0002@000320@00
6`000<<1000M00000P000<816`330Ad0E`0002@000330@007@000<D1000M00000P000<<17@350Ad0
E`0002@0003;0@0040000</1000M00000P000</1403;0Ad0E`0002@0003<0@0040000<`1000M0000
0P000<`1403<0Ad0E`0002@0002n0@0040000<81000@00000P000;h140320A00E`0002@000380@00
40000<`1000@00000P000<P1403<0A00E`0002@0003;0@007@000<h1000M00000P000</17@3>0Ad0
E`0002@0003E0@002P000=D1000M00000P000=D12P3E0Ad0E`0002@0003F0@002P000=H1000M0000
0P000=H12P3F0Ad0E`0002@0003B0@002P000=H1000:00000P000=812P3F0@X0E`0002@0003B0@00
7@000=T1000M00000P000=817@3I0Ad0E`0002@0003O0@004P000=l1000C00000P000=l14P3O0A<0
E`0002@0003N0@004`000=l1000C00000P000=l14`3N0A<0E`0002@0003N0@004P000=h1000C0000
0P000=h14`3N0A80E`0002@0003N0@004@000=l1000B00000P000=h14P3O0A40E`0002@0003O0@00
40000>41000A00000P000=l14@3Q0A00E`0002@0003Q0@0040000>@1000@00000P000>41403T0A00
E`0002@0003T0@0040000>H1000A00000P000>@1403V0A40E`0002@0003V0@004@000>L1000B0000
0P000>H14@3W0A80E`0002@0003W0@004P000>P1000D00000P000>L14P3X0A@0E`0002@0003X0@00
50000>P1000K00000P000>P1503X0A/0E`0002@0003X0@006`000>T1000M00000P000>P16`3Y0Ad0
E`0002@0003Y0@007@000>X1000M00000P000>T17@3Z0Ad0E`0002@0003W0@004P000>L1000K0000
0P000>L14P3W0A/0E`0002@0003W0@006`000>P1000M00000P000>L16`3X0Ad0E`0002@0003X0@00
7@000>X1000M00000P000>P17@3Z0Ad0E`0002@0003Z0@007@000>/1000M00000P000>X17@3[0Ad0
E`0002@0003V0@0050000>L1000E00000P000>L1503V0AD0E`0002@0003Q0@005@000>H1000F0000
0P000>H15@3Q0AH0E`0002@0003N0@005P000>41000G00000P000>415P3N0AL0E`0002@0003M0@00
5`000=h1000I00000P000=h15`3M0AT0E`0002@0003M0@006@000=d1000K00000P000=d16@3M0A/0
E`0002@0003M0@006`000=h1000M00000P000=d16`3N0Ad0E`0002@0003N0@007@000>41000M0000
0P000=h17@3Q0Ad0E`0002@0003Q0@007@000><1000M00000P000>417@3S0Ad0E`0002@0003S0@00
7@000>D1000M00000P000><17@3U0Ad0E`0002@0003U0@006`000>L1000M00000P000>D17@3W0A/0
E`0002@0003O0@005P000>41000G00000P000>415P3O0AL0E`0002@0003N0@005`000=l1000I0000
0P000=l15`3N0AT0E`0002@0003N0@006@000=h1000K00000P000=h16@3N0A/0E`0002@0003N0@00
6`000=l1000M00000P000=h16`3O0Ad0E`0002@0003O0@007@000>41000M00000P000=l17@3Q0Ad0
E`0002@0003a0@002P000?41000J00000P000?412P3a0AX0E`0002@0003a0@006P000?81000M0000
0P000?416P3b0Ad0E`0002@0003b0@007@000?@1000M00000P000?817@3d0Ad0E`0002@0003d0@00
7@000?D1000M00000P000?@17@3e0Ad0E`0002@0003e0@007@000?L1000M00000P000?D17@3g0Ad0
E`0002@0003g0@006`000?P1000M00000P000?L17@3h0A/0E`0002@0003b0@002P000?81000J0000
0P000?812P3b0AX0E`0002@0003b0@006P000?<1000M00000P000?816P3c0Ad0E`0002@0003c0@00
7@000?@1000M00000P000?<17@3d0Ad0E`0002@0003^0@0040000?D1000@00000P000>h1403e0A00
E`0002@0003m0@005P0000P2000F00000P000?d15P080QH0E`0002@000080P00500000P2000F0000
0P0000P25P080Q@0E`0002@000070P004P0000P2000D00000P0000P250070Q80E`0002@000060P00
4@0000L2000B00000P0000L24P060Q40E`0002@000040P00400000H2000A00000P0000H24@040Q00
E`0002@000020P00400000@2000@00000P0000@240020Q00E`0002@0003o0@0040000082000A0000
0P000082403o0A40E`0002@0003m0@004@000?l1000C00000P000?l14@3m0A<0E`0002@0003l0@00
4`000?d1000F00000P000?d14`3l0AH0E`0002@0003l0@005P000?`1000H00000P000?`15P3l0AP0
E`0002@0003l0@0060000?d1000K00000P000?`1603m0A/0E`0002@0003m0@006`000?l1000M0000
0P000?d16`3o0Ad0E`0002@0003o0@007@000082000M00000P000?l17@020Qd0E`0002@000020P00
7@0000<2000M00000P0000827@030Qd0E`0002@000030P007@0000H2000M00000P0000<27@060Qd0
E`0002@000060P006`0000P2000M00000P0000H27@080Q/0E`0002@000070P004`0000L2000F0000
0P0000L25P070Q<0E`0002@000060P004@0000L2000C00000P0000L24`060Q40E`0002@000000P00
40000082000A00000P00008240000Q40E`0002@0003n0@004@000002000C00000P0000024@3n0A<0
E`0002@0003m0@004`000?h1000F00000P000?h14`3m0AH0E`0002@0003m0@005P000?d1000H0000
0P000?d15P3m0AP0E`0002@0003m0@0060000?h1000K00000P000?d1603n0A/0E`0002@0003n0@00
6`000002000M00000P000?h16`000Qd0E`0002@000000P007@000082000M00000P0000027@020Qd0
E`0002@0000H0P002P0001P2000M00000P0001P22P0H0Qd0E`0002@0000I0P002P0001T2000M0000
0P0001T22P0I0Qd0E`0002@0000F0P004@0001P2000C00000P0001P24`0F0Q40E`0002@0000D0P00
400001H2000A00000P0001H24@0D0Q00E`0002@0000C0P00400001@2000@00000P0001@2400C0Q00
E`0002@0000@0P00400001<2000A00000P0001<2400@0Q40E`0002@0000>0P004@000102000C0000
0P0001024@0>0Q<0E`0002@0000=0P004`0000h2000F00000P0000h24`0=0QH0E`0002@0000=0P00
5P0000d2000H00000P0000d25P0=0QP0E`0002@0000=0P00600000h2000K00000P0000d2600>0Q/0
E`0002@0000>0P006`000102000M00000P0000h26`0@0Qd0E`0002@0000@0P007@0001<2000M0000
0P0001027@0C0Qd0E`0002@0000C0P007@0001@2000M00000P0001<27@0D0Qd0E`0002@0000D0P00
7@0001H2000M00000P0001@27@0F0Qd0E`0002@0000F0P006`0001P2000M00000P0001H27@0H0Q/0
E`0002@0000A0P00400001<2000A00000P0001<2400A0Q40E`0002@0000?0P004@000142000C0000
0P0001424@0?0Q<0E`0002@0000>0P004`0000l2000F00000P0000l24`0>0QH0E`0002@0000>0P00
5P0000h2000H00000P0000h25P0>0QP0E`0002@0000>0P00600000l2000K00000P0000h2600?0Q/0
E`0002@0000?0P006`000142000M00000P0000l26`0A0Qd0E`0002@0000A0P007@0001<2000M0000
0P0001427@0C0Qd0E`0002@0000E0P002P0001T2000:00000P0001D22P0I0PX0E`0002@0000H0P00
7@0001/2000M00000P0001P27@0K0Qd0E`0002@0000/0P00400002`2000T00000P0002`2400/0R@0
E`0002@0000]0P00400002d2000T00000P0002d2400]0R@0E`0002@0000]0P004@0002l2000C0000
0P0002d24`0_0Q40E`0002@0000_0P0040000302000A00000P0002l24@0`0Q00E`0002@0000`0P00
40000382000@00000P000302400b0Q00E`0002@0000b0P00400003D2000A00000P000382400e0Q40
E`0002@0000e0P004@0003L2000C00000P0003D24@0g0Q<0E`0002@0000g0P004`0003P2000F0000
0P0003L24`0h0QH0E`0002@0000h0P005P0003P2000H00000P0003P25P0h0QP0E`0002@0000g0P00
600003P2000K00000P0003P2600g0Q/0E`0002@0000e0P006`0003L2000M00000P0003L26`0e0Qd0
E`0002@0000b0P007@0003D2000M00000P0003D27@0b0Qd0E`0002@0000`0P007@000382000M0000
0P0003827@0`0Qd0E`0002@0000_0P007@000302000M00000P0003027@0_0Qd0E`0002@0000]0P00
6`0002l2000M00000P0002l27@0]0Q/0E`0002@0000b0P00400003@2000A00000P000382400d0Q40
E`0002@0000d0P004@0003H2000C00000P0003@24@0f0Q<0E`0002@0000f0P004`0003L2000F0000
0P0003H24`0g0QH0E`0002@0000g0P005P0003L2000H00000P0003L25P0g0QP0E`0002@0000f0P00
600003L2000K00000P0003L2600f0Q/0E`0002@0000d0P006`0003H2000M00000P0003H26`0d0Qd0
E`0002@0000b0P007@0003@2000M00000P0003@27@0b0Qd0E`0002@0000Y0P00400002d2000@0000
0P0002T2400]0Q00E`0002@0000Y0P0090000302000T00000P0002T2900`0R@0E`0002@0000n0P00
400003h2000M00000P0003h2400n0Qd0E`0002@0000o0P00400003l2000M00000P0003l2400o0Qd0
E`0002@0000o0P004`000402000F00000P0003l25P100Q<0E`0002@000100P004@000482000C0000
0P0004024`120Q40E`0002@000120P00400004<2000A00000P0004824@130Q00E`0002@000130P00
400004H2000@00000P0004<240160Q00E`0002@000160P00400004L2000A00000P0004H240170Q40
E`0002@000170P004@0004L2000B00000P0004L24@170Q80E`0002@000160P004P0004L2000C0000
0P0004L24P160Q<0E`0002@000150P004P0004H2000C00000P0004H24`150Q80E`0002@000150P00
4@0004H2000B00000P0004D24P160Q40E`0002@0000k0P00400003l2000@00000P0003/2400o0Q00
E`0002@0000k0P007@000482000M00000P0003/27@120Qd0E`0002@0001=0P002P0004h2000;0000
0P0004h22P1=0P/0E`0002@0001=0P002`0004h2000<00000P0004d22`1>0P`0E`0002@0001>0P00
2`0004l2000<00000P0004h2301?0P/0E`0002@0001>0P002P0004l2000;00000P0004l22`1>0PX0
E`0002@0001>0P00400004h2000M00000P0004h2401>0Qd0E`0002@0001?0P00400004l2000M0000
0P0004l2401?0Qd0E`0002@0001;0P00400004l2000@00000P0004/2401?0Q00E`0002@0001;0P00
7@000582000M00000P0004/27@1B0Qd0E`0002@0001O0P004`000602000D00000P0006024`1O0Q@0
E`0002@0001O0P0050000602000E00000P0005l2501P0QD0E`0002@0001P0P0050000642000E0000
0P0006025@1Q0Q@0E`0002@0001Q0P004`000642000D00000P000642501Q0Q<0E`0002@0001O0P00
4@000642000C00000P0006424`1O0Q40E`0002@0001M0P00400005l2000A00000P0005l24@1M0Q00
E`0002@0001K0P00400005d2000@00000P0005d2401K0Q00E`0002@0001H0P00400005/2000A0000
0P0005/2401H0Q40E`0002@0001F0P004@0005P2000C00000P0005P24@1F0Q<0E`0002@0001E0P00
4`0005H2000F00000P0005H24`1E0QH0E`0002@0001E0P005P0005D2000H00000P0005D25P1E0QP0
E`0002@0001E0P00600005H2000K00000P0005D2601F0Q/0E`0002@0001F0P006`0005P2000M0000
0P0005H26`1H0Qd0E`0002@0001H0P007@0005/2000M00000P0005P27@1K0Qd0E`0002@0001K0P00
7@0005`2000M00000P0005/27@1L0Qd0E`0002@0001L0P007@0005l2000M00000P0005`27@1O0Qd0
E`0002@0001O0P006`000642000M00000P0005l27@1Q0Q/0E`0002@0001I0P00400005/2000A0000
0P0005/2401I0Q40E`0002@0001G0P004@0005T2000C00000P0005T24@1G0Q<0E`0002@0001F0P00
4`0005L2000F00000P0005L24`1F0QH0E`0002@0001F0P005P0005H2000H00000P0005H25P1F0QP0
E`0002@0001F0P00600005L2000K00000P0005H2601G0Q/0E`0002@0001G0P006`0005T2000M0000
0P0005L26`1I0Qd0E`0002@0001I0P007@0005/2000M00000P0005T27@1K0Qd0E`0002@0001W0P00
5P000782000F00000P0006L25P1b0QH0E`0002@0001b0P0050000782000F00000P0007825P1b0Q@0
E`0002@0001a0P004P000782000D00000P000782501a0Q80E`0002@0001`0P004@000742000B0000
0P0007424P1`0Q40E`0002@0001^0P0040000702000A00000P0007024@1^0Q00E`0002@0001/0P00
400006h2000@00000P0006h2401/0Q00E`0002@0001Y0P00400006`2000A00000P0006`2401Y0Q40
E`0002@0001W0P004@0006T2000C00000P0006T24@1W0Q<0E`0002@0001V0P004`0006L2000F0000
0P0006L24`1V0QH0E`0002@0001V0P005P0006H2000H00000P0006H25P1V0QP0E`0002@0001V0P00
600006L2000K00000P0006H2601W0Q/0E`0002@0001W0P006`0006T2000M00000P0006L26`1Y0Qd0
E`0002@0001Y0P007@0006`2000M00000P0006T27@1/0Qd0E`0002@0001/0P007@0006d2000M0000
0P0006`27@1]0Qd0E`0002@0001]0P007@000702000M00000P0006d27@1`0Qd0E`0002@0001`0P00
6`000782000M00000P0007027@1b0Q/0E`0002@0001a0P004`000742000F00000P0007425P1a0Q<0
E`0002@0001`0P004@000742000C00000P0007424`1`0Q40E`0002@0001Z0P00400006`2000A0000
0P0006`2401Z0Q40E`0002@0001X0P004@0006X2000C00000P0006X24@1X0Q<0E`0002@0001W0P00
4`0006P2000F00000P0006P24`1W0QH0E`0002@0001W0P005P0006L2000H00000P0006L25P1W0QP0
E`0002@0001W0P00600006P2000K00000P0006L2601X0Q/0E`0002@0001X0P006`0006X2000M0000
0P0006P26`1Z0Qd0E`0002@0001Z0P007@0006`2000M00000P0006X27@1/0Qd0E`0002@000230P00
400008<2000T00000P0008<240230R@0E`0002@000240P00400008@2000T00000P0008@240240R@0
E`0002@000240P004@0008H2000C00000P0008@24`260Q40E`0002@000260P00400008L2000A0000
0P0008H24@270Q00E`0002@000270P00400008T2000@00000P0008L240290Q00E`0002@000290P00
400008`2000A00000P0008T2402<0Q40E`0002@0002<0P004@0008h2000C00000P0008`24@2>0Q<0
E`0002@0002>0P004`0008l2000F00000P0008h24`2?0QH0E`0002@0002?0P005P0008l2000H0000
0P0008l25P2?0QP0E`0002@0002>0P00600008l2000K00000P0008l2602>0Q/0E`0002@0002<0P00
6`0008h2000M00000P0008h26`2<0Qd0E`0002@000290P007@0008`2000M00000P0008`27@290Qd0
E`0002@000270P007@0008T2000M00000P0008T27@270Qd0E`0002@000260P007@0008L2000M0000
0P0008L27@260Qd0E`0002@000240P006`0008H2000M00000P0008H27@240Q/0E`0002@000290P00
400008/2000A00000P0008T2402;0Q40E`0002@0002;0P004@0008d2000C00000P0008/24@2=0Q<0
E`0002@0002=0P004`0008h2000F00000P0008d24`2>0QH0E`0002@0002>0P005P0008h2000H0000
0P0008h25P2>0QP0E`0002@0002=0P00600008h2000K00000P0008h2602=0Q/0E`0002@0002;0P00
6`0008d2000M00000P0008d26`2;0Qd0E`0002@000290P007@0008/2000M00000P0008/27@290Qd0
E`0002@000200P00400008@2000@00000P00080240240Q00E`0002@000200P00900008L2000T0000
0P00080290270R@0E`0002@0002F0P004P0009H2000C00000P0009H24P2F0Q<0E`0002@0002E0P00
4`0009H2000C00000P0009H24`2E0Q<0E`0002@0002E0P004P0009D2000C00000P0009D24`2E0Q80
E`0002@0002E0P004@0009H2000B00000P0009D24P2F0Q40E`0002@0002F0P00400009P2000A0000
0P0009H24@2H0Q00E`0002@0002H0P00400009/2000@00000P0009P2402K0Q00E`0002@0002K0P00
400009d2000A00000P0009/2402M0Q40E`0002@0002M0P004@0009h2000B00000P0009d24@2N0Q80
E`0002@0002N0P004P0009l2000D00000P0009h24P2O0Q@0E`0002@0002O0P00500009l2000K0000
0P0009l2502O0Q/0E`0002@0002O0P006`000:02000M00000P0009l26`2P0Qd0E`0002@0002P0P00
7@000:42000M00000P000:027@2Q0Qd0E`0002@0002N0P004P0009h2000K00000P0009h24P2N0Q/0
E`0002@0002N0P006`0009l2000M00000P0009h26`2O0Qd0E`0002@0002O0P007@000:42000M0000
0P0009l27@2Q0Qd0E`0002@0002Q0P007@000:82000M00000P000:427@2R0Qd0E`0002@0002M0P00
500009h2000E00000P0009h2502M0QD0E`0002@0002H0P005@0009d2000F00000P0009d25@2H0QH0
E`0002@0002E0P005P0009P2000G00000P0009P25P2E0QL0E`0002@0002D0P005`0009D2000I0000
0P0009D25`2D0QT0E`0002@0002D0P006@0009@2000K00000P0009@26@2D0Q/0E`0002@0002D0P00
6`0009D2000M00000P0009@26`2E0Qd0E`0002@0002E0P007@0009P2000M00000P0009D27@2H0Qd0
E`0002@0002H0P007@0009X2000M00000P0009P27@2J0Qd0E`0002@0002J0P007@0009`2000M0000
0P0009X27@2L0Qd0E`0002@0002L0P006`0009h2000M00000P0009`27@2N0Q/0E`0002@0002F0P00
5P0009P2000G00000P0009P25P2F0QL0E`0002@0002E0P005`0009H2000I00000P0009H25`2E0QT0
E`0002@0002E0P006@0009D2000K00000P0009D26@2E0Q/0E`0002@0002E0P006`0009H2000M0000
0P0009D26`2F0Qd0E`0002@0002F0P007@0009P2000M00000P0009H27@2H0Qd0E`0002@0002X0P00
2P000:P2000J00000P000:P22P2X0QX0E`0002@0002X0P006P000:T2000M00000P000:P26P2Y0Qd0
E`0002@0002Y0P007@000:/2000M00000P000:T27@2[0Qd0E`0002@0002[0P007@000:`2000M0000
0P000:/27@2/0Qd0E`0002@0002/0P007@000:h2000M00000P000:`27@2^0Qd0E`0002@0002^0P00
6`000:l2000M00000P000:h27@2_0Q/0E`0002@0002Y0P002P000:T2000J00000P000:T22P2Y0QX0
E`0002@0002Y0P006P000:X2000M00000P000:T26P2Z0Qd0E`0002@0002Z0P007@000:/2000M0000
0P000:X27@2[0Qd0E`0002@0002U0P0040000:`2000@00000P000:D2402/0Q00E`0002@0002e0P00
2P000;D2000M00000P000;D22P2e0Qd0E`0002@0002f0P002P000;H2000M00000P000;H22P2f0Qd0
E`0002@0002f0P004@000;P2000C00000P000;H24`2h0Q40E`0002@0002h0P0040000;X2000A0000
0P000;P24@2j0Q00E`0002@0002j0P0040000;`2000@00000P000;X2402l0Q00E`0002@0002l0P00
40000;l2000A00000P000;`2402o0Q40E`0002@0002o0P004@000<02000C00000P000;l24@300Q<0
E`0002@000300P004`000<02000M00000P000<024`300Qd0E`0002@0002l0P0040000;h2000A0000
0P000;`2402n0Q40E`0002@0002n0P004@000;l2000C00000P000;h24@2o0Q<0E`0002@0002o0P00
4`000;l2000M00000P000;l24`2o0Qd0E`0002@0002b0P002P000;H2000:00000P000;822P2f0PX0
E`0002@0002b0P007@000;T2000M00000P000;827@2i0Qd0E`0002@0002l0P007@000<82000M0000
0P000;`27@320Qd0E`0002@0003B0P0040000=H2000M00000P000=82403F0Qd0E`0002@0003C0P00
40000=H2000K00000P000=<2403F0Q/0E`0002@0003F0P0040000=T2000M00000P000=T2403F0Qd0
E`0002@0003I0P0040000=d2000M00000P000=T2403M0Qd0E`0002@0003J0P0040000=d2000K0000
0P000=X2403M0Q/0E`0002@0003M0P0040000>02000M00000P000>02403M0Qd0E`0002@0003?0P00
40000=H2000@00000P000<l2403F0Q00E`0002@0003N0P0040000><2000@00000P000=h2403S0Q00
E`0002@0003W0P002P000>P2000;00000P000>P22P3W0P/0E`0002@0003W0P002`000>P2000<0000
0P000>L22`3X0P`0E`0002@0003X0P002`000>T2000<00000P000>P2303Y0P/0E`0002@0003X0P00
2P000>T2000;00000P000>T22`3X0PX0E`0002@0003X0P0040000>P2000M00000P000>P2403X0Qd0
E`0002@0003Y0P0040000>T2000M00000P000>T2403Y0Qd0E`0002@0003U0P0040000>T2000@0000
0P000>D2403Y0Q00E`0002@0003U0P007@000>`2000M00000P000>D27@3/0Qd0E`0002@0003b0P00
2P000?82000J00000P000?822P3b0QX0E`0002@0003b0P006P000?<2000M00000P000?826P3c0Qd0
E`0002@0003c0P007@000?D2000M00000P000?<27@3e0Qd0E`0002@0003e0P007@000?H2000M0000
0P000?D27@3f0Qd0E`0002@0003f0P007@000?P2000M00000P000?H27@3h0Qd0E`0002@0003h0P00
6`000?T2000M00000P000?P27@3i0Q/0E`0002@0003c0P002P000?<2000J00000P000?<22P3c0QX0
E`0002@0003c0P006P000?@2000M00000P000?<26P3d0Qd0E`0002@0003d0P007@000?D2000M0000
0P000?@27@3e0Qd0E`0002@0003_0P0040000?H2000@00000P000>l2403f0Q00E`0002@0003o0P00
2P000?l2000M00000P000?l22P3o0Qd0E`0002@000000`002P000003000M00000P0000032P000ad0
E`0002@000000`004@000083000C00000P0000034`020a40E`0002@000020`00400000@3000A0000
0P0000834@040a00E`0002@000040`00400000H3000@00000P0000@340060a00E`0002@000060`00
400000T3000A00000P0000H340090a40E`0002@000090`004@0000X3000C00000P0000T34@0:0a<0
E`0002@0000:0`004`0000X3000M00000P0000X34`0:0ad0E`0002@000060`00400000P3000A0000
0P0000H340080a40E`0002@000080`004@0000T3000C00000P0000P34@090a<0E`0002@000090`00
4`0000T3000M00000P0000T34`090ad0E`0002@0003l0P002P000003000:00000P000?`22P000`X0
E`0002@0003l0P007@0000<3000M00000P000?`27@030ad0E`0002@000060`007@0000`3000M0000
0P0000H37@0<0ad0E`0002@0000U0`004`0002H3000D00000P0002H34`0U0a@0E`0002@0000U0`00
500002H3000E00000P0002D3500V0aD0E`0002@0000V0`00500002L3000E00000P0002H35@0W0a@0
E`0002@0000W0`004`0002L3000D00000P0002L3500W0a<0E`0002@0000U0`004@0002L3000C0000
0P0002L34`0U0a40E`0002@0000S0`00400002D3000A00000P0002D34@0S0a00E`0002@0000Q0`00
400002<3000@00000P0002<3400Q0a00E`0002@0000N0`0040000243000A00000P000243400N0a40
E`0002@0000L0`004@0001h3000C00000P0001h34@0L0a<0E`0002@0000K0`004`0001`3000F0000
0P0001`34`0K0aH0E`0002@0000K0`005P0001/3000H00000P0001/35P0K0aP0E`0002@0000K0`00
600001`3000K00000P0001/3600L0a/0E`0002@0000L0`006`0001h3000M00000P0001`36`0N0ad0
E`0002@0000N0`007@000243000M00000P0001h37@0Q0ad0E`0002@0000Q0`007@000283000M0000
0P0002437@0R0ad0E`0002@0000R0`007@0002D3000M00000P0002837@0U0ad0E`0002@0000U0`00
6`0002L3000M00000P0002D37@0W0a/0E`0002@0000O0`0040000243000A00000P000243400O0a40
E`0002@0000M0`004@0001l3000C00000P0001l34@0M0a<0E`0002@0000L0`004`0001d3000F0000
0P0001d34`0L0aH0E`0002@0000L0`005P0001`3000H00000P0001`35P0L0aP0E`0002@0000L0`00
600001d3000K00000P0001`3600M0a/0E`0002@0000M0`006`0001l3000M00000P0001d36`0O0ad0
E`0002@0000O0`007@000243000M00000P0001l37@0Q0ad0E`0002@0000]0`004P0003d3000B0000
0P0002d34P0m0a80E`0002@0000]0`00600003d3000H00000P0002d3600m0aP0E`0002@000140`00
700004D3000M00000P0004D370140ad0E`0002@000140`007@0004D3000M00000P0004@37@150ad0
E`0002@000150`007@0004H3000M00000P0004D37@160ad0E`0002@000150`00700004H3000M0000
0P0004H37@150a`0E`0002@0001?0`002P000583000;00000P0005832P1?0`/0E`0002@0001=0`00
2`0004l3000>00000P0004l32`1=0`h0E`0002@0001<0`003P0004d3000B00000P0004d33P1<0a80
E`0002@0001<0`004P0004`3000E00000P0004`34P1<0aD0E`0002@0001<0`005@0004d3000J0000
0P0004`35@1=0aX0E`0002@0001=0`006P0004l3000M00000P0004d36P1?0ad0E`0002@0001?0`00
7@000583000M00000P0004l37@1B0ad0E`0002@0001B0`007@0005<3000M00000P0005837@1C0ad0
E`0002@0001C0`007@0005H3000M00000P0005<37@1F0ad0E`0002@0001F0`006P0005P3000M0000
0P0005H37@1H0aX0E`0002@0001H0`005@0005T3000J00000P0005P36P1I0aD0E`0002@0001I0`00
4P0005T3000E00000P0005T35@1I0a80E`0002@0001H0`003P0005T3000B00000P0005T34P1H0`h0
E`0002@0001F0`002`0005P3000>00000P0005P33P1F0`/0E`0002@0001C0`002P0005H3000;0000
0P0005H32`1C0`X0E`0002@0001B0`002P0005<3000:00000P0005<32P1B0`X0E`0002@0001@0`00
2P000583000;00000P0005832P1@0`/0E`0002@0001?0`002`000503000<00000P0005032`1?0``0
E`0002@0001>0`00300004l3000>00000P0004l3301>0`h0E`0002@0001=0`003P0004h3000B0000
0P0004h33P1=0a80E`0002@0001=0`004P0004d3000E00000P0004d34P1=0aD0E`0002@0001=0`00
5@0004h3000J00000P0004d35@1>0aX0E`0002@0001>0`006P0004l3000L00000P0004h36P1?0a`0
E`0002@0001?0`0070000503000M00000P0004l3701@0ad0E`0002@0001@0`007@000583000M0000
0P0005037@1B0ad0E`0002@0001C0`007@0005D3000M00000P0005<37@1E0ad0E`0002@0001E0`00
700005H3000M00000P0005D37@1F0a`0E`0002@0001F0`006P0005L3000L00000P0005H3701G0aX0
E`0002@0001G0`005@0005P3000J00000P0005L36P1H0aD0E`0002@0001H0`004P0005P3000E0000
0P0005P35@1H0a80E`0002@0001G0`003P0005P3000B00000P0005P34P1G0`h0E`0002@0001F0`00
300005L3000>00000P0005L33P1F0``0E`0002@0001E0`002`0005H3000<00000P0005H3301E0`/0
E`0002@0001C0`002P0005D3000;00000P0005D32`1C0`X0E`0002@0001Q0`003@0006<3000>0000
0P0006433P1S0`d0E`0002@0001S0`002P0006D3000=00000P0006<33@1U0`X0E`0002@0001U0`00
2P0006D3000M00000P0006D32P1U0ad0E`0002@0001T0`002`0006@3000M00000P0006@32`1T0ad0
E`0002@0001Q0`007@0006T3000M00000P0006437@1Y0ad0E`0002@0002L0000O0<00:40001l0`00
0P000:40O0>L07`3E`0002@0002L0000:`<00:40000[0`000P000:40:`>L02/3E`0002@0002L0000
fP800:40003J0P000P000:40fP:L0=X2E`0002@0002L0000R@800:4000290P000P000:40R@:L08T2
E`0002@0002L0000=`800:40000g0P000P000:40=`:L03L2E`0002@0002L0000iP400:40003V0@00
0P000:40iP6L0>H1E`0002@0002L0000U@400:40002E0@000P000:40U@6L09D1E`0002@0002L0000
A0400:4000140@000P000:40A06L04@1E`0002@0002L0000lP000:40003b00000P000:40lP2L0?80
E`0002@0002L0000X@000:40002Q00000P000:40X@2L0:40E`0002@0002L0000D0000:40001@0000
0P000:40D02L0500E`0002@0000@00002P8002<0000:0P000P0001002P8S00X2E`0002@0000@0000
2@8002<000090P000P0001002@8S00T2E`0002@0000@00000P800100000=0P000P0001003@8@0082
E`0002@0000@0000o`40014000020P000P0001000P8A0?l1E`0002@0000A0000oP400180003o0@00
0P000140o`4B0?h1E`0002@0000B0000o@4001@0003n0@000P000180oP4D0?d1E`0002@0000D0000
o@4001L0003m0@000P0001@0o@4G0?d1E`0002@0000G0000o@4001P0003n0@000P0001L0o@4H0?h1
E`0002@0000H0000oP4001T0003o0@000P0001P0oP4I0?l1E`0002@0000I0000o`4001X000020P00
0P0001T0o`4J0082E`0002@0000J00000P8001X000090P000P0001X00P8J00T2E`0002@0000@0000
0080014000020P000P0001000P8A0002E`0002@0000A0000o`40018000000P000P000140008B0?l1
E`0002@0000B0000oP4001@0003o0@000P000180o`4D0?h1E`0002@0000D0000oP4001L0003n0@00
0P0001@0oP4G0?h1E`0002@0000G0000oP4001P0003o0@000P0001L0oP4H0?l1E`0002@0000H0000
o`4001T000000P000P0001P0o`4I0002E`0002@0000I0000008001X000020P000P0001T0008J0082
E`0002@0000S00001`8002<0000=0P000P0002<03@8S00L2E`0002@0000G0000mP4002<0003f0@00
0P0001L0mP4S0?H1E`0002@0000G0000m@4002<0003e0@000P0001L0m@4S0?D1E`0002@0000I0000
m04001`0003e0@000P0001`0m@4I0?@1E`0002@0000G0000l`4001T0003d0@000P0001T0m04G0?<1
E`0002@0000G0000l@4001L0003c0@000P0001L0l`4G0?41E`0002@0000G0000kP4001L0003a0@00
0P0001L0l@4G0>h1E`0002@0000G0000k@4001L0003^0@000P0001L0kP4G0>d1E`0002@0000G0000
k@4001P0003]0@000P0001L0k@4H0>d1E`0002@0000H0000k@4001T0003^0@000P0001P0k@4I0>h1
E`0002@0000H0000kP4001T0003_0@000P0001T0kP4H0>l1E`0002@0000G0000kP4001P0003_0@00
0P0001P0k`4G0>h1E`0002@0000G0000m@4001L0003i0@000P0001L0n@4G0?D1E`0002@0000S0000
l`4002<0003i0@000P0002<0n@4S0?<1E`0002@0000@0000iP400140003W0@000P000100iP4A0>L1
E`0002@0000A0000iP400180003W0@000P000140i`4B0>H1E`0002@0000A0000i@400180003V0@00
0P000180iP4A0>D1E`0002@0000@0000i@400140003V0@000P000140i@4@0>H1E`0002@0000G0000
iP4002<0003V0@000P0001L0iP4S0>H1E`0002@0000G0000i@4002<0003U0@000P0001L0i@4S0>D1
E`0002@0000G0000i@4001L0003Y0@000P0001L0j@4G0>D1E`0002@0000S0000h`4002<0003Y0@00
0P0002<0j@4S0><1E`0002@0000I0000dP4001X0003C0@000P0001T0dP4J0=<1E`0002@0000J0000
dP4001/0003C0@000P0001X0d`4K0=81E`0002@0000J0000d@4001/0003B0@000P0001/0dP4J0=41
E`0002@0000I0000d@4001X0003A0@000P0001X0d@4I0=41E`0002@0000G0000d@4001T0003C0@00
0P0001T0d@4G0=<1E`0002@0000G0000d`4001L0003E0@000P0001L0d`4G0=D1E`0002@0000G0000
e@4001L0003H0@000P0001L0e@4G0=P1E`0002@0000G0000f04001L0003J0@000P0001L0f04G0=X1
E`0002@0000G0000fP4001T0003L0@000P0001L0fP4I0=`1E`0002@0000I0000g04001`0003M0@00
0P0001T0g04L0=d1E`0002@0000L0000g@4001h0003M0@000P0001`0g@4N0=d1E`0002@0000N0000
g0400200003M0@000P0001h0g@4P0=`1E`0002@0000P0000fP400280003L0@000P000200g04R0=X1
E`0002@0000R0000f04002<0003J0@000P000280fP4S0=P1E`0002@0000S0000eP4002<0003H0@00
0P0002<0f04S0=H1E`0002@0000R0000d`4002<0003F0@000P0002<0eP4R0=<1E`0002@0000P0000
d@400280003C0@000P000280d`4P0=41E`0002@0000G0000f04001L0003I0@000P0001L0f04G0=T1
E`0002@0000G0000f@4001T0003K0@000P0001L0f@4I0=/1E`0002@0000I0000f`4001`0003L0@00
0P0001T0f`4L0=`1E`0002@0000L0000g04001h0003L0@000P0001`0g04N0=`1E`0002@0000N0000
f`400200003L0@000P0001h0g04P0=/1E`0002@0000P0000f@400280003K0@000P000200f`4R0=T1
E`0002@0000R0000f04002<0003I0@000P000280f@4S0=P1E`0002@0000L0000`04001`0003;0@00
0P0001`0b`4L0<01E`0002@0000J0000`04001`000300@000P0001`0`04J0<01E`0002@0000H0000
`04001X000310@000P0001X0`04H0<41E`0002@0000G0000`@4001P000320@000P0001P0`@4G0<81
E`0002@0000G0000`P4001L000340@000P0001L0`P4G0<@1E`0002@0000G0000a04001L000370@00
0P0001L0a04G0<L1E`0002@0000G0000a`4001L000390@000P0001L0a`4G0<T1E`0002@0000G0000
b@4001T0003;0@000P0001L0b@4I0</1E`0002@0000I0000b`4001`0003<0@000P0001T0b`4L0<`1
E`0002@0000L0000c04001h0003<0@000P0001`0c04N0<`1E`0002@0000N0000b`400200003<0@00
0P0001h0c04P0</1E`0002@0000P0000b@400280003;0@000P000200b`4R0<T1E`0002@0000R0000
a`4002<000390@000P000280b@4S0<L1E`0002@0000S0000a@4002<000370@000P0002<0a`4S0<D1
E`0002@0000R0000`P4002<000350@000P0002<0a@4R0<81E`0002@0000P0000`040028000320@00
0P000280`P4P0<01E`0002@0000I0000`@4001`000310@000P0001`0`@4I0<41E`0002@0000G0000
`@4001T000320@000P0001T0`@4G0<81E`0002@0000G0000a`4001L000380@000P0001L0a`4G0<P1
E`0002@0000G0000b04001T0003:0@000P0001L0b04I0<X1E`0002@0000I0000bP4001`0003;0@00
0P0001T0bP4L0</1E`0002@0000L0000b`4001h0003;0@000P0001`0b`4N0</1E`0002@0000N0000
bP400200003;0@000P0001h0b`4P0<X1E`0002@0000P0000b0400280003:0@000P000200bP4R0<P1
E`0002@0000R0000a`4002<000380@000P000280b04S0<L1E`0002@000140000OP<005@0001n0`00
0P0004@0OP=D07h3E`0002@0001N0000L`<00640001d0`000P000640L`=N07@3E`0002@0001L0000
M0<005h0001g0`000P0005h0M0=L07L3E`0002@0001K0000M`<005`0001k0`000P0005`0M`=K07/3
E`0002@0001K0000N`<005/0001n0`000P0005/0N`=K07h3E`0002@0001K0000OP<005`000230`00
0P0005/0OP=L08<3E`0002@0001L0000P`<005h000260`000P0005`0P`=N08H3E`0002@0001N0000
QP<0064000260`000P0005h0QP=Q08H3E`0002@0001Q0000QP<0068000260`000P000640QP=R08H3
E`0002@0001R0000QP<006D000260`000P000680QP=U08H3E`0002@0001U0000P`<006L000260`00
0P0006D0QP=W08<3E`0002@0001W0000OP<006P000230`000P0006L0P`=X07h3E`0002@0001X0000
N`<006P0001n0`000P0006P0OP=X07/3E`0002@0001W0000M`<006P0001k0`000P0006P0N`=W07L3
E`0002@0001U0000M0<006L0001g0`000P0006L0M`=U07@3E`0002@0001R0000L`<006D0001d0`00
0P0006D0M0=R07<3E`0002@0001Q0000L`<00680001c0`000P000680L`=Q07<3E`0002@0001O0000
L`<00640001d0`000P000640L`=O07@3E`0002@0001N0000M0<005l0001e0`000P0005l0M0=N07D3
E`0002@0001M0000M@<005h0001g0`000P0005h0M@=M07L3E`0002@0001L0000M`<005d0001k0`00
0P0005d0M`=L07/3E`0002@0001L0000N`<005`0001n0`000P0005`0N`=L07h3E`0002@0001L0000
OP<005d000230`000P0005`0OP=M08<3E`0002@0001M0000P`<005h000250`000P0005d0P`=N08D3
E`0002@0001N0000Q@<005l000260`000P0005h0Q@=O08H3E`0002@0001O0000QP<0064000260`00
0P0005l0QP=Q08H3E`0002@0001R0000QP<006@000260`000P000680QP=T08H3E`0002@0001T0000
Q@<006D000260`000P0006@0QP=U08D3E`0002@0001U0000P`<006H000250`000P0006D0Q@=V08<3
E`0002@0001V0000OP<006L000230`000P0006H0P`=W07h3E`0002@0001W0000N`<006L0001n0`00
0P0006L0OP=W07/3E`0002@0001V0000M`<006L0001k0`000P0006L0N`=V07L3E`0002@0001U0000
M@<006H0001g0`000P0006H0M`=U07D3E`0002@0001T0000M0<006D0001e0`000P0006D0M@=T07@3
E`0002@0001R0000L`<006@0001d0`000P0006@0M0=R07<3E`0002@0001]0000Q@<006h000260`00
0P0006h0Q@=]08H3E`0002@0001]0000QP<006h000260`000P0006d0QP=^08H3E`0002@0001^0000
QP<006l000260`000P0006h0QP=_08H3E`0002@0001^0000Q@<006l000260`000P0006l0QP=^08D3
E`0002@0001h0000L`<007/0001d0`000P0007/0L`=h07@3E`0002@0001f0000M0<007P0001g0`00
0P0007P0M0=f07L3E`0002@0001e0000M`<007H0001k0`000P0007H0M`=e07/3E`0002@0001e0000
N`<007D0001n0`000P0007D0N`=e07h3E`0002@0001e0000OP<007H000230`000P0007D0OP=f08<3
E`0002@0001f0000P`<007P000260`000P0007H0P`=h08H3E`0002@0001h0000QP<007/000260`00
0P0007P0QP=k08H3E`0002@0001k0000QP<007`000260`000P0007/0QP=l08H3E`0002@0001l0000
QP<007l000260`000P0007`0QP=o08H3E`0002@0001o0000P`<0084000260`000P0007l0QP>108<3
E`0002@000210000OP<0088000230`000P000840P`>207h3E`0002@000220000N`<00880001n0`00
0P000880OP>207/3E`0002@000210000M`<00880001k0`000P000880N`>107L3E`0002@0001o0000
M0<00840001g0`000P000840M`=o07@3E`0002@0001l0000L`<007l0001d0`000P0007l0M0=l07<3
E`0002@0001k0000L`<007`0001c0`000P0007`0L`=k07<3E`0002@0001i0000L`<007/0001d0`00
0P0007/0L`=i07@3E`0002@0001h0000M0<007T0001e0`000P0007T0M0=h07D3E`0002@0001g0000
M@<007P0001g0`000P0007P0M@=g07L3E`0002@0001f0000M`<007L0001k0`000P0007L0M`=f07/3
E`0002@0001f0000N`<007H0001n0`000P0007H0N`=f07h3E`0002@0001f0000OP<007L000230`00
0P0007H0OP=g08<3E`0002@0001g0000P`<007P000250`000P0007L0P`=h08D3E`0002@0001h0000
Q@<007T000260`000P0007P0Q@=i08H3E`0002@0001i0000QP<007/000260`000P0007T0QP=k08H3
E`0002@0001l0000QP<007h000260`000P0007`0QP=n08H3E`0002@0001n0000Q@<007l000260`00
0P0007h0QP=o08D3E`0002@0001o0000P`<0080000250`000P0007l0Q@>008<3E`0002@000200000
OP<0084000230`000P000800P`>107h3E`0002@000210000N`<00840001n0`000P000840OP>107/3
E`0002@000200000M`<00840001k0`000P000840N`>007L3E`0002@0001o0000M@<00800001g0`00
0P000800M`=o07D3E`0002@0001n0000M0<007l0001e0`000P0007l0M@=n07@3E`0002@0001l0000
L`<007h0001d0`000P0007h0M0=l07<3E`0002@000290000L`<008`0001d0`000P0008`0L`>907@3
E`0002@000280000M0<008T0001f0`000P0008T0M0>807H3E`0002@000280000MP<008P0001h0`00
0P0008P0MP>807P3E`0002@000280000N0<008T0001j0`000P0008P0N0>907X3E`0002@000290000
NP<008`0001k0`000P0008T0NP><07/3E`0002@0002<0000N`<008l0001k0`000P0008`0N`>?07/3
E`0002@0002?0000NP<00980001k0`000P0008l0N`>B07X3E`0002@0002B0000N0<009<0001j0`00
0P000980NP>C07P3E`0002@0002C0000MP<009<0001h0`000P0009<0N0>C07H3E`0002@0002B0000
M0<009<0001f0`000P0009<0MP>B07@3E`0002@0002?0000L`<00980001d0`000P000980M0>?07<3
E`0002@0002<0000L`<008l0001c0`000P0008l0L`><07<3E`0002@0002:0000L`<008`0001d0`00
0P0008`0L`>:07@3E`0002@000290000M0<008X0001f0`000P0008X0M0>907H3E`0002@000290000
MP<008T0001h0`000P0008T0MP>907P3E`0002@000290000N0<008X0001j0`000P0008T0N0>:07X3
E`0002@0002:0000NP<008`0001k0`000P0008X0NP><07/3E`0002@0002?0000NP<00940001k0`00
0P0008l0N`>A07X3E`0002@0002A0000N0<00980001j0`000P000940NP>B07P3E`0002@0002B0000
MP<00980001h0`000P000980N0>B07H3E`0002@0002A0000M0<00980001f0`000P000980MP>A07@3
E`0002@0002?0000L`<00940001d0`000P000940M0>?07<3E`0002@000290000N`<008`0001l0`00
0P0008`0N`>907`3E`0002@000280000O0<008T0001m0`000P0008T0O0>807d3E`0002@000270000
O@<008P0001o0`000P0008P0O@>707l3E`0002@000270000O`<008L000230`000P0008L0O`>708<3
E`0002@000270000P`<008P000250`000P0008L0P`>808D3E`0002@000280000Q@<008T000260`00
0P0008P0Q@>908H3E`0002@000290000QP<008`000260`000P0008T0QP><08H3E`0002@0002<0000
QP<008l000260`000P0008`0QP>?08H3E`0002@0002?0000QP<0098000260`000P0008l0QP>B08H3
E`0002@0002B0000Q@<009<000260`000P000980QP>C08D3E`0002@0002C0000P`<009@000250`00
0P0009<0Q@>D08<3E`0002@0002D0000O`<009@000230`000P0009@0P`>D07l3E`0002@0002C0000
O@<009@0001o0`000P0009@0O`>C07d3E`0002@0002B0000O0<009<0001m0`000P0009<0O@>B07`3
E`0002@0002?0000N`<00980001l0`000P000980O0>?07/3E`0002@0002:0000N`<008`0001l0`00
0P0008`0N`>:07`3E`0002@000290000O0<008X0001m0`000P0008X0O0>907d3E`0002@000280000
O@<008T0001o0`000P0008T0O@>807l3E`0002@000280000O`<008P000230`000P0008P0O`>808<3
E`0002@000280000P`<008T000250`000P0008P0P`>908D3E`0002@000290000Q@<008X000260`00
0P0008T0Q@>:08H3E`0002@0002:0000QP<008`000260`000P0008X0QP><08H3E`0002@0002?0000
QP<0094000260`000P0008l0QP>A08H3E`0002@0002A0000Q@<0098000260`000P000940QP>B08D3
E`0002@0002B0000P`<009<000250`000P000980Q@>C08<3E`0002@0002C0000O`<009<000230`00
0P0009<0P`>C07l3E`0002@0002B0000O@<009<0001o0`000P0009<0O`>B07d3E`0002@0002A0000
O0<00980001m0`000P000980O@>A07`3E`0002@0002?0000N`<00940001l0`000P000940O0>?07/3
E`0002@000140000;@<005@0000]0`000P0004@0;@=D02d3E`0002@0001N00008P<00640000S0`00
0P0006408P=N02<3E`0002@0001L00008`<005h0000V0`000P0005h08`=L02H3E`0002@0001K0000
9P<005`0000Z0`000P0005`09P=K02X3E`0002@0001K0000:P<005/0000]0`000P0005/0:P=K02d3
E`0002@0001K0000;@<005`0000b0`000P0005/0;@=L0383E`0002@0001L0000<P<005h0000e0`00
0P0005`0<P=N03D3E`0002@0001N0000=@<00640000e0`000P0005h0=@=Q03D3E`0002@0001Q0000
=@<00680000e0`000P000640=@=R03D3E`0002@0001R0000=@<006D0000e0`000P000680=@=U03D3
E`0002@0001U0000<P<006L0000e0`000P0006D0=@=W0383E`0002@0001W0000;@<006P0000b0`00
0P0006L0<P=X02d3E`0002@0001X0000:P<006P0000]0`000P0006P0;@=X02X3E`0002@0001W0000
9P<006P0000Z0`000P0006P0:P=W02H3E`0002@0001U00008`<006L0000V0`000P0006L09P=U02<3
E`0002@0001R00008P<006D0000S0`000P0006D08`=R0283E`0002@0001Q00008P<00680000R0`00
0P0006808P=Q0283E`0002@0001O00008P<00640000S0`000P0006408P=O02<3E`0002@0001N0000
8`<005l0000T0`000P0005l08`=N02@3E`0002@0001M000090<005h0000V0`000P0005h090=M02H3
E`0002@0001L00009P<005d0000Z0`000P0005d09P=L02X3E`0002@0001L0000:P<005`0000]0`00
0P0005`0:P=L02d3E`0002@0001L0000;@<005d0000b0`000P0005`0;@=M0383E`0002@0001M0000
<P<005h0000d0`000P0005d0<P=N03@3E`0002@0001N0000=0<005l0000e0`000P0005h0=0=O03D3
E`0002@0001O0000=@<00640000e0`000P0005l0=@=Q03D3E`0002@0001R0000=@<006@0000e0`00
0P000680=@=T03D3E`0002@0001T0000=0<006D0000e0`000P0006@0=@=U03@3E`0002@0001U0000
<P<006H0000d0`000P0006D0=0=V0383E`0002@0001V0000;@<006L0000b0`000P0006H0<P=W02d3
E`0002@0001W0000:P<006L0000]0`000P0006L0;@=W02X3E`0002@0001V00009P<006L0000Z0`00
0P0006L0:P=V02H3E`0002@0001U000090<006H0000V0`000P0006H09P=U02@3E`0002@0001T0000
8`<006D0000T0`000P0006D090=T02<3E`0002@0001R00008P<006@0000S0`000P0006@08`=R0283
E`0002@0001]0000=0<006h0000e0`000P0006h0=0=]03D3E`0002@0001]0000=@<006h0000e0`00
0P0006d0=@=^03D3E`0002@0001^0000=@<006l0000e0`000P0006h0=@=_03D3E`0002@0001^0000
=0<006l0000e0`000P0006l0=@=^03@3E`0002@0001h00008P<007/0000S0`000P0007/08P=h02<3
E`0002@0001f00008`<007P0000V0`000P0007P08`=f02H3E`0002@0001e00009P<007H0000Z0`00
0P0007H09P=e02X3E`0002@0001e0000:P<007D0000]0`000P0007D0:P=e02d3E`0002@0001e0000
;@<007H0000b0`000P0007D0;@=f0383E`0002@0001f0000<P<007P0000e0`000P0007H0<P=h03D3
E`0002@0001h0000=@<007/0000e0`000P0007P0=@=k03D3E`0002@0001k0000=@<007`0000e0`00
0P0007/0=@=l03D3E`0002@0001l0000=@<007l0000e0`000P0007`0=@=o03D3E`0002@0001o0000
<P<00840000e0`000P0007l0=@>10383E`0002@000210000;@<00880000b0`000P000840<P>202d3
E`0002@000220000:P<00880000]0`000P000880;@>202X3E`0002@0002100009P<00880000Z0`00
0P000880:P>102H3E`0002@0001o00008`<00840000V0`000P0008409P=o02<3E`0002@0001l0000
8P<007l0000S0`000P0007l08`=l0283E`0002@0001k00008P<007`0000R0`000P0007`08P=k0283
E`0002@0001i00008P<007/0000S0`000P0007/08P=i02<3E`0002@0001h00008`<007T0000T0`00
0P0007T08`=h02@3E`0002@0001g000090<007P0000V0`000P0007P090=g02H3E`0002@0001f0000
9P<007L0000Z0`000P0007L09P=f02X3E`0002@0001f0000:P<007H0000]0`000P0007H0:P=f02d3
E`0002@0001f0000;@<007L0000b0`000P0007H0;@=g0383E`0002@0001g0000<P<007P0000d0`00
0P0007L0<P=h03@3E`0002@0001h0000=0<007T0000e0`000P0007P0=0=i03D3E`0002@0001i0000
=@<007/0000e0`000P0007T0=@=k03D3E`0002@0001l0000=@<007h0000e0`000P0007`0=@=n03D3
E`0002@0001n0000=0<007l0000e0`000P0007h0=@=o03@3E`0002@0001o0000<P<00800000d0`00
0P0007l0=0>00383E`0002@000200000;@<00840000b0`000P000800<P>102d3E`0002@000210000
:P<00840000]0`000P000840;@>102X3E`0002@0002000009P<00840000Z0`000P000840:P>002H3
E`0002@0001o000090<00800000V0`000P0008009P=o02@3E`0002@0001n00008`<007l0000T0`00
0P0007l090=n02<3E`0002@0001l00008P<007h0000S0`000P0007h08`=l0283E`0002@000270000
8P<008L0000W0`000P0008L08P>702L3E`0002@00027000090<008P0000V0`000P0008L09P>802@3
E`0002@0002800008P<008X0000T0`000P0008P090>:0283E`0002@0002:00008P<008`0000R0`00
0P0008X08P><0283E`0002@0002<00008P<00900000U0`000P0008`08P>@02D3E`0002@0002@0000
9@<00980000U0`000P0009009@>B02D3E`0002@0002B000090<009<0000U0`000P0009809@>C02@3
E`0002@0002C00008P<009@0000T0`000P0009<090>D0283E`0002@0002800008`<008X0000T0`00
0P0008P090>:02<3E`0002@0002:00008`<008`0000S0`000P0008X08`><02<3E`0002@0002<0000
8`<00900000U0`000P0008`08`>@02D3E`0002@0002D00008P<009@0000U0`000P0009@08P>D02D3
E`0002@0002C00009@<009@0000W0`000P0009@09@>C02L3E`0002@0002?00009`<009<0000/0`00
0P0009<09`>?02`3E`0002@0002>0000;0<008l0000^0`000P0008l0;0>>02h3E`0002@0002=0000
;P<008h0000a0`000P0008h0;P>=0343E`0002@0002=0000<@<008d0000e0`000P0008d0<@>=03D3
E`0002@0002>00009`<009<0000/0`000P0009<09`>>02`3E`0002@0002=0000;0<008h0000^0`00
0P0008h0;0>=02h3E`0002@0002=0000;P<008d0000a0`000P0008d0;P>=0343E`0002@0002=0000
<@<008d0000e0`000P0008d0<@>=03D3E`0002@000140000f`8005@0003K0P000P0004@0f`9D0=/2
E`0002@0001N0000d0800640003A0P000P000640d09N0=42E`0002@0001L0000d@8005h0003D0P00
0P0005h0d@9L0=@2E`0002@0001K0000e08005`0003H0P000P0005`0e09K0=P2E`0002@0001K0000
f08005/0003K0P000P0005/0f09K0=/2E`0002@0001K0000f`8005`0003P0P000P0005/0f`9L0>02
E`0002@0001L0000h08005h0003S0P000P0005`0h09N0><2E`0002@0001N0000h`800640003S0P00
0P0005h0h`9Q0><2E`0002@0001Q0000h`800680003S0P000P000640h`9R0><2E`0002@0001R0000
h`8006D0003S0P000P000680h`9U0><2E`0002@0001U0000h08006L0003S0P000P0006D0h`9W0>02
E`0002@0001W0000f`8006P0003P0P000P0006L0h09X0=/2E`0002@0001X0000f08006P0003K0P00
0P0006P0f`9X0=P2E`0002@0001W0000e08006P0003H0P000P0006P0f09W0=@2E`0002@0001U0000
d@8006L0003D0P000P0006L0e09U0=42E`0002@0001R0000d08006D0003A0P000P0006D0d@9R0=02
E`0002@0001Q0000d0800680003@0P000P000680d09Q0=02E`0002@0001O0000d0800640003A0P00
0P000640d09O0=42E`0002@0001N0000d@8005l0003B0P000P0005l0d@9N0=82E`0002@0001M0000
dP8005h0003D0P000P0005h0dP9M0=@2E`0002@0001L0000e08005d0003H0P000P0005d0e09L0=P2
E`0002@0001L0000f08005`0003K0P000P0005`0f09L0=/2E`0002@0001L0000f`8005d0003P0P00
0P0005`0f`9M0>02E`0002@0001M0000h08005h0003R0P000P0005d0h09N0>82E`0002@0001N0000
hP8005l0003S0P000P0005h0hP9O0><2E`0002@0001O0000h`800640003S0P000P0005l0h`9Q0><2
E`0002@0001R0000h`8006@0003S0P000P000680h`9T0><2E`0002@0001T0000hP8006D0003S0P00
0P0006@0h`9U0>82E`0002@0001U0000h08006H0003R0P000P0006D0hP9V0>02E`0002@0001V0000
f`8006L0003P0P000P0006H0h09W0=/2E`0002@0001W0000f08006L0003K0P000P0006L0f`9W0=P2
E`0002@0001V0000e08006L0003H0P000P0006L0f09V0=@2E`0002@0001U0000dP8006H0003D0P00
0P0006H0e09U0=82E`0002@0001T0000d@8006D0003B0P000P0006D0dP9T0=42E`0002@0001R0000
d08006@0003A0P000P0006@0d@9R0=02E`0002@0001]0000hP8006h0003S0P000P0006h0hP9]0><2
E`0002@0001]0000h`8006h0003S0P000P0006d0h`9^0><2E`0002@0001^0000h`8006l0003S0P00
0P0006h0h`9_0><2E`0002@0001^0000hP8006l0003S0P000P0006l0h`9^0>82E`0002@0001h0000
d08007/0003A0P000P0007/0d09h0=42E`0002@0001f0000d@8007P0003D0P000P0007P0d@9f0=@2
E`0002@0001e0000e08007H0003H0P000P0007H0e09e0=P2E`0002@0001e0000f08007D0003K0P00
0P0007D0f09e0=/2E`0002@0001e0000f`8007H0003P0P000P0007D0f`9f0>02E`0002@0001f0000
h08007P0003S0P000P0007H0h09h0><2E`0002@0001h0000h`8007/0003S0P000P0007P0h`9k0><2
E`0002@0001k0000h`8007`0003S0P000P0007/0h`9l0><2E`0002@0001l0000h`8007l0003S0P00
0P0007`0h`9o0><2E`0002@0001o0000h0800840003S0P000P0007l0h`:10>02E`0002@000210000
f`800880003P0P000P000840h0:20=/2E`0002@000220000f0800880003K0P000P000880f`:20=P2
E`0002@000210000e0800880003H0P000P000880f0:10=@2E`0002@0001o0000d@800840003D0P00
0P000840e09o0=42E`0002@0001l0000d08007l0003A0P000P0007l0d@9l0=02E`0002@0001k0000
d08007`0003@0P000P0007`0d09k0=02E`0002@0001i0000d08007/0003A0P000P0007/0d09i0=42
E`0002@0001h0000d@8007T0003B0P000P0007T0d@9h0=82E`0002@0001g0000dP8007P0003D0P00
0P0007P0dP9g0=@2E`0002@0001f0000e08007L0003H0P000P0007L0e09f0=P2E`0002@0001f0000
f08007H0003K0P000P0007H0f09f0=/2E`0002@0001f0000f`8007L0003P0P000P0007H0f`9g0>02
E`0002@0001g0000h08007P0003R0P000P0007L0h09h0>82E`0002@0001h0000hP8007T0003S0P00
0P0007P0hP9i0><2E`0002@0001i0000h`8007/0003S0P000P0007T0h`9k0><2E`0002@0001l0000
h`8007h0003S0P000P0007`0h`9n0><2E`0002@0001n0000hP8007l0003S0P000P0007h0h`9o0>82
E`0002@0001o0000h0800800003R0P000P0007l0hP:00>02E`0002@000200000f`800840003P0P00
0P000800h0:10=/2E`0002@000210000f0800840003K0P000P000840f`:10=P2E`0002@000200000
e0800840003H0P000P000840f0:00=@2E`0002@0001o0000dP800800003D0P000P000800e09o0=82
E`0002@0001n0000d@8007l0003B0P000P0007l0dP9n0=42E`0002@0001l0000d08007h0003A0P00
0P0007h0d@9l0=02E`0002@0002A0000d`800980003D0P000P000980d`:A0=@2E`0002@0002A0000
e0800980003E0P000P000940e0:B0=D2E`0002@0002B0000e08009<0003E0P000P000980e@:C0=@2
E`0002@0002C0000d`8009<0003D0P000P0009<0e0:C0=<2E`0002@0002B0000d@8009<0003C0P00
0P0009<0d`:B0=42E`0002@0002@0000d0800980003A0P000P000980d@:@0=02E`0002@0002=0000
d0800900003@0P000P000900d0:=0=02E`0002@0002;0000d08008d0003A0P000P0008d0d0:;0=42
E`0002@000290000d@8008/0003C0P000P0008/0d@:90=<2E`0002@000280000d`8008T0003E0P00
0P0008T0d`:80=D2E`0002@000270000e@8008P0003H0P000P0008P0e@:70=P2E`0002@000270000
f08008L0003N0P000P0008L0f0:70=h2E`0002@000270000gP8008P0003Q0P000P0008L0gP:80>42
E`0002@000280000h@8008X0003S0P000P0008P0h@::0><2E`0002@0002:0000h`8008d0003S0P00
0P0008X0h`:=0><2E`0002@0002=0000h`8008h0003S0P000P0008d0h`:>0><2E`0002@0002>0000
h`800940003S0P000P0008h0h`:A0><2E`0002@0002A0000h@8009<0003S0P000P000940h`:C0>42
E`0002@0002C0000gP8009@0003Q0P000P0009<0h@:D0=h2E`0002@0002D0000g@8009@0003N0P00
0P0009@0gP:D0=d2E`0002@0002C0000fP8009@0003M0P000P0009@0g@:C0=X2E`0002@0002A0000
f08009<0003J0P000P0009<0fP:A0=P2E`0002@0002>0000e`800940003H0P000P000940f0:>0=L2
E`0002@0002=0000e`8008h0003G0P000P0008h0e`:=0=L2E`0002@0002;0000e`8008d0003H0P00
0P0008d0e`:;0=P2E`0002@000290000f08008/0003J0P000P0008/0f0:90=X2E`0002@000280000
fP8008T0003M0P000P0008T0fP:80=d2E`0002@0002<0000d08008d0003A0P000P0008d0d0:<0=42
E`0002@0002:0000d@8008`0003C0P000P0008`0d@::0=<2E`0002@000290000d`8008X0003E0P00
0P0008X0d`:90=D2E`0002@000280000e@8008T0003H0P000P0008T0e@:80=P2E`0002@000280000
f08008P0003N0P000P0008P0f0:80=h2E`0002@000280000gP8008T0003Q0P000P0008P0gP:90>42
E`0002@000290000h@8008/0003S0P000P0008T0h@:;0><2E`0002@0002;0000h`8008d0003S0P00
0P0008/0h`:=0><2E`0002@0002>0000h`800900003S0P000P0008h0h`:@0><2E`0002@0002@0000
h@800980003S0P000P000900h`:B0>42E`0002@0002B0000gP8009<0003Q0P000P000980h@:C0=h2
E`0002@0002C0000g@8009<0003N0P000P0009<0gP:C0=d2E`0002@0002B0000fP8009<0003M0P00
0P0009<0g@:B0=X2E`0002@0002@0000f0800980003J0P000P000980fP:@0=P2E`0002@0002>0000
e`800900003H0P000P000900f0:>0=L2E`0002@000140000RP8005@0002:0P000P0004@0RP9D08X2
E`0002@0001N0000O`80064000200P000P000640O`9N0802E`0002@0001L0000P08005h000230P00
0P0005h0P09L08<2E`0002@0001K0000P`8005`000270P000P0005`0P`9K08L2E`0002@0001K0000
Q`8005/0002:0P000P0005/0Q`9K08X2E`0002@0001K0000RP8005`0002?0P000P0005/0RP9L08l2
E`0002@0001L0000S`8005h0002B0P000P0005`0S`9N0982E`0002@0001N0000TP800640002B0P00
0P0005h0TP9Q0982E`0002@0001Q0000TP800680002B0P000P000640TP9R0982E`0002@0001R0000
TP8006D0002B0P000P000680TP9U0982E`0002@0001U0000S`8006L0002B0P000P0006D0TP9W08l2
E`0002@0001W0000RP8006P0002?0P000P0006L0S`9X08X2E`0002@0001X0000Q`8006P0002:0P00
0P0006P0RP9X08L2E`0002@0001W0000P`8006P000270P000P0006P0Q`9W08<2E`0002@0001U0000
P08006L000230P000P0006L0P`9U0802E`0002@0001R0000O`8006D000200P000P0006D0P09R07l2
E`0002@0001Q0000O`800680001o0P000P000680O`9Q07l2E`0002@0001O0000O`80064000200P00
0P000640O`9O0802E`0002@0001N0000P08005l000210P000P0005l0P09N0842E`0002@0001M0000
P@8005h000230P000P0005h0P@9M08<2E`0002@0001L0000P`8005d000270P000P0005d0P`9L08L2
E`0002@0001L0000Q`8005`0002:0P000P0005`0Q`9L08X2E`0002@0001L0000RP8005d0002?0P00
0P0005`0RP9M08l2E`0002@0001M0000S`8005h0002A0P000P0005d0S`9N0942E`0002@0001N0000
T@8005l0002B0P000P0005h0T@9O0982E`0002@0001O0000TP800640002B0P000P0005l0TP9Q0982
E`0002@0001R0000TP8006@0002B0P000P000680TP9T0982E`0002@0001T0000T@8006D0002B0P00
0P0006@0TP9U0942E`0002@0001U0000S`8006H0002A0P000P0006D0T@9V08l2E`0002@0001V0000
RP8006L0002?0P000P0006H0S`9W08X2E`0002@0001W0000Q`8006L0002:0P000P0006L0RP9W08L2
E`0002@0001V0000P`8006L000270P000P0006L0Q`9V08<2E`0002@0001U0000P@8006H000230P00
0P0006H0P`9U0842E`0002@0001T0000P08006D000210P000P0006D0P@9T0802E`0002@0001R0000
O`8006@000200P000P0006@0P09R07l2E`0002@0001]0000T@8006h0002B0P000P0006h0T@9]0982
E`0002@0001]0000TP8006h0002B0P000P0006d0TP9^0982E`0002@0001^0000TP8006l0002B0P00
0P0006h0TP9_0982E`0002@0001^0000T@8006l0002B0P000P0006l0TP9^0942E`0002@0001h0000
O`8007/000200P000P0007/0O`9h0802E`0002@0001f0000P08007P000230P000P0007P0P09f08<2
E`0002@0001e0000P`8007H000270P000P0007H0P`9e08L2E`0002@0001e0000Q`8007D0002:0P00
0P0007D0Q`9e08X2E`0002@0001e0000RP8007H0002?0P000P0007D0RP9f08l2E`0002@0001f0000
S`8007P0002B0P000P0007H0S`9h0982E`0002@0001h0000TP8007/0002B0P000P0007P0TP9k0982
E`0002@0001k0000TP8007`0002B0P000P0007/0TP9l0982E`0002@0001l0000TP8007l0002B0P00
0P0007`0TP9o0982E`0002@0001o0000S`800840002B0P000P0007l0TP:108l2E`0002@000210000
RP800880002?0P000P000840S`:208X2E`0002@000220000Q`800880002:0P000P000880RP:208L2
E`0002@000210000P`80088000270P000P000880Q`:108<2E`0002@0001o0000P080084000230P00
0P000840P`9o0802E`0002@0001l0000O`8007l000200P000P0007l0P09l07l2E`0002@0001k0000
O`8007`0001o0P000P0007`0O`9k07l2E`0002@0001i0000O`8007/000200P000P0007/0O`9i0802
E`0002@0001h0000P08007T000210P000P0007T0P09h0842E`0002@0001g0000P@8007P000230P00
0P0007P0P@9g08<2E`0002@0001f0000P`8007L000270P000P0007L0P`9f08L2E`0002@0001f0000
Q`8007H0002:0P000P0007H0Q`9f08X2E`0002@0001f0000RP8007L0002?0P000P0007H0RP9g08l2
E`0002@0001g0000S`8007P0002A0P000P0007L0S`9h0942E`0002@0001h0000T@8007T0002B0P00
0P0007P0T@9i0982E`0002@0001i0000TP8007/0002B0P000P0007T0TP9k0982E`0002@0001l0000
TP8007h0002B0P000P0007`0TP9n0982E`0002@0001n0000T@8007l0002B0P000P0007h0TP9o0942
E`0002@0001o0000S`800800002A0P000P0007l0T@:008l2E`0002@000200000RP800840002?0P00
0P000800S`:108X2E`0002@000210000Q`800840002:0P000P000840RP:108L2E`0002@000200000
P`80084000270P000P000840Q`:008<2E`0002@0001o0000P@80080000230P000P000800P`9o0842
E`0002@0001n0000P08007l000210P000P0007l0P@9n0802E`0002@0001l0000O`8007h000200P00
0P0007h0P09l07l2E`0002@000270000O`8008T000280P000P0008T0O`:708P2E`0002@000270000
QP8008T000280P000P0008L0R0:908H2E`0002@000290000Q@8008`000260P000P0008T0QP:<08D2
E`0002@0002<0000Q@8008h000250P000P0008`0Q@:>08D2E`0002@0002>0000Q@80094000260P00
0P0008h0Q@:A08H2E`0002@0002A0000QP8009<000280P000P000940QP:C08P2E`0002@0002C0000
R08009@0002;0P000P0009<0R0:D08/2E`0002@0002D0000R`8009@0002=0P000P0009@0R`:D08d2
E`0002@0002C0000S@8009@0002@0P000P0009@0S@:C0902E`0002@0002A0000T08009<0002B0P00
0P0009<0T0:A0982E`0002@0002>0000TP800940002B0P000P000940TP:>0982E`0002@0002<0000
TP8008h0002B0P000P0008h0TP:<0982E`0002@000290000TP8008`0002B0P000P0008`0TP:90982
E`0002@000280000T@8008T0002B0P000P0008T0TP:80942E`0002@000270000S`8008P0002A0P00
0P0008P0T@:708l2E`0002@000270000SP8008L0002?0P000P0008L0S`:708h2E`0002@000270000
S@8008P0002>0P000P0008L0SP:808d2E`0002@000280000S@8008T0002>0P000P0008P0S@:908h2
E`0002@000280000SP8008T0002?0P000P0008T0SP:808l2E`0002@0002>0000Q@80090000260P00
0P0008h0Q@:@08H2E`0002@0002@0000QP80098000280P000P000900QP:B08P2E`0002@0002B0000
R08009<0002;0P000P000980R0:C08/2E`0002@0002C0000R`8009<0002=0P000P0009<0R`:C08d2
E`0002@0002B0000S@8009<0002@0P000P0009<0S@:B0902E`0002@0002@0000T0800980002B0P00
0P000980T0:@0982E`0002@0002>0000TP800900002B0P000P000900TP:>0982E`0002@000290000
O`800980001o0P000P0008T0O`:B07l2E`0002@000290000P08008d000200P000P0008T0P0:=0802
E`0002@0002=0000O`80098000200P000P0008d0P0:B07l2E`0002@000140000>@8005@0000i0P00
0P0004@0>@9D03T2E`0002@0001N0000;P800640000_0P000P000640;P9N02l2E`0002@0001L0000
;`8005h0000b0P000P0005h0;`9L0382E`0002@0001K0000<P8005`0000f0P000P0005`0<P9K03H2
E`0002@0001K0000=P8005/0000i0P000P0005/0=P9K03T2E`0002@0001K0000>@8005`0000n0P00
0P0005/0>@9L03h2E`0002@0001L0000?P8005h000110P000P0005`0?P9N0442E`0002@0001N0000
@@80064000110P000P0005h0@@9Q0442E`0002@0001Q0000@@80068000110P000P000640@@9R0442
E`0002@0001R0000@@8006D000110P000P000680@@9U0442E`0002@0001U0000?P8006L000110P00
0P0006D0@@9W03h2E`0002@0001W0000>@8006P0000n0P000P0006L0?P9X03T2E`0002@0001X0000
=P8006P0000i0P000P0006P0>@9X03H2E`0002@0001W0000<P8006P0000f0P000P0006P0=P9W0382
E`0002@0001U0000;`8006L0000b0P000P0006L0<P9U02l2E`0002@0001R0000;P8006D0000_0P00
0P0006D0;`9R02h2E`0002@0001Q0000;P800680000^0P000P000680;P9Q02h2E`0002@0001O0000
;P800640000_0P000P000640;P9O02l2E`0002@0001N0000;`8005l0000`0P000P0005l0;`9N0302
E`0002@0001M0000<08005h0000b0P000P0005h0<09M0382E`0002@0001L0000<P8005d0000f0P00
0P0005d0<P9L03H2E`0002@0001L0000=P8005`0000i0P000P0005`0=P9L03T2E`0002@0001L0000
>@8005d0000n0P000P0005`0>@9M03h2E`0002@0001M0000?P8005h000100P000P0005d0?P9N0402
E`0002@0001N0000@08005l000110P000P0005h0@09O0442E`0002@0001O0000@@80064000110P00
0P0005l0@@9Q0442E`0002@0001R0000@@8006@000110P000P000680@@9T0442E`0002@0001T0000
@08006D000110P000P0006@0@@9U0402E`0002@0001U0000?P8006H000100P000P0006D0@09V03h2
E`0002@0001V0000>@8006L0000n0P000P0006H0?P9W03T2E`0002@0001W0000=P8006L0000i0P00
0P0006L0>@9W03H2E`0002@0001V0000<P8006L0000f0P000P0006L0=P9V0382E`0002@0001U0000
<08006H0000b0P000P0006H0<P9U0302E`0002@0001T0000;`8006D0000`0P000P0006D0<09T02l2
E`0002@0001R0000;P8006@0000_0P000P0006@0;`9R02h2E`0002@0001]0000@08006h000110P00
0P0006h0@09]0442E`0002@0001]0000@@8006h000110P000P0006d0@@9^0442E`0002@0001^0000
@@8006l000110P000P0006h0@@9_0442E`0002@0001^0000@08006l000110P000P0006l0@@9^0402
E`0002@0001h0000;P8007/0000_0P000P0007/0;P9h02l2E`0002@0001f0000;`8007P0000b0P00
0P0007P0;`9f0382E`0002@0001e0000<P8007H0000f0P000P0007H0<P9e03H2E`0002@0001e0000
=P8007D0000i0P000P0007D0=P9e03T2E`0002@0001e0000>@8007H0000n0P000P0007D0>@9f03h2
E`0002@0001f0000?P8007P000110P000P0007H0?P9h0442E`0002@0001h0000@@8007/000110P00
0P0007P0@@9k0442E`0002@0001k0000@@8007`000110P000P0007/0@@9l0442E`0002@0001l0000
@@8007l000110P000P0007`0@@9o0442E`0002@0001o0000?P80084000110P000P0007l0@@:103h2
E`0002@000210000>@800880000n0P000P000840?P:203T2E`0002@000220000=P800880000i0P00
0P000880>@:203H2E`0002@000210000<P800880000f0P000P000880=P:10382E`0002@0001o0000
;`800840000b0P000P000840<P9o02l2E`0002@0001l0000;P8007l0000_0P000P0007l0;`9l02h2
E`0002@0001k0000;P8007`0000^0P000P0007`0;P9k02h2E`0002@0001i0000;P8007/0000_0P00
0P0007/0;P9i02l2E`0002@0001h0000;`8007T0000`0P000P0007T0;`9h0302E`0002@0001g0000
<08007P0000b0P000P0007P0<09g0382E`0002@0001f0000<P8007L0000f0P000P0007L0<P9f03H2
E`0002@0001f0000=P8007H0000i0P000P0007H0=P9f03T2E`0002@0001f0000>@8007L0000n0P00
0P0007H0>@9g03h2E`0002@0001g0000?P8007P000100P000P0007L0?P9h0402E`0002@0001h0000
@08007T000110P000P0007P0@09i0442E`0002@0001i0000@@8007/000110P000P0007T0@@9k0442
E`0002@0001l0000@@8007h000110P000P0007`0@@9n0442E`0002@0001n0000@08007l000110P00
0P0007h0@@9o0402E`0002@0001o0000?P80080000100P000P0007l0@0:003h2E`0002@000200000
>@800840000n0P000P000800?P:103T2E`0002@000210000=P800840000i0P000P000840>@:103H2
E`0002@000200000<P800840000f0P000P000840=P:00382E`0002@0001o0000<0800800000b0P00
0P000800<P9o0302E`0002@0001n0000;`8007l0000`0P000P0007l0<09n02l2E`0002@0001l0000
;P8007h0000_0P000P0007h0;`9l02h2E`0002@0002?0000<08008l000110P000P0008l0<0:?0442
E`0002@0002@0000;P80090000110P000P000900;P:@0442E`0002@000260000;P800900000l0P00
0P000900;P:603`2E`0002@000260000?08009D0000l0P000P0008H0?0:E03`2E`0002@0002=0000
@@8009<000110P000P0008d0@@:C0442E`0002@000140000i`4005@0003W0@000P0004@0i`5D0>L1
E`0002@0001N0000g0400640003M0@000P000640g05N0=d1E`0002@0001L0000g@4005h0003P0@00
0P0005h0g@5L0>01E`0002@0001K0000h04005`0003T0@000P0005`0h05K0>@1E`0002@0001K0000
i04005/0003W0@000P0005/0i05K0>L1E`0002@0001K0000i`4005`0003/0@000P0005/0i`5L0>`1
E`0002@0001L0000k04005h0003_0@000P0005`0k05N0>l1E`0002@0001N0000k`400640003_0@00
0P0005h0k`5Q0>l1E`0002@0001Q0000k`400680003_0@000P000640k`5R0>l1E`0002@0001R0000
k`4006D0003_0@000P000680k`5U0>l1E`0002@0001U0000k04006L0003_0@000P0006D0k`5W0>`1
E`0002@0001W0000i`4006P0003/0@000P0006L0k05X0>L1E`0002@0001X0000i04006P0003W0@00
0P0006P0i`5X0>@1E`0002@0001W0000h04006P0003T0@000P0006P0i05W0>01E`0002@0001U0000
g@4006L0003P0@000P0006L0h05U0=d1E`0002@0001R0000g04006D0003M0@000P0006D0g@5R0=`1
E`0002@0001Q0000g0400680003L0@000P000680g05Q0=`1E`0002@0001O0000g0400640003M0@00
0P000640g05O0=d1E`0002@0001N0000g@4005l0003N0@000P0005l0g@5N0=h1E`0002@0001M0000
gP4005h0003P0@000P0005h0gP5M0>01E`0002@0001L0000h04005d0003T0@000P0005d0h05L0>@1
E`0002@0001L0000i04005`0003W0@000P0005`0i05L0>L1E`0002@0001L0000i`4005d0003/0@00
0P0005`0i`5M0>`1E`0002@0001M0000k04005h0003^0@000P0005d0k05N0>h1E`0002@0001N0000
kP4005l0003_0@000P0005h0kP5O0>l1E`0002@0001O0000k`400640003_0@000P0005l0k`5Q0>l1
E`0002@0001R0000k`4006@0003_0@000P000680k`5T0>l1E`0002@0001T0000kP4006D0003_0@00
0P0006@0k`5U0>h1E`0002@0001U0000k04006H0003^0@000P0006D0kP5V0>`1E`0002@0001V0000
i`4006L0003/0@000P0006H0k05W0>L1E`0002@0001W0000i04006L0003W0@000P0006L0i`5W0>@1
E`0002@0001V0000h04006L0003T0@000P0006L0i05V0>01E`0002@0001U0000gP4006H0003P0@00
0P0006H0h05U0=h1E`0002@0001T0000g@4006D0003N0@000P0006D0gP5T0=d1E`0002@0001R0000
g04006@0003M0@000P0006@0g@5R0=`1E`0002@0001]0000kP4006h0003_0@000P0006h0kP5]0>l1
E`0002@0001]0000k`4006h0003_0@000P0006d0k`5^0>l1E`0002@0001^0000k`4006l0003_0@00
0P0006h0k`5_0>l1E`0002@0001^0000kP4006l0003_0@000P0006l0k`5^0>h1E`0002@0001h0000
g04007/0003M0@000P0007/0g05h0=d1E`0002@0001f0000g@4007P0003P0@000P0007P0g@5f0>01
E`0002@0001e0000h04007H0003T0@000P0007H0h05e0>@1E`0002@0001e0000i04007D0003W0@00
0P0007D0i05e0>L1E`0002@0001e0000i`4007H0003/0@000P0007D0i`5f0>`1E`0002@0001f0000
k04007P0003_0@000P0007H0k05h0>l1E`0002@0001h0000k`4007/0003_0@000P0007P0k`5k0>l1
E`0002@0001k0000k`4007`0003_0@000P0007/0k`5l0>l1E`0002@0001l0000k`4007l0003_0@00
0P0007`0k`5o0>l1E`0002@0001o0000k0400840003_0@000P0007l0k`610>`1E`0002@000210000
i`400880003/0@000P000840k0620>L1E`0002@000220000i0400880003W0@000P000880i`620>@1
E`0002@000210000h0400880003T0@000P000880i0610>01E`0002@0001o0000g@400840003P0@00
0P000840h05o0=d1E`0002@0001l0000g04007l0003M0@000P0007l0g@5l0=`1E`0002@0001k0000
g04007`0003L0@000P0007`0g05k0=`1E`0002@0001i0000g04007/0003M0@000P0007/0g05i0=d1
E`0002@0001h0000g@4007T0003N0@000P0007T0g@5h0=h1E`0002@0001g0000gP4007P0003P0@00
0P0007P0gP5g0>01E`0002@0001f0000h04007L0003T0@000P0007L0h05f0>@1E`0002@0001f0000
i04007H0003W0@000P0007H0i05f0>L1E`0002@0001f0000i`4007L0003/0@000P0007H0i`5g0>`1
E`0002@0001g0000k04007P0003^0@000P0007L0k05h0>h1E`0002@0001h0000kP4007T0003_0@00
0P0007P0kP5i0>l1E`0002@0001i0000k`4007/0003_0@000P0007T0k`5k0>l1E`0002@0001l0000
k`4007h0003_0@000P0007`0k`5n0>l1E`0002@0001n0000kP4007l0003_0@000P0007h0k`5o0>h1
E`0002@0001o0000k0400800003^0@000P0007l0kP600>`1E`0002@000200000i`400840003/0@00
0P000800k0610>L1E`0002@000210000i0400840003W0@000P000840i`610>@1E`0002@000200000
h0400840003T0@000P000840i0600>01E`0002@0001o0000gP400800003P0@000P000800h05o0=h1
E`0002@0001n0000g@4007l0003N0@000P0007l0gP5n0=d1E`0002@0001l0000g04007h0003M0@00
0P0007h0g@5l0=`1E`0002@000280000h04008T0003Q0@000P0008P0h0690>41E`0002@000280000
h@4008T0003Q0@000P0008T0h@680>41E`0002@000270000h@4008P0003Q0@000P0008P0h@670>41
E`0002@000270000h04008L0003Q0@000P0008L0h@670>01E`0002@000270000gP4008P0003P0@00
0P0008L0h0680=h1E`0002@000280000g@4008T0003N0@000P0008P0gP690=d1E`0002@000290000
g04008`0003M0@000P0008T0g@6<0=`1E`0002@0002<0000g04008l0003L0@000P0008`0g06?0=`1
E`0002@0002?0000g0400980003M0@000P0008l0g06B0=d1E`0002@0002B0000g@4009<0003O0@00
0P000980g@6C0=l1E`0002@0002C0000g`4009<0003Q0@000P0009<0g`6C0>41E`0002@0002B0000
h@4009<0003S0@000P0009<0h@6B0><1E`0002@0002?0000h`400980003T0@000P000980h`6?0>@1
E`0002@0002=0000i04008l0003T0@000P0008l0i06=0>@1E`0002@0002?0000g0400940003M0@00
0P0008l0g06A0=d1E`0002@0002A0000g@400980003O0@000P000940g@6B0=l1E`0002@0002B0000
g`400980003Q0@000P000980g`6B0>41E`0002@0002A0000h@400980003S0@000P000980h@6A0><1
E`0002@0002?0000h`400940003T0@000P000940h`6?0>@1E`0002@0002?0000i0400940003U0@00
0P0008l0i06A0>D1E`0002@0002A0000i@4009<0003W0@000P000940i@6C0>L1E`0002@0002C0000
i`4009@0003Y0@000P0009<0i`6D0>T1E`0002@0002D0000j@4009@0003/0@000P0009@0j@6D0>`1
E`0002@0002C0000k04009@0003^0@000P0009@0k06C0>h1E`0002@0002B0000kP4009<0003_0@00
0P0009<0kP6B0>l1E`0002@0002?0000k`400980003_0@000P000980k`6?0>l1E`0002@0002<0000
k`4008l0003_0@000P0008l0k`6<0>l1E`0002@000290000k`4008`0003_0@000P0008`0k`690>l1
E`0002@000280000kP4008T0003_0@000P0008T0k`680>h1E`0002@000270000k04008P0003^0@00
0P0008P0kP670>`1E`0002@000270000j`4008L0003/0@000P0008L0k0670>/1E`0002@000270000
jP4008P0003[0@000P0008L0j`680>X1E`0002@000280000jP4008T0003[0@000P0008P0jP690>/1
E`0002@000280000j`4008T0003/0@000P0008T0j`680>`1E`0002@0002B0000iP4009<0003Y0@00
0P000980iP6C0>T1E`0002@0002C0000j@4009<0003/0@000P0009<0j@6C0>`1E`0002@0002B0000
k04009<0003^0@000P0009<0k06B0>h1E`0002@0002A0000kP400980003_0@000P000980kP6A0>l1
E`0002@0002?0000k`400940003_0@000P000940k`6?0>l1E`0002@000140000UP4005@0002F0@00
0P0004@0UP5D09H1E`0002@0001N0000R`400640002<0@000P000640R`5N08`1E`0002@0001L0000
S04005h0002?0@000P0005h0S05L08l1E`0002@0001K0000S`4005`0002C0@000P0005`0S`5K09<1
E`0002@0001K0000T`4005/0002F0@000P0005/0T`5K09H1E`0002@0001K0000UP4005`0002K0@00
0P0005/0UP5L09/1E`0002@0001L0000V`4005h0002N0@000P0005`0V`5N09h1E`0002@0001N0000
WP400640002N0@000P0005h0WP5Q09h1E`0002@0001Q0000WP400680002N0@000P000640WP5R09h1
E`0002@0001R0000WP4006D0002N0@000P000680WP5U09h1E`0002@0001U0000V`4006L0002N0@00
0P0006D0WP5W09/1E`0002@0001W0000UP4006P0002K0@000P0006L0V`5X09H1E`0002@0001X0000
T`4006P0002F0@000P0006P0UP5X09<1E`0002@0001W0000S`4006P0002C0@000P0006P0T`5W08l1
E`0002@0001U0000S04006L0002?0@000P0006L0S`5U08`1E`0002@0001R0000R`4006D0002<0@00
0P0006D0S05R08/1E`0002@0001Q0000R`400680002;0@000P000680R`5Q08/1E`0002@0001O0000
R`400640002<0@000P000640R`5O08`1E`0002@0001N0000S04005l0002=0@000P0005l0S05N08d1
E`0002@0001M0000S@4005h0002?0@000P0005h0S@5M08l1E`0002@0001L0000S`4005d0002C0@00
0P0005d0S`5L09<1E`0002@0001L0000T`4005`0002F0@000P0005`0T`5L09H1E`0002@0001L0000
UP4005d0002K0@000P0005`0UP5M09/1E`0002@0001M0000V`4005h0002M0@000P0005d0V`5N09d1
E`0002@0001N0000W@4005l0002N0@000P0005h0W@5O09h1E`0002@0001O0000WP400640002N0@00
0P0005l0WP5Q09h1E`0002@0001R0000WP4006@0002N0@000P000680WP5T09h1E`0002@0001T0000
W@4006D0002N0@000P0006@0WP5U09d1E`0002@0001U0000V`4006H0002M0@000P0006D0W@5V09/1
E`0002@0001V0000UP4006L0002K0@000P0006H0V`5W09H1E`0002@0001W0000T`4006L0002F0@00
0P0006L0UP5W09<1E`0002@0001V0000S`4006L0002C0@000P0006L0T`5V08l1E`0002@0001U0000
S@4006H0002?0@000P0006H0S`5U08d1E`0002@0001T0000S04006D0002=0@000P0006D0S@5T08`1
E`0002@0001R0000R`4006@0002<0@000P0006@0S05R08/1E`0002@0001]0000W@4006h0002N0@00
0P0006h0W@5]09h1E`0002@0001]0000WP4006h0002N0@000P0006d0WP5^09h1E`0002@0001^0000
WP4006l0002N0@000P0006h0WP5_09h1E`0002@0001^0000W@4006l0002N0@000P0006l0WP5^09d1
E`0002@0001h0000R`4007/0002<0@000P0007/0R`5h08`1E`0002@0001f0000S04007P0002?0@00
0P0007P0S05f08l1E`0002@0001e0000S`4007H0002C0@000P0007H0S`5e09<1E`0002@0001e0000
T`4007D0002F0@000P0007D0T`5e09H1E`0002@0001e0000UP4007H0002K0@000P0007D0UP5f09/1
E`0002@0001f0000V`4007P0002N0@000P0007H0V`5h09h1E`0002@0001h0000WP4007/0002N0@00
0P0007P0WP5k09h1E`0002@0001k0000WP4007`0002N0@000P0007/0WP5l09h1E`0002@0001l0000
WP4007l0002N0@000P0007`0WP5o09h1E`0002@0001o0000V`400840002N0@000P0007l0WP6109/1
E`0002@000210000UP400880002K0@000P000840V`6209H1E`0002@000220000T`400880002F0@00
0P000880UP6209<1E`0002@000210000S`400880002C0@000P000880T`6108l1E`0002@0001o0000
S0400840002?0@000P000840S`5o08`1E`0002@0001l0000R`4007l0002<0@000P0007l0S05l08/1
E`0002@0001k0000R`4007`0002;0@000P0007`0R`5k08/1E`0002@0001i0000R`4007/0002<0@00
0P0007/0R`5i08`1E`0002@0001h0000S04007T0002=0@000P0007T0S05h08d1E`0002@0001g0000
S@4007P0002?0@000P0007P0S@5g08l1E`0002@0001f0000S`4007L0002C0@000P0007L0S`5f09<1
E`0002@0001f0000T`4007H0002F0@000P0007H0T`5f09H1E`0002@0001f0000UP4007L0002K0@00
0P0007H0UP5g09/1E`0002@0001g0000V`4007P0002M0@000P0007L0V`5h09d1E`0002@0001h0000
W@4007T0002N0@000P0007P0W@5i09h1E`0002@0001i0000WP4007/0002N0@000P0007T0WP5k09h1
E`0002@0001l0000WP4007h0002N0@000P0007`0WP5n09h1E`0002@0001n0000W@4007l0002N0@00
0P0007h0WP5o09d1E`0002@0001o0000V`400800002M0@000P0007l0W@6009/1E`0002@000200000
UP400840002K0@000P000800V`6109H1E`0002@000210000T`400840002F0@000P000840UP6109<1
E`0002@000200000S`400840002C0@000P000840T`6008l1E`0002@0001o0000S@400800002?0@00
0P000800S`5o08d1E`0002@0001n0000S04007l0002=0@000P0007l0S@5n08`1E`0002@0001l0000
R`4007h0002<0@000P0007h0S05l08/1E`0002@000280000S`4008T0002@0@000P0008P0S`690901
E`0002@000280000T04008T0002@0@000P0008T0T0680901E`0002@000270000T04008P0002@0@00
0P0008P0T0670901E`0002@000270000S`4008L0002@0@000P0008L0T06708l1E`0002@000270000
S@4008P0002?0@000P0008L0S`6808d1E`0002@000280000S04008T0002=0@000P0008P0S@6908`1
E`0002@000290000R`4008`0002<0@000P0008T0S06<08/1E`0002@0002<0000R`4008l0002;0@00
0P0008`0R`6?08/1E`0002@0002?0000R`400980002<0@000P0008l0R`6B08`1E`0002@0002B0000
S04009<0002=0@000P000980S06C08d1E`0002@0002C0000S@4009@0002?0@000P0009<0S@6D08l1
E`0002@0002D0000S`4009@0002@0@000P0009@0S`6D0901E`0002@0002C0000T04009@0002B0@00
0P0009@0T06C0981E`0002@0002@0000TP4009<0002D0@000P0009<0TP6@09@1E`0002@0002<0000
U0400900002F0@000P000900U06<09H1E`0002@0002:0000UP4008`0002G0@000P0008`0UP6:09L1
E`0002@000280000U`4008X0002I0@000P0008X0U`6809T1E`0002@000270000V@4008P0002L0@00
0P0008P0V@6709`1E`0002@000270000W04008L0002N0@000P0008L0W06709h1E`0002@0002?0000
R`400940002<0@000P0008l0R`6A08`1E`0002@0002A0000S0400980002=0@000P000940S06B08d1
E`0002@0002B0000S@4009<0002?0@000P000980S@6C08l1E`0002@0002C0000S`4009<0002@0@00
0P0009<0S`6C0901E`0002@0002B0000T04009<0002B0@000P0009<0T06B0981E`0002@0002?0000
TP400980002D0@000P000980TP6?09@1E`0002@0002<0000U04008l0002F0@000P0008l0U06<09H1
E`0002@000270000W04008P0002M0@000P0008L0W@6809`1E`0002@000280000W04008X0002L0@00
0P0008P0W06:09`1E`0002@0002:0000W04008h0002N0@000P0008X0W06>09h1E`0002@0002>0000
WP400940002N0@000P0008h0WP6A09h1E`0002@0002A0000W@4009<0002N0@000P000940WP6C09d1
E`0002@0002C0000W04009@0002M0@000P0009<0W@6D09`1E`0002@0002:0000W04008h0002N0@00
0P0008X0W06>09h1E`0002@0002>0000WP400980002N0@000P0008h0WP6B09h1E`0002@0002B0000
WP4009<0002N0@000P000980WP6C09h1E`0002@0002C0000W04009@0002N0@000P0009<0WP6D09`1
E`0002@0002D0000VP4009@0002L0@000P0009@0W06D09X1E`0002@000140000A@4005@000150@00
0P0004@0A@5D04D1E`0002@0001N0000>P400640000k0@000P000640>P5N03/1E`0002@0001L0000
>`4005h0000n0@000P0005h0>`5L03h1E`0002@0001K0000?P4005`000120@000P0005`0?P5K0481
E`0002@0001K0000@P4005/000150@000P0005/0@P5K04D1E`0002@0001K0000A@4005`0001:0@00
0P0005/0A@5L04X1E`0002@0001L0000BP4005h0001=0@000P0005`0BP5N04d1E`0002@0001N0000
C@400640001=0@000P0005h0C@5Q04d1E`0002@0001Q0000C@400680001=0@000P000640C@5R04d1
E`0002@0001R0000C@4006D0001=0@000P000680C@5U04d1E`0002@0001U0000BP4006L0001=0@00
0P0006D0C@5W04X1E`0002@0001W0000A@4006P0001:0@000P0006L0BP5X04D1E`0002@0001X0000
@P4006P000150@000P0006P0A@5X0481E`0002@0001W0000?P4006P000120@000P0006P0@P5W03h1
E`0002@0001U0000>`4006L0000n0@000P0006L0?P5U03/1E`0002@0001R0000>P4006D0000k0@00
0P0006D0>`5R03X1E`0002@0001Q0000>P400680000j0@000P000680>P5Q03X1E`0002@0001O0000
>P400640000k0@000P000640>P5O03/1E`0002@0001N0000>`4005l0000l0@000P0005l0>`5N03`1
E`0002@0001M0000?04005h0000n0@000P0005h0?05M03h1E`0002@0001L0000?P4005d000120@00
0P0005d0?P5L0481E`0002@0001L0000@P4005`000150@000P0005`0@P5L04D1E`0002@0001L0000
A@4005d0001:0@000P0005`0A@5M04X1E`0002@0001M0000BP4005h0001<0@000P0005d0BP5N04`1
E`0002@0001N0000C04005l0001=0@000P0005h0C05O04d1E`0002@0001O0000C@400640001=0@00
0P0005l0C@5Q04d1E`0002@0001R0000C@4006@0001=0@000P000680C@5T04d1E`0002@0001T0000
C04006D0001=0@000P0006@0C@5U04`1E`0002@0001U0000BP4006H0001<0@000P0006D0C05V04X1
E`0002@0001V0000A@4006L0001:0@000P0006H0BP5W04D1E`0002@0001W0000@P4006L000150@00
0P0006L0A@5W0481E`0002@0001V0000?P4006L000120@000P0006L0@P5V03h1E`0002@0001U0000
?04006H0000n0@000P0006H0?P5U03`1E`0002@0001T0000>`4006D0000l0@000P0006D0?05T03/1
E`0002@0001R0000>P4006@0000k0@000P0006@0>`5R03X1E`0002@0001]0000C04006h0001=0@00
0P0006h0C05]04d1E`0002@0001]0000C@4006h0001=0@000P0006d0C@5^04d1E`0002@0001^0000
C@4006l0001=0@000P0006h0C@5_04d1E`0002@0001^0000C04006l0001=0@000P0006l0C@5^04`1
E`0002@0001h0000>P4007/0000k0@000P0007/0>P5h03/1E`0002@0001f0000>`4007P0000n0@00
0P0007P0>`5f03h1E`0002@0001e0000?P4007H000120@000P0007H0?P5e0481E`0002@0001e0000
@P4007D000150@000P0007D0@P5e04D1E`0002@0001e0000A@4007H0001:0@000P0007D0A@5f04X1
E`0002@0001f0000BP4007P0001=0@000P0007H0BP5h04d1E`0002@0001h0000C@4007/0001=0@00
0P0007P0C@5k04d1E`0002@0001k0000C@4007`0001=0@000P0007/0C@5l04d1E`0002@0001l0000
C@4007l0001=0@000P0007`0C@5o04d1E`0002@0001o0000BP400840001=0@000P0007l0C@6104X1
E`0002@000210000A@400880001:0@000P000840BP6204D1E`0002@000220000@P40088000150@00
0P000880A@620481E`0002@000210000?P40088000120@000P000880@P6103h1E`0002@0001o0000
>`400840000n0@000P000840?P5o03/1E`0002@0001l0000>P4007l0000k0@000P0007l0>`5l03X1
E`0002@0001k0000>P4007`0000j0@000P0007`0>P5k03X1E`0002@0001i0000>P4007/0000k0@00
0P0007/0>P5i03/1E`0002@0001h0000>`4007T0000l0@000P0007T0>`5h03`1E`0002@0001g0000
?04007P0000n0@000P0007P0?05g03h1E`0002@0001f0000?P4007L000120@000P0007L0?P5f0481
E`0002@0001f0000@P4007H000150@000P0007H0@P5f04D1E`0002@0001f0000A@4007L0001:0@00
0P0007H0A@5g04X1E`0002@0001g0000BP4007P0001<0@000P0007L0BP5h04`1E`0002@0001h0000
C04007T0001=0@000P0007P0C05i04d1E`0002@0001i0000C@4007/0001=0@000P0007T0C@5k04d1
E`0002@0001l0000C@4007h0001=0@000P0007`0C@5n04d1E`0002@0001n0000C04007l0001=0@00
0P0007h0C@5o04`1E`0002@0001o0000BP400800001<0@000P0007l0C06004X1E`0002@000200000
A@400840001:0@000P000800BP6104D1E`0002@000210000@P40084000150@000P000840A@610481
E`0002@000200000?P40084000120@000P000840@P6003h1E`0002@0001o0000?0400800000n0@00
0P000800?P5o03`1E`0002@0001n0000>`4007l0000l0@000P0007l0?05n03/1E`0002@0001l0000
>P4007h0000k0@000P0007h0>`5l03X1E`0002@0002:0000?@4008`0000n0@000P0008X0?P6<03d1
E`0002@0002<0000>P4008h0000m0@000P0008`0?@6>03X1E`0002@0002>0000>P4008h0001=0@00
0P0008h0>P6>04d1E`0002@0002=0000>`4008d0001=0@000P0008d0>`6=04d1E`0002@0002:0000
C@400980001=0@000P0008X0C@6B04d1E`0002@0001N0000j@000640003Z00000P000640j@1N0>X0
E`0002@0001L0000jP0005h0003]00000P0005h0jP1L0>d0E`0002@0001K0000k@0005`0003a0000
0P0005`0k@1K0?40E`0002@0001K0000l@0005/0003d00000P0005/0l@1K0?@0E`0002@0001K0000
m00005`0003i00000P0005/0m01L0?T0E`0002@0001L0000n@0005h0003l00000P0005`0n@1N0?`0
E`0002@0001N0000o0000640003l00000P0005h0o01Q0?`0E`0002@0001Q0000o0000680003l0000
0P000640o01R0?`0E`0002@0001R0000o00006D0003l00000P000680o01U0?`0E`0002@0001U0000
n@0006L0003l00000P0006D0o01W0?T0E`0002@0001W0000m00006P0003i00000P0006L0n@1X0?@0
E`0002@0001X0000l@0006P0003d00000P0006P0m01X0?40E`0002@0001W0000k@0006P0003a0000
0P0006P0l@1W0>d0E`0002@0001U0000jP0006L0003]00000P0006L0k@1U0>X0E`0002@0001R0000
j@0006D0003Z00000P0006D0jP1R0>T0E`0002@0001Q0000j@000680003Y00000P000680j@1Q0>T0
E`0002@0001O0000j@000640003Z00000P000640j@1O0>X0E`0002@0001N0000jP0005l0003[0000
0P0005l0jP1N0>/0E`0002@0001M0000j`0005h0003]00000P0005h0j`1M0>d0E`0002@0001L0000
k@0005d0003a00000P0005d0k@1L0?40E`0002@0001L0000l@0005`0003d00000P0005`0l@1L0?@0
E`0002@0001L0000m00005d0003i00000P0005`0m01M0?T0E`0002@0001M0000n@0005h0003k0000
0P0005d0n@1N0?/0E`0002@0001N0000n`0005l0003l00000P0005h0n`1O0?`0E`0002@0001O0000
o0000640003l00000P0005l0o01Q0?`0E`0002@0001R0000o00006@0003l00000P000680o01T0?`0
E`0002@0001T0000n`0006D0003l00000P0006@0o01U0?/0E`0002@0001U0000n@0006H0003k0000
0P0006D0n`1V0?T0E`0002@0001V0000m00006L0003i00000P0006H0n@1W0?@0E`0002@0001W0000
l@0006L0003d00000P0006L0m01W0?40E`0002@0001V0000k@0006L0003a00000P0006L0l@1V0>d0
E`0002@0001U0000j`0006H0003]00000P0006H0k@1U0>/0E`0002@0001T0000jP0006D0003[0000
0P0006D0j`1T0>X0E`0002@0001R0000j@0006@0003Z00000P0006@0jP1R0>T0E`0002@0001]0000
n`0006h0003l00000P0006h0n`1]0?`0E`0002@0001]0000o00006h0003l00000P0006d0o01^0?`0
E`0002@0001^0000o00006l0003l00000P0006h0o01_0?`0E`0002@0001^0000n`0006l0003l0000
0P0006l0o01^0?/0E`0002@0001h0000j@0007/0003Z00000P0007/0j@1h0>X0E`0002@0001f0000
jP0007P0003]00000P0007P0jP1f0>d0E`0002@0001e0000k@0007H0003a00000P0007H0k@1e0?40
E`0002@0001e0000l@0007D0003d00000P0007D0l@1e0?@0E`0002@0001e0000m00007H0003i0000
0P0007D0m01f0?T0E`0002@0001f0000n@0007P0003l00000P0007H0n@1h0?`0E`0002@0001h0000
o00007/0003l00000P0007P0o01k0?`0E`0002@0001k0000o00007`0003l00000P0007/0o01l0?`0
E`0002@0001l0000o00007l0003l00000P0007`0o01o0?`0E`0002@0001o0000n@000840003l0000
0P0007l0o0210?T0E`0002@000210000m0000880003i00000P000840n@220?@0E`0002@000220000
l@000880003d00000P000880m0220?40E`0002@000210000k@000880003a00000P000880l@210>d0
E`0002@0001o0000jP000840003]00000P000840k@1o0>X0E`0002@0001l0000j@0007l0003Z0000
0P0007l0jP1l0>T0E`0002@0001k0000j@0007`0003Y00000P0007`0j@1k0>T0E`0002@0001i0000
j@0007/0003Z00000P0007/0j@1i0>X0E`0002@0001h0000jP0007T0003[00000P0007T0jP1h0>/0
E`0002@0001g0000j`0007P0003]00000P0007P0j`1g0>d0E`0002@0001f0000k@0007L0003a0000
0P0007L0k@1f0?40E`0002@0001f0000l@0007H0003d00000P0007H0l@1f0?@0E`0002@0001f0000
m00007L0003i00000P0007H0m01g0?T0E`0002@0001g0000n@0007P0003k00000P0007L0n@1h0?/0
E`0002@0001h0000n`0007T0003l00000P0007P0n`1i0?`0E`0002@0001i0000o00007/0003l0000
0P0007T0o01k0?`0E`0002@0001l0000o00007h0003l00000P0007`0o01n0?`0E`0002@0001n0000
n`0007l0003l00000P0007h0o01o0?/0E`0002@0001o0000n@000800003k00000P0007l0n`200?T0
E`0002@000200000m0000840003i00000P000800n@210?@0E`0002@000210000l@000840003d0000
0P000840m0210?40E`0002@000200000k@000840003a00000P000840l@200>d0E`0002@0001o0000
j`000800003]00000P000800k@1o0>/0E`0002@0001n0000jP0007l0003[00000P0007l0j`1n0>X0
E`0002@0001l0000j@0007h0003Z00000P0007h0jP1l0>T0E`0002@0002:0000j@0008d0003Z0000
0P0008d0j@2:0>X0E`0002@000280000jP0008X0003]00000P0008X0jP280>d0E`0002@000270000
k@0008P0003a00000P0008P0k@270?40E`0002@000270000l@0008L0003d00000P0008L0l@270?@0
E`0002@000270000m00008P0003i00000P0008L0m0280?T0E`0002@000280000n@0008X0003l0000
0P0008P0n@2:0?`0E`0002@0002:0000o00008d0003l00000P0008X0o02=0?`0E`0002@0002=0000
o00008h0003l00000P0008d0o02>0?`0E`0002@0002>0000o0000940003l00000P0008h0o02A0?`0
E`0002@0002A0000n@0009<0003l00000P000940o02C0?T0E`0002@0002C0000m00009@0003i0000
0P0009<0n@2D0?@0E`0002@0002D0000l@0009@0003d00000P0009@0m02D0?40E`0002@0002C0000
k@0009@0003a00000P0009@0l@2C0>d0E`0002@0002A0000jP0009<0003]00000P0009<0k@2A0>X0
E`0002@0002>0000j@000940003Z00000P000940jP2>0>T0E`0002@0002=0000j@0008h0003Y0000
0P0008h0j@2=0>T0E`0002@0002;0000j@0008d0003Z00000P0008d0j@2;0>X0E`0002@0002:0000
jP0008/0003[00000P0008/0jP2:0>/0E`0002@000290000j`0008X0003]00000P0008X0j`290>d0
E`0002@000280000k@0008T0003a00000P0008T0k@280?40E`0002@000280000l@0008P0003d0000
0P0008P0l@280?@0E`0002@000280000m00008T0003i00000P0008P0m0290?T0E`0002@000290000
n@0008X0003k00000P0008T0n@2:0?/0E`0002@0002:0000n`0008/0003l00000P0008X0n`2;0?`0
E`0002@0002;0000o00008d0003l00000P0008/0o02=0?`0E`0002@0002>0000o0000900003l0000
0P0008h0o02@0?`0E`0002@0002@0000n`000940003l00000P000900o02A0?/0E`0002@0002A0000
n@000980003k00000P000940n`2B0?T0E`0002@0002B0000m00009<0003i00000P000980n@2C0?@0
E`0002@0002C0000l@0009<0003d00000P0009<0m02C0?40E`0002@0002B0000k@0009<0003a0000
0P0009<0l@2B0>d0E`0002@0002A0000j`000980003]00000P000980k@2A0>/0E`0002@0002@0000
jP000940003[00000P000940j`2@0>X0E`0002@0002>0000j@000900003Z00000P000900jP2>0>T0
E`0002@0001N0000U`000640002H00000P000640U`1N09P0E`0002@0001L0000V00005h0002K0000
0P0005h0V01L09/0E`0002@0001K0000V`0005`0002O00000P0005`0V`1K09l0E`0002@0001K0000
W`0005/0002R00000P0005/0W`1K0:80E`0002@0001K0000XP0005`0002W00000P0005/0XP1L0:L0
E`0002@0001L0000Y`0005h0002Z00000P0005`0Y`1N0:X0E`0002@0001N0000ZP000640002Z0000
0P0005h0ZP1Q0:X0E`0002@0001Q0000ZP000680002Z00000P000640ZP1R0:X0E`0002@0001R0000
ZP0006D0002Z00000P000680ZP1U0:X0E`0002@0001U0000Y`0006L0002Z00000P0006D0ZP1W0:L0
E`0002@0001W0000XP0006P0002W00000P0006L0Y`1X0:80E`0002@0001X0000W`0006P0002R0000
0P0006P0XP1X09l0E`0002@0001W0000V`0006P0002O00000P0006P0W`1W09/0E`0002@0001U0000
V00006L0002K00000P0006L0V`1U09P0E`0002@0001R0000U`0006D0002H00000P0006D0V01R09L0
E`0002@0001Q0000U`000680002G00000P000680U`1Q09L0E`0002@0001O0000U`000640002H0000
0P000640U`1O09P0E`0002@0001N0000V00005l0002I00000P0005l0V01N09T0E`0002@0001M0000
V@0005h0002K00000P0005h0V@1M09/0E`0002@0001L0000V`0005d0002O00000P0005d0V`1L09l0
E`0002@0001L0000W`0005`0002R00000P0005`0W`1L0:80E`0002@0001L0000XP0005d0002W0000
0P0005`0XP1M0:L0E`0002@0001M0000Y`0005h0002Y00000P0005d0Y`1N0:T0E`0002@0001N0000
Z@0005l0002Z00000P0005h0Z@1O0:X0E`0002@0001O0000ZP000640002Z00000P0005l0ZP1Q0:X0
E`0002@0001R0000ZP0006@0002Z00000P000680ZP1T0:X0E`0002@0001T0000Z@0006D0002Z0000
0P0006@0ZP1U0:T0E`0002@0001U0000Y`0006H0002Y00000P0006D0Z@1V0:L0E`0002@0001V0000
XP0006L0002W00000P0006H0Y`1W0:80E`0002@0001W0000W`0006L0002R00000P0006L0XP1W09l0
E`0002@0001V0000V`0006L0002O00000P0006L0W`1V09/0E`0002@0001U0000V@0006H0002K0000
0P0006H0V`1U09T0E`0002@0001T0000V00006D0002I00000P0006D0V@1T09P0E`0002@0001R0000
U`0006@0002H00000P0006@0V01R09L0E`0002@0001]0000Z@0006h0002Z00000P0006h0Z@1]0:X0
E`0002@0001]0000ZP0006h0002Z00000P0006d0ZP1^0:X0E`0002@0001^0000ZP0006l0002Z0000
0P0006h0ZP1_0:X0E`0002@0001^0000Z@0006l0002Z00000P0006l0ZP1^0:T0E`0002@0001h0000
U`0007/0002H00000P0007/0U`1h09P0E`0002@0001f0000V00007P0002K00000P0007P0V01f09/0
E`0002@0001e0000V`0007H0002O00000P0007H0V`1e09l0E`0002@0001e0000W`0007D0002R0000
0P0007D0W`1e0:80E`0002@0001e0000XP0007H0002W00000P0007D0XP1f0:L0E`0002@0001f0000
Y`0007P0002Z00000P0007H0Y`1h0:X0E`0002@0001h0000ZP0007/0002Z00000P0007P0ZP1k0:X0
E`0002@0001k0000ZP0007`0002Z00000P0007/0ZP1l0:X0E`0002@0001l0000ZP0007l0002Z0000
0P0007`0ZP1o0:X0E`0002@0001o0000Y`000840002Z00000P0007l0ZP210:L0E`0002@000210000
XP000880002W00000P000840Y`220:80E`0002@000220000W`000880002R00000P000880XP2209l0
E`0002@000210000V`000880002O00000P000880W`2109/0E`0002@0001o0000V0000840002K0000
0P000840V`1o09P0E`0002@0001l0000U`0007l0002H00000P0007l0V01l09L0E`0002@0001k0000
U`0007`0002G00000P0007`0U`1k09L0E`0002@0001i0000U`0007/0002H00000P0007/0U`1i09P0
E`0002@0001h0000V00007T0002I00000P0007T0V01h09T0E`0002@0001g0000V@0007P0002K0000
0P0007P0V@1g09/0E`0002@0001f0000V`0007L0002O00000P0007L0V`1f09l0E`0002@0001f0000
W`0007H0002R00000P0007H0W`1f0:80E`0002@0001f0000XP0007L0002W00000P0007H0XP1g0:L0
E`0002@0001g0000Y`0007P0002Y00000P0007L0Y`1h0:T0E`0002@0001h0000Z@0007T0002Z0000
0P0007P0Z@1i0:X0E`0002@0001i0000ZP0007/0002Z00000P0007T0ZP1k0:X0E`0002@0001l0000
ZP0007h0002Z00000P0007`0ZP1n0:X0E`0002@0001n0000Z@0007l0002Z00000P0007h0ZP1o0:T0
E`0002@0001o0000Y`000800002Y00000P0007l0Z@200:L0E`0002@000200000XP000840002W0000
0P000800Y`210:80E`0002@000210000W`000840002R00000P000840XP2109l0E`0002@000200000
V`000840002O00000P000840W`2009/0E`0002@0001o0000V@000800002K00000P000800V`1o09T0
E`0002@0001n0000V00007l0002I00000P0007l0V@1n09P0E`0002@0001l0000U`0007h0002H0000
0P0007h0V01l09L0E`0002@0002:0000VP0008`0002K00000P0008X0V`2<09X0E`0002@0002<0000
U`0008h0002J00000P0008`0VP2>09L0E`0002@0002>0000U`0008h0002Z00000P0008h0U`2>0:X0
E`0002@0002=0000V00008d0002Z00000P0008d0V02=0:X0E`0002@0002:0000ZP000980002Z0000
0P0008X0ZP2B0:X0E`0002@0001N0000AP000640001700000P000640AP1N04L0E`0002@0001L0000
A`0005h0001:00000P0005h0A`1L04X0E`0002@0001K0000BP0005`0001>00000P0005`0BP1K04h0
E`0002@0001K0000CP0005/0001A00000P0005/0CP1K0540E`0002@0001K0000D@0005`0001F0000
0P0005/0D@1L05H0E`0002@0001L0000EP0005h0001I00000P0005`0EP1N05T0E`0002@0001N0000
F@000640001I00000P0005h0F@1Q05T0E`0002@0001Q0000F@000680001I00000P000640F@1R05T0
E`0002@0001R0000F@0006D0001I00000P000680F@1U05T0E`0002@0001U0000EP0006L0001I0000
0P0006D0F@1W05H0E`0002@0001W0000D@0006P0001F00000P0006L0EP1X0540E`0002@0001X0000
CP0006P0001A00000P0006P0D@1X04h0E`0002@0001W0000BP0006P0001>00000P0006P0CP1W04X0
E`0002@0001U0000A`0006L0001:00000P0006L0BP1U04L0E`0002@0001R0000AP0006D000170000
0P0006D0A`1R04H0E`0002@0001Q0000AP000680001600000P000680AP1Q04H0E`0002@0001O0000
AP000640001700000P000640AP1O04L0E`0002@0001N0000A`0005l0001800000P0005l0A`1N04P0
E`0002@0001M0000B00005h0001:00000P0005h0B01M04X0E`0002@0001L0000BP0005d0001>0000
0P0005d0BP1L04h0E`0002@0001L0000CP0005`0001A00000P0005`0CP1L0540E`0002@0001L0000
D@0005d0001F00000P0005`0D@1M05H0E`0002@0001M0000EP0005h0001H00000P0005d0EP1N05P0
E`0002@0001N0000F00005l0001I00000P0005h0F01O05T0E`0002@0001O0000F@000640001I0000
0P0005l0F@1Q05T0E`0002@0001R0000F@0006@0001I00000P000680F@1T05T0E`0002@0001T0000
F00006D0001I00000P0006@0F@1U05P0E`0002@0001U0000EP0006H0001H00000P0006D0F01V05H0
E`0002@0001V0000D@0006L0001F00000P0006H0EP1W0540E`0002@0001W0000CP0006L0001A0000
0P0006L0D@1W04h0E`0002@0001V0000BP0006L0001>00000P0006L0CP1V04X0E`0002@0001U0000
B00006H0001:00000P0006H0BP1U04P0E`0002@0001T0000A`0006D0001800000P0006D0B01T04L0
E`0002@0001R0000AP0006@0001700000P0006@0A`1R04H0E`0002@0001]0000F00006h0001I0000
0P0006h0F01]05T0E`0002@0001]0000F@0006h0001I00000P0006d0F@1^05T0E`0002@0001^0000
F@0006l0001I00000P0006h0F@1_05T0E`0002@0001^0000F00006l0001I00000P0006l0F@1^05P0
E`0002@0001h0000AP0007/0001700000P0007/0AP1h04L0E`0002@0001f0000A`0007P0001:0000
0P0007P0A`1f04X0E`0002@0001e0000BP0007H0001>00000P0007H0BP1e04h0E`0002@0001e0000
CP0007D0001A00000P0007D0CP1e0540E`0002@0001e0000D@0007H0001F00000P0007D0D@1f05H0
E`0002@0001f0000EP0007P0001I00000P0007H0EP1h05T0E`0002@0001h0000F@0007/0001I0000
0P0007P0F@1k05T0E`0002@0001k0000F@0007`0001I00000P0007/0F@1l05T0E`0002@0001l0000
F@0007l0001I00000P0007`0F@1o05T0E`0002@0001o0000EP000840001I00000P0007l0F@2105H0
E`0002@000210000D@000880001F00000P000840EP220540E`0002@000220000CP000880001A0000
0P000880D@2204h0E`0002@000210000BP000880001>00000P000880CP2104X0E`0002@0001o0000
A`000840001:00000P000840BP1o04L0E`0002@0001l0000AP0007l0001700000P0007l0A`1l04H0
E`0002@0001k0000AP0007`0001600000P0007`0AP1k04H0E`0002@0001i0000AP0007/000170000
0P0007/0AP1i04L0E`0002@0001h0000A`0007T0001800000P0007T0A`1h04P0E`0002@0001g0000
B00007P0001:00000P0007P0B01g04X0E`0002@0001f0000BP0007L0001>00000P0007L0BP1f04h0
E`0002@0001f0000CP0007H0001A00000P0007H0CP1f0540E`0002@0001f0000D@0007L0001F0000
0P0007H0D@1g05H0E`0002@0001g0000EP0007P0001H00000P0007L0EP1h05P0E`0002@0001h0000
F00007T0001I00000P0007P0F01i05T0E`0002@0001i0000F@0007/0001I00000P0007T0F@1k05T0
E`0002@0001l0000F@0007h0001I00000P0007`0F@1n05T0E`0002@0001n0000F00007l0001I0000
0P0007h0F@1o05P0E`0002@0001o0000EP000800001H00000P0007l0F02005H0E`0002@000200000
D@000840001F00000P000800EP210540E`0002@000210000CP000840001A00000P000840D@2104h0
E`0002@000200000BP000840001>00000P000840CP2004X0E`0002@0001o0000B0000800001:0000
0P000800BP1o04P0E`0002@0001n0000A`0007l0001800000P0007l0B01n04L0E`0002@0001l0000
AP0007h0001700000P0007h0A`1l04H0E`0002@000280000BP0008T0001;00000P0008P0BP2904/0
E`0002@000280000B`0008T0001;00000P0008T0B`2804/0E`0002@000270000B`0008P0001;0000
0P0008P0B`2704/0E`0002@000270000BP0008L0001;00000P0008L0B`2704X0E`0002@000270000
B00008P0001:00000P0008L0BP2804P0E`0002@000280000A`0008T0001800000P0008P0B02904L0
E`0002@000290000AP0008`0001700000P0008T0A`2<04H0E`0002@0002<0000AP0008l000160000
0P0008`0AP2?04H0E`0002@0002?0000AP000980001700000P0008l0AP2B04L0E`0002@0002B0000
A`0009<0001800000P000980A`2C04P0E`0002@0002C0000B00009@0001:00000P0009<0B02D04X0
E`0002@0002D0000BP0009@0001;00000P0009@0BP2D04/0E`0002@0002C0000B`0009@0001=0000
0P0009@0B`2C04d0E`0002@0002@0000C@0009<0001?00000P0009<0C@2@04l0E`0002@0002<0000
C`000900001A00000P000900C`2<0540E`0002@0002:0000D@0008`0001B00000P0008`0D@2:0580
E`0002@000280000DP0008X0001D00000P0008X0DP2805@0E`0002@000270000E00008P0001G0000
0P0008P0E02705L0E`0002@000270000E`0008L0001I00000P0008L0E`2705T0E`0002@0002?0000
AP000940001700000P0008l0AP2A04L0E`0002@0002A0000A`000980001800000P000940A`2B04P0
E`0002@0002B0000B00009<0001:00000P000980B02C04X0E`0002@0002C0000BP0009<0001;0000
0P0009<0BP2C04/0E`0002@0002B0000B`0009<0001=00000P0009<0B`2B04d0E`0002@0002?0000
C@000980001?00000P000980C@2?04l0E`0002@0002<0000C`0008l0001A00000P0008l0C`2<0540
E`0002@000270000E`0008P0001H00000P0008L0F02805L0E`0002@000280000E`0008X0001G0000
0P0008P0E`2:05L0E`0002@0002:0000E`0008h0001I00000P0008X0E`2>05T0E`0002@0002>0000
F@000940001I00000P0008h0F@2A05T0E`0002@0002A0000F00009<0001I00000P000940F@2C05P0
E`0002@0002C0000E`0009@0001H00000P0009<0F02D05L0E`0002@0002:0000E`0008h0001I0000
0P0008X0E`2>05T0E`0002@0002>0000F@000980001I00000P0008h0F@2B05T0E`0002@0002B0000
F@0009<0001I00000P000980F@2C05T0E`0002@0002C0000E`0009@0001I00000P0009<0F@2D05L0
E`0002@0002D0000E@0009@0001G00000P0009@0E`2D05D0E`0002@0002V0000Q@<00:H0002=0`00
0P000:H0Q@>V08d3E`0002@0003N0000Q@<00=h0002=0`000P000=h0Q@?N08d3E`0002@0000G0@00
Q@<001L1002=0`000P0001L1Q@<G0Hd3E`0002@0001?0@00Q@<004l1002=0`000P0004l1Q@=?0Hd3
E`0002@000280@00Q@<008P1002=0`000P0008P1Q@>80Hd3E`0002@000300@00Q@<00<01002=0`00
0P000<01Q@?00Hd3E`0002@0003h0@00Q@<00?P1002=0`000P000?P1Q@?h0Hd3E`0002@0000a0P00
Q@<00342002=0`000P000342Q@<a0Xd3E`0002@0001Y0P00Q@<006T2002=0`000P0006T2Q@=Y0Xd3
E`0002@0002R0P00Q@<00:82002=0`000P000:82Q@>R0Xd3E`0002@0003J0P00Q@<00=X2002=0`00
0P000=X2Q@?J0Xd3E`0002@0000B0`00Q@<00183002=0`000P000183Q@<B0hd3E`0002@0001;0`00
Q@<004/3002=0`000P0004/3Q@=;0hd3E`0002@000230`00Q@<008<3002=0`000P0008<3Q@>30hd3
E`0002@0002k0`00Q@<00;/3002=0`000P000;/3Q@>k0hd3E`0002@0003d0`00Q@<00?@3002=0`00
0P000?@3Q@?d0hd3E`0002@0000/1000Q@<002`4002=0`000P0002`4Q@</18d3E`0002@0001U1000
Q@<006D4002=0`000P0006D4Q@=U18d3E`0002@0002M1000Q@<009d4002=0`000P0009d4Q@>M18d3
E`0002@0003E1000Q@<00=D4002=0`000P000=D4Q@?E18d3E`0002@0002Q0P00b`<00:42003N0`00
0P000:42b`>Q0]h3E`0002@0002Q0P00b`<00:42003N0`000P000:42b`>Q0]h3E`0002@0002J0P00
b`<009/2003@0`000P0009/2b`>J0]03E`0002@0002J0P00b`<009X2003@0`000P0009X2d0>J0//3
E`0002@0002J0P00b`<00:P2003;0`000P0009X2b`>X0//3E`0002@0002X0P00b`<00:P2003@0`00
0P000:P2b`>X0]03E`0002@0002W0P00b`<00:P2003@0`000P000:P2d0>W0//3E`0002@0002N0P00
gP<00:@2003N0`000P0009h2gP>T0]h3E`0002@0002]0P00b`<00:h2003<0`000P000:h2b`>]0/`3
E`0002@0002]0P00c0<00:h2003=0`000P000:d2c0>^0/d3E`0002@0002^0P00c0<00:l2003=0`00
0P000:h2c@>_0/`3E`0002@0002^0P00b`<00:l2003<0`000P000:l2c0>^0//3E`0002@0002^0P00
d@<00:h2003N0`000P000:h2d@>^0]h3E`0002@0002_0P00d@<00:l2003N0`000P000:l2d@>_0]h3
E`0002@0002[0P00d@<00:l2003A0`000P000:/2d@>_0]43E`0002@0002[0P00gP<00;82003N0`00
0P000:/2gP>b0]h3E`0002@0002h0P00d@<00;P2003N0`000P000;P2d@>h0]h3E`0002@0002i0P00
d@<00;T2003N0`000P000;T2d@>i0]h3E`0002@0002i0P00dP<00;/2003D0`000P000;T2e0>k0]83
E`0002@0002k0P00d@<00;d2003B0`000P000;/2dP>m0]43E`0002@0002m0P00d@<00;l2003A0`00
0P000;d2d@>o0]43E`0002@0002o0P00d@<00<82003B0`000P000;l2d@?20]83E`0002@000320P00
dP<00<<2003D0`000P000<82dP?30]@3E`0002@000330P00e0<00<<2003N0`000P000<<2e0?30]h3
E`0002@0002o0P00d@<00<42003B0`000P000;l2d@?10]83E`0002@000310P00dP<00<82003D0`00
0P000<42dP?20]@3E`0002@000320P00e0<00<82003N0`000P000<82e0?20]h3E`0002@000330P00
dP<00<@2003D0`000P000<<2e0?40]83E`0002@000340P00d@<00<L2003B0`000P000<@2dP?70]43
E`0002@000370P00d@<00<T2003A0`000P000<L2d@?90]43E`0002@000390P00d@<00<`2003B0`00
0P000<T2d@?<0]83E`0002@0003<0P00dP<00<d2003D0`000P000<`2dP?=0]@3E`0002@0003=0P00
e0<00<d2003N0`000P000<d2e0?=0]h3E`0002@000390P00d@<00</2003B0`000P000<T2d@?;0]83
E`0002@0003;0P00dP<00<`2003D0`000P000</2dP?<0]@3E`0002@0003<0P00e0<00<`2003N0`00
0P000<`2e0?<0]h3E`0002@0002e0P00d@<00;T2003A0`000P000;D2d@>i0]43E`0002@0002e0P00
gP<00;`2003N0`000P000;D2gP>l0]h3E`0002@0002o0P00gP<00<D2003N0`000P000;l2gP?50]h3
E`0002@000390P00gP<00<l2003N0`000P000<T2gP??0]h3E`0002@0003D0P00e`<00=l2003G0`00
0P000=@2e`?O0]L3E`0002@0003O0P00e@<00=l2003G0`000P000=l2e`?O0]D3E`0002@0003N0P00
d`<00=l2003E0`000P000=l2e@?N0]<3E`0002@0003M0P00dP<00=h2003C0`000P000=h2d`?M0]83
E`0002@0003K0P00d@<00=d2003B0`000P000=d2dP?K0]43E`0002@0003I0P00d@<00=/2003A0`00
0P000=/2d@?I0]43E`0002@0003F0P00d@<00=T2003B0`000P000=T2d@?F0]83E`0002@0003D0P00
dP<00=H2003D0`000P000=H2dP?D0]@3E`0002@0003C0P00e0<00=@2003G0`000P000=@2e0?C0]L3
E`0002@0003C0P00e`<00=<2003I0`000P000=<2e`?C0]T3E`0002@0003C0P00f@<00=@2003L0`00
0P000=<2f@?D0]`3E`0002@0003D0P00g0<00=H2003N0`000P000=@2g0?F0]h3E`0002@0003F0P00
gP<00=T2003N0`000P000=H2gP?I0]h3E`0002@0003I0P00gP<00=X2003N0`000P000=T2gP?J0]h3
E`0002@0003J0P00gP<00=d2003N0`000P000=X2gP?M0]h3E`0002@0003M0P00g0<00=l2003N0`00
0P000=d2gP?O0]`3E`0002@0003N0P00e0<00=h2003G0`000P000=h2e`?N0]@3E`0002@0003M0P00
dP<00=h2003D0`000P000=h2e0?M0]83E`0002@0003G0P00d@<00=T2003B0`000P000=T2d@?G0]83
E`0002@0003E0P00dP<00=L2003D0`000P000=L2dP?E0]@3E`0002@0003D0P00e0<00=D2003G0`00
0P000=D2e0?D0]L3E`0002@0003D0P00e`<00=@2003I0`000P000=@2e`?D0]T3E`0002@0003D0P00
f@<00=D2003L0`000P000=@2f@?E0]`3E`0002@0003E0P00g0<00=L2003N0`000P000=D2g0?G0]h3
E`0002@0003G0P00gP<00=T2003N0`000P000=L2gP?I0]h3E`0002@0002R0000WP<00:@0002O0`00
0P000:80W`>T09h3E`0002@0002T0000V`<00:H0002N0`000P000:@0WP>V09/3E`0002@0002V0000
V`<00:H0002^0`000P000:H0V`>V0:h3E`0002@0002U0000W0<00:D0002^0`000P000:D0W0>U0:h3
E`0002@0002R0000[P<00:X0002^0`000P000:80[P>Z0:h3E`0002@0003I0000W`<00=X0002P0`00
0P000=T0W`?J0:03E`0002@0003I0000X0<00=X0002P0`000P000=X0X0?I0:03E`0002@0003H0000
X0<00=T0002P0`000P000=T0X0?H0:03E`0002@0003H0000W`<00=P0002P0`000P000=P0X0?H09l3
E`0002@0003H0000W@<00=T0002O0`000P000=P0W`?I09d3E`0002@0003I0000W0<00=X0002M0`00
0P000=T0W@?J09`3E`0002@0003J0000V`<00=d0002L0`000P000=X0W0?M09/3E`0002@0003M0000
V`<00>00002K0`000P000=d0V`?P09/3E`0002@0003P0000V`<00><0002L0`000P000>00V`?S09`3
E`0002@0003S0000W0<00>@0002M0`000P000><0W0?T09d3E`0002@0003T0000W@<00>D0002O0`00
0P000>@0W@?U09l3E`0002@0003U0000W`<00>D0002P0`000P000>D0W`?U0:03E`0002@0003T0000
X0<00>D0002R0`000P000>D0X0?T0:83E`0002@0003Q0000XP<00>@0002T0`000P000>@0XP?Q0:@3
E`0002@0003M0000Y0<00>40002V0`000P000>40Y0?M0:H3E`0002@0003K0000YP<00=d0002W0`00
0P000=d0YP?K0:L3E`0002@0003I0000Y`<00=/0002Y0`000P000=/0Y`?I0:T3E`0002@0003H0000
Z@<00=T0002/0`000P000=T0Z@?H0:`3E`0002@0003H0000[0<00=P0002^0`000P000=P0[0?H0:h3
E`0002@0003P0000V`<00>80002L0`000P000>00V`?R09`3E`0002@0003R0000W0<00><0002M0`00
0P000>80W0?S09d3E`0002@0003S0000W@<00>@0002O0`000P000><0W@?T09l3E`0002@0003T0000
W`<00>@0002P0`000P000>@0W`?T0:03E`0002@0003S0000X0<00>@0002R0`000P000>@0X0?S0:83
E`0002@0003P0000XP<00><0002T0`000P000><0XP?P0:@3E`0002@0003M0000Y0<00>00002V0`00
0P000>00Y0?M0:H3E`0002@0003H0000[0<00=T0002]0`000P000=P0[@?I0:`3E`0002@0003I0000
[0<00=/0002/0`000P000=T0[0?K0:`3E`0002@0003K0000[0<00=l0002^0`000P000=/0[0?O0:h3
E`0002@0003O0000[P<00>80002^0`000P000=l0[P?R0:h3E`0002@0003R0000[@<00>@0002^0`00
0P000>80[P?T0:d3E`0002@0003T0000[0<00>D0002]0`000P000>@0[@?U0:`3E`0002@0003K0000
[0<00=l0002^0`000P000=/0[0?O0:h3E`0002@0003O0000[P<00><0002^0`000P000=l0[P?S0:h3
E`0002@0003S0000[P<00>@0002^0`000P000><0[P?T0:h3E`0002@0003T0000[0<00>D0002^0`00
0P000>@0[P?U0:`3E`0002@0003U0000ZP<00>D0002/0`000P000>D0[0?U0:X3E`0002@0000A0@00
W`<00181002P0`000P000141W`<B0J03E`0002@0000A0@00X0<00181002P0`000P000181X0<A0J03
E`0002@0000@0@00X0<00141002P0`000P000141X0<@0J03E`0002@0000@0@00W`<00101002P0`00
0P000101X0<@0Il3E`0002@0000@0@00W@<00141002O0`000P000101W`<A0Id3E`0002@0000A0@00
W0<00181002M0`000P000141W@<B0I`3E`0002@0000B0@00V`<001D1002L0`000P000181W0<E0I/3
E`0002@0000E0@00V`<001P1002K0`000P0001D1V`<H0I/3E`0002@0000H0@00V`<001/1002L0`00
0P0001P1V`<K0I`3E`0002@0000K0@00W0<001`1002N0`000P0001/1W0<L0Ih3E`0002@0000L0@00
WP<001`1002P0`000P0001`1WP<L0J03E`0002@0000K0@00X0<001`1002R0`000P0001`1X0<K0J83
E`0002@0000H0@00XP<001/1002S0`000P0001/1XP<H0J<3E`0002@0000F0@00X`<001P1002S0`00
0P0001P1X`<F0J<3E`0002@0000H0@00V`<001X1002L0`000P0001P1V`<J0I`3E`0002@0000J0@00
W0<001/1002N0`000P0001X1W0<K0Ih3E`0002@0000K0@00WP<001/1002P0`000P0001/1WP<K0J03
E`0002@0000J0@00X0<001/1002R0`000P0001/1X0<J0J83E`0002@0000H0@00XP<001X1002S0`00
0P0001X1XP<H0J<3E`0002@0000H0@00X`<001X1002T0`000P0001P1X`<J0J@3E`0002@0000J0@00
Y0<001`1002V0`000P0001X1Y0<L0JH3E`0002@0000L0@00YP<001d1002X0`000P0001`1YP<M0JP3
E`0002@0000M0@00Z0<001d1002[0`000P0001d1Z0<M0J/3E`0002@0000L0@00Z`<001d1002]0`00
0P0001d1Z`<L0Jd3E`0002@0000K0@00[@<001`1002^0`000P0001`1[@<K0Jh3E`0002@0000H0@00
[P<001/1002^0`000P0001/1[P<H0Jh3E`0002@0000E0@00[P<001P1002^0`000P0001P1[P<E0Jh3
E`0002@0000B0@00[P<001D1002^0`000P0001D1[P<B0Jh3E`0002@0000A0@00[@<00181002^0`00
0P000181[P<A0Jd3E`0002@0000@0@00Z`<00141002]0`000P000141[@<@0J/3E`0002@0000@0@00
ZP<00101002[0`000P000101Z`<@0JX3E`0002@0000@0@00Z@<00141002Z0`000P000101ZP<A0JT3
E`0002@0000A0@00Z@<00181002Z0`000P000141Z@<B0JX3E`0002@0000A0@00ZP<00181002[0`00
0P000181ZP<A0J/3E`0002@0000K0@00Y@<001`1002X0`000P0001/1Y@<L0JP3E`0002@0000L0@00
Z0<001`1002[0`000P0001`1Z0<L0J/3E`0002@0000K0@00Z`<001`1002]0`000P0001`1Z`<K0Jd3
E`0002@0000J0@00[@<001/1002^0`000P0001/1[@<J0Jh3E`0002@0000H0@00[P<001X1002^0`00
0P0001X1[P<H0Jh3E`0002@0001@0@00W@<00501002^0`000P000501W@=@0Jh3E`0002@0001A0@00
V`<00541002^0`000P000541V`=A0Jh3E`0002@000170@00V`<00541002Y0`000P000541V`=70JT3
E`0002@000170@00Z@<005H1002Y0`000P0004L1Z@=F0JT3E`0002@0001>0@00[P<005@1002^0`00
0P0004h1[P=D0Jh3E`0002@000210@00V`<008<1002T0`000P0008<1V`>10J@3E`0002@000210@00
XP<008<1002T0`000P000841Y0>30J83E`0002@000230@00X@<008H1002R0`000P0008<1XP>60J43
E`0002@000260@00X@<008P1002Q0`000P0008H1X@>80J43E`0002@000280@00X@<008/1002R0`00
0P0008P1X@>;0J83E`0002@0002;0@00XP<008d1002T0`000P0008/1XP>=0J@3E`0002@0002=0@00
Y0<008h1002W0`000P0008d1Y0>>0JL3E`0002@0002>0@00Y`<008h1002Y0`000P0008h1Y`>>0JT3
E`0002@0002=0@00Z@<008h1002/0`000P0008h1Z@>=0J`3E`0002@0002;0@00[0<008d1002^0`00
0P0008d1[0>;0Jh3E`0002@000280@00[P<008/1002^0`000P0008/1[P>80Jh3E`0002@000260@00
[P<008P1002^0`000P0008P1[P>60Jh3E`0002@000230@00[P<008H1002^0`000P0008H1[P>30Jh3
E`0002@000220@00[@<008<1002^0`000P0008<1[P>20Jd3E`0002@000210@00Z`<00881002]0`00
0P000881[@>10J/3E`0002@000210@00ZP<00841002[0`000P000841Z`>10JX3E`0002@000210@00
Z@<00881002Z0`000P000841ZP>20JT3E`0002@000220@00Z@<008<1002Z0`000P000881Z@>30JX3
E`0002@000220@00ZP<008<1002[0`000P0008<1ZP>20J/3E`0002@000280@00X@<008X1002R0`00
0P0008P1X@>:0J83E`0002@0002:0@00XP<008`1002T0`000P0008X1XP><0J@3E`0002@0002<0@00
Y0<008d1002W0`000P0008`1Y0>=0JL3E`0002@0002=0@00Y`<008d1002Y0`000P0008d1Y`>=0JT3
E`0002@0002<0@00Z@<008d1002/0`000P0008d1Z@><0J`3E`0002@0002:0@00[0<008`1002^0`00
0P0008`1[0>:0Jh3E`0002@000280@00[P<008X1002^0`000P0008X1[P>80Jh3E`0002@000230@00
V`<008`1002K0`000P0008<1V`><0I/3E`0002@000230@00W0<008L1002L0`000P0008<1W0>70I`3
E`0002@000270@00V`<008`1002L0`000P0008L1W0><0I/3E`0002@000330@00WP<00<@1002O0`00
0P000<@1WP?30Il3E`0002@000330@00W`<00<@1002P0`000P000<<1W`?40J03E`0002@000340@00
W`<00<D1002P0`000P000<@1X0?50Il3E`0002@000350@00WP<00<D1002O0`000P000<D1W`?50Ih3
E`0002@000340@00W0<00<D1002N0`000P000<D1WP?40I`3E`0002@000320@00V`<00<@1002L0`00
0P000<@1W0?20I/3E`0002@0002o0@00V`<00<81002K0`000P000<81V`>o0I/3E`0002@0002m0@00
V`<00;l1002L0`000P000;l1V`>m0I`3E`0002@0002k0@00W0<00;d1002N0`000P000;d1W0>k0Ih3
E`0002@0002j0@00WP<00;/1002P0`000P000;/1WP>j0J03E`0002@0002i0@00X0<00;X1002S0`00
0P000;X1X0>i0J<3E`0002@0002i0@00X`<00;T1002Y0`000P000;T1X`>i0JT3E`0002@0002i0@00
Z@<00;X1002/0`000P000;T1Z@>j0J`3E`0002@0002j0@00[0<00;`1002^0`000P000;X1[0>l0Jh3
E`0002@0002l0@00[P<00;l1002^0`000P000;`1[P>o0Jh3E`0002@0002o0@00[P<00<01002^0`00
0P000;l1[P?00Jh3E`0002@000300@00[P<00<<1002^0`000P000<01[P?30Jh3E`0002@000330@00
[0<00<D1002^0`000P000<<1[P?50J`3E`0002@000350@00Z@<00<H1002/0`000P000<D1[0?60JT3
E`0002@000360@00Z0<00<H1002Y0`000P000<H1Z@?60JP3E`0002@000350@00Y@<00<H1002X0`00
0P000<H1Z0?50JD3E`0002@000330@00X`<00<D1002U0`000P000<D1Y@?30J<3E`0002@000300@00
XP<00<<1002S0`000P000<<1X`?00J83E`0002@0002o0@00XP<00<01002R0`000P000<01XP>o0J83
E`0002@0002m0@00XP<00;l1002S0`000P000;l1XP>m0J<3E`0002@0002k0@00X`<00;d1002U0`00
0P000;d1X`>k0JD3E`0002@0002j0@00Y@<00;/1002X0`000P000;/1Y@>j0JP3E`0002@0002n0@00
V`<00;l1002L0`000P000;l1V`>n0I`3E`0002@0002l0@00W0<00;h1002N0`000P000;h1W0>l0Ih3
E`0002@0002k0@00WP<00;`1002P0`000P000;`1WP>k0J03E`0002@0002j0@00X0<00;/1002S0`00
0P000;/1X0>j0J<3E`0002@0002j0@00X`<00;X1002Y0`000P000;X1X`>j0JT3E`0002@0002j0@00
Z@<00;/1002/0`000P000;X1Z@>k0J`3E`0002@0002k0@00[0<00;d1002^0`000P000;/1[0>m0Jh3
E`0002@0002m0@00[P<00;l1002^0`000P000;d1[P>o0Jh3E`0002@000300@00[P<00<81002^0`00
0P000<01[P?20Jh3E`0002@000320@00[0<00<@1002^0`000P000<81[P?40J`3E`0002@000340@00
Z@<00<D1002/0`000P000<@1[0?50JT3E`0002@000350@00Z0<00<D1002Y0`000P000<D1Z@?50JP3
E`0002@000340@00Y@<00<D1002X0`000P000<D1Z0?40JD3E`0002@000320@00X`<00<@1002U0`00
0P000<@1Y@?20J<3E`0002@000300@00XP<00<81002S0`000P000<81X`?00J83E`0002@0003a0@00
V`<00?41002P0`000P000?41V`?a0J03E`0002@0003a0@00W@<00?81002O0`000P000?41W`?b0Id3
E`0002@0003b0@00V`<00?@1002M0`000P000?81W@?d0I/3E`0002@0003d0@00V`<00?H1002K0`00
0P000?@1V`?f0I/3E`0002@0003f0@00V`<00?X1002N0`000P000?H1V`?j0Ih3E`0002@0003j0@00
WP<00?`1002N0`000P000?X1WP?l0Ih3E`0002@0003l0@00W@<00?d1002N0`000P000?`1WP?m0Id3
E`0002@0003m0@00V`<00?h1002M0`000P000?d1W@?n0I/3E`0002@0003b0@00W0<00?@1002M0`00
0P000?81W@?d0I`3E`0002@0003d0@00W0<00?H1002L0`000P000?@1W0?f0I`3E`0002@0003f0@00
W0<00?X1002N0`000P000?H1W0?j0Ih3E`0002@0003n0@00V`<00?h1002N0`000P000?h1V`?n0Ih3
E`0002@0003m0@00WP<00?h1002P0`000P000?h1WP?m0J03E`0002@0003i0@00X0<00?d1002U0`00
0P000?d1X0?i0JD3E`0002@0003h0@00Y@<00?T1002W0`000P000?T1Y@?h0JL3E`0002@0003g0@00
Y`<00?P1002Z0`000P000?P1Y`?g0JX3E`0002@0003g0@00ZP<00?L1002^0`000P000?L1ZP?g0Jh3
E`0002@0003h0@00X0<00?d1002U0`000P000?d1X0?h0JD3E`0002@0003g0@00Y@<00?P1002W0`00
0P000?P1Y@?g0JL3E`0002@0003g0@00Y`<00?L1002Z0`000P000?L1Y`?g0JX3E`0002@0003g0@00
ZP<00?L1002^0`000P000?L1ZP?g0Jh3E`0002@0000/0P00V`<002l2002L0`000P0002l2V`</0Y`3
E`0002@0000[0P00W0<002`2002N0`000P0002`2W0<[0Yh3E`0002@0000[0P00WP<002/2002P0`00
0P0002/2WP<[0Z03E`0002@0000[0P00X0<002`2002R0`000P0002/2X0</0Z83E`0002@0000/0P00
XP<002l2002S0`000P0002`2XP<_0Z<3E`0002@0000_0P00X`<00382002S0`000P0002l2X`<b0Z<3
E`0002@0000b0P00XP<003D2002S0`000P000382X`<e0Z83E`0002@0000e0P00X0<003H2002R0`00
0P0003D2XP<f0Z03E`0002@0000f0P00WP<003H2002P0`000P0003H2X0<f0Yh3E`0002@0000e0P00
W0<003H2002N0`000P0003H2WP<e0Y`3E`0002@0000b0P00V`<003D2002L0`000P0003D2W0<b0Y/3
E`0002@0000_0P00V`<00382002K0`000P000382V`<_0Y/3E`0002@0000]0P00V`<002l2002L0`00
0P0002l2V`<]0Y`3E`0002@0000/0P00W0<002d2002N0`000P0002d2W0</0Yh3E`0002@0000/0P00
WP<002`2002P0`000P0002`2WP</0Z03E`0002@0000/0P00X0<002d2002R0`000P0002`2X0<]0Z83
E`0002@0000]0P00XP<002l2002S0`000P0002d2XP<_0Z<3E`0002@0000b0P00XP<003@2002S0`00
0P000382X`<d0Z83E`0002@0000d0P00X0<003D2002R0`000P0003@2XP<e0Z03E`0002@0000e0P00
WP<003D2002P0`000P0003D2X0<e0Yh3E`0002@0000d0P00W0<003D2002N0`000P0003D2WP<d0Y`3
E`0002@0000b0P00V`<003@2002L0`000P0003@2W0<b0Y/3E`0002@0000/0P00X`<002l2002T0`00
0P0002l2X`</0Z@3E`0002@0000[0P00Y0<002`2002U0`000P0002`2Y0<[0ZD3E`0002@0000Z0P00
Y@<002/2002W0`000P0002/2Y@<Z0ZL3E`0002@0000Z0P00Y`<002X2002[0`000P0002X2Y`<Z0Z/3
E`0002@0000Z0P00Z`<002/2002]0`000P0002X2Z`<[0Zd3E`0002@0000[0P00[@<002`2002^0`00
0P0002/2[@</0Zh3E`0002@0000/0P00[P<002l2002^0`000P0002`2[P<_0Zh3E`0002@0000_0P00
[P<00382002^0`000P0002l2[P<b0Zh3E`0002@0000b0P00[P<003D2002^0`000P000382[P<e0Zh3
E`0002@0000e0P00[@<003H2002^0`000P0003D2[P<f0Zd3E`0002@0000f0P00Z`<003L2002]0`00
0P0003H2[@<g0Z/3E`0002@0000g0P00Y`<003L2002[0`000P0003L2Z`<g0ZL3E`0002@0000f0P00
Y@<003L2002W0`000P0003L2Y`<f0ZD3E`0002@0000e0P00Y0<003H2002U0`000P0003H2Y@<e0Z@3
E`0002@0000b0P00X`<003D2002T0`000P0003D2Y0<b0Z<3E`0002@0000]0P00X`<002l2002T0`00
0P0002l2X`<]0Z@3E`0002@0000/0P00Y0<002d2002U0`000P0002d2Y0</0ZD3E`0002@0000[0P00
Y@<002`2002W0`000P0002`2Y@<[0ZL3E`0002@0000[0P00Y`<002/2002[0`000P0002/2Y`<[0Z/3
E`0002@0000[0P00Z`<002`2002]0`000P0002/2Z`</0Zd3E`0002@0000/0P00[@<002d2002^0`00
0P0002`2[@<]0Zh3E`0002@0000]0P00[P<002l2002^0`000P0002d2[P<_0Zh3E`0002@0000b0P00
[P<003@2002^0`000P000382[P<d0Zh3E`0002@0000d0P00[@<003D2002^0`000P0003@2[P<e0Zd3
E`0002@0000e0P00Z`<003H2002]0`000P0003D2[@<f0Z/3E`0002@0000f0P00Y`<003H2002[0`00
0P0003H2Z`<f0ZL3E`0002@0000e0P00Y@<003H2002W0`000P0003H2Y`<e0ZD3E`0002@0000d0P00
Y0<003D2002U0`000P0003D2Y@<d0Z@3E`0002@0000b0P00X`<003@2002T0`000P0003@2Y0<b0Z<3
E`0002@0001]0P00X@<006h2002T0`000P0006h2X@=]0Z@3E`0002@0001[0P00Y0<006d2002V0`00
0P0006d2Y0=[0ZH3E`0002@0001X0P00YP<006/2002W0`000P0006/2YP=X0ZL3E`0002@0001X0P00
Y`<006P2002W0`000P0006P2Y`=X0ZL3E`0002@0001U0P00YP<006P2002W0`000P0006P2Y`=U0ZH3
E`0002@0001S0P00Y0<006D2002V0`000P0006D2YP=S0Z@3E`0002@0001R0P00X@<006<2002T0`00
0P0006<2Y0=R0Z43E`0002@0001R0P00X0<00682002Q0`000P000682X@=R0Z03E`0002@0001R0P00
WP<006<2002P0`000P000682X0=S0Yh3E`0002@0001S0P00W0<006D2002N0`000P0006<2WP=U0Y`3
E`0002@0001U0P00V`<006P2002L0`000P0006D2W0=X0Y/3E`0002@0001X0P00V`<006T2002K0`00
0P0006P2V`=Y0Y/3E`0002@0001Y0P00V`<006`2002L0`000P0006T2V`=/0Y`3E`0002@0001/0P00
W0<006h2002N0`000P0006`2W0=^0Yh3E`0002@0001^0P00WP<006l2002P0`000P0006h2WP=_0Z03
E`0002@0001_0P00X0<006l2002V0`000P0006l2X0=_0ZH3E`0002@0001^0P00YP<006l2002Z0`00
0P0006l2YP=^0ZX3E`0002@0001]0P00ZP<006h2002/0`000P0006h2ZP=]0Z`3E`0002@0001[0P00
[0<006d2002^0`000P0006d2[0=[0Zh3E`0002@0001X0P00[P<006/2002^0`000P0006/2[P=X0Zh3
E`0002@0001V0P00[P<006P2002^0`000P0006P2[P=V0Zh3E`0002@0001T0P00[P<006H2002^0`00
0P0006H2[P=T0Zh3E`0002@0001S0P00[0<006@2002^0`000P0006@2[P=S0Z`3E`0002@0001S0P00
Z`<006<2002/0`000P0006<2[0=S0Z/3E`0002@0001S0P00ZP<006@2002[0`000P0006<2Z`=T0ZX3
E`0002@0001T0P00ZP<006D2002[0`000P0006@2ZP=U0Z/3E`0002@0001T0P00Z`<006D2002/0`00
0P0006D2Z`=T0Z`3E`0002@0001V0P00YP<006P2002W0`000P0006P2Y`=V0ZH3E`0002@0001T0P00
Y0<006H2002V0`000P0006H2YP=T0Z@3E`0002@0001S0P00X@<006@2002T0`000P0006@2Y0=S0Z43
E`0002@0001S0P00X0<006<2002Q0`000P0006<2X@=S0Z03E`0002@0001S0P00WP<006@2002P0`00
0P0006<2X0=T0Yh3E`0002@0001T0P00W0<006H2002N0`000P0006@2WP=V0Y`3E`0002@0001V0P00
V`<006P2002L0`000P0006H2W0=X0Y/3E`0002@0001Y0P00V`<006/2002L0`000P0006T2V`=[0Y`3
E`0002@0001[0P00W0<006d2002N0`000P0006/2W0=]0Yh3E`0002@0001]0P00WP<006h2002P0`00
0P0006d2WP=^0Z03E`0002@0001^0P00X0<006h2002V0`000P0006h2X0=^0ZH3E`0002@0001]0P00
YP<006h2002Z0`000P0006h2YP=]0ZX3E`0002@0001/0P00ZP<006d2002/0`000P0006d2ZP=/0Z`3
E`0002@0001Z0P00[0<006`2002^0`000P0006`2[0=Z0Zh3E`0002@0001X0P00[P<006X2002^0`00
0P0006X2[P=X0Zh3E`0002@0002E0P00WP<009L2002O0`000P0009D2W`>G0Yh3E`0002@0002G0P00
V`<009T2002N0`000P0009L2WP>I0Y/3E`0002@0002I0P00V`<009T2002^0`000P0009T2V`>I0Zh3
E`0002@0002H0P00W0<009P2002^0`000P0009P2W0>H0Zh3E`0002@0002E0P00[P<009d2002^0`00
0P0009D2[P>M0Zh3E`0002@0002W0P00V`<00:X2002L0`000P000:X2V`>W0Y`3E`0002@0002U0P00
W0<00:L2002O0`000P000:L2W0>U0Yl3E`0002@0002T0P00W`<00:D2002S0`000P000:D2W`>T0Z<3
E`0002@0002T0P00X`<00:@2002V0`000P000:@2X`>T0ZH3E`0002@0002T0P00YP<00:D2002[0`00
0P000:@2YP>U0Z/3E`0002@0002U0P00Z`<00:L2002^0`000P000:D2Z`>W0Zh3E`0002@0002W0P00
[P<00:X2002^0`000P000:L2[P>Z0Zh3E`0002@0002Z0P00[P<00:/2002^0`000P000:X2[P>[0Zh3
E`0002@0002[0P00[P<00:h2002^0`000P000:/2[P>^0Zh3E`0002@0002^0P00Z`<00;02002^0`00
0P000:h2[P>`0Z/3E`0002@0002`0P00YP<00;42002[0`000P000;02Z`>a0ZH3E`0002@0002a0P00
X`<00;42002V0`000P000;42YP>a0Z<3E`0002@0002`0P00W`<00;42002S0`000P000;42X`>`0Yl3
E`0002@0002^0P00W0<00;02002O0`000P000;02W`>^0Y`3E`0002@0002[0P00V`<00:h2002L0`00
0P000:h2W0>[0Y/3E`0002@0002Z0P00V`<00:/2002K0`000P000:/2V`>Z0Y/3E`0002@0002X0P00
V`<00:X2002L0`000P000:X2V`>X0Y`3E`0002@0002W0P00W0<00:P2002M0`000P000:P2W0>W0Yd3
E`0002@0002V0P00W@<00:L2002O0`000P000:L2W@>V0Yl3E`0002@0002U0P00W`<00:H2002S0`00
0P000:H2W`>U0Z<3E`0002@0002U0P00X`<00:D2002V0`000P000:D2X`>U0ZH3E`0002@0002U0P00
YP<00:H2002[0`000P000:D2YP>V0Z/3E`0002@0002V0P00Z`<00:L2002]0`000P000:H2Z`>W0Zd3
E`0002@0002W0P00[@<00:P2002^0`000P000:L2[@>X0Zh3E`0002@0002X0P00[P<00:X2002^0`00
0P000:P2[P>Z0Zh3E`0002@0002[0P00[P<00:d2002^0`000P000:/2[P>]0Zh3E`0002@0002]0P00
[@<00:h2002^0`000P000:d2[P>^0Zd3E`0002@0002^0P00Z`<00:l2002]0`000P000:h2[@>_0Z/3
E`0002@0002_0P00YP<00;02002[0`000P000:l2Z`>`0ZH3E`0002@0002`0P00X`<00;02002V0`00
0P000;02YP>`0Z<3E`0002@0002_0P00W`<00;02002S0`000P000;02X`>_0Yl3E`0002@0002^0P00
W@<00:l2002O0`000P000:l2W`>^0Yd3E`0002@0002]0P00W0<00:h2002M0`000P000:h2W@>]0Y`3
E`0002@0002[0P00V`<00:d2002L0`000P000:d2W0>[0Y/3E`0002@0003=0P00WP<00<l2002O0`00
0P000<d2W`??0Yh3E`0002@0003?0P00V`<00=42002N0`000P000<l2WP?A0Y/3E`0002@0003A0P00
V`<00=42002^0`000P000=42V`?A0Zh3E`0002@0003@0P00W0<00=02002^0`000P000=02W0?@0Zh3
E`0002@0003=0P00[P<00=D2002^0`000P000<d2[P?E0Zh3E`0002@0003O0P00WP<00>42002O0`00
0P000=l2W`?Q0Yh3E`0002@0003Q0P00V`<00><2002N0`000P000>42WP?S0Y/3E`0002@0003S0P00
V`<00><2002^0`000P000><2V`?S0Zh3E`0002@0003R0P00W0<00>82002^0`000P000>82W0?R0Zh3
E`0002@0003O0P00[P<00>L2002^0`000P000=l2[P?W0Zh3E`0002@000050`00WP<000L3002O0`00
0P0000D3W`<70ih3E`0002@000070`00V`<000T3002N0`000P0000L3WP<90i/3E`0002@000090`00
V`<000T3002^0`000P0000T3V`<90jh3E`0002@000080`00W0<000P3002^0`000P0000P3W0<80jh3
E`0002@000050`00[P<000d3002^0`000P0000D3[P<=0jh3E`0002@0000E0`00W`<001H3002P0`00
0P0001D3W`<F0j03E`0002@0000E0`00X0<001H3002P0`000P0001H3X0<E0j03E`0002@0000D0`00
X0<001D3002P0`000P0001D3X0<D0j03E`0002@0000D0`00W`<001@3002P0`000P0001@3X0<D0il3
E`0002@0000D0`00W@<001D3002O0`000P0001@3W`<E0id3E`0002@0000E0`00W0<001H3002M0`00
0P0001D3W@<F0i`3E`0002@0000F0`00V`<001T3002L0`000P0001H3W0<I0i/3E`0002@0000I0`00
V`<001`3002K0`000P0001T3V`<L0i/3E`0002@0000L0`00V`<001l3002L0`000P0001`3V`<O0i`3
E`0002@0000O0`00W0<00203002M0`000P0001l3W0<P0id3E`0002@0000P0`00W@<00243002O0`00
0P000203W@<Q0il3E`0002@0000Q0`00W`<00243002P0`000P000243W`<Q0j03E`0002@0000P0`00
X0<00243002R0`000P000243X0<P0j83E`0002@0000M0`00XP<00203002T0`000P000203XP<M0j@3
E`0002@0000I0`00Y0<001d3002V0`000P0001d3Y0<I0jH3E`0002@0000G0`00YP<001T3002W0`00
0P0001T3YP<G0jL3E`0002@0000E0`00Y`<001L3002Y0`000P0001L3Y`<E0jT3E`0002@0000D0`00
Z@<001D3002/0`000P0001D3Z@<D0j`3E`0002@0000D0`00[0<001@3002^0`000P0001@3[0<D0jh3
E`0002@0000L0`00V`<001h3002L0`000P0001`3V`<N0i`3E`0002@0000N0`00W0<001l3002M0`00
0P0001h3W0<O0id3E`0002@0000O0`00W@<00203002O0`000P0001l3W@<P0il3E`0002@0000P0`00
W`<00203002P0`000P000203W`<P0j03E`0002@0000O0`00X0<00203002R0`000P000203X0<O0j83
E`0002@0000L0`00XP<001l3002T0`000P0001l3XP<L0j@3E`0002@0000I0`00Y0<001`3002V0`00
0P0001`3Y0<I0jH3E`0002@0000D0`00[0<001D3002]0`000P0001@3[@<E0j`3E`0002@0000E0`00
[0<001L3002/0`000P0001D3[0<G0j`3E`0002@0000G0`00[0<001/3002^0`000P0001L3[0<K0jh3
E`0002@0000K0`00[P<001h3002^0`000P0001/3[P<N0jh3E`0002@0000N0`00[@<00203002^0`00
0P0001h3[P<P0jd3E`0002@0000P0`00[0<00243002]0`000P000203[@<Q0j`3E`0002@0000G0`00
[0<001/3002^0`000P0001L3[0<K0jh3E`0002@0000K0`00[P<001l3002^0`000P0001/3[P<O0jh3
E`0002@0000O0`00[P<00203002^0`000P0001l3[P<P0jh3E`0002@0000P0`00[0<00243002^0`00
0P000203[P<Q0j`3E`0002@0000Q0`00ZP<00243002/0`000P000243[0<Q0jX3E`0002@0000n0`00
WP<00403002O0`000P0003h3W`=00ih3E`0002@000100`00V`<00483002N0`000P000403WP=20i/3
E`0002@000120`00V`<00483002^0`000P000483V`=20jh3E`0002@000110`00W0<00443002^0`00
0P000443W0=10jh3E`0002@0000n0`00[P<004H3002^0`000P0003h3[P=60jh3E`0002@0001>0`00
W`<004l3002P0`000P0004h3W`=?0j03E`0002@0001>0`00X0<004l3002P0`000P0004l3X0=>0j03
E`0002@0001=0`00X0<004h3002P0`000P0004h3X0==0j03E`0002@0001=0`00W`<004d3002P0`00
0P0004d3X0==0il3E`0002@0001=0`00W@<004h3002O0`000P0004d3W`=>0id3E`0002@0001>0`00
W0<004l3002M0`000P0004h3W@=?0i`3E`0002@0001?0`00V`<00583002L0`000P0004l3W0=B0i/3
E`0002@0001B0`00V`<005D3002K0`000P000583V`=E0i/3E`0002@0001E0`00V`<005P3002L0`00
0P0005D3V`=H0i`3E`0002@0001H0`00W0<005T3002N0`000P0005P3W0=I0ih3E`0002@0001I0`00
WP<005T3002P0`000P0005T3WP=I0j03E`0002@0001H0`00X0<005T3002R0`000P0005T3X0=H0j83
E`0002@0001E0`00XP<005P3002S0`000P0005P3XP=E0j<3E`0002@0001C0`00X`<005D3002S0`00
0P0005D3X`=C0j<3E`0002@0001E0`00V`<005L3002L0`000P0005D3V`=G0i`3E`0002@0001G0`00
W0<005P3002N0`000P0005L3W0=H0ih3E`0002@0001H0`00WP<005P3002P0`000P0005P3WP=H0j03
E`0002@0001G0`00X0<005P3002R0`000P0005P3X0=G0j83E`0002@0001E0`00XP<005L3002S0`00
0P0005L3XP=E0j<3E`0002@0001E0`00X`<005L3002T0`000P0005D3X`=G0j@3E`0002@0001G0`00
Y0<005T3002V0`000P0005L3Y0=I0jH3E`0002@0001I0`00YP<005X3002X0`000P0005T3YP=J0jP3
E`0002@0001J0`00Z0<005X3002[0`000P0005X3Z0=J0j/3E`0002@0001I0`00Z`<005X3002]0`00
0P0005X3Z`=I0jd3E`0002@0001H0`00[@<005T3002^0`000P0005T3[@=H0jh3E`0002@0001E0`00
[P<005P3002^0`000P0005P3[P=E0jh3E`0002@0001B0`00[P<005D3002^0`000P0005D3[P=B0jh3
E`0002@0001?0`00[P<00583002^0`000P000583[P=?0jh3E`0002@0001>0`00[@<004l3002^0`00
0P0004l3[P=>0jd3E`0002@0001=0`00Z`<004h3002]0`000P0004h3[@==0j/3E`0002@0001=0`00
ZP<004d3002[0`000P0004d3Z`==0jX3E`0002@0001=0`00Z@<004h3002Z0`000P0004d3ZP=>0jT3
E`0002@0001>0`00Z@<004l3002Z0`000P0004h3Z@=?0jX3E`0002@0001>0`00ZP<004l3002[0`00
0P0004l3ZP=>0j/3E`0002@0001H0`00Y@<005T3002X0`000P0005P3Y@=I0jP3E`0002@0001I0`00
Z0<005T3002[0`000P0005T3Z0=I0j/3E`0002@0001H0`00Z`<005T3002]0`000P0005T3Z`=H0jd3
E`0002@0001G0`00[@<005P3002^0`000P0005P3[@=G0jh3E`0002@0001E0`00[P<005L3002^0`00
0P0005L3[P=E0jh3E`0002@0001f0`00WP<007P3002O0`000P0007H3W`=h0ih3E`0002@0001h0`00
V`<007X3002N0`000P0007P3WP=j0i/3E`0002@0001j0`00V`<007X3002^0`000P0007X3V`=j0jh3
E`0002@0001i0`00W0<007T3002^0`000P0007T3W0=i0jh3E`0002@0001f0`00[P<007h3002^0`00
0P0007H3[P=n0jh3E`0002@0002=0`00W@<008d3002^0`000P0008d3W@>=0jh3E`0002@0002>0`00
V`<008h3002^0`000P0008h3V`>>0jh3E`0002@000240`00V`<008h3002Y0`000P0008h3V`>40jT3
E`0002@000240`00Z@<009<3002Y0`000P0008@3Z@>C0jT3E`0002@0002;0`00[P<00943002^0`00
0P0008/3[P>A0jh3E`0002@0002_0`00WP<00;43002O0`000P000:l3W`>a0ih3E`0002@0002a0`00
V`<00;<3002N0`000P000;43WP>c0i/3E`0002@0002c0`00V`<00;<3002^0`000P000;<3V`>c0jh3
E`0002@0002b0`00W0<00;83002^0`000P000;83W0>b0jh3E`0002@0002_0`00[P<00;L3002^0`00
0P000:l3[P>g0jh3E`0002@0002m0`00V`<00;l3002T0`000P000;l3V`>m0j@3E`0002@0002m0`00
XP<00;l3002T0`000P000;d3Y0>o0j83E`0002@0002o0`00X@<00<83002R0`000P000;l3XP?20j43
E`0002@000320`00X@<00<@3002Q0`000P000<83X@?40j43E`0002@000340`00X@<00<L3002R0`00
0P000<@3X@?70j83E`0002@000370`00XP<00<T3002T0`000P000<L3XP?90j@3E`0002@000390`00
Y0<00<X3002W0`000P000<T3Y0?:0jL3E`0002@0003:0`00Y`<00<X3002Y0`000P000<X3Y`?:0jT3
E`0002@000390`00Z@<00<X3002/0`000P000<X3Z@?90j`3E`0002@000370`00[0<00<T3002^0`00
0P000<T3[0?70jh3E`0002@000340`00[P<00<L3002^0`000P000<L3[P?40jh3E`0002@000320`00
[P<00<@3002^0`000P000<@3[P?20jh3E`0002@0002o0`00[P<00<83002^0`000P000<83[P>o0jh3
E`0002@0002n0`00[@<00;l3002^0`000P000;l3[P>n0jd3E`0002@0002m0`00Z`<00;h3002]0`00
0P000;h3[@>m0j/3E`0002@0002m0`00ZP<00;d3002[0`000P000;d3Z`>m0jX3E`0002@0002m0`00
Z@<00;h3002Z0`000P000;d3ZP>n0jT3E`0002@0002n0`00Z@<00;l3002Z0`000P000;h3Z@>o0jX3
E`0002@0002n0`00ZP<00;l3002[0`000P000;l3ZP>n0j/3E`0002@000340`00X@<00<H3002R0`00
0P000<@3X@?60j83E`0002@000360`00XP<00<P3002T0`000P000<H3XP?80j@3E`0002@000380`00
Y0<00<T3002W0`000P000<P3Y0?90jL3E`0002@000390`00Y`<00<T3002Y0`000P000<T3Y`?90jT3
E`0002@000380`00Z@<00<T3002/0`000P000<T3Z@?80j`3E`0002@000360`00[0<00<P3002^0`00
0P000<P3[0?60jh3E`0002@000340`00[P<00<H3002^0`000P000<H3[P?40jh3E`0002@0002o0`00
V`<00<P3002K0`000P000;l3V`?80i/3E`0002@0002o0`00W0<00<<3002L0`000P000;l3W0?30i`3
E`0002@000330`00V`<00<P3002L0`000P000<<3W0?80i/3E`0002@0003W0`00WP<00>T3002O0`00
0P000>L3W`?Y0ih3E`0002@0003Y0`00V`<00>/3002N0`000P000>T3WP?[0i/3E`0002@0003[0`00
V`<00>/3002^0`000P000>/3V`?[0jh3E`0002@0003Z0`00W0<00>X3002^0`000P000>X3W0?Z0jh3
E`0002@0003W0`00[P<00>l3002^0`000P000>L3[P?_0jh3E`0002@000001000WP<00044002O0`00
0P000044WP<019l3E`0002@000001000W`<00044002P0`000P000004W`<11:03E`0002@000011000
W`<00084002P0`000P000044X0<219l3E`0002@000021000WP<00084002O0`000P000084W`<219h3
E`0002@000011000W0<00084002N0`000P000084WP<119`3E`0002@0003o0`00V`<00044002L0`00
0P000044W0?o0i/3E`0002@0003l0`00V`<00?l3002K0`000P000?l3V`?l0i/3E`0002@0003j0`00
V`<00?`3002L0`000P000?`3V`?j0i`3E`0002@0003h0`00W0<00?X3002N0`000P000?X3W0?h0ih3
E`0002@0003g0`00WP<00?P3002P0`000P000?P3WP?g0j03E`0002@0003f0`00X0<00?L3002S0`00
0P000?L3X0?f0j<3E`0002@0003f0`00X`<00?H3002Y0`000P000?H3X`?f0jT3E`0002@0003f0`00
Z@<00?L3002/0`000P000?H3Z@?g0j`3E`0002@0003g0`00[0<00?T3002^0`000P000?L3[0?i0jh3
E`0002@0003i0`00[P<00?`3002^0`000P000?T3[P?l0jh3E`0002@0003l0`00[P<00?d3002^0`00
0P000?`3[P?m0jh3E`0002@0003m0`00[P<00004002^0`000P000?d3[P<01:h3E`0002@000001000
[0<00084002^0`000P000004[P<21:`3E`0002@000021000Z@<000<4002/0`000P000084[0<31:T3
E`0002@000031000Z0<000<4002Y0`000P0000<4Z@<31:P3E`0002@000021000Y@<000<4002X0`00
0P0000<4Z0<21:D3E`0002@000001000X`<00084002U0`000P000084Y@<01:<3E`0002@0003m0`00
XP<00004002S0`000P000004X`?m0j83E`0002@0003l0`00XP<00?d3002R0`000P000?d3XP?l0j83
E`0002@0003j0`00XP<00?`3002S0`000P000?`3XP?j0j<3E`0002@0003h0`00X`<00?X3002U0`00
0P000?X3X`?h0jD3E`0002@0003g0`00Y@<00?P3002X0`000P000?P3Y@?g0jP3E`0002@0003k0`00
V`<00?`3002L0`000P000?`3V`?k0i`3E`0002@0003i0`00W0<00?/3002N0`000P000?/3W0?i0ih3
E`0002@0003h0`00WP<00?T3002P0`000P000?T3WP?h0j03E`0002@0003g0`00X0<00?P3002S0`00
0P000?P3X0?g0j<3E`0002@0003g0`00X`<00?L3002Y0`000P000?L3X`?g0jT3E`0002@0003g0`00
Z@<00?P3002/0`000P000?L3Z@?h0j`3E`0002@0003h0`00[0<00?X3002^0`000P000?P3[0?j0jh3
E`0002@0003j0`00[P<00?`3002^0`000P000?X3[P?l0jh3E`0002@0003m0`00[P<00?l3002^0`00
0P000?d3[P?o0jh3E`0002@0003o0`00[0<00044002^0`000P000?l3[P<11:`3E`0002@000011000
Z@<00084002/0`000P000044[0<21:T3E`0002@000021000Z0<00084002Y0`000P000084Z@<21:P3
E`0002@000011000Y@<00084002X0`000P000084Z0<11:D3E`0002@0003o0`00X`<00044002U0`00
0P000044Y@?o0j<3E`0002@0003m0`00XP<00?l3002S0`000P000?l3X`?m0j83E`0002@0000O1000
WP<00244002O0`000P0001l4W`<Q19h3E`0002@0000Q1000V`<002<4002N0`000P000244WP<S19/3
E`0002@0000S1000V`<002<4002^0`000P0002<4V`<S1:h3E`0002@0000R1000W0<00284002^0`00
0P000284W0<R1:h3E`0002@0000O1000[P<002L4002^0`000P0001l4[P<W1:h3E`0002@0000^1000
V`<002h4002P0`000P0002h4V`<^1:03E`0002@0000^1000W@<002l4002O0`000P0002h4W`<_19d3
E`0002@0000_1000V`<00344002M0`000P0002l4W@<a19/3E`0002@0000a1000V`<003<4002K0`00
0P000344V`<c19/3E`0002@0000c1000V`<003L4002N0`000P0003<4V`<g19h3E`0002@0000g1000
WP<003T4002N0`000P0003L4WP<i19h3E`0002@0000i1000W@<003X4002N0`000P0003T4WP<j19d3
E`0002@0000j1000V`<003/4002M0`000P0003X4W@<k19/3E`0002@0000_1000W0<00344002M0`00
0P0002l4W@<a19`3E`0002@0000a1000W0<003<4002L0`000P000344W0<c19`3E`0002@0000c1000
W0<003L4002N0`000P0003<4W0<g19h3E`0002@0000k1000V`<003/4002N0`000P0003/4V`<k19h3
E`0002@0000j1000WP<003/4002P0`000P0003/4WP<j1:03E`0002@0000f1000X0<003X4002U0`00
0P0003X4X0<f1:D3E`0002@0000e1000Y@<003H4002W0`000P0003H4Y@<e1:L3E`0002@0000d1000
Y`<003D4002Z0`000P0003D4Y`<d1:X3E`0002@0000d1000ZP<003@4002^0`000P0003@4ZP<d1:h3
E`0002@0000e1000X0<003X4002U0`000P0003X4X0<e1:D3E`0002@0000d1000Y@<003D4002W0`00
0P0003D4Y@<d1:L3E`0002@0000d1000Y`<003@4002Z0`000P0003@4Y`<d1:X3E`0002@0000d1000
ZP<003@4002^0`000P0003@4ZP<d1:h3E`0002@0001H1000WP<005X4002O0`000P0005P4W`=J19h3
E`0002@0001J1000V`<005`4002N0`000P0005X4WP=L19/3E`0002@0001L1000V`<005`4002^0`00
0P0005`4V`=L1:h3E`0002@0001K1000W0<005/4002^0`000P0005/4W0=K1:h3E`0002@0001H1000
[P<00604002^0`000P0005P4[P=P1:h3E`0002@0001Y1000V`<006`4002L0`000P0006`4V`=Y19`3
E`0002@0001X1000W0<006T4002N0`000P0006T4W0=X19h3E`0002@0001X1000WP<006P4002P0`00
0P0006P4WP=X1:03E`0002@0001X1000X0<006T4002R0`000P0006P4X0=Y1:83E`0002@0001Y1000
XP<006`4002S0`000P0006T4XP=/1:<3E`0002@0001/1000X`<006l4002S0`000P0006`4X`=_1:<3
E`0002@0001_1000XP<00784002S0`000P0006l4X`=b1:83E`0002@0001b1000X0<007<4002R0`00
0P000784XP=c1:03E`0002@0001c1000WP<007<4002P0`000P0007<4X0=c19h3E`0002@0001b1000
W0<007<4002N0`000P0007<4WP=b19`3E`0002@0001_1000V`<00784002L0`000P000784W0=_19/3
E`0002@0001/1000V`<006l4002K0`000P0006l4V`=/19/3E`0002@0001Z1000V`<006`4002L0`00
0P0006`4V`=Z19`3E`0002@0001Y1000W0<006X4002N0`000P0006X4W0=Y19h3E`0002@0001Y1000
WP<006T4002P0`000P0006T4WP=Y1:03E`0002@0001Y1000X0<006X4002R0`000P0006T4X0=Z1:83
E`0002@0001Z1000XP<006`4002S0`000P0006X4XP=/1:<3E`0002@0001_1000XP<00744002S0`00
0P0006l4X`=a1:83E`0002@0001a1000X0<00784002R0`000P000744XP=b1:03E`0002@0001b1000
WP<00784002P0`000P000784X0=b19h3E`0002@0001a1000W0<00784002N0`000P000784WP=a19`3
E`0002@0001_1000V`<00744002L0`000P000744W0=_19/3E`0002@0001Y1000X`<006`4002T0`00
0P0006`4X`=Y1:@3E`0002@0001X1000Y0<006T4002U0`000P0006T4Y0=X1:D3E`0002@0001W1000
Y@<006P4002W0`000P0006P4Y@=W1:L3E`0002@0001W1000Y`<006L4002[0`000P0006L4Y`=W1:/3
E`0002@0001W1000Z`<006P4002]0`000P0006L4Z`=X1:d3E`0002@0001X1000[@<006T4002^0`00
0P0006P4[@=Y1:h3E`0002@0001Y1000[P<006`4002^0`000P0006T4[P=/1:h3E`0002@0001/1000
[P<006l4002^0`000P0006`4[P=_1:h3E`0002@0001_1000[P<00784002^0`000P0006l4[P=b1:h3
E`0002@0001b1000[@<007<4002^0`000P000784[P=c1:d3E`0002@0001c1000Z`<007@4002]0`00
0P0007<4[@=d1:/3E`0002@0001d1000Y`<007@4002[0`000P0007@4Z`=d1:L3E`0002@0001c1000
Y@<007@4002W0`000P0007@4Y`=c1:D3E`0002@0001b1000Y0<007<4002U0`000P0007<4Y@=b1:@3
E`0002@0001_1000X`<00784002T0`000P000784Y0=_1:<3E`0002@0001Z1000X`<006`4002T0`00
0P0006`4X`=Z1:@3E`0002@0001Y1000Y0<006X4002U0`000P0006X4Y0=Y1:D3E`0002@0001X1000
Y@<006T4002W0`000P0006T4Y@=X1:L3E`0002@0001X1000Y`<006P4002[0`000P0006P4Y`=X1:/3
E`0002@0001X1000Z`<006T4002]0`000P0006P4Z`=Y1:d3E`0002@0001Y1000[@<006X4002^0`00
0P0006T4[@=Z1:h3E`0002@0001Z1000[P<006`4002^0`000P0006X4[P=/1:h3E`0002@0001_1000
[P<00744002^0`000P0006l4[P=a1:h3E`0002@0001a1000[@<00784002^0`000P000744[P=b1:d3
E`0002@0001b1000Z`<007<4002]0`000P000784[@=c1:/3E`0002@0001c1000Y`<007<4002[0`00
0P0007<4Z`=c1:L3E`0002@0001b1000Y@<007<4002W0`000P0007<4Y`=b1:D3E`0002@0001a1000
Y0<00784002U0`000P000784Y@=a1:@3E`0002@0001_1000X`<00744002T0`000P000744Y0=_1:<3
E`0002@0002@1000WP<00984002O0`000P000904W`>B19h3E`0002@0002B1000V`<009@4002N0`00
0P000984WP>D19/3E`0002@0002D1000V`<009@4002^0`000P0009@4V`>D1:h3E`0002@0002C1000
W0<009<4002^0`000P0009<4W0>C1:h3E`0002@0002@1000[P<009P4002^0`000P000904[P>H1:h3
E`0002@0002Z1000X@<00:/4002T0`000P000:/4X@>Z1:@3E`0002@0002X1000Y0<00:X4002V0`00
0P000:X4Y0>X1:H3E`0002@0002U1000YP<00:P4002W0`000P000:P4YP>U1:L3E`0002@0002U1000
Y`<00:D4002W0`000P000:D4Y`>U1:L3E`0002@0002R1000YP<00:D4002W0`000P000:D4Y`>R1:H3
E`0002@0002P1000Y0<00:84002V0`000P000:84YP>P1:@3E`0002@0002O1000X@<00:04002T0`00
0P000:04Y0>O1:43E`0002@0002O1000X0<009l4002Q0`000P0009l4X@>O1:03E`0002@0002O1000
WP<00:04002P0`000P0009l4X0>P19h3E`0002@0002P1000W0<00:84002N0`000P000:04WP>R19`3
E`0002@0002R1000V`<00:D4002L0`000P000:84W0>U19/3E`0002@0002U1000V`<00:H4002K0`00
0P000:D4V`>V19/3E`0002@0002V1000V`<00:T4002L0`000P000:H4V`>Y19`3E`0002@0002Y1000
W0<00:/4002N0`000P000:T4W0>[19h3E`0002@0002[1000WP<00:`4002P0`000P000:/4WP>/1:03
E`0002@0002/1000X0<00:`4002V0`000P000:`4X0>/1:H3E`0002@0002[1000YP<00:`4002Z0`00
0P000:`4YP>[1:X3E`0002@0002Z1000ZP<00:/4002/0`000P000:/4ZP>Z1:`3E`0002@0002X1000
[0<00:X4002^0`000P000:X4[0>X1:h3E`0002@0002U1000[P<00:P4002^0`000P000:P4[P>U1:h3
E`0002@0002S1000[P<00:D4002^0`000P000:D4[P>S1:h3E`0002@0002Q1000[P<00:<4002^0`00
0P000:<4[P>Q1:h3E`0002@0002P1000[0<00:44002^0`000P000:44[P>P1:`3E`0002@0002P1000
Z`<00:04002/0`000P000:04[0>P1:/3E`0002@0002P1000ZP<00:44002[0`000P000:04Z`>Q1:X3
E`0002@0002Q1000ZP<00:84002[0`000P000:44ZP>R1:/3E`0002@0002Q1000Z`<00:84002/0`00
0P000:84Z`>Q1:`3E`0002@0002S1000YP<00:D4002W0`000P000:D4Y`>S1:H3E`0002@0002Q1000
Y0<00:<4002V0`000P000:<4YP>Q1:@3E`0002@0002P1000X@<00:44002T0`000P000:44Y0>P1:43
E`0002@0002P1000X0<00:04002Q0`000P000:04X@>P1:03E`0002@0002P1000WP<00:44002P0`00
0P000:04X0>Q19h3E`0002@0002Q1000W0<00:<4002N0`000P000:44WP>S19`3E`0002@0002S1000
V`<00:D4002L0`000P000:<4W0>U19/3E`0002@0002V1000V`<00:P4002L0`000P000:H4V`>X19`3
E`0002@0002X1000W0<00:X4002N0`000P000:P4W0>Z19h3E`0002@0002Z1000WP<00:/4002P0`00
0P000:X4WP>[1:03E`0002@0002[1000X0<00:/4002V0`000P000:/4X0>[1:H3E`0002@0002Z1000
YP<00:/4002Z0`000P000:/4YP>Z1:X3E`0002@0002Y1000ZP<00:X4002/0`000P000:X4ZP>Y1:`3
E`0002@0002W1000[0<00:T4002^0`000P000:T4[0>W1:h3E`0002@0002U1000[P<00:L4002^0`00
0P000:L4[P>U1:h3E`0002@000361000W`<00<L4002P0`000P000<H4W`?71:03E`0002@000361000
X0<00<L4002P0`000P000<L4X0?61:03E`0002@000351000X0<00<H4002P0`000P000<H4X0?51:03
E`0002@000351000W`<00<D4002P0`000P000<D4X0?519l3E`0002@000351000W@<00<H4002O0`00
0P000<D4W`?619d3E`0002@000361000W0<00<L4002M0`000P000<H4W@?719`3E`0002@000371000
V`<00<X4002L0`000P000<L4W0?:19/3E`0002@0003:1000V`<00<d4002K0`000P000<X4V`?=19/3
E`0002@0003=1000V`<00=04002L0`000P000<d4V`?@19`3E`0002@0003@1000W0<00=44002M0`00
0P000=04W0?A19d3E`0002@0003A1000W@<00=84002O0`000P000=44W@?B19l3E`0002@0003B1000
W`<00=84002P0`000P000=84W`?B1:03E`0002@0003A1000X0<00=84002R0`000P000=84X0?A1:83
E`0002@0003>1000XP<00=44002T0`000P000=44XP?>1:@3E`0002@0003:1000Y0<00<h4002V0`00
0P000<h4Y0?:1:H3E`0002@000381000YP<00<X4002W0`000P000<X4YP?81:L3E`0002@000361000
Y`<00<P4002Y0`000P000<P4Y`?61:T3E`0002@000351000Z@<00<H4002/0`000P000<H4Z@?51:`3
E`0002@000351000[0<00<D4002^0`000P000<D4[0?51:h3E`0002@0003=1000V`<00<l4002L0`00
0P000<d4V`??19`3E`0002@0003?1000W0<00=04002M0`000P000<l4W0?@19d3E`0002@0003@1000
W@<00=44002O0`000P000=04W@?A19l3E`0002@0003A1000W`<00=44002P0`000P000=44W`?A1:03
E`0002@0003@1000X0<00=44002R0`000P000=44X0?@1:83E`0002@0003=1000XP<00=04002T0`00
0P000=04XP?=1:@3E`0002@0003:1000Y0<00<d4002V0`000P000<d4Y0?:1:H3E`0002@000351000
[0<00<H4002]0`000P000<D4[@?61:`3E`0002@000361000[0<00<P4002/0`000P000<H4[0?81:`3
E`0002@000381000[0<00<`4002^0`000P000<P4[0?<1:h3E`0002@0003<1000[P<00<l4002^0`00
0P000<`4[P??1:h3E`0002@0003?1000[@<00=44002^0`000P000<l4[P?A1:d3E`0002@0003A1000
[0<00=84002]0`000P000=44[@?B1:`3E`0002@000381000[0<00<`4002^0`000P000<P4[0?<1:h3
E`0002@0003<1000[P<00=04002^0`000P000<`4[P?@1:h3E`0002@0003@1000[P<00=44002^0`00
0P000=04[P?A1:h3E`0002@0003A1000[0<00=84002^0`000P000=44[P?B1:`3E`0002@0003B1000
ZP<00=84002/0`000P000=84[0?B1:X3E`0002@0003J1000V`<00=d4002L0`000P000=d4V`?J19`3
E`0002@0003H1000W0<00=X4002O0`000P000=X4W0?H19l3E`0002@0003G1000W`<00=P4002S0`00
0P000=P4W`?G1:<3E`0002@0003G1000X`<00=L4002V0`000P000=L4X`?G1:H3E`0002@0003G1000
YP<00=P4002[0`000P000=L4YP?H1:/3E`0002@0003H1000Z`<00=X4002^0`000P000=P4Z`?J1:h3
E`0002@0003J1000[P<00=d4002^0`000P000=X4[P?M1:h3E`0002@0003M1000[P<00=h4002^0`00
0P000=d4[P?N1:h3E`0002@0003N1000[P<00>44002^0`000P000=h4[P?Q1:h3E`0002@0003Q1000
Z`<00><4002^0`000P000>44[P?S1:/3E`0002@0003S1000YP<00>@4002[0`000P000><4Z`?T1:H3
E`0002@0003T1000X`<00>@4002V0`000P000>@4YP?T1:<3E`0002@0003S1000W`<00>@4002S0`00
0P000>@4X`?S19l3E`0002@0003Q1000W0<00><4002O0`000P000><4W`?Q19`3E`0002@0003N1000
V`<00>44002L0`000P000>44W0?N19/3E`0002@0003M1000V`<00=h4002K0`000P000=h4V`?M19/3
E`0002@0003K1000V`<00=d4002L0`000P000=d4V`?K19`3E`0002@0003J1000W0<00=/4002M0`00
0P000=/4W0?J19d3E`0002@0003I1000W@<00=X4002O0`000P000=X4W@?I19l3E`0002@0003H1000
W`<00=T4002S0`000P000=T4W`?H1:<3E`0002@0003H1000X`<00=P4002V0`000P000=P4X`?H1:H3
E`0002@0003H1000YP<00=T4002[0`000P000=P4YP?I1:/3E`0002@0003I1000Z`<00=X4002]0`00
0P000=T4Z`?J1:d3E`0002@0003J1000[@<00=/4002^0`000P000=X4[@?K1:h3E`0002@0003K1000
[P<00=d4002^0`000P000=/4[P?M1:h3E`0002@0003N1000[P<00>04002^0`000P000=h4[P?P1:h3
E`0002@0003P1000[@<00>44002^0`000P000>04[P?Q1:d3E`0002@0003Q1000Z`<00>84002]0`00
0P000>44[@?R1:/3E`0002@0003R1000YP<00><4002[0`000P000>84Z`?S1:H3E`0002@0003S1000
X`<00><4002V0`000P000><4YP?S1:<3E`0002@0003R1000W`<00><4002S0`000P000><4X`?R19l3
E`0002@0003Q1000W@<00>84002O0`000P000>84W`?Q19d3E`0002@0003P1000W0<00>44002M0`00
0P000>44W@?P19`3E`0002@0003N1000V`<00>04002L0`000P000>04W0?N19/39`0001P000030000
00000000000000009@0000`000030000:00000`000010000F`0007`0002R0000H`400:X0001/0@00
0@0001H0000F0000ZP1W0JX0IP6Y06D1Z@1T0JL0I06W06<1Y@1S0J@0I06S06@1X`1U0J80I@6R06X1
X`1Z0J<0J`6T06/1Y01/0JL0K06X06/1Z@1[0JT0JP6Z06T1ZP1W0E/0001`0000fP0009X0003R0000
XP000040000C00004`000>80WP3R09`0h@2K0>00V`3P09X0g@2J0=`0V`3K09/0f`2L0=X0W@3J0:00
f`2P0=/0X@3L0:80h@2R0>40X@3R0:00hP2O0>80WP1K0000L00001<1001N0@006`4006L100010000
4`0001<0000K0F816P5Q0AX1H04I0El1605O0AP1GP4F0Eh15@5O0A@1G`4C0F014`5U0A@1IP4E0FL1
605W0AT1IP4J0FD16P5T0A/1I04K0F81F`000800001;0@00I`4005<1001`0@000@0001L0000G0000
D`5[0E<1J@5B0FT1DP5X0E41J05A0FL1C@5W0Dd1J05<0FP1C05Y0D/1J@5;0Fh1C05^0D`1K`5=0Fl1
CP5`0E01L05A0Fl1DP5_0E81KP5C0Fh1D`5/0E<1J`5K0000L00008<1003l0000R`4000@100010000
4`0001<0002;0@01R`7m08X1o@2:0O`0Q@7l08D1o@240Od0Q07o08<1o`230@41Q0420H@10`650@<1
Q@440HX1106:0@<1R`430H/10@6;0@01F`000700002l0@00e0400<@1003L0@000@0001<0000C0000
a07H0L@1e`730MH1``7E0L81e@720M@1_P7D0Kd1e@6l0MH1_07J0Kd1f`6n0M`1`P7L0L81f`730M/1
``7J0L@1fP740MT1a07H0E/000200000m04001P2003l0@008@800040000G00005`000?`170;l0AX2
n`4J0_/16@;j0AT2n@4H0_L160;g0AT2m@4I0_D16P;d0A/2m04N0_D17`;e0B02mP4P0_L18@;j0B42
nP4P0_/180;k0Al2o04O0_`17@;l0A`2F`000700000/0P00^08003D200300P000@0001<0000C0000
=@:l0SD2^`8d0[X2=0:i0S<2^08^0[P2;@:i0Rd2^`8/0[`2;@:m0Rd2_`8^0[l2;`;00S82`08c0[l2
=0:o0S@2_@8e0[d2=@:l0U/000200000I@800?h1001]0P001`800040000G00005`0006d20P9]0P42
K0800V`2o`5[0_l1JP;n0FL2oP5W0_l1IP;o0FH2009U0P02I@850VH21@9V0PH2I`860VP21`9Z0PL2
JP860V`21P9/0PD2K@840Vd20`9]0P82F`0007`0002M0P00``400:D2003<0@000@0001H0000F0000
Y@;70JD2a@6T0/D1Y0;40J<2a06R0/<1X0;30J02a06O0/@1WP;50Id2aP6M0/T1WP;:0Il2b`6P0/`1
X`;<0J<2b`6T0//1Y0;:0JD2bP6U0/T1Y@;70E/0001d0000eP8007`2003N0P00Q@800040000D0000
50000=h2P0;N0Wl2g@9o0]d2OP;L0Wd2f`9l0]P2O0;H0Wd2e`9m0]H2OP;F0X<2e`:40]T2Q0;J0XD2
f`:40]`2Q0;M0X<2g@:20]h2P@;N0X02F`000600000>0`00DP8001H3001J0P000@0000l0000?0000
5P=F0QH3E08E0e<250=B0Q03DP8?0e<23P=D0Ph3F08?0eT240=J0Q@3FP8E0eT25P=H0QH3E`8F0eH2
F`00070000160`00]04004h3002l0@000@0001<0000C0000CP>h0Dh3]@5=0kD1C@>d0DP3]0580kD1
A`>e0DL3]P560kL1AP>i0DL3^P570k/1B0>k0DP3_05=0k`1C@>k0Dh3^`5>0kT1CP>h0E/0001h0000
O`<00:@100270`00[@400040000E00005@0008L3Z0670jL1QP>V0HD3Y@640jD1Q0>T0H83Y0610jD1
P0>U0H03YP5o0jH1O`>[0H03Z`600j`1P@>/0H43[@640jd1Q@>/0HH3Z`670jX1Q`>X0E/0001h0000
]`<005h2002o0`00I`800040000E00005@000;l3HP:o0f02_P=O0[d3G`:m0eh2^P=N0[T3G`:h0el2
^0=P0[L3H@:g0f@2^0=U0[P3IP:j0fH2^`=W0[`3IP:n0fH2_P=U0[l3I@:o0f<2_`=R0U/0001/0000
l0<003l2003h0`00A`800040000B00004P000?P3@`;h0d82m`=10_L3@0;f0d02mP<o0_43?`;a0d02
l0=00_03AP;a0dH2l@=70_H3A`;f0dH2m`=60_L3A@;h0d@2n0=30U/0001l0000:0@00<L2000`1000
d0800040000F00005P000304b`8`1<X2;`C90Rl4b08^1<P2;@C70RX4a`8Z1<P2:@C80RT4b@8X1<T2
:0C>0RT4cP8Y1<l2:`C?0R`4d08]1<l2;`C?0Rl4cP8`1<d2<0C<0S04b`9K0000N0000604002A0P00
J0@009X2000100005@0001D0001X19D2J0BC0VL4TP9V1982IPBA0V<4T@9R1982H@BB0V44T`9P19@2
H0BG0V44V09R19T2H`BI0VD4VP9U19T2I`BI0VL4V09X19P2J0BF0VP4U@9K0000O00009T400120`00
X@@004/3000100005P0001H0002Q14H3X@A50j04A0>O14<3WPA20i/4@P>K14<3VPA30iX4A0>I14@3
V@A90iX4B@>J14X3V`A:0i`4B`>N14/3WPA:0il4BP>P14T3X@A80j44A`>Q14H3F`0006P0003A1000
b`800=T4003C0P000@000140000A0000f@C?0]T4c@;H1<`2e`C;0]<4b`;B1<`2dPC=0]44cP;A1=42
dPCA0]84dP;C1=<2e`CC0]P4dP;I1=42f@C@0]T4c`9G0000L0000:H0002N0000e@@004H3000E0000
YP1W0Mh0WP0G0F81C`5[0HP100700MP1n04L0S42_09Y0P82XP;70MX2P08B0eH2B`>h0H<3Z06k0f82
m0=30R`4b`9U19D2W@A60mD4c`;E1=02E`0003@0002Q0000B0000=X400250`001P000:40Q0>Q04P0
fPA80=X4Q0>Q08@3X@250bD0000<00001@00P2P0000<00000`0002D0000<00002000P2P0000<0000
0P0002D0000<00003@00P0h0000D000000000100000D0000
\>"], "Graphics",
 ImageSize->{251.25, 198.938},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}}],

Cell["\<\
Below are the actual and average simulated trade directions. Actuals are \
indicated by a dot; estimated are indicated by the bars.\
\>", "Text"],

Cell[GraphicsData["Metafile", "\<\
CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@000@f@0@0006`00000000000000?D4003/0`00
000000000037FP00idL00215CDH00040T0d10587000300000000000000000000@0H00;04000U0@00
g000000000000000000008Qh101PF`<02@000100003f1000k@<000X0000@000000000000000D0000
300000d0000U0000300000P0080W00006000004000000000oooo0000000U000030000040000[0000
6000000000000000mP@00>d3000[00006000000000000000mP@00>d3000V00007000008000000000
0@000000000000009@0000`000020000E`0002@000060@002P0000`1000M00000P0000`12P060Ad0
E`0002@0000<0@002P0001<1000M00000P0000`12P0C0Ad0E`0002@0000<0@003@000181000M0000
0P0000`13@0B0Ad0E`0002@000080@0060000101000H00000P0000P1600@0AP0E`0002@000040@00
7@0000X1000M00000P0000@17@0:0Ad0E`0002@0000?0@007@0001@1000M00000P0000l17@0D0Ad0
E`0002@0000R0@004`0002<1000D00000P0002<14`0R0A@0E`0002@0000R0@00500002<1000E0000
0P000281500S0AD0E`0002@0000S0@00500002@1000E00000P0002<15@0T0A@0E`0002@0000T0@00
4`0002@1000D00000P0002@1500T0A<0E`0002@0000R0@004@0002@1000C00000P0002@14`0R0A40
E`0002@0000P0@0040000281000A00000P0002814@0P0A00E`0002@0000N0@0040000201000@0000
0P000201400N0A00E`0002@0000K0@00400001h1000A00000P0001h1400K0A40E`0002@0000I0@00
4@0001/1000C00000P0001/14@0I0A<0E`0002@0000H0@004`0001T1000F00000P0001T14`0H0AH0
E`0002@0000H0@005P0001P1000H00000P0001P15P0H0AP0E`0002@0000H0@00600001T1000K0000
0P0001P1600I0A/0E`0002@0000I0@006`0001/1000M00000P0001T16`0K0Ad0E`0002@0000K0@00
7@0001h1000M00000P0001/17@0N0Ad0E`0002@0000N0@007@0001l1000M00000P0001h17@0O0Ad0
E`0002@0000O0@007@000281000M00000P0001l17@0R0Ad0E`0002@0000R0@006`0002@1000M0000
0P0002817@0T0A/0E`0002@0000L0@00400001h1000A00000P0001h1400L0A40E`0002@0000J0@00
4@0001`1000C00000P0001`14@0J0A<0E`0002@0000I0@004`0001X1000F00000P0001X14`0I0AH0
E`0002@0000I0@005P0001T1000H00000P0001T15P0I0AP0E`0002@0000I0@00600001X1000K0000
0P0001T1600J0A/0E`0002@0000J0@006`0001`1000M00000P0001X16`0L0Ad0E`0002@0000L0@00
7@0001h1000M00000P0001`17@0N0Ad0E`0002@0000[0@002P0002/1000J00000P0002/12P0[0AX0
E`0002@0000[0@006P0002`1000M00000P0002/16P0/0Ad0E`0002@0000/0@007@0002h1000M0000
0P0002`17@0^0Ad0E`0002@0000^0@007@0002l1000M00000P0002h17@0_0Ad0E`0002@0000_0@00
7@000341000M00000P0002l17@0a0Ad0E`0002@0000a0@006`000381000M00000P0003417@0b0A/0
E`0002@0000/0@002P0002`1000J00000P0002`12P0/0AX0E`0002@0000/0@006P0002d1000M0000
0P0002`16P0]0Ad0E`0002@0000]0@007@0002h1000M00000P0002d17@0^0Ad0E`0002@0000X0@00
400002l1000@00000P0002P1400_0A00E`0002@0000h0@00400003P1000K00000P0003P1400h0A/0
E`0002@0000h0@006`0003T1000M00000P0003P16`0i0Ad0E`0002@0000i0@007@0003`1000M0000
0P0003T17@0l0Ad0E`0002@0000l0@007@0003d1000M00000P0003`17@0m0Ad0E`0002@0000m0@00
7@000401000M00000P0003d17@100Ad0E`0002@000100@006`000481000M00000P0004017@120A/0
E`0002@0000i0@00400003T1000K00000P0003T1400i0A/0E`0002@0000i0@006`0003X1000M0000
0P0003T16`0j0Ad0E`0002@0000j0@007@0003`1000M00000P0003X17@0l0Ad0E`0002@000120@00
40000481000M00000P00048140120Ad0E`0002@000130@00400004<1000M00000P0004<140130Ad0
E`0002@0000e0@00400003T1000@00000P0003D1400i0A00E`0002@0000o0@00400004<1000@0000
0P0003l140130A00E`0002@000120@007@0004D1000M00000P0004817@150Ad0E`0002@0001<0@00
4P0004`1000C00000P0004`14P1<0A<0E`0002@0001;0@004`0004`1000C00000P0004`14`1;0A<0
E`0002@0001;0@004P0004/1000C00000P0004/14`1;0A80E`0002@0001;0@004@0004`1000B0000
0P0004/14P1<0A40E`0002@0001<0@00400004h1000A00000P0004`14@1>0A00E`0002@0001>0@00
40000541000@00000P0004h1401A0A00E`0002@0001A0@00400005<1000A00000P000541401C0A40
E`0002@0001C0@004@0005@1000B00000P0005<14@1D0A80E`0002@0001D0@004P0005D1000D0000
0P0005@14P1E0A@0E`0002@0001E0@00500005D1000K00000P0005D1501E0A/0E`0002@0001E0@00
6`0005H1000M00000P0005D16`1F0Ad0E`0002@0001F0@007@0005L1000M00000P0005H17@1G0Ad0
E`0002@0001D0@004P0005@1000K00000P0005@14P1D0A/0E`0002@0001D0@006`0005D1000M0000
0P0005@16`1E0Ad0E`0002@0001E0@007@0005L1000M00000P0005D17@1G0Ad0E`0002@0001G0@00
7@0005P1000M00000P0005L17@1H0Ad0E`0002@0001C0@00500005@1000E00000P0005@1501C0AD0
E`0002@0001>0@005@0005<1000F00000P0005<15@1>0AH0E`0002@0001;0@005P0004h1000G0000
0P0004h15P1;0AL0E`0002@0001:0@005`0004/1000I00000P0004/15`1:0AT0E`0002@0001:0@00
6@0004X1000K00000P0004X16@1:0A/0E`0002@0001:0@006`0004/1000M00000P0004X16`1;0Ad0
E`0002@0001;0@007@0004h1000M00000P0004/17@1>0Ad0E`0002@0001>0@007@000501000M0000
0P0004h17@1@0Ad0E`0002@0001@0@007@000581000M00000P0005017@1B0Ad0E`0002@0001B0@00
6`0005@1000M00000P0005817@1D0A/0E`0002@0001<0@005P0004h1000G00000P0004h15P1<0AL0
E`0002@0001;0@005`0004`1000I00000P0004`15`1;0AT0E`0002@0001;0@006@0004/1000K0000
0P0004/16@1;0A/0E`0002@0001;0@006`0004`1000M00000P0004/16`1<0Ad0E`0002@0001<0@00
7@0004h1000M00000P0004`17@1>0Ad0E`0002@0001M0@002P0005d1000M00000P0005d12P1M0Ad0
E`0002@0001N0@002P0005h1000M00000P0005h12P1N0Ad0E`0002@0001J0@002P0005h1000:0000
0P0005X12P1N0@X0E`0002@0001J0@007@000641000M00000P0005X17@1Q0Ad0E`0002@0001a0@00
4P000741000C00000P0007414P1a0A<0E`0002@0001`0@004`000741000C00000P0007414`1`0A<0
E`0002@0001`0@004P000701000C00000P0007014`1`0A80E`0002@0001`0@004@000741000B0000
0P0007014P1a0A40E`0002@0001a0@00400007<1000A00000P0007414@1c0A00E`0002@0001c0@00
400007H1000@00000P0007<1401f0A00E`0002@0001f0@00400007P1000A00000P0007H1401h0A40
E`0002@0001h0@004@0007T1000B00000P0007P14@1i0A80E`0002@0001i0@004P0007X1000D0000
0P0007T14P1j0A@0E`0002@0001j0@00500007X1000K00000P0007X1501j0A/0E`0002@0001j0@00
6`0007/1000M00000P0007X16`1k0Ad0E`0002@0001k0@007@0007`1000M00000P0007/17@1l0Ad0
E`0002@0001i0@004P0007T1000K00000P0007T14P1i0A/0E`0002@0001i0@006`0007X1000M0000
0P0007T16`1j0Ad0E`0002@0001j0@007@0007`1000M00000P0007X17@1l0Ad0E`0002@0001l0@00
7@0007d1000M00000P0007`17@1m0Ad0E`0002@0001h0@00500007T1000E00000P0007T1501h0AD0
E`0002@0001c0@005@0007P1000F00000P0007P15@1c0AH0E`0002@0001`0@005P0007<1000G0000
0P0007<15P1`0AL0E`0002@0001_0@005`000701000I00000P0007015`1_0AT0E`0002@0001_0@00
6@0006l1000K00000P0006l16@1_0A/0E`0002@0001_0@006`000701000M00000P0006l16`1`0Ad0
E`0002@0001`0@007@0007<1000M00000P0007017@1c0Ad0E`0002@0001c0@007@0007D1000M0000
0P0007<17@1e0Ad0E`0002@0001e0@007@0007L1000M00000P0007D17@1g0Ad0E`0002@0001g0@00
6`0007T1000M00000P0007L17@1i0A/0E`0002@0001a0@005P0007<1000G00000P0007<15P1a0AL0
E`0002@0001`0@005`000741000I00000P0007415`1`0AT0E`0002@0001`0@006@000701000K0000
0P0007016@1`0A/0E`0002@0001`0@006`000741000M00000P0007016`1a0Ad0E`0002@0001a0@00
7@0007<1000M00000P0007417@1c0Ad0E`0002@000230@00400008<1000M00000P0008<140230Ad0
E`0002@000240@00400008@1000M00000P0008@140240Ad0E`0002@000240@004@0008H1000C0000
0P0008@14`260A40E`0002@000260@00400008P1000A00000P0008H14@280A00E`0002@000280@00
400008X1000@00000P0008P1402:0A00E`0002@0002:0@00400008d1000A00000P0008X1402=0A40
E`0002@0002=0@004@0008h1000C00000P0008d14@2>0A<0E`0002@0002>0@004`0008h1000M0000
0P0008h14`2>0Ad0E`0002@0002:0@00400008`1000A00000P0008X1402<0A40E`0002@0002<0@00
4@0008d1000C00000P0008`14@2=0A<0E`0002@0002=0@004`0008d1000M00000P0008d14`2=0Ad0
E`0002@000200@00400008@1000@00000P00080140240A00E`0002@000200@007@0008L1000M0000
0P0008017@270Ad0E`0002@0002:0@007@000901000M00000P0008X17@2@0Ad0E`0002@0002P0@00
2P000:01000M00000P000:012P2P0Ad0E`0002@0002Q0@002P000:41000M00000P000:412P2Q0Ad0
E`0002@0002N0@004@000:01000C00000P000:014`2N0A40E`0002@0002L0@00400009h1000A0000
0P0009h14@2L0A00E`0002@0002K0@00400009`1000@00000P0009`1402K0A00E`0002@0002H0@00
400009/1000A00000P0009/1402H0A40E`0002@0002F0@004@0009P1000C00000P0009P14@2F0A<0
E`0002@0002E0@004`0009H1000F00000P0009H14`2E0AH0E`0002@0002E0@005P0009D1000H0000
0P0009D15P2E0AP0E`0002@0002E0@00600009H1000K00000P0009D1602F0A/0E`0002@0002F0@00
6`0009P1000M00000P0009H16`2H0Ad0E`0002@0002H0@007@0009/1000M00000P0009P17@2K0Ad0
E`0002@0002K0@007@0009`1000M00000P0009/17@2L0Ad0E`0002@0002L0@007@0009h1000M0000
0P0009`17@2N0Ad0E`0002@0002N0@006`000:01000M00000P0009h17@2P0A/0E`0002@0002I0@00
400009/1000A00000P0009/1402I0A40E`0002@0002G0@004@0009T1000C00000P0009T14@2G0A<0
E`0002@0002F0@004`0009L1000F00000P0009L14`2F0AH0E`0002@0002F0@005P0009H1000H0000
0P0009H15P2F0AP0E`0002@0002F0@00600009L1000K00000P0009H1602G0A/0E`0002@0002G0@00
6`0009T1000M00000P0009L16`2I0Ad0E`0002@0002I0@007@0009/1000M00000P0009T17@2K0Ad0
E`0002@0002M0@002P000:41000:00000P0009d12P2Q0@X0E`0002@0002P0@007@000:<1000M0000
0P000:017@2S0Ad0E`0002@0002j0@0040000;/1000B00000P000;X14P2k0A00E`0002@0002k0@00
40000;/1000D00000P000;/1402k0A@0E`0002@0002j0@004P000;/1000D00000P000;/1502j0A80
E`0002@0002i0@004@000;X1000B00000P000;X14P2i0A40E`0002@0002g0@0040000;T1000A0000
0P000;T14@2g0A00E`0002@0002d0@0040000;L1000@00000P000;L1402d0A00E`0002@0002b0@00
40000;@1000A00000P000;@1402b0A40E`0002@0002a0@004@000;81000B00000P000;814@2a0A80
E`0002@0002a0@004P000;41000D00000P000;414P2a0A@0E`0002@0002a0@0050000;81000E0000
0P000;41502b0AD0E`0002@0002b0@005@000;@1000F00000P000;815@2d0AH0E`0002@0002d0@00
5P000;P1000H00000P000;@15P2h0AP0E`0002@0002h0@0060000;X1000I00000P000;P1602j0AT0
E`0002@0002j0@006@000;/1000J00000P000;X16@2k0AX0E`0002@0002a0@004`000;81000D0000
0P000;414`2b0A@0E`0002@0002b0@0050000;@1000E00000P000;81502d0AD0E`0002@0002d0@00
5@000;P1000G00000P000;@15@2h0AL0E`0002@0002h0@005`000;X1000H00000P000;P15`2j0AP0
E`0002@0002j0@0060000;/1000I00000P000;X1602k0AT0E`0002@0002k0@006@000;/1000L0000
0P000;/16@2k0A`0E`0002@0002j0@0070000;/1000M00000P000;/1702j0Ad0E`0002@0002h0@00
7@000;X1000M00000P000;X17@2h0Ad0E`0002@0002e0@007@000;P1000M00000P000;P17@2e0Ad0
E`0002@0002c0@007@000;D1000M00000P000;D17@2c0Ad0E`0002@0002b0@0070000;<1000M0000
0P000;<17@2b0A`0E`0002@0002a0@006P000;81000L00000P000;81702a0AX0E`0002@0002a0@00
6P000;41000M00000P000;416P2a0Ad0E`0002@0002a0@0070000;81000M00000P000;417@2b0A`0
E`0002@000320@002P000<<1000;00000P000<<12P320@/0E`0002@000320@002`000<<1000<0000
0P000<812`330@`0E`0002@000330@002`000<@1000<00000P000<<130340@/0E`0002@000330@00
2P000<@1000;00000P000<@12`330@X0E`0002@000330@0040000<<1000M00000P000<<140330Ad0
E`0002@000340@0040000<@1000M00000P000<@140340Ad0E`0002@000300@0040000<@1000@0000
0P000<0140340A00E`0002@000300@007@000<L1000M00000P000<017@370Ad0E`0002@0003<0@00
40000<`1000M00000P000<`1403<0Ad0E`0002@0003=0@0040000<d1000M00000P000<d1403=0Ad0
E`0002@0003=0@004@000<l1000C00000P000<d14`3?0A40E`0002@0003?0@0040000=41000A0000
0P000<l14@3A0A00E`0002@0003A0@0040000=<1000@00000P000=41403C0A00E`0002@0003C0@00
40000=H1000A00000P000=<1403F0A40E`0002@0003F0@004@000=L1000C00000P000=H14@3G0A<0
E`0002@0003G0@004`000=L1000M00000P000=L14`3G0Ad0E`0002@0003C0@0040000=D1000A0000
0P000=<1403E0A40E`0002@0003E0@004@000=H1000C00000P000=D14@3F0A<0E`0002@0003F0@00
4`000=H1000M00000P000=H14`3F0Ad0E`0002@0003G0@004@000=P1000C00000P000=L14`3H0A40
E`0002@0003H0@0040000=/1000A00000P000=P14@3K0A00E`0002@0003K0@0040000=d1000@0000
0P000=/1403M0A00E`0002@0003M0@0040000>01000A00000P000=d1403P0A40E`0002@0003P0@00
4@000>41000C00000P000>014@3Q0A<0E`0002@0003Q0@004`000>41000M00000P000>414`3Q0Ad0
E`0002@0003M0@0040000=l1000A00000P000=d1403O0A40E`0002@0003O0@004@000>01000C0000
0P000=l14@3P0A<0E`0002@0003P0@004`000>01000M00000P000>014`3P0Ad0E`0002@000390@00
40000<d1000@00000P000<T1403=0A00E`0002@000390@007@000=01000M00000P000<T17@3@0Ad0
E`0002@0003C0@007@000=T1000M00000P000=<17@3I0Ad0E`0002@0003M0@007@000><1000M0000
0P000=d17@3S0Ad0E`0002@0003Z0@0040000>X1000K00000P000>X1403Z0A/0E`0002@0003Z0@00
6`000>/1000M00000P000>X16`3[0Ad0E`0002@0003[0@007@000>h1000M00000P000>/17@3^0Ad0
E`0002@0003^0@007@000>l1000M00000P000>h17@3_0Ad0E`0002@0003_0@007@000?81000M0000
0P000>l17@3b0Ad0E`0002@0003b0@006`000?@1000M00000P000?817@3d0A/0E`0002@0003[0@00
40000>/1000K00000P000>/1403[0A/0E`0002@0003[0@006`000>`1000M00000P000>/16`3/0Ad0
E`0002@0003/0@007@000>h1000M00000P000>`17@3^0Ad0E`0002@0003d0@0040000?@1000M0000
0P000?@1403d0Ad0E`0002@0003e0@0040000?D1000M00000P000?D1403e0Ad0E`0002@0003W0@00
40000>/1000@00000P000>L1403[0A00E`0002@0003a0@0040000?D1000@00000P000?41403e0A00
E`0002@0003d0@007@000?L1000M00000P000?@17@3g0Ad0E`0002@0003m0@002P000?d1000M0000
0P000?d12P3m0Ad0E`0002@0003n0@002P000?h1000M00000P000?h12P3n0Ad0E`0002@0003j0@00
2P000?h1000:00000P000?X12P3n0@X0E`0002@0003j0@007@000042000M00000P000?X17@010Qd0
E`0002@000070P004P0000L2000C00000P0000L24P070Q<0E`0002@000060P004`0000L2000C0000
0P0000L24`060Q<0E`0002@000060P004P0000H2000C00000P0000H24`060Q80E`0002@000060P00
4@0000L2000B00000P0000H24P070Q40E`0002@000070P00400000T2000A00000P0000L24@090Q00
E`0002@000090P00400000`2000@00000P0000T2400<0Q00E`0002@0000<0P00400000h2000A0000
0P0000`2400>0Q40E`0002@0000>0P004@0000l2000B00000P0000h24@0?0Q80E`0002@0000?0P00
4P000102000D00000P0000l24P0@0Q@0E`0002@0000@0P0050000102000K00000P000102500@0Q/0
E`0002@0000@0P006`000142000M00000P0001026`0A0Qd0E`0002@0000A0P007@000182000M0000
0P0001427@0B0Qd0E`0002@0000?0P004P0000l2000K00000P0000l24P0?0Q/0E`0002@0000?0P00
6`000102000M00000P0000l26`0@0Qd0E`0002@0000@0P007@000182000M00000P0001027@0B0Qd0
E`0002@0000B0P007@0001<2000M00000P0001827@0C0Qd0E`0002@0000>0P00500000l2000E0000
0P0000l2500>0QD0E`0002@000090P005@0000h2000F00000P0000h25@090QH0E`0002@000060P00
5P0000T2000G00000P0000T25P060QL0E`0002@000050P005`0000H2000I00000P0000H25`050QT0
E`0002@000050P006@0000D2000K00000P0000D26@050Q/0E`0002@000050P006`0000H2000M0000
0P0000D26`060Qd0E`0002@000060P007@0000T2000M00000P0000H27@090Qd0E`0002@000090P00
7@0000/2000M00000P0000T27@0;0Qd0E`0002@0000;0P007@0000d2000M00000P0000/27@0=0Qd0
E`0002@0000=0P006`0000l2000M00000P0000d27@0?0Q/0E`0002@000070P005P0000T2000G0000
0P0000T25P070QL0E`0002@000060P005`0000L2000I00000P0000L25`060QT0E`0002@000060P00
6@0000H2000K00000P0000H26@060Q/0E`0002@000060P006`0000L2000M00000P0000H26`070Qd0
E`0002@000070P007@0000T2000M00000P0000L27@090Qd0E`0002@0000I0P002P0001T2000J0000
0P0001T22P0I0QX0E`0002@0000I0P006P0001X2000M00000P0001T26P0J0Qd0E`0002@0000J0P00
7@0001`2000M00000P0001X27@0L0Qd0E`0002@0000L0P007@0001d2000M00000P0001`27@0M0Qd0
E`0002@0000M0P007@0001l2000M00000P0001d27@0O0Qd0E`0002@0000O0P006`000202000M0000
0P0001l27@0P0Q/0E`0002@0000J0P002P0001X2000J00000P0001X22P0J0QX0E`0002@0000J0P00
6P0001/2000M00000P0001X26P0K0Qd0E`0002@0000K0P007@0001`2000M00000P0001/27@0L0Qd0
E`0002@0000F0P00400001d2000@00000P0001H2400M0Q00E`0002@0000U0P005P000302000F0000
0P0002D25P0`0QH0E`0002@0000`0P0050000302000F00000P0003025P0`0Q@0E`0002@0000_0P00
4P000302000D00000P000302500_0Q80E`0002@0000^0P004@0002l2000B00000P0002l24P0^0Q40
E`0002@0000/0P00400002h2000A00000P0002h24@0/0Q00E`0002@0000Z0P00400002`2000@0000
0P0002`2400Z0Q00E`0002@0000W0P00400002X2000A00000P0002X2400W0Q40E`0002@0000U0P00
4@0002L2000C00000P0002L24@0U0Q<0E`0002@0000T0P004`0002D2000F00000P0002D24`0T0QH0
E`0002@0000T0P005P0002@2000H00000P0002@25P0T0QP0E`0002@0000T0P00600002D2000K0000
0P0002@2600U0Q/0E`0002@0000U0P006`0002L2000M00000P0002D26`0W0Qd0E`0002@0000W0P00
7@0002X2000M00000P0002L27@0Z0Qd0E`0002@0000Z0P007@0002/2000M00000P0002X27@0[0Qd0
E`0002@0000[0P007@0002h2000M00000P0002/27@0^0Qd0E`0002@0000^0P006`000302000M0000
0P0002h27@0`0Q/0E`0002@0000_0P004`0002l2000F00000P0002l25P0_0Q<0E`0002@0000^0P00
4@0002l2000C00000P0002l24`0^0Q40E`0002@0000X0P00400002X2000A00000P0002X2400X0Q40
E`0002@0000V0P004@0002P2000C00000P0002P24@0V0Q<0E`0002@0000U0P004`0002H2000F0000
0P0002H24`0U0QH0E`0002@0000U0P005P0002D2000H00000P0002D25P0U0QP0E`0002@0000U0P00
600002H2000K00000P0002D2600V0Q/0E`0002@0000V0P006`0002P2000M00000P0002H26`0X0Qd0
E`0002@0000X0P007@0002X2000M00000P0002P27@0Z0Qd0E`0002@000100P002P000402000M0000
0P0004022P100Qd0E`0002@000110P002P000442000M00000P0004422P110Qd0E`0002@0000n0P00
4@000402000C00000P0004024`0n0Q40E`0002@0000l0P00400003h2000A00000P0003h24@0l0Q00
E`0002@0000k0P00400003`2000@00000P0003`2400k0Q00E`0002@0000h0P00400003/2000A0000
0P0003/2400h0Q40E`0002@0000f0P004@0003P2000C00000P0003P24@0f0Q<0E`0002@0000e0P00
4`0003H2000F00000P0003H24`0e0QH0E`0002@0000e0P005P0003D2000H00000P0003D25P0e0QP0
E`0002@0000e0P00600003H2000K00000P0003D2600f0Q/0E`0002@0000f0P006`0003P2000M0000
0P0003H26`0h0Qd0E`0002@0000h0P007@0003/2000M00000P0003P27@0k0Qd0E`0002@0000k0P00
7@0003`2000M00000P0003/27@0l0Qd0E`0002@0000l0P007@0003h2000M00000P0003`27@0n0Qd0
E`0002@0000n0P006`000402000M00000P0003h27@100Q/0E`0002@0000i0P00400003/2000A0000
0P0003/2400i0Q40E`0002@0000g0P004@0003T2000C00000P0003T24@0g0Q<0E`0002@0000f0P00
4`0003L2000F00000P0003L24`0f0QH0E`0002@0000f0P005P0003H2000H00000P0003H25P0f0QP0
E`0002@0000f0P00600003L2000K00000P0003H2600g0Q/0E`0002@0000g0P006`0003T2000M0000
0P0003L26`0i0Qd0E`0002@0000i0P007@0003/2000M00000P0003T27@0k0Qd0E`0002@0000m0P00
2P000442000:00000P0003d22P110PX0E`0002@000100P007@0004<2000M00000P0004027@130Qd0
E`0002@0001D0P002P0005@2000J00000P0005@22P1D0QX0E`0002@0001D0P006P0005D2000M0000
0P0005@26P1E0Qd0E`0002@0001E0P007@0005L2000M00000P0005D27@1G0Qd0E`0002@0001G0P00
7@0005P2000M00000P0005L27@1H0Qd0E`0002@0001H0P007@0005X2000M00000P0005P27@1J0Qd0
E`0002@0001J0P006`0005/2000M00000P0005X27@1K0Q/0E`0002@0001E0P002P0005D2000J0000
0P0005D22P1E0QX0E`0002@0001E0P006P0005H2000M00000P0005D26P1F0Qd0E`0002@0001F0P00
7@0005L2000M00000P0005H27@1G0Qd0E`0002@0001A0P00400005P2000@00000P000542401H0Q00
E`0002@0001Q0P0040000642000M00000P000642401Q0Qd0E`0002@0001R0P0040000682000M0000
0P000682401R0Qd0E`0002@0001R0P004`0006<2000F00000P0006825P1S0Q<0E`0002@0001S0P00
4@0006D2000C00000P0006<24`1U0Q40E`0002@0001U0P00400006H2000A00000P0006D24@1V0Q00
E`0002@0001V0P00400006T2000@00000P0006H2401Y0Q00E`0002@0001Y0P00400006X2000A0000
0P0006T2401Z0Q40E`0002@0001Z0P004@0006X2000B00000P0006X24@1Z0Q80E`0002@0001Y0P00
4P0006X2000C00000P0006X24P1Y0Q<0E`0002@0001X0P004P0006T2000C00000P0006T24`1X0Q80
E`0002@0001X0P004@0006T2000B00000P0006P24P1Y0Q40E`0002@0001N0P0040000682000@0000
0P0005h2401R0Q00E`0002@0001N0P007@0006D2000M00000P0005h27@1U0Qd0E`0002@0001a0P00
4P000742000C00000P0007424P1a0Q<0E`0002@0001`0P004`000742000C00000P0007424`1`0Q<0
E`0002@0001`0P004P000702000C00000P0007024`1`0Q80E`0002@0001`0P004@000742000B0000
0P0007024P1a0Q40E`0002@0001a0P00400007<2000A00000P0007424@1c0Q00E`0002@0001c0P00
400007H2000@00000P0007<2401f0Q00E`0002@0001f0P00400007P2000A00000P0007H2401h0Q40
E`0002@0001h0P004@0007T2000B00000P0007P24@1i0Q80E`0002@0001i0P004P0007X2000D0000
0P0007T24P1j0Q@0E`0002@0001j0P00500007X2000K00000P0007X2501j0Q/0E`0002@0001j0P00
6`0007/2000M00000P0007X26`1k0Qd0E`0002@0001k0P007@0007`2000M00000P0007/27@1l0Qd0
E`0002@0001i0P004P0007T2000K00000P0007T24P1i0Q/0E`0002@0001i0P006`0007X2000M0000
0P0007T26`1j0Qd0E`0002@0001j0P007@0007`2000M00000P0007X27@1l0Qd0E`0002@0001l0P00
7@0007d2000M00000P0007`27@1m0Qd0E`0002@0001h0P00500007T2000E00000P0007T2501h0QD0
E`0002@0001c0P005@0007P2000F00000P0007P25@1c0QH0E`0002@0001`0P005P0007<2000G0000
0P0007<25P1`0QL0E`0002@0001_0P005`000702000I00000P0007025`1_0QT0E`0002@0001_0P00
6@0006l2000K00000P0006l26@1_0Q/0E`0002@0001_0P006`000702000M00000P0006l26`1`0Qd0
E`0002@0001`0P007@0007<2000M00000P0007027@1c0Qd0E`0002@0001c0P007@0007D2000M0000
0P0007<27@1e0Qd0E`0002@0001e0P007@0007L2000M00000P0007D27@1g0Qd0E`0002@0001g0P00
6`0007T2000M00000P0007L27@1i0Q/0E`0002@0001a0P005P0007<2000G00000P0007<25P1a0QL0
E`0002@0001`0P005`000742000I00000P0007425`1`0QT0E`0002@0001`0P006@000702000K0000
0P0007026@1`0Q/0E`0002@0001`0P006`000742000M00000P0007026`1a0Qd0E`0002@0001a0P00
7@0007<2000M00000P0007427@1c0Qd0E`0002@0002;0P002P0008/2000M00000P0008/22P2;0Qd0
E`0002@0002<0P002P0008`2000M00000P0008`22P2<0Qd0E`0002@000290P004@0008/2000C0000
0P0008/24`290Q40E`0002@000270P00400008T2000A00000P0008T24@270Q00E`0002@000260P00
400008L2000@00000P0008L240260Q00E`0002@000230P00400008H2000A00000P0008H240230Q40
E`0002@000210P004@0008<2000C00000P0008<24@210Q<0E`0002@000200P004`000842000F0000
0P0008424`200QH0E`0002@000200P005P000802000H00000P0008025P200QP0E`0002@000200P00
60000842000K00000P00080260210Q/0E`0002@000210P006`0008<2000M00000P0008426`230Qd0
E`0002@000230P007@0008H2000M00000P0008<27@260Qd0E`0002@000260P007@0008L2000M0000
0P0008H27@270Qd0E`0002@000270P007@0008T2000M00000P0008L27@290Qd0E`0002@000290P00
6`0008/2000M00000P0008T27@2;0Q/0E`0002@000240P00400008H2000A00000P0008H240240Q40
E`0002@000220P004@0008@2000C00000P0008@24@220Q<0E`0002@000210P004`000882000F0000
0P0008824`210QH0E`0002@000210P005P000842000H00000P0008425P210QP0E`0002@000210P00
60000882000K00000P00084260220Q/0E`0002@000220P006`0008@2000M00000P0008826`240Qd0
E`0002@000240P007@0008H2000M00000P0008@27@260Qd0E`0002@000280P002P0008`2000:0000
0P0008P22P2<0PX0E`0002@0002;0P007@0008h2000M00000P0008/27@2>0Qd0E`0002@0002D0P00
5P0009l2000F00000P0009@25P2O0QH0E`0002@0002O0P00500009l2000F00000P0009l25P2O0Q@0
E`0002@0002N0P004P0009l2000D00000P0009l2502N0Q80E`0002@0002M0P004@0009h2000B0000
0P0009h24P2M0Q40E`0002@0002K0P00400009d2000A00000P0009d24@2K0Q00E`0002@0002I0P00
400009/2000@00000P0009/2402I0Q00E`0002@0002F0P00400009T2000A00000P0009T2402F0Q40
E`0002@0002D0P004@0009H2000C00000P0009H24@2D0Q<0E`0002@0002C0P004`0009@2000F0000
0P0009@24`2C0QH0E`0002@0002C0P005P0009<2000H00000P0009<25P2C0QP0E`0002@0002C0P00
600009@2000K00000P0009<2602D0Q/0E`0002@0002D0P006`0009H2000M00000P0009@26`2F0Qd0
E`0002@0002F0P007@0009T2000M00000P0009H27@2I0Qd0E`0002@0002I0P007@0009X2000M0000
0P0009T27@2J0Qd0E`0002@0002J0P007@0009d2000M00000P0009X27@2M0Qd0E`0002@0002M0P00
6`0009l2000M00000P0009d27@2O0Q/0E`0002@0002N0P004`0009h2000F00000P0009h25P2N0Q<0
E`0002@0002M0P004@0009h2000C00000P0009h24`2M0Q40E`0002@0002G0P00400009T2000A0000
0P0009T2402G0Q40E`0002@0002E0P004@0009L2000C00000P0009L24@2E0Q<0E`0002@0002D0P00
4`0009D2000F00000P0009D24`2D0QH0E`0002@0002D0P005P0009@2000H00000P0009@25P2D0QP0
E`0002@0002D0P00600009D2000K00000P0009@2602E0Q/0E`0002@0002E0P006`0009L2000M0000
0P0009D26`2G0Qd0E`0002@0002G0P007@0009T2000M00000P0009L27@2I0Qd0E`0002@0002i0P00
2P000;T2000M00000P000;T22P2i0Qd0E`0002@0002j0P002P000;X2000M00000P000;X22P2j0Qd0
E`0002@0002g0P004@000;T2000C00000P000;T24`2g0Q40E`0002@0002e0P0040000;L2000A0000
0P000;L24@2e0Q00E`0002@0002d0P0040000;D2000@00000P000;D2402d0Q00E`0002@0002a0P00
40000;@2000A00000P000;@2402a0Q40E`0002@0002_0P004@000;42000C00000P000;424@2_0Q<0
E`0002@0002^0P004`000:l2000F00000P000:l24`2^0QH0E`0002@0002^0P005P000:h2000H0000
0P000:h25P2^0QP0E`0002@0002^0P0060000:l2000K00000P000:h2602_0Q/0E`0002@0002_0P00
6`000;42000M00000P000:l26`2a0Qd0E`0002@0002a0P007@000;@2000M00000P000;427@2d0Qd0
E`0002@0002d0P007@000;D2000M00000P000;@27@2e0Qd0E`0002@0002e0P007@000;L2000M0000
0P000;D27@2g0Qd0E`0002@0002g0P006`000;T2000M00000P000;L27@2i0Q/0E`0002@0002b0P00
40000;@2000A00000P000;@2402b0Q40E`0002@0002`0P004@000;82000C00000P000;824@2`0Q<0
E`0002@0002_0P004`000;02000F00000P000;024`2_0QH0E`0002@0002_0P005P000:l2000H0000
0P000:l25P2_0QP0E`0002@0002_0P0060000;02000K00000P000:l2602`0Q/0E`0002@0002`0P00
6`000;82000M00000P000;026`2b0Qd0E`0002@0002b0P007@000;@2000M00000P000;827@2d0Qd0
E`0002@0002f0P002P000;X2000:00000P000;H22P2j0PX0E`0002@0002i0P007@000;`2000M0000
0P000;T27@2l0Qd0E`0002@000320P002P000<<2000;00000P000<<22P320P/0E`0002@000320P00
2`000<<2000<00000P000<822`330P`0E`0002@000330P002`000<@2000<00000P000<<230340P/0
E`0002@000330P002P000<@2000;00000P000<@22`330PX0E`0002@000330P0040000<<2000M0000
0P000<<240330Qd0E`0002@000340P0040000<@2000M00000P000<@240340Qd0E`0002@000300P00
40000<@2000@00000P000<0240340Q00E`0002@000300P007@000<L2000M00000P000<027@370Qd0
E`0002@0003<0P0040000<`2000M00000P000<`2403<0Qd0E`0002@0003=0P0040000<d2000M0000
0P000<d2403=0Qd0E`0002@0003=0P004`000<h2000F00000P000<d25P3>0Q<0E`0002@0003>0P00
4@000=02000C00000P000<h24`3@0Q40E`0002@0003@0P0040000=42000A00000P000=024@3A0Q00
E`0002@0003A0P0040000=@2000@00000P000=42403D0Q00E`0002@0003D0P0040000=D2000A0000
0P000=@2403E0Q40E`0002@0003E0P004@000=D2000B00000P000=D24@3E0Q80E`0002@0003D0P00
4P000=D2000C00000P000=D24P3D0Q<0E`0002@0003C0P004P000=@2000C00000P000=@24`3C0Q80
E`0002@0003C0P004@000=@2000B00000P000=<24P3D0Q40E`0002@000390P0040000<d2000@0000
0P000<T2403=0Q00E`0002@000390P007@000=02000M00000P000<T27@3@0Qd0E`0002@0003K0P00
5P000>H2000F00000P000=/25P3V0QH0E`0002@0003V0P0050000>H2000F00000P000>H25P3V0Q@0
E`0002@0003U0P004P000>H2000D00000P000>H2503U0Q80E`0002@0003T0P004@000>D2000B0000
0P000>D24P3T0Q40E`0002@0003R0P0040000>@2000A00000P000>@24@3R0Q00E`0002@0003P0P00
40000>82000@00000P000>82403P0Q00E`0002@0003M0P0040000>02000A00000P000>02403M0Q40
E`0002@0003K0P004@000=d2000C00000P000=d24@3K0Q<0E`0002@0003J0P004`000=/2000F0000
0P000=/24`3J0QH0E`0002@0003J0P005P000=X2000H00000P000=X25P3J0QP0E`0002@0003J0P00
60000=/2000K00000P000=X2603K0Q/0E`0002@0003K0P006`000=d2000M00000P000=/26`3M0Qd0
E`0002@0003M0P007@000>02000M00000P000=d27@3P0Qd0E`0002@0003P0P007@000>42000M0000
0P000>027@3Q0Qd0E`0002@0003Q0P007@000>@2000M00000P000>427@3T0Qd0E`0002@0003T0P00
6`000>H2000M00000P000>@27@3V0Q/0E`0002@0003U0P004`000>D2000F00000P000>D25P3U0Q<0
E`0002@0003T0P004@000>D2000C00000P000>D24`3T0Q40E`0002@0003N0P0040000>02000A0000
0P000>02403N0Q40E`0002@0003L0P004@000=h2000C00000P000=h24@3L0Q<0E`0002@0003K0P00
4`000=`2000F00000P000=`24`3K0QH0E`0002@0003K0P005P000=/2000H00000P000=/25P3K0QP0
E`0002@0003K0P0060000=`2000K00000P000=/2603L0Q/0E`0002@0003L0P006`000=h2000M0000
0P000=`26`3N0Qd0E`0002@0003N0P007@000>02000M00000P000=h27@3P0Qd0E`0002@0003e0P00
4`000?H2000D00000P000?H24`3e0Q@0E`0002@0003e0P0050000?H2000E00000P000?D2503f0QD0
E`0002@0003f0P0050000?L2000E00000P000?H25@3g0Q@0E`0002@0003g0P004`000?L2000D0000
0P000?L2503g0Q<0E`0002@0003e0P004@000?L2000C00000P000?L24`3e0Q40E`0002@0003c0P00
40000?D2000A00000P000?D24@3c0Q00E`0002@0003a0P0040000?<2000@00000P000?<2403a0Q00
E`0002@0003^0P0040000?42000A00000P000?42403^0Q40E`0002@0003/0P004@000>h2000C0000
0P000>h24@3/0Q<0E`0002@0003[0P004`000>`2000F00000P000>`24`3[0QH0E`0002@0003[0P00
5P000>/2000H00000P000>/25P3[0QP0E`0002@0003[0P0060000>`2000K00000P000>/2603/0Q/0
E`0002@0003/0P006`000>h2000M00000P000>`26`3^0Qd0E`0002@0003^0P007@000?42000M0000
0P000>h27@3a0Qd0E`0002@0003a0P007@000?82000M00000P000?427@3b0Qd0E`0002@0003b0P00
7@000?D2000M00000P000?827@3e0Qd0E`0002@0003e0P006`000?L2000M00000P000?D27@3g0Q/0
E`0002@0003_0P0040000?42000A00000P000?42403_0Q40E`0002@0003]0P004@000>l2000C0000
0P000>l24@3]0Q<0E`0002@0003/0P004`000>d2000F00000P000>d24`3/0QH0E`0002@0003/0P00
5P000>`2000H00000P000>`25P3/0QP0E`0002@0003/0P0060000>d2000K00000P000>`2603]0Q/0
E`0002@0003]0P006`000>l2000M00000P000>d26`3_0Qd0E`0002@0003_0P007@000?42000M0000
0P000>l27@3a0Qd0E`0002@0003m0P002P000?d2000J00000P000?d22P3m0QX0E`0002@0003m0P00
6P000?h2000M00000P000?d26P3n0Qd0E`0002@0003n0P007@000003000M00000P000?h27@000ad0
E`0002@000000`007@000043000M00000P0000037@010ad0E`0002@000010`007@0000<3000M0000
0P0000437@030ad0E`0002@000030`006`0000@3000M00000P0000<37@040a/0E`0002@0003n0P00
2P000?h2000J00000P000?h22P3n0QX0E`0002@0003n0P006P000?l2000M00000P000?h26P3o0Qd0
E`0002@0003o0P007@000003000M00000P000?l27@000ad0E`0002@0003j0P0040000043000@0000
0P000?X240010a00E`0002@0000:0`002P0000/3000;00000P0000/32P0:0`/0E`0002@0000:0`00
2`0000/3000<00000P0000X32`0;0``0E`0002@0000;0`002`0000`3000<00000P0000/3300<0`/0
E`0002@0000;0`002P0000`3000;00000P0000`32`0;0`X0E`0002@0000;0`00400000/3000M0000
0P0000/3400;0ad0E`0002@0000<0`00400000`3000M00000P0000`3400<0ad0E`0002@000080`00
400000`3000@00000P0000P3400<0a00E`0002@000080`007@0000l3000M00000P0000P37@0?0ad0
E`0002@0000F0`00400001T3000A00000P0001T3400F0a40E`0002@0000D0`004@0001H3000C0000
0P0001H34@0D0a<0E`0002@0000C0`004`0001@3000F00000P0001@34`0C0aH0E`0002@0000C0`00
5P0001<3000H00000P0001<35P0C0aP0E`0002@0000C0`00600001@3000K00000P0001<3600D0a/0
E`0002@0000D0`006`0001H3000M00000P0001@36`0F0ad0E`0002@0000F0`007@0001T3000M0000
0P0001H37@0I0ad0E`0002@0000I0`007@0001X3000M00000P0001T37@0J0ad0E`0002@0000J0`00
7@0001d3000M00000P0001X37@0M0ad0E`0002@0000M0`006`0001l3000M00000P0001d37@0O0a/0
E`0002@0000O0`0060000203000K00000P0001l36`0P0aP0E`0002@0000P0`005P000203000H0000
0P000203600P0aH0E`0002@0000O0`004`000203000F00000P0002035P0O0a<0E`0002@0000M0`00
4@0001l3000C00000P0001l34`0M0a40E`0002@0000J0`00400001d3000A00000P0001d34@0J0a00
E`0002@0000I0`00400001X3000@00000P0001X3400I0a00E`0002@0000G0`00400001T3000A0000
0P0001T3400G0a40E`0002@0000E0`004@0001L3000C00000P0001L34@0E0a<0E`0002@0000D0`00
4`0001D3000F00000P0001D34`0D0aH0E`0002@0000D0`005P0001@3000H00000P0001@35P0D0aP0
E`0002@0000D0`00600001D3000K00000P0001@3600E0a/0E`0002@0000E0`006`0001L3000M0000
0P0001D36`0G0ad0E`0002@0000G0`007@0001T3000M00000P0001L37@0I0ad0E`0002@0000J0`00
7@0001`3000M00000P0001X37@0L0ad0E`0002@0000L0`006`0001h3000M00000P0001`37@0N0a/0
E`0002@0000N0`00600001l3000K00000P0001h36`0O0aP0E`0002@0000O0`005P0001l3000H0000
0P0001l3600O0aH0E`0002@0000N0`004`0001l3000F00000P0001l35P0N0a<0E`0002@0000L0`00
4@0001h3000C00000P0001h34`0L0a40E`0002@0000J0`00400001`3000A00000P0001`34@0J0a00
E`0002@0000V0`00400002H3000M00000P0002H3400V0ad0E`0002@0000W0`00400002L3000M0000
0P0002L3400W0ad0E`0002@0000W0`004@0002T3000C00000P0002L34`0Y0a40E`0002@0000Y0`00
400002/3000A00000P0002T34@0[0a00E`0002@0000[0`00400002d3000@00000P0002/3400]0a00
E`0002@0000]0`0040000303000A00000P0002d3400`0a40E`0002@0000`0`004@000343000C0000
0P0003034@0a0a<0E`0002@0000a0`004`000343000M00000P0003434`0a0ad0E`0002@0000]0`00
400002l3000A00000P0002d3400_0a40E`0002@0000_0`004@000303000C00000P0002l34@0`0a<0
E`0002@0000`0`004`000303000M00000P0003034`0`0ad0E`0002@0000S0`00400002L3000@0000
0P0002<3400W0a00E`0002@0000S0`007@0002X3000M00000P0002<37@0Z0ad0E`0002@0000]0`00
7@0003<3000M00000P0002d37@0c0ad0E`0002@000110`0040000483000B00000P0004434P120a00
E`0002@000120`0040000483000D00000P00048340120a@0E`0002@000110`004P000483000D0000
0P00048350110a80E`0002@000100`004@000443000B00000P0004434P100a40E`0002@0000n0`00
40000403000A00000P0004034@0n0a00E`0002@0000k0`00400003h3000@00000P0003h3400k0a00
E`0002@0000i0`00400003/3000A00000P0003/3400i0a40E`0002@0000h0`004@0003T3000B0000
0P0003T34@0h0a80E`0002@0000h0`004P0003P3000D00000P0003P34P0h0a@0E`0002@0000h0`00
500003T3000E00000P0003P3500i0aD0E`0002@0000i0`005@0003/3000F00000P0003T35@0k0aH0
E`0002@0000k0`005P0003l3000H00000P0003/35P0o0aP0E`0002@0000o0`0060000443000I0000
0P0003l360110aT0E`0002@000110`006@000483000J00000P0004436@120aX0E`0002@0000h0`00
4`0003T3000D00000P0003P34`0i0a@0E`0002@0000i0`00500003/3000E00000P0003T3500k0aD0
E`0002@0000k0`005@0003l3000G00000P0003/35@0o0aL0E`0002@0000o0`005`000443000H0000
0P0003l35`110aP0E`0002@000110`0060000483000I00000P00044360120aT0E`0002@000120`00
6@000483000L00000P0004836@120a`0E`0002@000110`0070000483000M00000P00048370110ad0
E`0002@0000o0`007@000443000M00000P0004437@0o0ad0E`0002@0000l0`007@0003l3000M0000
0P0003l37@0l0ad0E`0002@0000j0`007@0003`3000M00000P0003`37@0j0ad0E`0002@0000i0`00
700003X3000M00000P0003X37@0i0a`0E`0002@0000h0`006P0003T3000L00000P0003T3700h0aX0
E`0002@0000h0`006P0003P3000M00000P0003P36P0h0ad0E`0002@0000h0`00700003T3000M0000
0P0003P37@0i0a`0E`0002@0001B0`00400005H3000M00000P000583401F0ad0E`0002@0001C0`00
400005H3000K00000P0005<3401F0a/0E`0002@0001F0`00400005T3000M00000P0005T3401F0ad0
E`0002@0001I0`00400005d3000M00000P0005T3401M0ad0E`0002@0001J0`00400005d3000K0000
0P0005X3401M0a/0E`0002@0001M0`0040000603000M00000P000603401M0ad0E`0002@0001?0`00
400005H3000@00000P0004l3401F0a00E`0002@0001N0`00400006<3000@00000P0005h3401S0a00
E`0002@0001X0`002P0006T3000;00000P0006T32P1X0`/0E`0002@0001X0`002`0006T3000<0000
0P0006P32`1Y0``0E`0002@0001Y0`002`0006X3000<00000P0006T3301Z0`/0E`0002@0001Y0`00
2P0006X3000;00000P0006X32`1Y0`X0E`0002@0001Y0`00400006T3000M00000P0006T3401Y0ad0
E`0002@0001Z0`00400006X3000M00000P0006X3401Z0ad0E`0002@0001V0`00400006X3000@0000
0P0006H3401Z0a00E`0002@0001V0`007@0006d3000M00000P0006H37@1]0ad0E`0002@0001c0`00
2P0007<3000J00000P0007<32P1c0aX0E`0002@0001c0`006P0007@3000M00000P0007<36P1d0ad0
E`0002@0001d0`007@0007H3000M00000P0007@37@1f0ad0E`0002@0001f0`007@0007L3000M0000
0P0007H37@1g0ad0E`0002@0001g0`007@0007T3000M00000P0007L37@1i0ad0E`0002@0001i0`00
6`0007X3000M00000P0007T37@1j0a/0E`0002@0001d0`002P0007@3000J00000P0007@32P1d0aX0
E`0002@0001d0`006P0007D3000M00000P0007@36P1e0ad0E`0002@0001e0`007@0007H3000M0000
0P0007D37@1f0ad0E`0002@0001`0`00400007L3000@00000P000703401g0a00E`0002@000200`00
2P000803000M00000P0008032P200ad0E`0002@000210`002P000843000M00000P0008432P210ad0
E`0002@000210`004@0008<3000C00000P0008434`230a40E`0002@000230`00400008D3000A0000
0P0008<34@250a00E`0002@000250`00400008L3000@00000P0008D340270a00E`0002@000270`00
400008X3000A00000P0008L3402:0a40E`0002@0002:0`004@0008/3000C00000P0008X34@2;0a<0
E`0002@0002;0`004`0008/3000M00000P0008/34`2;0ad0E`0002@000270`00400008T3000A0000
0P0008L340290a40E`0002@000290`004@0008X3000C00000P0008T34@2:0a<0E`0002@0002:0`00
4`0008X3000M00000P0008X34`2:0ad0E`0002@0001m0`002P000843000:00000P0007d32P210`X0
E`0002@0001m0`007@0008@3000M00000P0007d37@240ad0E`0002@000270`007@0008d3000M0000
0P0008L37@2=0ad0E`0002@0002V0`004`000:L3000D00000P000:L34`2V0a@0E`0002@0002V0`00
50000:L3000E00000P000:H3502W0aD0E`0002@0002W0`0050000:P3000E00000P000:L35@2X0a@0
E`0002@0002X0`004`000:P3000D00000P000:P3502X0a<0E`0002@0002V0`004@000:P3000C0000
0P000:P34`2V0a40E`0002@0002T0`0040000:H3000A00000P000:H34@2T0a00E`0002@0002R0`00
40000:@3000@00000P000:@3402R0a00E`0002@0002O0`0040000:83000A00000P000:83402O0a40
E`0002@0002M0`004@0009l3000C00000P0009l34@2M0a<0E`0002@0002L0`004`0009d3000F0000
0P0009d34`2L0aH0E`0002@0002L0`005P0009`3000H00000P0009`35P2L0aP0E`0002@0002L0`00
600009d3000K00000P0009`3602M0a/0E`0002@0002M0`006`0009l3000M00000P0009d36`2O0ad0
E`0002@0002O0`007@000:83000M00000P0009l37@2R0ad0E`0002@0002R0`007@000:<3000M0000
0P000:837@2S0ad0E`0002@0002S0`007@000:H3000M00000P000:<37@2V0ad0E`0002@0002V0`00
6`000:P3000M00000P000:H37@2X0a/0E`0002@0002P0`0040000:83000A00000P000:83402P0a40
E`0002@0002N0`004@000:03000C00000P000:034@2N0a<0E`0002@0002M0`004`0009h3000F0000
0P0009h34`2M0aH0E`0002@0002M0`005P0009d3000H00000P0009d35P2M0aP0E`0002@0002M0`00
600009h3000K00000P0009d3602N0a/0E`0002@0002N0`006`000:03000M00000P0009h36`2P0ad0
E`0002@0002P0`007@000:83000M00000P000:037@2R0ad0E`0002@0002]0`004P000;d3000B0000
0P000:d34P2m0a80E`0002@0002]0`0060000;d3000H00000P000:d3602m0aP0E`0002@000350`00
70000<H3000M00000P000<H370350ad0E`0002@000350`007@000<H3000M00000P000<D37@360ad0
E`0002@000360`007@000<L3000M00000P000<H37@370ad0E`0002@000360`0070000<L3000M0000
0P000<L37@360a`0E`0002@0003@0`002P000=<3000;00000P000=<32P3@0`/0E`0002@0003>0`00
2`000=03000>00000P000=032`3>0`h0E`0002@0003=0`003P000<h3000B00000P000<h33P3=0a80
E`0002@0003=0`004P000<d3000E00000P000<d34P3=0aD0E`0002@0003=0`005@000<h3000J0000
0P000<d35@3>0aX0E`0002@0003>0`006P000=03000M00000P000<h36P3@0ad0E`0002@0003@0`00
7@000=<3000M00000P000=037@3C0ad0E`0002@0003C0`007@000=@3000M00000P000=<37@3D0ad0
E`0002@0003D0`007@000=L3000M00000P000=@37@3G0ad0E`0002@0003G0`006P000=T3000M0000
0P000=L37@3I0aX0E`0002@0003I0`005@000=X3000J00000P000=T36P3J0aD0E`0002@0003J0`00
4P000=X3000E00000P000=X35@3J0a80E`0002@0003I0`003P000=X3000B00000P000=X34P3I0`h0
E`0002@0003G0`002`000=T3000>00000P000=T33P3G0`/0E`0002@0003D0`002P000=L3000;0000
0P000=L32`3D0`X0E`0002@0003C0`002P000=@3000:00000P000=@32P3C0`X0E`0002@0003A0`00
2P000=<3000;00000P000=<32P3A0`/0E`0002@0003@0`002`000=43000<00000P000=432`3@0``0
E`0002@0003?0`0030000=03000>00000P000=03303?0`h0E`0002@0003>0`003P000<l3000B0000
0P000<l33P3>0a80E`0002@0003>0`004P000<h3000E00000P000<h34P3>0aD0E`0002@0003>0`00
5@000<l3000J00000P000<h35@3?0aX0E`0002@0003?0`006P000=03000L00000P000<l36P3@0a`0
E`0002@0003@0`0070000=43000M00000P000=03703A0ad0E`0002@0003A0`007@000=<3000M0000
0P000=437@3C0ad0E`0002@0003D0`007@000=H3000M00000P000=@37@3F0ad0E`0002@0003F0`00
70000=L3000M00000P000=H37@3G0a`0E`0002@0003G0`006P000=P3000L00000P000=L3703H0aX0
E`0002@0003H0`005@000=T3000J00000P000=P36P3I0aD0E`0002@0003I0`004P000=T3000E0000
0P000=T35@3I0a80E`0002@0003H0`003P000=T3000B00000P000=T34P3H0`h0E`0002@0003G0`00
30000=P3000>00000P000=P33P3G0``0E`0002@0003F0`002`000=L3000<00000P000=L3303F0`/0
E`0002@0003D0`002P000=H3000;00000P000=H32`3D0`X0E`0002@0003Q0`003@000><3000>0000
0P000>433P3S0`d0E`0002@0003S0`002P000>D3000=00000P000><33@3U0`X0E`0002@0003U0`00
2P000>D3000M00000P000>D32P3U0ad0E`0002@0003T0`002`000>@3000M00000P000>@32`3T0ad0
E`0002@0003Q0`007@000>T3000M00000P000>437@3Y0ad0E`0002@0003:0000O0<00<l0001l0`00
0P000<l0O0?:07`3E`0002@0003:0000iP400<l0003V0@000P000<l0iP7:0>H1E`0002@0003:0000
D0000<l0001@00000P000<l0D03:0500E`0002@0000@00005@8002<0000E0P000P0001005@8S01D2
E`0002@0000@0000508002<0000D0P000P000100508S01@2E`0002@0000F00003`8001d0000?0P00
0P0001H03`8M00l2E`0002@0000@00002@800100000H0P000P000100608@00T2E`0002@0000@0000
2@8001H000090P000P0001002@8F00T2E`0002@0000@00002@8001H0000:0P000P0001H02@8@00X2
E`0002@0000I00003`8001T0000D0P000P0001T0508I00l2E`0002@0000S00002@8002<0000H0P00
0P0002<0608S00T2E`0002@0000N00002@8002<000090P000P0002<02@8N00T2E`0002@0000N0000
2@8002<0000:0P000P0001h02@8S00X2E`0002@0000=0000o`4000d000020P000P0000d0o`4=0082
E`0002@0000=00000P8002T000020P000P0000d00P8Y0082E`0002@0000Y0000o`4002T000020P00
0P0002T00P8Y0?l1E`0002@0000=00000@8002T000010P000P0000d00@8Y0042E`0002@0000G0000
k04002T0003/0@000P0001L0k04Y0>`1E`0002@0000G0000j`4002T0003[0@000P0001L0j`4Y0>/1
E`0002@0000G0000k04001T0003^0@000P0001T0k04G0>h1E`0002@0000G0000kP4001L0003`0@00
0P0001L0kP4G0?01E`0002@0000G0000l04001L0003b0@000P0001L0l04G0?81E`0002@0000G0000
lP4001L0003d0@000P0001L0lP4G0?@1E`0002@0000G0000m04001T0003f0@000P0001L0m04I0?H1
E`0002@0000I0000mP4001`0003g0@000P0001T0mP4L0?L1E`0002@0000L0000m`4001h0003g0@00
0P0001`0m`4N0?L1E`0002@0000N0000mP400200003g0@000P0001h0m`4P0?H1E`0002@0000P0000
m0400280003f0@000P000200mP4R0?@1E`0002@0000R0000lP4002<0003d0@000P000280m04S0?81
E`0002@0000S0000l04002<0003b0@000P0002<0lP4S0?01E`0002@0000R0000kP4002<0003`0@00
0P0002<0l04R0>h1E`0002@0000P0000k0400280003^0@000P000280kP4P0>`1E`0002@0000G0000
lP4001L0003c0@000P0001L0lP4G0?<1E`0002@0000G0000l`4001T0003e0@000P0001L0l`4I0?D1
E`0002@0000I0000m@4001`0003f0@000P0001T0m@4L0?H1E`0002@0000L0000mP4001h0003f0@00
0P0001`0mP4N0?H1E`0002@0000N0000m@400200003f0@000P0001h0mP4P0?D1E`0002@0000P0000
l`400280003e0@000P000200m@4R0?<1E`0002@0000R0000lP4002<0003c0@000P000280l`4S0?81
E`0002@0000Y0000j04002T0003_0@000P0002T0k`4Y0>P1E`0002@0000=0000g@4000h0003O0@00
0P0000d0g@4>0=l1E`0002@0000>0000g`400140003P0@000P0000h0g`4A0>01E`0002@0000A0000
h04001D0003R0@000P000140h04E0>81E`0002@0000E0000hP4001T0003S0@000P0001D0hP4I0><1
E`0002@0000I0000h`4001d0003S0@000P0001T0h`4M0><1E`0002@0000M0000hP400240003S0@00
0P0001d0h`4Q0>81E`0002@0000Q0000h04002D0003R0@000P000240hP4U0>01E`0002@0000U0000
g`4002P0003P0@000P0002D0h04X0=l1E`0002@0000X0000g@4002T0003O0@000P0002P0g`4Y0=d1
E`0002@0000>0000g`400180003P0@000P0000h0g`4B0>01E`0002@0000B0000h04001D0003Q0@00
0P000180h04E0>41E`0002@0000E0000h@4001T0003R0@000P0001D0h@4I0>81E`0002@0000I0000
hP4001d0003R0@000P0001T0hP4M0>81E`0002@0000M0000h@400240003R0@000P0001d0hP4Q0>41
E`0002@0000Q0000h04002@0003Q0@000P000240h@4T0>01E`0002@0000T0000g`4002P0003P0@00
0P0002@0h04X0=l1E`0002@0000@0000e@4001l0003E0@000P000100e@4O0=D1E`0002@0000O0000
e0400280003E0@000P0001l0e@4R0=@1E`0002@0000R0000d`4002<0003D0@000P000280e04S0=<1
E`0002@0000S0000d@4002<0003C0@000P0002<0d`4S0=41E`0002@0000R0000c`4002<0003A0@00
0P0002<0d@4R0<l1E`0002@0000P0000cP400280003?0@000P000280c`4P0<h1E`0002@0000@0000
e04001l0003D0@000P000100e04O0=@1E`0002@0000O0000d`400280003D0@000P0001l0e04R0=<1
E`0002@0000R0000d`4002<0003C0@000P000280d`4S0=<1E`0002@0000G0000d@4001L0003H0@00
0P0001L0f04G0=41E`0002@0000=0000a`4000h000390@000P0000d0b@4>0<L1E`0002@0000>0000
a@40014000370@000P0000h0a`4A0<D1E`0002@0000A0000a04001D000350@000P000140a@4E0<@1
E`0002@0000E0000``4001T000340@000P0001D0a04I0<<1E`0002@0000I0000``4001d000330@00
0P0001T0``4M0<<1E`0002@0000M0000``40024000340@000P0001d0``4Q0<@1E`0002@0000Q0000
a04002D000350@000P000240a04U0<D1E`0002@0000U0000a@4002P000370@000P0002D0a@4X0<L1
E`0002@0000X0000a`4002T000390@000P0002P0a`4Y0<T1E`0002@0000>0000a@40018000370@00
0P0000h0a`4B0<D1E`0002@0000B0000a@4001D000350@000P000180a@4E0<D1E`0002@0000E0000
a04001T000350@000P0001D0a@4I0<@1E`0002@0000I0000a04001d000340@000P0001T0a04M0<@1
E`0002@0000M0000a040024000350@000P0001d0a04Q0<D1E`0002@0000Q0000a@4002@000350@00
0P000240a@4T0<D1E`0002@0000T0000a@4002P000370@000P0002@0a@4X0<L1E`0002@0000=0000
]`4000d0002j0@000P0000d0^P4=0;L1E`0002@0000=0000]`4002T0002g0@000P0000d0]`4Y0;L1
E`0002@0000Y0000]`4002T0002j0@000P0002T0]`4Y0;X1E`0002@0000=0000^04002T0002h0@00
0P0000d0^04Y0;P1E`0002@000160000OP<005H0001n0`000P0004H0OP=F07h3E`0002@0001O0000
MP<00640001g0`000P0005l0M`=Q07H3E`0002@0001Q0000L`<006<0001f0`000P000640MP=S07<3
E`0002@0001S0000L`<006<000260`000P0006<0L`=S08H3E`0002@0001R0000M0<0068000260`00
0P000680M0=R08H3E`0002@0001O0000QP<006L000260`000P0005l0QP=W08H3E`0002@0001n0000
K`<007l0001a0`000P0007l0K`=n0743E`0002@0001l0000L@<007h0001d0`000P0007h0L@=l07@3
E`0002@0001j0000M0<007`0001h0`000P0007`0M0=j07P3E`0002@0001i0000N0<007X0001l0`00
0P0007X0N0=i07`3E`0002@0001i0000O0<007T000200`000P0007T0O0=i0803E`0002@0001i0000
P0<007X000250`000P0007T0P0=j08D3E`0002@0001j0000Q@<007`000280`000P0007X0Q@=l08P3
E`0002@0001l0000R0<007h0002;0`000P0007`0R0=n08/3E`0002@0001n0000R`<007l0002=0`00
0P0007h0R`=o08d3E`0002@0001l0000L@<007h0001e0`000P0007h0L@=l07D3E`0002@0001k0000
M@<007`0001h0`000P0007`0M@=k07P3E`0002@0001j0000N0<007/0001l0`000P0007/0N0=j07`3
E`0002@0001j0000O0<007X000200`000P0007X0O0=j0803E`0002@0001j0000P0<007/000250`00
0P0007X0P0=k08D3E`0002@0001k0000Q@<007`000270`000P0007/0Q@=l08L3E`0002@0001l0000
Q`<007h0002;0`000P0007`0Q`=n08/3E`0002@0002A0000L`<00980001f0`000P000940MP>B07<3
E`0002@0002B0000L`<00980001h0`000P000980L`>B07P3E`0002@0002A0000MP<00980001h0`00
0P000980N0>A07H3E`0002@0002?0000M0<00940001f0`000P000940MP>?07@3E`0002@0002<0000
L`<008l0001d0`000P0008l0M0><07<3E`0002@0002:0000L`<008`0001c0`000P0008`0L`>:07<3
E`0002@000270000L`<008X0001d0`000P0008X0L`>707@3E`0002@000250000M0<008L0001f0`00
0P0008L0M0>507H3E`0002@000250000MP<008D0001h0`000P0008D0MP>507P3E`0002@000250000
N0<008H0001i0`000P0008D0N0>607T3E`0002@000260000N@<008L0001j0`000P0008H0N@>707X3
E`0002@000270000NP<008T0001k0`000P0008L0NP>907/3E`0002@000290000N`<008h0001m0`00
0P0008T0N`>>07d3E`0002@0002>0000O@<00900001n0`000P0008h0O@>@07h3E`0002@0002@0000
OP<0098000200`000P000900OP>B0803E`0002@000250000N0<008L0001i0`000P0008D0N0>707T3
E`0002@000270000N@<008T0001j0`000P0008L0N@>907X3E`0002@000290000NP<008h0001l0`00
0P0008T0NP>>07`3E`0002@0002>0000O0<00900001m0`000P0008h0O0>@07d3E`0002@0002@0000
O@<00940001n0`000P000900O@>A07h3E`0002@0002A0000OP<0098000200`000P000940OP>B0803
E`0002@0002B0000P0<0098000240`000P000980P0>B08@3E`0002@0002@0000Q0<0098000260`00
0P000980Q0>@08H3E`0002@0002=0000QP<0090000260`000P000900QP>=08H3E`0002@0002;0000
QP<008d000260`000P0008d0QP>;08H3E`0002@000280000QP<008/000260`000P0008/0QP>808H3
E`0002@000260000Q0<008P000260`000P0008P0QP>608@3E`0002@000250000P@<008H000240`00
0P0008H0Q0>50843E`0002@000250000P@<008D000260`000P0008D0P@>508H3E`0002@000250000
Q0<008H000260`000P0008D0QP>608@3E`0002@0002H0000O`<00:<0001o0`000P0009P0O`>S07l3
E`0002@0002S0000O@<00:<0001o0`000P000:<0O`>S07d3E`0002@0002R0000N`<00:<0001m0`00
0P000:<0O@>R07/3E`0002@0002Q0000NP<00:80001k0`000P000:80N`>Q07X3E`0002@0002O0000
N@<00:40001j0`000P000:40NP>O07T3E`0002@0002M0000N@<009l0001i0`000P0009l0N@>M07T3
E`0002@0002J0000N@<009d0001j0`000P0009d0N@>J07X3E`0002@0002H0000NP<009X0001l0`00
0P0009X0NP>H07`3E`0002@0002G0000O0<009P0001o0`000P0009P0O0>G07l3E`0002@0002G0000
O`<009L000210`000P0009L0O`>G0843E`0002@0002G0000P@<009P000240`000P0009L0P@>H08@3
E`0002@0002H0000Q0<009X000260`000P0009P0Q0>J08H3E`0002@0002J0000QP<009d000260`00
0P0009X0QP>M08H3E`0002@0002M0000QP<009h000260`000P0009d0QP>N08H3E`0002@0002N0000
QP<00:4000260`000P0009h0QP>Q08H3E`0002@0002Q0000Q0<00:<000260`000P000:40QP>S08@3
E`0002@0002R0000O0<00:80001o0`000P000:80O`>R07`3E`0002@0002Q0000NP<00:80001l0`00
0P000:80O0>Q07X3E`0002@0002K0000N@<009d0001j0`000P0009d0N@>K07X3E`0002@0002I0000
NP<009/0001l0`000P0009/0NP>I07`3E`0002@0002H0000O0<009T0001o0`000P0009T0O0>H07l3
E`0002@0002H0000O`<009P000210`000P0009P0O`>H0843E`0002@0002H0000P@<009T000240`00
0P0009P0P@>I08@3E`0002@0002I0000Q0<009/000260`000P0009T0Q0>K08H3E`0002@0002K0000
QP<009d000260`000P0009/0QP>M08H3E`0002@0002Y0000L`<00:T000260`000P000:T0L`>Y08H3
E`0002@0002Z0000L`<00:X000260`000P000:X0L`>Z08H3E`0002@0002V0000L`<00:X0001c0`00
0P000:H0L`>Z07<3E`0002@0002V0000QP<00:d000260`000P000:H0QP>]08H3E`0002@0002c0000
L`<00;<000260`000P000;<0L`>c08H3E`0002@0002d0000L`<00;@000260`000P000;@0L`>d08H3
E`0002@0002`0000L`<00;@0001c0`000P000;00L`>d07<3E`0002@0002`0000QP<00;L000260`00
0P000;00QP>g08H3E`0002@0002k0000K`<00;d0001a0`000P000;/0K`>m0743E`0002@0002m0000
L@<00;l0001d0`000P000;d0L@>o07@3E`0002@0002o0000M0<00<40001h0`000P000;l0M0?107P3
E`0002@000310000N0<00<40001l0`000P000<40N0?107`3E`0002@000310000O0<00<4000200`00
0P000<40O0?10803E`0002@000310000P0<00<4000250`000P000<40P0?108D3E`0002@0002o0000
Q@<00<4000280`000P000<40Q@>o08P3E`0002@0002m0000R0<00;l0002;0`000P000;l0R0>m08/3
E`0002@0002k0000R`<00;d0002=0`000P000;d0R`>k08d3E`0002@0002m0000L@<00;l0001e0`00
0P000;d0L@>o07D3E`0002@0002o0000M@<00<00001h0`000P000;l0M@?007P3E`0002@000300000
N0<00<40001l0`000P000<00N0?107`3E`0002@000310000O0<00<4000200`000P000<40O0?10803
E`0002@000300000P0<00<4000250`000P000<40P0?008D3E`0002@0002o0000Q@<00<0000270`00
0P000<00Q@>o08L3E`0002@0002m0000Q`<00;l0002;0`000P000;l0Q`>m08/3E`0002@0001<0000
B@0004`0001I00000P0004`0B@1<05T0E`0002@000140000D@0005@0001A00000P0004@0D@1D0540
E`0002@0001N0000B@000600001:00000P0005h0BP1P04T0E`0002@0001P0000AP00068000190000
0P000600B@1R04H0E`0002@0001R0000AP000680001I00000P000680AP1R05T0E`0002@0001Q0000
A`000640001I00000P000640A`1Q05T0E`0002@0001N0000F@0006H0001I00000P0005h0F@1V05T0
E`0002@0001l0000@P0007d0001400000P0007d0@P1l04@0E`0002@0001j0000A00007`000170000
0P0007`0A01j04L0E`0002@0001h0000A`0007X0001;00000P0007X0A`1h04/0E`0002@0001g0000
B`0007P0001?00000P0007P0B`1g04l0E`0002@0001g0000C`0007L0001C00000P0007L0C`1g05<0
E`0002@0001g0000D`0007P0001H00000P0007L0D`1h05P0E`0002@0001h0000F00007X0001K0000
0P0007P0F01j05/0E`0002@0001j0000F`0007`0001N00000P0007X0F`1l05h0E`0002@0001l0000
GP0007d0001P00000P0007`0GP1m0600E`0002@0001j0000A00007`0001800000P0007`0A01j04P0
E`0002@0001i0000B00007X0001;00000P0007X0B01i04/0E`0002@0001h0000B`0007T0001?0000
0P0007T0B`1h04l0E`0002@0001h0000C`0007P0001C00000P0007P0C`1h05<0E`0002@0001h0000
D`0007T0001H00000P0007P0D`1i05P0E`0002@0001i0000F00007X0001J00000P0007T0F01j05X0
E`0002@0001j0000FP0007`0001N00000P0007X0FP1l05h0E`0002@000250000AP0008D0001I0000
0P0008D0AP2505T0E`0002@000260000AP0008H0001I00000P0008H0AP2605T0E`0002@000220000
AP0008d0001600000P000880AP2=04H0E`0002@0002=0000AP000900001700000P0008d0AP2@04L0
E`0002@0002@0000A`000940001800000P000900A`2A04P0E`0002@0002A0000B0000940001:0000
0P000940B02A04X0E`0002@0002A0000BP000940001;00000P000940BP2A04/0E`0002@0002A0000
B`000940001=00000P000940B`2A04d0E`0002@0002@0000C@000940001>00000P000940C@2@04h0
E`0002@0002=0000CP000900001?00000P000900CP2=04l0E`0002@0002=0000AP0008l000170000
0P0008d0AP2?04L0E`0002@0002?0000A`000900001800000P0008l0A`2@04P0E`0002@0002@0000
B0000940001:00000P000900B02A04X0E`0002@0002A0000BP000940001;00000P000940BP2A04/0
E`0002@0002@0000B`000940001=00000P000940B`2@04d0E`0002@0002?0000C@000900001>0000
0P000900C@2?04h0E`0002@0002=0000CP0008l0001?00000P0008l0CP2=04l0E`0002@000260000
C`0008d0001?00000P0008H0C`2=04l0E`0002@0002=0000C`000900001@00000P0008d0C`2@0500
E`0002@0002@0000D0000940001A00000P000900D02A0540E`0002@0002A0000D@000940001C0000
0P000940D@2A05<0E`0002@0002A0000D`000940001F00000P000940D`2A05H0E`0002@0002A0000
EP000940001H00000P000940EP2A05P0E`0002@0002@0000F0000940001I00000P000940F02@05T0
E`0002@0002=0000F@000900001I00000P000900F@2=05T0E`0002@000220000F@0008d0001I0000
0P0008d0F@2205T0E`0002@0002=0000C`0008l0001@00000P0008d0C`2?0500E`0002@0002?0000
D0000900001A00000P0008l0D02@0540E`0002@0002@0000D@000940001C00000P000900D@2A05<0
E`0002@0002A0000D`000940001F00000P000940D`2A05H0E`0002@0002@0000EP000940001H0000
0P000940EP2@05P0E`0002@0002?0000F0000900001I00000P000900F02?05T0E`0002@0002=0000
F@0008l0001I00000P0008l0F@2=05T0E`0002@0002I0000C00009T0001G00000P0009T0C02I05L0
E`0002@0002I0000E`0009X0001I00000P0009T0E`2J05T0E`0002@0002J0000F@0009d0001I0000
0P0009X0F@2M05T0E`0002@0002M0000F@0009h0001I00000P0009d0F@2N05T0E`0002@0002N0000
F@000:40001I00000P0009h0F@2Q05T0E`0002@0002Q0000E`000:<0001I00000P000:40F@2S05L0
E`0002@0002J0000C00009X0001G00000P0009X0C02J05L0E`0002@0002J0000E`0009/0001I0000
0P0009X0E`2K05T0E`0002@0002K0000F@0009d0001I00000P0009/0F@2M05T0E`0002@0002S0000
C0000:<0001I00000P000:<0C02S05T0E`0002@0002T0000C0000:@0001I00000P000:@0C02T05T0
E`0002@0002F0000C00009X0001<00000P0009H0C02J04`0E`0002@0002P0000C0000:@0001<0000
0P000:00C02T04`0E`0002@0002S0000F@000:H0001I00000P000:<0F@2V05T0E`0002@0002[0000
C0000;00001I00000P000:/0C02`05T0E`0002@0002/0000C0000;00001H00000P000:`0C02`05P0
E`0002@0002`0000C0000;H0001I00000P000;H0C02`05T0E`0002@0002_0000F@000;00001M0000
0P000;00F@2_05d0E`0002@0002]0000G@000:l0001O00000P000:l0G@2]05l0E`0002@0002[0000
G`000:d0001P00000P000:d0G`2[0600E`0002@0002Z0000H0000:/0001P00000P000:/0H02Z0600
E`0002@0002Y0000G`000:X0001P00000P000:X0H02Y05l0E`0002@0002Y0000GP000:X0001O0000
0P000:T0G`2Z05h0E`0002@0002Z0000GP000:/0001O00000P000:X0GP2[05l0E`0002@0002Y0000
C0000:l0001<00000P000:T0C02_04`0E`0002@0002b0000C0000;P0001<00000P000;80C02h04`0
E`0002@0002k0000@P000;d0001400000P000;/0@P2m04@0E`0002@0002m0000A0000;l000170000
0P000;d0A02o04L0E`0002@0002o0000A`000<40001;00000P000;l0A`3104/0E`0002@000310000
B`000<40001?00000P000<40B`3104l0E`0002@000310000C`000<40001C00000P000<40C`3105<0
E`0002@000310000D`000<40001H00000P000<40D`3105P0E`0002@0002o0000F0000<40001K0000
0P000<40F02o05/0E`0002@0002m0000F`000;l0001N00000P000;l0F`2m05h0E`0002@0002k0000
GP000;d0001P00000P000;d0GP2k0600E`0002@0002m0000A0000;l0001800000P000;d0A02o04P0
E`0002@0002o0000B0000<00001;00000P000;l0B03004/0E`0002@000300000B`000<40001?0000
0P000<00B`3104l0E`0002@000310000C`000<40001C00000P000<40C`3105<0E`0002@000300000
D`000<40001H00000P000<40D`3005P0E`0002@0002o0000F0000<00001J00000P000<00F02o05X0
E`0002@0002m0000FP000;l0001N00000P000;l0FP2m05h0E`0002@0003E0000Q@<00=D0002=0`00
0P000=D0Q@?E08d3E`0002@0000;0@00Q@<000/1002=0`000P0000/1Q@<;0Hd3E`0002@000100@00
Q@<00401002=0`000P000401Q@=00Hd3E`0002@0001f0@00Q@<007H1002=0`000P0007H1Q@=f0Hd3
E`0002@0002/0@00Q@<00:`1002=0`000P000:`1Q@>/0Hd3E`0002@0003R0@00Q@<00>81002=0`00
0P000>81Q@?R0Hd3E`0002@0000H0P00Q@<001P2002=0`000P0001P2Q@<H0Xd3E`0002@0001>0P00
Q@<004h2002=0`000P0004h2Q@=>0Xd3E`0002@000240P00Q@<008@2002=0`000P0008@2Q@>40Xd3
E`0002@0002j0P00Q@<00;X2002=0`000P000;X2Q@>j0Xd3E`0002@0003`0P00Q@<00?02002=0`00
0P000?02Q@?`0Xd3E`0002@0000V0`00Q@<002H3002=0`000P0002H3Q@<V0hd3E`0002@0001L0`00
Q@<005`3002=0`000P0005`3Q@=L0hd3E`0002@0002B0`00Q@<00983002=0`000P000983Q@>B0hd3
E`0002@000380`00Q@<00<P3002=0`000P000<P3Q@?80hd3E`0002@0003n0`00Q@<00?h3002=0`00
0P000?h3Q@?n0hd3E`0002@0000d1000Q@<003@4002=0`000P0003@4Q@<d18d3E`0002@0001Y1000
Q@<006T4002=0`000P0006T4Q@=Y18d3E`0002@0002O1000Q@<009l4002=0`000P0009l4Q@>O18d3
E`0002@0003E1000Q@<00=D4002=0`000P000=D4Q@?E18d3E`0002@0002h0P00b`<00;P2003N0`00
0P000;P2b`>h0]h3E`0002@0002h0P00b`<00;P2003N0`000P000;P2b`>h0]h3E`0002@0002a0P00
b`<00;82003@0`000P000;82b`>a0]03E`0002@0002a0P00b`<00;42003@0`000P000;42d0>a0//3
E`0002@0002a0P00b`<00;l2003;0`000P000;42b`>o0//3E`0002@0002o0P00b`<00;l2003@0`00
0P000;l2b`>o0]03E`0002@0002n0P00b`<00;l2003@0`000P000;l2d0>n0//3E`0002@0002e0P00
gP<00;/2003N0`000P000;D2gP>k0]h3E`0002@000340P00b`<00<D2003<0`000P000<D2b`?40/`3
E`0002@000340P00c0<00<D2003=0`000P000<@2c0?50/d3E`0002@000350P00c0<00<H2003=0`00
0P000<D2c@?60/`3E`0002@000350P00b`<00<H2003<0`000P000<H2c0?50//3E`0002@000350P00
d@<00<D2003N0`000P000<D2d@?50]h3E`0002@000360P00d@<00<H2003N0`000P000<H2d@?60]h3
E`0002@000320P00d@<00<H2003A0`000P000<82d@?60]43E`0002@000320P00gP<00<T2003N0`00
0P000<82gP?90]h3E`0002@0003?0P00d@<00<l2003N0`000P000<l2d@??0]h3E`0002@0003@0P00
d@<00=02003N0`000P000=02d@?@0]h3E`0002@0003@0P00dP<00=82003D0`000P000=02e0?B0]83
E`0002@0003B0P00d@<00=@2003B0`000P000=82dP?D0]43E`0002@0003D0P00d@<00=H2003A0`00
0P000=@2d@?F0]43E`0002@0003F0P00d@<00=T2003B0`000P000=H2d@?I0]83E`0002@0003I0P00
dP<00=X2003D0`000P000=T2dP?J0]@3E`0002@0003J0P00e0<00=X2003N0`000P000=X2e0?J0]h3
E`0002@0003F0P00d@<00=P2003B0`000P000=H2d@?H0]83E`0002@0003H0P00dP<00=T2003D0`00
0P000=P2dP?I0]@3E`0002@0003I0P00e0<00=T2003N0`000P000=T2e0?I0]h3E`0002@0003J0P00
dP<00=/2003D0`000P000=X2e0?K0]83E`0002@0003K0P00d@<00=h2003B0`000P000=/2dP?N0]43
E`0002@0003N0P00d@<00>02003A0`000P000=h2d@?P0]43E`0002@0003P0P00d@<00><2003B0`00
0P000>02d@?S0]83E`0002@0003S0P00dP<00>@2003D0`000P000><2dP?T0]@3E`0002@0003T0P00
e0<00>@2003N0`000P000>@2e0?T0]h3E`0002@0003P0P00d@<00>82003B0`000P000>02d@?R0]83
E`0002@0003R0P00dP<00><2003D0`000P000>82dP?S0]@3E`0002@0003S0P00e0<00><2003N0`00
0P000><2e0?S0]h3E`0002@0003<0P00d@<00=02003A0`000P000<`2d@?@0]43E`0002@0003<0P00
gP<00=<2003N0`000P000<`2gP?C0]h3E`0002@0003F0P00gP<00=`2003N0`000P000=H2gP?L0]h3
E`0002@0003P0P00gP<00>H2003N0`000P000>02gP?V0]h3E`0002@0003/0P00e`<00?L2003G0`00
0P000>`2e`?g0]L3E`0002@0003g0P00e@<00?L2003G0`000P000?L2e`?g0]D3E`0002@0003f0P00
d`<00?L2003E0`000P000?L2e@?f0]<3E`0002@0003e0P00dP<00?H2003C0`000P000?H2d`?e0]83
E`0002@0003c0P00d@<00?D2003B0`000P000?D2dP?c0]43E`0002@0003a0P00d@<00?<2003A0`00
0P000?<2d@?a0]43E`0002@0003^0P00d@<00?42003B0`000P000?42d@?^0]83E`0002@0003/0P00
dP<00>h2003D0`000P000>h2dP?/0]@3E`0002@0003[0P00e0<00>`2003G0`000P000>`2e0?[0]L3
E`0002@0003[0P00e`<00>/2003I0`000P000>/2e`?[0]T3E`0002@0003[0P00f@<00>`2003L0`00
0P000>/2f@?/0]`3E`0002@0003/0P00g0<00>h2003N0`000P000>`2g0?^0]h3E`0002@0003^0P00
gP<00?42003N0`000P000>h2gP?a0]h3E`0002@0003a0P00gP<00?82003N0`000P000?42gP?b0]h3
E`0002@0003b0P00gP<00?D2003N0`000P000?82gP?e0]h3E`0002@0003e0P00g0<00?L2003N0`00
0P000?D2gP?g0]`3E`0002@0003f0P00e0<00?H2003G0`000P000?H2e`?f0]@3E`0002@0003e0P00
dP<00?H2003D0`000P000?H2e0?e0]83E`0002@0003_0P00d@<00?42003B0`000P000?42d@?_0]83
E`0002@0003]0P00dP<00>l2003D0`000P000>l2dP?]0]@3E`0002@0003/0P00e0<00>d2003G0`00
0P000>d2e0?/0]L3E`0002@0003/0P00e`<00>`2003I0`000P000>`2e`?/0]T3E`0002@0003/0P00
f@<00>d2003L0`000P000>`2f@?]0]`3E`0002@0003]0P00g0<00>l2003N0`000P000>d2g0?_0]h3
E`0002@0003_0P00gP<00?42003N0`000P000>l2gP?a0]h3E`0002@0003A0000WP<00=<0002O0`00
0P000=40W`?C09h3E`0002@0003C0000V`<00=D0002N0`000P000=<0WP?E09/3E`0002@0003E0000
V`<00=D0002^0`000P000=D0V`?E0:h3E`0002@0003D0000W0<00=@0002^0`000P000=@0W0?D0:h3
E`0002@0003A0000[P<00=T0002^0`000P000=40[P?I0:h3E`0002@000050@00W`<000H1002P0`00
0P0000D1W`<60J03E`0002@000050@00X0<000H1002P0`000P0000H1X0<50J03E`0002@000040@00
X0<000D1002P0`000P0000D1X0<40J03E`0002@000040@00W`<000@1002P0`000P0000@1X0<40Il3
E`0002@000040@00W@<000D1002O0`000P0000@1W`<50Id3E`0002@000050@00W0<000H1002M0`00
0P0000D1W@<60I`3E`0002@000060@00V`<000T1002L0`000P0000H1W0<90I/3E`0002@000090@00
V`<000`1002K0`000P0000T1V`<<0I/3E`0002@0000<0@00V`<000l1002L0`000P0000`1V`<?0I`3
E`0002@0000?0@00W0<00101002M0`000P0000l1W0<@0Id3E`0002@0000@0@00W@<00141002O0`00
0P000101W@<A0Il3E`0002@0000A0@00W`<00141002P0`000P000141W`<A0J03E`0002@0000@0@00
X0<00141002R0`000P000141X0<@0J83E`0002@0000=0@00XP<00101002T0`000P000101XP<=0J@3
E`0002@000090@00Y0<000d1002V0`000P0000d1Y0<90JH3E`0002@000070@00YP<000T1002W0`00
0P0000T1YP<70JL3E`0002@000050@00Y`<000L1002Y0`000P0000L1Y`<50JT3E`0002@000040@00
Z@<000D1002/0`000P0000D1Z@<40J`3E`0002@000040@00[0<000@1002^0`000P0000@1[0<40Jh3
E`0002@0000<0@00V`<000h1002L0`000P0000`1V`<>0I`3E`0002@0000>0@00W0<000l1002M0`00
0P0000h1W0<?0Id3E`0002@0000?0@00W@<00101002O0`000P0000l1W@<@0Il3E`0002@0000@0@00
W`<00101002P0`000P000101W`<@0J03E`0002@0000?0@00X0<00101002R0`000P000101X0<?0J83
E`0002@0000<0@00XP<000l1002T0`000P0000l1XP<<0J@3E`0002@000090@00Y0<000`1002V0`00
0P0000`1Y0<90JH3E`0002@000040@00[0<000D1002]0`000P0000@1[@<50J`3E`0002@000050@00
[0<000L1002/0`000P0000D1[0<70J`3E`0002@000070@00[0<000/1002^0`000P0000L1[0<;0Jh3
E`0002@0000;0@00[P<000h1002^0`000P0000/1[P<>0Jh3E`0002@0000>0@00[@<00101002^0`00
0P0000h1[P<@0Jd3E`0002@0000@0@00[0<00141002]0`000P000101[@<A0J`3E`0002@000070@00
[0<000/1002^0`000P0000L1[0<;0Jh3E`0002@0000;0@00[P<000l1002^0`000P0000/1[P<?0Jh3
E`0002@0000?0@00[P<00101002^0`000P0000l1[P<@0Jh3E`0002@0000@0@00[0<00141002^0`00
0P000101[P<A0J`3E`0002@0000A0@00ZP<00141002/0`000P000141[0<A0JX3E`0002@0000k0@00
W`<003`1002P0`000P0003/1W`<l0J03E`0002@0000k0@00X0<003`1002P0`000P0003`1X0<k0J03
E`0002@0000j0@00X0<003/1002P0`000P0003/1X0<j0J03E`0002@0000j0@00W`<003X1002P0`00
0P0003X1X0<j0Il3E`0002@0000j0@00W@<003/1002O0`000P0003X1W`<k0Id3E`0002@0000k0@00
W0<003`1002M0`000P0003/1W@<l0I`3E`0002@0000l0@00V`<003l1002L0`000P0003`1W0<o0I/3
E`0002@0000o0@00V`<00481002K0`000P0003l1V`=20I/3E`0002@000120@00V`<004D1002L0`00
0P000481V`=50I`3E`0002@000150@00W0<004H1002N0`000P0004D1W0=60Ih3E`0002@000160@00
WP<004H1002P0`000P0004H1WP=60J03E`0002@000150@00X0<004H1002R0`000P0004H1X0=50J83
E`0002@000120@00XP<004D1002S0`000P0004D1XP=20J<3E`0002@000100@00X`<00481002S0`00
0P000481X`=00J<3E`0002@000120@00V`<004@1002L0`000P000481V`=40I`3E`0002@000140@00
W0<004D1002N0`000P0004@1W0=50Ih3E`0002@000150@00WP<004D1002P0`000P0004D1WP=50J03
E`0002@000140@00X0<004D1002R0`000P0004D1X0=40J83E`0002@000120@00XP<004@1002S0`00
0P0004@1XP=20J<3E`0002@000120@00X`<004@1002T0`000P000481X`=40J@3E`0002@000140@00
Y0<004H1002V0`000P0004@1Y0=60JH3E`0002@000160@00YP<004L1002X0`000P0004H1YP=70JP3
E`0002@000170@00Z0<004L1002[0`000P0004L1Z0=70J/3E`0002@000160@00Z`<004L1002]0`00
0P0004L1Z`=60Jd3E`0002@000150@00[@<004H1002^0`000P0004H1[@=50Jh3E`0002@000120@00
[P<004D1002^0`000P0004D1[P=20Jh3E`0002@0000o0@00[P<00481002^0`000P000481[P<o0Jh3
E`0002@0000l0@00[P<003l1002^0`000P0003l1[P<l0Jh3E`0002@0000k0@00[@<003`1002^0`00
0P0003`1[P<k0Jd3E`0002@0000j0@00Z`<003/1002]0`000P0003/1[@<j0J/3E`0002@0000j0@00
ZP<003X1002[0`000P0003X1Z`<j0JX3E`0002@0000j0@00Z@<003/1002Z0`000P0003X1ZP<k0JT3
E`0002@0000k0@00Z@<003`1002Z0`000P0003/1Z@<l0JX3E`0002@0000k0@00ZP<003`1002[0`00
0P0003`1ZP<k0J/3E`0002@000150@00Y@<004H1002X0`000P0004D1Y@=60JP3E`0002@000160@00
Z0<004H1002[0`000P0004H1Z0=60J/3E`0002@000150@00Z`<004H1002]0`000P0004H1Z`=50Jd3
E`0002@000140@00[@<004D1002^0`000P0004D1[@=40Jh3E`0002@000120@00[P<004@1002^0`00
0P0004@1[P=20Jh3E`0002@0001h0@00W@<007P1002^0`000P0007P1W@=h0Jh3E`0002@0001i0@00
V`<007T1002^0`000P0007T1V`=i0Jh3E`0002@0001_0@00V`<007T1002Y0`000P0007T1V`=_0JT3
E`0002@0001_0@00Z@<007h1002Y0`000P0006l1Z@=n0JT3E`0002@0001f0@00[P<007`1002^0`00
0P0007H1[P=l0Jh3E`0002@0002U0@00V`<00:L1002T0`000P000:L1V`>U0J@3E`0002@0002U0@00
XP<00:L1002T0`000P000:D1Y0>W0J83E`0002@0002W0@00X@<00:X1002R0`000P000:L1XP>Z0J43
E`0002@0002Z0@00X@<00:`1002Q0`000P000:X1X@>/0J43E`0002@0002/0@00X@<00:l1002R0`00
0P000:`1X@>_0J83E`0002@0002_0@00XP<00;41002T0`000P000:l1XP>a0J@3E`0002@0002a0@00
Y0<00;81002W0`000P000;41Y0>b0JL3E`0002@0002b0@00Y`<00;81002Y0`000P000;81Y`>b0JT3
E`0002@0002a0@00Z@<00;81002/0`000P000;81Z@>a0J`3E`0002@0002_0@00[0<00;41002^0`00
0P000;41[0>_0Jh3E`0002@0002/0@00[P<00:l1002^0`000P000:l1[P>/0Jh3E`0002@0002Z0@00
[P<00:`1002^0`000P000:`1[P>Z0Jh3E`0002@0002W0@00[P<00:X1002^0`000P000:X1[P>W0Jh3
E`0002@0002V0@00[@<00:L1002^0`000P000:L1[P>V0Jd3E`0002@0002U0@00Z`<00:H1002]0`00
0P000:H1[@>U0J/3E`0002@0002U0@00ZP<00:D1002[0`000P000:D1Z`>U0JX3E`0002@0002U0@00
Z@<00:H1002Z0`000P000:D1ZP>V0JT3E`0002@0002V0@00Z@<00:L1002Z0`000P000:H1Z@>W0JX3
E`0002@0002V0@00ZP<00:L1002[0`000P000:L1ZP>V0J/3E`0002@0002/0@00X@<00:h1002R0`00
0P000:`1X@>^0J83E`0002@0002^0@00XP<00;01002T0`000P000:h1XP>`0J@3E`0002@0002`0@00
Y0<00;41002W0`000P000;01Y0>a0JL3E`0002@0002a0@00Y`<00;41002Y0`000P000;41Y`>a0JT3
E`0002@0002`0@00Z@<00;41002/0`000P000;41Z@>`0J`3E`0002@0002^0@00[0<00;01002^0`00
0P000;01[0>^0Jh3E`0002@0002/0@00[P<00:h1002^0`000P000:h1[P>/0Jh3E`0002@0002W0@00
V`<00;01002K0`000P000:L1V`>`0I/3E`0002@0002W0@00W0<00:/1002L0`000P000:L1W0>[0I`3
E`0002@0002[0@00V`<00;01002L0`000P000:/1W0>`0I/3E`0002@0003U0@00WP<00>H1002O0`00
0P000>H1WP?U0Il3E`0002@0003U0@00W`<00>H1002P0`000P000>D1W`?V0J03E`0002@0003V0@00
W`<00>L1002P0`000P000>H1X0?W0Il3E`0002@0003W0@00WP<00>L1002O0`000P000>L1W`?W0Ih3
E`0002@0003V0@00W0<00>L1002N0`000P000>L1WP?V0I`3E`0002@0003T0@00V`<00>H1002L0`00
0P000>H1W0?T0I/3E`0002@0003Q0@00V`<00>@1002K0`000P000>@1V`?Q0I/3E`0002@0003O0@00
V`<00>41002L0`000P000>41V`?O0I`3E`0002@0003M0@00W0<00=l1002N0`000P000=l1W0?M0Ih3
E`0002@0003L0@00WP<00=d1002P0`000P000=d1WP?L0J03E`0002@0003K0@00X0<00=`1002S0`00
0P000=`1X0?K0J<3E`0002@0003K0@00X`<00=/1002Y0`000P000=/1X`?K0JT3E`0002@0003K0@00
Z@<00=`1002/0`000P000=/1Z@?L0J`3E`0002@0003L0@00[0<00=h1002^0`000P000=`1[0?N0Jh3
E`0002@0003N0@00[P<00>41002^0`000P000=h1[P?Q0Jh3E`0002@0003Q0@00[P<00>81002^0`00
0P000>41[P?R0Jh3E`0002@0003R0@00[P<00>D1002^0`000P000>81[P?U0Jh3E`0002@0003U0@00
[0<00>L1002^0`000P000>D1[P?W0J`3E`0002@0003W0@00Z@<00>P1002/0`000P000>L1[0?X0JT3
E`0002@0003X0@00Z0<00>P1002Y0`000P000>P1Z@?X0JP3E`0002@0003W0@00Y@<00>P1002X0`00
0P000>P1Z0?W0JD3E`0002@0003U0@00X`<00>L1002U0`000P000>L1Y@?U0J<3E`0002@0003R0@00
XP<00>D1002S0`000P000>D1X`?R0J83E`0002@0003Q0@00XP<00>81002R0`000P000>81XP?Q0J83
E`0002@0003O0@00XP<00>41002S0`000P000>41XP?O0J<3E`0002@0003M0@00X`<00=l1002U0`00
0P000=l1X`?M0JD3E`0002@0003L0@00Y@<00=d1002X0`000P000=d1Y@?L0JP3E`0002@0003P0@00
V`<00>41002L0`000P000>41V`?P0I`3E`0002@0003N0@00W0<00>01002N0`000P000>01W0?N0Ih3
E`0002@0003M0@00WP<00=h1002P0`000P000=h1WP?M0J03E`0002@0003L0@00X0<00=d1002S0`00
0P000=d1X0?L0J<3E`0002@0003L0@00X`<00=`1002Y0`000P000=`1X`?L0JT3E`0002@0003L0@00
Z@<00=d1002/0`000P000=`1Z@?M0J`3E`0002@0003M0@00[0<00=l1002^0`000P000=d1[0?O0Jh3
E`0002@0003O0@00[P<00>41002^0`000P000=l1[P?Q0Jh3E`0002@0003R0@00[P<00>@1002^0`00
0P000>81[P?T0Jh3E`0002@0003T0@00[0<00>H1002^0`000P000>@1[P?V0J`3E`0002@0003V0@00
Z@<00>L1002/0`000P000>H1[0?W0JT3E`0002@0003W0@00Z0<00>L1002Y0`000P000>L1Z@?W0JP3
E`0002@0003V0@00Y@<00>L1002X0`000P000>L1Z0?V0JD3E`0002@0003T0@00X`<00>H1002U0`00
0P000>H1Y@?T0J<3E`0002@0003R0@00XP<00>@1002S0`000P000>@1X`?R0J83E`0002@0000A0P00
V`<00142002P0`000P000142V`<A0Z03E`0002@0000A0P00W@<00182002O0`000P000142W`<B0Yd3
E`0002@0000B0P00V`<001@2002M0`000P000182W@<D0Y/3E`0002@0000D0P00V`<001H2002K0`00
0P0001@2V`<F0Y/3E`0002@0000F0P00V`<001X2002N0`000P0001H2V`<J0Yh3E`0002@0000J0P00
WP<001`2002N0`000P0001X2WP<L0Yh3E`0002@0000L0P00W@<001d2002N0`000P0001`2WP<M0Yd3
E`0002@0000M0P00V`<001h2002M0`000P0001d2W@<N0Y/3E`0002@0000B0P00W0<001@2002M0`00
0P000182W@<D0Y`3E`0002@0000D0P00W0<001H2002L0`000P0001@2W0<F0Y`3E`0002@0000F0P00
W0<001X2002N0`000P0001H2W0<J0Yh3E`0002@0000N0P00V`<001h2002N0`000P0001h2V`<N0Yh3
E`0002@0000M0P00WP<001h2002P0`000P0001h2WP<M0Z03E`0002@0000I0P00X0<001d2002U0`00
0P0001d2X0<I0ZD3E`0002@0000H0P00Y@<001T2002W0`000P0001T2Y@<H0ZL3E`0002@0000G0P00
Y`<001P2002Z0`000P0001P2Y`<G0ZX3E`0002@0000G0P00ZP<001L2002^0`000P0001L2ZP<G0Zh3
E`0002@0000H0P00X0<001d2002U0`000P0001d2X0<H0ZD3E`0002@0000G0P00Y@<001P2002W0`00
0P0001P2Y@<G0ZL3E`0002@0000G0P00Y`<001L2002Z0`000P0001L2Y`<G0ZX3E`0002@0000G0P00
ZP<001L2002^0`000P0001L2ZP<G0Zh3E`0002@000190P00V`<004`2002L0`000P0004`2V`=90Y`3
E`0002@000180P00W0<004T2002N0`000P0004T2W0=80Yh3E`0002@000180P00WP<004P2002P0`00
0P0004P2WP=80Z03E`0002@000180P00X0<004T2002R0`000P0004P2X0=90Z83E`0002@000190P00
XP<004`2002S0`000P0004T2XP=<0Z<3E`0002@0001<0P00X`<004l2002S0`000P0004`2X`=?0Z<3
E`0002@0001?0P00XP<00582002S0`000P0004l2X`=B0Z83E`0002@0001B0P00X0<005<2002R0`00
0P000582XP=C0Z03E`0002@0001C0P00WP<005<2002P0`000P0005<2X0=C0Yh3E`0002@0001B0P00
W0<005<2002N0`000P0005<2WP=B0Y`3E`0002@0001?0P00V`<00582002L0`000P000582W0=?0Y/3
E`0002@0001<0P00V`<004l2002K0`000P0004l2V`=<0Y/3E`0002@0001:0P00V`<004`2002L0`00
0P0004`2V`=:0Y`3E`0002@000190P00W0<004X2002N0`000P0004X2W0=90Yh3E`0002@000190P00
WP<004T2002P0`000P0004T2WP=90Z03E`0002@000190P00X0<004X2002R0`000P0004T2X0=:0Z83
E`0002@0001:0P00XP<004`2002S0`000P0004X2XP=<0Z<3E`0002@0001?0P00XP<00542002S0`00
0P0004l2X`=A0Z83E`0002@0001A0P00X0<00582002R0`000P000542XP=B0Z03E`0002@0001B0P00
WP<00582002P0`000P000582X0=B0Yh3E`0002@0001A0P00W0<00582002N0`000P000582WP=A0Y`3
E`0002@0001?0P00V`<00542002L0`000P000542W0=?0Y/3E`0002@000190P00X`<004`2002T0`00
0P0004`2X`=90Z@3E`0002@000180P00Y0<004T2002U0`000P0004T2Y0=80ZD3E`0002@000170P00
Y@<004P2002W0`000P0004P2Y@=70ZL3E`0002@000170P00Y`<004L2002[0`000P0004L2Y`=70Z/3
E`0002@000170P00Z`<004P2002]0`000P0004L2Z`=80Zd3E`0002@000180P00[@<004T2002^0`00
0P0004P2[@=90Zh3E`0002@000190P00[P<004`2002^0`000P0004T2[P=<0Zh3E`0002@0001<0P00
[P<004l2002^0`000P0004`2[P=?0Zh3E`0002@0001?0P00[P<00582002^0`000P0004l2[P=B0Zh3
E`0002@0001B0P00[@<005<2002^0`000P000582[P=C0Zd3E`0002@0001C0P00Z`<005@2002]0`00
0P0005<2[@=D0Z/3E`0002@0001D0P00Y`<005@2002[0`000P0005@2Z`=D0ZL3E`0002@0001C0P00
Y@<005@2002W0`000P0005@2Y`=C0ZD3E`0002@0001B0P00Y0<005<2002U0`000P0005<2Y@=B0Z@3
E`0002@0001?0P00X`<00582002T0`000P000582Y0=?0Z<3E`0002@0001:0P00X`<004`2002T0`00
0P0004`2X`=:0Z@3E`0002@000190P00Y0<004X2002U0`000P0004X2Y0=90ZD3E`0002@000180P00
Y@<004T2002W0`000P0004T2Y@=80ZL3E`0002@000180P00Y`<004P2002[0`000P0004P2Y`=80Z/3
E`0002@000180P00Z`<004T2002]0`000P0004P2Z`=90Zd3E`0002@000190P00[@<004X2002^0`00
0P0004T2[@=:0Zh3E`0002@0001:0P00[P<004`2002^0`000P0004X2[P=<0Zh3E`0002@0001?0P00
[P<00542002^0`000P0004l2[P=A0Zh3E`0002@0001A0P00[@<00582002^0`000P000542[P=B0Zd3
E`0002@0001B0P00Z`<005<2002]0`000P000582[@=C0Z/3E`0002@0001C0P00Y`<005<2002[0`00
0P0005<2Z`=C0ZL3E`0002@0001B0P00Y@<005<2002W0`000P0005<2Y`=B0ZD3E`0002@0001A0P00
Y0<00582002U0`000P000582Y@=A0Z@3E`0002@0001?0P00X`<00542002T0`000P000542Y0=?0Z<3
E`0002@000280P00X@<008T2002T0`000P0008T2X@>80Z@3E`0002@000260P00Y0<008P2002V0`00
0P0008P2Y0>60ZH3E`0002@000230P00YP<008H2002W0`000P0008H2YP>30ZL3E`0002@000230P00
Y`<008<2002W0`000P0008<2Y`>30ZL3E`0002@000200P00YP<008<2002W0`000P0008<2Y`>00ZH3
E`0002@0001n0P00Y0<00802002V0`000P000802YP=n0Z@3E`0002@0001m0P00X@<007h2002T0`00
0P0007h2Y0=m0Z43E`0002@0001m0P00X0<007d2002Q0`000P0007d2X@=m0Z03E`0002@0001m0P00
WP<007h2002P0`000P0007d2X0=n0Yh3E`0002@0001n0P00W0<00802002N0`000P0007h2WP>00Y`3
E`0002@000200P00V`<008<2002L0`000P000802W0>30Y/3E`0002@000230P00V`<008@2002K0`00
0P0008<2V`>40Y/3E`0002@000240P00V`<008L2002L0`000P0008@2V`>70Y`3E`0002@000270P00
W0<008T2002N0`000P0008L2W0>90Yh3E`0002@000290P00WP<008X2002P0`000P0008T2WP>:0Z03
E`0002@0002:0P00X0<008X2002V0`000P0008X2X0>:0ZH3E`0002@000290P00YP<008X2002Z0`00
0P0008X2YP>90ZX3E`0002@000280P00ZP<008T2002/0`000P0008T2ZP>80Z`3E`0002@000260P00
[0<008P2002^0`000P0008P2[0>60Zh3E`0002@000230P00[P<008H2002^0`000P0008H2[P>30Zh3
E`0002@000210P00[P<008<2002^0`000P0008<2[P>10Zh3E`0002@0001o0P00[P<00842002^0`00
0P000842[P=o0Zh3E`0002@0001n0P00[0<007l2002^0`000P0007l2[P=n0Z`3E`0002@0001n0P00
Z`<007h2002/0`000P0007h2[0=n0Z/3E`0002@0001n0P00ZP<007l2002[0`000P0007h2Z`=o0ZX3
E`0002@0001o0P00ZP<00802002[0`000P0007l2ZP>00Z/3E`0002@0001o0P00Z`<00802002/0`00
0P000802Z`=o0Z`3E`0002@000210P00YP<008<2002W0`000P0008<2Y`>10ZH3E`0002@0001o0P00
Y0<00842002V0`000P000842YP=o0Z@3E`0002@0001n0P00X@<007l2002T0`000P0007l2Y0=n0Z43
E`0002@0001n0P00X0<007h2002Q0`000P0007h2X@=n0Z03E`0002@0001n0P00WP<007l2002P0`00
0P0007h2X0=o0Yh3E`0002@0001o0P00W0<00842002N0`000P0007l2WP>10Y`3E`0002@000210P00
V`<008<2002L0`000P000842W0>30Y/3E`0002@000240P00V`<008H2002L0`000P0008@2V`>60Y`3
E`0002@000260P00W0<008P2002N0`000P0008H2W0>80Yh3E`0002@000280P00WP<008T2002P0`00
0P0008P2WP>90Z03E`0002@000290P00X0<008T2002V0`000P0008T2X0>90ZH3E`0002@000280P00
YP<008T2002Z0`000P0008T2YP>80ZX3E`0002@000270P00ZP<008P2002/0`000P0008P2ZP>70Z`3
E`0002@000250P00[0<008L2002^0`000P0008L2[0>50Zh3E`0002@000230P00[P<008D2002^0`00
0P0008D2[P>30Zh3E`0002@0002]0P00WP<00:l2002O0`000P000:d2W`>_0Yh3E`0002@0002_0P00
V`<00;42002N0`000P000:l2WP>a0Y/3E`0002@0002a0P00V`<00;42002^0`000P000;42V`>a0Zh3
E`0002@0002`0P00W0<00;02002^0`000P000;02W0>`0Zh3E`0002@0002]0P00[P<00;D2002^0`00
0P000:d2[P>e0Zh3E`0002@0002o0P00V`<00<82002L0`000P000<82V`>o0Y`3E`0002@0002m0P00
W0<00;l2002O0`000P000;l2W0>m0Yl3E`0002@0002l0P00W`<00;d2002S0`000P000;d2W`>l0Z<3
E`0002@0002l0P00X`<00;`2002V0`000P000;`2X`>l0ZH3E`0002@0002l0P00YP<00;d2002[0`00
0P000;`2YP>m0Z/3E`0002@0002m0P00Z`<00;l2002^0`000P000;d2Z`>o0Zh3E`0002@0002o0P00
[P<00<82002^0`000P000;l2[P?20Zh3E`0002@000320P00[P<00<<2002^0`000P000<82[P?30Zh3
E`0002@000330P00[P<00<H2002^0`000P000<<2[P?60Zh3E`0002@000360P00Z`<00<P2002^0`00
0P000<H2[P?80Z/3E`0002@000380P00YP<00<T2002[0`000P000<P2Z`?90ZH3E`0002@000390P00
X`<00<T2002V0`000P000<T2YP?90Z<3E`0002@000380P00W`<00<T2002S0`000P000<T2X`?80Yl3
E`0002@000360P00W0<00<P2002O0`000P000<P2W`?60Y`3E`0002@000330P00V`<00<H2002L0`00
0P000<H2W0?30Y/3E`0002@000320P00V`<00<<2002K0`000P000<<2V`?20Y/3E`0002@000300P00
V`<00<82002L0`000P000<82V`?00Y`3E`0002@0002o0P00W0<00<02002M0`000P000<02W0>o0Yd3
E`0002@0002n0P00W@<00;l2002O0`000P000;l2W@>n0Yl3E`0002@0002m0P00W`<00;h2002S0`00
0P000;h2W`>m0Z<3E`0002@0002m0P00X`<00;d2002V0`000P000;d2X`>m0ZH3E`0002@0002m0P00
YP<00;h2002[0`000P000;d2YP>n0Z/3E`0002@0002n0P00Z`<00;l2002]0`000P000;h2Z`>o0Zd3
E`0002@0002o0P00[@<00<02002^0`000P000;l2[@?00Zh3E`0002@000300P00[P<00<82002^0`00
0P000<02[P?20Zh3E`0002@000330P00[P<00<D2002^0`000P000<<2[P?50Zh3E`0002@000350P00
[@<00<H2002^0`000P000<D2[P?60Zd3E`0002@000360P00Z`<00<L2002]0`000P000<H2[@?70Z/3
E`0002@000370P00YP<00<P2002[0`000P000<L2Z`?80ZH3E`0002@000380P00X`<00<P2002V0`00
0P000<P2YP?80Z<3E`0002@000370P00W`<00<P2002S0`000P000<P2X`?70Yl3E`0002@000360P00
W@<00<L2002O0`000P000<L2W`?60Yd3E`0002@000350P00W0<00<H2002M0`000P000<H2W@?50Y`3
E`0002@000330P00V`<00<D2002L0`000P000<D2W0?30Y/3E`0002@0003S0P00WP<00>D2002O0`00
0P000><2W`?U0Yh3E`0002@0003U0P00V`<00>L2002N0`000P000>D2WP?W0Y/3E`0002@0003W0P00
V`<00>L2002^0`000P000>L2V`?W0Zh3E`0002@0003V0P00W0<00>H2002^0`000P000>H2W0?V0Zh3
E`0002@0003S0P00[P<00>/2002^0`000P000><2[P?[0Zh3E`0002@0003e0P00WP<00?L2002O0`00
0P000?D2W`?g0Yh3E`0002@0003g0P00V`<00?T2002N0`000P000?L2WP?i0Y/3E`0002@0003i0P00
V`<00?T2002^0`000P000?T2V`?i0Zh3E`0002@0003h0P00W0<00?P2002^0`000P000?P2W0?h0Zh3
E`0002@0003e0P00[P<00?d2002^0`000P000?D2[P?m0Zh3E`0002@0000I0`00WP<001/3002O0`00
0P0001T3W`<K0ih3E`0002@0000K0`00V`<001d3002N0`000P0001/3WP<M0i/3E`0002@0000M0`00
V`<001d3002^0`000P0001d3V`<M0jh3E`0002@0000L0`00W0<001`3002^0`000P0001`3W0<L0jh3
E`0002@0000I0`00[P<00243002^0`000P0001T3[P<Q0jh3E`0002@0000Y0`00W`<002X3002P0`00
0P0002T3W`<Z0j03E`0002@0000Y0`00X0<002X3002P0`000P0002X3X0<Y0j03E`0002@0000X0`00
X0<002T3002P0`000P0002T3X0<X0j03E`0002@0000X0`00W`<002P3002P0`000P0002P3X0<X0il3
E`0002@0000X0`00W@<002T3002O0`000P0002P3W`<Y0id3E`0002@0000Y0`00W0<002X3002M0`00
0P0002T3W@<Z0i`3E`0002@0000Z0`00V`<002d3002L0`000P0002X3W0<]0i/3E`0002@0000]0`00
V`<00303002K0`000P0002d3V`<`0i/3E`0002@0000`0`00V`<003<3002L0`000P000303V`<c0i`3
E`0002@0000c0`00W0<003@3002M0`000P0003<3W0<d0id3E`0002@0000d0`00W@<003D3002O0`00
0P0003@3W@<e0il3E`0002@0000e0`00W`<003D3002P0`000P0003D3W`<e0j03E`0002@0000d0`00
X0<003D3002R0`000P0003D3X0<d0j83E`0002@0000a0`00XP<003@3002T0`000P0003@3XP<a0j@3
E`0002@0000]0`00Y0<00343002V0`000P000343Y0<]0jH3E`0002@0000[0`00YP<002d3002W0`00
0P0002d3YP<[0jL3E`0002@0000Y0`00Y`<002/3002Y0`000P0002/3Y`<Y0jT3E`0002@0000X0`00
Z@<002T3002/0`000P0002T3Z@<X0j`3E`0002@0000X0`00[0<002P3002^0`000P0002P3[0<X0jh3
E`0002@0000`0`00V`<00383002L0`000P000303V`<b0i`3E`0002@0000b0`00W0<003<3002M0`00
0P000383W0<c0id3E`0002@0000c0`00W@<003@3002O0`000P0003<3W@<d0il3E`0002@0000d0`00
W`<003@3002P0`000P0003@3W`<d0j03E`0002@0000c0`00X0<003@3002R0`000P0003@3X0<c0j83
E`0002@0000`0`00XP<003<3002T0`000P0003<3XP<`0j@3E`0002@0000]0`00Y0<00303002V0`00
0P000303Y0<]0jH3E`0002@0000X0`00[0<002T3002]0`000P0002P3[@<Y0j`3E`0002@0000Y0`00
[0<002/3002/0`000P0002T3[0<[0j`3E`0002@0000[0`00[0<002l3002^0`000P0002/3[0<_0jh3
E`0002@0000_0`00[P<00383002^0`000P0002l3[P<b0jh3E`0002@0000b0`00[@<003@3002^0`00
0P000383[P<d0jd3E`0002@0000d0`00[0<003D3002]0`000P0003@3[@<e0j`3E`0002@0000[0`00
[0<002l3002^0`000P0002/3[0<_0jh3E`0002@0000_0`00[P<003<3002^0`000P0002l3[P<c0jh3
E`0002@0000c0`00[P<003@3002^0`000P0003<3[P<d0jh3E`0002@0000d0`00[0<003D3002^0`00
0P0003@3[P<e0j`3E`0002@0000e0`00ZP<003D3002/0`000P0003D3[0<e0jX3E`0002@0001?0`00
WP<00543002O0`000P0004l3W`=A0ih3E`0002@0001A0`00V`<005<3002N0`000P000543WP=C0i/3
E`0002@0001C0`00V`<005<3002^0`000P0005<3V`=C0jh3E`0002@0001B0`00W0<00583002^0`00
0P000583W0=B0jh3E`0002@0001?0`00[P<005L3002^0`000P0004l3[P=G0jh3E`0002@0001O0`00
W`<00603002P0`000P0005l3W`=P0j03E`0002@0001O0`00X0<00603002P0`000P000603X0=O0j03
E`0002@0001N0`00X0<005l3002P0`000P0005l3X0=N0j03E`0002@0001N0`00W`<005h3002P0`00
0P0005h3X0=N0il3E`0002@0001N0`00W@<005l3002O0`000P0005h3W`=O0id3E`0002@0001O0`00
W0<00603002M0`000P0005l3W@=P0i`3E`0002@0001P0`00V`<006<3002L0`000P000603W0=S0i/3
E`0002@0001S0`00V`<006H3002K0`000P0006<3V`=V0i/3E`0002@0001V0`00V`<006T3002L0`00
0P0006H3V`=Y0i`3E`0002@0001Y0`00W0<006X3002N0`000P0006T3W0=Z0ih3E`0002@0001Z0`00
WP<006X3002P0`000P0006X3WP=Z0j03E`0002@0001Y0`00X0<006X3002R0`000P0006X3X0=Y0j83
E`0002@0001V0`00XP<006T3002S0`000P0006T3XP=V0j<3E`0002@0001T0`00X`<006H3002S0`00
0P0006H3X`=T0j<3E`0002@0001V0`00V`<006P3002L0`000P0006H3V`=X0i`3E`0002@0001X0`00
W0<006T3002N0`000P0006P3W0=Y0ih3E`0002@0001Y0`00WP<006T3002P0`000P0006T3WP=Y0j03
E`0002@0001X0`00X0<006T3002R0`000P0006T3X0=X0j83E`0002@0001V0`00XP<006P3002S0`00
0P0006P3XP=V0j<3E`0002@0001V0`00X`<006P3002T0`000P0006H3X`=X0j@3E`0002@0001X0`00
Y0<006X3002V0`000P0006P3Y0=Z0jH3E`0002@0001Z0`00YP<006/3002X0`000P0006X3YP=[0jP3
E`0002@0001[0`00Z0<006/3002[0`000P0006/3Z0=[0j/3E`0002@0001Z0`00Z`<006/3002]0`00
0P0006/3Z`=Z0jd3E`0002@0001Y0`00[@<006X3002^0`000P0006X3[@=Y0jh3E`0002@0001V0`00
[P<006T3002^0`000P0006T3[P=V0jh3E`0002@0001S0`00[P<006H3002^0`000P0006H3[P=S0jh3
E`0002@0001P0`00[P<006<3002^0`000P0006<3[P=P0jh3E`0002@0001O0`00[@<00603002^0`00
0P000603[P=O0jd3E`0002@0001N0`00Z`<005l3002]0`000P0005l3[@=N0j/3E`0002@0001N0`00
ZP<005h3002[0`000P0005h3Z`=N0jX3E`0002@0001N0`00Z@<005l3002Z0`000P0005h3ZP=O0jT3
E`0002@0001O0`00Z@<00603002Z0`000P0005l3Z@=P0jX3E`0002@0001O0`00ZP<00603002[0`00
0P000603ZP=O0j/3E`0002@0001Y0`00Y@<006X3002X0`000P0006T3Y@=Z0jP3E`0002@0001Z0`00
Z0<006X3002[0`000P0006X3Z0=Z0j/3E`0002@0001Y0`00Z`<006X3002]0`000P0006X3Z`=Y0jd3
E`0002@0001X0`00[@<006T3002^0`000P0006T3[@=X0jh3E`0002@0001V0`00[P<006P3002^0`00
0P0006P3[P=V0jh3E`0002@000250`00WP<008L3002O0`000P0008D3W`>70ih3E`0002@000270`00
V`<008T3002N0`000P0008L3WP>90i/3E`0002@000290`00V`<008T3002^0`000P0008T3V`>90jh3
E`0002@000280`00W0<008P3002^0`000P0008P3W0>80jh3E`0002@000250`00[P<008d3002^0`00
0P0008D3[P>=0jh3E`0002@0002L0`00W@<009`3002^0`000P0009`3W@>L0jh3E`0002@0002M0`00
V`<009d3002^0`000P0009d3V`>M0jh3E`0002@0002C0`00V`<009d3002Y0`000P0009d3V`>C0jT3
E`0002@0002C0`00Z@<00:83002Y0`000P0009<3Z@>R0jT3E`0002@0002J0`00[P<00:03002^0`00
0P0009X3[P>P0jh3E`0002@0002k0`00WP<00;d3002O0`000P000;/3W`>m0ih3E`0002@0002m0`00
V`<00;l3002N0`000P000;d3WP>o0i/3E`0002@0002o0`00V`<00;l3002^0`000P000;l3V`>o0jh3
E`0002@0002n0`00W0<00;h3002^0`000P000;h3W0>n0jh3E`0002@0002k0`00[P<00<<3002^0`00
0P000;/3[P?30jh3E`0002@0003:0`00V`<00<`3002T0`000P000<`3V`?:0j@3E`0002@0003:0`00
XP<00<`3002T0`000P000<X3Y0?<0j83E`0002@0003<0`00X@<00<l3002R0`000P000<`3XP??0j43
E`0002@0003?0`00X@<00=43002Q0`000P000<l3X@?A0j43E`0002@0003A0`00X@<00=@3002R0`00
0P000=43X@?D0j83E`0002@0003D0`00XP<00=H3002T0`000P000=@3XP?F0j@3E`0002@0003F0`00
Y0<00=L3002W0`000P000=H3Y0?G0jL3E`0002@0003G0`00Y`<00=L3002Y0`000P000=L3Y`?G0jT3
E`0002@0003F0`00Z@<00=L3002/0`000P000=L3Z@?F0j`3E`0002@0003D0`00[0<00=H3002^0`00
0P000=H3[0?D0jh3E`0002@0003A0`00[P<00=@3002^0`000P000=@3[P?A0jh3E`0002@0003?0`00
[P<00=43002^0`000P000=43[P??0jh3E`0002@0003<0`00[P<00<l3002^0`000P000<l3[P?<0jh3
E`0002@0003;0`00[@<00<`3002^0`000P000<`3[P?;0jd3E`0002@0003:0`00Z`<00</3002]0`00
0P000</3[@?:0j/3E`0002@0003:0`00ZP<00<X3002[0`000P000<X3Z`?:0jX3E`0002@0003:0`00
Z@<00</3002Z0`000P000<X3ZP?;0jT3E`0002@0003;0`00Z@<00<`3002Z0`000P000</3Z@?<0jX3
E`0002@0003;0`00ZP<00<`3002[0`000P000<`3ZP?;0j/3E`0002@0003A0`00X@<00=<3002R0`00
0P000=43X@?C0j83E`0002@0003C0`00XP<00=D3002T0`000P000=<3XP?E0j@3E`0002@0003E0`00
Y0<00=H3002W0`000P000=D3Y0?F0jL3E`0002@0003F0`00Y`<00=H3002Y0`000P000=H3Y`?F0jT3
E`0002@0003E0`00Z@<00=H3002/0`000P000=H3Z@?E0j`3E`0002@0003C0`00[0<00=D3002^0`00
0P000=D3[0?C0jh3E`0002@0003A0`00[P<00=<3002^0`000P000=<3[P?A0jh3E`0002@0003<0`00
V`<00=D3002K0`000P000<`3V`?E0i/3E`0002@0003<0`00W0<00=03002L0`000P000<`3W0?@0i`3
E`0002@0003@0`00V`<00=D3002L0`000P000=03W0?E0i/3E`0002@0003a0`00WP<00?<3002O0`00
0P000?43W`?c0ih3E`0002@0003c0`00V`<00?D3002N0`000P000?<3WP?e0i/3E`0002@0003e0`00
V`<00?D3002^0`000P000?D3V`?e0jh3E`0002@0003d0`00W0<00?@3002^0`000P000?@3W0?d0jh3
E`0002@0003a0`00[P<00?T3002^0`000P000?43[P?i0jh3E`0002@0000:1000WP<000/4002O0`00
0P0000/4WP<:19l3E`0002@0000:1000W`<000/4002P0`000P0000X4W`<;1:03E`0002@0000;1000
W`<000`4002P0`000P0000/4X0<<19l3E`0002@0000<1000WP<000`4002O0`000P0000`4W`<<19h3
E`0002@0000;1000W0<000`4002N0`000P0000`4WP<;19`3E`0002@000091000V`<000/4002L0`00
0P0000/4W0<919/3E`0002@000061000V`<000T4002K0`000P0000T4V`<619/3E`0002@000041000
V`<000H4002L0`000P0000H4V`<419`3E`0002@000021000W0<000@4002N0`000P0000@4W0<219h3
E`0002@000011000WP<00084002P0`000P000084WP<11:03E`0002@000001000X0<00044002S0`00
0P000044X0<01:<3E`0002@000001000X`<00004002Y0`000P000004X`<01:T3E`0002@000001000
Z@<00044002/0`000P000004Z@<11:`3E`0002@000011000[0<000<4002^0`000P000044[0<31:h3
E`0002@000031000[P<000H4002^0`000P0000<4[P<61:h3E`0002@000061000[P<000L4002^0`00
0P0000H4[P<71:h3E`0002@000071000[P<000X4002^0`000P0000L4[P<:1:h3E`0002@0000:1000
[0<000`4002^0`000P0000X4[P<<1:`3E`0002@0000<1000Z@<000d4002/0`000P0000`4[0<=1:T3
E`0002@0000=1000Z0<000d4002Y0`000P0000d4Z@<=1:P3E`0002@0000<1000Y@<000d4002X0`00
0P0000d4Z0<<1:D3E`0002@0000:1000X`<000`4002U0`000P0000`4Y@<:1:<3E`0002@000071000
XP<000X4002S0`000P0000X4X`<71:83E`0002@000061000XP<000L4002R0`000P0000L4XP<61:83
E`0002@000041000XP<000H4002S0`000P0000H4XP<41:<3E`0002@000021000X`<000@4002U0`00
0P0000@4X`<21:D3E`0002@000011000Y@<00084002X0`000P000084Y@<11:P3E`0002@000051000
V`<000H4002L0`000P0000H4V`<519`3E`0002@000031000W0<000D4002N0`000P0000D4W0<319h3
E`0002@000021000WP<000<4002P0`000P0000<4WP<21:03E`0002@000011000X0<00084002S0`00
0P000084X0<11:<3E`0002@000011000X`<00044002Y0`000P000044X`<11:T3E`0002@000011000
Z@<00084002/0`000P000044Z@<21:`3E`0002@000021000[0<000@4002^0`000P000084[0<41:h3
E`0002@000041000[P<000H4002^0`000P0000@4[P<61:h3E`0002@000071000[P<000T4002^0`00
0P0000L4[P<91:h3E`0002@000091000[0<000/4002^0`000P0000T4[P<;1:`3E`0002@0000;1000
Z@<000`4002/0`000P0000/4[0<<1:T3E`0002@0000<1000Z0<000`4002Y0`000P0000`4Z@<<1:P3
E`0002@0000;1000Y@<000`4002X0`000P0000`4Z0<;1:D3E`0002@000091000X`<000/4002U0`00
0P0000/4Y@<91:<3E`0002@000071000XP<000T4002S0`000P0000T4X`<71:83E`0002@0000W1000
WP<002T4002O0`000P0002L4W`<Y19h3E`0002@0000Y1000V`<002/4002N0`000P0002T4WP<[19/3
E`0002@0000[1000V`<002/4002^0`000P0002/4V`<[1:h3E`0002@0000Z1000W0<002X4002^0`00
0P0002X4W0<Z1:h3E`0002@0000W1000[P<002l4002^0`000P0002L4[P<_1:h3E`0002@0000f1000
V`<003H4002P0`000P0003H4V`<f1:03E`0002@0000f1000W@<003L4002O0`000P0003H4W`<g19d3
E`0002@0000g1000V`<003T4002M0`000P0003L4W@<i19/3E`0002@0000i1000V`<003/4002K0`00
0P0003T4V`<k19/3E`0002@0000k1000V`<003l4002N0`000P0003/4V`<o19h3E`0002@0000o1000
WP<00444002N0`000P0003l4WP=119h3E`0002@000111000W@<00484002N0`000P000444WP=219d3
E`0002@000121000V`<004<4002M0`000P000484W@=319/3E`0002@0000g1000W0<003T4002M0`00
0P0003L4W@<i19`3E`0002@0000i1000W0<003/4002L0`000P0003T4W0<k19`3E`0002@0000k1000
W0<003l4002N0`000P0003/4W0<o19h3E`0002@000131000V`<004<4002N0`000P0004<4V`=319h3
E`0002@000121000WP<004<4002P0`000P0004<4WP=21:03E`0002@0000n1000X0<00484002U0`00
0P000484X0<n1:D3E`0002@0000m1000Y@<003h4002W0`000P0003h4Y@<m1:L3E`0002@0000l1000
Y`<003d4002Z0`000P0003d4Y`<l1:X3E`0002@0000l1000ZP<003`4002^0`000P0003`4ZP<l1:h3
E`0002@0000m1000X0<00484002U0`000P000484X0<m1:D3E`0002@0000l1000Y@<003d4002W0`00
0P0003d4Y@<l1:L3E`0002@0000l1000Y`<003`4002Z0`000P0003`4Y`<l1:X3E`0002@0000l1000
ZP<003`4002^0`000P0003`4ZP<l1:h3E`0002@0001M1000WP<005l4002O0`000P0005d4W`=O19h3
E`0002@0001O1000V`<00644002N0`000P0005l4WP=Q19/3E`0002@0001Q1000V`<00644002^0`00
0P000644V`=Q1:h3E`0002@0001P1000W0<00604002^0`000P000604W0=P1:h3E`0002@0001M1000
[P<006D4002^0`000P0005d4[P=U1:h3E`0002@0001]1000V`<00704002L0`000P000704V`=]19`3
E`0002@0001/1000W0<006d4002N0`000P0006d4W0=/19h3E`0002@0001/1000WP<006`4002P0`00
0P0006`4WP=/1:03E`0002@0001/1000X0<006d4002R0`000P0006`4X0=]1:83E`0002@0001]1000
XP<00704002S0`000P0006d4XP=`1:<3E`0002@0001`1000X`<007<4002S0`000P000704X`=c1:<3
E`0002@0001c1000XP<007H4002S0`000P0007<4X`=f1:83E`0002@0001f1000X0<007L4002R0`00
0P0007H4XP=g1:03E`0002@0001g1000WP<007L4002P0`000P0007L4X0=g19h3E`0002@0001f1000
W0<007L4002N0`000P0007L4WP=f19`3E`0002@0001c1000V`<007H4002L0`000P0007H4W0=c19/3
E`0002@0001`1000V`<007<4002K0`000P0007<4V`=`19/3E`0002@0001^1000V`<00704002L0`00
0P000704V`=^19`3E`0002@0001]1000W0<006h4002N0`000P0006h4W0=]19h3E`0002@0001]1000
WP<006d4002P0`000P0006d4WP=]1:03E`0002@0001]1000X0<006h4002R0`000P0006d4X0=^1:83
E`0002@0001^1000XP<00704002S0`000P0006h4XP=`1:<3E`0002@0001c1000XP<007D4002S0`00
0P0007<4X`=e1:83E`0002@0001e1000X0<007H4002R0`000P0007D4XP=f1:03E`0002@0001f1000
WP<007H4002P0`000P0007H4X0=f19h3E`0002@0001e1000W0<007H4002N0`000P0007H4WP=e19`3
E`0002@0001c1000V`<007D4002L0`000P0007D4W0=c19/3E`0002@0001]1000X`<00704002T0`00
0P000704X`=]1:@3E`0002@0001/1000Y0<006d4002U0`000P0006d4Y0=/1:D3E`0002@0001[1000
Y@<006`4002W0`000P0006`4Y@=[1:L3E`0002@0001[1000Y`<006/4002[0`000P0006/4Y`=[1:/3
E`0002@0001[1000Z`<006`4002]0`000P0006/4Z`=/1:d3E`0002@0001/1000[@<006d4002^0`00
0P0006`4[@=]1:h3E`0002@0001]1000[P<00704002^0`000P0006d4[P=`1:h3E`0002@0001`1000
[P<007<4002^0`000P000704[P=c1:h3E`0002@0001c1000[P<007H4002^0`000P0007<4[P=f1:h3
E`0002@0001f1000[@<007L4002^0`000P0007H4[P=g1:d3E`0002@0001g1000Z`<007P4002]0`00
0P0007L4[@=h1:/3E`0002@0001h1000Y`<007P4002[0`000P0007P4Z`=h1:L3E`0002@0001g1000
Y@<007P4002W0`000P0007P4Y`=g1:D3E`0002@0001f1000Y0<007L4002U0`000P0007L4Y@=f1:@3
E`0002@0001c1000X`<007H4002T0`000P0007H4Y0=c1:<3E`0002@0001^1000X`<00704002T0`00
0P000704X`=^1:@3E`0002@0001]1000Y0<006h4002U0`000P0006h4Y0=]1:D3E`0002@0001/1000
Y@<006d4002W0`000P0006d4Y@=/1:L3E`0002@0001/1000Y`<006`4002[0`000P0006`4Y`=/1:/3
E`0002@0001/1000Z`<006d4002]0`000P0006`4Z`=]1:d3E`0002@0001]1000[@<006h4002^0`00
0P0006d4[@=^1:h3E`0002@0001^1000[P<00704002^0`000P0006h4[P=`1:h3E`0002@0001c1000
[P<007D4002^0`000P0007<4[P=e1:h3E`0002@0001e1000[@<007H4002^0`000P0007D4[P=f1:d3
E`0002@0001f1000Z`<007L4002]0`000P0007H4[@=g1:/3E`0002@0001g1000Y`<007L4002[0`00
0P0007L4Z`=g1:L3E`0002@0001f1000Y@<007L4002W0`000P0007L4Y`=f1:D3E`0002@0001e1000
Y0<007H4002U0`000P0007H4Y@=e1:@3E`0002@0001c1000X`<007D4002T0`000P0007D4Y0=c1:<3
E`0002@0002C1000WP<009D4002O0`000P0009<4W`>E19h3E`0002@0002E1000V`<009L4002N0`00
0P0009D4WP>G19/3E`0002@0002G1000V`<009L4002^0`000P0009L4V`>G1:h3E`0002@0002F1000
W0<009H4002^0`000P0009H4W0>F1:h3E`0002@0002C1000[P<009/4002^0`000P0009<4[P>K1:h3
E`0002@0002/1000X@<00:d4002T0`000P000:d4X@>/1:@3E`0002@0002Z1000Y0<00:`4002V0`00
0P000:`4Y0>Z1:H3E`0002@0002W1000YP<00:X4002W0`000P000:X4YP>W1:L3E`0002@0002W1000
Y`<00:L4002W0`000P000:L4Y`>W1:L3E`0002@0002T1000YP<00:L4002W0`000P000:L4Y`>T1:H3
E`0002@0002R1000Y0<00:@4002V0`000P000:@4YP>R1:@3E`0002@0002Q1000X@<00:84002T0`00
0P000:84Y0>Q1:43E`0002@0002Q1000X0<00:44002Q0`000P000:44X@>Q1:03E`0002@0002Q1000
WP<00:84002P0`000P000:44X0>R19h3E`0002@0002R1000W0<00:@4002N0`000P000:84WP>T19`3
E`0002@0002T1000V`<00:L4002L0`000P000:@4W0>W19/3E`0002@0002W1000V`<00:P4002K0`00
0P000:L4V`>X19/3E`0002@0002X1000V`<00:/4002L0`000P000:P4V`>[19`3E`0002@0002[1000
W0<00:d4002N0`000P000:/4W0>]19h3E`0002@0002]1000WP<00:h4002P0`000P000:d4WP>^1:03
E`0002@0002^1000X0<00:h4002V0`000P000:h4X0>^1:H3E`0002@0002]1000YP<00:h4002Z0`00
0P000:h4YP>]1:X3E`0002@0002/1000ZP<00:d4002/0`000P000:d4ZP>/1:`3E`0002@0002Z1000
[0<00:`4002^0`000P000:`4[0>Z1:h3E`0002@0002W1000[P<00:X4002^0`000P000:X4[P>W1:h3
E`0002@0002U1000[P<00:L4002^0`000P000:L4[P>U1:h3E`0002@0002S1000[P<00:D4002^0`00
0P000:D4[P>S1:h3E`0002@0002R1000[0<00:<4002^0`000P000:<4[P>R1:`3E`0002@0002R1000
Z`<00:84002/0`000P000:84[0>R1:/3E`0002@0002R1000ZP<00:<4002[0`000P000:84Z`>S1:X3
E`0002@0002S1000ZP<00:@4002[0`000P000:<4ZP>T1:/3E`0002@0002S1000Z`<00:@4002/0`00
0P000:@4Z`>S1:`3E`0002@0002U1000YP<00:L4002W0`000P000:L4Y`>U1:H3E`0002@0002S1000
Y0<00:D4002V0`000P000:D4YP>S1:@3E`0002@0002R1000X@<00:<4002T0`000P000:<4Y0>R1:43
E`0002@0002R1000X0<00:84002Q0`000P000:84X@>R1:03E`0002@0002R1000WP<00:<4002P0`00
0P000:84X0>S19h3E`0002@0002S1000W0<00:D4002N0`000P000:<4WP>U19`3E`0002@0002U1000
V`<00:L4002L0`000P000:D4W0>W19/3E`0002@0002X1000V`<00:X4002L0`000P000:P4V`>Z19`3
E`0002@0002Z1000W0<00:`4002N0`000P000:X4W0>/19h3E`0002@0002/1000WP<00:d4002P0`00
0P000:`4WP>]1:03E`0002@0002]1000X0<00:d4002V0`000P000:d4X0>]1:H3E`0002@0002/1000
YP<00:d4002Z0`000P000:d4YP>/1:X3E`0002@0002[1000ZP<00:`4002/0`000P000:`4ZP>[1:`3
E`0002@0002Y1000[0<00:/4002^0`000P000:/4[0>Y1:h3E`0002@0002W1000[P<00:T4002^0`00
0P000:T4[P>W1:h3E`0002@000361000W`<00<L4002P0`000P000<H4W`?71:03E`0002@000361000
X0<00<L4002P0`000P000<L4X0?61:03E`0002@000351000X0<00<H4002P0`000P000<H4X0?51:03
E`0002@000351000W`<00<D4002P0`000P000<D4X0?519l3E`0002@000351000W@<00<H4002O0`00
0P000<D4W`?619d3E`0002@000361000W0<00<L4002M0`000P000<H4W@?719`3E`0002@000371000
V`<00<X4002L0`000P000<L4W0?:19/3E`0002@0003:1000V`<00<d4002K0`000P000<X4V`?=19/3
E`0002@0003=1000V`<00=04002L0`000P000<d4V`?@19`3E`0002@0003@1000W0<00=44002M0`00
0P000=04W0?A19d3E`0002@0003A1000W@<00=84002O0`000P000=44W@?B19l3E`0002@0003B1000
W`<00=84002P0`000P000=84W`?B1:03E`0002@0003A1000X0<00=84002R0`000P000=84X0?A1:83
E`0002@0003>1000XP<00=44002T0`000P000=44XP?>1:@3E`0002@0003:1000Y0<00<h4002V0`00
0P000<h4Y0?:1:H3E`0002@000381000YP<00<X4002W0`000P000<X4YP?81:L3E`0002@000361000
Y`<00<P4002Y0`000P000<P4Y`?61:T3E`0002@000351000Z@<00<H4002/0`000P000<H4Z@?51:`3
E`0002@000351000[0<00<D4002^0`000P000<D4[0?51:h3E`0002@0003=1000V`<00<l4002L0`00
0P000<d4V`??19`3E`0002@0003?1000W0<00=04002M0`000P000<l4W0?@19d3E`0002@0003@1000
W@<00=44002O0`000P000=04W@?A19l3E`0002@0003A1000W`<00=44002P0`000P000=44W`?A1:03
E`0002@0003@1000X0<00=44002R0`000P000=44X0?@1:83E`0002@0003=1000XP<00=04002T0`00
0P000=04XP?=1:@3E`0002@0003:1000Y0<00<d4002V0`000P000<d4Y0?:1:H3E`0002@000351000
[0<00<H4002]0`000P000<D4[@?61:`3E`0002@000361000[0<00<P4002/0`000P000<H4[0?81:`3
E`0002@000381000[0<00<`4002^0`000P000<P4[0?<1:h3E`0002@0003<1000[P<00<l4002^0`00
0P000<`4[P??1:h3E`0002@0003?1000[@<00=44002^0`000P000<l4[P?A1:d3E`0002@0003A1000
[0<00=84002]0`000P000=44[@?B1:`3E`0002@000381000[0<00<`4002^0`000P000<P4[0?<1:h3
E`0002@0003<1000[P<00=04002^0`000P000<`4[P?@1:h3E`0002@0003@1000[P<00=44002^0`00
0P000=04[P?A1:h3E`0002@0003A1000[0<00=84002^0`000P000=44[P?B1:`3E`0002@0003B1000
ZP<00=84002/0`000P000=84[0?B1:X3E`0002@0003J1000V`<00=d4002L0`000P000=d4V`?J19`3
E`0002@0003H1000W0<00=X4002O0`000P000=X4W0?H19l3E`0002@0003G1000W`<00=P4002S0`00
0P000=P4W`?G1:<3E`0002@0003G1000X`<00=L4002V0`000P000=L4X`?G1:H3E`0002@0003G1000
YP<00=P4002[0`000P000=L4YP?H1:/3E`0002@0003H1000Z`<00=X4002^0`000P000=P4Z`?J1:h3
E`0002@0003J1000[P<00=d4002^0`000P000=X4[P?M1:h3E`0002@0003M1000[P<00=h4002^0`00
0P000=d4[P?N1:h3E`0002@0003N1000[P<00>44002^0`000P000=h4[P?Q1:h3E`0002@0003Q1000
Z`<00><4002^0`000P000>44[P?S1:/3E`0002@0003S1000YP<00>@4002[0`000P000><4Z`?T1:H3
E`0002@0003T1000X`<00>@4002V0`000P000>@4YP?T1:<3E`0002@0003S1000W`<00>@4002S0`00
0P000>@4X`?S19l3E`0002@0003Q1000W0<00><4002O0`000P000><4W`?Q19`3E`0002@0003N1000
V`<00>44002L0`000P000>44W0?N19/3E`0002@0003M1000V`<00=h4002K0`000P000=h4V`?M19/3
E`0002@0003K1000V`<00=d4002L0`000P000=d4V`?K19`3E`0002@0003J1000W0<00=/4002M0`00
0P000=/4W0?J19d3E`0002@0003I1000W@<00=X4002O0`000P000=X4W@?I19l3E`0002@0003H1000
W`<00=T4002S0`000P000=T4W`?H1:<3E`0002@0003H1000X`<00=P4002V0`000P000=P4X`?H1:H3
E`0002@0003H1000YP<00=T4002[0`000P000=P4YP?I1:/3E`0002@0003I1000Z`<00=X4002]0`00
0P000=T4Z`?J1:d3E`0002@0003J1000[@<00=/4002^0`000P000=X4[@?K1:h3E`0002@0003K1000
[P<00=d4002^0`000P000=/4[P?M1:h3E`0002@0003N1000[P<00>04002^0`000P000=h4[P?P1:h3
E`0002@0003P1000[@<00>44002^0`000P000>04[P?Q1:d3E`0002@0003Q1000Z`<00>84002]0`00
0P000>44[@?R1:/3E`0002@0003R1000YP<00><4002[0`000P000>84Z`?S1:H3E`0002@0003S1000
X`<00><4002V0`000P000><4YP?S1:<3E`0002@0003R1000W`<00><4002S0`000P000><4X`?R19l3
E`0002@0003Q1000W@<00>84002O0`000P000>84W`?Q19d3E`0002@0003P1000W0<00>44002M0`00
0P000>44W@?P19`3E`0002@0003N1000V`<00>04002L0`000P000>04W0?N19/3E`0002@0003E0000
iP400=D4003V0@000P000=D0iP7E1>H19@0000`000080020:00000`0000200009`0001P000020000
00000000000000009@0000`000020000:00000`000010000EP000300003C0000iP400=P0001G0`00
1@000=<0iP7H0>H1f01G0m<0E`?C0>H19@0000`000080020EP00030000090@00DP0000h1003V0@00
1@0000h1iP490NH12@5B00h1DP0>0NH19@0000`000080020EP000300000n0@00iP4004<1001L0`00
1@0003h1iP530NH1@`5L0ch1G0<n0NH19@0000`000080020EP000300001d0@00iP4007T1000i0`00
1@0007@1iP5i0NH1N@4i0g@1>@=d0NH19@0000`000080020EP000300002Z0@00G@000:l1003V0@00
1@000:l1iP6Z0NH1ZP5M0:l1G@2_0NH19@0000`000080020EP000300003P0@00iP400>D1003O0P00
1@000>01iP7U0NH1i@7O0^01g`;P0NH19@0000`000080020EP000300000F0P00iP4001/2002@0P00
1@0001H2iP4K0^H16`:@0QH2T08F0^H19@0000`000080020EP000300001<0P00iP400542001j0`00
1@0004`2iP5A0^H1D@9j0d`2NP=<0^H19@0000`000080020EP00030000220P00O@0008L2003V0@00
1@0008L2iP620^H1PP9m08L2O@270^H19@0000`000080020EP000300002h0P00E`000;d2003V0@00
1@000;d2iP6h0^H1^09G0;d2E`2m0^H19@0000`000080020EP000300003^0P00iP400?<2001b0`00
1@000>h2iP7c0^H1l`9b0nh2LP?^0^H19@0000`000080020EP000300000T0`00iP4002T3001=0`00
1@0002@3iP4Y0nH1:@==0b@3C@<T0nH19@0000`000080020EP000300001J0`00I`0005l3003V0@00
1@0005l3iP5J0nH1FP=W05l3I`1O0nH19@0000`000080020EP000300002@0`00E`0009D3003V0@00
1@0009D3iP6@0nH1T0=G09D3E`2E0nH19@0000`000080020EP00030000360`00iP400</300080`00
1@000<H3iP7;0nH1b`<80lH320?60nH19@0000`000080020EP000300003l0`00FP400044003V0@00
1@000044iP7l0nH1o0=J0@44FP411>H19@0000`000080020EP000300000b1000iP4003L4003G0P00
1@000384iP4g1>H1=`CG0S84e`8b1>H19@0000`000080020EP000300001W1000^P0006`4003V0@00
1@0006`4iP5W1>H1I`Bj06`4^P1/1>H19@0000`000080020EP000300002M1000iP400:84001[0`00
1@0009d4iP6R1>H1XPA[0id4J`>M1>H19@0000`000080020EP000300003C1000;`400=P4003V0@00
1@000=P4iP7C1>H1d`@_0MP4;`7H1>H19P0001`0000100000000004000000000000002D0000<0000
0@0005/0002@0000cP0007H3003J0000P`<00040000K00006`000=X0O0?J07T3f@1i0mP0N0?G07L3
eP1g0mD0MP?C07L3d@1g0m00N0?@07T3c`1i0ll0N`?>07`3c`1n0ll0P0?@0843d0220m40PP?B08<3
e`230mL0PP?H0883f@210mX0P0?J07h3fP1l0e/0002@0000104004X0000@0@00EP000040000K0000
6`000101D00@0Dd03`5<00h1B`0=0D/03@5:00P1BP070D/01P5;00H1C0050Dd01@5>00@1C`040E40
1@5B00D1D`060E@01P5E00L1E@080EH03@5F00d1E@0>0ED03`5D0101D`0@0E40405@05/0002H0000
>P4007H300160@00P`<00040000M00007@0004H1O0=60GX3A@5i0d@1N0=30GL3@@5g0d01MP<o0GL3
?@5g0c`1N0<l0GT3>`5i0c/1NP<j0G/3>P5n0c/1O`<k0H03?0610c`1PP<m0H83?P630d81P`=30H83
A0620dD1P@=50H03AP5o0dH1OP=60G`3F`0009P0001`0@00MP<007`100230`000@0001d0000M0000
O05l0g`1NP=k0GT3NP5h0gT1M`=g0GL3MP5f0gD1M`=c0GL3LP5h0g81N@=a0GT3L@5j0g01N`=`0Gh3
L@5o0g41P0=b0H43LP620g<1PP=d0H<3N0630gT1PP=j0H83N`610g/1P0=l0Gl3O05n0g`1O0=K0000
T0000:H1001:0000/P4005H0000100006`0001/0002b0E00/P5>0;41C@2a0D`0/05;0:l1B`2^0DX0
ZP5:0:T1B`2X0D/0Y`5<0:L1CP2V0Dl0YP5A0:L1DP2W0E@0Z05E0:T1E@2Z0EH0[P5F0:l1E@2`0ED0
/@5D0;41D`2b0E80/P5A0;81D01K0000S0000=`1001f0`00j04008<3000100006P0001X0003X0G`3
j05j0nL1N@?V0GP3i@5g0n<1M`?R0GH3h@5g0ml1M`?N0GP3g@5i0m`1NP?L0Gl3g@600md1P@?N0H83
g`620n01P`?T0H<3i@620nH1PP?W0H43i`600nP1O`?X0Gh3j05l0e/0002<00004P8007H3000N0P00
P`<00040000J00006P0001h2O0<N0WX37@9i0a`2N0<K0WL36@9g0aP2MP<G0WL35@9g0a@2N0<C0WT3
4P9j0a82O`<C0X034`:10a@2PP<E0X835P:30aX2P`<K0X8370:20ad2P@<M0X037P9o0ah2OP<N0W`3
F`0008`000180P00MP<005@200230`000@0001X0000J0000E09l0e@2NP=C0WT3DP9h0e42M`=?0WL3
CP9f0dd2M`=;0WL3BP9h0dT2N@=80WX3B09o0dT2P0=90X43BP:20d/2PP=<0X<3D0:30e42PP=B0X83
D`:10e<2P0=D0Wl3E09n0e@2O0=K0000U00007h2001:0000RP8005H000010000700001`0002:0U00
RP9?08T2CP290Td0R09<08P2B`270T/0QP9:0882BP210T/0P09;07l2C01o0Td0OP9>07h2DP1o0U<0
O`9D0802E@210UD0PP9F08H2EP270UD0R09E08P2E0290U<0R@9B08X2D@2:0U00F`0009@0002d0P00
BP000<02001F00000@0001`0000L0000`09@0<02C`2o0Th0_`9=0;h2C02n0T/0_@9;0;`2BP2h0TX0
]`9;0;H2B`2e0T`0]@9=0;@2CP2d0U80]@9C0;D2E02f0UD0]`9E0;P2EP2l0UH0_@9E0;h2E@2n0U@0
_`9C0;l2DP300U40`09@05/0002D0000jP8007H3003f0P00P`<00040000L000070000?H2O0?f0W/3
m@9j0oD2N@?d0WT3m09h0o<2M`?a0WL3l09f0nh2M`?]0WL3k09h0n/2N@?Z0WX3jP9o0n/2P0?[0X43
k0:20nd2PP?]0X<3lP:30o<2PP?d0X83m0:10oD2P0?e0Wl3mP9n0oH2O0=K0000R0000203001f0`00
;0<008<3000100006@0001T0000/0g`3:`=k0b/3N@<Z0gT3:P=h0bT3M`<W0gL39P=f0b@3M`<S0gL3
8P=h0b43N@<P0gT380>00b43P@<R0h838`>20b<3P`<X0h<3:@>20bX3PP<Z0h43:`>00b/3OP</0g`3
F`0007@0001F0`00BP000683001F00000@0001@0000D0000HP=@0643C`1Q0dd0H0=<05l3B`1N0dX0
F@=:05P3B`1G0d`0EP==05H3D`1G0e@0F0=E05T3EP1N0eH0G`=E0603E01Q0e<0H@=A0683D01K0000
L00008`3001:0000U`<005H0000100004`0001<0002G0e00U`==09H3C02E0d/0U0=:08l3BP2>0d/0
S@=<08`3C@2<0e<0S@=D08h3E@2?0eH0U0=F09D3E@2F0e@0U`=C09L3D@2G0e00F`0008P000310`00
MP<00<d300230`000@0001T0000I0000c@=l0ld3N@?<0gT3b`=h0l/3M`?90gL3b0=f0lH3M`?40gL3
a0=h0l<3N@?20gT3`P=k0l43O0?20gh3`P>00l<3P@?40h83a@>30lX3P`?;0h83c0>10ld3P0?=0gh3
c@=l0e/0002D0000m`<007H300031000P`<00040000L0000700000<4O0<317T30PAi0`44N0<017L3
o`=g0oh3MP?l0gL3nP=g0oT3N0?i0gT3n0=i0oP3NP?g0g/3m`=n0oP3O`?h0h03n@>10oT3PP?j0h83
n`>30`04P`<018830@B20`84P@<318030`An0`<4O0=K0000V00002d4001f0`00>@@008<300010000
7@0001d0000i17`3>@Aj0cP4N@<g17P3=PAg0cD4M`<d17H3<PAg0c04M`<_17P3;`Ai0bh4N@<^17X3
;@Ak0bd4OP<^17l3;PB00bl4P@<_1883<0B20c44P`<e18<3=PB20cL4PP<h1843>0B00cT4O`<i17h3
>@Al0e/0002H0000H`@004X0001_1000EP000040000M00007@0006l4D01_14h0KPA=06h4C01]14/0
K0A;06/4BP1W14X0IPA;06D4B`1U14`0I0A=06@4CP1S14l0H`AA06@4DP1T15<0I@AD06D4E@1V15D0
I`AF06/4EP1/15D0K@AE06h4E01^15<0K`AB06l4D@1_1500F`0009P0002I1000MP<00:D400230`00
0@0001d0000M0000Y@Al0jD4NP>T17T3X`Ah0j84M`>P17L3W`Af0ih4M`>L17L3V`Ah0i/4N@>J17T3
VPAj0iT4N`>I17h3VPAo0iX4P0>K1843V`B20i`4PP>M18<3X@B30j84PP>S1883Y0B10j@4P0>U17l3
Y@An0jD4O0=K0000T0000<l4001:0000f`@005H0000100006`0001/0003K1500f`A>0=X4C@3J14`0
f@A;0=P4B`3G14X0d`A:0=84B`3A14/0d0A<0=04C@3?14h0c`AB0=04D`3@15@0d@AE0=84E@3C15H0
e`AF0=P4E@3I15D0fPAD0=X4D`3K1580f`AA0=/4D01G0000=0000=0000180000fP@008D300060000
d0240m00B03J14P0fPB40m00Q0?@08D39@0000`000050020:00000`0000200009@0000`000080020
:00000`0000100009@0000`0000=00203P0001@000000000400001@0
\>"], "Graphics",
 ImageSize->{249.75, 197.75},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}}],

Cell[TextData[{
 "Note that estimates by and large agree with with actuals, at least in \
direction (",
 Cell[BoxData[
  SubscriptBox["q", "16"]]],
 " is the sole exception). "
}], "Text"],

Cell[TextData[{
 "Now, consider the same analysis, with the cost parameter changed to ",
 Cell[BoxData[
  RowBox[{"c", "=", "0.005"}]]],
 ", i.e., one-half the previous value. The figure below shows the prices. \
Notice that the bid-ask bounce is much less visually evident."
}], "Text"],

Cell[GraphicsData["Metafile", "\<\
CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@000DgD0@0006`00000000000000?D4003/0`00
000000000037FP00idL00215CDH00040e4d10289000400000000000000000000@0H00;04000U0@00
g000000000000000000008Qh101PF`<02@000100003f1000k@<000X0000@000000000000000D0000
300000d0000U0000300000P0080W00006000004000000000oooo0000000U000030000040000[0000
6000000000000000mP@00>d3000[00006000000000000000mP@00>d3000V00007000008000000000
0@000000000000009@0000`000020000E`0002@000210@002P000881000=00000P0008413@220@X0
E`0002@000220@002P000881000?00000P0008812P220@l0E`0002@000210@003@000881000?0000
0P0008813`210@d0E`0002@0001o0@002`000841000=00000P0008413@1o0@/0E`0002@0001l0@00
2P0007l1000;00000P0007l12`1l0@X0E`0002@0001j0@002P0007`1000:00000P0007`12P1j0@X0
E`0002@0001g0@002P0007X1000;00000P0007X12P1g0@/0E`0002@0001e0@002`0007L1000=0000
0P0007L12`1e0@d0E`0002@0001e0@003@0007D1000?00000P0007D13@1e0@l0E`0002@0001e0@00
3`0007H1000@00000P0007D13`1f0A00E`0002@0001f0@00400007L1000A00000P0007H1401g0A40
E`0002@0001g0@004@0007T1000B00000P0007L14@1i0A80E`0002@0001i0@004P0007h1000D0000
0P0007T14P1n0A@0E`0002@0001n0@0050000801000E00000P0007h150200AD0E`0002@000200@00
5@000881000G00000P0008015@220AL0E`0002@0001e0@003`0007L1000@00000P0007D13`1g0A00
E`0002@0001g0@00400007T1000A00000P0007L1401i0A40E`0002@0001i0@004@0007h1000C0000
0P0007T14@1n0A<0E`0002@0001n0@004`000801000D00000P0007h14`200A@0E`0002@000200@00
50000841000E00000P00080150210AD0E`0002@000210@005@000881000G00000P0008415@220AL0
E`0002@000220@005`000881000K00000P0008815`220A/0E`0002@000200@006`000881000M0000
0P0008816`200Ad0E`0002@0001m0@007@000801000M00000P0008017@1m0Ad0E`0002@0001k0@00
7@0007d1000M00000P0007d17@1k0Ad0E`0002@0001h0@007@0007/1000M00000P0007/17@1h0Ad0
E`0002@0001f0@006`0007P1000M00000P0007P17@1f0A/0E`0002@0001e0@00600007H1000K0000
0P0007H16`1e0AP0E`0002@0001e0@00600007D1000M00000P0007D1601e0Ad0E`0002@0001e0@00
6`0007H1000M00000P0007D17@1f0A/0E`0002@000270@002P0008P1000;00000P0008P12P270@/0
E`0002@000270@002`0008P1000<00000P0008L12`280@`0E`0002@000280@002`0008T1000<0000
0P0008P130290@/0E`0002@000280@002P0008T1000;00000P0008T12`280@X0E`0002@000280@00
400008P1000M00000P0008P140280Ad0E`0002@000290@00400008T1000M00000P0008T140290Ad0
E`0002@000250@00400008T1000@00000P0008D140290A00E`0002@000250@007@0008`1000M0000
0P0008D17@2<0Ad0E`0002@0002B0@0040000981000M00000P000981402B0Ad0E`0002@0002C0@00
400009<1000M00000P0009<1402C0Ad0E`0002@0002C0@004@0009D1000C00000P0009<14`2E0A40
E`0002@0002E0@00400009L1000A00000P0009D14@2G0A00E`0002@0002G0@00400009T1000@0000
0P0009L1402I0A00E`0002@0002I0@00400009`1000A00000P0009T1402L0A40E`0002@0002L0@00
4@0009d1000C00000P0009`14@2M0A<0E`0002@0002M0@004`0009d1000M00000P0009d14`2M0Ad0
E`0002@0002I0@00400009/1000A00000P0009T1402K0A40E`0002@0002K0@004@0009`1000C0000
0P0009/14@2L0A<0E`0002@0002L0@004`0009`1000M00000P0009`14`2L0Ad0E`0002@0002M0@00
4@0009h1000C00000P0009d14`2N0A40E`0002@0002N0@0040000:41000A00000P0009h14@2Q0A00
E`0002@0002Q0@0040000:<1000@00000P000:41402S0A00E`0002@0002S0@0040000:H1000A0000
0P000:<1402V0A40E`0002@0002V0@004@000:L1000C00000P000:H14@2W0A<0E`0002@0002W0@00
4`000:L1000M00000P000:L14`2W0Ad0E`0002@0002S0@0040000:D1000A00000P000:<1402U0A40
E`0002@0002U0@004@000:H1000C00000P000:D14@2V0A<0E`0002@0002V0@004`000:H1000M0000
0P000:H14`2V0Ad0E`0002@0002?0@00400009<1000@00000P0008l1402C0A00E`0002@0002?0@00
7@0009H1000M00000P0008l17@2F0Ad0E`0002@0002I0@007@0009l1000M00000P0009T17@2O0Ad0
E`0002@0002S0@007@000:T1000M00000P000:<17@2Y0Ad0E`0002@0002`0@0040000;01000K0000
0P000;01402`0A/0E`0002@0002`0@006`000;41000M00000P000;016`2a0Ad0E`0002@0002a0@00
7@000;@1000M00000P000;417@2d0Ad0E`0002@0002d0@007@000;D1000M00000P000;@17@2e0Ad0
E`0002@0002e0@007@000;P1000M00000P000;D17@2h0Ad0E`0002@0002h0@006`000;X1000M0000
0P000;P17@2j0A/0E`0002@0002a0@0040000;41000K00000P000;41402a0A/0E`0002@0002a0@00
6`000;81000M00000P000;416`2b0Ad0E`0002@0002b0@007@000;@1000M00000P000;817@2d0Ad0
E`0002@0002j0@0040000;X1000M00000P000;X1402j0Ad0E`0002@0002k0@0040000;/1000M0000
0P000;/1402k0Ad0E`0002@0002]0@0040000;41000@00000P000:d1402a0A00E`0002@0002g0@00
40000;/1000@00000P000;L1402k0A00E`0002@0002j0@007@000;d1000M00000P000;X17@2m0Ad0
E`0002@000330@002P000<<1000M00000P000<<12P330Ad0E`0002@000340@002P000<@1000M0000
0P000<@12P340Ad0E`0002@000300@002P000<@1000:00000P000<012P340@X0E`0002@000300@00
7@000<L1000M00000P000<017@370Ad0E`0002@0003=0@004P000<d1000C00000P000<d14P3=0A<0
E`0002@0003<0@004`000<d1000C00000P000<d14`3<0A<0E`0002@0003<0@004P000<`1000C0000
0P000<`14`3<0A80E`0002@0003<0@004@000<d1000B00000P000<`14P3=0A40E`0002@0003=0@00
40000<l1000A00000P000<d14@3?0A00E`0002@0003?0@0040000=81000@00000P000<l1403B0A00
E`0002@0003B0@0040000=@1000A00000P000=81403D0A40E`0002@0003D0@004@000=D1000B0000
0P000=@14@3E0A80E`0002@0003E0@004P000=H1000D00000P000=D14P3F0A@0E`0002@0003F0@00
50000=H1000K00000P000=H1503F0A/0E`0002@0003F0@006`000=L1000M00000P000=H16`3G0Ad0
E`0002@0003G0@007@000=P1000M00000P000=L17@3H0Ad0E`0002@0003E0@004P000=D1000K0000
0P000=D14P3E0A/0E`0002@0003E0@006`000=H1000M00000P000=D16`3F0Ad0E`0002@0003F0@00
7@000=P1000M00000P000=H17@3H0Ad0E`0002@0003H0@007@000=T1000M00000P000=P17@3I0Ad0
E`0002@0003D0@0050000=D1000E00000P000=D1503D0AD0E`0002@0003?0@005@000=@1000F0000
0P000=@15@3?0AH0E`0002@0003<0@005P000<l1000G00000P000<l15P3<0AL0E`0002@0003;0@00
5`000<`1000I00000P000<`15`3;0AT0E`0002@0003;0@006@000</1000K00000P000</16@3;0A/0
E`0002@0003;0@006`000<`1000M00000P000</16`3<0Ad0E`0002@0003<0@007@000<l1000M0000
0P000<`17@3?0Ad0E`0002@0003?0@007@000=41000M00000P000<l17@3A0Ad0E`0002@0003A0@00
7@000=<1000M00000P000=417@3C0Ad0E`0002@0003C0@006`000=D1000M00000P000=<17@3E0A/0
E`0002@0003=0@005P000<l1000G00000P000<l15P3=0AL0E`0002@0003<0@005`000<d1000I0000
0P000<d15`3<0AT0E`0002@0003<0@006@000<`1000K00000P000<`16@3<0A/0E`0002@0003<0@00
6`000<d1000M00000P000<`16`3=0Ad0E`0002@0003=0@007@000<l1000M00000P000<d17@3?0Ad0
E`0002@0003O0@002P000=l1000J00000P000=l12P3O0AX0E`0002@0003O0@006P000>01000M0000
0P000=l16P3P0Ad0E`0002@0003P0@007@000>81000M00000P000>017@3R0Ad0E`0002@0003R0@00
7@000><1000M00000P000>817@3S0Ad0E`0002@0003S0@007@000>D1000M00000P000><17@3U0Ad0
E`0002@0003U0@006`000>H1000M00000P000>D17@3V0A/0E`0002@0003P0@002P000>01000J0000
0P000>012P3P0AX0E`0002@0003P0@006P000>41000M00000P000>016P3Q0Ad0E`0002@0003Q0@00
7@000>81000M00000P000>417@3R0Ad0E`0002@0003L0@0040000><1000@00000P000=`1403S0A00
E`0002@0003[0@005P000?H1000F00000P000>/15P3f0AH0E`0002@0003f0@0050000?H1000F0000
0P000?H15P3f0A@0E`0002@0003e0@004P000?H1000D00000P000?H1503e0A80E`0002@0003d0@00
4@000?D1000B00000P000?D14P3d0A40E`0002@0003b0@0040000?@1000A00000P000?@14@3b0A00
E`0002@0003`0@0040000?81000@00000P000?81403`0A00E`0002@0003]0@0040000?01000A0000
0P000?01403]0A40E`0002@0003[0@004@000>d1000C00000P000>d14@3[0A<0E`0002@0003Z0@00
4`000>/1000F00000P000>/14`3Z0AH0E`0002@0003Z0@005P000>X1000H00000P000>X15P3Z0AP0
E`0002@0003Z0@0060000>/1000K00000P000>X1603[0A/0E`0002@0003[0@006`000>d1000M0000
0P000>/16`3]0Ad0E`0002@0003]0@007@000?01000M00000P000>d17@3`0Ad0E`0002@0003`0@00
7@000?41000M00000P000?017@3a0Ad0E`0002@0003a0@007@000?@1000M00000P000?417@3d0Ad0
E`0002@0003d0@006`000?H1000M00000P000?@17@3f0A/0E`0002@0003e0@004`000?D1000F0000
0P000?D15P3e0A<0E`0002@0003d0@004@000?D1000C00000P000?D14`3d0A40E`0002@0003^0@00
40000?01000A00000P000?01403^0A40E`0002@0003/0@004@000>h1000C00000P000>h14@3/0A<0
E`0002@0003[0@004`000>`1000F00000P000>`14`3[0AH0E`0002@0003[0@005P000>/1000H0000
0P000>/15P3[0AP0E`0002@0003[0@0060000>`1000K00000P000>/1603/0A/0E`0002@0003/0@00
6`000>h1000M00000P000>`16`3^0Ad0E`0002@0003^0@007@000?01000M00000P000>h17@3`0Ad0
E`0002@000060P002P0000H2000M00000P0000H22P060Qd0E`0002@000070P002P0000L2000M0000
0P0000L22P070Qd0E`0002@000040P004@0000H2000C00000P0000H24`040Q40E`0002@000020P00
400000@2000A00000P0000@24@020Q00E`0002@000010P0040000082000@00000P00008240010Q00
E`0002@0003n0@0040000042000A00000P000042403n0A40E`0002@0003l0@004@000?h1000C0000
0P000?h14@3l0A<0E`0002@0003k0@004`000?`1000F00000P000?`14`3k0AH0E`0002@0003k0@00
5P000?/1000H00000P000?/15P3k0AP0E`0002@0003k0@0060000?`1000K00000P000?/1603l0A/0
E`0002@0003l0@006`000?h1000M00000P000?`16`3n0Ad0E`0002@0003n0@007@000042000M0000
0P000?h17@010Qd0E`0002@000010P007@000082000M00000P0000427@020Qd0E`0002@000020P00
7@0000@2000M00000P0000827@040Qd0E`0002@000040P006`0000H2000M00000P0000@27@060Q/0
E`0002@0003o0@0040000042000A00000P000042403o0A40E`0002@0003m0@004@000?l1000C0000
0P000?l14@3m0A<0E`0002@0003l0@004`000?d1000F00000P000?d14`3l0AH0E`0002@0003l0@00
5P000?`1000H00000P000?`15P3l0AP0E`0002@0003l0@0060000?d1000K00000P000?`1603m0A/0
E`0002@0003m0@006`000?l1000M00000P000?d16`3o0Ad0E`0002@0003o0@007@000042000M0000
0P000?l17@010Qd0E`0002@000030P002P0000L2000:00000P0000<22P070PX0E`0002@000060P00
7@0000T2000M00000P0000H27@090Qd0E`0002@0000J0P00400001X2000T00000P0001X2400J0R@0
E`0002@0000K0P00400001/2000T00000P0001/2400K0R@0E`0002@0000K0P004@0001d2000C0000
0P0001/24`0M0Q40E`0002@0000M0P00400001h2000A00000P0001d24@0N0Q00E`0002@0000N0P00
40000202000@00000P0001h2400P0Q00E`0002@0000P0P00400002<2000A00000P000202400S0Q40
E`0002@0000S0P004@0002D2000C00000P0002<24@0U0Q<0E`0002@0000U0P004`0002H2000F0000
0P0002D24`0V0QH0E`0002@0000V0P005P0002H2000H00000P0002H25P0V0QP0E`0002@0000U0P00
600002H2000K00000P0002H2600U0Q/0E`0002@0000S0P006`0002D2000M00000P0002D26`0S0Qd0
E`0002@0000P0P007@0002<2000M00000P0002<27@0P0Qd0E`0002@0000N0P007@000202000M0000
0P0002027@0N0Qd0E`0002@0000M0P007@0001h2000M00000P0001h27@0M0Qd0E`0002@0000K0P00
6`0001d2000M00000P0001d27@0K0Q/0E`0002@0000P0P0040000282000A00000P000202400R0Q40
E`0002@0000R0P004@0002@2000C00000P0002824@0T0Q<0E`0002@0000T0P004`0002D2000F0000
0P0002@24`0U0QH0E`0002@0000U0P005P0002D2000H00000P0002D25P0U0QP0E`0002@0000T0P00
600002D2000K00000P0002D2600T0Q/0E`0002@0000R0P006`0002@2000M00000P0002@26`0R0Qd0
E`0002@0000P0P007@000282000M00000P0002827@0P0Qd0E`0002@0000G0P00400001/2000@0000
0P0001L2400K0Q00E`0002@0000G0P00900001h2000T00000P0001L2900N0R@0E`0002@0000]0P00
400002d2000M00000P0002d2400]0Qd0E`0002@0000^0P00400002h2000M00000P0002h2400^0Qd0
E`0002@0000^0P004`0002l2000F00000P0002h25P0_0Q<0E`0002@0000_0P004@000342000C0000
0P0002l24`0a0Q40E`0002@0000a0P0040000382000A00000P0003424@0b0Q00E`0002@0000b0P00
400003D2000@00000P000382400e0Q00E`0002@0000e0P00400003H2000A00000P0003D2400f0Q40
E`0002@0000f0P004@0003H2000B00000P0003H24@0f0Q80E`0002@0000e0P004P0003H2000C0000
0P0003H24P0e0Q<0E`0002@0000d0P004P0003D2000C00000P0003D24`0d0Q80E`0002@0000d0P00
4@0003D2000B00000P0003@24P0e0Q40E`0002@0000Z0P00400002h2000@00000P0002X2400^0Q00
E`0002@0000Z0P007@000342000M00000P0002X27@0a0Qd0E`0002@0000k0P002P0003`2000;0000
0P0003`22P0k0P/0E`0002@0000k0P002`0003`2000<00000P0003/22`0l0P`0E`0002@0000l0P00
2`0003d2000<00000P0003`2300m0P/0E`0002@0000l0P002P0003d2000;00000P0003d22`0l0PX0
E`0002@0000l0P00400003`2000M00000P0003`2400l0Qd0E`0002@0000m0P00400003d2000M0000
0P0003d2400m0Qd0E`0002@0000i0P00400003d2000@00000P0003T2400m0Q00E`0002@0000i0P00
7@000402000M00000P0003T27@100Qd0E`0002@0001>0P004`0004l2000D00000P0004l24`1>0Q@0
E`0002@0001>0P00500004l2000E00000P0004h2501?0QD0E`0002@0001?0P0050000502000E0000
0P0004l25@1@0Q@0E`0002@0001@0P004`000502000D00000P000502501@0Q<0E`0002@0001>0P00
4@000502000C00000P0005024`1>0Q40E`0002@0001<0P00400004h2000A00000P0004h24@1<0Q00
E`0002@0001:0P00400004`2000@00000P0004`2401:0Q00E`0002@000170P00400004X2000A0000
0P0004X240170Q40E`0002@000150P004@0004L2000C00000P0004L24@150Q<0E`0002@000140P00
4`0004D2000F00000P0004D24`140QH0E`0002@000140P005P0004@2000H00000P0004@25P140QP0
E`0002@000140P00600004D2000K00000P0004@260150Q/0E`0002@000150P006`0004L2000M0000
0P0004D26`170Qd0E`0002@000170P007@0004X2000M00000P0004L27@1:0Qd0E`0002@0001:0P00
7@0004/2000M00000P0004X27@1;0Qd0E`0002@0001;0P007@0004h2000M00000P0004/27@1>0Qd0
E`0002@0001>0P006`000502000M00000P0004h27@1@0Q/0E`0002@000180P00400004X2000A0000
0P0004X240180Q40E`0002@000160P004@0004P2000C00000P0004P24@160Q<0E`0002@000150P00
4`0004H2000F00000P0004H24`150QH0E`0002@000150P005P0004D2000H00000P0004D25P150QP0
E`0002@000150P00600004H2000K00000P0004D260160Q/0E`0002@000160P006`0004P2000M0000
0P0004H26`180Qd0E`0002@000180P007@0004X2000M00000P0004P27@1:0Qd0E`0002@0001E0P00
5P000602000F00000P0005D25P1P0QH0E`0002@0001P0P0050000602000F00000P0006025P1P0Q@0
E`0002@0001O0P004P000602000D00000P000602501O0Q80E`0002@0001N0P004@0005l2000B0000
0P0005l24P1N0Q40E`0002@0001L0P00400005h2000A00000P0005h24@1L0Q00E`0002@0001J0P00
400005`2000@00000P0005`2401J0Q00E`0002@0001G0P00400005X2000A00000P0005X2401G0Q40
E`0002@0001E0P004@0005L2000C00000P0005L24@1E0Q<0E`0002@0001D0P004`0005D2000F0000
0P0005D24`1D0QH0E`0002@0001D0P005P0005@2000H00000P0005@25P1D0QP0E`0002@0001D0P00
600005D2000K00000P0005@2601E0Q/0E`0002@0001E0P006`0005L2000M00000P0005D26`1G0Qd0
E`0002@0001G0P007@0005X2000M00000P0005L27@1J0Qd0E`0002@0001J0P007@0005/2000M0000
0P0005X27@1K0Qd0E`0002@0001K0P007@0005h2000M00000P0005/27@1N0Qd0E`0002@0001N0P00
6`000602000M00000P0005h27@1P0Q/0E`0002@0001O0P004`0005l2000F00000P0005l25P1O0Q<0
E`0002@0001N0P004@0005l2000C00000P0005l24`1N0Q40E`0002@0001H0P00400005X2000A0000
0P0005X2401H0Q40E`0002@0001F0P004@0005P2000C00000P0005P24@1F0Q<0E`0002@0001E0P00
4`0005H2000F00000P0005H24`1E0QH0E`0002@0001E0P005P0005D2000H00000P0005D25P1E0QP0
E`0002@0001E0P00600005H2000K00000P0005D2601F0Q/0E`0002@0001F0P006`0005P2000M0000
0P0005H26`1H0Qd0E`0002@0001H0P007@0005X2000M00000P0005P27@1J0Qd0E`0002@0001a0P00
40000742000T00000P000742401a0R@0E`0002@0001b0P0040000782000T00000P000782401b0R@0
E`0002@0001b0P004@0007@2000C00000P0007824`1d0Q40E`0002@0001d0P00400007D2000A0000
0P0007@24@1e0Q00E`0002@0001e0P00400007L2000@00000P0007D2401g0Q00E`0002@0001g0P00
400007X2000A00000P0007L2401j0Q40E`0002@0001j0P004@0007`2000C00000P0007X24@1l0Q<0
E`0002@0001l0P004`0007d2000F00000P0007`24`1m0QH0E`0002@0001m0P005P0007d2000H0000
0P0007d25P1m0QP0E`0002@0001l0P00600007d2000K00000P0007d2601l0Q/0E`0002@0001j0P00
6`0007`2000M00000P0007`26`1j0Qd0E`0002@0001g0P007@0007X2000M00000P0007X27@1g0Qd0
E`0002@0001e0P007@0007L2000M00000P0007L27@1e0Qd0E`0002@0001d0P007@0007D2000M0000
0P0007D27@1d0Qd0E`0002@0001b0P006`0007@2000M00000P0007@27@1b0Q/0E`0002@0001g0P00
400007T2000A00000P0007L2401i0Q40E`0002@0001i0P004@0007/2000C00000P0007T24@1k0Q<0
E`0002@0001k0P004`0007`2000F00000P0007/24`1l0QH0E`0002@0001l0P005P0007`2000H0000
0P0007`25P1l0QP0E`0002@0001k0P00600007`2000K00000P0007`2601k0Q/0E`0002@0001i0P00
6`0007/2000M00000P0007/26`1i0Qd0E`0002@0001g0P007@0007T2000M00000P0007T27@1g0Qd0
E`0002@0001^0P0040000782000@00000P0006h2401b0Q00E`0002@0001^0P00900007D2000T0000
0P0006h2901e0R@0E`0002@000240P004P0008@2000C00000P0008@24P240Q<0E`0002@000230P00
4`0008@2000C00000P0008@24`230Q<0E`0002@000230P004P0008<2000C00000P0008<24`230Q80
E`0002@000230P004@0008@2000B00000P0008<24P240Q40E`0002@000240P00400008H2000A0000
0P0008@24@260Q00E`0002@000260P00400008T2000@00000P0008H240290Q00E`0002@000290P00
400008/2000A00000P0008T2402;0Q40E`0002@0002;0P004@0008`2000B00000P0008/24@2<0Q80
E`0002@0002<0P004P0008d2000D00000P0008`24P2=0Q@0E`0002@0002=0P00500008d2000K0000
0P0008d2502=0Q/0E`0002@0002=0P006`0008h2000M00000P0008d26`2>0Qd0E`0002@0002>0P00
7@0008l2000M00000P0008h27@2?0Qd0E`0002@0002<0P004P0008`2000K00000P0008`24P2<0Q/0
E`0002@0002<0P006`0008d2000M00000P0008`26`2=0Qd0E`0002@0002=0P007@0008l2000M0000
0P0008d27@2?0Qd0E`0002@0002?0P007@000902000M00000P0008l27@2@0Qd0E`0002@0002;0P00
500008`2000E00000P0008`2502;0QD0E`0002@000260P005@0008/2000F00000P0008/25@260QH0
E`0002@000230P005P0008H2000G00000P0008H25P230QL0E`0002@000220P005`0008<2000I0000
0P0008<25`220QT0E`0002@000220P006@000882000K00000P0008826@220Q/0E`0002@000220P00
6`0008<2000M00000P0008826`230Qd0E`0002@000230P007@0008H2000M00000P0008<27@260Qd0
E`0002@000260P007@0008P2000M00000P0008H27@280Qd0E`0002@000280P007@0008X2000M0000
0P0008P27@2:0Qd0E`0002@0002:0P006`0008`2000M00000P0008X27@2<0Q/0E`0002@000240P00
5P0008H2000G00000P0008H25P240QL0E`0002@000230P005`0008@2000I00000P0008@25`230QT0
E`0002@000230P006@0008<2000K00000P0008<26@230Q/0E`0002@000230P006`0008@2000M0000
0P0008<26`240Qd0E`0002@000240P007@0008H2000M00000P0008@27@260Qd0E`0002@0002F0P00
2P0009H2000J00000P0009H22P2F0QX0E`0002@0002F0P006P0009L2000M00000P0009H26P2G0Qd0
E`0002@0002G0P007@0009T2000M00000P0009L27@2I0Qd0E`0002@0002I0P007@0009X2000M0000
0P0009T27@2J0Qd0E`0002@0002J0P007@0009`2000M00000P0009X27@2L0Qd0E`0002@0002L0P00
6`0009d2000M00000P0009`27@2M0Q/0E`0002@0002G0P002P0009L2000J00000P0009L22P2G0QX0
E`0002@0002G0P006P0009P2000M00000P0009L26P2H0Qd0E`0002@0002H0P007@0009T2000M0000
0P0009P27@2I0Qd0E`0002@0002C0P00400009X2000@00000P0009<2402J0Q00E`0002@0002S0P00
2P000:<2000M00000P000:<22P2S0Qd0E`0002@0002T0P002P000:@2000M00000P000:@22P2T0Qd0
E`0002@0002T0P004@000:H2000C00000P000:@24`2V0Q40E`0002@0002V0P0040000:P2000A0000
0P000:H24@2X0Q00E`0002@0002X0P0040000:X2000@00000P000:P2402Z0Q00E`0002@0002Z0P00
40000:d2000A00000P000:X2402]0Q40E`0002@0002]0P004@000:h2000C00000P000:d24@2^0Q<0
E`0002@0002^0P004`000:h2000M00000P000:h24`2^0Qd0E`0002@0002Z0P0040000:`2000A0000
0P000:X2402/0Q40E`0002@0002/0P004@000:d2000C00000P000:`24@2]0Q<0E`0002@0002]0P00
4`000:d2000M00000P000:d24`2]0Qd0E`0002@0002P0P002P000:@2000:00000P000:022P2T0PX0
E`0002@0002P0P007@000:L2000M00000P000:027@2W0Qd0E`0002@0002Z0P007@000;02000M0000
0P000:X27@2`0Qd0E`0002@000300P0040000<@2000M00000P000<0240340Qd0E`0002@000310P00
40000<@2000K00000P000<4240340Q/0E`0002@000340P0040000<L2000M00000P000<L240340Qd0
E`0002@000370P0040000</2000M00000P000<L2403;0Qd0E`0002@000380P0040000</2000K0000
0P000<P2403;0Q/0E`0002@0003;0P0040000<h2000M00000P000<h2403;0Qd0E`0002@0002m0P00
40000<@2000@00000P000;d240340Q00E`0002@0003<0P0040000=42000@00000P000<`2403A0Q00
E`0002@0003E0P002P000=H2000;00000P000=H22P3E0P/0E`0002@0003E0P002`000=H2000<0000
0P000=D22`3F0P`0E`0002@0003F0P002`000=L2000<00000P000=H2303G0P/0E`0002@0003F0P00
2P000=L2000;00000P000=L22`3F0PX0E`0002@0003F0P0040000=H2000M00000P000=H2403F0Qd0
E`0002@0003G0P0040000=L2000M00000P000=L2403G0Qd0E`0002@0003C0P0040000=L2000@0000
0P000=<2403G0Q00E`0002@0003C0P007@000=X2000M00000P000=<27@3J0Qd0E`0002@0003P0P00
2P000>02000J00000P000>022P3P0QX0E`0002@0003P0P006P000>42000M00000P000>026P3Q0Qd0
E`0002@0003Q0P007@000><2000M00000P000>427@3S0Qd0E`0002@0003S0P007@000>@2000M0000
0P000><27@3T0Qd0E`0002@0003T0P007@000>H2000M00000P000>@27@3V0Qd0E`0002@0003V0P00
6`000>L2000M00000P000>H27@3W0Q/0E`0002@0003Q0P002P000>42000J00000P000>422P3Q0QX0
E`0002@0003Q0P006P000>82000M00000P000>426P3R0Qd0E`0002@0003R0P007@000><2000M0000
0P000>827@3S0Qd0E`0002@0003M0P0040000>@2000@00000P000=d2403T0Q00E`0002@0003^0P00
2P000>h2000M00000P000>h22P3^0Qd0E`0002@0003_0P002P000>l2000M00000P000>l22P3_0Qd0
E`0002@0003_0P004@000?42000C00000P000>l24`3a0Q40E`0002@0003a0P0040000?<2000A0000
0P000?424@3c0Q00E`0002@0003c0P0040000?D2000@00000P000?<2403e0Q00E`0002@0003e0P00
40000?P2000A00000P000?D2403h0Q40E`0002@0003h0P004@000?T2000C00000P000?P24@3i0Q<0
E`0002@0003i0P004`000?T2000M00000P000?T24`3i0Qd0E`0002@0003e0P0040000?L2000A0000
0P000?D2403g0Q40E`0002@0003g0P004@000?P2000C00000P000?L24@3h0Q<0E`0002@0003h0P00
4`000?P2000M00000P000?P24`3h0Qd0E`0002@0003[0P002P000>l2000:00000P000>/22P3_0PX0
E`0002@0003[0P007@000?82000M00000P000>/27@3b0Qd0E`0002@0003e0P007@000?/2000M0000
0P000?D27@3k0Qd0E`0002@0000C0`004`0001@3000D00000P0001@34`0C0a@0E`0002@0000C0`00
500001@3000E00000P0001<3500D0aD0E`0002@0000D0`00500001D3000E00000P0001@35@0E0a@0
E`0002@0000E0`004`0001D3000D00000P0001D3500E0a<0E`0002@0000C0`004@0001D3000C0000
0P0001D34`0C0a40E`0002@0000A0`00400001<3000A00000P0001<34@0A0a00E`0002@0000?0`00
40000143000@00000P000143400?0a00E`0002@0000<0`00400000l3000A00000P0000l3400<0a40
E`0002@0000:0`004@0000`3000C00000P0000`34@0:0a<0E`0002@000090`004`0000X3000F0000
0P0000X34`090aH0E`0002@000090`005P0000T3000H00000P0000T35P090aP0E`0002@000090`00
600000X3000K00000P0000T3600:0a/0E`0002@0000:0`006`0000`3000M00000P0000X36`0<0ad0
E`0002@0000<0`007@0000l3000M00000P0000`37@0?0ad0E`0002@0000?0`007@000103000M0000
0P0000l37@0@0ad0E`0002@0000@0`007@0001<3000M00000P0001037@0C0ad0E`0002@0000C0`00
6`0001D3000M00000P0001<37@0E0a/0E`0002@0000=0`00400000l3000A00000P0000l3400=0a40
E`0002@0000;0`004@0000d3000C00000P0000d34@0;0a<0E`0002@0000:0`004`0000/3000F0000
0P0000/34`0:0aH0E`0002@0000:0`005P0000X3000H00000P0000X35P0:0aP0E`0002@0000:0`00
600000/3000K00000P0000X3600;0a/0E`0002@0000;0`006`0000d3000M00000P0000/36`0=0ad0
E`0002@0000=0`007@0000l3000M00000P0000d37@0?0ad0E`0002@0000K0`004P0002/3000B0000
0P0001/34P0[0a80E`0002@0000K0`00600002/3000H00000P0001/3600[0aP0E`0002@0000d0`00
2P0003L3000;00000P0003L32P0d0`/0E`0002@0000b0`002`0003@3000>00000P0003@32`0b0`h0
E`0002@0000a0`003P000383000B00000P0003833P0a0a80E`0002@0000a0`004P000343000E0000
0P0003434P0a0aD0E`0002@0000a0`005@000383000J00000P0003435@0b0aX0E`0002@0000b0`00
6P0003@3000M00000P0003836P0d0ad0E`0002@0000d0`007@0003L3000M00000P0003@37@0g0ad0
E`0002@0000g0`007@0003P3000M00000P0003L37@0h0ad0E`0002@0000h0`007@0003/3000M0000
0P0003P37@0k0ad0E`0002@0000k0`006P0003d3000M00000P0003/37@0m0aX0E`0002@0000m0`00
5@0003h3000J00000P0003d36P0n0aD0E`0002@0000n0`004P0003h3000E00000P0003h35@0n0a80
E`0002@0000m0`003P0003h3000B00000P0003h34P0m0`h0E`0002@0000k0`002`0003d3000>0000
0P0003d33P0k0`/0E`0002@0000h0`002P0003/3000;00000P0003/32`0h0`X0E`0002@0000g0`00
2P0003P3000:00000P0003P32P0g0`X0E`0002@0000e0`002P0003L3000;00000P0003L32P0e0`/0
E`0002@0000d0`002`0003D3000<00000P0003D32`0d0``0E`0002@0000c0`00300003@3000>0000
0P0003@3300c0`h0E`0002@0000b0`003P0003<3000B00000P0003<33P0b0a80E`0002@0000b0`00
4P000383000E00000P0003834P0b0aD0E`0002@0000b0`005@0003<3000J00000P0003835@0c0aX0
E`0002@0000c0`006P0003@3000L00000P0003<36P0d0a`0E`0002@0000d0`00700003D3000M0000
0P0003@3700e0ad0E`0002@0000e0`007@0003L3000M00000P0003D37@0g0ad0E`0002@0000h0`00
7@0003X3000M00000P0003P37@0j0ad0E`0002@0000j0`00700003/3000M00000P0003X37@0k0a`0
E`0002@0000k0`006P0003`3000L00000P0003/3700l0aX0E`0002@0000l0`005@0003d3000J0000
0P0003`36P0m0aD0E`0002@0000m0`004P0003d3000E00000P0003d35@0m0a80E`0002@0000l0`00
3P0003d3000B00000P0003d34P0l0`h0E`0002@0000k0`00300003`3000>00000P0003`33P0k0``0
E`0002@0000j0`002`0003/3000<00000P0003/3300j0`/0E`0002@0000h0`002P0003X3000;0000
0P0003X32`0h0`X0E`0002@000140`00700004D3000M00000P0004D370140ad0E`0002@000140`00
7@0004D3000M00000P0004@37@150ad0E`0002@000150`007@0004H3000M00000P0004D37@160ad0
E`0002@000150`00700004H3000M00000P0004H37@150a`0E`0002@0001?0`002P000583000;0000
0P0005832P1?0`/0E`0002@0001=0`002`0004l3000>00000P0004l32`1=0`h0E`0002@0001<0`00
3P0004d3000B00000P0004d33P1<0a80E`0002@0001<0`004P0004`3000E00000P0004`34P1<0aD0
E`0002@0001<0`005@0004d3000J00000P0004`35@1=0aX0E`0002@0001=0`006P0004l3000M0000
0P0004d36P1?0ad0E`0002@0001?0`007@000583000M00000P0004l37@1B0ad0E`0002@0001B0`00
7@0005<3000M00000P0005837@1C0ad0E`0002@0001C0`007@0005H3000M00000P0005<37@1F0ad0
E`0002@0001F0`006P0005P3000M00000P0005H37@1H0aX0E`0002@0001H0`005@0005T3000J0000
0P0005P36P1I0aD0E`0002@0001I0`004P0005T3000E00000P0005T35@1I0a80E`0002@0001H0`00
3P0005T3000B00000P0005T34P1H0`h0E`0002@0001F0`002`0005P3000>00000P0005P33P1F0`/0
E`0002@0001C0`002P0005H3000;00000P0005H32`1C0`X0E`0002@0001B0`002P0005<3000:0000
0P0005<32P1B0`X0E`0002@0001@0`002P000583000;00000P0005832P1@0`/0E`0002@0001?0`00
2`000503000<00000P0005032`1?0``0E`0002@0001>0`00300004l3000>00000P0004l3301>0`h0
E`0002@0001=0`003P0004h3000B00000P0004h33P1=0a80E`0002@0001=0`004P0004d3000E0000
0P0004d34P1=0aD0E`0002@0001=0`005@0004h3000J00000P0004d35@1>0aX0E`0002@0001>0`00
6P0004l3000L00000P0004h36P1?0a`0E`0002@0001?0`0070000503000M00000P0004l3701@0ad0
E`0002@0001@0`007@000583000M00000P0005037@1B0ad0E`0002@0001C0`007@0005D3000M0000
0P0005<37@1E0ad0E`0002@0001E0`00700005H3000M00000P0005D37@1F0a`0E`0002@0001F0`00
6P0005L3000L00000P0005H3701G0aX0E`0002@0001G0`005@0005P3000J00000P0005L36P1H0aD0
E`0002@0001H0`004P0005P3000E00000P0005P35@1H0a80E`0002@0001G0`003P0005P3000B0000
0P0005P34P1G0`h0E`0002@0001F0`00300005L3000>00000P0005L33P1F0``0E`0002@0001E0`00
2`0005H3000<00000P0005H3301E0`/0E`0002@0001C0`002P0005D3000;00000P0005D32`1C0`X0
E`0002@0001Q0`002P0006@3000;00000P0006@32P1Q0`/0E`0002@0001O0`002`000643000>0000
0P0006432`1O0`h0E`0002@0001N0`003P0005l3000B00000P0005l33P1N0a80E`0002@0001N0`00
4P0005h3000E00000P0005h34P1N0aD0E`0002@0001N0`005@0005l3000J00000P0005h35@1O0aX0
E`0002@0001O0`006P000643000M00000P0005l36P1Q0ad0E`0002@0001Q0`007@0006@3000M0000
0P0006437@1T0ad0E`0002@0001T0`007@0006D3000M00000P0006@37@1U0ad0E`0002@0001U0`00
7@0006P3000M00000P0006D37@1X0ad0E`0002@0001X0`006P0006X3000M00000P0006P37@1Z0aX0
E`0002@0001Z0`005@0006/3000J00000P0006X36P1[0aD0E`0002@0001[0`004P0006/3000E0000
0P0006/35@1[0a80E`0002@0001Z0`003P0006/3000B00000P0006/34P1Z0`h0E`0002@0001X0`00
2`0006X3000>00000P0006X33P1X0`/0E`0002@0001U0`002P0006P3000;00000P0006P32`1U0`X0
E`0002@0001T0`002P0006D3000:00000P0006D32P1T0`X0E`0002@0001R0`002P0006@3000;0000
0P0006@32P1R0`/0E`0002@0001Q0`002`000683000<00000P0006832`1Q0``0E`0002@0001P0`00
30000643000>00000P000643301P0`h0E`0002@0001O0`003P000603000B00000P0006033P1O0a80
E`0002@0001O0`004P0005l3000E00000P0005l34P1O0aD0E`0002@0001O0`005@000603000J0000
0P0005l35@1P0aX0E`0002@0001P0`006P000643000L00000P0006036P1Q0a`0E`0002@0001Q0`00
70000683000M00000P000643701R0ad0E`0002@0001R0`007@0006@3000M00000P0006837@1T0ad0
E`0002@0001U0`007@0006L3000M00000P0006D37@1W0ad0E`0002@0001W0`00700006P3000M0000
0P0006L37@1X0a`0E`0002@0001X0`006P0006T3000L00000P0006P3701Y0aX0E`0002@0001Y0`00
5@0006X3000J00000P0006T36P1Z0aD0E`0002@0001Z0`004P0006X3000E00000P0006X35@1Z0a80
E`0002@0001Y0`003P0006X3000B00000P0006X34P1Y0`h0E`0002@0001X0`00300006T3000>0000
0P0006T33P1X0``0E`0002@0001W0`002`0006P3000<00000P0006P3301W0`/0E`0002@0001U0`00
2P0006L3000;00000P0006L32`1U0`X0E`0002@0001`0`002P000783000C00000P0007832P1`0a<0
E`0002@0001`0`004@000783000C00000P0007034`1b0a40E`0002@0001b0`00400007D3000A0000
0P0007834@1e0a00E`0002@0001e0`00400007L3000@00000P0007D3401g0a00E`0002@0001g0`00
400007X3000A00000P0007L3401j0a40E`0002@0001j0`004@0007`3000C00000P0007X34@1l0a<0
E`0002@0001l0`004`0007d3000F00000P0007`34`1m0aH0E`0002@0001m0`005P0007d3000H0000
0P0007d35P1m0aP0E`0002@0001l0`00600007d3000K00000P0007d3601l0a/0E`0002@0001j0`00
6`0007`3000M00000P0007`36`1j0ad0E`0002@0001g0`007@0007X3000M00000P0007X37@1g0ad0
E`0002@0001e0`007@0007L3000M00000P0007L37@1e0ad0E`0002@0001b0`007@0007D3000M0000
0P0007D37@1b0ad0E`0002@0001a0`0070000783000M00000P0007837@1a0a`0E`0002@0001`0`00
6P000743000L00000P000743701`0aX0E`0002@0001`0`006@000703000J00000P0007036P1`0aT0
E`0002@0001`0`0060000743000I00000P0007036@1a0aP0E`0002@0001a0`0060000783000I0000
0P000743601b0aT0E`0002@0001a0`006@000783000J00000P0007836@1a0aX0E`0002@0001g0`00
400007T3000A00000P0007L3401i0a40E`0002@0001i0`004@0007/3000C00000P0007T34@1k0a<0
E`0002@0001k0`004`0007`3000F00000P0007/34`1l0aH0E`0002@0001l0`005P0007`3000H0000
0P0007`35P1l0aP0E`0002@0001k0`00600007`3000K00000P0007`3601k0a/0E`0002@0001i0`00
6`0007/3000M00000P0007/36`1i0ad0E`0002@0001g0`007@0007T3000M00000P0007T37@1g0ad0
E`0002@0001b0`002P0007/3000:00000P0007832P1k0`X0E`0002@0001b0`002`0007H3000;0000
0P0007832`1f0`/0E`0002@0001f0`002P0007/3000;00000P0007H32`1k0`X0E`0002@0002L0000
O0<00:40001l0`000P000:40O0>L07`3E`0002@0002L00005`<00:40000G0`000P000:405`>L01L3
E`0002@0002L0000/@800:40002a0P000P000:40/@:L0;42E`0002@0002L0000C0800:40001<0P00
0P000:40C0:L04`2E`0002@0002L0000iP400:40003V0@000P000:40iP6L0>H1E`0002@0002L0000
P0400:4000200@000P000:40P06L0801E`0002@0002L00006`400:40000K0@000P000:406`6L01/1
E`0002@0002L0000]@000:40002e00000P000:40]@2L0;D0E`0002@0002L0000D0000:40001@0000
0P000:40D02L0500E`0002@0000@00002P8002<0000:0P000P0001002P8S00X2E`0002@0000@0000
2@8002<000090P000P0001002@8S00T2E`0002@0000@00000P800100000=0P000P0001003@8@0082
E`0002@0000@0000o`40014000020P000P0001000P8A0?l1E`0002@0000A0000oP400180003o0@00
0P000140o`4B0?h1E`0002@0000B0000o@4001@0003n0@000P000180oP4D0?d1E`0002@0000D0000
o@4001L0003m0@000P0001@0o@4G0?d1E`0002@0000G0000o@4001P0003n0@000P0001L0o@4H0?h1
E`0002@0000H0000oP4001T0003o0@000P0001P0oP4I0?l1E`0002@0000I0000o`4001X000020P00
0P0001T0o`4J0082E`0002@0000J00000P8001X000090P000P0001X00P8J00T2E`0002@0000@0000
0080014000020P000P0001000P8A0002E`0002@0000A0000o`40018000000P000P000140008B0?l1
E`0002@0000B0000oP4001@0003o0@000P000180o`4D0?h1E`0002@0000D0000oP4001L0003n0@00
0P0001@0oP4G0?h1E`0002@0000G0000oP4001P0003o0@000P0001L0oP4H0?l1E`0002@0000H0000
o`4001T000000P000P0001P0o`4I0002E`0002@0000I0000008001X000020P000P0001T0008J0082
E`0002@0000S00001`8002<0000=0P000P0002<03@8S00L2E`0002@0000G0000mP4002<0003f0@00
0P0001L0mP4S0?H1E`0002@0000G0000m@4002<0003e0@000P0001L0m@4S0?D1E`0002@0000I0000
m04001`0003e0@000P0001`0m@4I0?@1E`0002@0000G0000l`4001T0003d0@000P0001T0m04G0?<1
E`0002@0000G0000l@4001L0003c0@000P0001L0l`4G0?41E`0002@0000G0000kP4001L0003a0@00
0P0001L0l@4G0>h1E`0002@0000G0000k@4001L0003^0@000P0001L0kP4G0>d1E`0002@0000G0000
k@4001P0003]0@000P0001L0k@4H0>d1E`0002@0000H0000k@4001T0003^0@000P0001P0k@4I0>h1
E`0002@0000H0000kP4001T0003_0@000P0001T0kP4H0>l1E`0002@0000G0000kP4001P0003_0@00
0P0001P0k`4G0>h1E`0002@0000G0000m@4001L0003i0@000P0001L0n@4G0?D1E`0002@0000S0000
l`4002<0003i0@000P0002<0n@4S0?<1E`0002@0000@0000iP400140003W0@000P000100iP4A0>L1
E`0002@0000A0000iP400180003W0@000P000140i`4B0>H1E`0002@0000A0000i@400180003V0@00
0P000180iP4A0>D1E`0002@0000@0000i@400140003V0@000P000140i@4@0>H1E`0002@0000G0000
iP4002<0003V0@000P0001L0iP4S0>H1E`0002@0000G0000i@4002<0003U0@000P0001L0i@4S0>D1
E`0002@0000G0000i@4001L0003Y0@000P0001L0j@4G0>D1E`0002@0000S0000h`4002<0003Y0@00
0P0002<0j@4S0><1E`0002@0000I0000dP4001X0003C0@000P0001T0dP4J0=<1E`0002@0000J0000
dP4001/0003C0@000P0001X0d`4K0=81E`0002@0000J0000d@4001/0003B0@000P0001/0dP4J0=41
E`0002@0000I0000d@4001X0003A0@000P0001X0d@4I0=41E`0002@0000G0000d@4001T0003C0@00
0P0001T0d@4G0=<1E`0002@0000G0000d`4001L0003E0@000P0001L0d`4G0=D1E`0002@0000G0000
e@4001L0003H0@000P0001L0e@4G0=P1E`0002@0000G0000f04001L0003J0@000P0001L0f04G0=X1
E`0002@0000G0000fP4001T0003L0@000P0001L0fP4I0=`1E`0002@0000I0000g04001`0003M0@00
0P0001T0g04L0=d1E`0002@0000L0000g@4001h0003M0@000P0001`0g@4N0=d1E`0002@0000N0000
g0400200003M0@000P0001h0g@4P0=`1E`0002@0000P0000fP400280003L0@000P000200g04R0=X1
E`0002@0000R0000f04002<0003J0@000P000280fP4S0=P1E`0002@0000S0000eP4002<0003H0@00
0P0002<0f04S0=H1E`0002@0000R0000d`4002<0003F0@000P0002<0eP4R0=<1E`0002@0000P0000
d@400280003C0@000P000280d`4P0=41E`0002@0000G0000f04001L0003I0@000P0001L0f04G0=T1
E`0002@0000G0000f@4001T0003K0@000P0001L0f@4I0=/1E`0002@0000I0000f`4001`0003L0@00
0P0001T0f`4L0=`1E`0002@0000L0000g04001h0003L0@000P0001`0g04N0=`1E`0002@0000N0000
f`400200003L0@000P0001h0g04P0=/1E`0002@0000P0000f@400280003K0@000P000200f`4R0=T1
E`0002@0000R0000f04002<0003I0@000P000280f@4S0=P1E`0002@0000L0000`04001`0003;0@00
0P0001`0b`4L0<01E`0002@0000J0000`04001`000300@000P0001`0`04J0<01E`0002@0000H0000
`04001X000310@000P0001X0`04H0<41E`0002@0000G0000`@4001P000320@000P0001P0`@4G0<81
E`0002@0000G0000`P4001L000340@000P0001L0`P4G0<@1E`0002@0000G0000a04001L000370@00
0P0001L0a04G0<L1E`0002@0000G0000a`4001L000390@000P0001L0a`4G0<T1E`0002@0000G0000
b@4001T0003;0@000P0001L0b@4I0</1E`0002@0000I0000b`4001`0003<0@000P0001T0b`4L0<`1
E`0002@0000L0000c04001h0003<0@000P0001`0c04N0<`1E`0002@0000N0000b`400200003<0@00
0P0001h0c04P0</1E`0002@0000P0000b@400280003;0@000P000200b`4R0<T1E`0002@0000R0000
a`4002<000390@000P000280b@4S0<L1E`0002@0000S0000a@4002<000370@000P0002<0a`4S0<D1
E`0002@0000R0000`P4002<000350@000P0002<0a@4R0<81E`0002@0000P0000`040028000320@00
0P000280`P4P0<01E`0002@0000I0000`@4001`000310@000P0001`0`@4I0<41E`0002@0000G0000
`@4001T000320@000P0001T0`@4G0<81E`0002@0000G0000a`4001L000380@000P0001L0a`4G0<P1
E`0002@0000G0000b04001T0003:0@000P0001L0b04I0<X1E`0002@0000I0000bP4001`0003;0@00
0P0001T0bP4L0</1E`0002@0000L0000b`4001h0003;0@000P0001`0b`4N0</1E`0002@0000N0000
bP400200003;0@000P0001h0b`4P0<X1E`0002@0000P0000b0400280003:0@000P000200bP4R0<P1
E`0002@0000R0000a`4002<000380@000P000280b04S0<L1E`0002@000140000OP<005@0001n0`00
0P0004@0OP=D07h3E`0002@0001N0000L`<00640001d0`000P000640L`=N07@3E`0002@0001L0000
M0<005h0001g0`000P0005h0M0=L07L3E`0002@0001K0000M`<005`0001k0`000P0005`0M`=K07/3
E`0002@0001K0000N`<005/0001n0`000P0005/0N`=K07h3E`0002@0001K0000OP<005`000230`00
0P0005/0OP=L08<3E`0002@0001L0000P`<005h000260`000P0005`0P`=N08H3E`0002@0001N0000
QP<0064000260`000P0005h0QP=Q08H3E`0002@0001Q0000QP<0068000260`000P000640QP=R08H3
E`0002@0001R0000QP<006D000260`000P000680QP=U08H3E`0002@0001U0000P`<006L000260`00
0P0006D0QP=W08<3E`0002@0001W0000OP<006P000230`000P0006L0P`=X07h3E`0002@0001X0000
N`<006P0001n0`000P0006P0OP=X07/3E`0002@0001W0000M`<006P0001k0`000P0006P0N`=W07L3
E`0002@0001U0000M0<006L0001g0`000P0006L0M`=U07@3E`0002@0001R0000L`<006D0001d0`00
0P0006D0M0=R07<3E`0002@0001Q0000L`<00680001c0`000P000680L`=Q07<3E`0002@0001O0000
L`<00640001d0`000P000640L`=O07@3E`0002@0001N0000M0<005l0001e0`000P0005l0M0=N07D3
E`0002@0001M0000M@<005h0001g0`000P0005h0M@=M07L3E`0002@0001L0000M`<005d0001k0`00
0P0005d0M`=L07/3E`0002@0001L0000N`<005`0001n0`000P0005`0N`=L07h3E`0002@0001L0000
OP<005d000230`000P0005`0OP=M08<3E`0002@0001M0000P`<005h000250`000P0005d0P`=N08D3
E`0002@0001N0000Q@<005l000260`000P0005h0Q@=O08H3E`0002@0001O0000QP<0064000260`00
0P0005l0QP=Q08H3E`0002@0001R0000QP<006@000260`000P000680QP=T08H3E`0002@0001T0000
Q@<006D000260`000P0006@0QP=U08D3E`0002@0001U0000P`<006H000250`000P0006D0Q@=V08<3
E`0002@0001V0000OP<006L000230`000P0006H0P`=W07h3E`0002@0001W0000N`<006L0001n0`00
0P0006L0OP=W07/3E`0002@0001V0000M`<006L0001k0`000P0006L0N`=V07L3E`0002@0001U0000
M@<006H0001g0`000P0006H0M`=U07D3E`0002@0001T0000M0<006D0001e0`000P0006D0M@=T07@3
E`0002@0001R0000L`<006@0001d0`000P0006@0M0=R07<3E`0002@0001]0000Q@<006h000260`00
0P0006h0Q@=]08H3E`0002@0001]0000QP<006h000260`000P0006d0QP=^08H3E`0002@0001^0000
QP<006l000260`000P0006h0QP=_08H3E`0002@0001^0000Q@<006l000260`000P0006l0QP=^08D3
E`0002@0001h0000L`<007/0001d0`000P0007/0L`=h07@3E`0002@0001f0000M0<007P0001g0`00
0P0007P0M0=f07L3E`0002@0001e0000M`<007H0001k0`000P0007H0M`=e07/3E`0002@0001e0000
N`<007D0001n0`000P0007D0N`=e07h3E`0002@0001e0000OP<007H000230`000P0007D0OP=f08<3
E`0002@0001f0000P`<007P000260`000P0007H0P`=h08H3E`0002@0001h0000QP<007/000260`00
0P0007P0QP=k08H3E`0002@0001k0000QP<007`000260`000P0007/0QP=l08H3E`0002@0001l0000
QP<007l000260`000P0007`0QP=o08H3E`0002@0001o0000P`<0084000260`000P0007l0QP>108<3
E`0002@000210000OP<0088000230`000P000840P`>207h3E`0002@000220000N`<00880001n0`00
0P000880OP>207/3E`0002@000210000M`<00880001k0`000P000880N`>107L3E`0002@0001o0000
M0<00840001g0`000P000840M`=o07@3E`0002@0001l0000L`<007l0001d0`000P0007l0M0=l07<3
E`0002@0001k0000L`<007`0001c0`000P0007`0L`=k07<3E`0002@0001i0000L`<007/0001d0`00
0P0007/0L`=i07@3E`0002@0001h0000M0<007T0001e0`000P0007T0M0=h07D3E`0002@0001g0000
M@<007P0001g0`000P0007P0M@=g07L3E`0002@0001f0000M`<007L0001k0`000P0007L0M`=f07/3
E`0002@0001f0000N`<007H0001n0`000P0007H0N`=f07h3E`0002@0001f0000OP<007L000230`00
0P0007H0OP=g08<3E`0002@0001g0000P`<007P000250`000P0007L0P`=h08D3E`0002@0001h0000
Q@<007T000260`000P0007P0Q@=i08H3E`0002@0001i0000QP<007/000260`000P0007T0QP=k08H3
E`0002@0001l0000QP<007h000260`000P0007`0QP=n08H3E`0002@0001n0000Q@<007l000260`00
0P0007h0QP=o08D3E`0002@0001o0000P`<0080000250`000P0007l0Q@>008<3E`0002@000200000
OP<0084000230`000P000800P`>107h3E`0002@000210000N`<00840001n0`000P000840OP>107/3
E`0002@000200000M`<00840001k0`000P000840N`>007L3E`0002@0001o0000M@<00800001g0`00
0P000800M`=o07D3E`0002@0001n0000M0<007l0001e0`000P0007l0M@=n07@3E`0002@0001l0000
L`<007h0001d0`000P0007h0M0=l07<3E`0002@000270000L`<008L0001h0`000P0008L0L`>707P3
E`0002@000270000M@<008P0001g0`000P0008L0M`>807D3E`0002@000280000L`<008X0001e0`00
0P0008P0M@>:07<3E`0002@0002:0000L`<008`0001c0`000P0008X0L`><07<3E`0002@0002<0000
L`<00900001f0`000P0008`0L`>@07H3E`0002@0002@0000MP<00980001f0`000P000900MP>B07H3
E`0002@0002B0000M@<009<0001f0`000P000980MP>C07D3E`0002@0002C0000L`<009@0001e0`00
0P0009<0M@>D07<3E`0002@000280000M0<008X0001e0`000P0008P0M@>:07@3E`0002@0002:0000
M0<008`0001d0`000P0008X0M0><07@3E`0002@0002<0000M0<00900001f0`000P0008`0M0>@07H3
E`0002@0002D0000L`<009@0001f0`000P0009@0L`>D07H3E`0002@0002C0000MP<009@0001h0`00
0P0009@0MP>C07P3E`0002@0002?0000N0<009<0001m0`000P0009<0N0>?07d3E`0002@0002>0000
O@<008l0001o0`000P0008l0O@>>07l3E`0002@0002=0000O`<008h000220`000P0008h0O`>=0883
E`0002@0002=0000PP<008d000260`000P0008d0PP>=08H3E`0002@0002>0000N0<009<0001m0`00
0P0009<0N0>>07d3E`0002@0002=0000O@<008h0001o0`000P0008h0O@>=07l3E`0002@0002=0000
O`<008d000220`000P0008d0O`>=0883E`0002@0002=0000PP<008d000260`000P0008d0PP>=08H3
E`0002@00014000060<005@0000H0`000P0004@060=D01P3E`0002@0001N00003@<00640000>0`00
0P0006403@=N00h3E`0002@0001L00003P<005h0000A0`000P0005h03P=L0143E`0002@0001K0000
4@<005`0000E0`000P0005`04@=K01D3E`0002@0001K00005@<005/0000H0`000P0005/05@=K01P3
E`0002@0001K000060<005`0000M0`000P0005/060=L01d3E`0002@0001L00007@<005h0000P0`00
0P0005`07@=N0203E`0002@0001N000080<00640000P0`000P0005h080=Q0203E`0002@0001Q0000
80<00680000P0`000P00064080=R0203E`0002@0001R000080<006D0000P0`000P00068080=U0203
E`0002@0001U00007@<006L0000P0`000P0006D080=W01d3E`0002@0001W000060<006P0000M0`00
0P0006L07@=X01P3E`0002@0001X00005@<006P0000H0`000P0006P060=X01D3E`0002@0001W0000
4@<006P0000E0`000P0006P05@=W0143E`0002@0001U00003P<006L0000A0`000P0006L04@=U00h3
E`0002@0001R00003@<006D0000>0`000P0006D03P=R00d3E`0002@0001Q00003@<00680000=0`00
0P0006803@=Q00d3E`0002@0001O00003@<00640000>0`000P0006403@=O00h3E`0002@0001N0000
3P<005l0000?0`000P0005l03P=N00l3E`0002@0001M00003`<005h0000A0`000P0005h03`=M0143
E`0002@0001L00004@<005d0000E0`000P0005d04@=L01D3E`0002@0001L00005@<005`0000H0`00
0P0005`05@=L01P3E`0002@0001L000060<005d0000M0`000P0005`060=M01d3E`0002@0001M0000
7@<005h0000O0`000P0005d07@=N01l3E`0002@0001N00007`<005l0000P0`000P0005h07`=O0203
E`0002@0001O000080<00640000P0`000P0005l080=Q0203E`0002@0001R000080<006@0000P0`00
0P00068080=T0203E`0002@0001T00007`<006D0000P0`000P0006@080=U01l3E`0002@0001U0000
7@<006H0000O0`000P0006D07`=V01d3E`0002@0001V000060<006L0000M0`000P0006H07@=W01P3
E`0002@0001W00005@<006L0000H0`000P0006L060=W01D3E`0002@0001V00004@<006L0000E0`00
0P0006L05@=V0143E`0002@0001U00003`<006H0000A0`000P0006H04@=U00l3E`0002@0001T0000
3P<006D0000?0`000P0006D03`=T00h3E`0002@0001R00003@<006@0000>0`000P0006@03P=R00d3
E`0002@0001]00007`<006h0000P0`000P0006h07`=]0203E`0002@0001]000080<006h0000P0`00
0P0006d080=^0203E`0002@0001^000080<006l0000P0`000P0006h080=_0203E`0002@0001^0000
7`<006l0000P0`000P0006l080=^01l3E`0002@0001h00003@<007/0000>0`000P0007/03@=h00h3
E`0002@0001f00003P<007P0000A0`000P0007P03P=f0143E`0002@0001e00004@<007H0000E0`00
0P0007H04@=e01D3E`0002@0001e00005@<007D0000H0`000P0007D05@=e01P3E`0002@0001e0000
60<007H0000M0`000P0007D060=f01d3E`0002@0001f00007@<007P0000P0`000P0007H07@=h0203
E`0002@0001h000080<007/0000P0`000P0007P080=k0203E`0002@0001k000080<007`0000P0`00
0P0007/080=l0203E`0002@0001l000080<007l0000P0`000P0007`080=o0203E`0002@0001o0000
7@<00840000P0`000P0007l080>101d3E`0002@00021000060<00880000M0`000P0008407@>201P3
E`0002@0002200005@<00880000H0`000P00088060>201D3E`0002@0002100004@<00880000E0`00
0P0008805@>10143E`0002@0001o00003P<00840000A0`000P0008404@=o00h3E`0002@0001l0000
3@<007l0000>0`000P0007l03P=l00d3E`0002@0001k00003@<007`0000=0`000P0007`03@=k00d3
E`0002@0001i00003@<007/0000>0`000P0007/03@=i00h3E`0002@0001h00003P<007T0000?0`00
0P0007T03P=h00l3E`0002@0001g00003`<007P0000A0`000P0007P03`=g0143E`0002@0001f0000
4@<007L0000E0`000P0007L04@=f01D3E`0002@0001f00005@<007H0000H0`000P0007H05@=f01P3
E`0002@0001f000060<007L0000M0`000P0007H060=g01d3E`0002@0001g00007@<007P0000O0`00
0P0007L07@=h01l3E`0002@0001h00007`<007T0000P0`000P0007P07`=i0203E`0002@0001i0000
80<007/0000P0`000P0007T080=k0203E`0002@0001l000080<007h0000P0`000P0007`080=n0203
E`0002@0001n00007`<007l0000P0`000P0007h080=o01l3E`0002@0001o00007@<00800000O0`00
0P0007l07`>001d3E`0002@00020000060<00840000M0`000P0008007@>101P3E`0002@000210000
5@<00840000H0`000P00084060>101D3E`0002@0002000004@<00840000E0`000P0008405@>00143
E`0002@0001o00003`<00800000A0`000P0008004@=o00l3E`0002@0001n00003P<007l0000?0`00
0P0007l03`=n00h3E`0002@0001l00003@<007h0000>0`000P0007h03P=l00d3E`0002@0002A0000
40<00980000A0`000P00098040>A0143E`0002@0002A00004@<00980000B0`000P0009404@>B0183
E`0002@0002B00004@<009<0000B0`000P0009804P>C0143E`0002@0002C000040<009<0000A0`00
0P0009<04@>C0103E`0002@0002B00003P<009<0000@0`000P0009<040>B00h3E`0002@0002@0000
3@<00980000>0`000P0009803P>@00d3E`0002@0002=00003@<00900000=0`000P0009003@>=00d3
E`0002@0002;00003@<008d0000>0`000P0008d03@>;00h3E`0002@0002900003P<008/0000@0`00
0P0008/03P>90103E`0002@00028000040<008T0000B0`000P0008T040>80183E`0002@000270000
4P<008P0000E0`000P0008P04P>701D3E`0002@0002700005@<008L0000K0`000P0008L05@>701/3
E`0002@0002700006`<008P0000N0`000P0008L06`>801h3E`0002@0002800007P<008X0000P0`00
0P0008P07P>:0203E`0002@0002:000080<008d0000P0`000P0008X080>=0203E`0002@0002=0000
80<008h0000P0`000P0008d080>>0203E`0002@0002>000080<00940000P0`000P0008h080>A0203
E`0002@0002A00007P<009<0000P0`000P00094080>C01h3E`0002@0002C00006`<009@0000N0`00
0P0009<07P>D01/3E`0002@0002D00006P<009@0000K0`000P0009@06`>D01X3E`0002@0002C0000
5`<009@0000J0`000P0009@06P>C01L3E`0002@0002A00005@<009<0000G0`000P0009<05`>A01D3
E`0002@0002>000050<00940000E0`000P0009405@>>01@3E`0002@0002=000050<008h0000D0`00
0P0008h050>=01@3E`0002@0002;000050<008d0000E0`000P0008d050>;01D3E`0002@000290000
5@<008/0000G0`000P0008/05@>901L3E`0002@0002800005`<008T0000J0`000P0008T05`>801X3
E`0002@0002<00003@<008d0000>0`000P0008d03@><00h3E`0002@0002:00003P<008`0000@0`00
0P0008`03P>:0103E`0002@00029000040<008X0000B0`000P0008X040>90183E`0002@000280000
4P<008T0000E0`000P0008T04P>801D3E`0002@0002800005@<008P0000K0`000P0008P05@>801/3
E`0002@0002800006`<008T0000N0`000P0008P06`>901h3E`0002@0002900007P<008/0000P0`00
0P0008T07P>;0203E`0002@0002;000080<008d0000P0`000P0008/080>=0203E`0002@0002>0000
80<00900000P0`000P0008h080>@0203E`0002@0002@00007P<00980000P0`000P00090080>B01h3
E`0002@0002B00006`<009<0000N0`000P0009807P>C01/3E`0002@0002C00006P<009<0000K0`00
0P0009<06`>C01X3E`0002@0002B00005`<009<0000J0`000P0009<06P>B01L3E`0002@0002@0000
5@<00980000G0`000P0009805`>@01D3E`0002@0002>000050<00900000E0`000P0009005@>>01@3
E`0002@000140000/`8005@0002c0P000P0004@0/`9D0;<2E`0002@0001N0000Z0800640002Y0P00
0P000640Z09N0:T2E`0002@0001L0000Z@8005h0002/0P000P0005h0Z@9L0:`2E`0002@0001K0000
[08005`0002`0P000P0005`0[09K0;02E`0002@0001K0000/08005/0002c0P000P0005/0/09K0;<2
E`0002@0001K0000/`8005`0002h0P000P0005/0/`9L0;P2E`0002@0001L0000^08005h0002k0P00
0P0005`0^09N0;/2E`0002@0001N0000^`800640002k0P000P0005h0^`9Q0;/2E`0002@0001Q0000
^`800680002k0P000P000640^`9R0;/2E`0002@0001R0000^`8006D0002k0P000P000680^`9U0;/2
E`0002@0001U0000^08006L0002k0P000P0006D0^`9W0;P2E`0002@0001W0000/`8006P0002h0P00
0P0006L0^09X0;<2E`0002@0001X0000/08006P0002c0P000P0006P0/`9X0;02E`0002@0001W0000
[08006P0002`0P000P0006P0/09W0:`2E`0002@0001U0000Z@8006L0002/0P000P0006L0[09U0:T2
E`0002@0001R0000Z08006D0002Y0P000P0006D0Z@9R0:P2E`0002@0001Q0000Z0800680002X0P00
0P000680Z09Q0:P2E`0002@0001O0000Z0800640002Y0P000P000640Z09O0:T2E`0002@0001N0000
Z@8005l0002Z0P000P0005l0Z@9N0:X2E`0002@0001M0000ZP8005h0002/0P000P0005h0ZP9M0:`2
E`0002@0001L0000[08005d0002`0P000P0005d0[09L0;02E`0002@0001L0000/08005`0002c0P00
0P0005`0/09L0;<2E`0002@0001L0000/`8005d0002h0P000P0005`0/`9M0;P2E`0002@0001M0000
^08005h0002j0P000P0005d0^09N0;X2E`0002@0001N0000^P8005l0002k0P000P0005h0^P9O0;/2
E`0002@0001O0000^`800640002k0P000P0005l0^`9Q0;/2E`0002@0001R0000^`8006@0002k0P00
0P000680^`9T0;/2E`0002@0001T0000^P8006D0002k0P000P0006@0^`9U0;X2E`0002@0001U0000
^08006H0002j0P000P0006D0^P9V0;P2E`0002@0001V0000/`8006L0002h0P000P0006H0^09W0;<2
E`0002@0001W0000/08006L0002c0P000P0006L0/`9W0;02E`0002@0001V0000[08006L0002`0P00
0P0006L0/09V0:`2E`0002@0001U0000ZP8006H0002/0P000P0006H0[09U0:X2E`0002@0001T0000
Z@8006D0002Z0P000P0006D0ZP9T0:T2E`0002@0001R0000Z08006@0002Y0P000P0006@0Z@9R0:P2
E`0002@0001]0000^P8006h0002k0P000P0006h0^P9]0;/2E`0002@0001]0000^`8006h0002k0P00
0P0006d0^`9^0;/2E`0002@0001^0000^`8006l0002k0P000P0006h0^`9_0;/2E`0002@0001^0000
^P8006l0002k0P000P0006l0^`9^0;X2E`0002@0001h0000Z08007/0002Y0P000P0007/0Z09h0:T2
E`0002@0001f0000Z@8007P0002/0P000P0007P0Z@9f0:`2E`0002@0001e0000[08007H0002`0P00
0P0007H0[09e0;02E`0002@0001e0000/08007D0002c0P000P0007D0/09e0;<2E`0002@0001e0000
/`8007H0002h0P000P0007D0/`9f0;P2E`0002@0001f0000^08007P0002k0P000P0007H0^09h0;/2
E`0002@0001h0000^`8007/0002k0P000P0007P0^`9k0;/2E`0002@0001k0000^`8007`0002k0P00
0P0007/0^`9l0;/2E`0002@0001l0000^`8007l0002k0P000P0007`0^`9o0;/2E`0002@0001o0000
^0800840002k0P000P0007l0^`:10;P2E`0002@000210000/`800880002h0P000P000840^0:20;<2
E`0002@000220000/0800880002c0P000P000880/`:20;02E`0002@000210000[0800880002`0P00
0P000880/0:10:`2E`0002@0001o0000Z@800840002/0P000P000840[09o0:T2E`0002@0001l0000
Z08007l0002Y0P000P0007l0Z@9l0:P2E`0002@0001k0000Z08007`0002X0P000P0007`0Z09k0:P2
E`0002@0001i0000Z08007/0002Y0P000P0007/0Z09i0:T2E`0002@0001h0000Z@8007T0002Z0P00
0P0007T0Z@9h0:X2E`0002@0001g0000ZP8007P0002/0P000P0007P0ZP9g0:`2E`0002@0001f0000
[08007L0002`0P000P0007L0[09f0;02E`0002@0001f0000/08007H0002c0P000P0007H0/09f0;<2
E`0002@0001f0000/`8007L0002h0P000P0007H0/`9g0;P2E`0002@0001g0000^08007P0002j0P00
0P0007L0^09h0;X2E`0002@0001h0000^P8007T0002k0P000P0007P0^P9i0;/2E`0002@0001i0000
^`8007/0002k0P000P0007T0^`9k0;/2E`0002@0001l0000^`8007h0002k0P000P0007`0^`9n0;/2
E`0002@0001n0000^P8007l0002k0P000P0007h0^`9o0;X2E`0002@0001o0000^0800800002j0P00
0P0007l0^P:00;P2E`0002@000200000/`800840002h0P000P000800^0:10;<2E`0002@000210000
/0800840002c0P000P000840/`:10;02E`0002@000200000[0800840002`0P000P000840/0:00:`2
E`0002@0001o0000ZP800800002/0P000P000800[09o0:X2E`0002@0001n0000Z@8007l0002Z0P00
0P0007l0ZP9n0:T2E`0002@0001l0000Z08007h0002Y0P000P0007h0Z@9l0:P2E`0002@000270000
Z08008T0002a0P000P0008T0Z0:70;42E`0002@000270000[`8008T0002a0P000P0008L0/@:90:l2
E`0002@000290000[P8008`0002_0P000P0008T0[`:<0:h2E`0002@0002<0000[P8008h0002^0P00
0P0008`0[P:>0:h2E`0002@0002>0000[P800940002_0P000P0008h0[P:A0:l2E`0002@0002A0000
[`8009<0002a0P000P000940[`:C0;42E`0002@0002C0000/@8009@0002d0P000P0009<0/@:D0;@2
E`0002@0002D0000]08009@0002f0P000P0009@0]0:D0;H2E`0002@0002C0000]P8009@0002i0P00
0P0009@0]P:C0;T2E`0002@0002A0000^@8009<0002k0P000P0009<0^@:A0;/2E`0002@0002>0000
^`800940002k0P000P000940^`:>0;/2E`0002@0002<0000^`8008h0002k0P000P0008h0^`:<0;/2
E`0002@000290000^`8008`0002k0P000P0008`0^`:90;/2E`0002@000280000^P8008T0002k0P00
0P0008T0^`:80;X2E`0002@000270000^08008P0002j0P000P0008P0^P:70;P2E`0002@000270000
]`8008L0002h0P000P0008L0^0:70;L2E`0002@000270000]P8008P0002g0P000P0008L0]`:80;H2
E`0002@000280000]P8008T0002g0P000P0008P0]P:90;L2E`0002@000280000]`8008T0002h0P00
0P0008T0]`:80;P2E`0002@0002>0000[P800900002_0P000P0008h0[P:@0:l2E`0002@0002@0000
[`800980002a0P000P000900[`:B0;42E`0002@0002B0000/@8009<0002d0P000P000980/@:C0;@2
E`0002@0002C0000]08009<0002f0P000P0009<0]0:C0;H2E`0002@0002B0000]P8009<0002i0P00
0P0009<0]P:B0;T2E`0002@0002@0000^@800980002k0P000P000980^@:@0;/2E`0002@0002>0000
^`800900002k0P000P000900^`:>0;/2E`0002@000290000Z0800980002X0P000P0008T0Z0:B0:P2
E`0002@000290000Z@8008d0002Y0P000P0008T0Z@:=0:T2E`0002@0002=0000Z0800980002Y0P00
0P0008d0Z@:B0:P2E`0002@000140000C@8005@0001=0P000P0004@0C@9D04d2E`0002@0001N0000
@P80064000130P000P000640@P9N04<2E`0002@0001L0000@`8005h000160P000P0005h0@`9L04H2
E`0002@0001K0000AP8005`0001:0P000P0005`0AP9K04X2E`0002@0001K0000BP8005/0001=0P00
0P0005/0BP9K04d2E`0002@0001K0000C@8005`0001B0P000P0005/0C@9L0582E`0002@0001L0000
DP8005h0001E0P000P0005`0DP9N05D2E`0002@0001N0000E@800640001E0P000P0005h0E@9Q05D2
E`0002@0001Q0000E@800680001E0P000P000640E@9R05D2E`0002@0001R0000E@8006D0001E0P00
0P000680E@9U05D2E`0002@0001U0000DP8006L0001E0P000P0006D0E@9W0582E`0002@0001W0000
C@8006P0001B0P000P0006L0DP9X04d2E`0002@0001X0000BP8006P0001=0P000P0006P0C@9X04X2
E`0002@0001W0000AP8006P0001:0P000P0006P0BP9W04H2E`0002@0001U0000@`8006L000160P00
0P0006L0AP9U04<2E`0002@0001R0000@P8006D000130P000P0006D0@`9R0482E`0002@0001Q0000
@P80068000120P000P000680@P9Q0482E`0002@0001O0000@P80064000130P000P000640@P9O04<2
E`0002@0001N0000@`8005l000140P000P0005l0@`9N04@2E`0002@0001M0000A08005h000160P00
0P0005h0A09M04H2E`0002@0001L0000AP8005d0001:0P000P0005d0AP9L04X2E`0002@0001L0000
BP8005`0001=0P000P0005`0BP9L04d2E`0002@0001L0000C@8005d0001B0P000P0005`0C@9M0582
E`0002@0001M0000DP8005h0001D0P000P0005d0DP9N05@2E`0002@0001N0000E08005l0001E0P00
0P0005h0E09O05D2E`0002@0001O0000E@800640001E0P000P0005l0E@9Q05D2E`0002@0001R0000
E@8006@0001E0P000P000680E@9T05D2E`0002@0001T0000E08006D0001E0P000P0006@0E@9U05@2
E`0002@0001U0000DP8006H0001D0P000P0006D0E09V0582E`0002@0001V0000C@8006L0001B0P00
0P0006H0DP9W04d2E`0002@0001W0000BP8006L0001=0P000P0006L0C@9W04X2E`0002@0001V0000
AP8006L0001:0P000P0006L0BP9V04H2E`0002@0001U0000A08006H000160P000P0006H0AP9U04@2
E`0002@0001T0000@`8006D000140P000P0006D0A09T04<2E`0002@0001R0000@P8006@000130P00
0P0006@0@`9R0482E`0002@0001]0000E08006h0001E0P000P0006h0E09]05D2E`0002@0001]0000
E@8006h0001E0P000P0006d0E@9^05D2E`0002@0001^0000E@8006l0001E0P000P0006h0E@9_05D2
E`0002@0001^0000E08006l0001E0P000P0006l0E@9^05@2E`0002@0001h0000@P8007/000130P00
0P0007/0@P9h04<2E`0002@0001f0000@`8007P000160P000P0007P0@`9f04H2E`0002@0001e0000
AP8007H0001:0P000P0007H0AP9e04X2E`0002@0001e0000BP8007D0001=0P000P0007D0BP9e04d2
E`0002@0001e0000C@8007H0001B0P000P0007D0C@9f0582E`0002@0001f0000DP8007P0001E0P00
0P0007H0DP9h05D2E`0002@0001h0000E@8007/0001E0P000P0007P0E@9k05D2E`0002@0001k0000
E@8007`0001E0P000P0007/0E@9l05D2E`0002@0001l0000E@8007l0001E0P000P0007`0E@9o05D2
E`0002@0001o0000DP800840001E0P000P0007l0E@:10582E`0002@000210000C@800880001B0P00
0P000840DP:204d2E`0002@000220000BP800880001=0P000P000880C@:204X2E`0002@000210000
AP800880001:0P000P000880BP:104H2E`0002@0001o0000@`80084000160P000P000840AP9o04<2
E`0002@0001l0000@P8007l000130P000P0007l0@`9l0482E`0002@0001k0000@P8007`000120P00
0P0007`0@P9k0482E`0002@0001i0000@P8007/000130P000P0007/0@P9i04<2E`0002@0001h0000
@`8007T000140P000P0007T0@`9h04@2E`0002@0001g0000A08007P000160P000P0007P0A09g04H2
E`0002@0001f0000AP8007L0001:0P000P0007L0AP9f04X2E`0002@0001f0000BP8007H0001=0P00
0P0007H0BP9f04d2E`0002@0001f0000C@8007L0001B0P000P0007H0C@9g0582E`0002@0001g0000
DP8007P0001D0P000P0007L0DP9h05@2E`0002@0001h0000E08007T0001E0P000P0007P0E09i05D2
E`0002@0001i0000E@8007/0001E0P000P0007T0E@9k05D2E`0002@0001l0000E@8007h0001E0P00
0P0007`0E@9n05D2E`0002@0001n0000E08007l0001E0P000P0007h0E@9o05@2E`0002@0001o0000
DP800800001D0P000P0007l0E0:00582E`0002@000200000C@800840001B0P000P000800DP:104d2
E`0002@000210000BP800840001=0P000P000840C@:104X2E`0002@000200000AP800840001:0P00
0P000840BP:004H2E`0002@0001o0000A080080000160P000P000800AP9o04@2E`0002@0001n0000
@`8007l000140P000P0007l0A09n04<2E`0002@0001l0000@P8007h000130P000P0007h0@`9l0482
E`0002@0002?0000A08008l0001E0P000P0008l0A0:?05D2E`0002@0002@0000@P800900001E0P00
0P000900@P:@05D2E`0002@000260000@P800900001@0P000P000900@P:60502E`0002@000260000
D08009D0001@0P000P0008H0D0:E0502E`0002@0002=0000E@8009<0001E0P000P0008d0E@:C05D2
E`0002@000140000i`4005@0003W0@000P0004@0i`5D0>L1E`0002@0001N0000g0400640003M0@00
0P000640g05N0=d1E`0002@0001L0000g@4005h0003P0@000P0005h0g@5L0>01E`0002@0001K0000
h04005`0003T0@000P0005`0h05K0>@1E`0002@0001K0000i04005/0003W0@000P0005/0i05K0>L1
E`0002@0001K0000i`4005`0003/0@000P0005/0i`5L0>`1E`0002@0001L0000k04005h0003_0@00
0P0005`0k05N0>l1E`0002@0001N0000k`400640003_0@000P0005h0k`5Q0>l1E`0002@0001Q0000
k`400680003_0@000P000640k`5R0>l1E`0002@0001R0000k`4006D0003_0@000P000680k`5U0>l1
E`0002@0001U0000k04006L0003_0@000P0006D0k`5W0>`1E`0002@0001W0000i`4006P0003/0@00
0P0006L0k05X0>L1E`0002@0001X0000i04006P0003W0@000P0006P0i`5X0>@1E`0002@0001W0000
h04006P0003T0@000P0006P0i05W0>01E`0002@0001U0000g@4006L0003P0@000P0006L0h05U0=d1
E`0002@0001R0000g04006D0003M0@000P0006D0g@5R0=`1E`0002@0001Q0000g0400680003L0@00
0P000680g05Q0=`1E`0002@0001O0000g0400640003M0@000P000640g05O0=d1E`0002@0001N0000
g@4005l0003N0@000P0005l0g@5N0=h1E`0002@0001M0000gP4005h0003P0@000P0005h0gP5M0>01
E`0002@0001L0000h04005d0003T0@000P0005d0h05L0>@1E`0002@0001L0000i04005`0003W0@00
0P0005`0i05L0>L1E`0002@0001L0000i`4005d0003/0@000P0005`0i`5M0>`1E`0002@0001M0000
k04005h0003^0@000P0005d0k05N0>h1E`0002@0001N0000kP4005l0003_0@000P0005h0kP5O0>l1
E`0002@0001O0000k`400640003_0@000P0005l0k`5Q0>l1E`0002@0001R0000k`4006@0003_0@00
0P000680k`5T0>l1E`0002@0001T0000kP4006D0003_0@000P0006@0k`5U0>h1E`0002@0001U0000
k04006H0003^0@000P0006D0kP5V0>`1E`0002@0001V0000i`4006L0003/0@000P0006H0k05W0>L1
E`0002@0001W0000i04006L0003W0@000P0006L0i`5W0>@1E`0002@0001V0000h04006L0003T0@00
0P0006L0i05V0>01E`0002@0001U0000gP4006H0003P0@000P0006H0h05U0=h1E`0002@0001T0000
g@4006D0003N0@000P0006D0gP5T0=d1E`0002@0001R0000g04006@0003M0@000P0006@0g@5R0=`1
E`0002@0001]0000kP4006h0003_0@000P0006h0kP5]0>l1E`0002@0001]0000k`4006h0003_0@00
0P0006d0k`5^0>l1E`0002@0001^0000k`4006l0003_0@000P0006h0k`5_0>l1E`0002@0001^0000
kP4006l0003_0@000P0006l0k`5^0>h1E`0002@0001h0000g04007/0003M0@000P0007/0g05h0=d1
E`0002@0001f0000g@4007P0003P0@000P0007P0g@5f0>01E`0002@0001e0000h04007H0003T0@00
0P0007H0h05e0>@1E`0002@0001e0000i04007D0003W0@000P0007D0i05e0>L1E`0002@0001e0000
i`4007H0003/0@000P0007D0i`5f0>`1E`0002@0001f0000k04007P0003_0@000P0007H0k05h0>l1
E`0002@0001h0000k`4007/0003_0@000P0007P0k`5k0>l1E`0002@0001k0000k`4007`0003_0@00
0P0007/0k`5l0>l1E`0002@0001l0000k`4007l0003_0@000P0007`0k`5o0>l1E`0002@0001o0000
k0400840003_0@000P0007l0k`610>`1E`0002@000210000i`400880003/0@000P000840k0620>L1
E`0002@000220000i0400880003W0@000P000880i`620>@1E`0002@000210000h0400880003T0@00
0P000880i0610>01E`0002@0001o0000g@400840003P0@000P000840h05o0=d1E`0002@0001l0000
g04007l0003M0@000P0007l0g@5l0=`1E`0002@0001k0000g04007`0003L0@000P0007`0g05k0=`1
E`0002@0001i0000g04007/0003M0@000P0007/0g05i0=d1E`0002@0001h0000g@4007T0003N0@00
0P0007T0g@5h0=h1E`0002@0001g0000gP4007P0003P0@000P0007P0gP5g0>01E`0002@0001f0000
h04007L0003T0@000P0007L0h05f0>@1E`0002@0001f0000i04007H0003W0@000P0007H0i05f0>L1
E`0002@0001f0000i`4007L0003/0@000P0007H0i`5g0>`1E`0002@0001g0000k04007P0003^0@00
0P0007L0k05h0>h1E`0002@0001h0000kP4007T0003_0@000P0007P0kP5i0>l1E`0002@0001i0000
k`4007/0003_0@000P0007T0k`5k0>l1E`0002@0001l0000k`4007h0003_0@000P0007`0k`5n0>l1
E`0002@0001n0000kP4007l0003_0@000P0007h0k`5o0>h1E`0002@0001o0000k0400800003^0@00
0P0007l0kP600>`1E`0002@000200000i`400840003/0@000P000800k0610>L1E`0002@000210000
i0400840003W0@000P000840i`610>@1E`0002@000200000h0400840003T0@000P000840i0600>01
E`0002@0001o0000gP400800003P0@000P000800h05o0=h1E`0002@0001n0000g@4007l0003N0@00
0P0007l0gP5n0=d1E`0002@0001l0000g04007h0003M0@000P0007h0g@5l0=`1E`0002@000280000
h04008T0003Q0@000P0008P0h0690>41E`0002@000280000h@4008T0003Q0@000P0008T0h@680>41
E`0002@000270000h@4008P0003Q0@000P0008P0h@670>41E`0002@000270000h04008L0003Q0@00
0P0008L0h@670>01E`0002@000270000gP4008P0003P0@000P0008L0h0680=h1E`0002@000280000
g@4008T0003N0@000P0008P0gP690=d1E`0002@000290000g04008`0003M0@000P0008T0g@6<0=`1
E`0002@0002<0000g04008l0003L0@000P0008`0g06?0=`1E`0002@0002?0000g0400980003M0@00
0P0008l0g06B0=d1E`0002@0002B0000g@4009<0003O0@000P000980g@6C0=l1E`0002@0002C0000
g`4009<0003Q0@000P0009<0g`6C0>41E`0002@0002B0000h@4009<0003S0@000P0009<0h@6B0><1
E`0002@0002?0000h`400980003T0@000P000980h`6?0>@1E`0002@0002=0000i04008l0003T0@00
0P0008l0i06=0>@1E`0002@0002?0000g0400940003M0@000P0008l0g06A0=d1E`0002@0002A0000
g@400980003O0@000P000940g@6B0=l1E`0002@0002B0000g`400980003Q0@000P000980g`6B0>41
E`0002@0002A0000h@400980003S0@000P000980h@6A0><1E`0002@0002?0000h`400940003T0@00
0P000940h`6?0>@1E`0002@0002?0000i0400940003U0@000P0008l0i06A0>D1E`0002@0002A0000
i@4009<0003W0@000P000940i@6C0>L1E`0002@0002C0000i`4009@0003Y0@000P0009<0i`6D0>T1
E`0002@0002D0000j@4009@0003/0@000P0009@0j@6D0>`1E`0002@0002C0000k04009@0003^0@00
0P0009@0k06C0>h1E`0002@0002B0000kP4009<0003_0@000P0009<0kP6B0>l1E`0002@0002?0000
k`400980003_0@000P000980k`6?0>l1E`0002@0002<0000k`4008l0003_0@000P0008l0k`6<0>l1
E`0002@000290000k`4008`0003_0@000P0008`0k`690>l1E`0002@000280000kP4008T0003_0@00
0P0008T0k`680>h1E`0002@000270000k04008P0003^0@000P0008P0kP670>`1E`0002@000270000
j`4008L0003/0@000P0008L0k0670>/1E`0002@000270000jP4008P0003[0@000P0008L0j`680>X1
E`0002@000280000jP4008T0003[0@000P0008P0jP690>/1E`0002@000280000j`4008T0003/0@00
0P0008T0j`680>`1E`0002@0002B0000iP4009<0003Y0@000P000980iP6C0>T1E`0002@0002C0000
j@4009<0003/0@000P0009<0j@6C0>`1E`0002@0002B0000k04009<0003^0@000P0009<0k06B0>h1
E`0002@0002A0000kP400980003_0@000P000980kP6A0>l1E`0002@0002?0000k`400940003_0@00
0P000940k`6?0>l1E`0002@000140000PP4005@000220@000P0004@0PP5D0881E`0002@0001N0000
M`400640001h0@000P000640M`5N07P1E`0002@0001L0000N04005h0001k0@000P0005h0N05L07/1
E`0002@0001K0000N`4005`0001o0@000P0005`0N`5K07l1E`0002@0001K0000O`4005/000220@00
0P0005/0O`5K0881E`0002@0001K0000PP4005`000270@000P0005/0PP5L08L1E`0002@0001L0000
Q`4005h0002:0@000P0005`0Q`5N08X1E`0002@0001N0000RP400640002:0@000P0005h0RP5Q08X1
E`0002@0001Q0000RP400680002:0@000P000640RP5R08X1E`0002@0001R0000RP4006D0002:0@00
0P000680RP5U08X1E`0002@0001U0000Q`4006L0002:0@000P0006D0RP5W08L1E`0002@0001W0000
PP4006P000270@000P0006L0Q`5X0881E`0002@0001X0000O`4006P000220@000P0006P0PP5X07l1
E`0002@0001W0000N`4006P0001o0@000P0006P0O`5W07/1E`0002@0001U0000N04006L0001k0@00
0P0006L0N`5U07P1E`0002@0001R0000M`4006D0001h0@000P0006D0N05R07L1E`0002@0001Q0000
M`400680001g0@000P000680M`5Q07L1E`0002@0001O0000M`400640001h0@000P000640M`5O07P1
E`0002@0001N0000N04005l0001i0@000P0005l0N05N07T1E`0002@0001M0000N@4005h0001k0@00
0P0005h0N@5M07/1E`0002@0001L0000N`4005d0001o0@000P0005d0N`5L07l1E`0002@0001L0000
O`4005`000220@000P0005`0O`5L0881E`0002@0001L0000PP4005d000270@000P0005`0PP5M08L1
E`0002@0001M0000Q`4005h000290@000P0005d0Q`5N08T1E`0002@0001N0000R@4005l0002:0@00
0P0005h0R@5O08X1E`0002@0001O0000RP400640002:0@000P0005l0RP5Q08X1E`0002@0001R0000
RP4006@0002:0@000P000680RP5T08X1E`0002@0001T0000R@4006D0002:0@000P0006@0RP5U08T1
E`0002@0001U0000Q`4006H000290@000P0006D0R@5V08L1E`0002@0001V0000PP4006L000270@00
0P0006H0Q`5W0881E`0002@0001W0000O`4006L000220@000P0006L0PP5W07l1E`0002@0001V0000
N`4006L0001o0@000P0006L0O`5V07/1E`0002@0001U0000N@4006H0001k0@000P0006H0N`5U07T1
E`0002@0001T0000N04006D0001i0@000P0006D0N@5T07P1E`0002@0001R0000M`4006@0001h0@00
0P0006@0N05R07L1E`0002@0001]0000R@4006h0002:0@000P0006h0R@5]08X1E`0002@0001]0000
RP4006h0002:0@000P0006d0RP5^08X1E`0002@0001^0000RP4006l0002:0@000P0006h0RP5_08X1
E`0002@0001^0000R@4006l0002:0@000P0006l0RP5^08T1E`0002@0001h0000M`4007/0001h0@00
0P0007/0M`5h07P1E`0002@0001f0000N04007P0001k0@000P0007P0N05f07/1E`0002@0001e0000
N`4007H0001o0@000P0007H0N`5e07l1E`0002@0001e0000O`4007D000220@000P0007D0O`5e0881
E`0002@0001e0000PP4007H000270@000P0007D0PP5f08L1E`0002@0001f0000Q`4007P0002:0@00
0P0007H0Q`5h08X1E`0002@0001h0000RP4007/0002:0@000P0007P0RP5k08X1E`0002@0001k0000
RP4007`0002:0@000P0007/0RP5l08X1E`0002@0001l0000RP4007l0002:0@000P0007`0RP5o08X1
E`0002@0001o0000Q`400840002:0@000P0007l0RP6108L1E`0002@000210000PP40088000270@00
0P000840Q`620881E`0002@000220000O`40088000220@000P000880PP6207l1E`0002@000210000
N`400880001o0@000P000880O`6107/1E`0002@0001o0000N0400840001k0@000P000840N`5o07P1
E`0002@0001l0000M`4007l0001h0@000P0007l0N05l07L1E`0002@0001k0000M`4007`0001g0@00
0P0007`0M`5k07L1E`0002@0001i0000M`4007/0001h0@000P0007/0M`5i07P1E`0002@0001h0000
N04007T0001i0@000P0007T0N05h07T1E`0002@0001g0000N@4007P0001k0@000P0007P0N@5g07/1
E`0002@0001f0000N`4007L0001o0@000P0007L0N`5f07l1E`0002@0001f0000O`4007H000220@00
0P0007H0O`5f0881E`0002@0001f0000PP4007L000270@000P0007H0PP5g08L1E`0002@0001g0000
Q`4007P000290@000P0007L0Q`5h08T1E`0002@0001h0000R@4007T0002:0@000P0007P0R@5i08X1
E`0002@0001i0000RP4007/0002:0@000P0007T0RP5k08X1E`0002@0001l0000RP4007h0002:0@00
0P0007`0RP5n08X1E`0002@0001n0000R@4007l0002:0@000P0007h0RP5o08T1E`0002@0001o0000
Q`40080000290@000P0007l0R@6008L1E`0002@000200000PP40084000270@000P000800Q`610881
E`0002@000210000O`40084000220@000P000840PP6107l1E`0002@000200000N`400840001o0@00
0P000840O`6007/1E`0002@0001o0000N@400800001k0@000P000800N`5o07T1E`0002@0001n0000
N04007l0001i0@000P0007l0N@5n07P1E`0002@0001l0000M`4007h0001h0@000P0007h0N05l07L1
E`0002@000280000N`4008T0001l0@000P0008P0N`6907`1E`0002@000280000O04008T0001l0@00
0P0008T0O06807`1E`0002@000270000O04008P0001l0@000P0008P0O06707`1E`0002@000270000
N`4008L0001l0@000P0008L0O06707/1E`0002@000270000N@4008P0001k0@000P0008L0N`6807T1
E`0002@000280000N04008T0001i0@000P0008P0N@6907P1E`0002@000290000M`4008`0001h0@00
0P0008T0N06<07L1E`0002@0002<0000M`4008l0001g0@000P0008`0M`6?07L1E`0002@0002?0000
M`400980001h0@000P0008l0M`6B07P1E`0002@0002B0000N04009<0001i0@000P000980N06C07T1
E`0002@0002C0000N@4009@0001k0@000P0009<0N@6D07/1E`0002@0002D0000N`4009@0001l0@00
0P0009@0N`6D07`1E`0002@0002C0000O04009@0001n0@000P0009@0O06C07h1E`0002@0002@0000
OP4009<000200@000P0009<0OP6@0801E`0002@0002<0000P040090000220@000P000900P06<0881
E`0002@0002:0000PP4008`000230@000P0008`0PP6:08<1E`0002@000280000P`4008X000250@00
0P0008X0P`6808D1E`0002@000270000Q@4008P000280@000P0008P0Q@6708P1E`0002@000270000
R04008L0002:0@000P0008L0R06708X1E`0002@0002?0000M`400940001h0@000P0008l0M`6A07P1
E`0002@0002A0000N0400980001i0@000P000940N06B07T1E`0002@0002B0000N@4009<0001k0@00
0P000980N@6C07/1E`0002@0002C0000N`4009<0001l0@000P0009<0N`6C07`1E`0002@0002B0000
O04009<0001n0@000P0009<0O06B07h1E`0002@0002?0000OP40098000200@000P000980OP6?0801
E`0002@0002<0000P04008l000220@000P0008l0P06<0881E`0002@000270000R04008P000290@00
0P0008L0R@6808P1E`0002@000280000R04008X000280@000P0008P0R06:08P1E`0002@0002:0000
R04008h0002:0@000P0008X0R06>08X1E`0002@0002>0000RP400940002:0@000P0008h0RP6A08X1
E`0002@0002A0000R@4009<0002:0@000P000940RP6C08T1E`0002@0002C0000R04009@000290@00
0P0009<0R@6D08P1E`0002@0002:0000R04008h0002:0@000P0008X0R06>08X1E`0002@0002>0000
RP400980002:0@000P0008h0RP6B08X1E`0002@0002B0000RP4009<0002:0@000P000980RP6C08X1
E`0002@0002C0000R04009@0002:0@000P0009<0RP6D08P1E`0002@0002D0000QP4009@000280@00
0P0009@0R06D08H1E`0002@000140000704005@0000L0@000P0004@0705D01`1E`0002@0001N0000
4@400640000B0@000P0006404@5N0181E`0002@0001L00004P4005h0000E0@000P0005h04P5L01D1
E`0002@0001K00005@4005`0000I0@000P0005`05@5K01T1E`0002@0001K00006@4005/0000L0@00
0P0005/06@5K01`1E`0002@0001K0000704005`0000Q0@000P0005/0705L0241E`0002@0001L0000
8@4005h0000T0@000P0005`08@5N02@1E`0002@0001N000090400640000T0@000P0005h0905Q02@1
E`0002@0001Q000090400680000T0@000P000640905R02@1E`0002@0001R0000904006D0000T0@00
0P000680905U02@1E`0002@0001U00008@4006L0000T0@000P0006D0905W0241E`0002@0001W0000
704006P0000Q0@000P0006L08@5X01`1E`0002@0001X00006@4006P0000L0@000P0006P0705X01T1
E`0002@0001W00005@4006P0000I0@000P0006P06@5W01D1E`0002@0001U00004P4006L0000E0@00
0P0006L05@5U0181E`0002@0001R00004@4006D0000B0@000P0006D04P5R0141E`0002@0001Q0000
4@400680000A0@000P0006804@5Q0141E`0002@0001O00004@400640000B0@000P0006404@5O0181
E`0002@0001N00004P4005l0000C0@000P0005l04P5N01<1E`0002@0001M00004`4005h0000E0@00
0P0005h04`5M01D1E`0002@0001L00005@4005d0000I0@000P0005d05@5L01T1E`0002@0001L0000
6@4005`0000L0@000P0005`06@5L01`1E`0002@0001L0000704005d0000Q0@000P0005`0705M0241
E`0002@0001M00008@4005h0000S0@000P0005d08@5N02<1E`0002@0001N00008`4005l0000T0@00
0P0005h08`5O02@1E`0002@0001O000090400640000T0@000P0005l0905Q02@1E`0002@0001R0000
904006@0000T0@000P000680905T02@1E`0002@0001T00008`4006D0000T0@000P0006@0905U02<1
E`0002@0001U00008@4006H0000S0@000P0006D08`5V0241E`0002@0001V0000704006L0000Q0@00
0P0006H08@5W01`1E`0002@0001W00006@4006L0000L0@000P0006L0705W01T1E`0002@0001V0000
5@4006L0000I0@000P0006L06@5V01D1E`0002@0001U00004`4006H0000E0@000P0006H05@5U01<1
E`0002@0001T00004P4006D0000C0@000P0006D04`5T0181E`0002@0001R00004@4006@0000B0@00
0P0006@04P5R0141E`0002@0001]00008`4006h0000T0@000P0006h08`5]02@1E`0002@0001]0000
904006h0000T0@000P0006d0905^02@1E`0002@0001^0000904006l0000T0@000P0006h0905_02@1
E`0002@0001^00008`4006l0000T0@000P0006l0905^02<1E`0002@0001h00004@4007/0000B0@00
0P0007/04@5h0181E`0002@0001f00004P4007P0000E0@000P0007P04P5f01D1E`0002@0001e0000
5@4007H0000I0@000P0007H05@5e01T1E`0002@0001e00006@4007D0000L0@000P0007D06@5e01`1
E`0002@0001e0000704007H0000Q0@000P0007D0705f0241E`0002@0001f00008@4007P0000T0@00
0P0007H08@5h02@1E`0002@0001h0000904007/0000T0@000P0007P0905k02@1E`0002@0001k0000
904007`0000T0@000P0007/0905l02@1E`0002@0001l0000904007l0000T0@000P0007`0905o02@1
E`0002@0001o00008@400840000T0@000P0007l090610241E`0002@00021000070400880000Q0@00
0P0008408@6201`1E`0002@0002200006@400880000L0@000P000880706201T1E`0002@000210000
5@400880000I0@000P0008806@6101D1E`0002@0001o00004P400840000E0@000P0008405@5o0181
E`0002@0001l00004@4007l0000B0@000P0007l04P5l0141E`0002@0001k00004@4007`0000A0@00
0P0007`04@5k0141E`0002@0001i00004@4007/0000B0@000P0007/04@5i0181E`0002@0001h0000
4P4007T0000C0@000P0007T04P5h01<1E`0002@0001g00004`4007P0000E0@000P0007P04`5g01D1
E`0002@0001f00005@4007L0000I0@000P0007L05@5f01T1E`0002@0001f00006@4007H0000L0@00
0P0007H06@5f01`1E`0002@0001f0000704007L0000Q0@000P0007H0705g0241E`0002@0001g0000
8@4007P0000S0@000P0007L08@5h02<1E`0002@0001h00008`4007T0000T0@000P0007P08`5i02@1
E`0002@0001i0000904007/0000T0@000P0007T0905k02@1E`0002@0001l0000904007h0000T0@00
0P0007`0905n02@1E`0002@0001n00008`4007l0000T0@000P0007h0905o02<1E`0002@0001o0000
8@400800000S0@000P0007l08`600241E`0002@00020000070400840000Q0@000P0008008@6101`1
E`0002@0002100006@400840000L0@000P000840706101T1E`0002@0002000005@400840000I0@00
0P0008406@6001D1E`0002@0001o00004`400800000E0@000P0008005@5o01<1E`0002@0001n0000
4P4007l0000C0@000P0007l04`5n0181E`0002@0001l00004@4007h0000B0@000P0007h04P5l0141
E`0002@0002:0000504008`0000E0@000P0008X05@6<01@1E`0002@0002<00004@4008h0000D0@00
0P0008`0506>0141E`0002@0002>00004@4008h0000T0@000P0008h04@6>02@1E`0002@0002=0000
4P4008d0000T0@000P0008d04P6=02@1E`0002@0002:000090400980000T0@000P0008X0906B02@1
E`0002@0001N0000[0000640002]00000P000640[01N0:d0E`0002@0001L0000[@0005h0002`0000
0P0005h0[@1L0;00E`0002@0001K0000/00005`0002d00000P0005`0/01K0;@0E`0002@0001K0000
]00005/0002g00000P0005/0]01K0;L0E`0002@0001K0000]`0005`0002l00000P0005/0]`1L0;`0
E`0002@0001L0000_00005h0002o00000P0005`0_01N0;l0E`0002@0001N0000_`000640002o0000
0P0005h0_`1Q0;l0E`0002@0001Q0000_`000680002o00000P000640_`1R0;l0E`0002@0001R0000
_`0006D0002o00000P000680_`1U0;l0E`0002@0001U0000_00006L0002o00000P0006D0_`1W0;`0
E`0002@0001W0000]`0006P0002l00000P0006L0_01X0;L0E`0002@0001X0000]00006P0002g0000
0P0006P0]`1X0;@0E`0002@0001W0000/00006P0002d00000P0006P0]01W0;00E`0002@0001U0000
[@0006L0002`00000P0006L0/01U0:d0E`0002@0001R0000[00006D0002]00000P0006D0[@1R0:`0
E`0002@0001Q0000[0000680002/00000P000680[01Q0:`0E`0002@0001O0000[0000640002]0000
0P000640[01O0:d0E`0002@0001N0000[@0005l0002^00000P0005l0[@1N0:h0E`0002@0001M0000
[P0005h0002`00000P0005h0[P1M0;00E`0002@0001L0000/00005d0002d00000P0005d0/01L0;@0
E`0002@0001L0000]00005`0002g00000P0005`0]01L0;L0E`0002@0001L0000]`0005d0002l0000
0P0005`0]`1M0;`0E`0002@0001M0000_00005h0002n00000P0005d0_01N0;h0E`0002@0001N0000
_P0005l0002o00000P0005h0_P1O0;l0E`0002@0001O0000_`000640002o00000P0005l0_`1Q0;l0
E`0002@0001R0000_`0006@0002o00000P000680_`1T0;l0E`0002@0001T0000_P0006D0002o0000
0P0006@0_`1U0;h0E`0002@0001U0000_00006H0002n00000P0006D0_P1V0;`0E`0002@0001V0000
]`0006L0002l00000P0006H0_01W0;L0E`0002@0001W0000]00006L0002g00000P0006L0]`1W0;@0
E`0002@0001V0000/00006L0002d00000P0006L0]01V0;00E`0002@0001U0000[P0006H0002`0000
0P0006H0/01U0:h0E`0002@0001T0000[@0006D0002^00000P0006D0[P1T0:d0E`0002@0001R0000
[00006@0002]00000P0006@0[@1R0:`0E`0002@0001]0000_P0006h0002o00000P0006h0_P1]0;l0
E`0002@0001]0000_`0006h0002o00000P0006d0_`1^0;l0E`0002@0001^0000_`0006l0002o0000
0P0006h0_`1_0;l0E`0002@0001^0000_P0006l0002o00000P0006l0_`1^0;h0E`0002@0001h0000
[00007/0002]00000P0007/0[01h0:d0E`0002@0001f0000[@0007P0002`00000P0007P0[@1f0;00
E`0002@0001e0000/00007H0002d00000P0007H0/01e0;@0E`0002@0001e0000]00007D0002g0000
0P0007D0]01e0;L0E`0002@0001e0000]`0007H0002l00000P0007D0]`1f0;`0E`0002@0001f0000
_00007P0002o00000P0007H0_01h0;l0E`0002@0001h0000_`0007/0002o00000P0007P0_`1k0;l0
E`0002@0001k0000_`0007`0002o00000P0007/0_`1l0;l0E`0002@0001l0000_`0007l0002o0000
0P0007`0_`1o0;l0E`0002@0001o0000_0000840002o00000P0007l0_`210;`0E`0002@000210000
]`000880002l00000P000840_0220;L0E`0002@000220000]0000880002g00000P000880]`220;@0
E`0002@000210000/0000880002d00000P000880]0210;00E`0002@0001o0000[@000840002`0000
0P000840/01o0:d0E`0002@0001l0000[00007l0002]00000P0007l0[@1l0:`0E`0002@0001k0000
[00007`0002/00000P0007`0[01k0:`0E`0002@0001i0000[00007/0002]00000P0007/0[01i0:d0
E`0002@0001h0000[@0007T0002^00000P0007T0[@1h0:h0E`0002@0001g0000[P0007P0002`0000
0P0007P0[P1g0;00E`0002@0001f0000/00007L0002d00000P0007L0/01f0;@0E`0002@0001f0000
]00007H0002g00000P0007H0]01f0;L0E`0002@0001f0000]`0007L0002l00000P0007H0]`1g0;`0
E`0002@0001g0000_00007P0002n00000P0007L0_01h0;h0E`0002@0001h0000_P0007T0002o0000
0P0007P0_P1i0;l0E`0002@0001i0000_`0007/0002o00000P0007T0_`1k0;l0E`0002@0001l0000
_`0007h0002o00000P0007`0_`1n0;l0E`0002@0001n0000_P0007l0002o00000P0007h0_`1o0;h0
E`0002@0001o0000_0000800002n00000P0007l0_P200;`0E`0002@000200000]`000840002l0000
0P000800_0210;L0E`0002@000210000]0000840002g00000P000840]`210;@0E`0002@000200000
/0000840002d00000P000840]0200;00E`0002@0001o0000[P000800002`00000P000800/01o0:h0
E`0002@0001n0000[@0007l0002^00000P0007l0[P1n0:d0E`0002@0001l0000[00007h0002]0000
0P0007h0[@1l0:`0E`0002@0002:0000[00008d0002]00000P0008d0[02:0:d0E`0002@000280000
[@0008X0002`00000P0008X0[@280;00E`0002@000270000/00008P0002d00000P0008P0/0270;@0
E`0002@000270000]00008L0002g00000P0008L0]0270;L0E`0002@000270000]`0008P0002l0000
0P0008L0]`280;`0E`0002@000280000_00008X0002o00000P0008P0_02:0;l0E`0002@0002:0000
_`0008d0002o00000P0008X0_`2=0;l0E`0002@0002=0000_`0008h0002o00000P0008d0_`2>0;l0
E`0002@0002>0000_`000940002o00000P0008h0_`2A0;l0E`0002@0002A0000_00009<0002o0000
0P000940_`2C0;`0E`0002@0002C0000]`0009@0002l00000P0009<0_02D0;L0E`0002@0002D0000
]00009@0002g00000P0009@0]`2D0;@0E`0002@0002C0000/00009@0002d00000P0009@0]02C0;00
E`0002@0002A0000[@0009<0002`00000P0009<0/02A0:d0E`0002@0002>0000[0000940002]0000
0P000940[@2>0:`0E`0002@0002=0000[00008h0002/00000P0008h0[02=0:`0E`0002@0002;0000
[00008d0002]00000P0008d0[02;0:d0E`0002@0002:0000[@0008/0002^00000P0008/0[@2:0:h0
E`0002@000290000[P0008X0002`00000P0008X0[P290;00E`0002@000280000/00008T0002d0000
0P0008T0/0280;@0E`0002@000280000]00008P0002g00000P0008P0]0280;L0E`0002@000280000
]`0008T0002l00000P0008P0]`290;`0E`0002@000290000_00008X0002n00000P0008T0_02:0;h0
E`0002@0002:0000_P0008/0002o00000P0008X0_P2;0;l0E`0002@0002;0000_`0008d0002o0000
0P0008/0_`2=0;l0E`0002@0002>0000_`000900002o00000P0008h0_`2@0;l0E`0002@0002@0000
_P000940002o00000P000900_`2A0;h0E`0002@0002A0000_0000980002n00000P000940_P2B0;`0
E`0002@0002B0000]`0009<0002l00000P000980_02C0;L0E`0002@0002C0000]00009<0002g0000
0P0009<0]`2C0;@0E`0002@0002B0000/00009<0002d00000P0009<0]02B0;00E`0002@0002A0000
[P000980002`00000P000980/02A0:h0E`0002@0002@0000[@000940002^00000P000940[P2@0:d0
E`0002@0002>0000[0000900002]00000P000900[@2>0:`0E`0002@0001N0000AP00064000170000
0P000640AP1N04L0E`0002@0001L0000A`0005h0001:00000P0005h0A`1L04X0E`0002@0001K0000
BP0005`0001>00000P0005`0BP1K04h0E`0002@0001K0000CP0005/0001A00000P0005/0CP1K0540
E`0002@0001K0000D@0005`0001F00000P0005/0D@1L05H0E`0002@0001L0000EP0005h0001I0000
0P0005`0EP1N05T0E`0002@0001N0000F@000640001I00000P0005h0F@1Q05T0E`0002@0001Q0000
F@000680001I00000P000640F@1R05T0E`0002@0001R0000F@0006D0001I00000P000680F@1U05T0
E`0002@0001U0000EP0006L0001I00000P0006D0F@1W05H0E`0002@0001W0000D@0006P0001F0000
0P0006L0EP1X0540E`0002@0001X0000CP0006P0001A00000P0006P0D@1X04h0E`0002@0001W0000
BP0006P0001>00000P0006P0CP1W04X0E`0002@0001U0000A`0006L0001:00000P0006L0BP1U04L0
E`0002@0001R0000AP0006D0001700000P0006D0A`1R04H0E`0002@0001Q0000AP00068000160000
0P000680AP1Q04H0E`0002@0001O0000AP000640001700000P000640AP1O04L0E`0002@0001N0000
A`0005l0001800000P0005l0A`1N04P0E`0002@0001M0000B00005h0001:00000P0005h0B01M04X0
E`0002@0001L0000BP0005d0001>00000P0005d0BP1L04h0E`0002@0001L0000CP0005`0001A0000
0P0005`0CP1L0540E`0002@0001L0000D@0005d0001F00000P0005`0D@1M05H0E`0002@0001M0000
EP0005h0001H00000P0005d0EP1N05P0E`0002@0001N0000F00005l0001I00000P0005h0F01O05T0
E`0002@0001O0000F@000640001I00000P0005l0F@1Q05T0E`0002@0001R0000F@0006@0001I0000
0P000680F@1T05T0E`0002@0001T0000F00006D0001I00000P0006@0F@1U05P0E`0002@0001U0000
EP0006H0001H00000P0006D0F01V05H0E`0002@0001V0000D@0006L0001F00000P0006H0EP1W0540
E`0002@0001W0000CP0006L0001A00000P0006L0D@1W04h0E`0002@0001V0000BP0006L0001>0000
0P0006L0CP1V04X0E`0002@0001U0000B00006H0001:00000P0006H0BP1U04P0E`0002@0001T0000
A`0006D0001800000P0006D0B01T04L0E`0002@0001R0000AP0006@0001700000P0006@0A`1R04H0
E`0002@0001]0000F00006h0001I00000P0006h0F01]05T0E`0002@0001]0000F@0006h0001I0000
0P0006d0F@1^05T0E`0002@0001^0000F@0006l0001I00000P0006h0F@1_05T0E`0002@0001^0000
F00006l0001I00000P0006l0F@1^05P0E`0002@0001h0000AP0007/0001700000P0007/0AP1h04L0
E`0002@0001f0000A`0007P0001:00000P0007P0A`1f04X0E`0002@0001e0000BP0007H0001>0000
0P0007H0BP1e04h0E`0002@0001e0000CP0007D0001A00000P0007D0CP1e0540E`0002@0001e0000
D@0007H0001F00000P0007D0D@1f05H0E`0002@0001f0000EP0007P0001I00000P0007H0EP1h05T0
E`0002@0001h0000F@0007/0001I00000P0007P0F@1k05T0E`0002@0001k0000F@0007`0001I0000
0P0007/0F@1l05T0E`0002@0001l0000F@0007l0001I00000P0007`0F@1o05T0E`0002@0001o0000
EP000840001I00000P0007l0F@2105H0E`0002@000210000D@000880001F00000P000840EP220540
E`0002@000220000CP000880001A00000P000880D@2204h0E`0002@000210000BP000880001>0000
0P000880CP2104X0E`0002@0001o0000A`000840001:00000P000840BP1o04L0E`0002@0001l0000
AP0007l0001700000P0007l0A`1l04H0E`0002@0001k0000AP0007`0001600000P0007`0AP1k04H0
E`0002@0001i0000AP0007/0001700000P0007/0AP1i04L0E`0002@0001h0000A`0007T000180000
0P0007T0A`1h04P0E`0002@0001g0000B00007P0001:00000P0007P0B01g04X0E`0002@0001f0000
BP0007L0001>00000P0007L0BP1f04h0E`0002@0001f0000CP0007H0001A00000P0007H0CP1f0540
E`0002@0001f0000D@0007L0001F00000P0007H0D@1g05H0E`0002@0001g0000EP0007P0001H0000
0P0007L0EP1h05P0E`0002@0001h0000F00007T0001I00000P0007P0F01i05T0E`0002@0001i0000
F@0007/0001I00000P0007T0F@1k05T0E`0002@0001l0000F@0007h0001I00000P0007`0F@1n05T0
E`0002@0001n0000F00007l0001I00000P0007h0F@1o05P0E`0002@0001o0000EP000800001H0000
0P0007l0F02005H0E`0002@000200000D@000840001F00000P000800EP210540E`0002@000210000
CP000840001A00000P000840D@2104h0E`0002@000200000BP000840001>00000P000840CP2004X0
E`0002@0001o0000B0000800001:00000P000800BP1o04P0E`0002@0001n0000A`0007l000180000
0P0007l0B01n04L0E`0002@0001l0000AP0007h0001700000P0007h0A`1l04H0E`0002@0002:0000
B@0008`0001:00000P0008X0BP2<04T0E`0002@0002<0000AP0008h0001900000P0008`0B@2>04H0
E`0002@0002>0000AP0008h0001I00000P0008h0AP2>05T0E`0002@0002=0000A`0008d0001I0000
0P0008d0A`2=05T0E`0002@0002:0000F@000980001I00000P0008X0F@2B05T0E`0002@0002V0000
Q@<00:H0002=0`000P000:H0Q@>V08d3E`0002@0003N0000Q@<00=h0002=0`000P000=h0Q@?N08d3
E`0002@0000G0@00Q@<001L1002=0`000P0001L1Q@<G0Hd3E`0002@0001?0@00Q@<004l1002=0`00
0P0004l1Q@=?0Hd3E`0002@000280@00Q@<008P1002=0`000P0008P1Q@>80Hd3E`0002@000300@00
Q@<00<01002=0`000P000<01Q@?00Hd3E`0002@0003h0@00Q@<00?P1002=0`000P000?P1Q@?h0Hd3
E`0002@0000a0P00Q@<00342002=0`000P000342Q@<a0Xd3E`0002@0001Y0P00Q@<006T2002=0`00
0P0006T2Q@=Y0Xd3E`0002@0002R0P00Q@<00:82002=0`000P000:82Q@>R0Xd3E`0002@0003J0P00
Q@<00=X2002=0`000P000=X2Q@?J0Xd3E`0002@0000B0`00Q@<00183002=0`000P000183Q@<B0hd3
E`0002@0001;0`00Q@<004/3002=0`000P0004/3Q@=;0hd3E`0002@000230`00Q@<008<3002=0`00
0P0008<3Q@>30hd3E`0002@0002k0`00Q@<00;/3002=0`000P000;/3Q@>k0hd3E`0002@0003d0`00
Q@<00?@3002=0`000P000?@3Q@?d0hd3E`0002@0000/1000Q@<002`4002=0`000P0002`4Q@</18d3
E`0002@0001U1000Q@<006D4002=0`000P0006D4Q@=U18d3E`0002@0002M1000Q@<009d4002=0`00
0P0009d4Q@>M18d3E`0002@0003E1000Q@<00=D4002=0`000P000=D4Q@?E18d3E`0002@0002Q0P00
b`<00:42003N0`000P000:42b`>Q0]h3E`0002@0002Q0P00b`<00:42003N0`000P000:42b`>Q0]h3
E`0002@0002J0P00b`<009/2003@0`000P0009/2b`>J0]03E`0002@0002J0P00b`<009X2003@0`00
0P0009X2d0>J0//3E`0002@0002J0P00b`<00:P2003;0`000P0009X2b`>X0//3E`0002@0002X0P00
b`<00:P2003@0`000P000:P2b`>X0]03E`0002@0002W0P00b`<00:P2003@0`000P000:P2d0>W0//3
E`0002@0002N0P00gP<00:@2003N0`000P0009h2gP>T0]h3E`0002@0002]0P00b`<00:h2003<0`00
0P000:h2b`>]0/`3E`0002@0002]0P00c0<00:h2003=0`000P000:d2c0>^0/d3E`0002@0002^0P00
c0<00:l2003=0`000P000:h2c@>_0/`3E`0002@0002^0P00b`<00:l2003<0`000P000:l2c0>^0//3
E`0002@0002^0P00d@<00:h2003N0`000P000:h2d@>^0]h3E`0002@0002_0P00d@<00:l2003N0`00
0P000:l2d@>_0]h3E`0002@0002[0P00d@<00:l2003A0`000P000:/2d@>_0]43E`0002@0002[0P00
gP<00;82003N0`000P000:/2gP>b0]h3E`0002@0002h0P00d@<00;P2003N0`000P000;P2d@>h0]h3
E`0002@0002i0P00d@<00;T2003N0`000P000;T2d@>i0]h3E`0002@0002i0P00dP<00;/2003D0`00
0P000;T2e0>k0]83E`0002@0002k0P00d@<00;d2003B0`000P000;/2dP>m0]43E`0002@0002m0P00
d@<00;l2003A0`000P000;d2d@>o0]43E`0002@0002o0P00d@<00<82003B0`000P000;l2d@?20]83
E`0002@000320P00dP<00<<2003D0`000P000<82dP?30]@3E`0002@000330P00e0<00<<2003N0`00
0P000<<2e0?30]h3E`0002@0002o0P00d@<00<42003B0`000P000;l2d@?10]83E`0002@000310P00
dP<00<82003D0`000P000<42dP?20]@3E`0002@000320P00e0<00<82003N0`000P000<82e0?20]h3
E`0002@000330P00dP<00<@2003D0`000P000<<2e0?40]83E`0002@000340P00d@<00<L2003B0`00
0P000<@2dP?70]43E`0002@000370P00d@<00<T2003A0`000P000<L2d@?90]43E`0002@000390P00
d@<00<`2003B0`000P000<T2d@?<0]83E`0002@0003<0P00dP<00<d2003D0`000P000<`2dP?=0]@3
E`0002@0003=0P00e0<00<d2003N0`000P000<d2e0?=0]h3E`0002@000390P00d@<00</2003B0`00
0P000<T2d@?;0]83E`0002@0003;0P00dP<00<`2003D0`000P000</2dP?<0]@3E`0002@0003<0P00
e0<00<`2003N0`000P000<`2e0?<0]h3E`0002@0002e0P00d@<00;T2003A0`000P000;D2d@>i0]43
E`0002@0002e0P00gP<00;`2003N0`000P000;D2gP>l0]h3E`0002@0002o0P00gP<00<D2003N0`00
0P000;l2gP?50]h3E`0002@000390P00gP<00<l2003N0`000P000<T2gP??0]h3E`0002@0003D0P00
e`<00=l2003G0`000P000=@2e`?O0]L3E`0002@0003O0P00e@<00=l2003G0`000P000=l2e`?O0]D3
E`0002@0003N0P00d`<00=l2003E0`000P000=l2e@?N0]<3E`0002@0003M0P00dP<00=h2003C0`00
0P000=h2d`?M0]83E`0002@0003K0P00d@<00=d2003B0`000P000=d2dP?K0]43E`0002@0003I0P00
d@<00=/2003A0`000P000=/2d@?I0]43E`0002@0003F0P00d@<00=T2003B0`000P000=T2d@?F0]83
E`0002@0003D0P00dP<00=H2003D0`000P000=H2dP?D0]@3E`0002@0003C0P00e0<00=@2003G0`00
0P000=@2e0?C0]L3E`0002@0003C0P00e`<00=<2003I0`000P000=<2e`?C0]T3E`0002@0003C0P00
f@<00=@2003L0`000P000=<2f@?D0]`3E`0002@0003D0P00g0<00=H2003N0`000P000=@2g0?F0]h3
E`0002@0003F0P00gP<00=T2003N0`000P000=H2gP?I0]h3E`0002@0003I0P00gP<00=X2003N0`00
0P000=T2gP?J0]h3E`0002@0003J0P00gP<00=d2003N0`000P000=X2gP?M0]h3E`0002@0003M0P00
g0<00=l2003N0`000P000=d2gP?O0]`3E`0002@0003N0P00e0<00=h2003G0`000P000=h2e`?N0]@3
E`0002@0003M0P00dP<00=h2003D0`000P000=h2e0?M0]83E`0002@0003G0P00d@<00=T2003B0`00
0P000=T2d@?G0]83E`0002@0003E0P00dP<00=L2003D0`000P000=L2dP?E0]@3E`0002@0003D0P00
e0<00=D2003G0`000P000=D2e0?D0]L3E`0002@0003D0P00e`<00=@2003I0`000P000=@2e`?D0]T3
E`0002@0003D0P00f@<00=D2003L0`000P000=@2f@?E0]`3E`0002@0003E0P00g0<00=L2003N0`00
0P000=D2g0?G0]h3E`0002@0003G0P00gP<00=T2003N0`000P000=L2gP?I0]h3E`0002@0002R0000
WP<00:@0002O0`000P000:80W`>T09h3E`0002@0002T0000V`<00:H0002N0`000P000:@0WP>V09/3
E`0002@0002V0000V`<00:H0002^0`000P000:H0V`>V0:h3E`0002@0002U0000W0<00:D0002^0`00
0P000:D0W0>U0:h3E`0002@0002R0000[P<00:X0002^0`000P000:80[P>Z0:h3E`0002@0003I0000
W`<00=X0002P0`000P000=T0W`?J0:03E`0002@0003I0000X0<00=X0002P0`000P000=X0X0?I0:03
E`0002@0003H0000X0<00=T0002P0`000P000=T0X0?H0:03E`0002@0003H0000W`<00=P0002P0`00
0P000=P0X0?H09l3E`0002@0003H0000W@<00=T0002O0`000P000=P0W`?I09d3E`0002@0003I0000
W0<00=X0002M0`000P000=T0W@?J09`3E`0002@0003J0000V`<00=d0002L0`000P000=X0W0?M09/3
E`0002@0003M0000V`<00>00002K0`000P000=d0V`?P09/3E`0002@0003P0000V`<00><0002L0`00
0P000>00V`?S09`3E`0002@0003S0000W0<00>@0002M0`000P000><0W0?T09d3E`0002@0003T0000
W@<00>D0002O0`000P000>@0W@?U09l3E`0002@0003U0000W`<00>D0002P0`000P000>D0W`?U0:03
E`0002@0003T0000X0<00>D0002R0`000P000>D0X0?T0:83E`0002@0003Q0000XP<00>@0002T0`00
0P000>@0XP?Q0:@3E`0002@0003M0000Y0<00>40002V0`000P000>40Y0?M0:H3E`0002@0003K0000
YP<00=d0002W0`000P000=d0YP?K0:L3E`0002@0003I0000Y`<00=/0002Y0`000P000=/0Y`?I0:T3
E`0002@0003H0000Z@<00=T0002/0`000P000=T0Z@?H0:`3E`0002@0003H0000[0<00=P0002^0`00
0P000=P0[0?H0:h3E`0002@0003P0000V`<00>80002L0`000P000>00V`?R09`3E`0002@0003R0000
W0<00><0002M0`000P000>80W0?S09d3E`0002@0003S0000W@<00>@0002O0`000P000><0W@?T09l3
E`0002@0003T0000W`<00>@0002P0`000P000>@0W`?T0:03E`0002@0003S0000X0<00>@0002R0`00
0P000>@0X0?S0:83E`0002@0003P0000XP<00><0002T0`000P000><0XP?P0:@3E`0002@0003M0000
Y0<00>00002V0`000P000>00Y0?M0:H3E`0002@0003H0000[0<00=T0002]0`000P000=P0[@?I0:`3
E`0002@0003I0000[0<00=/0002/0`000P000=T0[0?K0:`3E`0002@0003K0000[0<00=l0002^0`00
0P000=/0[0?O0:h3E`0002@0003O0000[P<00>80002^0`000P000=l0[P?R0:h3E`0002@0003R0000
[@<00>@0002^0`000P000>80[P?T0:d3E`0002@0003T0000[0<00>D0002]0`000P000>@0[@?U0:`3
E`0002@0003K0000[0<00=l0002^0`000P000=/0[0?O0:h3E`0002@0003O0000[P<00><0002^0`00
0P000=l0[P?S0:h3E`0002@0003S0000[P<00>@0002^0`000P000><0[P?T0:h3E`0002@0003T0000
[0<00>D0002^0`000P000>@0[P?U0:`3E`0002@0003U0000ZP<00>D0002/0`000P000>D0[0?U0:X3
E`0002@0000A0@00W`<00181002P0`000P000141W`<B0J03E`0002@0000A0@00X0<00181002P0`00
0P000181X0<A0J03E`0002@0000@0@00X0<00141002P0`000P000141X0<@0J03E`0002@0000@0@00
W`<00101002P0`000P000101X0<@0Il3E`0002@0000@0@00W@<00141002O0`000P000101W`<A0Id3
E`0002@0000A0@00W0<00181002M0`000P000141W@<B0I`3E`0002@0000B0@00V`<001D1002L0`00
0P000181W0<E0I/3E`0002@0000E0@00V`<001P1002K0`000P0001D1V`<H0I/3E`0002@0000H0@00
V`<001/1002L0`000P0001P1V`<K0I`3E`0002@0000K0@00W0<001`1002N0`000P0001/1W0<L0Ih3
E`0002@0000L0@00WP<001`1002P0`000P0001`1WP<L0J03E`0002@0000K0@00X0<001`1002R0`00
0P0001`1X0<K0J83E`0002@0000H0@00XP<001/1002S0`000P0001/1XP<H0J<3E`0002@0000F0@00
X`<001P1002S0`000P0001P1X`<F0J<3E`0002@0000H0@00V`<001X1002L0`000P0001P1V`<J0I`3
E`0002@0000J0@00W0<001/1002N0`000P0001X1W0<K0Ih3E`0002@0000K0@00WP<001/1002P0`00
0P0001/1WP<K0J03E`0002@0000J0@00X0<001/1002R0`000P0001/1X0<J0J83E`0002@0000H0@00
XP<001X1002S0`000P0001X1XP<H0J<3E`0002@0000H0@00X`<001X1002T0`000P0001P1X`<J0J@3
E`0002@0000J0@00Y0<001`1002V0`000P0001X1Y0<L0JH3E`0002@0000L0@00YP<001d1002X0`00
0P0001`1YP<M0JP3E`0002@0000M0@00Z0<001d1002[0`000P0001d1Z0<M0J/3E`0002@0000L0@00
Z`<001d1002]0`000P0001d1Z`<L0Jd3E`0002@0000K0@00[@<001`1002^0`000P0001`1[@<K0Jh3
E`0002@0000H0@00[P<001/1002^0`000P0001/1[P<H0Jh3E`0002@0000E0@00[P<001P1002^0`00
0P0001P1[P<E0Jh3E`0002@0000B0@00[P<001D1002^0`000P0001D1[P<B0Jh3E`0002@0000A0@00
[@<00181002^0`000P000181[P<A0Jd3E`0002@0000@0@00Z`<00141002]0`000P000141[@<@0J/3
E`0002@0000@0@00ZP<00101002[0`000P000101Z`<@0JX3E`0002@0000@0@00Z@<00141002Z0`00
0P000101ZP<A0JT3E`0002@0000A0@00Z@<00181002Z0`000P000141Z@<B0JX3E`0002@0000A0@00
ZP<00181002[0`000P000181ZP<A0J/3E`0002@0000K0@00Y@<001`1002X0`000P0001/1Y@<L0JP3
E`0002@0000L0@00Z0<001`1002[0`000P0001`1Z0<L0J/3E`0002@0000K0@00Z`<001`1002]0`00
0P0001`1Z`<K0Jd3E`0002@0000J0@00[@<001/1002^0`000P0001/1[@<J0Jh3E`0002@0000H0@00
[P<001X1002^0`000P0001X1[P<H0Jh3E`0002@0001@0@00W@<00501002^0`000P000501W@=@0Jh3
E`0002@0001A0@00V`<00541002^0`000P000541V`=A0Jh3E`0002@000170@00V`<00541002Y0`00
0P000541V`=70JT3E`0002@000170@00Z@<005H1002Y0`000P0004L1Z@=F0JT3E`0002@0001>0@00
[P<005@1002^0`000P0004h1[P=D0Jh3E`0002@000210@00V`<008<1002T0`000P0008<1V`>10J@3
E`0002@000210@00XP<008<1002T0`000P000841Y0>30J83E`0002@000230@00X@<008H1002R0`00
0P0008<1XP>60J43E`0002@000260@00X@<008P1002Q0`000P0008H1X@>80J43E`0002@000280@00
X@<008/1002R0`000P0008P1X@>;0J83E`0002@0002;0@00XP<008d1002T0`000P0008/1XP>=0J@3
E`0002@0002=0@00Y0<008h1002W0`000P0008d1Y0>>0JL3E`0002@0002>0@00Y`<008h1002Y0`00
0P0008h1Y`>>0JT3E`0002@0002=0@00Z@<008h1002/0`000P0008h1Z@>=0J`3E`0002@0002;0@00
[0<008d1002^0`000P0008d1[0>;0Jh3E`0002@000280@00[P<008/1002^0`000P0008/1[P>80Jh3
E`0002@000260@00[P<008P1002^0`000P0008P1[P>60Jh3E`0002@000230@00[P<008H1002^0`00
0P0008H1[P>30Jh3E`0002@000220@00[@<008<1002^0`000P0008<1[P>20Jd3E`0002@000210@00
Z`<00881002]0`000P000881[@>10J/3E`0002@000210@00ZP<00841002[0`000P000841Z`>10JX3
E`0002@000210@00Z@<00881002Z0`000P000841ZP>20JT3E`0002@000220@00Z@<008<1002Z0`00
0P000881Z@>30JX3E`0002@000220@00ZP<008<1002[0`000P0008<1ZP>20J/3E`0002@000280@00
X@<008X1002R0`000P0008P1X@>:0J83E`0002@0002:0@00XP<008`1002T0`000P0008X1XP><0J@3
E`0002@0002<0@00Y0<008d1002W0`000P0008`1Y0>=0JL3E`0002@0002=0@00Y`<008d1002Y0`00
0P0008d1Y`>=0JT3E`0002@0002<0@00Z@<008d1002/0`000P0008d1Z@><0J`3E`0002@0002:0@00
[0<008`1002^0`000P0008`1[0>:0Jh3E`0002@000280@00[P<008X1002^0`000P0008X1[P>80Jh3
E`0002@000230@00V`<008`1002K0`000P0008<1V`><0I/3E`0002@000230@00W0<008L1002L0`00
0P0008<1W0>70I`3E`0002@000270@00V`<008`1002L0`000P0008L1W0><0I/3E`0002@000330@00
WP<00<@1002O0`000P000<@1WP?30Il3E`0002@000330@00W`<00<@1002P0`000P000<<1W`?40J03
E`0002@000340@00W`<00<D1002P0`000P000<@1X0?50Il3E`0002@000350@00WP<00<D1002O0`00
0P000<D1W`?50Ih3E`0002@000340@00W0<00<D1002N0`000P000<D1WP?40I`3E`0002@000320@00
V`<00<@1002L0`000P000<@1W0?20I/3E`0002@0002o0@00V`<00<81002K0`000P000<81V`>o0I/3
E`0002@0002m0@00V`<00;l1002L0`000P000;l1V`>m0I`3E`0002@0002k0@00W0<00;d1002N0`00
0P000;d1W0>k0Ih3E`0002@0002j0@00WP<00;/1002P0`000P000;/1WP>j0J03E`0002@0002i0@00
X0<00;X1002S0`000P000;X1X0>i0J<3E`0002@0002i0@00X`<00;T1002Y0`000P000;T1X`>i0JT3
E`0002@0002i0@00Z@<00;X1002/0`000P000;T1Z@>j0J`3E`0002@0002j0@00[0<00;`1002^0`00
0P000;X1[0>l0Jh3E`0002@0002l0@00[P<00;l1002^0`000P000;`1[P>o0Jh3E`0002@0002o0@00
[P<00<01002^0`000P000;l1[P?00Jh3E`0002@000300@00[P<00<<1002^0`000P000<01[P?30Jh3
E`0002@000330@00[0<00<D1002^0`000P000<<1[P?50J`3E`0002@000350@00Z@<00<H1002/0`00
0P000<D1[0?60JT3E`0002@000360@00Z0<00<H1002Y0`000P000<H1Z@?60JP3E`0002@000350@00
Y@<00<H1002X0`000P000<H1Z0?50JD3E`0002@000330@00X`<00<D1002U0`000P000<D1Y@?30J<3
E`0002@000300@00XP<00<<1002S0`000P000<<1X`?00J83E`0002@0002o0@00XP<00<01002R0`00
0P000<01XP>o0J83E`0002@0002m0@00XP<00;l1002S0`000P000;l1XP>m0J<3E`0002@0002k0@00
X`<00;d1002U0`000P000;d1X`>k0JD3E`0002@0002j0@00Y@<00;/1002X0`000P000;/1Y@>j0JP3
E`0002@0002n0@00V`<00;l1002L0`000P000;l1V`>n0I`3E`0002@0002l0@00W0<00;h1002N0`00
0P000;h1W0>l0Ih3E`0002@0002k0@00WP<00;`1002P0`000P000;`1WP>k0J03E`0002@0002j0@00
X0<00;/1002S0`000P000;/1X0>j0J<3E`0002@0002j0@00X`<00;X1002Y0`000P000;X1X`>j0JT3
E`0002@0002j0@00Z@<00;/1002/0`000P000;X1Z@>k0J`3E`0002@0002k0@00[0<00;d1002^0`00
0P000;/1[0>m0Jh3E`0002@0002m0@00[P<00;l1002^0`000P000;d1[P>o0Jh3E`0002@000300@00
[P<00<81002^0`000P000<01[P?20Jh3E`0002@000320@00[0<00<@1002^0`000P000<81[P?40J`3
E`0002@000340@00Z@<00<D1002/0`000P000<@1[0?50JT3E`0002@000350@00Z0<00<D1002Y0`00
0P000<D1Z@?50JP3E`0002@000340@00Y@<00<D1002X0`000P000<D1Z0?40JD3E`0002@000320@00
X`<00<@1002U0`000P000<@1Y@?20J<3E`0002@000300@00XP<00<81002S0`000P000<81X`?00J83
E`0002@0003a0@00V`<00?41002P0`000P000?41V`?a0J03E`0002@0003a0@00W@<00?81002O0`00
0P000?41W`?b0Id3E`0002@0003b0@00V`<00?@1002M0`000P000?81W@?d0I/3E`0002@0003d0@00
V`<00?H1002K0`000P000?@1V`?f0I/3E`0002@0003f0@00V`<00?X1002N0`000P000?H1V`?j0Ih3
E`0002@0003j0@00WP<00?`1002N0`000P000?X1WP?l0Ih3E`0002@0003l0@00W@<00?d1002N0`00
0P000?`1WP?m0Id3E`0002@0003m0@00V`<00?h1002M0`000P000?d1W@?n0I/3E`0002@0003b0@00
W0<00?@1002M0`000P000?81W@?d0I`3E`0002@0003d0@00W0<00?H1002L0`000P000?@1W0?f0I`3
E`0002@0003f0@00W0<00?X1002N0`000P000?H1W0?j0Ih3E`0002@0003n0@00V`<00?h1002N0`00
0P000?h1V`?n0Ih3E`0002@0003m0@00WP<00?h1002P0`000P000?h1WP?m0J03E`0002@0003i0@00
X0<00?d1002U0`000P000?d1X0?i0JD3E`0002@0003h0@00Y@<00?T1002W0`000P000?T1Y@?h0JL3
E`0002@0003g0@00Y`<00?P1002Z0`000P000?P1Y`?g0JX3E`0002@0003g0@00ZP<00?L1002^0`00
0P000?L1ZP?g0Jh3E`0002@0003h0@00X0<00?d1002U0`000P000?d1X0?h0JD3E`0002@0003g0@00
Y@<00?P1002W0`000P000?P1Y@?g0JL3E`0002@0003g0@00Y`<00?L1002Z0`000P000?L1Y`?g0JX3
E`0002@0003g0@00ZP<00?L1002^0`000P000?L1ZP?g0Jh3E`0002@0000/0P00V`<002l2002L0`00
0P0002l2V`</0Y`3E`0002@0000[0P00W0<002`2002N0`000P0002`2W0<[0Yh3E`0002@0000[0P00
WP<002/2002P0`000P0002/2WP<[0Z03E`0002@0000[0P00X0<002`2002R0`000P0002/2X0</0Z83
E`0002@0000/0P00XP<002l2002S0`000P0002`2XP<_0Z<3E`0002@0000_0P00X`<00382002S0`00
0P0002l2X`<b0Z<3E`0002@0000b0P00XP<003D2002S0`000P000382X`<e0Z83E`0002@0000e0P00
X0<003H2002R0`000P0003D2XP<f0Z03E`0002@0000f0P00WP<003H2002P0`000P0003H2X0<f0Yh3
E`0002@0000e0P00W0<003H2002N0`000P0003H2WP<e0Y`3E`0002@0000b0P00V`<003D2002L0`00
0P0003D2W0<b0Y/3E`0002@0000_0P00V`<00382002K0`000P000382V`<_0Y/3E`0002@0000]0P00
V`<002l2002L0`000P0002l2V`<]0Y`3E`0002@0000/0P00W0<002d2002N0`000P0002d2W0</0Yh3
E`0002@0000/0P00WP<002`2002P0`000P0002`2WP</0Z03E`0002@0000/0P00X0<002d2002R0`00
0P0002`2X0<]0Z83E`0002@0000]0P00XP<002l2002S0`000P0002d2XP<_0Z<3E`0002@0000b0P00
XP<003@2002S0`000P000382X`<d0Z83E`0002@0000d0P00X0<003D2002R0`000P0003@2XP<e0Z03
E`0002@0000e0P00WP<003D2002P0`000P0003D2X0<e0Yh3E`0002@0000d0P00W0<003D2002N0`00
0P0003D2WP<d0Y`3E`0002@0000b0P00V`<003@2002L0`000P0003@2W0<b0Y/3E`0002@0000/0P00
X`<002l2002T0`000P0002l2X`</0Z@3E`0002@0000[0P00Y0<002`2002U0`000P0002`2Y0<[0ZD3
E`0002@0000Z0P00Y@<002/2002W0`000P0002/2Y@<Z0ZL3E`0002@0000Z0P00Y`<002X2002[0`00
0P0002X2Y`<Z0Z/3E`0002@0000Z0P00Z`<002/2002]0`000P0002X2Z`<[0Zd3E`0002@0000[0P00
[@<002`2002^0`000P0002/2[@</0Zh3E`0002@0000/0P00[P<002l2002^0`000P0002`2[P<_0Zh3
E`0002@0000_0P00[P<00382002^0`000P0002l2[P<b0Zh3E`0002@0000b0P00[P<003D2002^0`00
0P000382[P<e0Zh3E`0002@0000e0P00[@<003H2002^0`000P0003D2[P<f0Zd3E`0002@0000f0P00
Z`<003L2002]0`000P0003H2[@<g0Z/3E`0002@0000g0P00Y`<003L2002[0`000P0003L2Z`<g0ZL3
E`0002@0000f0P00Y@<003L2002W0`000P0003L2Y`<f0ZD3E`0002@0000e0P00Y0<003H2002U0`00
0P0003H2Y@<e0Z@3E`0002@0000b0P00X`<003D2002T0`000P0003D2Y0<b0Z<3E`0002@0000]0P00
X`<002l2002T0`000P0002l2X`<]0Z@3E`0002@0000/0P00Y0<002d2002U0`000P0002d2Y0</0ZD3
E`0002@0000[0P00Y@<002`2002W0`000P0002`2Y@<[0ZL3E`0002@0000[0P00Y`<002/2002[0`00
0P0002/2Y`<[0Z/3E`0002@0000[0P00Z`<002`2002]0`000P0002/2Z`</0Zd3E`0002@0000/0P00
[@<002d2002^0`000P0002`2[@<]0Zh3E`0002@0000]0P00[P<002l2002^0`000P0002d2[P<_0Zh3
E`0002@0000b0P00[P<003@2002^0`000P000382[P<d0Zh3E`0002@0000d0P00[@<003D2002^0`00
0P0003@2[P<e0Zd3E`0002@0000e0P00Z`<003H2002]0`000P0003D2[@<f0Z/3E`0002@0000f0P00
Y`<003H2002[0`000P0003H2Z`<f0ZL3E`0002@0000e0P00Y@<003H2002W0`000P0003H2Y`<e0ZD3
E`0002@0000d0P00Y0<003D2002U0`000P0003D2Y@<d0Z@3E`0002@0000b0P00X`<003@2002T0`00
0P0003@2Y0<b0Z<3E`0002@0001]0P00X@<006h2002T0`000P0006h2X@=]0Z@3E`0002@0001[0P00
Y0<006d2002V0`000P0006d2Y0=[0ZH3E`0002@0001X0P00YP<006/2002W0`000P0006/2YP=X0ZL3
E`0002@0001X0P00Y`<006P2002W0`000P0006P2Y`=X0ZL3E`0002@0001U0P00YP<006P2002W0`00
0P0006P2Y`=U0ZH3E`0002@0001S0P00Y0<006D2002V0`000P0006D2YP=S0Z@3E`0002@0001R0P00
X@<006<2002T0`000P0006<2Y0=R0Z43E`0002@0001R0P00X0<00682002Q0`000P000682X@=R0Z03
E`0002@0001R0P00WP<006<2002P0`000P000682X0=S0Yh3E`0002@0001S0P00W0<006D2002N0`00
0P0006<2WP=U0Y`3E`0002@0001U0P00V`<006P2002L0`000P0006D2W0=X0Y/3E`0002@0001X0P00
V`<006T2002K0`000P0006P2V`=Y0Y/3E`0002@0001Y0P00V`<006`2002L0`000P0006T2V`=/0Y`3
E`0002@0001/0P00W0<006h2002N0`000P0006`2W0=^0Yh3E`0002@0001^0P00WP<006l2002P0`00
0P0006h2WP=_0Z03E`0002@0001_0P00X0<006l2002V0`000P0006l2X0=_0ZH3E`0002@0001^0P00
YP<006l2002Z0`000P0006l2YP=^0ZX3E`0002@0001]0P00ZP<006h2002/0`000P0006h2ZP=]0Z`3
E`0002@0001[0P00[0<006d2002^0`000P0006d2[0=[0Zh3E`0002@0001X0P00[P<006/2002^0`00
0P0006/2[P=X0Zh3E`0002@0001V0P00[P<006P2002^0`000P0006P2[P=V0Zh3E`0002@0001T0P00
[P<006H2002^0`000P0006H2[P=T0Zh3E`0002@0001S0P00[0<006@2002^0`000P0006@2[P=S0Z`3
E`0002@0001S0P00Z`<006<2002/0`000P0006<2[0=S0Z/3E`0002@0001S0P00ZP<006@2002[0`00
0P0006<2Z`=T0ZX3E`0002@0001T0P00ZP<006D2002[0`000P0006@2ZP=U0Z/3E`0002@0001T0P00
Z`<006D2002/0`000P0006D2Z`=T0Z`3E`0002@0001V0P00YP<006P2002W0`000P0006P2Y`=V0ZH3
E`0002@0001T0P00Y0<006H2002V0`000P0006H2YP=T0Z@3E`0002@0001S0P00X@<006@2002T0`00
0P0006@2Y0=S0Z43E`0002@0001S0P00X0<006<2002Q0`000P0006<2X@=S0Z03E`0002@0001S0P00
WP<006@2002P0`000P0006<2X0=T0Yh3E`0002@0001T0P00W0<006H2002N0`000P0006@2WP=V0Y`3
E`0002@0001V0P00V`<006P2002L0`000P0006H2W0=X0Y/3E`0002@0001Y0P00V`<006/2002L0`00
0P0006T2V`=[0Y`3E`0002@0001[0P00W0<006d2002N0`000P0006/2W0=]0Yh3E`0002@0001]0P00
WP<006h2002P0`000P0006d2WP=^0Z03E`0002@0001^0P00X0<006h2002V0`000P0006h2X0=^0ZH3
E`0002@0001]0P00YP<006h2002Z0`000P0006h2YP=]0ZX3E`0002@0001/0P00ZP<006d2002/0`00
0P0006d2ZP=/0Z`3E`0002@0001Z0P00[0<006`2002^0`000P0006`2[0=Z0Zh3E`0002@0001X0P00
[P<006X2002^0`000P0006X2[P=X0Zh3E`0002@0002E0P00WP<009L2002O0`000P0009D2W`>G0Yh3
E`0002@0002G0P00V`<009T2002N0`000P0009L2WP>I0Y/3E`0002@0002I0P00V`<009T2002^0`00
0P0009T2V`>I0Zh3E`0002@0002H0P00W0<009P2002^0`000P0009P2W0>H0Zh3E`0002@0002E0P00
[P<009d2002^0`000P0009D2[P>M0Zh3E`0002@0002W0P00V`<00:X2002L0`000P000:X2V`>W0Y`3
E`0002@0002U0P00W0<00:L2002O0`000P000:L2W0>U0Yl3E`0002@0002T0P00W`<00:D2002S0`00
0P000:D2W`>T0Z<3E`0002@0002T0P00X`<00:@2002V0`000P000:@2X`>T0ZH3E`0002@0002T0P00
YP<00:D2002[0`000P000:@2YP>U0Z/3E`0002@0002U0P00Z`<00:L2002^0`000P000:D2Z`>W0Zh3
E`0002@0002W0P00[P<00:X2002^0`000P000:L2[P>Z0Zh3E`0002@0002Z0P00[P<00:/2002^0`00
0P000:X2[P>[0Zh3E`0002@0002[0P00[P<00:h2002^0`000P000:/2[P>^0Zh3E`0002@0002^0P00
Z`<00;02002^0`000P000:h2[P>`0Z/3E`0002@0002`0P00YP<00;42002[0`000P000;02Z`>a0ZH3
E`0002@0002a0P00X`<00;42002V0`000P000;42YP>a0Z<3E`0002@0002`0P00W`<00;42002S0`00
0P000;42X`>`0Yl3E`0002@0002^0P00W0<00;02002O0`000P000;02W`>^0Y`3E`0002@0002[0P00
V`<00:h2002L0`000P000:h2W0>[0Y/3E`0002@0002Z0P00V`<00:/2002K0`000P000:/2V`>Z0Y/3
E`0002@0002X0P00V`<00:X2002L0`000P000:X2V`>X0Y`3E`0002@0002W0P00W0<00:P2002M0`00
0P000:P2W0>W0Yd3E`0002@0002V0P00W@<00:L2002O0`000P000:L2W@>V0Yl3E`0002@0002U0P00
W`<00:H2002S0`000P000:H2W`>U0Z<3E`0002@0002U0P00X`<00:D2002V0`000P000:D2X`>U0ZH3
E`0002@0002U0P00YP<00:H2002[0`000P000:D2YP>V0Z/3E`0002@0002V0P00Z`<00:L2002]0`00
0P000:H2Z`>W0Zd3E`0002@0002W0P00[@<00:P2002^0`000P000:L2[@>X0Zh3E`0002@0002X0P00
[P<00:X2002^0`000P000:P2[P>Z0Zh3E`0002@0002[0P00[P<00:d2002^0`000P000:/2[P>]0Zh3
E`0002@0002]0P00[@<00:h2002^0`000P000:d2[P>^0Zd3E`0002@0002^0P00Z`<00:l2002]0`00
0P000:h2[@>_0Z/3E`0002@0002_0P00YP<00;02002[0`000P000:l2Z`>`0ZH3E`0002@0002`0P00
X`<00;02002V0`000P000;02YP>`0Z<3E`0002@0002_0P00W`<00;02002S0`000P000;02X`>_0Yl3
E`0002@0002^0P00W@<00:l2002O0`000P000:l2W`>^0Yd3E`0002@0002]0P00W0<00:h2002M0`00
0P000:h2W@>]0Y`3E`0002@0002[0P00V`<00:d2002L0`000P000:d2W0>[0Y/3E`0002@0003=0P00
WP<00<l2002O0`000P000<d2W`??0Yh3E`0002@0003?0P00V`<00=42002N0`000P000<l2WP?A0Y/3
E`0002@0003A0P00V`<00=42002^0`000P000=42V`?A0Zh3E`0002@0003@0P00W0<00=02002^0`00
0P000=02W0?@0Zh3E`0002@0003=0P00[P<00=D2002^0`000P000<d2[P?E0Zh3E`0002@0003O0P00
WP<00>42002O0`000P000=l2W`?Q0Yh3E`0002@0003Q0P00V`<00><2002N0`000P000>42WP?S0Y/3
E`0002@0003S0P00V`<00><2002^0`000P000><2V`?S0Zh3E`0002@0003R0P00W0<00>82002^0`00
0P000>82W0?R0Zh3E`0002@0003O0P00[P<00>L2002^0`000P000=l2[P?W0Zh3E`0002@000050`00
WP<000L3002O0`000P0000D3W`<70ih3E`0002@000070`00V`<000T3002N0`000P0000L3WP<90i/3
E`0002@000090`00V`<000T3002^0`000P0000T3V`<90jh3E`0002@000080`00W0<000P3002^0`00
0P0000P3W0<80jh3E`0002@000050`00[P<000d3002^0`000P0000D3[P<=0jh3E`0002@0000E0`00
W`<001H3002P0`000P0001D3W`<F0j03E`0002@0000E0`00X0<001H3002P0`000P0001H3X0<E0j03
E`0002@0000D0`00X0<001D3002P0`000P0001D3X0<D0j03E`0002@0000D0`00W`<001@3002P0`00
0P0001@3X0<D0il3E`0002@0000D0`00W@<001D3002O0`000P0001@3W`<E0id3E`0002@0000E0`00
W0<001H3002M0`000P0001D3W@<F0i`3E`0002@0000F0`00V`<001T3002L0`000P0001H3W0<I0i/3
E`0002@0000I0`00V`<001`3002K0`000P0001T3V`<L0i/3E`0002@0000L0`00V`<001l3002L0`00
0P0001`3V`<O0i`3E`0002@0000O0`00W0<00203002M0`000P0001l3W0<P0id3E`0002@0000P0`00
W@<00243002O0`000P000203W@<Q0il3E`0002@0000Q0`00W`<00243002P0`000P000243W`<Q0j03
E`0002@0000P0`00X0<00243002R0`000P000243X0<P0j83E`0002@0000M0`00XP<00203002T0`00
0P000203XP<M0j@3E`0002@0000I0`00Y0<001d3002V0`000P0001d3Y0<I0jH3E`0002@0000G0`00
YP<001T3002W0`000P0001T3YP<G0jL3E`0002@0000E0`00Y`<001L3002Y0`000P0001L3Y`<E0jT3
E`0002@0000D0`00Z@<001D3002/0`000P0001D3Z@<D0j`3E`0002@0000D0`00[0<001@3002^0`00
0P0001@3[0<D0jh3E`0002@0000L0`00V`<001h3002L0`000P0001`3V`<N0i`3E`0002@0000N0`00
W0<001l3002M0`000P0001h3W0<O0id3E`0002@0000O0`00W@<00203002O0`000P0001l3W@<P0il3
E`0002@0000P0`00W`<00203002P0`000P000203W`<P0j03E`0002@0000O0`00X0<00203002R0`00
0P000203X0<O0j83E`0002@0000L0`00XP<001l3002T0`000P0001l3XP<L0j@3E`0002@0000I0`00
Y0<001`3002V0`000P0001`3Y0<I0jH3E`0002@0000D0`00[0<001D3002]0`000P0001@3[@<E0j`3
E`0002@0000E0`00[0<001L3002/0`000P0001D3[0<G0j`3E`0002@0000G0`00[0<001/3002^0`00
0P0001L3[0<K0jh3E`0002@0000K0`00[P<001h3002^0`000P0001/3[P<N0jh3E`0002@0000N0`00
[@<00203002^0`000P0001h3[P<P0jd3E`0002@0000P0`00[0<00243002]0`000P000203[@<Q0j`3
E`0002@0000G0`00[0<001/3002^0`000P0001L3[0<K0jh3E`0002@0000K0`00[P<001l3002^0`00
0P0001/3[P<O0jh3E`0002@0000O0`00[P<00203002^0`000P0001l3[P<P0jh3E`0002@0000P0`00
[0<00243002^0`000P000203[P<Q0j`3E`0002@0000Q0`00ZP<00243002/0`000P000243[0<Q0jX3
E`0002@0000n0`00WP<00403002O0`000P0003h3W`=00ih3E`0002@000100`00V`<00483002N0`00
0P000403WP=20i/3E`0002@000120`00V`<00483002^0`000P000483V`=20jh3E`0002@000110`00
W0<00443002^0`000P000443W0=10jh3E`0002@0000n0`00[P<004H3002^0`000P0003h3[P=60jh3
E`0002@0001>0`00W`<004l3002P0`000P0004h3W`=?0j03E`0002@0001>0`00X0<004l3002P0`00
0P0004l3X0=>0j03E`0002@0001=0`00X0<004h3002P0`000P0004h3X0==0j03E`0002@0001=0`00
W`<004d3002P0`000P0004d3X0==0il3E`0002@0001=0`00W@<004h3002O0`000P0004d3W`=>0id3
E`0002@0001>0`00W0<004l3002M0`000P0004h3W@=?0i`3E`0002@0001?0`00V`<00583002L0`00
0P0004l3W0=B0i/3E`0002@0001B0`00V`<005D3002K0`000P000583V`=E0i/3E`0002@0001E0`00
V`<005P3002L0`000P0005D3V`=H0i`3E`0002@0001H0`00W0<005T3002N0`000P0005P3W0=I0ih3
E`0002@0001I0`00WP<005T3002P0`000P0005T3WP=I0j03E`0002@0001H0`00X0<005T3002R0`00
0P0005T3X0=H0j83E`0002@0001E0`00XP<005P3002S0`000P0005P3XP=E0j<3E`0002@0001C0`00
X`<005D3002S0`000P0005D3X`=C0j<3E`0002@0001E0`00V`<005L3002L0`000P0005D3V`=G0i`3
E`0002@0001G0`00W0<005P3002N0`000P0005L3W0=H0ih3E`0002@0001H0`00WP<005P3002P0`00
0P0005P3WP=H0j03E`0002@0001G0`00X0<005P3002R0`000P0005P3X0=G0j83E`0002@0001E0`00
XP<005L3002S0`000P0005L3XP=E0j<3E`0002@0001E0`00X`<005L3002T0`000P0005D3X`=G0j@3
E`0002@0001G0`00Y0<005T3002V0`000P0005L3Y0=I0jH3E`0002@0001I0`00YP<005X3002X0`00
0P0005T3YP=J0jP3E`0002@0001J0`00Z0<005X3002[0`000P0005X3Z0=J0j/3E`0002@0001I0`00
Z`<005X3002]0`000P0005X3Z`=I0jd3E`0002@0001H0`00[@<005T3002^0`000P0005T3[@=H0jh3
E`0002@0001E0`00[P<005P3002^0`000P0005P3[P=E0jh3E`0002@0001B0`00[P<005D3002^0`00
0P0005D3[P=B0jh3E`0002@0001?0`00[P<00583002^0`000P000583[P=?0jh3E`0002@0001>0`00
[@<004l3002^0`000P0004l3[P=>0jd3E`0002@0001=0`00Z`<004h3002]0`000P0004h3[@==0j/3
E`0002@0001=0`00ZP<004d3002[0`000P0004d3Z`==0jX3E`0002@0001=0`00Z@<004h3002Z0`00
0P0004d3ZP=>0jT3E`0002@0001>0`00Z@<004l3002Z0`000P0004h3Z@=?0jX3E`0002@0001>0`00
ZP<004l3002[0`000P0004l3ZP=>0j/3E`0002@0001H0`00Y@<005T3002X0`000P0005P3Y@=I0jP3
E`0002@0001I0`00Z0<005T3002[0`000P0005T3Z0=I0j/3E`0002@0001H0`00Z`<005T3002]0`00
0P0005T3Z`=H0jd3E`0002@0001G0`00[@<005P3002^0`000P0005P3[@=G0jh3E`0002@0001E0`00
[P<005L3002^0`000P0005L3[P=E0jh3E`0002@0001f0`00WP<007P3002O0`000P0007H3W`=h0ih3
E`0002@0001h0`00V`<007X3002N0`000P0007P3WP=j0i/3E`0002@0001j0`00V`<007X3002^0`00
0P0007X3V`=j0jh3E`0002@0001i0`00W0<007T3002^0`000P0007T3W0=i0jh3E`0002@0001f0`00
[P<007h3002^0`000P0007H3[P=n0jh3E`0002@0002=0`00W@<008d3002^0`000P0008d3W@>=0jh3
E`0002@0002>0`00V`<008h3002^0`000P0008h3V`>>0jh3E`0002@000240`00V`<008h3002Y0`00
0P0008h3V`>40jT3E`0002@000240`00Z@<009<3002Y0`000P0008@3Z@>C0jT3E`0002@0002;0`00
[P<00943002^0`000P0008/3[P>A0jh3E`0002@0002_0`00WP<00;43002O0`000P000:l3W`>a0ih3
E`0002@0002a0`00V`<00;<3002N0`000P000;43WP>c0i/3E`0002@0002c0`00V`<00;<3002^0`00
0P000;<3V`>c0jh3E`0002@0002b0`00W0<00;83002^0`000P000;83W0>b0jh3E`0002@0002_0`00
[P<00;L3002^0`000P000:l3[P>g0jh3E`0002@0002m0`00V`<00;l3002T0`000P000;l3V`>m0j@3
E`0002@0002m0`00XP<00;l3002T0`000P000;d3Y0>o0j83E`0002@0002o0`00X@<00<83002R0`00
0P000;l3XP?20j43E`0002@000320`00X@<00<@3002Q0`000P000<83X@?40j43E`0002@000340`00
X@<00<L3002R0`000P000<@3X@?70j83E`0002@000370`00XP<00<T3002T0`000P000<L3XP?90j@3
E`0002@000390`00Y0<00<X3002W0`000P000<T3Y0?:0jL3E`0002@0003:0`00Y`<00<X3002Y0`00
0P000<X3Y`?:0jT3E`0002@000390`00Z@<00<X3002/0`000P000<X3Z@?90j`3E`0002@000370`00
[0<00<T3002^0`000P000<T3[0?70jh3E`0002@000340`00[P<00<L3002^0`000P000<L3[P?40jh3
E`0002@000320`00[P<00<@3002^0`000P000<@3[P?20jh3E`0002@0002o0`00[P<00<83002^0`00
0P000<83[P>o0jh3E`0002@0002n0`00[@<00;l3002^0`000P000;l3[P>n0jd3E`0002@0002m0`00
Z`<00;h3002]0`000P000;h3[@>m0j/3E`0002@0002m0`00ZP<00;d3002[0`000P000;d3Z`>m0jX3
E`0002@0002m0`00Z@<00;h3002Z0`000P000;d3ZP>n0jT3E`0002@0002n0`00Z@<00;l3002Z0`00
0P000;h3Z@>o0jX3E`0002@0002n0`00ZP<00;l3002[0`000P000;l3ZP>n0j/3E`0002@000340`00
X@<00<H3002R0`000P000<@3X@?60j83E`0002@000360`00XP<00<P3002T0`000P000<H3XP?80j@3
E`0002@000380`00Y0<00<T3002W0`000P000<P3Y0?90jL3E`0002@000390`00Y`<00<T3002Y0`00
0P000<T3Y`?90jT3E`0002@000380`00Z@<00<T3002/0`000P000<T3Z@?80j`3E`0002@000360`00
[0<00<P3002^0`000P000<P3[0?60jh3E`0002@000340`00[P<00<H3002^0`000P000<H3[P?40jh3
E`0002@0002o0`00V`<00<P3002K0`000P000;l3V`?80i/3E`0002@0002o0`00W0<00<<3002L0`00
0P000;l3W0?30i`3E`0002@000330`00V`<00<P3002L0`000P000<<3W0?80i/3E`0002@0003W0`00
WP<00>T3002O0`000P000>L3W`?Y0ih3E`0002@0003Y0`00V`<00>/3002N0`000P000>T3WP?[0i/3
E`0002@0003[0`00V`<00>/3002^0`000P000>/3V`?[0jh3E`0002@0003Z0`00W0<00>X3002^0`00
0P000>X3W0?Z0jh3E`0002@0003W0`00[P<00>l3002^0`000P000>L3[P?_0jh3E`0002@000001000
WP<00044002O0`000P000044WP<019l3E`0002@000001000W`<00044002P0`000P000004W`<11:03
E`0002@000011000W`<00084002P0`000P000044X0<219l3E`0002@000021000WP<00084002O0`00
0P000084W`<219h3E`0002@000011000W0<00084002N0`000P000084WP<119`3E`0002@0003o0`00
V`<00044002L0`000P000044W0?o0i/3E`0002@0003l0`00V`<00?l3002K0`000P000?l3V`?l0i/3
E`0002@0003j0`00V`<00?`3002L0`000P000?`3V`?j0i`3E`0002@0003h0`00W0<00?X3002N0`00
0P000?X3W0?h0ih3E`0002@0003g0`00WP<00?P3002P0`000P000?P3WP?g0j03E`0002@0003f0`00
X0<00?L3002S0`000P000?L3X0?f0j<3E`0002@0003f0`00X`<00?H3002Y0`000P000?H3X`?f0jT3
E`0002@0003f0`00Z@<00?L3002/0`000P000?H3Z@?g0j`3E`0002@0003g0`00[0<00?T3002^0`00
0P000?L3[0?i0jh3E`0002@0003i0`00[P<00?`3002^0`000P000?T3[P?l0jh3E`0002@0003l0`00
[P<00?d3002^0`000P000?`3[P?m0jh3E`0002@0003m0`00[P<00004002^0`000P000?d3[P<01:h3
E`0002@000001000[0<00084002^0`000P000004[P<21:`3E`0002@000021000Z@<000<4002/0`00
0P000084[0<31:T3E`0002@000031000Z0<000<4002Y0`000P0000<4Z@<31:P3E`0002@000021000
Y@<000<4002X0`000P0000<4Z0<21:D3E`0002@000001000X`<00084002U0`000P000084Y@<01:<3
E`0002@0003m0`00XP<00004002S0`000P000004X`?m0j83E`0002@0003l0`00XP<00?d3002R0`00
0P000?d3XP?l0j83E`0002@0003j0`00XP<00?`3002S0`000P000?`3XP?j0j<3E`0002@0003h0`00
X`<00?X3002U0`000P000?X3X`?h0jD3E`0002@0003g0`00Y@<00?P3002X0`000P000?P3Y@?g0jP3
E`0002@0003k0`00V`<00?`3002L0`000P000?`3V`?k0i`3E`0002@0003i0`00W0<00?/3002N0`00
0P000?/3W0?i0ih3E`0002@0003h0`00WP<00?T3002P0`000P000?T3WP?h0j03E`0002@0003g0`00
X0<00?P3002S0`000P000?P3X0?g0j<3E`0002@0003g0`00X`<00?L3002Y0`000P000?L3X`?g0jT3
E`0002@0003g0`00Z@<00?P3002/0`000P000?L3Z@?h0j`3E`0002@0003h0`00[0<00?X3002^0`00
0P000?P3[0?j0jh3E`0002@0003j0`00[P<00?`3002^0`000P000?X3[P?l0jh3E`0002@0003m0`00
[P<00?l3002^0`000P000?d3[P?o0jh3E`0002@0003o0`00[0<00044002^0`000P000?l3[P<11:`3
E`0002@000011000Z@<00084002/0`000P000044[0<21:T3E`0002@000021000Z0<00084002Y0`00
0P000084Z@<21:P3E`0002@000011000Y@<00084002X0`000P000084Z0<11:D3E`0002@0003o0`00
X`<00044002U0`000P000044Y@?o0j<3E`0002@0003m0`00XP<00?l3002S0`000P000?l3X`?m0j83
E`0002@0000O1000WP<00244002O0`000P0001l4W`<Q19h3E`0002@0000Q1000V`<002<4002N0`00
0P000244WP<S19/3E`0002@0000S1000V`<002<4002^0`000P0002<4V`<S1:h3E`0002@0000R1000
W0<00284002^0`000P000284W0<R1:h3E`0002@0000O1000[P<002L4002^0`000P0001l4[P<W1:h3
E`0002@0000^1000V`<002h4002P0`000P0002h4V`<^1:03E`0002@0000^1000W@<002l4002O0`00
0P0002h4W`<_19d3E`0002@0000_1000V`<00344002M0`000P0002l4W@<a19/3E`0002@0000a1000
V`<003<4002K0`000P000344V`<c19/3E`0002@0000c1000V`<003L4002N0`000P0003<4V`<g19h3
E`0002@0000g1000WP<003T4002N0`000P0003L4WP<i19h3E`0002@0000i1000W@<003X4002N0`00
0P0003T4WP<j19d3E`0002@0000j1000V`<003/4002M0`000P0003X4W@<k19/3E`0002@0000_1000
W0<00344002M0`000P0002l4W@<a19`3E`0002@0000a1000W0<003<4002L0`000P000344W0<c19`3
E`0002@0000c1000W0<003L4002N0`000P0003<4W0<g19h3E`0002@0000k1000V`<003/4002N0`00
0P0003/4V`<k19h3E`0002@0000j1000WP<003/4002P0`000P0003/4WP<j1:03E`0002@0000f1000
X0<003X4002U0`000P0003X4X0<f1:D3E`0002@0000e1000Y@<003H4002W0`000P0003H4Y@<e1:L3
E`0002@0000d1000Y`<003D4002Z0`000P0003D4Y`<d1:X3E`0002@0000d1000ZP<003@4002^0`00
0P0003@4ZP<d1:h3E`0002@0000e1000X0<003X4002U0`000P0003X4X0<e1:D3E`0002@0000d1000
Y@<003D4002W0`000P0003D4Y@<d1:L3E`0002@0000d1000Y`<003@4002Z0`000P0003@4Y`<d1:X3
E`0002@0000d1000ZP<003@4002^0`000P0003@4ZP<d1:h3E`0002@0001H1000WP<005X4002O0`00
0P0005P4W`=J19h3E`0002@0001J1000V`<005`4002N0`000P0005X4WP=L19/3E`0002@0001L1000
V`<005`4002^0`000P0005`4V`=L1:h3E`0002@0001K1000W0<005/4002^0`000P0005/4W0=K1:h3
E`0002@0001H1000[P<00604002^0`000P0005P4[P=P1:h3E`0002@0001Y1000V`<006`4002L0`00
0P0006`4V`=Y19`3E`0002@0001X1000W0<006T4002N0`000P0006T4W0=X19h3E`0002@0001X1000
WP<006P4002P0`000P0006P4WP=X1:03E`0002@0001X1000X0<006T4002R0`000P0006P4X0=Y1:83
E`0002@0001Y1000XP<006`4002S0`000P0006T4XP=/1:<3E`0002@0001/1000X`<006l4002S0`00
0P0006`4X`=_1:<3E`0002@0001_1000XP<00784002S0`000P0006l4X`=b1:83E`0002@0001b1000
X0<007<4002R0`000P000784XP=c1:03E`0002@0001c1000WP<007<4002P0`000P0007<4X0=c19h3
E`0002@0001b1000W0<007<4002N0`000P0007<4WP=b19`3E`0002@0001_1000V`<00784002L0`00
0P000784W0=_19/3E`0002@0001/1000V`<006l4002K0`000P0006l4V`=/19/3E`0002@0001Z1000
V`<006`4002L0`000P0006`4V`=Z19`3E`0002@0001Y1000W0<006X4002N0`000P0006X4W0=Y19h3
E`0002@0001Y1000WP<006T4002P0`000P0006T4WP=Y1:03E`0002@0001Y1000X0<006X4002R0`00
0P0006T4X0=Z1:83E`0002@0001Z1000XP<006`4002S0`000P0006X4XP=/1:<3E`0002@0001_1000
XP<00744002S0`000P0006l4X`=a1:83E`0002@0001a1000X0<00784002R0`000P000744XP=b1:03
E`0002@0001b1000WP<00784002P0`000P000784X0=b19h3E`0002@0001a1000W0<00784002N0`00
0P000784WP=a19`3E`0002@0001_1000V`<00744002L0`000P000744W0=_19/3E`0002@0001Y1000
X`<006`4002T0`000P0006`4X`=Y1:@3E`0002@0001X1000Y0<006T4002U0`000P0006T4Y0=X1:D3
E`0002@0001W1000Y@<006P4002W0`000P0006P4Y@=W1:L3E`0002@0001W1000Y`<006L4002[0`00
0P0006L4Y`=W1:/3E`0002@0001W1000Z`<006P4002]0`000P0006L4Z`=X1:d3E`0002@0001X1000
[@<006T4002^0`000P0006P4[@=Y1:h3E`0002@0001Y1000[P<006`4002^0`000P0006T4[P=/1:h3
E`0002@0001/1000[P<006l4002^0`000P0006`4[P=_1:h3E`0002@0001_1000[P<00784002^0`00
0P0006l4[P=b1:h3E`0002@0001b1000[@<007<4002^0`000P000784[P=c1:d3E`0002@0001c1000
Z`<007@4002]0`000P0007<4[@=d1:/3E`0002@0001d1000Y`<007@4002[0`000P0007@4Z`=d1:L3
E`0002@0001c1000Y@<007@4002W0`000P0007@4Y`=c1:D3E`0002@0001b1000Y0<007<4002U0`00
0P0007<4Y@=b1:@3E`0002@0001_1000X`<00784002T0`000P000784Y0=_1:<3E`0002@0001Z1000
X`<006`4002T0`000P0006`4X`=Z1:@3E`0002@0001Y1000Y0<006X4002U0`000P0006X4Y0=Y1:D3
E`0002@0001X1000Y@<006T4002W0`000P0006T4Y@=X1:L3E`0002@0001X1000Y`<006P4002[0`00
0P0006P4Y`=X1:/3E`0002@0001X1000Z`<006T4002]0`000P0006P4Z`=Y1:d3E`0002@0001Y1000
[@<006X4002^0`000P0006T4[@=Z1:h3E`0002@0001Z1000[P<006`4002^0`000P0006X4[P=/1:h3
E`0002@0001_1000[P<00744002^0`000P0006l4[P=a1:h3E`0002@0001a1000[@<00784002^0`00
0P000744[P=b1:d3E`0002@0001b1000Z`<007<4002]0`000P000784[@=c1:/3E`0002@0001c1000
Y`<007<4002[0`000P0007<4Z`=c1:L3E`0002@0001b1000Y@<007<4002W0`000P0007<4Y`=b1:D3
E`0002@0001a1000Y0<00784002U0`000P000784Y@=a1:@3E`0002@0001_1000X`<00744002T0`00
0P000744Y0=_1:<3E`0002@0002@1000WP<00984002O0`000P000904W`>B19h3E`0002@0002B1000
V`<009@4002N0`000P000984WP>D19/3E`0002@0002D1000V`<009@4002^0`000P0009@4V`>D1:h3
E`0002@0002C1000W0<009<4002^0`000P0009<4W0>C1:h3E`0002@0002@1000[P<009P4002^0`00
0P000904[P>H1:h3E`0002@0002Z1000X@<00:/4002T0`000P000:/4X@>Z1:@3E`0002@0002X1000
Y0<00:X4002V0`000P000:X4Y0>X1:H3E`0002@0002U1000YP<00:P4002W0`000P000:P4YP>U1:L3
E`0002@0002U1000Y`<00:D4002W0`000P000:D4Y`>U1:L3E`0002@0002R1000YP<00:D4002W0`00
0P000:D4Y`>R1:H3E`0002@0002P1000Y0<00:84002V0`000P000:84YP>P1:@3E`0002@0002O1000
X@<00:04002T0`000P000:04Y0>O1:43E`0002@0002O1000X0<009l4002Q0`000P0009l4X@>O1:03
E`0002@0002O1000WP<00:04002P0`000P0009l4X0>P19h3E`0002@0002P1000W0<00:84002N0`00
0P000:04WP>R19`3E`0002@0002R1000V`<00:D4002L0`000P000:84W0>U19/3E`0002@0002U1000
V`<00:H4002K0`000P000:D4V`>V19/3E`0002@0002V1000V`<00:T4002L0`000P000:H4V`>Y19`3
E`0002@0002Y1000W0<00:/4002N0`000P000:T4W0>[19h3E`0002@0002[1000WP<00:`4002P0`00
0P000:/4WP>/1:03E`0002@0002/1000X0<00:`4002V0`000P000:`4X0>/1:H3E`0002@0002[1000
YP<00:`4002Z0`000P000:`4YP>[1:X3E`0002@0002Z1000ZP<00:/4002/0`000P000:/4ZP>Z1:`3
E`0002@0002X1000[0<00:X4002^0`000P000:X4[0>X1:h3E`0002@0002U1000[P<00:P4002^0`00
0P000:P4[P>U1:h3E`0002@0002S1000[P<00:D4002^0`000P000:D4[P>S1:h3E`0002@0002Q1000
[P<00:<4002^0`000P000:<4[P>Q1:h3E`0002@0002P1000[0<00:44002^0`000P000:44[P>P1:`3
E`0002@0002P1000Z`<00:04002/0`000P000:04[0>P1:/3E`0002@0002P1000ZP<00:44002[0`00
0P000:04Z`>Q1:X3E`0002@0002Q1000ZP<00:84002[0`000P000:44ZP>R1:/3E`0002@0002Q1000
Z`<00:84002/0`000P000:84Z`>Q1:`3E`0002@0002S1000YP<00:D4002W0`000P000:D4Y`>S1:H3
E`0002@0002Q1000Y0<00:<4002V0`000P000:<4YP>Q1:@3E`0002@0002P1000X@<00:44002T0`00
0P000:44Y0>P1:43E`0002@0002P1000X0<00:04002Q0`000P000:04X@>P1:03E`0002@0002P1000
WP<00:44002P0`000P000:04X0>Q19h3E`0002@0002Q1000W0<00:<4002N0`000P000:44WP>S19`3
E`0002@0002S1000V`<00:D4002L0`000P000:<4W0>U19/3E`0002@0002V1000V`<00:P4002L0`00
0P000:H4V`>X19`3E`0002@0002X1000W0<00:X4002N0`000P000:P4W0>Z19h3E`0002@0002Z1000
WP<00:/4002P0`000P000:X4WP>[1:03E`0002@0002[1000X0<00:/4002V0`000P000:/4X0>[1:H3
E`0002@0002Z1000YP<00:/4002Z0`000P000:/4YP>Z1:X3E`0002@0002Y1000ZP<00:X4002/0`00
0P000:X4ZP>Y1:`3E`0002@0002W1000[0<00:T4002^0`000P000:T4[0>W1:h3E`0002@0002U1000
[P<00:L4002^0`000P000:L4[P>U1:h3E`0002@000361000W`<00<L4002P0`000P000<H4W`?71:03
E`0002@000361000X0<00<L4002P0`000P000<L4X0?61:03E`0002@000351000X0<00<H4002P0`00
0P000<H4X0?51:03E`0002@000351000W`<00<D4002P0`000P000<D4X0?519l3E`0002@000351000
W@<00<H4002O0`000P000<D4W`?619d3E`0002@000361000W0<00<L4002M0`000P000<H4W@?719`3
E`0002@000371000V`<00<X4002L0`000P000<L4W0?:19/3E`0002@0003:1000V`<00<d4002K0`00
0P000<X4V`?=19/3E`0002@0003=1000V`<00=04002L0`000P000<d4V`?@19`3E`0002@0003@1000
W0<00=44002M0`000P000=04W0?A19d3E`0002@0003A1000W@<00=84002O0`000P000=44W@?B19l3
E`0002@0003B1000W`<00=84002P0`000P000=84W`?B1:03E`0002@0003A1000X0<00=84002R0`00
0P000=84X0?A1:83E`0002@0003>1000XP<00=44002T0`000P000=44XP?>1:@3E`0002@0003:1000
Y0<00<h4002V0`000P000<h4Y0?:1:H3E`0002@000381000YP<00<X4002W0`000P000<X4YP?81:L3
E`0002@000361000Y`<00<P4002Y0`000P000<P4Y`?61:T3E`0002@000351000Z@<00<H4002/0`00
0P000<H4Z@?51:`3E`0002@000351000[0<00<D4002^0`000P000<D4[0?51:h3E`0002@0003=1000
V`<00<l4002L0`000P000<d4V`??19`3E`0002@0003?1000W0<00=04002M0`000P000<l4W0?@19d3
E`0002@0003@1000W@<00=44002O0`000P000=04W@?A19l3E`0002@0003A1000W`<00=44002P0`00
0P000=44W`?A1:03E`0002@0003@1000X0<00=44002R0`000P000=44X0?@1:83E`0002@0003=1000
XP<00=04002T0`000P000=04XP?=1:@3E`0002@0003:1000Y0<00<d4002V0`000P000<d4Y0?:1:H3
E`0002@000351000[0<00<H4002]0`000P000<D4[@?61:`3E`0002@000361000[0<00<P4002/0`00
0P000<H4[0?81:`3E`0002@000381000[0<00<`4002^0`000P000<P4[0?<1:h3E`0002@0003<1000
[P<00<l4002^0`000P000<`4[P??1:h3E`0002@0003?1000[@<00=44002^0`000P000<l4[P?A1:d3
E`0002@0003A1000[0<00=84002]0`000P000=44[@?B1:`3E`0002@000381000[0<00<`4002^0`00
0P000<P4[0?<1:h3E`0002@0003<1000[P<00=04002^0`000P000<`4[P?@1:h3E`0002@0003@1000
[P<00=44002^0`000P000=04[P?A1:h3E`0002@0003A1000[0<00=84002^0`000P000=44[P?B1:`3
E`0002@0003B1000ZP<00=84002/0`000P000=84[0?B1:X3E`0002@0003J1000V`<00=d4002L0`00
0P000=d4V`?J19`3E`0002@0003H1000W0<00=X4002O0`000P000=X4W0?H19l3E`0002@0003G1000
W`<00=P4002S0`000P000=P4W`?G1:<3E`0002@0003G1000X`<00=L4002V0`000P000=L4X`?G1:H3
E`0002@0003G1000YP<00=P4002[0`000P000=L4YP?H1:/3E`0002@0003H1000Z`<00=X4002^0`00
0P000=P4Z`?J1:h3E`0002@0003J1000[P<00=d4002^0`000P000=X4[P?M1:h3E`0002@0003M1000
[P<00=h4002^0`000P000=d4[P?N1:h3E`0002@0003N1000[P<00>44002^0`000P000=h4[P?Q1:h3
E`0002@0003Q1000Z`<00><4002^0`000P000>44[P?S1:/3E`0002@0003S1000YP<00>@4002[0`00
0P000><4Z`?T1:H3E`0002@0003T1000X`<00>@4002V0`000P000>@4YP?T1:<3E`0002@0003S1000
W`<00>@4002S0`000P000>@4X`?S19l3E`0002@0003Q1000W0<00><4002O0`000P000><4W`?Q19`3
E`0002@0003N1000V`<00>44002L0`000P000>44W0?N19/3E`0002@0003M1000V`<00=h4002K0`00
0P000=h4V`?M19/3E`0002@0003K1000V`<00=d4002L0`000P000=d4V`?K19`3E`0002@0003J1000
W0<00=/4002M0`000P000=/4W0?J19d3E`0002@0003I1000W@<00=X4002O0`000P000=X4W@?I19l3
E`0002@0003H1000W`<00=T4002S0`000P000=T4W`?H1:<3E`0002@0003H1000X`<00=P4002V0`00
0P000=P4X`?H1:H3E`0002@0003H1000YP<00=T4002[0`000P000=P4YP?I1:/3E`0002@0003I1000
Z`<00=X4002]0`000P000=T4Z`?J1:d3E`0002@0003J1000[@<00=/4002^0`000P000=X4[@?K1:h3
E`0002@0003K1000[P<00=d4002^0`000P000=/4[P?M1:h3E`0002@0003N1000[P<00>04002^0`00
0P000=h4[P?P1:h3E`0002@0003P1000[@<00>44002^0`000P000>04[P?Q1:d3E`0002@0003Q1000
Z`<00>84002]0`000P000>44[@?R1:/3E`0002@0003R1000YP<00><4002[0`000P000>84Z`?S1:H3
E`0002@0003S1000X`<00><4002V0`000P000><4YP?S1:<3E`0002@0003R1000W`<00><4002S0`00
0P000><4X`?R19l3E`0002@0003Q1000W@<00>84002O0`000P000>84W`?Q19d3E`0002@0003P1000
W0<00>44002M0`000P000>44W@?P19`3E`0002@0003N1000V`<00>04002L0`000P000>04W0?N19/3
9`0001P00003000000000000000000009@0000`000030000:00000`000010000F`0006P0002R0000
4@400:X0000I0@000@000140000A0000ZP0E0JX0506Y01<1Z@0B0JP04@6T0141X`0B0J804`6R01L1
X`0H0J@06@6X01T1Z@0H0JT05`6Z01L1ZP0F0JX05@5K0000L0000=X0001k0000hP0008<000010000
4`0001<0003R07l0hP1m0>40O03P07/0g01k0=`0O03K07`0f`1m0=X0O@3J0840f`210=/0PP3L0880
g0230>00P`3Q0880hP210>80P03R07l0F`0006`0000C0@002`4001/1000C0@000@000180000B0000
6`4?0A/13P4J0@d16P4<0AT1304I0@/1504;0A@1304C0@`14`4B0A@14P4D0A<16@4C0AT14P4J0A81
6P4A0A/1404K0@l1F`000600001;0@005P4005<1000N0@000@0000l0000?0000D`4J0E<1605B0AL1
D@4F0Dd15P5<0AL1B`4H0D/1705<0Ad1C@4N0E417P5B0Ad1D`4L0E<16`5C0AX1F`0007P000230@00
m@0008/1003n00000@0001D0000E0000R`7i08/1m`2:0OH0R@7e08H1m@250OH0Q07g08@1n0230OT0
P`7j08@1n`240O`0Q@7m08H1o@260Oh0R07n08T1o@2:0Od0R`7l08/1nP2;0OT0F`0006`0002l0@00
WP400<@1002V0@000@000180000B0000a06R0L@1X0730J01``6O0L81W`720Ih1_P6N0Kd1W`6l0J01
_06T0Kd1Y@6n0JH1`P6V0L81Y@730JD1``6T0L@1X`740J81F`0007P0003d0@00l`400?`1003l0@00
0@0001D0000E0000o07g0O`1m@7k0O@1nP7d0OX1l`7g0O<1mP7d0OD1m07e0OD1m07f0O@1n@7e0OX1
m@7k0OL1n`7h0O`1n@7k0O/1n`7k0OX1o07j0O`1n07l0OL1F`000800000/0P00^P8003D200330P00
0@0001L0000G0000=@:n0SD2_@8d0[d2=0:l0S<2^`8b0[X2;`:j0Rh2^`8]0[`2;@:m0R`2_P8]0[l2
;@;10Rh2`P8_0/82<0;30S42``8b0/82<`;20S@2`@8d0/02=@:o0SD2_P9K0000K00006D2000h0P00
K@800402000100004P000180001]0S`2K@8k0V`2>P9/0ST2J`8i0VX2>09W0SP2I`8i0VH2>@9U0SX2
I@8o0VH2?`9V0T02K0900V`2?`9]0Sh2K@8m0Vd2?09K0000L00009d2003_0@00Y@800?L100010000
4`0001<0002U0_<1Y@;a0J@2l06T0^l1W`;_0Ih2l06N0_41W@;a0Id2m06N0_D1WP;f0Il2mP6P0_L1
X`;g0J<2mP6T0_H1Y@;e0JD2m06U0_<1F`0006`0003F0P00L0800=h2001h0P000@000180000B0000
gP9d0]h2L`;M0W82g@9a0]`2L@;L0W02f09`0]L2L@;F0W42eP9g0]L2M`;H0WP2g09h0]`2M`;M0WL2
g@9f0]h2M@;N0W@2F`000600000>0`00>`8001H300130P000@0000l0000?00005P<o0QH3?@8E0c`2
50<k0Q03>`8?0c`23P<m0Ph3@@8?0d8240=30Q@3@`8E0d825P=10QH3@08F0cl2F`00070000160`00
f`4004h3003S0@000@0001<0000C0000CP?O0Dh3g05=0m`1C@?K0DP3f`580m`1A`?L0DL3gP560mh1
AP?P0DL3h@570n81B0?R0DP3h`5=0n<1C@?R0Dh3hP5>0n01CP?O0E/0001`0000O`<00<P100270`00
d0400040000C00004`0008L3c0670lX1QP?:0HH3b@650lT1Q@?80H43b0600lT1O`?:0Gl3cP600ll1
P@?@0HD3d0650ll1QP??0HH3cP670lh1Q`?=0HL3c05K0000P0000;L3001:0P00_`<005<200010000
5`0001L0002o0dh2_`=<0[h3C0:n0d/2_@=;0[d3BP:j0dX2^@=;0[P3B`:h0d`2]`==0[L3D0:h0e42
^0=B0[X3DP:j0e<2_0=C0[d3DP:n0e82_P=A0[l3D@:o0dl2_`=>0U/0001d0000l0<002<2003h0`00
;0800040000D000050000?P39`;h0bH2m`<V0_L39@;f0b@2m@<S0_838`;a0b@2l0<U0_03:P;a0b/2
lP<[0_<3;0;d0b`2m@<[0_H3:`;g0bX2m`<Y0_P3:0;h0bL2F`0006P0000X1000cP800304003F0P00
0@000140000A0000<0CB0S04d08_1=02;`C>0RT4cP8Y1<l2:0C@0RP4e08Y1=D2:PCE0RX4eP8]1=H2
;PCE0Rl4e@8_1=@2<0CC0S04dP9K0000L0000604003`0P00J0@00?P2000100004`0001<0001X1?@2
J0Cb0VL4l@9V1?02HPC`0V84l@9Q1?42H@Cb0V04lP9P1?D2H@Cf0V44m`9R1?L2HPCh0VH4n09W1?L2
J0Cf0VP4m@9X1?@2F`0007@0002I1000I`<00:44001`0`000@0001@0000D0000X@A[0j44JP>P16T3
W`AX0ih4J0>M16L3W0AX0iX4J0>J16T3V@AY0iT4KP>J16l3V`A_0i/4L0>N1703W`A_0j04K`>P16h3
X@A]0j44J`=K0000J0000=44000h0`00f@@00403000100004@000140003I13`3f@@j0mP4>@?G13P3
d`@h0m84>@?B13X3d@@j0m44?P?B13h3dP@o0m<4@0?G1403f0@o0mT4?P?I13d3f@@l0eL0001`0000
YP0007l0003E1000J`<001D0002V01D1gP1o01L13`5?0AX1R07i0<01XP7h0OL1<@:n0VT2?0:R0_<1
fP9d0Q83?`9;0ml1P`?<0K/3CP;d0bL2;0CB0VD4m0:M16/3e@@l0mD4?@=G0000=0000:4000180000
fP@008D300060000X@240j40B03J14P0fPB40j40Q0>Q08D39@0000`000050020:00000`000030000
9@0000`000080020:00000`0000200009@0000`0000=00203P0001@000000000400001@0
\>"], "Graphics",
 ImageSize->{242.75, 192.25},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}}],

Cell["\<\
Here are the actual and estimated trade direction indicators:\
\>", "Text"],

Cell[GraphicsData["Metafile", "\<\
CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@000AZ00@0006`00000000000000?D4003/0`00
000000000037FP00idL00215CDH00040P1X10:h7000300000000000000000000@0H00;04000U0@00
g000000000000000000008Qh101PF`<02@000100003f1000k@<000X0000@000000000000000D0000
300000d0000U0000300000P0080W00006000004000000000oooo0000000U000030000040000[0000
6000000000000000mP@00>d3000[00006000000000000000mP@00>d3000V00007000008000000000
0@000000000000009@0000`000020000E`0002@0003d00002P000?X0000M00000P000?X02P3d01d0
E`0002@0003j00002P000041000M00000P000?X02P010Ad0E`0002@0003j00003@000001000M0000
0P000?X03@000Ad0E`0002@0003f000060000?h0000H00000P000?H0603n01P0E`0002@0003b0000
7@000?P0000M00000P000?807@3h01d0E`0002@0003m00007@000081000M00000P000?d07@020Ad0
E`0002@0000@0@004`000141000D00000P0001414`0@0A@0E`0002@0000@0@0050000141000E0000
0P000101500A0AD0E`0002@0000A0@0050000181000E00000P0001415@0B0A@0E`0002@0000B0@00
4`000181000D00000P000181500B0A<0E`0002@0000@0@004@000181000C00000P0001814`0@0A40
E`0002@0000>0@0040000101000A00000P0001014@0>0A00E`0002@0000<0@00400000h1000@0000
0P0000h1400<0A00E`0002@000090@00400000`1000A00000P0000`140090A40E`0002@000070@00
4@0000T1000C00000P0000T14@070A<0E`0002@000060@004`0000L1000F00000P0000L14`060AH0
E`0002@000060@005P0000H1000H00000P0000H15P060AP0E`0002@000060@00600000L1000K0000
0P0000H160070A/0E`0002@000070@006`0000T1000M00000P0000L16`090Ad0E`0002@000090@00
7@0000`1000M00000P0000T17@0<0Ad0E`0002@0000<0@007@0000d1000M00000P0000`17@0=0Ad0
E`0002@0000=0@007@000101000M00000P0000d17@0@0Ad0E`0002@0000@0@006`000181000M0000
0P0001017@0B0A/0E`0002@0000:0@00400000`1000A00000P0000`1400:0A40E`0002@000080@00
4@0000X1000C00000P0000X14@080A<0E`0002@000070@004`0000P1000F00000P0000P14`070AH0
E`0002@000070@005P0000L1000H00000P0000L15P070AP0E`0002@000070@00600000P1000K0000
0P0000L160080A/0E`0002@000080@006`0000X1000M00000P0000P16`0:0Ad0E`0002@0000:0@00
7@0000`1000M00000P0000X17@0<0Ad0E`0002@0000I0@002P0001T1000J00000P0001T12P0I0AX0
E`0002@0000I0@006P0001X1000M00000P0001T16P0J0Ad0E`0002@0000J0@007@0001`1000M0000
0P0001X17@0L0Ad0E`0002@0000L0@007@0001d1000M00000P0001`17@0M0Ad0E`0002@0000M0@00
7@0001l1000M00000P0001d17@0O0Ad0E`0002@0000O0@006`000201000M00000P0001l17@0P0A/0
E`0002@0000J0@002P0001X1000J00000P0001X12P0J0AX0E`0002@0000J0@006P0001/1000M0000
0P0001X16P0K0Ad0E`0002@0000K0@007@0001`1000M00000P0001/17@0L0Ad0E`0002@0000F0@00
400001d1000@00000P0001H1400M0A00E`0002@0000V0@00400002H1000K00000P0002H1400V0A/0
E`0002@0000V0@006`0002L1000M00000P0002H16`0W0Ad0E`0002@0000W0@007@0002X1000M0000
0P0002L17@0Z0Ad0E`0002@0000Z0@007@0002/1000M00000P0002X17@0[0Ad0E`0002@0000[0@00
7@0002h1000M00000P0002/17@0^0Ad0E`0002@0000^0@006`000301000M00000P0002h17@0`0A/0
E`0002@0000W0@00400002L1000K00000P0002L1400W0A/0E`0002@0000W0@006`0002P1000M0000
0P0002L16`0X0Ad0E`0002@0000X0@007@0002X1000M00000P0002P17@0Z0Ad0E`0002@0000`0@00
40000301000M00000P000301400`0Ad0E`0002@0000a0@0040000341000M00000P000341400a0Ad0
E`0002@0000S0@00400002L1000@00000P0002<1400W0A00E`0002@0000]0@0040000341000@0000
0P0002d1400a0A00E`0002@0000`0@007@0003<1000M00000P0003017@0c0Ad0E`0002@0000j0@00
4P0003X1000C00000P0003X14P0j0A<0E`0002@0000i0@004`0003X1000C00000P0003X14`0i0A<0
E`0002@0000i0@004P0003T1000C00000P0003T14`0i0A80E`0002@0000i0@004@0003X1000B0000
0P0003T14P0j0A40E`0002@0000j0@00400003`1000A00000P0003X14@0l0A00E`0002@0000l0@00
400003l1000@00000P0003`1400o0A00E`0002@0000o0@0040000441000A00000P0003l140110A40
E`0002@000110@004@000481000B00000P0004414@120A80E`0002@000120@004P0004<1000D0000
0P0004814P130A@0E`0002@000130@00500004<1000K00000P0004<150130A/0E`0002@000130@00
6`0004@1000M00000P0004<16`140Ad0E`0002@000140@007@0004D1000M00000P0004@17@150Ad0
E`0002@000120@004P000481000K00000P0004814P120A/0E`0002@000120@006`0004<1000M0000
0P0004816`130Ad0E`0002@000130@007@0004D1000M00000P0004<17@150Ad0E`0002@000150@00
7@0004H1000M00000P0004D17@160Ad0E`0002@000110@0050000481000E00000P00048150110AD0
E`0002@0000l0@005@000441000F00000P0004415@0l0AH0E`0002@0000i0@005P0003`1000G0000
0P0003`15P0i0AL0E`0002@0000h0@005`0003T1000I00000P0003T15`0h0AT0E`0002@0000h0@00
6@0003P1000K00000P0003P16@0h0A/0E`0002@0000h0@006`0003T1000M00000P0003P16`0i0Ad0
E`0002@0000i0@007@0003`1000M00000P0003T17@0l0Ad0E`0002@0000l0@007@0003h1000M0000
0P0003`17@0n0Ad0E`0002@0000n0@007@000401000M00000P0003h17@100Ad0E`0002@000100@00
6`000481000M00000P0004017@120A/0E`0002@0000j0@005P0003`1000G00000P0003`15P0j0AL0
E`0002@0000i0@005`0003X1000I00000P0003X15`0i0AT0E`0002@0000i0@006@0003T1000K0000
0P0003T16@0i0A/0E`0002@0000i0@006`0003X1000M00000P0003T16`0j0Ad0E`0002@0000j0@00
7@0003`1000M00000P0003X17@0l0Ad0E`0002@0001<0@002P0004`1000M00000P0004`12P1<0Ad0
E`0002@0001=0@002P0004d1000M00000P0004d12P1=0Ad0E`0002@000190@002P0004d1000:0000
0P0004T12P1=0@X0E`0002@000190@007@000501000M00000P0004T17@1@0Ad0E`0002@0001O0@00
4P0005l1000C00000P0005l14P1O0A<0E`0002@0001N0@004`0005l1000C00000P0005l14`1N0A<0
E`0002@0001N0@004P0005h1000C00000P0005h14`1N0A80E`0002@0001N0@004@0005l1000B0000
0P0005h14P1O0A40E`0002@0001O0@0040000641000A00000P0005l14@1Q0A00E`0002@0001Q0@00
400006@1000@00000P000641401T0A00E`0002@0001T0@00400006H1000A00000P0006@1401V0A40
E`0002@0001V0@004@0006L1000B00000P0006H14@1W0A80E`0002@0001W0@004P0006P1000D0000
0P0006L14P1X0A@0E`0002@0001X0@00500006P1000K00000P0006P1501X0A/0E`0002@0001X0@00
6`0006T1000M00000P0006P16`1Y0Ad0E`0002@0001Y0@007@0006X1000M00000P0006T17@1Z0Ad0
E`0002@0001W0@004P0006L1000K00000P0006L14P1W0A/0E`0002@0001W0@006`0006P1000M0000
0P0006L16`1X0Ad0E`0002@0001X0@007@0006X1000M00000P0006P17@1Z0Ad0E`0002@0001Z0@00
7@0006/1000M00000P0006X17@1[0Ad0E`0002@0001V0@00500006L1000E00000P0006L1501V0AD0
E`0002@0001Q0@005@0006H1000F00000P0006H15@1Q0AH0E`0002@0001N0@005P000641000G0000
0P0006415P1N0AL0E`0002@0001M0@005`0005h1000I00000P0005h15`1M0AT0E`0002@0001M0@00
6@0005d1000K00000P0005d16@1M0A/0E`0002@0001M0@006`0005h1000M00000P0005d16`1N0Ad0
E`0002@0001N0@007@000641000M00000P0005h17@1Q0Ad0E`0002@0001Q0@007@0006<1000M0000
0P0006417@1S0Ad0E`0002@0001S0@007@0006D1000M00000P0006<17@1U0Ad0E`0002@0001U0@00
6`0006L1000M00000P0006D17@1W0A/0E`0002@0001O0@005P000641000G00000P0006415P1O0AL0
E`0002@0001N0@005`0005l1000I00000P0005l15`1N0AT0E`0002@0001N0@006@0005h1000K0000
0P0005h16@1N0A/0E`0002@0001N0@006`0005l1000M00000P0005h16`1O0Ad0E`0002@0001O0@00
7@000641000M00000P0005l17@1Q0Ad0E`0002@0001a0@0040000741000M00000P000741401a0Ad0
E`0002@0001b0@0040000781000M00000P000781401b0Ad0E`0002@0001b0@004@0007@1000C0000
0P0007814`1d0A40E`0002@0001d0@00400007H1000A00000P0007@14@1f0A00E`0002@0001f0@00
400007P1000@00000P0007H1401h0A00E`0002@0001h0@00400007/1000A00000P0007P1401k0A40
E`0002@0001k0@004@0007`1000C00000P0007/14@1l0A<0E`0002@0001l0@004`0007`1000M0000
0P0007`14`1l0Ad0E`0002@0001h0@00400007X1000A00000P0007P1401j0A40E`0002@0001j0@00
4@0007/1000C00000P0007X14@1k0A<0E`0002@0001k0@004`0007/1000M00000P0007/14`1k0Ad0
E`0002@0001^0@0040000781000@00000P0006h1401b0A00E`0002@0001^0@007@0007D1000M0000
0P0006h17@1e0Ad0E`0002@0001h0@007@0007h1000M00000P0007P17@1n0Ad0E`0002@0002>0@00
2P0008h1000M00000P0008h12P2>0Ad0E`0002@0002?0@002P0008l1000M00000P0008l12P2?0Ad0
E`0002@0002<0@004@0008h1000C00000P0008h14`2<0A40E`0002@0002:0@00400008`1000A0000
0P0008`14@2:0A00E`0002@000290@00400008X1000@00000P0008X140290A00E`0002@000260@00
400008T1000A00000P0008T140260A40E`0002@000240@004@0008H1000C00000P0008H14@240A<0
E`0002@000230@004`0008@1000F00000P0008@14`230AH0E`0002@000230@005P0008<1000H0000
0P0008<15P230AP0E`0002@000230@00600008@1000K00000P0008<160240A/0E`0002@000240@00
6`0008H1000M00000P0008@16`260Ad0E`0002@000260@007@0008T1000M00000P0008H17@290Ad0
E`0002@000290@007@0008X1000M00000P0008T17@2:0Ad0E`0002@0002:0@007@0008`1000M0000
0P0008X17@2<0Ad0E`0002@0002<0@006`0008h1000M00000P0008`17@2>0A/0E`0002@000270@00
400008T1000A00000P0008T140270A40E`0002@000250@004@0008L1000C00000P0008L14@250A<0
E`0002@000240@004`0008D1000F00000P0008D14`240AH0E`0002@000240@005P0008@1000H0000
0P0008@15P240AP0E`0002@000240@00600008D1000K00000P0008@160250A/0E`0002@000250@00
6`0008L1000M00000P0008D16`270Ad0E`0002@000270@007@0008T1000M00000P0008L17@290Ad0
E`0002@0002;0@002P0008l1000:00000P0008/12P2?0@X0E`0002@0002>0@007@000941000M0000
0P0008h17@2A0Ad0E`0002@0002Y0@0040000:X1000B00000P000:T14P2Z0A00E`0002@0002Z0@00
40000:X1000D00000P000:X1402Z0A@0E`0002@0002Y0@004P000:X1000D00000P000:X1502Y0A80
E`0002@0002X0@004@000:T1000B00000P000:T14P2X0A40E`0002@0002V0@0040000:P1000A0000
0P000:P14@2V0A00E`0002@0002S0@0040000:H1000@00000P000:H1402S0A00E`0002@0002Q0@00
40000:<1000A00000P000:<1402Q0A40E`0002@0002P0@004@000:41000B00000P000:414@2P0A80
E`0002@0002P0@004P000:01000D00000P000:014P2P0A@0E`0002@0002P0@0050000:41000E0000
0P000:01502Q0AD0E`0002@0002Q0@005@000:<1000F00000P000:415@2S0AH0E`0002@0002S0@00
5P000:L1000H00000P000:<15P2W0AP0E`0002@0002W0@0060000:T1000I00000P000:L1602Y0AT0
E`0002@0002Y0@006@000:X1000J00000P000:T16@2Z0AX0E`0002@0002P0@004`000:41000D0000
0P000:014`2Q0A@0E`0002@0002Q0@0050000:<1000E00000P000:41502S0AD0E`0002@0002S0@00
5@000:L1000G00000P000:<15@2W0AL0E`0002@0002W0@005`000:T1000H00000P000:L15`2Y0AP0
E`0002@0002Y0@0060000:X1000I00000P000:T1602Z0AT0E`0002@0002Z0@006@000:X1000L0000
0P000:X16@2Z0A`0E`0002@0002Y0@0070000:X1000M00000P000:X1702Y0Ad0E`0002@0002W0@00
7@000:T1000M00000P000:T17@2W0Ad0E`0002@0002T0@007@000:L1000M00000P000:L17@2T0Ad0
E`0002@0002R0@007@000:@1000M00000P000:@17@2R0Ad0E`0002@0002Q0@0070000:81000M0000
0P000:817@2Q0A`0E`0002@0002P0@006P000:41000L00000P000:41702P0AX0E`0002@0002P0@00
6P000:01000M00000P000:016P2P0Ad0E`0002@0002P0@0070000:41000M00000P000:017@2Q0A`0
E`0002@0002`0@002P000;41000;00000P000;412P2`0@/0E`0002@0002`0@002`000;41000<0000
0P000;012`2a0@`0E`0002@0002a0@002`000;81000<00000P000;41302b0@/0E`0002@0002a0@00
2P000;81000;00000P000;812`2a0@X0E`0002@0002a0@0040000;41000M00000P000;41402a0Ad0
E`0002@0002b0@0040000;81000M00000P000;81402b0Ad0E`0002@0002^0@0040000;81000@0000
0P000:h1402b0A00E`0002@0002^0@007@000;D1000M00000P000:h17@2e0Ad0E`0002@0002k0@00
40000;/1000M00000P000;/1402k0Ad0E`0002@0002l0@0040000;`1000M00000P000;`1402l0Ad0
E`0002@0002l0@004@000;h1000C00000P000;`14`2n0A40E`0002@0002n0@0040000<01000A0000
0P000;h14@300A00E`0002@000300@0040000<81000@00000P000<0140320A00E`0002@000320@00
40000<D1000A00000P000<8140350A40E`0002@000350@004@000<H1000C00000P000<D14@360A<0
E`0002@000360@004`000<H1000M00000P000<H14`360Ad0E`0002@000320@0040000<@1000A0000
0P000<8140340A40E`0002@000340@004@000<D1000C00000P000<@14@350A<0E`0002@000350@00
4`000<D1000M00000P000<D14`350Ad0E`0002@000360@004@000<L1000C00000P000<H14`370A40
E`0002@000370@0040000<X1000A00000P000<L14@3:0A00E`0002@0003:0@0040000<`1000@0000
0P000<X1403<0A00E`0002@0003<0@0040000<l1000A00000P000<`1403?0A40E`0002@0003?0@00
4@000=01000C00000P000<l14@3@0A<0E`0002@0003@0@004`000=01000M00000P000=014`3@0Ad0
E`0002@0003<0@0040000<h1000A00000P000<`1403>0A40E`0002@0003>0@004@000<l1000C0000
0P000<h14@3?0A<0E`0002@0003?0@004`000<l1000M00000P000<l14`3?0Ad0E`0002@0002h0@00
40000;`1000@00000P000;P1402l0A00E`0002@0002h0@007@000;l1000M00000P000;P17@2o0Ad0
E`0002@000320@007@000<P1000M00000P000<817@380Ad0E`0002@0003<0@007@000=81000M0000
0P000<`17@3B0Ad0E`0002@0003H0@0040000=P1000K00000P000=P1403H0A/0E`0002@0003H0@00
6`000=T1000M00000P000=P16`3I0Ad0E`0002@0003I0@007@000=`1000M00000P000=T17@3L0Ad0
E`0002@0003L0@007@000=d1000M00000P000=`17@3M0Ad0E`0002@0003M0@007@000>01000M0000
0P000=d17@3P0Ad0E`0002@0003P0@006`000>81000M00000P000>017@3R0A/0E`0002@0003I0@00
40000=T1000K00000P000=T1403I0A/0E`0002@0003I0@006`000=X1000M00000P000=T16`3J0Ad0
E`0002@0003J0@007@000=`1000M00000P000=X17@3L0Ad0E`0002@0003R0@0040000>81000M0000
0P000>81403R0Ad0E`0002@0003S0@0040000><1000M00000P000><1403S0Ad0E`0002@0003E0@00
40000=T1000@00000P000=D1403I0A00E`0002@0003O0@0040000><1000@00000P000=l1403S0A00
E`0002@0003R0@007@000>D1000M00000P000>817@3U0Ad0E`0002@0003/0@002P000>`1000M0000
0P000>`12P3/0Ad0E`0002@0003]0@002P000>d1000M00000P000>d12P3]0Ad0E`0002@0003Y0@00
2P000>d1000:00000P000>T12P3]0@X0E`0002@0003Y0@007@000?01000M00000P000>T17@3`0Ad0
E`0002@0003e0@004P000?D1000C00000P000?D14P3e0A<0E`0002@0003d0@004`000?D1000C0000
0P000?D14`3d0A<0E`0002@0003d0@004P000?@1000C00000P000?@14`3d0A80E`0002@0003d0@00
4@000?D1000B00000P000?@14P3e0A40E`0002@0003e0@0040000?L1000A00000P000?D14@3g0A00
E`0002@0003g0@0040000?X1000@00000P000?L1403j0A00E`0002@0003j0@0040000?`1000A0000
0P000?X1403l0A40E`0002@0003l0@004@000?d1000B00000P000?`14@3m0A80E`0002@0003m0@00
4P000?h1000D00000P000?d14P3n0A@0E`0002@0003n0@0050000?h1000K00000P000?h1503n0A/0
E`0002@0003n0@006`000?l1000M00000P000?h16`3o0Ad0E`0002@0003o0@007@000002000M0000
0P000?l17@000Qd0E`0002@0003m0@004P000?d1000K00000P000?d14P3m0A/0E`0002@0003m0@00
6`000?h1000M00000P000?d16`3n0Ad0E`0002@0003n0@007@000002000M00000P000?h17@000Qd0
E`0002@000000P007@000042000M00000P0000027@010Qd0E`0002@0003l0@0050000?d1000E0000
0P000?d1503l0AD0E`0002@0003g0@005@000?`1000F00000P000?`15@3g0AH0E`0002@0003d0@00
5P000?L1000G00000P000?L15P3d0AL0E`0002@0003c0@005`000?@1000I00000P000?@15`3c0AT0
E`0002@0003c0@006@000?<1000K00000P000?<16@3c0A/0E`0002@0003c0@006`000?@1000M0000
0P000?<16`3d0Ad0E`0002@0003d0@007@000?L1000M00000P000?@17@3g0Ad0E`0002@0003g0@00
7@000?T1000M00000P000?L17@3i0Ad0E`0002@0003i0@007@000?/1000M00000P000?T17@3k0Ad0
E`0002@0003k0@006`000?d1000M00000P000?/17@3m0A/0E`0002@0003e0@005P000?L1000G0000
0P000?L15P3e0AL0E`0002@0003d0@005`000?D1000I00000P000?D15`3d0AT0E`0002@0003d0@00
6@000?@1000K00000P000?@16@3d0A/0E`0002@0003d0@006`000?D1000M00000P000?@16`3e0Ad0
E`0002@0003e0@007@000?L1000M00000P000?D17@3g0Ad0E`0002@000070P002P0000L2000J0000
0P0000L22P070QX0E`0002@000070P006P0000P2000M00000P0000L26P080Qd0E`0002@000080P00
7@0000X2000M00000P0000P27@0:0Qd0E`0002@0000:0P007@0000/2000M00000P0000X27@0;0Qd0
E`0002@0000;0P007@0000d2000M00000P0000/27@0=0Qd0E`0002@0000=0P006`0000h2000M0000
0P0000d27@0>0Q/0E`0002@000080P002P0000P2000J00000P0000P22P080QX0E`0002@000080P00
6P0000T2000M00000P0000P26P090Qd0E`0002@000090P007@0000X2000M00000P0000T27@0:0Qd0
E`0002@000040P00400000/2000@00000P0000@2400;0Q00E`0002@0000D0P005P0001l2000F0000
0P0001@25P0O0QH0E`0002@0000O0P00500001l2000F00000P0001l25P0O0Q@0E`0002@0000N0P00
4P0001l2000D00000P0001l2500N0Q80E`0002@0000M0P004@0001h2000B00000P0001h24P0M0Q40
E`0002@0000K0P00400001d2000A00000P0001d24@0K0Q00E`0002@0000I0P00400001/2000@0000
0P0001/2400I0Q00E`0002@0000F0P00400001T2000A00000P0001T2400F0Q40E`0002@0000D0P00
4@0001H2000C00000P0001H24@0D0Q<0E`0002@0000C0P004`0001@2000F00000P0001@24`0C0QH0
E`0002@0000C0P005P0001<2000H00000P0001<25P0C0QP0E`0002@0000C0P00600001@2000K0000
0P0001<2600D0Q/0E`0002@0000D0P006`0001H2000M00000P0001@26`0F0Qd0E`0002@0000F0P00
7@0001T2000M00000P0001H27@0I0Qd0E`0002@0000I0P007@0001X2000M00000P0001T27@0J0Qd0
E`0002@0000J0P007@0001d2000M00000P0001X27@0M0Qd0E`0002@0000M0P006`0001l2000M0000
0P0001d27@0O0Q/0E`0002@0000N0P004`0001h2000F00000P0001h25P0N0Q<0E`0002@0000M0P00
4@0001h2000C00000P0001h24`0M0Q40E`0002@0000G0P00400001T2000A00000P0001T2400G0Q40
E`0002@0000E0P004@0001L2000C00000P0001L24@0E0Q<0E`0002@0000D0P004`0001D2000F0000
0P0001D24`0D0QH0E`0002@0000D0P005P0001@2000H00000P0001@25P0D0QP0E`0002@0000D0P00
600001D2000K00000P0001@2600E0Q/0E`0002@0000E0P006`0001L2000M00000P0001D26`0G0Qd0
E`0002@0000G0P007@0001T2000M00000P0001L27@0I0Qd0E`0002@0000^0P002P0002h2000M0000
0P0002h22P0^0Qd0E`0002@0000_0P002P0002l2000M00000P0002l22P0_0Qd0E`0002@0000/0P00
4@0002h2000C00000P0002h24`0/0Q40E`0002@0000Z0P00400002`2000A00000P0002`24@0Z0Q00
E`0002@0000Y0P00400002X2000@00000P0002X2400Y0Q00E`0002@0000V0P00400002T2000A0000
0P0002T2400V0Q40E`0002@0000T0P004@0002H2000C00000P0002H24@0T0Q<0E`0002@0000S0P00
4`0002@2000F00000P0002@24`0S0QH0E`0002@0000S0P005P0002<2000H00000P0002<25P0S0QP0
E`0002@0000S0P00600002@2000K00000P0002<2600T0Q/0E`0002@0000T0P006`0002H2000M0000
0P0002@26`0V0Qd0E`0002@0000V0P007@0002T2000M00000P0002H27@0Y0Qd0E`0002@0000Y0P00
7@0002X2000M00000P0002T27@0Z0Qd0E`0002@0000Z0P007@0002`2000M00000P0002X27@0/0Qd0
E`0002@0000/0P006`0002h2000M00000P0002`27@0^0Q/0E`0002@0000W0P00400002T2000A0000
0P0002T2400W0Q40E`0002@0000U0P004@0002L2000C00000P0002L24@0U0Q<0E`0002@0000T0P00
4`0002D2000F00000P0002D24`0T0QH0E`0002@0000T0P005P0002@2000H00000P0002@25P0T0QP0
E`0002@0000T0P00600002D2000K00000P0002@2600U0Q/0E`0002@0000U0P006`0002L2000M0000
0P0002D26`0W0Qd0E`0002@0000W0P007@0002T2000M00000P0002L27@0Y0Qd0E`0002@0000[0P00
2P0002l2000:00000P0002/22P0_0PX0E`0002@0000^0P007@000342000M00000P0002h27@0a0Qd0
E`0002@000120P002P000482000J00000P0004822P120QX0E`0002@000120P006P0004<2000M0000
0P0004826P130Qd0E`0002@000130P007@0004D2000M00000P0004<27@150Qd0E`0002@000150P00
7@0004H2000M00000P0004D27@160Qd0E`0002@000160P007@0004P2000M00000P0004H27@180Qd0
E`0002@000180P006`0004T2000M00000P0004P27@190Q/0E`0002@000130P002P0004<2000J0000
0P0004<22P130QX0E`0002@000130P006P0004@2000M00000P0004<26P140Qd0E`0002@000140P00
7@0004D2000M00000P0004@27@150Qd0E`0002@0000o0P00400004H2000@00000P0003l240160Q00
E`0002@0001@0P0040000502000M00000P000502401@0Qd0E`0002@0001A0P0040000542000M0000
0P000542401A0Qd0E`0002@0001A0P004`000582000F00000P0005425P1B0Q<0E`0002@0001B0P00
4@0005@2000C00000P0005824`1D0Q40E`0002@0001D0P00400005D2000A00000P0005@24@1E0Q00
E`0002@0001E0P00400005P2000@00000P0005D2401H0Q00E`0002@0001H0P00400005T2000A0000
0P0005P2401I0Q40E`0002@0001I0P004@0005T2000B00000P0005T24@1I0Q80E`0002@0001H0P00
4P0005T2000C00000P0005T24P1H0Q<0E`0002@0001G0P004P0005P2000C00000P0005P24`1G0Q80
E`0002@0001G0P004@0005P2000B00000P0005L24P1H0Q40E`0002@0001=0P0040000542000@0000
0P0004d2401A0Q00E`0002@0001=0P007@0005@2000M00000P0004d27@1D0Qd0E`0002@0001O0P00
4P0005l2000C00000P0005l24P1O0Q<0E`0002@0001N0P004`0005l2000C00000P0005l24`1N0Q<0
E`0002@0001N0P004P0005h2000C00000P0005h24`1N0Q80E`0002@0001N0P004@0005l2000B0000
0P0005h24P1O0Q40E`0002@0001O0P0040000642000A00000P0005l24@1Q0Q00E`0002@0001Q0P00
400006@2000@00000P000642401T0Q00E`0002@0001T0P00400006H2000A00000P0006@2401V0Q40
E`0002@0001V0P004@0006L2000B00000P0006H24@1W0Q80E`0002@0001W0P004P0006P2000D0000
0P0006L24P1X0Q@0E`0002@0001X0P00500006P2000K00000P0006P2501X0Q/0E`0002@0001X0P00
6`0006T2000M00000P0006P26`1Y0Qd0E`0002@0001Y0P007@0006X2000M00000P0006T27@1Z0Qd0
E`0002@0001W0P004P0006L2000K00000P0006L24P1W0Q/0E`0002@0001W0P006`0006P2000M0000
0P0006L26`1X0Qd0E`0002@0001X0P007@0006X2000M00000P0006P27@1Z0Qd0E`0002@0001Z0P00
7@0006/2000M00000P0006X27@1[0Qd0E`0002@0001V0P00500006L2000E00000P0006L2501V0QD0
E`0002@0001Q0P005@0006H2000F00000P0006H25@1Q0QH0E`0002@0001N0P005P000642000G0000
0P0006425P1N0QL0E`0002@0001M0P005`0005h2000I00000P0005h25`1M0QT0E`0002@0001M0P00
6@0005d2000K00000P0005d26@1M0Q/0E`0002@0001M0P006`0005h2000M00000P0005d26`1N0Qd0
E`0002@0001N0P007@000642000M00000P0005h27@1Q0Qd0E`0002@0001Q0P007@0006<2000M0000
0P0006427@1S0Qd0E`0002@0001S0P007@0006D2000M00000P0006<27@1U0Qd0E`0002@0001U0P00
6`0006L2000M00000P0006D27@1W0Q/0E`0002@0001O0P005P000642000G00000P0006425P1O0QL0
E`0002@0001N0P005`0005l2000I00000P0005l25`1N0QT0E`0002@0001N0P006@0005h2000K0000
0P0005h26@1N0Q/0E`0002@0001N0P006`0005l2000M00000P0005h26`1O0Qd0E`0002@0001O0P00
7@000642000M00000P0005l27@1Q0Qd0E`0002@0001j0P002P0007X2000M00000P0007X22P1j0Qd0
E`0002@0001k0P002P0007/2000M00000P0007/22P1k0Qd0E`0002@0001h0P004@0007X2000C0000
0P0007X24`1h0Q40E`0002@0001f0P00400007P2000A00000P0007P24@1f0Q00E`0002@0001e0P00
400007H2000@00000P0007H2401e0Q00E`0002@0001b0P00400007D2000A00000P0007D2401b0Q40
E`0002@0001`0P004@000782000C00000P0007824@1`0Q<0E`0002@0001_0P004`000702000F0000
0P0007024`1_0QH0E`0002@0001_0P005P0006l2000H00000P0006l25P1_0QP0E`0002@0001_0P00
60000702000K00000P0006l2601`0Q/0E`0002@0001`0P006`000782000M00000P0007026`1b0Qd0
E`0002@0001b0P007@0007D2000M00000P0007827@1e0Qd0E`0002@0001e0P007@0007H2000M0000
0P0007D27@1f0Qd0E`0002@0001f0P007@0007P2000M00000P0007H27@1h0Qd0E`0002@0001h0P00
6`0007X2000M00000P0007P27@1j0Q/0E`0002@0001c0P00400007D2000A00000P0007D2401c0Q40
E`0002@0001a0P004@0007<2000C00000P0007<24@1a0Q<0E`0002@0001`0P004`000742000F0000
0P0007424`1`0QH0E`0002@0001`0P005P000702000H00000P0007025P1`0QP0E`0002@0001`0P00
60000742000K00000P000702601a0Q/0E`0002@0001a0P006`0007<2000M00000P0007426`1c0Qd0
E`0002@0001c0P007@0007D2000M00000P0007<27@1e0Qd0E`0002@0001g0P002P0007/2000:0000
0P0007L22P1k0PX0E`0002@0001j0P007@0007d2000M00000P0007X27@1m0Qd0E`0002@000220P00
5P0008d2000F00000P0008825P2=0QH0E`0002@0002=0P00500008d2000F00000P0008d25P2=0Q@0
E`0002@0002<0P004P0008d2000D00000P0008d2502<0Q80E`0002@0002;0P004@0008`2000B0000
0P0008`24P2;0Q40E`0002@000290P00400008/2000A00000P0008/24@290Q00E`0002@000270P00
400008T2000@00000P0008T240270Q00E`0002@000240P00400008L2000A00000P0008L240240Q40
E`0002@000220P004@0008@2000C00000P0008@24@220Q<0E`0002@000210P004`000882000F0000
0P0008824`210QH0E`0002@000210P005P000842000H00000P0008425P210QP0E`0002@000210P00
60000882000K00000P00084260220Q/0E`0002@000220P006`0008@2000M00000P0008826`240Qd0
E`0002@000240P007@0008L2000M00000P0008@27@270Qd0E`0002@000270P007@0008P2000M0000
0P0008L27@280Qd0E`0002@000280P007@0008/2000M00000P0008P27@2;0Qd0E`0002@0002;0P00
6`0008d2000M00000P0008/27@2=0Q/0E`0002@0002<0P004`0008`2000F00000P0008`25P2<0Q<0
E`0002@0002;0P004@0008`2000C00000P0008`24`2;0Q40E`0002@000250P00400008L2000A0000
0P0008L240250Q40E`0002@000230P004@0008D2000C00000P0008D24@230Q<0E`0002@000220P00
4`0008<2000F00000P0008<24`220QH0E`0002@000220P005P000882000H00000P0008825P220QP0
E`0002@000220P00600008<2000K00000P00088260230Q/0E`0002@000230P006`0008D2000M0000
0P0008<26`250Qd0E`0002@000250P007@0008L2000M00000P0008D27@270Qd0E`0002@0002W0P00
2P000:L2000M00000P000:L22P2W0Qd0E`0002@0002X0P002P000:P2000M00000P000:P22P2X0Qd0
E`0002@0002U0P004@000:L2000C00000P000:L24`2U0Q40E`0002@0002S0P0040000:D2000A0000
0P000:D24@2S0Q00E`0002@0002R0P0040000:<2000@00000P000:<2402R0Q00E`0002@0002O0P00
40000:82000A00000P000:82402O0Q40E`0002@0002M0P004@0009l2000C00000P0009l24@2M0Q<0
E`0002@0002L0P004`0009d2000F00000P0009d24`2L0QH0E`0002@0002L0P005P0009`2000H0000
0P0009`25P2L0QP0E`0002@0002L0P00600009d2000K00000P0009`2602M0Q/0E`0002@0002M0P00
6`0009l2000M00000P0009d26`2O0Qd0E`0002@0002O0P007@000:82000M00000P0009l27@2R0Qd0
E`0002@0002R0P007@000:<2000M00000P000:827@2S0Qd0E`0002@0002S0P007@000:D2000M0000
0P000:<27@2U0Qd0E`0002@0002U0P006`000:L2000M00000P000:D27@2W0Q/0E`0002@0002P0P00
40000:82000A00000P000:82402P0Q40E`0002@0002N0P004@000:02000C00000P000:024@2N0Q<0
E`0002@0002M0P004`0009h2000F00000P0009h24`2M0QH0E`0002@0002M0P005P0009d2000H0000
0P0009d25P2M0QP0E`0002@0002M0P00600009h2000K00000P0009d2602N0Q/0E`0002@0002N0P00
6`000:02000M00000P0009h26`2P0Qd0E`0002@0002P0P007@000:82000M00000P000:027@2R0Qd0
E`0002@0002T0P002P000:P2000:00000P000:@22P2X0PX0E`0002@0002W0P007@000:X2000M0000
0P000:L27@2Z0Qd0E`0002@0002`0P002P000;42000;00000P000;422P2`0P/0E`0002@0002`0P00
2`000;42000<00000P000;022`2a0P`0E`0002@0002a0P002`000;82000<00000P000;42302b0P/0
E`0002@0002a0P002P000;82000;00000P000;822`2a0PX0E`0002@0002a0P0040000;42000M0000
0P000;42402a0Qd0E`0002@0002b0P0040000;82000M00000P000;82402b0Qd0E`0002@0002^0P00
40000;82000@00000P000:h2402b0Q00E`0002@0002^0P007@000;D2000M00000P000:h27@2e0Qd0
E`0002@0002k0P0040000;/2000M00000P000;/2402k0Qd0E`0002@0002l0P0040000;`2000M0000
0P000;`2402l0Qd0E`0002@0002l0P004`000;d2000F00000P000;`25P2m0Q<0E`0002@0002m0P00
4@000;l2000C00000P000;d24`2o0Q40E`0002@0002o0P0040000<02000A00000P000;l24@300Q00
E`0002@000300P0040000<<2000@00000P000<0240330Q00E`0002@000330P0040000<@2000A0000
0P000<<240340Q40E`0002@000340P004@000<@2000B00000P000<@24@340Q80E`0002@000330P00
4P000<@2000C00000P000<@24P330Q<0E`0002@000320P004P000<<2000C00000P000<<24`320Q80
E`0002@000320P004@000<<2000B00000P000<824P330Q40E`0002@0002h0P0040000;`2000@0000
0P000;P2402l0Q00E`0002@0002h0P007@000;l2000M00000P000;P27@2o0Qd0E`0002@000390P00
5P000=@2000F00000P000<T25P3D0QH0E`0002@0003D0P0050000=@2000F00000P000=@25P3D0Q@0
E`0002@0003C0P004P000=@2000D00000P000=@2503C0Q80E`0002@0003B0P004@000=<2000B0000
0P000=<24P3B0Q40E`0002@0003@0P0040000=82000A00000P000=824@3@0Q00E`0002@0003>0P00
40000=02000@00000P000=02403>0Q00E`0002@0003;0P0040000<h2000A00000P000<h2403;0Q40
E`0002@000390P004@000</2000C00000P000</24@390Q<0E`0002@000380P004`000<T2000F0000
0P000<T24`380QH0E`0002@000380P005P000<P2000H00000P000<P25P380QP0E`0002@000380P00
60000<T2000K00000P000<P260390Q/0E`0002@000390P006`000</2000M00000P000<T26`3;0Qd0
E`0002@0003;0P007@000<h2000M00000P000</27@3>0Qd0E`0002@0003>0P007@000<l2000M0000
0P000<h27@3?0Qd0E`0002@0003?0P007@000=82000M00000P000<l27@3B0Qd0E`0002@0003B0P00
6`000=@2000M00000P000=827@3D0Q/0E`0002@0003C0P004`000=<2000F00000P000=<25P3C0Q<0
E`0002@0003B0P004@000=<2000C00000P000=<24`3B0Q40E`0002@0003<0P0040000<h2000A0000
0P000<h2403<0Q40E`0002@0003:0P004@000<`2000C00000P000<`24@3:0Q<0E`0002@000390P00
4`000<X2000F00000P000<X24`390QH0E`0002@000390P005P000<T2000H00000P000<T25P390QP0
E`0002@000390P0060000<X2000K00000P000<T2603:0Q/0E`0002@0003:0P006`000<`2000M0000
0P000<X26`3<0Qd0E`0002@0003<0P007@000<h2000M00000P000<`27@3>0Qd0E`0002@0003S0P00
4`000>@2000D00000P000>@24`3S0Q@0E`0002@0003S0P0050000>@2000E00000P000><2503T0QD0
E`0002@0003T0P0050000>D2000E00000P000>@25@3U0Q@0E`0002@0003U0P004`000>D2000D0000
0P000>D2503U0Q<0E`0002@0003S0P004@000>D2000C00000P000>D24`3S0Q40E`0002@0003Q0P00
40000><2000A00000P000><24@3Q0Q00E`0002@0003O0P0040000>42000@00000P000>42403O0Q00
E`0002@0003L0P0040000=l2000A00000P000=l2403L0Q40E`0002@0003J0P004@000=`2000C0000
0P000=`24@3J0Q<0E`0002@0003I0P004`000=X2000F00000P000=X24`3I0QH0E`0002@0003I0P00
5P000=T2000H00000P000=T25P3I0QP0E`0002@0003I0P0060000=X2000K00000P000=T2603J0Q/0
E`0002@0003J0P006`000=`2000M00000P000=X26`3L0Qd0E`0002@0003L0P007@000=l2000M0000
0P000=`27@3O0Qd0E`0002@0003O0P007@000>02000M00000P000=l27@3P0Qd0E`0002@0003P0P00
7@000><2000M00000P000>027@3S0Qd0E`0002@0003S0P006`000>D2000M00000P000><27@3U0Q/0
E`0002@0003M0P0040000=l2000A00000P000=l2403M0Q40E`0002@0003K0P004@000=d2000C0000
0P000=d24@3K0Q<0E`0002@0003J0P004`000=/2000F00000P000=/24`3J0QH0E`0002@0003J0P00
5P000=X2000H00000P000=X25P3J0QP0E`0002@0003J0P0060000=/2000K00000P000=X2603K0Q/0
E`0002@0003K0P006`000=d2000M00000P000=/26`3M0Qd0E`0002@0003M0P007@000=l2000M0000
0P000=d27@3O0Qd0E`0002@0003/0P002P000>`2000J00000P000>`22P3/0QX0E`0002@0003/0P00
6P000>d2000M00000P000>`26P3]0Qd0E`0002@0003]0P007@000>l2000M00000P000>d27@3_0Qd0
E`0002@0003_0P007@000?02000M00000P000>l27@3`0Qd0E`0002@0003`0P007@000?82000M0000
0P000?027@3b0Qd0E`0002@0003b0P006`000?<2000M00000P000?827@3c0Q/0E`0002@0003]0P00
2P000>d2000J00000P000>d22P3]0QX0E`0002@0003]0P006P000>h2000M00000P000>d26P3^0Qd0
E`0002@0003^0P007@000>l2000M00000P000>h27@3_0Qd0E`0002@0003Y0P0040000?02000@0000
0P000>T2403`0Q00E`0002@0003h0P002P000?T2000;00000P000?T22P3h0P/0E`0002@0003h0P00
2`000?T2000<00000P000?P22`3i0P`0E`0002@0003i0P002`000?X2000<00000P000?T2303j0P/0
E`0002@0003i0P002P000?X2000;00000P000?X22`3i0PX0E`0002@0003i0P0040000?T2000M0000
0P000?T2403i0Qd0E`0002@0003j0P0040000?X2000M00000P000?X2403j0Qd0E`0002@0003f0P00
40000?X2000@00000P000?H2403j0Q00E`0002@0003f0P007@000?d2000M00000P000?H27@3m0Qd0
E`0002@000040`00400000L3000A00000P0000L340040a40E`0002@000020`004@0000@3000C0000
0P0000@34@020a<0E`0002@000010`004`000083000F00000P0000834`010aH0E`0002@000010`00
5P000043000H00000P0000435P010aP0E`0002@000010`0060000083000K00000P00004360020a/0
E`0002@000020`006`0000@3000M00000P0000836`040ad0E`0002@000040`007@0000L3000M0000
0P0000@37@070ad0E`0002@000070`007@0000P3000M00000P0000L37@080ad0E`0002@000080`00
7@0000/3000M00000P0000P37@0;0ad0E`0002@0000;0`006`0000d3000M00000P0000/37@0=0a/0
E`0002@0000=0`00600000h3000K00000P0000d36`0>0aP0E`0002@0000>0`005P0000h3000H0000
0P0000h3600>0aH0E`0002@0000=0`004`0000h3000F00000P0000h35P0=0a<0E`0002@0000;0`00
4@0000d3000C00000P0000d34`0;0a40E`0002@000080`00400000/3000A00000P0000/34@080a00
E`0002@000070`00400000P3000@00000P0000P340070a00E`0002@000050`00400000L3000A0000
0P0000L340050a40E`0002@000030`004@0000D3000C00000P0000D34@030a<0E`0002@000020`00
4`0000<3000F00000P0000<34`020aH0E`0002@000020`005P000083000H00000P0000835P020aP0
E`0002@000020`00600000<3000K00000P00008360030a/0E`0002@000030`006`0000D3000M0000
0P0000<36`050ad0E`0002@000050`007@0000L3000M00000P0000D37@070ad0E`0002@000080`00
7@0000X3000M00000P0000P37@0:0ad0E`0002@0000:0`006`0000`3000M00000P0000X37@0<0a/0
E`0002@0000<0`00600000d3000K00000P0000`36`0=0aP0E`0002@0000=0`005P0000d3000H0000
0P0000d3600=0aH0E`0002@0000<0`004`0000d3000F00000P0000d35P0<0a<0E`0002@0000:0`00
4@0000`3000C00000P0000`34`0:0a40E`0002@000080`00400000X3000A00000P0000X34@080a00
E`0002@0000E0`00400001D3000M00000P0001D3400E0ad0E`0002@0000F0`00400001H3000M0000
0P0001H3400F0ad0E`0002@0000F0`004@0001P3000C00000P0001H34`0H0a40E`0002@0000H0`00
400001X3000A00000P0001P34@0J0a00E`0002@0000J0`00400001`3000@00000P0001X3400L0a00
E`0002@0000L0`00400001l3000A00000P0001`3400O0a40E`0002@0000O0`004@000203000C0000
0P0001l34@0P0a<0E`0002@0000P0`004`000203000M00000P0002034`0P0ad0E`0002@0000L0`00
400001h3000A00000P0001`3400N0a40E`0002@0000N0`004@0001l3000C00000P0001h34@0O0a<0
E`0002@0000O0`004`0001l3000M00000P0001l34`0O0ad0E`0002@0000B0`00400001H3000@0000
0P000183400F0a00E`0002@0000B0`007@0001T3000M00000P0001837@0I0ad0E`0002@0000L0`00
7@000283000M00000P0001`37@0R0ad0E`0002@0000_0`0040000303000B00000P0002l34P0`0a00
E`0002@0000`0`0040000303000D00000P000303400`0a@0E`0002@0000_0`004P000303000D0000
0P000303500_0a80E`0002@0000^0`004@0002l3000B00000P0002l34P0^0a40E`0002@0000/0`00
400002h3000A00000P0002h34@0/0a00E`0002@0000Y0`00400002`3000@00000P0002`3400Y0a00
E`0002@0000W0`00400002T3000A00000P0002T3400W0a40E`0002@0000V0`004@0002L3000B0000
0P0002L34@0V0a80E`0002@0000V0`004P0002H3000D00000P0002H34P0V0a@0E`0002@0000V0`00
500002L3000E00000P0002H3500W0aD0E`0002@0000W0`005@0002T3000F00000P0002L35@0Y0aH0
E`0002@0000Y0`005P0002d3000H00000P0002T35P0]0aP0E`0002@0000]0`00600002l3000I0000
0P0002d3600_0aT0E`0002@0000_0`006@000303000J00000P0002l36@0`0aX0E`0002@0000V0`00
4`0002L3000D00000P0002H34`0W0a@0E`0002@0000W0`00500002T3000E00000P0002L3500Y0aD0
E`0002@0000Y0`005@0002d3000G00000P0002T35@0]0aL0E`0002@0000]0`005`0002l3000H0000
0P0002d35`0_0aP0E`0002@0000_0`0060000303000I00000P0002l3600`0aT0E`0002@0000`0`00
6@000303000L00000P0003036@0`0a`0E`0002@0000_0`0070000303000M00000P000303700_0ad0
E`0002@0000]0`007@0002l3000M00000P0002l37@0]0ad0E`0002@0000Z0`007@0002d3000M0000
0P0002d37@0Z0ad0E`0002@0000X0`007@0002X3000M00000P0002X37@0X0ad0E`0002@0000W0`00
700002P3000M00000P0002P37@0W0a`0E`0002@0000V0`006P0002L3000L00000P0002L3700V0aX0
E`0002@0000V0`006P0002H3000M00000P0002H36P0V0ad0E`0002@0000V0`00700002L3000M0000
0P0002H37@0W0a`0E`0002@000100`00400004@3000M00000P00040340140ad0E`0002@000110`00
400004@3000K00000P00044340140a/0E`0002@000140`00400004L3000M00000P0004L340140ad0
E`0002@000170`00400004/3000M00000P0004L3401;0ad0E`0002@000180`00400004/3000K0000
0P0004P3401;0a/0E`0002@0001;0`00400004h3000M00000P0004h3401;0ad0E`0002@0000m0`00
400004@3000@00000P0003d340140a00E`0002@0001<0`0040000543000@00000P0004`3401A0a00
E`0002@0001F0`002P0005L3000;00000P0005L32P1F0`/0E`0002@0001F0`002`0005L3000<0000
0P0005H32`1G0``0E`0002@0001G0`002`0005P3000<00000P0005L3301H0`/0E`0002@0001G0`00
2P0005P3000;00000P0005P32`1G0`X0E`0002@0001G0`00400005L3000M00000P0005L3401G0ad0
E`0002@0001H0`00400005P3000M00000P0005P3401H0ad0E`0002@0001D0`00400005P3000@0000
0P0005@3401H0a00E`0002@0001D0`007@0005/3000M00000P0005@37@1K0ad0E`0002@0001Q0`00
2P000643000J00000P0006432P1Q0aX0E`0002@0001Q0`006P000683000M00000P0006436P1R0ad0
E`0002@0001R0`007@0006@3000M00000P0006837@1T0ad0E`0002@0001T0`007@0006D3000M0000
0P0006@37@1U0ad0E`0002@0001U0`007@0006L3000M00000P0006D37@1W0ad0E`0002@0001W0`00
6`0006P3000M00000P0006L37@1X0a/0E`0002@0001R0`002P000683000J00000P0006832P1R0aX0
E`0002@0001R0`006P0006<3000M00000P0006836P1S0ad0E`0002@0001S0`007@0006@3000M0000
0P0006<37@1T0ad0E`0002@0001N0`00400006D3000@00000P0005h3401U0a00E`0002@0001^0`00
2P0006h3000M00000P0006h32P1^0ad0E`0002@0001_0`002P0006l3000M00000P0006l32P1_0ad0
E`0002@0001_0`004@000743000C00000P0006l34`1a0a40E`0002@0001a0`00400007<3000A0000
0P0007434@1c0a00E`0002@0001c0`00400007D3000@00000P0007<3401e0a00E`0002@0001e0`00
400007P3000A00000P0007D3401h0a40E`0002@0001h0`004@0007T3000C00000P0007P34@1i0a<0
E`0002@0001i0`004`0007T3000M00000P0007T34`1i0ad0E`0002@0001e0`00400007L3000A0000
0P0007D3401g0a40E`0002@0001g0`004@0007P3000C00000P0007L34@1h0a<0E`0002@0001h0`00
4`0007P3000M00000P0007P34`1h0ad0E`0002@0001[0`002P0006l3000:00000P0006/32P1_0`X0
E`0002@0001[0`007@000783000M00000P0006/37@1b0ad0E`0002@0001e0`007@0007/3000M0000
0P0007D37@1k0ad0E`0002@0002D0`004`0009D3000D00000P0009D34`2D0a@0E`0002@0002D0`00
500009D3000E00000P0009@3502E0aD0E`0002@0002E0`00500009H3000E00000P0009D35@2F0a@0
E`0002@0002F0`004`0009H3000D00000P0009H3502F0a<0E`0002@0002D0`004@0009H3000C0000
0P0009H34`2D0a40E`0002@0002B0`00400009@3000A00000P0009@34@2B0a00E`0002@0002@0`00
40000983000@00000P000983402@0a00E`0002@0002=0`0040000903000A00000P000903402=0a40
E`0002@0002;0`004@0008d3000C00000P0008d34@2;0a<0E`0002@0002:0`004`0008/3000F0000
0P0008/34`2:0aH0E`0002@0002:0`005P0008X3000H00000P0008X35P2:0aP0E`0002@0002:0`00
600008/3000K00000P0008X3602;0a/0E`0002@0002;0`006`0008d3000M00000P0008/36`2=0ad0
E`0002@0002=0`007@000903000M00000P0008d37@2@0ad0E`0002@0002@0`007@000943000M0000
0P0009037@2A0ad0E`0002@0002A0`007@0009@3000M00000P0009437@2D0ad0E`0002@0002D0`00
6`0009H3000M00000P0009@37@2F0a/0E`0002@0002>0`0040000903000A00000P000903402>0a40
E`0002@0002<0`004@0008h3000C00000P0008h34@2<0a<0E`0002@0002;0`004`0008`3000F0000
0P0008`34`2;0aH0E`0002@0002;0`005P0008/3000H00000P0008/35P2;0aP0E`0002@0002;0`00
600008`3000K00000P0008/3602<0a/0E`0002@0002<0`006`0008h3000M00000P0008`36`2>0ad0
E`0002@0002>0`007@000903000M00000P0008h37@2@0ad0E`0002@0002L0`004P000:`3000B0000
0P0009`34P2/0a80E`0002@0002L0`0060000:`3000H00000P0009`3602/0aP0E`0002@0002e0`00
2P000;P3000;00000P000;P32P2e0`/0E`0002@0002c0`002`000;D3000>00000P000;D32`2c0`h0
E`0002@0002b0`003P000;<3000B00000P000;<33P2b0a80E`0002@0002b0`004P000;83000E0000
0P000;834P2b0aD0E`0002@0002b0`005@000;<3000J00000P000;835@2c0aX0E`0002@0002c0`00
6P000;D3000M00000P000;<36P2e0ad0E`0002@0002e0`007@000;P3000M00000P000;D37@2h0ad0
E`0002@0002h0`007@000;T3000M00000P000;P37@2i0ad0E`0002@0002i0`007@000;`3000M0000
0P000;T37@2l0ad0E`0002@0002l0`006P000;h3000M00000P000;`37@2n0aX0E`0002@0002n0`00
5@000;l3000J00000P000;h36P2o0aD0E`0002@0002o0`004P000;l3000E00000P000;l35@2o0a80
E`0002@0002n0`003P000;l3000B00000P000;l34P2n0`h0E`0002@0002l0`002`000;h3000>0000
0P000;h33P2l0`/0E`0002@0002i0`002P000;`3000;00000P000;`32`2i0`X0E`0002@0002h0`00
2P000;T3000:00000P000;T32P2h0`X0E`0002@0002f0`002P000;P3000;00000P000;P32P2f0`/0
E`0002@0002e0`002`000;H3000<00000P000;H32`2e0``0E`0002@0002d0`0030000;D3000>0000
0P000;D3302d0`h0E`0002@0002c0`003P000;@3000B00000P000;@33P2c0a80E`0002@0002c0`00
4P000;<3000E00000P000;<34P2c0aD0E`0002@0002c0`005@000;@3000J00000P000;<35@2d0aX0
E`0002@0002d0`006P000;D3000L00000P000;@36P2e0a`0E`0002@0002e0`0070000;H3000M0000
0P000;D3702f0ad0E`0002@0002f0`007@000;P3000M00000P000;H37@2h0ad0E`0002@0002i0`00
7@000;/3000M00000P000;T37@2k0ad0E`0002@0002k0`0070000;`3000M00000P000;/37@2l0a`0
E`0002@0002l0`006P000;d3000L00000P000;`3702m0aX0E`0002@0002m0`005@000;h3000J0000
0P000;d36P2n0aD0E`0002@0002n0`004P000;h3000E00000P000;h35@2n0a80E`0002@0002m0`00
3P000;h3000B00000P000;h34P2m0`h0E`0002@0002l0`0030000;d3000>00000P000;d33P2l0``0
E`0002@0002k0`002`000;`3000<00000P000;`3302k0`/0E`0002@0002i0`002P000;/3000;0000
0P000;/32`2i0`X0E`0002@000350`0070000<H3000M00000P000<H370350ad0E`0002@000350`00
7@000<H3000M00000P000<D37@360ad0E`0002@000360`007@000<L3000M00000P000<H37@370ad0
E`0002@000360`0070000<L3000M00000P000<L37@360a`0E`0002@0003@0`002P000=<3000;0000
0P000=<32P3@0`/0E`0002@0003>0`002`000=03000>00000P000=032`3>0`h0E`0002@0003=0`00
3P000<h3000B00000P000<h33P3=0a80E`0002@0003=0`004P000<d3000E00000P000<d34P3=0aD0
E`0002@0003=0`005@000<h3000J00000P000<d35@3>0aX0E`0002@0003>0`006P000=03000M0000
0P000<h36P3@0ad0E`0002@0003@0`007@000=<3000M00000P000=037@3C0ad0E`0002@0003C0`00
7@000=@3000M00000P000=<37@3D0ad0E`0002@0003D0`007@000=L3000M00000P000=@37@3G0ad0
E`0002@0003G0`006P000=T3000M00000P000=L37@3I0aX0E`0002@0003I0`005@000=X3000J0000
0P000=T36P3J0aD0E`0002@0003J0`004P000=X3000E00000P000=X35@3J0a80E`0002@0003I0`00
3P000=X3000B00000P000=X34P3I0`h0E`0002@0003G0`002`000=T3000>00000P000=T33P3G0`/0
E`0002@0003D0`002P000=L3000;00000P000=L32`3D0`X0E`0002@0003C0`002P000=@3000:0000
0P000=@32P3C0`X0E`0002@0003A0`002P000=<3000;00000P000=<32P3A0`/0E`0002@0003@0`00
2`000=43000<00000P000=432`3@0``0E`0002@0003?0`0030000=03000>00000P000=03303?0`h0
E`0002@0003>0`003P000<l3000B00000P000<l33P3>0a80E`0002@0003>0`004P000<h3000E0000
0P000<h34P3>0aD0E`0002@0003>0`005@000<l3000J00000P000<h35@3?0aX0E`0002@0003?0`00
6P000=03000L00000P000<l36P3@0a`0E`0002@0003@0`0070000=43000M00000P000=03703A0ad0
E`0002@0003A0`007@000=<3000M00000P000=437@3C0ad0E`0002@0003D0`007@000=H3000M0000
0P000=@37@3F0ad0E`0002@0003F0`0070000=L3000M00000P000=H37@3G0a`0E`0002@0003G0`00
6P000=P3000L00000P000=L3703H0aX0E`0002@0003H0`005@000=T3000J00000P000=P36P3I0aD0
E`0002@0003I0`004P000=T3000E00000P000=T35@3I0a80E`0002@0003H0`003P000=T3000B0000
0P000=T34P3H0`h0E`0002@0003G0`0030000=P3000>00000P000=P33P3G0``0E`0002@0003F0`00
2`000=L3000<00000P000=L3303F0`/0E`0002@0003D0`002P000=H3000;00000P000=H32`3D0`X0
E`0002@0003Q0`002P000>@3000;00000P000>@32P3Q0`/0E`0002@0003O0`002`000>43000>0000
0P000>432`3O0`h0E`0002@0003N0`003P000=l3000B00000P000=l33P3N0a80E`0002@0003N0`00
4P000=h3000E00000P000=h34P3N0aD0E`0002@0003N0`005@000=l3000J00000P000=h35@3O0aX0
E`0002@0003O0`006P000>43000M00000P000=l36P3Q0ad0E`0002@0003Q0`007@000>@3000M0000
0P000>437@3T0ad0E`0002@0003T0`007@000>D3000M00000P000>@37@3U0ad0E`0002@0003U0`00
7@000>P3000M00000P000>D37@3X0ad0E`0002@0003X0`006P000>X3000M00000P000>P37@3Z0aX0
E`0002@0003Z0`005@000>/3000J00000P000>X36P3[0aD0E`0002@0003[0`004P000>/3000E0000
0P000>/35@3[0a80E`0002@0003Z0`003P000>/3000B00000P000>/34P3Z0`h0E`0002@0003X0`00
2`000>X3000>00000P000>X33P3X0`/0E`0002@0003U0`002P000>P3000;00000P000>P32`3U0`X0
E`0002@0003T0`002P000>D3000:00000P000>D32P3T0`X0E`0002@0003R0`002P000>@3000;0000
0P000>@32P3R0`/0E`0002@0003Q0`002`000>83000<00000P000>832`3Q0``0E`0002@0003P0`00
30000>43000>00000P000>43303P0`h0E`0002@0003O0`003P000>03000B00000P000>033P3O0a80
E`0002@0003O0`004P000=l3000E00000P000=l34P3O0aD0E`0002@0003O0`005@000>03000J0000
0P000=l35@3P0aX0E`0002@0003P0`006P000>43000L00000P000>036P3Q0a`0E`0002@0003Q0`00
70000>83000M00000P000>43703R0ad0E`0002@0003R0`007@000>@3000M00000P000>837@3T0ad0
E`0002@0003U0`007@000>L3000M00000P000>D37@3W0ad0E`0002@0003W0`0070000>P3000M0000
0P000>L37@3X0a`0E`0002@0003X0`006P000>T3000L00000P000>P3703Y0aX0E`0002@0003Y0`00
5@000>X3000J00000P000>T36P3Z0aD0E`0002@0003Z0`004P000>X3000E00000P000>X35@3Z0a80
E`0002@0003Y0`003P000>X3000B00000P000>X34P3Y0`h0E`0002@0003X0`0030000>T3000>0000
0P000>T33P3X0``0E`0002@0003W0`002`000>P3000<00000P000>P3303W0`/0E`0002@0003U0`00
2P000>L3000;00000P000>L32`3U0`X0E`0002@0003`0`002P000?83000C00000P000?832P3`0a<0
E`0002@0003`0`004@000?83000C00000P000?034`3b0a40E`0002@0003b0`0040000?D3000A0000
0P000?834@3e0a00E`0002@0003e0`0040000?L3000@00000P000?D3403g0a00E`0002@0003g0`00
40000?X3000A00000P000?L3403j0a40E`0002@0003j0`004@000?`3000C00000P000?X34@3l0a<0
E`0002@0003l0`004`000?d3000F00000P000?`34`3m0aH0E`0002@0003m0`005P000?d3000H0000
0P000?d35P3m0aP0E`0002@0003l0`0060000?d3000K00000P000?d3603l0a/0E`0002@0003j0`00
6`000?`3000M00000P000?`36`3j0ad0E`0002@0003g0`007@000?X3000M00000P000?X37@3g0ad0
E`0002@0003e0`007@000?L3000M00000P000?L37@3e0ad0E`0002@0003b0`007@000?D3000M0000
0P000?D37@3b0ad0E`0002@0003a0`0070000?83000M00000P000?837@3a0a`0E`0002@0003`0`00
6P000?43000L00000P000?43703`0aX0E`0002@0003`0`006@000?03000J00000P000?036P3`0aT0
E`0002@0003`0`0060000?43000I00000P000?036@3a0aP0E`0002@0003a0`0060000?83000I0000
0P000?43603b0aT0E`0002@0003a0`006@000?83000J00000P000?836@3a0aX0E`0002@0003g0`00
40000?T3000A00000P000?L3403i0a40E`0002@0003i0`004@000?/3000C00000P000?T34@3k0a<0
E`0002@0003k0`004`000?`3000F00000P000?/34`3l0aH0E`0002@0003l0`005P000?`3000H0000
0P000?`35P3l0aP0E`0002@0003k0`0060000?`3000K00000P000?`3603k0a/0E`0002@0003i0`00
6`000?/3000M00000P000?/36`3i0ad0E`0002@0003g0`007@000?T3000M00000P000?T37@3g0ad0
E`0002@0003b0`002P000?/3000:00000P000?832P3k0`X0E`0002@0003b0`002`000?H3000;0000
0P000?832`3f0`/0E`0002@0003f0`002P000?/3000;00000P000?H32`3k0`X0E`0002@0003:0000
O0<00<l0001l0`000P000<l0O0?:07`3E`0002@0003:0000iP400<l0003V0@000P000<l0iP7:0>H1
E`0002@0003:0000D0000<l0001@00000P000<l0D03:0500E`0002@0000@00005@8002<0000E0P00
0P0001005@8S01D2E`0002@0000@0000508002<0000D0P000P000100508S01@2E`0002@0000F0000
3`8001d0000?0P000P0001H03`8M00l2E`0002@0000@00002@800100000H0P000P000100608@00T2
E`0002@0000@00002@8001H000090P000P0001002@8F00T2E`0002@0000@00002@8001H0000:0P00
0P0001H02@8@00X2E`0002@0000I00003`8001T0000D0P000P0001T0508I00l2E`0002@0000S0000
2@8002<0000H0P000P0002<0608S00T2E`0002@0000N00002@8002<000090P000P0002<02@8N00T2
E`0002@0000N00002@8002<0000:0P000P0001h02@8S00X2E`0002@0000=0000o`4000d000020P00
0P0000d0o`4=0082E`0002@0000=00000P8002T000020P000P0000d00P8Y0082E`0002@0000Y0000
o`4002T000020P000P0002T00P8Y0?l1E`0002@0000=00000@8002T000010P000P0000d00@8Y0042
E`0002@0000G0000k04002T0003/0@000P0001L0k04Y0>`1E`0002@0000G0000j`4002T0003[0@00
0P0001L0j`4Y0>/1E`0002@0000G0000k04001T0003^0@000P0001T0k04G0>h1E`0002@0000G0000
kP4001L0003`0@000P0001L0kP4G0?01E`0002@0000G0000l04001L0003b0@000P0001L0l04G0?81
E`0002@0000G0000lP4001L0003d0@000P0001L0lP4G0?@1E`0002@0000G0000m04001T0003f0@00
0P0001L0m04I0?H1E`0002@0000I0000mP4001`0003g0@000P0001T0mP4L0?L1E`0002@0000L0000
m`4001h0003g0@000P0001`0m`4N0?L1E`0002@0000N0000mP400200003g0@000P0001h0m`4P0?H1
E`0002@0000P0000m0400280003f0@000P000200mP4R0?@1E`0002@0000R0000lP4002<0003d0@00
0P000280m04S0?81E`0002@0000S0000l04002<0003b0@000P0002<0lP4S0?01E`0002@0000R0000
kP4002<0003`0@000P0002<0l04R0>h1E`0002@0000P0000k0400280003^0@000P000280kP4P0>`1
E`0002@0000G0000lP4001L0003c0@000P0001L0lP4G0?<1E`0002@0000G0000l`4001T0003e0@00
0P0001L0l`4I0?D1E`0002@0000I0000m@4001`0003f0@000P0001T0m@4L0?H1E`0002@0000L0000
mP4001h0003f0@000P0001`0mP4N0?H1E`0002@0000N0000m@400200003f0@000P0001h0mP4P0?D1
E`0002@0000P0000l`400280003e0@000P000200m@4R0?<1E`0002@0000R0000lP4002<0003c0@00
0P000280l`4S0?81E`0002@0000Y0000j04002T0003_0@000P0002T0k`4Y0>P1E`0002@0000=0000
g@4000h0003O0@000P0000d0g@4>0=l1E`0002@0000>0000g`400140003P0@000P0000h0g`4A0>01
E`0002@0000A0000h04001D0003R0@000P000140h04E0>81E`0002@0000E0000hP4001T0003S0@00
0P0001D0hP4I0><1E`0002@0000I0000h`4001d0003S0@000P0001T0h`4M0><1E`0002@0000M0000
hP400240003S0@000P0001d0h`4Q0>81E`0002@0000Q0000h04002D0003R0@000P000240hP4U0>01
E`0002@0000U0000g`4002P0003P0@000P0002D0h04X0=l1E`0002@0000X0000g@4002T0003O0@00
0P0002P0g`4Y0=d1E`0002@0000>0000g`400180003P0@000P0000h0g`4B0>01E`0002@0000B0000
h04001D0003Q0@000P000180h04E0>41E`0002@0000E0000h@4001T0003R0@000P0001D0h@4I0>81
E`0002@0000I0000hP4001d0003R0@000P0001T0hP4M0>81E`0002@0000M0000h@400240003R0@00
0P0001d0hP4Q0>41E`0002@0000Q0000h04002@0003Q0@000P000240h@4T0>01E`0002@0000T0000
g`4002P0003P0@000P0002@0h04X0=l1E`0002@0000@0000e@4001l0003E0@000P000100e@4O0=D1
E`0002@0000O0000e0400280003E0@000P0001l0e@4R0=@1E`0002@0000R0000d`4002<0003D0@00
0P000280e04S0=<1E`0002@0000S0000d@4002<0003C0@000P0002<0d`4S0=41E`0002@0000R0000
c`4002<0003A0@000P0002<0d@4R0<l1E`0002@0000P0000cP400280003?0@000P000280c`4P0<h1
E`0002@0000@0000e04001l0003D0@000P000100e04O0=@1E`0002@0000O0000d`400280003D0@00
0P0001l0e04R0=<1E`0002@0000R0000d`4002<0003C0@000P000280d`4S0=<1E`0002@0000G0000
d@4001L0003H0@000P0001L0f04G0=41E`0002@0000=0000a`4000h000390@000P0000d0b@4>0<L1
E`0002@0000>0000a@40014000370@000P0000h0a`4A0<D1E`0002@0000A0000a04001D000350@00
0P000140a@4E0<@1E`0002@0000E0000``4001T000340@000P0001D0a04I0<<1E`0002@0000I0000
``4001d000330@000P0001T0``4M0<<1E`0002@0000M0000``40024000340@000P0001d0``4Q0<@1
E`0002@0000Q0000a04002D000350@000P000240a04U0<D1E`0002@0000U0000a@4002P000370@00
0P0002D0a@4X0<L1E`0002@0000X0000a`4002T000390@000P0002P0a`4Y0<T1E`0002@0000>0000
a@40018000370@000P0000h0a`4B0<D1E`0002@0000B0000a@4001D000350@000P000180a@4E0<D1
E`0002@0000E0000a04001T000350@000P0001D0a@4I0<@1E`0002@0000I0000a04001d000340@00
0P0001T0a04M0<@1E`0002@0000M0000a040024000350@000P0001d0a04Q0<D1E`0002@0000Q0000
a@4002@000350@000P000240a@4T0<D1E`0002@0000T0000a@4002P000370@000P0002@0a@4X0<L1
E`0002@0000=0000]`4000d0002j0@000P0000d0^P4=0;L1E`0002@0000=0000]`4002T0002g0@00
0P0000d0]`4Y0;L1E`0002@0000Y0000]`4002T0002j0@000P0002T0]`4Y0;X1E`0002@0000=0000
^04002T0002h0@000P0000d0^04Y0;P1E`0002@000160000OP<005H0001n0`000P0004H0OP=F07h3
E`0002@0001O0000MP<00640001g0`000P0005l0M`=Q07H3E`0002@0001Q0000L`<006<0001f0`00
0P000640MP=S07<3E`0002@0001S0000L`<006<000260`000P0006<0L`=S08H3E`0002@0001R0000
M0<0068000260`000P000680M0=R08H3E`0002@0001O0000QP<006L000260`000P0005l0QP=W08H3
E`0002@0001n0000K`<007l0001a0`000P0007l0K`=n0743E`0002@0001l0000L@<007h0001d0`00
0P0007h0L@=l07@3E`0002@0001j0000M0<007`0001h0`000P0007`0M0=j07P3E`0002@0001i0000
N0<007X0001l0`000P0007X0N0=i07`3E`0002@0001i0000O0<007T000200`000P0007T0O0=i0803
E`0002@0001i0000P0<007X000250`000P0007T0P0=j08D3E`0002@0001j0000Q@<007`000280`00
0P0007X0Q@=l08P3E`0002@0001l0000R0<007h0002;0`000P0007`0R0=n08/3E`0002@0001n0000
R`<007l0002=0`000P0007h0R`=o08d3E`0002@0001l0000L@<007h0001e0`000P0007h0L@=l07D3
E`0002@0001k0000M@<007`0001h0`000P0007`0M@=k07P3E`0002@0001j0000N0<007/0001l0`00
0P0007/0N0=j07`3E`0002@0001j0000O0<007X000200`000P0007X0O0=j0803E`0002@0001j0000
P0<007/000250`000P0007X0P0=k08D3E`0002@0001k0000Q@<007`000270`000P0007/0Q@=l08L3
E`0002@0001l0000Q`<007h0002;0`000P0007`0Q`=n08/3E`0002@0002A0000L`<00980001f0`00
0P000940MP>B07<3E`0002@0002B0000L`<00980001h0`000P000980L`>B07P3E`0002@0002A0000
MP<00980001h0`000P000980N0>A07H3E`0002@0002?0000M0<00940001f0`000P000940MP>?07@3
E`0002@0002<0000L`<008l0001d0`000P0008l0M0><07<3E`0002@0002:0000L`<008`0001c0`00
0P0008`0L`>:07<3E`0002@000270000L`<008X0001d0`000P0008X0L`>707@3E`0002@000250000
M0<008L0001f0`000P0008L0M0>507H3E`0002@000250000MP<008D0001h0`000P0008D0MP>507P3
E`0002@000250000N0<008H0001i0`000P0008D0N0>607T3E`0002@000260000N@<008L0001j0`00
0P0008H0N@>707X3E`0002@000270000NP<008T0001k0`000P0008L0NP>907/3E`0002@000290000
N`<008h0001m0`000P0008T0N`>>07d3E`0002@0002>0000O@<00900001n0`000P0008h0O@>@07h3
E`0002@0002@0000OP<0098000200`000P000900OP>B0803E`0002@000250000N0<008L0001i0`00
0P0008D0N0>707T3E`0002@000270000N@<008T0001j0`000P0008L0N@>907X3E`0002@000290000
NP<008h0001l0`000P0008T0NP>>07`3E`0002@0002>0000O0<00900001m0`000P0008h0O0>@07d3
E`0002@0002@0000O@<00940001n0`000P000900O@>A07h3E`0002@0002A0000OP<0098000200`00
0P000940OP>B0803E`0002@0002B0000P0<0098000240`000P000980P0>B08@3E`0002@0002@0000
Q0<0098000260`000P000980Q0>@08H3E`0002@0002=0000QP<0090000260`000P000900QP>=08H3
E`0002@0002;0000QP<008d000260`000P0008d0QP>;08H3E`0002@000280000QP<008/000260`00
0P0008/0QP>808H3E`0002@000260000Q0<008P000260`000P0008P0QP>608@3E`0002@000250000
P@<008H000240`000P0008H0Q0>50843E`0002@000250000P@<008D000260`000P0008D0P@>508H3
E`0002@000250000Q0<008H000260`000P0008D0QP>608@3E`0002@0002H0000O`<00:<0001o0`00
0P0009P0O`>S07l3E`0002@0002S0000O@<00:<0001o0`000P000:<0O`>S07d3E`0002@0002R0000
N`<00:<0001m0`000P000:<0O@>R07/3E`0002@0002Q0000NP<00:80001k0`000P000:80N`>Q07X3
E`0002@0002O0000N@<00:40001j0`000P000:40NP>O07T3E`0002@0002M0000N@<009l0001i0`00
0P0009l0N@>M07T3E`0002@0002J0000N@<009d0001j0`000P0009d0N@>J07X3E`0002@0002H0000
NP<009X0001l0`000P0009X0NP>H07`3E`0002@0002G0000O0<009P0001o0`000P0009P0O0>G07l3
E`0002@0002G0000O`<009L000210`000P0009L0O`>G0843E`0002@0002G0000P@<009P000240`00
0P0009L0P@>H08@3E`0002@0002H0000Q0<009X000260`000P0009P0Q0>J08H3E`0002@0002J0000
QP<009d000260`000P0009X0QP>M08H3E`0002@0002M0000QP<009h000260`000P0009d0QP>N08H3
E`0002@0002N0000QP<00:4000260`000P0009h0QP>Q08H3E`0002@0002Q0000Q0<00:<000260`00
0P000:40QP>S08@3E`0002@0002R0000O0<00:80001o0`000P000:80O`>R07`3E`0002@0002Q0000
NP<00:80001l0`000P000:80O0>Q07X3E`0002@0002K0000N@<009d0001j0`000P0009d0N@>K07X3
E`0002@0002I0000NP<009/0001l0`000P0009/0NP>I07`3E`0002@0002H0000O0<009T0001o0`00
0P0009T0O0>H07l3E`0002@0002H0000O`<009P000210`000P0009P0O`>H0843E`0002@0002H0000
P@<009T000240`000P0009P0P@>I08@3E`0002@0002I0000Q0<009/000260`000P0009T0Q0>K08H3
E`0002@0002K0000QP<009d000260`000P0009/0QP>M08H3E`0002@0002Y0000L`<00:T000260`00
0P000:T0L`>Y08H3E`0002@0002Z0000L`<00:X000260`000P000:X0L`>Z08H3E`0002@0002V0000
L`<00:X0001c0`000P000:H0L`>Z07<3E`0002@0002V0000QP<00:d000260`000P000:H0QP>]08H3
E`0002@0002c0000L`<00;<000260`000P000;<0L`>c08H3E`0002@0002d0000L`<00;@000260`00
0P000;@0L`>d08H3E`0002@0002`0000L`<00;@0001c0`000P000;00L`>d07<3E`0002@0002`0000
QP<00;L000260`000P000;00QP>g08H3E`0002@0002k0000K`<00;d0001a0`000P000;/0K`>m0743
E`0002@0002m0000L@<00;l0001d0`000P000;d0L@>o07@3E`0002@0002o0000M0<00<40001h0`00
0P000;l0M0?107P3E`0002@000310000N0<00<40001l0`000P000<40N0?107`3E`0002@000310000
O0<00<4000200`000P000<40O0?10803E`0002@000310000P0<00<4000250`000P000<40P0?108D3
E`0002@0002o0000Q@<00<4000280`000P000<40Q@>o08P3E`0002@0002m0000R0<00;l0002;0`00
0P000;l0R0>m08/3E`0002@0002k0000R`<00;d0002=0`000P000;d0R`>k08d3E`0002@0002m0000
L@<00;l0001e0`000P000;d0L@>o07D3E`0002@0002o0000M@<00<00001h0`000P000;l0M@?007P3
E`0002@000300000N0<00<40001l0`000P000<00N0?107`3E`0002@000310000O0<00<4000200`00
0P000<40O0?10803E`0002@000300000P0<00<4000250`000P000<40P0?008D3E`0002@0002o0000
Q@<00<0000270`000P000<00Q@>o08L3E`0002@0002m0000Q`<00;l0002;0`000P000;l0Q`>m08/3
E`0002@0001<0000B@0004`0001I00000P0004`0B@1<05T0E`0002@000140000D@0005@0001A0000
0P0004@0D@1D0540E`0002@0001N0000B@000600001:00000P0005h0BP1P04T0E`0002@0001P0000
AP000680001900000P000600B@1R04H0E`0002@0001R0000AP000680001I00000P000680AP1R05T0
E`0002@0001Q0000A`000640001I00000P000640A`1Q05T0E`0002@0001N0000F@0006H0001I0000
0P0005h0F@1V05T0E`0002@0001l0000@P0007d0001400000P0007d0@P1l04@0E`0002@0001j0000
A00007`0001700000P0007`0A01j04L0E`0002@0001h0000A`0007X0001;00000P0007X0A`1h04/0
E`0002@0001g0000B`0007P0001?00000P0007P0B`1g04l0E`0002@0001g0000C`0007L0001C0000
0P0007L0C`1g05<0E`0002@0001g0000D`0007P0001H00000P0007L0D`1h05P0E`0002@0001h0000
F00007X0001K00000P0007P0F01j05/0E`0002@0001j0000F`0007`0001N00000P0007X0F`1l05h0
E`0002@0001l0000GP0007d0001P00000P0007`0GP1m0600E`0002@0001j0000A00007`000180000
0P0007`0A01j04P0E`0002@0001i0000B00007X0001;00000P0007X0B01i04/0E`0002@0001h0000
B`0007T0001?00000P0007T0B`1h04l0E`0002@0001h0000C`0007P0001C00000P0007P0C`1h05<0
E`0002@0001h0000D`0007T0001H00000P0007P0D`1i05P0E`0002@0001i0000F00007X0001J0000
0P0007T0F01j05X0E`0002@0001j0000FP0007`0001N00000P0007X0FP1l05h0E`0002@000250000
AP0008D0001I00000P0008D0AP2505T0E`0002@000260000AP0008H0001I00000P0008H0AP2605T0
E`0002@000220000AP0008d0001600000P000880AP2=04H0E`0002@0002=0000AP00090000170000
0P0008d0AP2@04L0E`0002@0002@0000A`000940001800000P000900A`2A04P0E`0002@0002A0000
B0000940001:00000P000940B02A04X0E`0002@0002A0000BP000940001;00000P000940BP2A04/0
E`0002@0002A0000B`000940001=00000P000940B`2A04d0E`0002@0002@0000C@000940001>0000
0P000940C@2@04h0E`0002@0002=0000CP000900001?00000P000900CP2=04l0E`0002@0002=0000
AP0008l0001700000P0008d0AP2?04L0E`0002@0002?0000A`000900001800000P0008l0A`2@04P0
E`0002@0002@0000B0000940001:00000P000900B02A04X0E`0002@0002A0000BP000940001;0000
0P000940BP2A04/0E`0002@0002@0000B`000940001=00000P000940B`2@04d0E`0002@0002?0000
C@000900001>00000P000900C@2?04h0E`0002@0002=0000CP0008l0001?00000P0008l0CP2=04l0
E`0002@000260000C`0008d0001?00000P0008H0C`2=04l0E`0002@0002=0000C`000900001@0000
0P0008d0C`2@0500E`0002@0002@0000D0000940001A00000P000900D02A0540E`0002@0002A0000
D@000940001C00000P000940D@2A05<0E`0002@0002A0000D`000940001F00000P000940D`2A05H0
E`0002@0002A0000EP000940001H00000P000940EP2A05P0E`0002@0002@0000F0000940001I0000
0P000940F02@05T0E`0002@0002=0000F@000900001I00000P000900F@2=05T0E`0002@000220000
F@0008d0001I00000P0008d0F@2205T0E`0002@0002=0000C`0008l0001@00000P0008d0C`2?0500
E`0002@0002?0000D0000900001A00000P0008l0D02@0540E`0002@0002@0000D@000940001C0000
0P000900D@2A05<0E`0002@0002A0000D`000940001F00000P000940D`2A05H0E`0002@0002@0000
EP000940001H00000P000940EP2@05P0E`0002@0002?0000F0000900001I00000P000900F02?05T0
E`0002@0002=0000F@0008l0001I00000P0008l0F@2=05T0E`0002@0002I0000C00009T0001G0000
0P0009T0C02I05L0E`0002@0002I0000E`0009X0001I00000P0009T0E`2J05T0E`0002@0002J0000
F@0009d0001I00000P0009X0F@2M05T0E`0002@0002M0000F@0009h0001I00000P0009d0F@2N05T0
E`0002@0002N0000F@000:40001I00000P0009h0F@2Q05T0E`0002@0002Q0000E`000:<0001I0000
0P000:40F@2S05L0E`0002@0002J0000C00009X0001G00000P0009X0C02J05L0E`0002@0002J0000
E`0009/0001I00000P0009X0E`2K05T0E`0002@0002K0000F@0009d0001I00000P0009/0F@2M05T0
E`0002@0002S0000C0000:<0001I00000P000:<0C02S05T0E`0002@0002T0000C0000:@0001I0000
0P000:@0C02T05T0E`0002@0002F0000C00009X0001<00000P0009H0C02J04`0E`0002@0002P0000
C0000:@0001<00000P000:00C02T04`0E`0002@0002S0000F@000:H0001I00000P000:<0F@2V05T0
E`0002@0002[0000C0000;00001I00000P000:/0C02`05T0E`0002@0002/0000C0000;00001H0000
0P000:`0C02`05P0E`0002@0002`0000C0000;H0001I00000P000;H0C02`05T0E`0002@0002_0000
F@000;00001M00000P000;00F@2_05d0E`0002@0002]0000G@000:l0001O00000P000:l0G@2]05l0
E`0002@0002[0000G`000:d0001P00000P000:d0G`2[0600E`0002@0002Z0000H0000:/0001P0000
0P000:/0H02Z0600E`0002@0002Y0000G`000:X0001P00000P000:X0H02Y05l0E`0002@0002Y0000
GP000:X0001O00000P000:T0G`2Z05h0E`0002@0002Z0000GP000:/0001O00000P000:X0GP2[05l0
E`0002@0002Y0000C0000:l0001<00000P000:T0C02_04`0E`0002@0002b0000C0000;P0001<0000
0P000;80C02h04`0E`0002@0002k0000@P000;d0001400000P000;/0@P2m04@0E`0002@0002m0000
A0000;l0001700000P000;d0A02o04L0E`0002@0002o0000A`000<40001;00000P000;l0A`3104/0
E`0002@000310000B`000<40001?00000P000<40B`3104l0E`0002@000310000C`000<40001C0000
0P000<40C`3105<0E`0002@000310000D`000<40001H00000P000<40D`3105P0E`0002@0002o0000
F0000<40001K00000P000<40F02o05/0E`0002@0002m0000F`000;l0001N00000P000;l0F`2m05h0
E`0002@0002k0000GP000;d0001P00000P000;d0GP2k0600E`0002@0002m0000A0000;l000180000
0P000;d0A02o04P0E`0002@0002o0000B0000<00001;00000P000;l0B03004/0E`0002@000300000
B`000<40001?00000P000<00B`3104l0E`0002@000310000C`000<40001C00000P000<40C`3105<0
E`0002@000300000D`000<40001H00000P000<40D`3005P0E`0002@0002o0000F0000<00001J0000
0P000<00F02o05X0E`0002@0002m0000FP000;l0001N00000P000;l0FP2m05h0E`0002@0003E0000
Q@<00=D0002=0`000P000=D0Q@?E08d3E`0002@0000;0@00Q@<000/1002=0`000P0000/1Q@<;0Hd3
E`0002@000100@00Q@<00401002=0`000P000401Q@=00Hd3E`0002@0001f0@00Q@<007H1002=0`00
0P0007H1Q@=f0Hd3E`0002@0002/0@00Q@<00:`1002=0`000P000:`1Q@>/0Hd3E`0002@0003R0@00
Q@<00>81002=0`000P000>81Q@?R0Hd3E`0002@0000H0P00Q@<001P2002=0`000P0001P2Q@<H0Xd3
E`0002@0001>0P00Q@<004h2002=0`000P0004h2Q@=>0Xd3E`0002@000240P00Q@<008@2002=0`00
0P0008@2Q@>40Xd3E`0002@0002j0P00Q@<00;X2002=0`000P000;X2Q@>j0Xd3E`0002@0003`0P00
Q@<00?02002=0`000P000?02Q@?`0Xd3E`0002@0000V0`00Q@<002H3002=0`000P0002H3Q@<V0hd3
E`0002@0001L0`00Q@<005`3002=0`000P0005`3Q@=L0hd3E`0002@0002B0`00Q@<00983002=0`00
0P000983Q@>B0hd3E`0002@000380`00Q@<00<P3002=0`000P000<P3Q@?80hd3E`0002@0003n0`00
Q@<00?h3002=0`000P000?h3Q@?n0hd3E`0002@0000d1000Q@<003@4002=0`000P0003@4Q@<d18d3
E`0002@0001Y1000Q@<006T4002=0`000P0006T4Q@=Y18d3E`0002@0002O1000Q@<009l4002=0`00
0P0009l4Q@>O18d3E`0002@0003E1000Q@<00=D4002=0`000P000=D4Q@?E18d3E`0002@0002h0P00
b`<00;P2003N0`000P000;P2b`>h0]h3E`0002@0002h0P00b`<00;P2003N0`000P000;P2b`>h0]h3
E`0002@0002a0P00b`<00;82003@0`000P000;82b`>a0]03E`0002@0002a0P00b`<00;42003@0`00
0P000;42d0>a0//3E`0002@0002a0P00b`<00;l2003;0`000P000;42b`>o0//3E`0002@0002o0P00
b`<00;l2003@0`000P000;l2b`>o0]03E`0002@0002n0P00b`<00;l2003@0`000P000;l2d0>n0//3
E`0002@0002e0P00gP<00;/2003N0`000P000;D2gP>k0]h3E`0002@000340P00b`<00<D2003<0`00
0P000<D2b`?40/`3E`0002@000340P00c0<00<D2003=0`000P000<@2c0?50/d3E`0002@000350P00
c0<00<H2003=0`000P000<D2c@?60/`3E`0002@000350P00b`<00<H2003<0`000P000<H2c0?50//3
E`0002@000350P00d@<00<D2003N0`000P000<D2d@?50]h3E`0002@000360P00d@<00<H2003N0`00
0P000<H2d@?60]h3E`0002@000320P00d@<00<H2003A0`000P000<82d@?60]43E`0002@000320P00
gP<00<T2003N0`000P000<82gP?90]h3E`0002@0003?0P00d@<00<l2003N0`000P000<l2d@??0]h3
E`0002@0003@0P00d@<00=02003N0`000P000=02d@?@0]h3E`0002@0003@0P00dP<00=82003D0`00
0P000=02e0?B0]83E`0002@0003B0P00d@<00=@2003B0`000P000=82dP?D0]43E`0002@0003D0P00
d@<00=H2003A0`000P000=@2d@?F0]43E`0002@0003F0P00d@<00=T2003B0`000P000=H2d@?I0]83
E`0002@0003I0P00dP<00=X2003D0`000P000=T2dP?J0]@3E`0002@0003J0P00e0<00=X2003N0`00
0P000=X2e0?J0]h3E`0002@0003F0P00d@<00=P2003B0`000P000=H2d@?H0]83E`0002@0003H0P00
dP<00=T2003D0`000P000=P2dP?I0]@3E`0002@0003I0P00e0<00=T2003N0`000P000=T2e0?I0]h3
E`0002@0003J0P00dP<00=/2003D0`000P000=X2e0?K0]83E`0002@0003K0P00d@<00=h2003B0`00
0P000=/2dP?N0]43E`0002@0003N0P00d@<00>02003A0`000P000=h2d@?P0]43E`0002@0003P0P00
d@<00><2003B0`000P000>02d@?S0]83E`0002@0003S0P00dP<00>@2003D0`000P000><2dP?T0]@3
E`0002@0003T0P00e0<00>@2003N0`000P000>@2e0?T0]h3E`0002@0003P0P00d@<00>82003B0`00
0P000>02d@?R0]83E`0002@0003R0P00dP<00><2003D0`000P000>82dP?S0]@3E`0002@0003S0P00
e0<00><2003N0`000P000><2e0?S0]h3E`0002@0003<0P00d@<00=02003A0`000P000<`2d@?@0]43
E`0002@0003<0P00gP<00=<2003N0`000P000<`2gP?C0]h3E`0002@0003F0P00gP<00=`2003N0`00
0P000=H2gP?L0]h3E`0002@0003P0P00gP<00>H2003N0`000P000>02gP?V0]h3E`0002@0003/0P00
e`<00?L2003G0`000P000>`2e`?g0]L3E`0002@0003g0P00e@<00?L2003G0`000P000?L2e`?g0]D3
E`0002@0003f0P00d`<00?L2003E0`000P000?L2e@?f0]<3E`0002@0003e0P00dP<00?H2003C0`00
0P000?H2d`?e0]83E`0002@0003c0P00d@<00?D2003B0`000P000?D2dP?c0]43E`0002@0003a0P00
d@<00?<2003A0`000P000?<2d@?a0]43E`0002@0003^0P00d@<00?42003B0`000P000?42d@?^0]83
E`0002@0003/0P00dP<00>h2003D0`000P000>h2dP?/0]@3E`0002@0003[0P00e0<00>`2003G0`00
0P000>`2e0?[0]L3E`0002@0003[0P00e`<00>/2003I0`000P000>/2e`?[0]T3E`0002@0003[0P00
f@<00>`2003L0`000P000>/2f@?/0]`3E`0002@0003/0P00g0<00>h2003N0`000P000>`2g0?^0]h3
E`0002@0003^0P00gP<00?42003N0`000P000>h2gP?a0]h3E`0002@0003a0P00gP<00?82003N0`00
0P000?42gP?b0]h3E`0002@0003b0P00gP<00?D2003N0`000P000?82gP?e0]h3E`0002@0003e0P00
g0<00?L2003N0`000P000?D2gP?g0]`3E`0002@0003f0P00e0<00?H2003G0`000P000?H2e`?f0]@3
E`0002@0003e0P00dP<00?H2003D0`000P000?H2e0?e0]83E`0002@0003_0P00d@<00?42003B0`00
0P000?42d@?_0]83E`0002@0003]0P00dP<00>l2003D0`000P000>l2dP?]0]@3E`0002@0003/0P00
e0<00>d2003G0`000P000>d2e0?/0]L3E`0002@0003/0P00e`<00>`2003I0`000P000>`2e`?/0]T3
E`0002@0003/0P00f@<00>d2003L0`000P000>`2f@?]0]`3E`0002@0003]0P00g0<00>l2003N0`00
0P000>d2g0?_0]h3E`0002@0003_0P00gP<00?42003N0`000P000>l2gP?a0]h3E`0002@0003A0000
WP<00=<0002O0`000P000=40W`?C09h3E`0002@0003C0000V`<00=D0002N0`000P000=<0WP?E09/3
E`0002@0003E0000V`<00=D0002^0`000P000=D0V`?E0:h3E`0002@0003D0000W0<00=@0002^0`00
0P000=@0W0?D0:h3E`0002@0003A0000[P<00=T0002^0`000P000=40[P?I0:h3E`0002@000050@00
W`<000H1002P0`000P0000D1W`<60J03E`0002@000050@00X0<000H1002P0`000P0000H1X0<50J03
E`0002@000040@00X0<000D1002P0`000P0000D1X0<40J03E`0002@000040@00W`<000@1002P0`00
0P0000@1X0<40Il3E`0002@000040@00W@<000D1002O0`000P0000@1W`<50Id3E`0002@000050@00
W0<000H1002M0`000P0000D1W@<60I`3E`0002@000060@00V`<000T1002L0`000P0000H1W0<90I/3
E`0002@000090@00V`<000`1002K0`000P0000T1V`<<0I/3E`0002@0000<0@00V`<000l1002L0`00
0P0000`1V`<?0I`3E`0002@0000?0@00W0<00101002M0`000P0000l1W0<@0Id3E`0002@0000@0@00
W@<00141002O0`000P000101W@<A0Il3E`0002@0000A0@00W`<00141002P0`000P000141W`<A0J03
E`0002@0000@0@00X0<00141002R0`000P000141X0<@0J83E`0002@0000=0@00XP<00101002T0`00
0P000101XP<=0J@3E`0002@000090@00Y0<000d1002V0`000P0000d1Y0<90JH3E`0002@000070@00
YP<000T1002W0`000P0000T1YP<70JL3E`0002@000050@00Y`<000L1002Y0`000P0000L1Y`<50JT3
E`0002@000040@00Z@<000D1002/0`000P0000D1Z@<40J`3E`0002@000040@00[0<000@1002^0`00
0P0000@1[0<40Jh3E`0002@0000<0@00V`<000h1002L0`000P0000`1V`<>0I`3E`0002@0000>0@00
W0<000l1002M0`000P0000h1W0<?0Id3E`0002@0000?0@00W@<00101002O0`000P0000l1W@<@0Il3
E`0002@0000@0@00W`<00101002P0`000P000101W`<@0J03E`0002@0000?0@00X0<00101002R0`00
0P000101X0<?0J83E`0002@0000<0@00XP<000l1002T0`000P0000l1XP<<0J@3E`0002@000090@00
Y0<000`1002V0`000P0000`1Y0<90JH3E`0002@000040@00[0<000D1002]0`000P0000@1[@<50J`3
E`0002@000050@00[0<000L1002/0`000P0000D1[0<70J`3E`0002@000070@00[0<000/1002^0`00
0P0000L1[0<;0Jh3E`0002@0000;0@00[P<000h1002^0`000P0000/1[P<>0Jh3E`0002@0000>0@00
[@<00101002^0`000P0000h1[P<@0Jd3E`0002@0000@0@00[0<00141002]0`000P000101[@<A0J`3
E`0002@000070@00[0<000/1002^0`000P0000L1[0<;0Jh3E`0002@0000;0@00[P<000l1002^0`00
0P0000/1[P<?0Jh3E`0002@0000?0@00[P<00101002^0`000P0000l1[P<@0Jh3E`0002@0000@0@00
[0<00141002^0`000P000101[P<A0J`3E`0002@0000A0@00ZP<00141002/0`000P000141[0<A0JX3
E`0002@0000k0@00W`<003`1002P0`000P0003/1W`<l0J03E`0002@0000k0@00X0<003`1002P0`00
0P0003`1X0<k0J03E`0002@0000j0@00X0<003/1002P0`000P0003/1X0<j0J03E`0002@0000j0@00
W`<003X1002P0`000P0003X1X0<j0Il3E`0002@0000j0@00W@<003/1002O0`000P0003X1W`<k0Id3
E`0002@0000k0@00W0<003`1002M0`000P0003/1W@<l0I`3E`0002@0000l0@00V`<003l1002L0`00
0P0003`1W0<o0I/3E`0002@0000o0@00V`<00481002K0`000P0003l1V`=20I/3E`0002@000120@00
V`<004D1002L0`000P000481V`=50I`3E`0002@000150@00W0<004H1002N0`000P0004D1W0=60Ih3
E`0002@000160@00WP<004H1002P0`000P0004H1WP=60J03E`0002@000150@00X0<004H1002R0`00
0P0004H1X0=50J83E`0002@000120@00XP<004D1002S0`000P0004D1XP=20J<3E`0002@000100@00
X`<00481002S0`000P000481X`=00J<3E`0002@000120@00V`<004@1002L0`000P000481V`=40I`3
E`0002@000140@00W0<004D1002N0`000P0004@1W0=50Ih3E`0002@000150@00WP<004D1002P0`00
0P0004D1WP=50J03E`0002@000140@00X0<004D1002R0`000P0004D1X0=40J83E`0002@000120@00
XP<004@1002S0`000P0004@1XP=20J<3E`0002@000120@00X`<004@1002T0`000P000481X`=40J@3
E`0002@000140@00Y0<004H1002V0`000P0004@1Y0=60JH3E`0002@000160@00YP<004L1002X0`00
0P0004H1YP=70JP3E`0002@000170@00Z0<004L1002[0`000P0004L1Z0=70J/3E`0002@000160@00
Z`<004L1002]0`000P0004L1Z`=60Jd3E`0002@000150@00[@<004H1002^0`000P0004H1[@=50Jh3
E`0002@000120@00[P<004D1002^0`000P0004D1[P=20Jh3E`0002@0000o0@00[P<00481002^0`00
0P000481[P<o0Jh3E`0002@0000l0@00[P<003l1002^0`000P0003l1[P<l0Jh3E`0002@0000k0@00
[@<003`1002^0`000P0003`1[P<k0Jd3E`0002@0000j0@00Z`<003/1002]0`000P0003/1[@<j0J/3
E`0002@0000j0@00ZP<003X1002[0`000P0003X1Z`<j0JX3E`0002@0000j0@00Z@<003/1002Z0`00
0P0003X1ZP<k0JT3E`0002@0000k0@00Z@<003`1002Z0`000P0003/1Z@<l0JX3E`0002@0000k0@00
ZP<003`1002[0`000P0003`1ZP<k0J/3E`0002@000150@00Y@<004H1002X0`000P0004D1Y@=60JP3
E`0002@000160@00Z0<004H1002[0`000P0004H1Z0=60J/3E`0002@000150@00Z`<004H1002]0`00
0P0004H1Z`=50Jd3E`0002@000140@00[@<004D1002^0`000P0004D1[@=40Jh3E`0002@000120@00
[P<004@1002^0`000P0004@1[P=20Jh3E`0002@0001h0@00W@<007P1002^0`000P0007P1W@=h0Jh3
E`0002@0001i0@00V`<007T1002^0`000P0007T1V`=i0Jh3E`0002@0001_0@00V`<007T1002Y0`00
0P0007T1V`=_0JT3E`0002@0001_0@00Z@<007h1002Y0`000P0006l1Z@=n0JT3E`0002@0001f0@00
[P<007`1002^0`000P0007H1[P=l0Jh3E`0002@0002U0@00V`<00:L1002T0`000P000:L1V`>U0J@3
E`0002@0002U0@00XP<00:L1002T0`000P000:D1Y0>W0J83E`0002@0002W0@00X@<00:X1002R0`00
0P000:L1XP>Z0J43E`0002@0002Z0@00X@<00:`1002Q0`000P000:X1X@>/0J43E`0002@0002/0@00
X@<00:l1002R0`000P000:`1X@>_0J83E`0002@0002_0@00XP<00;41002T0`000P000:l1XP>a0J@3
E`0002@0002a0@00Y0<00;81002W0`000P000;41Y0>b0JL3E`0002@0002b0@00Y`<00;81002Y0`00
0P000;81Y`>b0JT3E`0002@0002a0@00Z@<00;81002/0`000P000;81Z@>a0J`3E`0002@0002_0@00
[0<00;41002^0`000P000;41[0>_0Jh3E`0002@0002/0@00[P<00:l1002^0`000P000:l1[P>/0Jh3
E`0002@0002Z0@00[P<00:`1002^0`000P000:`1[P>Z0Jh3E`0002@0002W0@00[P<00:X1002^0`00
0P000:X1[P>W0Jh3E`0002@0002V0@00[@<00:L1002^0`000P000:L1[P>V0Jd3E`0002@0002U0@00
Z`<00:H1002]0`000P000:H1[@>U0J/3E`0002@0002U0@00ZP<00:D1002[0`000P000:D1Z`>U0JX3
E`0002@0002U0@00Z@<00:H1002Z0`000P000:D1ZP>V0JT3E`0002@0002V0@00Z@<00:L1002Z0`00
0P000:H1Z@>W0JX3E`0002@0002V0@00ZP<00:L1002[0`000P000:L1ZP>V0J/3E`0002@0002/0@00
X@<00:h1002R0`000P000:`1X@>^0J83E`0002@0002^0@00XP<00;01002T0`000P000:h1XP>`0J@3
E`0002@0002`0@00Y0<00;41002W0`000P000;01Y0>a0JL3E`0002@0002a0@00Y`<00;41002Y0`00
0P000;41Y`>a0JT3E`0002@0002`0@00Z@<00;41002/0`000P000;41Z@>`0J`3E`0002@0002^0@00
[0<00;01002^0`000P000;01[0>^0Jh3E`0002@0002/0@00[P<00:h1002^0`000P000:h1[P>/0Jh3
E`0002@0002W0@00V`<00;01002K0`000P000:L1V`>`0I/3E`0002@0002W0@00W0<00:/1002L0`00
0P000:L1W0>[0I`3E`0002@0002[0@00V`<00;01002L0`000P000:/1W0>`0I/3E`0002@0003U0@00
WP<00>H1002O0`000P000>H1WP?U0Il3E`0002@0003U0@00W`<00>H1002P0`000P000>D1W`?V0J03
E`0002@0003V0@00W`<00>L1002P0`000P000>H1X0?W0Il3E`0002@0003W0@00WP<00>L1002O0`00
0P000>L1W`?W0Ih3E`0002@0003V0@00W0<00>L1002N0`000P000>L1WP?V0I`3E`0002@0003T0@00
V`<00>H1002L0`000P000>H1W0?T0I/3E`0002@0003Q0@00V`<00>@1002K0`000P000>@1V`?Q0I/3
E`0002@0003O0@00V`<00>41002L0`000P000>41V`?O0I`3E`0002@0003M0@00W0<00=l1002N0`00
0P000=l1W0?M0Ih3E`0002@0003L0@00WP<00=d1002P0`000P000=d1WP?L0J03E`0002@0003K0@00
X0<00=`1002S0`000P000=`1X0?K0J<3E`0002@0003K0@00X`<00=/1002Y0`000P000=/1X`?K0JT3
E`0002@0003K0@00Z@<00=`1002/0`000P000=/1Z@?L0J`3E`0002@0003L0@00[0<00=h1002^0`00
0P000=`1[0?N0Jh3E`0002@0003N0@00[P<00>41002^0`000P000=h1[P?Q0Jh3E`0002@0003Q0@00
[P<00>81002^0`000P000>41[P?R0Jh3E`0002@0003R0@00[P<00>D1002^0`000P000>81[P?U0Jh3
E`0002@0003U0@00[0<00>L1002^0`000P000>D1[P?W0J`3E`0002@0003W0@00Z@<00>P1002/0`00
0P000>L1[0?X0JT3E`0002@0003X0@00Z0<00>P1002Y0`000P000>P1Z@?X0JP3E`0002@0003W0@00
Y@<00>P1002X0`000P000>P1Z0?W0JD3E`0002@0003U0@00X`<00>L1002U0`000P000>L1Y@?U0J<3
E`0002@0003R0@00XP<00>D1002S0`000P000>D1X`?R0J83E`0002@0003Q0@00XP<00>81002R0`00
0P000>81XP?Q0J83E`0002@0003O0@00XP<00>41002S0`000P000>41XP?O0J<3E`0002@0003M0@00
X`<00=l1002U0`000P000=l1X`?M0JD3E`0002@0003L0@00Y@<00=d1002X0`000P000=d1Y@?L0JP3
E`0002@0003P0@00V`<00>41002L0`000P000>41V`?P0I`3E`0002@0003N0@00W0<00>01002N0`00
0P000>01W0?N0Ih3E`0002@0003M0@00WP<00=h1002P0`000P000=h1WP?M0J03E`0002@0003L0@00
X0<00=d1002S0`000P000=d1X0?L0J<3E`0002@0003L0@00X`<00=`1002Y0`000P000=`1X`?L0JT3
E`0002@0003L0@00Z@<00=d1002/0`000P000=`1Z@?M0J`3E`0002@0003M0@00[0<00=l1002^0`00
0P000=d1[0?O0Jh3E`0002@0003O0@00[P<00>41002^0`000P000=l1[P?Q0Jh3E`0002@0003R0@00
[P<00>@1002^0`000P000>81[P?T0Jh3E`0002@0003T0@00[0<00>H1002^0`000P000>@1[P?V0J`3
E`0002@0003V0@00Z@<00>L1002/0`000P000>H1[0?W0JT3E`0002@0003W0@00Z0<00>L1002Y0`00
0P000>L1Z@?W0JP3E`0002@0003V0@00Y@<00>L1002X0`000P000>L1Z0?V0JD3E`0002@0003T0@00
X`<00>H1002U0`000P000>H1Y@?T0J<3E`0002@0003R0@00XP<00>@1002S0`000P000>@1X`?R0J83
E`0002@0000A0P00V`<00142002P0`000P000142V`<A0Z03E`0002@0000A0P00W@<00182002O0`00
0P000142W`<B0Yd3E`0002@0000B0P00V`<001@2002M0`000P000182W@<D0Y/3E`0002@0000D0P00
V`<001H2002K0`000P0001@2V`<F0Y/3E`0002@0000F0P00V`<001X2002N0`000P0001H2V`<J0Yh3
E`0002@0000J0P00WP<001`2002N0`000P0001X2WP<L0Yh3E`0002@0000L0P00W@<001d2002N0`00
0P0001`2WP<M0Yd3E`0002@0000M0P00V`<001h2002M0`000P0001d2W@<N0Y/3E`0002@0000B0P00
W0<001@2002M0`000P000182W@<D0Y`3E`0002@0000D0P00W0<001H2002L0`000P0001@2W0<F0Y`3
E`0002@0000F0P00W0<001X2002N0`000P0001H2W0<J0Yh3E`0002@0000N0P00V`<001h2002N0`00
0P0001h2V`<N0Yh3E`0002@0000M0P00WP<001h2002P0`000P0001h2WP<M0Z03E`0002@0000I0P00
X0<001d2002U0`000P0001d2X0<I0ZD3E`0002@0000H0P00Y@<001T2002W0`000P0001T2Y@<H0ZL3
E`0002@0000G0P00Y`<001P2002Z0`000P0001P2Y`<G0ZX3E`0002@0000G0P00ZP<001L2002^0`00
0P0001L2ZP<G0Zh3E`0002@0000H0P00X0<001d2002U0`000P0001d2X0<H0ZD3E`0002@0000G0P00
Y@<001P2002W0`000P0001P2Y@<G0ZL3E`0002@0000G0P00Y`<001L2002Z0`000P0001L2Y`<G0ZX3
E`0002@0000G0P00ZP<001L2002^0`000P0001L2ZP<G0Zh3E`0002@000190P00V`<004`2002L0`00
0P0004`2V`=90Y`3E`0002@000180P00W0<004T2002N0`000P0004T2W0=80Yh3E`0002@000180P00
WP<004P2002P0`000P0004P2WP=80Z03E`0002@000180P00X0<004T2002R0`000P0004P2X0=90Z83
E`0002@000190P00XP<004`2002S0`000P0004T2XP=<0Z<3E`0002@0001<0P00X`<004l2002S0`00
0P0004`2X`=?0Z<3E`0002@0001?0P00XP<00582002S0`000P0004l2X`=B0Z83E`0002@0001B0P00
X0<005<2002R0`000P000582XP=C0Z03E`0002@0001C0P00WP<005<2002P0`000P0005<2X0=C0Yh3
E`0002@0001B0P00W0<005<2002N0`000P0005<2WP=B0Y`3E`0002@0001?0P00V`<00582002L0`00
0P000582W0=?0Y/3E`0002@0001<0P00V`<004l2002K0`000P0004l2V`=<0Y/3E`0002@0001:0P00
V`<004`2002L0`000P0004`2V`=:0Y`3E`0002@000190P00W0<004X2002N0`000P0004X2W0=90Yh3
E`0002@000190P00WP<004T2002P0`000P0004T2WP=90Z03E`0002@000190P00X0<004X2002R0`00
0P0004T2X0=:0Z83E`0002@0001:0P00XP<004`2002S0`000P0004X2XP=<0Z<3E`0002@0001?0P00
XP<00542002S0`000P0004l2X`=A0Z83E`0002@0001A0P00X0<00582002R0`000P000542XP=B0Z03
E`0002@0001B0P00WP<00582002P0`000P000582X0=B0Yh3E`0002@0001A0P00W0<00582002N0`00
0P000582WP=A0Y`3E`0002@0001?0P00V`<00542002L0`000P000542W0=?0Y/3E`0002@000190P00
X`<004`2002T0`000P0004`2X`=90Z@3E`0002@000180P00Y0<004T2002U0`000P0004T2Y0=80ZD3
E`0002@000170P00Y@<004P2002W0`000P0004P2Y@=70ZL3E`0002@000170P00Y`<004L2002[0`00
0P0004L2Y`=70Z/3E`0002@000170P00Z`<004P2002]0`000P0004L2Z`=80Zd3E`0002@000180P00
[@<004T2002^0`000P0004P2[@=90Zh3E`0002@000190P00[P<004`2002^0`000P0004T2[P=<0Zh3
E`0002@0001<0P00[P<004l2002^0`000P0004`2[P=?0Zh3E`0002@0001?0P00[P<00582002^0`00
0P0004l2[P=B0Zh3E`0002@0001B0P00[@<005<2002^0`000P000582[P=C0Zd3E`0002@0001C0P00
Z`<005@2002]0`000P0005<2[@=D0Z/3E`0002@0001D0P00Y`<005@2002[0`000P0005@2Z`=D0ZL3
E`0002@0001C0P00Y@<005@2002W0`000P0005@2Y`=C0ZD3E`0002@0001B0P00Y0<005<2002U0`00
0P0005<2Y@=B0Z@3E`0002@0001?0P00X`<00582002T0`000P000582Y0=?0Z<3E`0002@0001:0P00
X`<004`2002T0`000P0004`2X`=:0Z@3E`0002@000190P00Y0<004X2002U0`000P0004X2Y0=90ZD3
E`0002@000180P00Y@<004T2002W0`000P0004T2Y@=80ZL3E`0002@000180P00Y`<004P2002[0`00
0P0004P2Y`=80Z/3E`0002@000180P00Z`<004T2002]0`000P0004P2Z`=90Zd3E`0002@000190P00
[@<004X2002^0`000P0004T2[@=:0Zh3E`0002@0001:0P00[P<004`2002^0`000P0004X2[P=<0Zh3
E`0002@0001?0P00[P<00542002^0`000P0004l2[P=A0Zh3E`0002@0001A0P00[@<00582002^0`00
0P000542[P=B0Zd3E`0002@0001B0P00Z`<005<2002]0`000P000582[@=C0Z/3E`0002@0001C0P00
Y`<005<2002[0`000P0005<2Z`=C0ZL3E`0002@0001B0P00Y@<005<2002W0`000P0005<2Y`=B0ZD3
E`0002@0001A0P00Y0<00582002U0`000P000582Y@=A0Z@3E`0002@0001?0P00X`<00542002T0`00
0P000542Y0=?0Z<3E`0002@000280P00X@<008T2002T0`000P0008T2X@>80Z@3E`0002@000260P00
Y0<008P2002V0`000P0008P2Y0>60ZH3E`0002@000230P00YP<008H2002W0`000P0008H2YP>30ZL3
E`0002@000230P00Y`<008<2002W0`000P0008<2Y`>30ZL3E`0002@000200P00YP<008<2002W0`00
0P0008<2Y`>00ZH3E`0002@0001n0P00Y0<00802002V0`000P000802YP=n0Z@3E`0002@0001m0P00
X@<007h2002T0`000P0007h2Y0=m0Z43E`0002@0001m0P00X0<007d2002Q0`000P0007d2X@=m0Z03
E`0002@0001m0P00WP<007h2002P0`000P0007d2X0=n0Yh3E`0002@0001n0P00W0<00802002N0`00
0P0007h2WP>00Y`3E`0002@000200P00V`<008<2002L0`000P000802W0>30Y/3E`0002@000230P00
V`<008@2002K0`000P0008<2V`>40Y/3E`0002@000240P00V`<008L2002L0`000P0008@2V`>70Y`3
E`0002@000270P00W0<008T2002N0`000P0008L2W0>90Yh3E`0002@000290P00WP<008X2002P0`00
0P0008T2WP>:0Z03E`0002@0002:0P00X0<008X2002V0`000P0008X2X0>:0ZH3E`0002@000290P00
YP<008X2002Z0`000P0008X2YP>90ZX3E`0002@000280P00ZP<008T2002/0`000P0008T2ZP>80Z`3
E`0002@000260P00[0<008P2002^0`000P0008P2[0>60Zh3E`0002@000230P00[P<008H2002^0`00
0P0008H2[P>30Zh3E`0002@000210P00[P<008<2002^0`000P0008<2[P>10Zh3E`0002@0001o0P00
[P<00842002^0`000P000842[P=o0Zh3E`0002@0001n0P00[0<007l2002^0`000P0007l2[P=n0Z`3
E`0002@0001n0P00Z`<007h2002/0`000P0007h2[0=n0Z/3E`0002@0001n0P00ZP<007l2002[0`00
0P0007h2Z`=o0ZX3E`0002@0001o0P00ZP<00802002[0`000P0007l2ZP>00Z/3E`0002@0001o0P00
Z`<00802002/0`000P000802Z`=o0Z`3E`0002@000210P00YP<008<2002W0`000P0008<2Y`>10ZH3
E`0002@0001o0P00Y0<00842002V0`000P000842YP=o0Z@3E`0002@0001n0P00X@<007l2002T0`00
0P0007l2Y0=n0Z43E`0002@0001n0P00X0<007h2002Q0`000P0007h2X@=n0Z03E`0002@0001n0P00
WP<007l2002P0`000P0007h2X0=o0Yh3E`0002@0001o0P00W0<00842002N0`000P0007l2WP>10Y`3
E`0002@000210P00V`<008<2002L0`000P000842W0>30Y/3E`0002@000240P00V`<008H2002L0`00
0P0008@2V`>60Y`3E`0002@000260P00W0<008P2002N0`000P0008H2W0>80Yh3E`0002@000280P00
WP<008T2002P0`000P0008P2WP>90Z03E`0002@000290P00X0<008T2002V0`000P0008T2X0>90ZH3
E`0002@000280P00YP<008T2002Z0`000P0008T2YP>80ZX3E`0002@000270P00ZP<008P2002/0`00
0P0008P2ZP>70Z`3E`0002@000250P00[0<008L2002^0`000P0008L2[0>50Zh3E`0002@000230P00
[P<008D2002^0`000P0008D2[P>30Zh3E`0002@0002]0P00WP<00:l2002O0`000P000:d2W`>_0Yh3
E`0002@0002_0P00V`<00;42002N0`000P000:l2WP>a0Y/3E`0002@0002a0P00V`<00;42002^0`00
0P000;42V`>a0Zh3E`0002@0002`0P00W0<00;02002^0`000P000;02W0>`0Zh3E`0002@0002]0P00
[P<00;D2002^0`000P000:d2[P>e0Zh3E`0002@0002o0P00V`<00<82002L0`000P000<82V`>o0Y`3
E`0002@0002m0P00W0<00;l2002O0`000P000;l2W0>m0Yl3E`0002@0002l0P00W`<00;d2002S0`00
0P000;d2W`>l0Z<3E`0002@0002l0P00X`<00;`2002V0`000P000;`2X`>l0ZH3E`0002@0002l0P00
YP<00;d2002[0`000P000;`2YP>m0Z/3E`0002@0002m0P00Z`<00;l2002^0`000P000;d2Z`>o0Zh3
E`0002@0002o0P00[P<00<82002^0`000P000;l2[P?20Zh3E`0002@000320P00[P<00<<2002^0`00
0P000<82[P?30Zh3E`0002@000330P00[P<00<H2002^0`000P000<<2[P?60Zh3E`0002@000360P00
Z`<00<P2002^0`000P000<H2[P?80Z/3E`0002@000380P00YP<00<T2002[0`000P000<P2Z`?90ZH3
E`0002@000390P00X`<00<T2002V0`000P000<T2YP?90Z<3E`0002@000380P00W`<00<T2002S0`00
0P000<T2X`?80Yl3E`0002@000360P00W0<00<P2002O0`000P000<P2W`?60Y`3E`0002@000330P00
V`<00<H2002L0`000P000<H2W0?30Y/3E`0002@000320P00V`<00<<2002K0`000P000<<2V`?20Y/3
E`0002@000300P00V`<00<82002L0`000P000<82V`?00Y`3E`0002@0002o0P00W0<00<02002M0`00
0P000<02W0>o0Yd3E`0002@0002n0P00W@<00;l2002O0`000P000;l2W@>n0Yl3E`0002@0002m0P00
W`<00;h2002S0`000P000;h2W`>m0Z<3E`0002@0002m0P00X`<00;d2002V0`000P000;d2X`>m0ZH3
E`0002@0002m0P00YP<00;h2002[0`000P000;d2YP>n0Z/3E`0002@0002n0P00Z`<00;l2002]0`00
0P000;h2Z`>o0Zd3E`0002@0002o0P00[@<00<02002^0`000P000;l2[@?00Zh3E`0002@000300P00
[P<00<82002^0`000P000<02[P?20Zh3E`0002@000330P00[P<00<D2002^0`000P000<<2[P?50Zh3
E`0002@000350P00[@<00<H2002^0`000P000<D2[P?60Zd3E`0002@000360P00Z`<00<L2002]0`00
0P000<H2[@?70Z/3E`0002@000370P00YP<00<P2002[0`000P000<L2Z`?80ZH3E`0002@000380P00
X`<00<P2002V0`000P000<P2YP?80Z<3E`0002@000370P00W`<00<P2002S0`000P000<P2X`?70Yl3
E`0002@000360P00W@<00<L2002O0`000P000<L2W`?60Yd3E`0002@000350P00W0<00<H2002M0`00
0P000<H2W@?50Y`3E`0002@000330P00V`<00<D2002L0`000P000<D2W0?30Y/3E`0002@0003S0P00
WP<00>D2002O0`000P000><2W`?U0Yh3E`0002@0003U0P00V`<00>L2002N0`000P000>D2WP?W0Y/3
E`0002@0003W0P00V`<00>L2002^0`000P000>L2V`?W0Zh3E`0002@0003V0P00W0<00>H2002^0`00
0P000>H2W0?V0Zh3E`0002@0003S0P00[P<00>/2002^0`000P000><2[P?[0Zh3E`0002@0003e0P00
WP<00?L2002O0`000P000?D2W`?g0Yh3E`0002@0003g0P00V`<00?T2002N0`000P000?L2WP?i0Y/3
E`0002@0003i0P00V`<00?T2002^0`000P000?T2V`?i0Zh3E`0002@0003h0P00W0<00?P2002^0`00
0P000?P2W0?h0Zh3E`0002@0003e0P00[P<00?d2002^0`000P000?D2[P?m0Zh3E`0002@0000I0`00
WP<001/3002O0`000P0001T3W`<K0ih3E`0002@0000K0`00V`<001d3002N0`000P0001/3WP<M0i/3
E`0002@0000M0`00V`<001d3002^0`000P0001d3V`<M0jh3E`0002@0000L0`00W0<001`3002^0`00
0P0001`3W0<L0jh3E`0002@0000I0`00[P<00243002^0`000P0001T3[P<Q0jh3E`0002@0000Y0`00
W`<002X3002P0`000P0002T3W`<Z0j03E`0002@0000Y0`00X0<002X3002P0`000P0002X3X0<Y0j03
E`0002@0000X0`00X0<002T3002P0`000P0002T3X0<X0j03E`0002@0000X0`00W`<002P3002P0`00
0P0002P3X0<X0il3E`0002@0000X0`00W@<002T3002O0`000P0002P3W`<Y0id3E`0002@0000Y0`00
W0<002X3002M0`000P0002T3W@<Z0i`3E`0002@0000Z0`00V`<002d3002L0`000P0002X3W0<]0i/3
E`0002@0000]0`00V`<00303002K0`000P0002d3V`<`0i/3E`0002@0000`0`00V`<003<3002L0`00
0P000303V`<c0i`3E`0002@0000c0`00W0<003@3002M0`000P0003<3W0<d0id3E`0002@0000d0`00
W@<003D3002O0`000P0003@3W@<e0il3E`0002@0000e0`00W`<003D3002P0`000P0003D3W`<e0j03
E`0002@0000d0`00X0<003D3002R0`000P0003D3X0<d0j83E`0002@0000a0`00XP<003@3002T0`00
0P0003@3XP<a0j@3E`0002@0000]0`00Y0<00343002V0`000P000343Y0<]0jH3E`0002@0000[0`00
YP<002d3002W0`000P0002d3YP<[0jL3E`0002@0000Y0`00Y`<002/3002Y0`000P0002/3Y`<Y0jT3
E`0002@0000X0`00Z@<002T3002/0`000P0002T3Z@<X0j`3E`0002@0000X0`00[0<002P3002^0`00
0P0002P3[0<X0jh3E`0002@0000`0`00V`<00383002L0`000P000303V`<b0i`3E`0002@0000b0`00
W0<003<3002M0`000P000383W0<c0id3E`0002@0000c0`00W@<003@3002O0`000P0003<3W@<d0il3
E`0002@0000d0`00W`<003@3002P0`000P0003@3W`<d0j03E`0002@0000c0`00X0<003@3002R0`00
0P0003@3X0<c0j83E`0002@0000`0`00XP<003<3002T0`000P0003<3XP<`0j@3E`0002@0000]0`00
Y0<00303002V0`000P000303Y0<]0jH3E`0002@0000X0`00[0<002T3002]0`000P0002P3[@<Y0j`3
E`0002@0000Y0`00[0<002/3002/0`000P0002T3[0<[0j`3E`0002@0000[0`00[0<002l3002^0`00
0P0002/3[0<_0jh3E`0002@0000_0`00[P<00383002^0`000P0002l3[P<b0jh3E`0002@0000b0`00
[@<003@3002^0`000P000383[P<d0jd3E`0002@0000d0`00[0<003D3002]0`000P0003@3[@<e0j`3
E`0002@0000[0`00[0<002l3002^0`000P0002/3[0<_0jh3E`0002@0000_0`00[P<003<3002^0`00
0P0002l3[P<c0jh3E`0002@0000c0`00[P<003@3002^0`000P0003<3[P<d0jh3E`0002@0000d0`00
[0<003D3002^0`000P0003@3[P<e0j`3E`0002@0000e0`00ZP<003D3002/0`000P0003D3[0<e0jX3
E`0002@0001?0`00WP<00543002O0`000P0004l3W`=A0ih3E`0002@0001A0`00V`<005<3002N0`00
0P000543WP=C0i/3E`0002@0001C0`00V`<005<3002^0`000P0005<3V`=C0jh3E`0002@0001B0`00
W0<00583002^0`000P000583W0=B0jh3E`0002@0001?0`00[P<005L3002^0`000P0004l3[P=G0jh3
E`0002@0001O0`00W`<00603002P0`000P0005l3W`=P0j03E`0002@0001O0`00X0<00603002P0`00
0P000603X0=O0j03E`0002@0001N0`00X0<005l3002P0`000P0005l3X0=N0j03E`0002@0001N0`00
W`<005h3002P0`000P0005h3X0=N0il3E`0002@0001N0`00W@<005l3002O0`000P0005h3W`=O0id3
E`0002@0001O0`00W0<00603002M0`000P0005l3W@=P0i`3E`0002@0001P0`00V`<006<3002L0`00
0P000603W0=S0i/3E`0002@0001S0`00V`<006H3002K0`000P0006<3V`=V0i/3E`0002@0001V0`00
V`<006T3002L0`000P0006H3V`=Y0i`3E`0002@0001Y0`00W0<006X3002N0`000P0006T3W0=Z0ih3
E`0002@0001Z0`00WP<006X3002P0`000P0006X3WP=Z0j03E`0002@0001Y0`00X0<006X3002R0`00
0P0006X3X0=Y0j83E`0002@0001V0`00XP<006T3002S0`000P0006T3XP=V0j<3E`0002@0001T0`00
X`<006H3002S0`000P0006H3X`=T0j<3E`0002@0001V0`00V`<006P3002L0`000P0006H3V`=X0i`3
E`0002@0001X0`00W0<006T3002N0`000P0006P3W0=Y0ih3E`0002@0001Y0`00WP<006T3002P0`00
0P0006T3WP=Y0j03E`0002@0001X0`00X0<006T3002R0`000P0006T3X0=X0j83E`0002@0001V0`00
XP<006P3002S0`000P0006P3XP=V0j<3E`0002@0001V0`00X`<006P3002T0`000P0006H3X`=X0j@3
E`0002@0001X0`00Y0<006X3002V0`000P0006P3Y0=Z0jH3E`0002@0001Z0`00YP<006/3002X0`00
0P0006X3YP=[0jP3E`0002@0001[0`00Z0<006/3002[0`000P0006/3Z0=[0j/3E`0002@0001Z0`00
Z`<006/3002]0`000P0006/3Z`=Z0jd3E`0002@0001Y0`00[@<006X3002^0`000P0006X3[@=Y0jh3
E`0002@0001V0`00[P<006T3002^0`000P0006T3[P=V0jh3E`0002@0001S0`00[P<006H3002^0`00
0P0006H3[P=S0jh3E`0002@0001P0`00[P<006<3002^0`000P0006<3[P=P0jh3E`0002@0001O0`00
[@<00603002^0`000P000603[P=O0jd3E`0002@0001N0`00Z`<005l3002]0`000P0005l3[@=N0j/3
E`0002@0001N0`00ZP<005h3002[0`000P0005h3Z`=N0jX3E`0002@0001N0`00Z@<005l3002Z0`00
0P0005h3ZP=O0jT3E`0002@0001O0`00Z@<00603002Z0`000P0005l3Z@=P0jX3E`0002@0001O0`00
ZP<00603002[0`000P000603ZP=O0j/3E`0002@0001Y0`00Y@<006X3002X0`000P0006T3Y@=Z0jP3
E`0002@0001Z0`00Z0<006X3002[0`000P0006X3Z0=Z0j/3E`0002@0001Y0`00Z`<006X3002]0`00
0P0006X3Z`=Y0jd3E`0002@0001X0`00[@<006T3002^0`000P0006T3[@=X0jh3E`0002@0001V0`00
[P<006P3002^0`000P0006P3[P=V0jh3E`0002@000250`00WP<008L3002O0`000P0008D3W`>70ih3
E`0002@000270`00V`<008T3002N0`000P0008L3WP>90i/3E`0002@000290`00V`<008T3002^0`00
0P0008T3V`>90jh3E`0002@000280`00W0<008P3002^0`000P0008P3W0>80jh3E`0002@000250`00
[P<008d3002^0`000P0008D3[P>=0jh3E`0002@0002L0`00W@<009`3002^0`000P0009`3W@>L0jh3
E`0002@0002M0`00V`<009d3002^0`000P0009d3V`>M0jh3E`0002@0002C0`00V`<009d3002Y0`00
0P0009d3V`>C0jT3E`0002@0002C0`00Z@<00:83002Y0`000P0009<3Z@>R0jT3E`0002@0002J0`00
[P<00:03002^0`000P0009X3[P>P0jh3E`0002@0002k0`00WP<00;d3002O0`000P000;/3W`>m0ih3
E`0002@0002m0`00V`<00;l3002N0`000P000;d3WP>o0i/3E`0002@0002o0`00V`<00;l3002^0`00
0P000;l3V`>o0jh3E`0002@0002n0`00W0<00;h3002^0`000P000;h3W0>n0jh3E`0002@0002k0`00
[P<00<<3002^0`000P000;/3[P?30jh3E`0002@0003:0`00V`<00<`3002T0`000P000<`3V`?:0j@3
E`0002@0003:0`00XP<00<`3002T0`000P000<X3Y0?<0j83E`0002@0003<0`00X@<00<l3002R0`00
0P000<`3XP??0j43E`0002@0003?0`00X@<00=43002Q0`000P000<l3X@?A0j43E`0002@0003A0`00
X@<00=@3002R0`000P000=43X@?D0j83E`0002@0003D0`00XP<00=H3002T0`000P000=@3XP?F0j@3
E`0002@0003F0`00Y0<00=L3002W0`000P000=H3Y0?G0jL3E`0002@0003G0`00Y`<00=L3002Y0`00
0P000=L3Y`?G0jT3E`0002@0003F0`00Z@<00=L3002/0`000P000=L3Z@?F0j`3E`0002@0003D0`00
[0<00=H3002^0`000P000=H3[0?D0jh3E`0002@0003A0`00[P<00=@3002^0`000P000=@3[P?A0jh3
E`0002@0003?0`00[P<00=43002^0`000P000=43[P??0jh3E`0002@0003<0`00[P<00<l3002^0`00
0P000<l3[P?<0jh3E`0002@0003;0`00[@<00<`3002^0`000P000<`3[P?;0jd3E`0002@0003:0`00
Z`<00</3002]0`000P000</3[@?:0j/3E`0002@0003:0`00ZP<00<X3002[0`000P000<X3Z`?:0jX3
E`0002@0003:0`00Z@<00</3002Z0`000P000<X3ZP?;0jT3E`0002@0003;0`00Z@<00<`3002Z0`00
0P000</3Z@?<0jX3E`0002@0003;0`00ZP<00<`3002[0`000P000<`3ZP?;0j/3E`0002@0003A0`00
X@<00=<3002R0`000P000=43X@?C0j83E`0002@0003C0`00XP<00=D3002T0`000P000=<3XP?E0j@3
E`0002@0003E0`00Y0<00=H3002W0`000P000=D3Y0?F0jL3E`0002@0003F0`00Y`<00=H3002Y0`00
0P000=H3Y`?F0jT3E`0002@0003E0`00Z@<00=H3002/0`000P000=H3Z@?E0j`3E`0002@0003C0`00
[0<00=D3002^0`000P000=D3[0?C0jh3E`0002@0003A0`00[P<00=<3002^0`000P000=<3[P?A0jh3
E`0002@0003<0`00V`<00=D3002K0`000P000<`3V`?E0i/3E`0002@0003<0`00W0<00=03002L0`00
0P000<`3W0?@0i`3E`0002@0003@0`00V`<00=D3002L0`000P000=03W0?E0i/3E`0002@0003a0`00
WP<00?<3002O0`000P000?43W`?c0ih3E`0002@0003c0`00V`<00?D3002N0`000P000?<3WP?e0i/3
E`0002@0003e0`00V`<00?D3002^0`000P000?D3V`?e0jh3E`0002@0003d0`00W0<00?@3002^0`00
0P000?@3W0?d0jh3E`0002@0003a0`00[P<00?T3002^0`000P000?43[P?i0jh3E`0002@0000:1000
WP<000/4002O0`000P0000/4WP<:19l3E`0002@0000:1000W`<000/4002P0`000P0000X4W`<;1:03
E`0002@0000;1000W`<000`4002P0`000P0000/4X0<<19l3E`0002@0000<1000WP<000`4002O0`00
0P0000`4W`<<19h3E`0002@0000;1000W0<000`4002N0`000P0000`4WP<;19`3E`0002@000091000
V`<000/4002L0`000P0000/4W0<919/3E`0002@000061000V`<000T4002K0`000P0000T4V`<619/3
E`0002@000041000V`<000H4002L0`000P0000H4V`<419`3E`0002@000021000W0<000@4002N0`00
0P0000@4W0<219h3E`0002@000011000WP<00084002P0`000P000084WP<11:03E`0002@000001000
X0<00044002S0`000P000044X0<01:<3E`0002@000001000X`<00004002Y0`000P000004X`<01:T3
E`0002@000001000Z@<00044002/0`000P000004Z@<11:`3E`0002@000011000[0<000<4002^0`00
0P000044[0<31:h3E`0002@000031000[P<000H4002^0`000P0000<4[P<61:h3E`0002@000061000
[P<000L4002^0`000P0000H4[P<71:h3E`0002@000071000[P<000X4002^0`000P0000L4[P<:1:h3
E`0002@0000:1000[0<000`4002^0`000P0000X4[P<<1:`3E`0002@0000<1000Z@<000d4002/0`00
0P0000`4[0<=1:T3E`0002@0000=1000Z0<000d4002Y0`000P0000d4Z@<=1:P3E`0002@0000<1000
Y@<000d4002X0`000P0000d4Z0<<1:D3E`0002@0000:1000X`<000`4002U0`000P0000`4Y@<:1:<3
E`0002@000071000XP<000X4002S0`000P0000X4X`<71:83E`0002@000061000XP<000L4002R0`00
0P0000L4XP<61:83E`0002@000041000XP<000H4002S0`000P0000H4XP<41:<3E`0002@000021000
X`<000@4002U0`000P0000@4X`<21:D3E`0002@000011000Y@<00084002X0`000P000084Y@<11:P3
E`0002@000051000V`<000H4002L0`000P0000H4V`<519`3E`0002@000031000W0<000D4002N0`00
0P0000D4W0<319h3E`0002@000021000WP<000<4002P0`000P0000<4WP<21:03E`0002@000011000
X0<00084002S0`000P000084X0<11:<3E`0002@000011000X`<00044002Y0`000P000044X`<11:T3
E`0002@000011000Z@<00084002/0`000P000044Z@<21:`3E`0002@000021000[0<000@4002^0`00
0P000084[0<41:h3E`0002@000041000[P<000H4002^0`000P0000@4[P<61:h3E`0002@000071000
[P<000T4002^0`000P0000L4[P<91:h3E`0002@000091000[0<000/4002^0`000P0000T4[P<;1:`3
E`0002@0000;1000Z@<000`4002/0`000P0000/4[0<<1:T3E`0002@0000<1000Z0<000`4002Y0`00
0P0000`4Z@<<1:P3E`0002@0000;1000Y@<000`4002X0`000P0000`4Z0<;1:D3E`0002@000091000
X`<000/4002U0`000P0000/4Y@<91:<3E`0002@000071000XP<000T4002S0`000P0000T4X`<71:83
E`0002@0000W1000WP<002T4002O0`000P0002L4W`<Y19h3E`0002@0000Y1000V`<002/4002N0`00
0P0002T4WP<[19/3E`0002@0000[1000V`<002/4002^0`000P0002/4V`<[1:h3E`0002@0000Z1000
W0<002X4002^0`000P0002X4W0<Z1:h3E`0002@0000W1000[P<002l4002^0`000P0002L4[P<_1:h3
E`0002@0000f1000V`<003H4002P0`000P0003H4V`<f1:03E`0002@0000f1000W@<003L4002O0`00
0P0003H4W`<g19d3E`0002@0000g1000V`<003T4002M0`000P0003L4W@<i19/3E`0002@0000i1000
V`<003/4002K0`000P0003T4V`<k19/3E`0002@0000k1000V`<003l4002N0`000P0003/4V`<o19h3
E`0002@0000o1000WP<00444002N0`000P0003l4WP=119h3E`0002@000111000W@<00484002N0`00
0P000444WP=219d3E`0002@000121000V`<004<4002M0`000P000484W@=319/3E`0002@0000g1000
W0<003T4002M0`000P0003L4W@<i19`3E`0002@0000i1000W0<003/4002L0`000P0003T4W0<k19`3
E`0002@0000k1000W0<003l4002N0`000P0003/4W0<o19h3E`0002@000131000V`<004<4002N0`00
0P0004<4V`=319h3E`0002@000121000WP<004<4002P0`000P0004<4WP=21:03E`0002@0000n1000
X0<00484002U0`000P000484X0<n1:D3E`0002@0000m1000Y@<003h4002W0`000P0003h4Y@<m1:L3
E`0002@0000l1000Y`<003d4002Z0`000P0003d4Y`<l1:X3E`0002@0000l1000ZP<003`4002^0`00
0P0003`4ZP<l1:h3E`0002@0000m1000X0<00484002U0`000P000484X0<m1:D3E`0002@0000l1000
Y@<003d4002W0`000P0003d4Y@<l1:L3E`0002@0000l1000Y`<003`4002Z0`000P0003`4Y`<l1:X3
E`0002@0000l1000ZP<003`4002^0`000P0003`4ZP<l1:h3E`0002@0001M1000WP<005l4002O0`00
0P0005d4W`=O19h3E`0002@0001O1000V`<00644002N0`000P0005l4WP=Q19/3E`0002@0001Q1000
V`<00644002^0`000P000644V`=Q1:h3E`0002@0001P1000W0<00604002^0`000P000604W0=P1:h3
E`0002@0001M1000[P<006D4002^0`000P0005d4[P=U1:h3E`0002@0001]1000V`<00704002L0`00
0P000704V`=]19`3E`0002@0001/1000W0<006d4002N0`000P0006d4W0=/19h3E`0002@0001/1000
WP<006`4002P0`000P0006`4WP=/1:03E`0002@0001/1000X0<006d4002R0`000P0006`4X0=]1:83
E`0002@0001]1000XP<00704002S0`000P0006d4XP=`1:<3E`0002@0001`1000X`<007<4002S0`00
0P000704X`=c1:<3E`0002@0001c1000XP<007H4002S0`000P0007<4X`=f1:83E`0002@0001f1000
X0<007L4002R0`000P0007H4XP=g1:03E`0002@0001g1000WP<007L4002P0`000P0007L4X0=g19h3
E`0002@0001f1000W0<007L4002N0`000P0007L4WP=f19`3E`0002@0001c1000V`<007H4002L0`00
0P0007H4W0=c19/3E`0002@0001`1000V`<007<4002K0`000P0007<4V`=`19/3E`0002@0001^1000
V`<00704002L0`000P000704V`=^19`3E`0002@0001]1000W0<006h4002N0`000P0006h4W0=]19h3
E`0002@0001]1000WP<006d4002P0`000P0006d4WP=]1:03E`0002@0001]1000X0<006h4002R0`00
0P0006d4X0=^1:83E`0002@0001^1000XP<00704002S0`000P0006h4XP=`1:<3E`0002@0001c1000
XP<007D4002S0`000P0007<4X`=e1:83E`0002@0001e1000X0<007H4002R0`000P0007D4XP=f1:03
E`0002@0001f1000WP<007H4002P0`000P0007H4X0=f19h3E`0002@0001e1000W0<007H4002N0`00
0P0007H4WP=e19`3E`0002@0001c1000V`<007D4002L0`000P0007D4W0=c19/3E`0002@0001]1000
X`<00704002T0`000P000704X`=]1:@3E`0002@0001/1000Y0<006d4002U0`000P0006d4Y0=/1:D3
E`0002@0001[1000Y@<006`4002W0`000P0006`4Y@=[1:L3E`0002@0001[1000Y`<006/4002[0`00
0P0006/4Y`=[1:/3E`0002@0001[1000Z`<006`4002]0`000P0006/4Z`=/1:d3E`0002@0001/1000
[@<006d4002^0`000P0006`4[@=]1:h3E`0002@0001]1000[P<00704002^0`000P0006d4[P=`1:h3
E`0002@0001`1000[P<007<4002^0`000P000704[P=c1:h3E`0002@0001c1000[P<007H4002^0`00
0P0007<4[P=f1:h3E`0002@0001f1000[@<007L4002^0`000P0007H4[P=g1:d3E`0002@0001g1000
Z`<007P4002]0`000P0007L4[@=h1:/3E`0002@0001h1000Y`<007P4002[0`000P0007P4Z`=h1:L3
E`0002@0001g1000Y@<007P4002W0`000P0007P4Y`=g1:D3E`0002@0001f1000Y0<007L4002U0`00
0P0007L4Y@=f1:@3E`0002@0001c1000X`<007H4002T0`000P0007H4Y0=c1:<3E`0002@0001^1000
X`<00704002T0`000P000704X`=^1:@3E`0002@0001]1000Y0<006h4002U0`000P0006h4Y0=]1:D3
E`0002@0001/1000Y@<006d4002W0`000P0006d4Y@=/1:L3E`0002@0001/1000Y`<006`4002[0`00
0P0006`4Y`=/1:/3E`0002@0001/1000Z`<006d4002]0`000P0006`4Z`=]1:d3E`0002@0001]1000
[@<006h4002^0`000P0006d4[@=^1:h3E`0002@0001^1000[P<00704002^0`000P0006h4[P=`1:h3
E`0002@0001c1000[P<007D4002^0`000P0007<4[P=e1:h3E`0002@0001e1000[@<007H4002^0`00
0P0007D4[P=f1:d3E`0002@0001f1000Z`<007L4002]0`000P0007H4[@=g1:/3E`0002@0001g1000
Y`<007L4002[0`000P0007L4Z`=g1:L3E`0002@0001f1000Y@<007L4002W0`000P0007L4Y`=f1:D3
E`0002@0001e1000Y0<007H4002U0`000P0007H4Y@=e1:@3E`0002@0001c1000X`<007D4002T0`00
0P0007D4Y0=c1:<3E`0002@0002C1000WP<009D4002O0`000P0009<4W`>E19h3E`0002@0002E1000
V`<009L4002N0`000P0009D4WP>G19/3E`0002@0002G1000V`<009L4002^0`000P0009L4V`>G1:h3
E`0002@0002F1000W0<009H4002^0`000P0009H4W0>F1:h3E`0002@0002C1000[P<009/4002^0`00
0P0009<4[P>K1:h3E`0002@0002/1000X@<00:d4002T0`000P000:d4X@>/1:@3E`0002@0002Z1000
Y0<00:`4002V0`000P000:`4Y0>Z1:H3E`0002@0002W1000YP<00:X4002W0`000P000:X4YP>W1:L3
E`0002@0002W1000Y`<00:L4002W0`000P000:L4Y`>W1:L3E`0002@0002T1000YP<00:L4002W0`00
0P000:L4Y`>T1:H3E`0002@0002R1000Y0<00:@4002V0`000P000:@4YP>R1:@3E`0002@0002Q1000
X@<00:84002T0`000P000:84Y0>Q1:43E`0002@0002Q1000X0<00:44002Q0`000P000:44X@>Q1:03
E`0002@0002Q1000WP<00:84002P0`000P000:44X0>R19h3E`0002@0002R1000W0<00:@4002N0`00
0P000:84WP>T19`3E`0002@0002T1000V`<00:L4002L0`000P000:@4W0>W19/3E`0002@0002W1000
V`<00:P4002K0`000P000:L4V`>X19/3E`0002@0002X1000V`<00:/4002L0`000P000:P4V`>[19`3
E`0002@0002[1000W0<00:d4002N0`000P000:/4W0>]19h3E`0002@0002]1000WP<00:h4002P0`00
0P000:d4WP>^1:03E`0002@0002^1000X0<00:h4002V0`000P000:h4X0>^1:H3E`0002@0002]1000
YP<00:h4002Z0`000P000:h4YP>]1:X3E`0002@0002/1000ZP<00:d4002/0`000P000:d4ZP>/1:`3
E`0002@0002Z1000[0<00:`4002^0`000P000:`4[0>Z1:h3E`0002@0002W1000[P<00:X4002^0`00
0P000:X4[P>W1:h3E`0002@0002U1000[P<00:L4002^0`000P000:L4[P>U1:h3E`0002@0002S1000
[P<00:D4002^0`000P000:D4[P>S1:h3E`0002@0002R1000[0<00:<4002^0`000P000:<4[P>R1:`3
E`0002@0002R1000Z`<00:84002/0`000P000:84[0>R1:/3E`0002@0002R1000ZP<00:<4002[0`00
0P000:84Z`>S1:X3E`0002@0002S1000ZP<00:@4002[0`000P000:<4ZP>T1:/3E`0002@0002S1000
Z`<00:@4002/0`000P000:@4Z`>S1:`3E`0002@0002U1000YP<00:L4002W0`000P000:L4Y`>U1:H3
E`0002@0002S1000Y0<00:D4002V0`000P000:D4YP>S1:@3E`0002@0002R1000X@<00:<4002T0`00
0P000:<4Y0>R1:43E`0002@0002R1000X0<00:84002Q0`000P000:84X@>R1:03E`0002@0002R1000
WP<00:<4002P0`000P000:84X0>S19h3E`0002@0002S1000W0<00:D4002N0`000P000:<4WP>U19`3
E`0002@0002U1000V`<00:L4002L0`000P000:D4W0>W19/3E`0002@0002X1000V`<00:X4002L0`00
0P000:P4V`>Z19`3E`0002@0002Z1000W0<00:`4002N0`000P000:X4W0>/19h3E`0002@0002/1000
WP<00:d4002P0`000P000:`4WP>]1:03E`0002@0002]1000X0<00:d4002V0`000P000:d4X0>]1:H3
E`0002@0002/1000YP<00:d4002Z0`000P000:d4YP>/1:X3E`0002@0002[1000ZP<00:`4002/0`00
0P000:`4ZP>[1:`3E`0002@0002Y1000[0<00:/4002^0`000P000:/4[0>Y1:h3E`0002@0002W1000
[P<00:T4002^0`000P000:T4[P>W1:h3E`0002@000361000W`<00<L4002P0`000P000<H4W`?71:03
E`0002@000361000X0<00<L4002P0`000P000<L4X0?61:03E`0002@000351000X0<00<H4002P0`00
0P000<H4X0?51:03E`0002@000351000W`<00<D4002P0`000P000<D4X0?519l3E`0002@000351000
W@<00<H4002O0`000P000<D4W`?619d3E`0002@000361000W0<00<L4002M0`000P000<H4W@?719`3
E`0002@000371000V`<00<X4002L0`000P000<L4W0?:19/3E`0002@0003:1000V`<00<d4002K0`00
0P000<X4V`?=19/3E`0002@0003=1000V`<00=04002L0`000P000<d4V`?@19`3E`0002@0003@1000
W0<00=44002M0`000P000=04W0?A19d3E`0002@0003A1000W@<00=84002O0`000P000=44W@?B19l3
E`0002@0003B1000W`<00=84002P0`000P000=84W`?B1:03E`0002@0003A1000X0<00=84002R0`00
0P000=84X0?A1:83E`0002@0003>1000XP<00=44002T0`000P000=44XP?>1:@3E`0002@0003:1000
Y0<00<h4002V0`000P000<h4Y0?:1:H3E`0002@000381000YP<00<X4002W0`000P000<X4YP?81:L3
E`0002@000361000Y`<00<P4002Y0`000P000<P4Y`?61:T3E`0002@000351000Z@<00<H4002/0`00
0P000<H4Z@?51:`3E`0002@000351000[0<00<D4002^0`000P000<D4[0?51:h3E`0002@0003=1000
V`<00<l4002L0`000P000<d4V`??19`3E`0002@0003?1000W0<00=04002M0`000P000<l4W0?@19d3
E`0002@0003@1000W@<00=44002O0`000P000=04W@?A19l3E`0002@0003A1000W`<00=44002P0`00
0P000=44W`?A1:03E`0002@0003@1000X0<00=44002R0`000P000=44X0?@1:83E`0002@0003=1000
XP<00=04002T0`000P000=04XP?=1:@3E`0002@0003:1000Y0<00<d4002V0`000P000<d4Y0?:1:H3
E`0002@000351000[0<00<H4002]0`000P000<D4[@?61:`3E`0002@000361000[0<00<P4002/0`00
0P000<H4[0?81:`3E`0002@000381000[0<00<`4002^0`000P000<P4[0?<1:h3E`0002@0003<1000
[P<00<l4002^0`000P000<`4[P??1:h3E`0002@0003?1000[@<00=44002^0`000P000<l4[P?A1:d3
E`0002@0003A1000[0<00=84002]0`000P000=44[@?B1:`3E`0002@000381000[0<00<`4002^0`00
0P000<P4[0?<1:h3E`0002@0003<1000[P<00=04002^0`000P000<`4[P?@1:h3E`0002@0003@1000
[P<00=44002^0`000P000=04[P?A1:h3E`0002@0003A1000[0<00=84002^0`000P000=44[P?B1:`3
E`0002@0003B1000ZP<00=84002/0`000P000=84[0?B1:X3E`0002@0003J1000V`<00=d4002L0`00
0P000=d4V`?J19`3E`0002@0003H1000W0<00=X4002O0`000P000=X4W0?H19l3E`0002@0003G1000
W`<00=P4002S0`000P000=P4W`?G1:<3E`0002@0003G1000X`<00=L4002V0`000P000=L4X`?G1:H3
E`0002@0003G1000YP<00=P4002[0`000P000=L4YP?H1:/3E`0002@0003H1000Z`<00=X4002^0`00
0P000=P4Z`?J1:h3E`0002@0003J1000[P<00=d4002^0`000P000=X4[P?M1:h3E`0002@0003M1000
[P<00=h4002^0`000P000=d4[P?N1:h3E`0002@0003N1000[P<00>44002^0`000P000=h4[P?Q1:h3
E`0002@0003Q1000Z`<00><4002^0`000P000>44[P?S1:/3E`0002@0003S1000YP<00>@4002[0`00
0P000><4Z`?T1:H3E`0002@0003T1000X`<00>@4002V0`000P000>@4YP?T1:<3E`0002@0003S1000
W`<00>@4002S0`000P000>@4X`?S19l3E`0002@0003Q1000W0<00><4002O0`000P000><4W`?Q19`3
E`0002@0003N1000V`<00>44002L0`000P000>44W0?N19/3E`0002@0003M1000V`<00=h4002K0`00
0P000=h4V`?M19/3E`0002@0003K1000V`<00=d4002L0`000P000=d4V`?K19`3E`0002@0003J1000
W0<00=/4002M0`000P000=/4W0?J19d3E`0002@0003I1000W@<00=X4002O0`000P000=X4W@?I19l3
E`0002@0003H1000W`<00=T4002S0`000P000=T4W`?H1:<3E`0002@0003H1000X`<00=P4002V0`00
0P000=P4X`?H1:H3E`0002@0003H1000YP<00=T4002[0`000P000=P4YP?I1:/3E`0002@0003I1000
Z`<00=X4002]0`000P000=T4Z`?J1:d3E`0002@0003J1000[@<00=/4002^0`000P000=X4[@?K1:h3
E`0002@0003K1000[P<00=d4002^0`000P000=/4[P?M1:h3E`0002@0003N1000[P<00>04002^0`00
0P000=h4[P?P1:h3E`0002@0003P1000[@<00>44002^0`000P000>04[P?Q1:d3E`0002@0003Q1000
Z`<00>84002]0`000P000>44[@?R1:/3E`0002@0003R1000YP<00><4002[0`000P000>84Z`?S1:H3
E`0002@0003S1000X`<00><4002V0`000P000><4YP?S1:<3E`0002@0003R1000W`<00><4002S0`00
0P000><4X`?R19l3E`0002@0003Q1000W@<00>84002O0`000P000>84W`?Q19d3E`0002@0003P1000
W0<00>44002M0`000P000>44W@?P19`3E`0002@0003N1000V`<00>04002L0`000P000>04W0?N19/3
E`0002@0003E0000iP400=D4003V0@000P000=D0iP7E1>H19@0000`000080020:00000`000020000
9`0001P00002000000000000000000009@0000`000020000:00000`000010000EP000300003C0000
iP400=P0002Y0P001@000=<0iP7H0>H1f02Y0]<0Z@;C0>H19@0000`000080020EP00030000090@00
T00000h1003V0@001@0000h1iP490NH12@6@00h1T00>0NH19@0000`000080020EP000300000n0@00
iP4004<1002B0P001@0003h1iP530NH1@`6B0Sh1TP8n0NH19@0000`000080020EP000300001d0@00
iP4007T1000Q0P001@0007@1iP5i0NH1N@4Q0W@18@9d0NH19@0000`000080020EP000300002Z0@00
c@000:l1003V0@001@000:l1iP6Z0NH1ZP7=0:l1c@2_0NH19@0000`000080020EP000300003P0@00
iP400>D1000M0P001@000>01iP7U0NH1i@4M0^017@;P0NH19@0000`000080020EP000300000F0P00
N@4001/2003V0@001@0001/2iP4F0^H15P9i0A/2N@4K0^H19@0000`000080020EP000300001<0P00
iP400542001=0`001@0004`2iP5A0^H1D@9=0d`2C@=<0^H19@0000`000080020EP00030000220P00
S@4008L2003V0@001@0008L2iP620^H1PP:=0HL2S@670^H19@0000`000080020EP000300002h0P00
d0000;d2003V0@001@000;d2iP6h0^H1^0;@0;d2d02m0^H19@0000`000080020EP000300003^0P00
iP400?<2003M0P001@000>h2iP7c0^H1l`;M0^h2g@;^0^H19@0000`000080020EP000300000T0`00
iP4002T300160P001@0002@3iP4Y0nH1:@=60R@3AP8T0nH19@0000`000080020EP000300001J0`00
?@4005l3003V0@001@0005l3iP5J0nH1FP<m0El3?@5O0nH19@0000`000080020EP000300002@0`00
l@0009D3003V0@001@0009D3iP6@0nH1T0?a09D3l@2E0nH19@0000`000080020EP00030000360`00
iP400</3002S0P001@000<H3iP7;0nH1b`>S0/H3X`;60nH19@0000`000080020EP000300003l0`00
h`000044003V0@001@000044iP7l0nH1o0?S0044h`011>H19@0000`000080020EP000300000b1000
iP4003L400290P001@000384iP4g1>H1=`B90S84R@8b1>H19@0000`000080020EP000300001W1000
[@4006`4003V0@001@0006`4iP5W1>H1I`B]0F`4[@5/1>H19@0000`000080020EP000300002M1000
iP400:84003M0P001@0009d4iP6R1>H1XPCM0Yd4g@:M1>H19@0000`000080020EP000300003C1000
b0400=P4003V0@001@000=P4iP7C1>H1d`C80MP4b07H1>H19P0001`0000100000000004000000000
000002D0000<00000@0005/0002@0000cP0007H3003J0000P`<00040000K00006`000=X0O0?J07T3
f@1i0mP0N0?G07L3eP1g0mD0MP?C07L3d@1g0m00N0?@07T3c`1i0ll0N`?>07`3c`1n0ll0P0?@0843
d0220m40PP?B08<3e`230mL0PP?H0883f@210mX0P0?J07h3fP1l0e/0002@0000104004X0000@0@00
EP000040000K00006`000101D00@0Dd03`5<00h1B`0=0D/03@5:00P1BP070D/01P5;00H1C0050Dd0
1@5>00@1C`040E401@5B00D1D`060E@01P5E00L1E@080EH03@5F00d1E@0>0ED03`5D0101D`0@0E40
405@05/0002H0000>P4007H300160@00P`<00040000M00007@0004H1O0=60GX3A@5i0d@1N0=30GL3
@@5g0d01MP<o0GL3?@5g0c`1N0<l0GT3>`5i0c/1NP<j0G/3>P5n0c/1O`<k0H03?0610c`1PP<m0H83
?P630d81P`=30H83A0620dD1P@=50H03AP5o0dH1OP=60G`3F`0009P0001`0@00MP<007`100230`00
0@0001d0000M0000O05l0g`1NP=k0GT3NP5h0gT1M`=g0GL3MP5f0gD1M`=c0GL3LP5h0g81N@=a0GT3
L@5j0g01N`=`0Gh3L@5o0g41P0=b0H43LP620g<1PP=d0H<3N0630gT1PP=j0H83N`610g/1P0=l0Gl3
O05n0g`1O0=K0000T0000:H1001:0000/P4005H0000100006`0001/0002b0E00/P5>0;41C@2a0D`0
/05;0:l1B`2^0DX0ZP5:0:T1B`2X0D/0Y`5<0:L1CP2V0Dl0YP5A0:L1DP2W0E@0Z05E0:T1E@2Z0EH0
[P5F0:l1E@2`0ED0/@5D0;41D`2b0E80/P5A0;81D01K0000S0000=`1001f0`00j04008<300010000
6P0001X0003X0G`3j05j0nL1N@?V0GP3i@5g0n<1M`?R0GH3h@5g0ml1M`?N0GP3g@5i0m`1NP?L0Gl3
g@600md1P@?N0H83g`620n01P`?T0H<3i@620nH1PP?W0H43i`600nP1O`?X0Gh3j05l0e/0002<0000
4P8007H3000N0P00P`<00040000J00006P0001h2O0<N0WX37@9i0a`2N0<K0WL36@9g0aP2MP<G0WL3
5@9g0a@2N0<C0WT34P9j0a82O`<C0X034`:10a@2PP<E0X835P:30aX2P`<K0X8370:20ad2P@<M0X03
7P9o0ah2OP<N0W`3F`0008`000180P00MP<005@200230`000@0001X0000J0000E09l0e@2NP=C0WT3
DP9h0e42M`=?0WL3CP9f0dd2M`=;0WL3BP9h0dT2N@=80WX3B09o0dT2P0=90X43BP:20d/2PP=<0X<3
D0:30e42PP=B0X83D`:10e<2P0=D0Wl3E09n0e@2O0=K0000U00007h2001:0000RP8005H000010000
700001`0002:0U00RP9?08T2CP290Td0R09<08P2B`270T/0QP9:0882BP210T/0P09;07l2C01o0Td0
OP9>07h2DP1o0U<0O`9D0802E@210UD0PP9F08H2EP270UD0R09E08P2E0290U<0R@9B08X2D@2:0U00
F`0009@0002d0P00BP000<02001F00000@0001`0000L0000`09@0<02C`2o0Th0_`9=0;h2C02n0T/0
_@9;0;`2BP2h0TX0]`9;0;H2B`2e0T`0]@9=0;@2CP2d0U80]@9C0;D2E02f0UD0]`9E0;P2EP2l0UH0
_@9E0;h2E@2n0U@0_`9C0;l2DP300U40`09@05/0002D0000jP8007H3003f0P00P`<00040000L0000
70000?H2O0?f0W/3m@9j0oD2N@?d0WT3m09h0o<2M`?a0WL3l09f0nh2M`?]0WL3k09h0n/2N@?Z0WX3
jP9o0n/2P0?[0X43k0:20nd2PP?]0X<3lP:30o<2PP?d0X83m0:10oD2P0?e0Wl3mP9n0oH2O0=K0000
R0000203001f0`00;0<008<3000100006@0001T0000/0g`3:`=k0b/3N@<Z0gT3:P=h0bT3M`<W0gL3
9P=f0b@3M`<S0gL38P=h0b43N@<P0gT380>00b43P@<R0h838`>20b<3P`<X0h<3:@>20bX3PP<Z0h43
:`>00b/3OP</0g`3F`0007@0001F0`00BP000683001F00000@0001@0000D0000HP=@0643C`1Q0dd0
H0=<05l3B`1N0dX0F@=:05P3B`1G0d`0EP==05H3D`1G0e@0F0=E05T3EP1N0eH0G`=E0603E01Q0e<0
H@=A0683D01K0000L00008`3001:0000U`<005H0000100004`0001<0002G0e00U`==09H3C02E0d/0
U0=:08l3BP2>0d/0S@=<08`3C@2<0e<0S@=D08h3E@2?0eH0U0=F09D3E@2F0e@0U`=C09L3D@2G0e00
F`0008P000310`00MP<00<d300230`000@0001T0000I0000c@=l0ld3N@?<0gT3b`=h0l/3M`?90gL3
b0=f0lH3M`?40gL3a0=h0l<3N@?20gT3`P=k0l43O0?20gh3`P>00l<3P@?40h83a@>30lX3P`?;0h83
c0>10ld3P0?=0gh3c@=l0e/0002D0000m`<007H300031000P`<00040000L0000700000<4O0<317T3
0PAi0`44N0<017L3o`=g0oh3MP?l0gL3nP=g0oT3N0?i0gT3n0=i0oP3NP?g0g/3m`=n0oP3O`?h0h03
n@>10oT3PP?j0h83n`>30`04P`<018830@B20`84P@<318030`An0`<4O0=K0000V00002d4001f0`00
>@@008<3000100007@0001d0000i17`3>@Aj0cP4N@<g17P3=PAg0cD4M`<d17H3<PAg0c04M`<_17P3
;`Ai0bh4N@<^17X3;@Ak0bd4OP<^17l3;PB00bl4P@<_1883<0B20c44P`<e18<3=PB20cL4PP<h1843
>0B00cT4O`<i17h3>@Al0e/0002H0000H`@004X0001_1000EP000040000M00007@0006l4D01_14h0
KPA=06h4C01]14/0K0A;06/4BP1W14X0IPA;06D4B`1U14`0I0A=06@4CP1S14l0H`AA06@4DP1T15<0
I@AD06D4E@1V15D0I`AF06/4EP1/15D0K@AE06h4E01^15<0K`AB06l4D@1_1500F`0009P0002I1000
MP<00:D400230`000@0001d0000M0000Y@Al0jD4NP>T17T3X`Ah0j84M`>P17L3W`Af0ih4M`>L17L3
V`Ah0i/4N@>J17T3VPAj0iT4N`>I17h3VPAo0iX4P0>K1843V`B20i`4PP>M18<3X@B30j84PP>S1883
Y0B10j@4P0>U17l3Y@An0jD4O0=K0000T0000<l4001:0000f`@005H0000100006`0001/0003K1500
f`A>0=X4C@3J14`0f@A;0=P4B`3G14X0d`A:0=84B`3A14/0d0A<0=04C@3?14h0c`AB0=04D`3@15@0
d@AE0=84E@3C15H0e`AF0=P4E@3I15D0fPAD0=X4D`3K1580f`AA0=/4D01G0000=0000=0000180000
fP@008D300060000d0240m00B03J14P0fPB40m00Q0?@08D39@0000`000050020:00000`000020000
9@0000`000080020:00000`0000100009@0000`0000=00203P0001@000000000400001@0
\>"], "Graphics",
 ImageSize->{242.25, 191.813},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}}],

Cell["\<\
The buy/sell classification accuracy here is weaker. There are more \
directionally incorrect inferences, and the directions that are correctly \
identified are weaker.  Just as we'd have a harder time picking out the buys \
and sells visually, the sampler has a tougher time classifying trades.\
\>", "Text"],

Cell[TextData[{
 "We'll now try things with ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["\[Sigma]", "u"], "=", "0.01"}]]],
 " and ",
 Cell[BoxData[
  RowBox[{"c", "=", "0.001"}]]],
 ". Here's the price path:"
}], "Text"],

Cell[GraphicsData["Metafile", "\<\
CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@000DZh0@0006`00000000000000?D4003/0`00
000000000037FP00idL00215CDH00040^4X100X9000400000000000000000000@0H00;04000U0@00
g000000000000000000008Qh101PF`<02@000100003f1000k@<000X0000@000000000000000D0000
300000d0000U0000300000P0080W00006000004000000000oooo0000000U000030000040000[0000
6000000000000000mP@00>d3000[00006000000000000000mP@00>d3000V00007000008000000000
0@000000000000009@0000`000020000E`0002@000210@002P000881000=00000P0008413@220@X0
E`0002@000220@002P000881000?00000P0008812P220@l0E`0002@000210@003@000881000?0000
0P0008813`210@d0E`0002@0001o0@002`000841000=00000P0008413@1o0@/0E`0002@0001l0@00
2P0007l1000;00000P0007l12`1l0@X0E`0002@0001j0@002P0007`1000:00000P0007`12P1j0@X0
E`0002@0001g0@002P0007X1000;00000P0007X12P1g0@/0E`0002@0001e0@002`0007L1000=0000
0P0007L12`1e0@d0E`0002@0001e0@003@0007D1000?00000P0007D13@1e0@l0E`0002@0001e0@00
3`0007H1000@00000P0007D13`1f0A00E`0002@0001f0@00400007L1000A00000P0007H1401g0A40
E`0002@0001g0@004@0007T1000B00000P0007L14@1i0A80E`0002@0001i0@004P0007h1000D0000
0P0007T14P1n0A@0E`0002@0001n0@0050000801000E00000P0007h150200AD0E`0002@000200@00
5@000881000G00000P0008015@220AL0E`0002@0001e0@003`0007L1000@00000P0007D13`1g0A00
E`0002@0001g0@00400007T1000A00000P0007L1401i0A40E`0002@0001i0@004@0007h1000C0000
0P0007T14@1n0A<0E`0002@0001n0@004`000801000D00000P0007h14`200A@0E`0002@000200@00
50000841000E00000P00080150210AD0E`0002@000210@005@000881000G00000P0008415@220AL0
E`0002@000220@005`000881000K00000P0008815`220A/0E`0002@000200@006`000881000M0000
0P0008816`200Ad0E`0002@0001m0@007@000801000M00000P0008017@1m0Ad0E`0002@0001k0@00
7@0007d1000M00000P0007d17@1k0Ad0E`0002@0001h0@007@0007/1000M00000P0007/17@1h0Ad0
E`0002@0001f0@006`0007P1000M00000P0007P17@1f0A/0E`0002@0001e0@00600007H1000K0000
0P0007H16`1e0AP0E`0002@0001e0@00600007D1000M00000P0007D1601e0Ad0E`0002@0001e0@00
6`0007H1000M00000P0007D17@1f0A/0E`0002@000270@002P0008P1000;00000P0008P12P270@/0
E`0002@000270@002`0008P1000<00000P0008L12`280@`0E`0002@000280@002`0008T1000<0000
0P0008P130290@/0E`0002@000280@002P0008T1000;00000P0008T12`280@X0E`0002@000280@00
400008P1000M00000P0008P140280Ad0E`0002@000290@00400008T1000M00000P0008T140290Ad0
E`0002@000250@00400008T1000@00000P0008D140290A00E`0002@000250@007@0008`1000M0000
0P0008D17@2<0Ad0E`0002@0002B0@0040000981000M00000P000981402B0Ad0E`0002@0002C0@00
400009<1000M00000P0009<1402C0Ad0E`0002@0002C0@004@0009D1000C00000P0009<14`2E0A40
E`0002@0002E0@00400009L1000A00000P0009D14@2G0A00E`0002@0002G0@00400009T1000@0000
0P0009L1402I0A00E`0002@0002I0@00400009`1000A00000P0009T1402L0A40E`0002@0002L0@00
4@0009d1000C00000P0009`14@2M0A<0E`0002@0002M0@004`0009d1000M00000P0009d14`2M0Ad0
E`0002@0002I0@00400009/1000A00000P0009T1402K0A40E`0002@0002K0@004@0009`1000C0000
0P0009/14@2L0A<0E`0002@0002L0@004`0009`1000M00000P0009`14`2L0Ad0E`0002@0002M0@00
4@0009h1000C00000P0009d14`2N0A40E`0002@0002N0@0040000:41000A00000P0009h14@2Q0A00
E`0002@0002Q0@0040000:<1000@00000P000:41402S0A00E`0002@0002S0@0040000:H1000A0000
0P000:<1402V0A40E`0002@0002V0@004@000:L1000C00000P000:H14@2W0A<0E`0002@0002W0@00
4`000:L1000M00000P000:L14`2W0Ad0E`0002@0002S0@0040000:D1000A00000P000:<1402U0A40
E`0002@0002U0@004@000:H1000C00000P000:D14@2V0A<0E`0002@0002V0@004`000:H1000M0000
0P000:H14`2V0Ad0E`0002@0002?0@00400009<1000@00000P0008l1402C0A00E`0002@0002?0@00
7@0009H1000M00000P0008l17@2F0Ad0E`0002@0002I0@007@0009l1000M00000P0009T17@2O0Ad0
E`0002@0002S0@007@000:T1000M00000P000:<17@2Y0Ad0E`0002@0002`0@0040000;01000K0000
0P000;01402`0A/0E`0002@0002`0@006`000;41000M00000P000;016`2a0Ad0E`0002@0002a0@00
7@000;@1000M00000P000;417@2d0Ad0E`0002@0002d0@007@000;D1000M00000P000;@17@2e0Ad0
E`0002@0002e0@007@000;P1000M00000P000;D17@2h0Ad0E`0002@0002h0@006`000;X1000M0000
0P000;P17@2j0A/0E`0002@0002a0@0040000;41000K00000P000;41402a0A/0E`0002@0002a0@00
6`000;81000M00000P000;416`2b0Ad0E`0002@0002b0@007@000;@1000M00000P000;817@2d0Ad0
E`0002@0002j0@0040000;X1000M00000P000;X1402j0Ad0E`0002@0002k0@0040000;/1000M0000
0P000;/1402k0Ad0E`0002@0002]0@0040000;41000@00000P000:d1402a0A00E`0002@0002g0@00
40000;/1000@00000P000;L1402k0A00E`0002@0002j0@007@000;d1000M00000P000;X17@2m0Ad0
E`0002@000330@002P000<<1000M00000P000<<12P330Ad0E`0002@000340@002P000<@1000M0000
0P000<@12P340Ad0E`0002@000300@002P000<@1000:00000P000<012P340@X0E`0002@000300@00
7@000<L1000M00000P000<017@370Ad0E`0002@0003=0@004P000<d1000C00000P000<d14P3=0A<0
E`0002@0003<0@004`000<d1000C00000P000<d14`3<0A<0E`0002@0003<0@004P000<`1000C0000
0P000<`14`3<0A80E`0002@0003<0@004@000<d1000B00000P000<`14P3=0A40E`0002@0003=0@00
40000<l1000A00000P000<d14@3?0A00E`0002@0003?0@0040000=81000@00000P000<l1403B0A00
E`0002@0003B0@0040000=@1000A00000P000=81403D0A40E`0002@0003D0@004@000=D1000B0000
0P000=@14@3E0A80E`0002@0003E0@004P000=H1000D00000P000=D14P3F0A@0E`0002@0003F0@00
50000=H1000K00000P000=H1503F0A/0E`0002@0003F0@006`000=L1000M00000P000=H16`3G0Ad0
E`0002@0003G0@007@000=P1000M00000P000=L17@3H0Ad0E`0002@0003E0@004P000=D1000K0000
0P000=D14P3E0A/0E`0002@0003E0@006`000=H1000M00000P000=D16`3F0Ad0E`0002@0003F0@00
7@000=P1000M00000P000=H17@3H0Ad0E`0002@0003H0@007@000=T1000M00000P000=P17@3I0Ad0
E`0002@0003D0@0050000=D1000E00000P000=D1503D0AD0E`0002@0003?0@005@000=@1000F0000
0P000=@15@3?0AH0E`0002@0003<0@005P000<l1000G00000P000<l15P3<0AL0E`0002@0003;0@00
5`000<`1000I00000P000<`15`3;0AT0E`0002@0003;0@006@000</1000K00000P000</16@3;0A/0
E`0002@0003;0@006`000<`1000M00000P000</16`3<0Ad0E`0002@0003<0@007@000<l1000M0000
0P000<`17@3?0Ad0E`0002@0003?0@007@000=41000M00000P000<l17@3A0Ad0E`0002@0003A0@00
7@000=<1000M00000P000=417@3C0Ad0E`0002@0003C0@006`000=D1000M00000P000=<17@3E0A/0
E`0002@0003=0@005P000<l1000G00000P000<l15P3=0AL0E`0002@0003<0@005`000<d1000I0000
0P000<d15`3<0AT0E`0002@0003<0@006@000<`1000K00000P000<`16@3<0A/0E`0002@0003<0@00
6`000<d1000M00000P000<`16`3=0Ad0E`0002@0003=0@007@000<l1000M00000P000<d17@3?0Ad0
E`0002@0003O0@002P000=l1000J00000P000=l12P3O0AX0E`0002@0003O0@006P000>01000M0000
0P000=l16P3P0Ad0E`0002@0003P0@007@000>81000M00000P000>017@3R0Ad0E`0002@0003R0@00
7@000><1000M00000P000>817@3S0Ad0E`0002@0003S0@007@000>D1000M00000P000><17@3U0Ad0
E`0002@0003U0@006`000>H1000M00000P000>D17@3V0A/0E`0002@0003P0@002P000>01000J0000
0P000>012P3P0AX0E`0002@0003P0@006P000>41000M00000P000>016P3Q0Ad0E`0002@0003Q0@00
7@000>81000M00000P000>417@3R0Ad0E`0002@0003L0@0040000><1000@00000P000=`1403S0A00
E`0002@0003[0@005P000?H1000F00000P000>/15P3f0AH0E`0002@0003f0@0050000?H1000F0000
0P000?H15P3f0A@0E`0002@0003e0@004P000?H1000D00000P000?H1503e0A80E`0002@0003d0@00
4@000?D1000B00000P000?D14P3d0A40E`0002@0003b0@0040000?@1000A00000P000?@14@3b0A00
E`0002@0003`0@0040000?81000@00000P000?81403`0A00E`0002@0003]0@0040000?01000A0000
0P000?01403]0A40E`0002@0003[0@004@000>d1000C00000P000>d14@3[0A<0E`0002@0003Z0@00
4`000>/1000F00000P000>/14`3Z0AH0E`0002@0003Z0@005P000>X1000H00000P000>X15P3Z0AP0
E`0002@0003Z0@0060000>/1000K00000P000>X1603[0A/0E`0002@0003[0@006`000>d1000M0000
0P000>/16`3]0Ad0E`0002@0003]0@007@000?01000M00000P000>d17@3`0Ad0E`0002@0003`0@00
7@000?41000M00000P000?017@3a0Ad0E`0002@0003a0@007@000?@1000M00000P000?417@3d0Ad0
E`0002@0003d0@006`000?H1000M00000P000?@17@3f0A/0E`0002@0003e0@004`000?D1000F0000
0P000?D15P3e0A<0E`0002@0003d0@004@000?D1000C00000P000?D14`3d0A40E`0002@0003^0@00
40000?01000A00000P000?01403^0A40E`0002@0003/0@004@000>h1000C00000P000>h14@3/0A<0
E`0002@0003[0@004`000>`1000F00000P000>`14`3[0AH0E`0002@0003[0@005P000>/1000H0000
0P000>/15P3[0AP0E`0002@0003[0@0060000>`1000K00000P000>/1603/0A/0E`0002@0003/0@00
6`000>h1000M00000P000>`16`3^0Ad0E`0002@0003^0@007@000?01000M00000P000>h17@3`0Ad0
E`0002@000060P002P0000H2000M00000P0000H22P060Qd0E`0002@000070P002P0000L2000M0000
0P0000L22P070Qd0E`0002@000040P004@0000H2000C00000P0000H24`040Q40E`0002@000020P00
400000@2000A00000P0000@24@020Q00E`0002@000010P0040000082000@00000P00008240010Q00
E`0002@0003n0@0040000042000A00000P000042403n0A40E`0002@0003l0@004@000?h1000C0000
0P000?h14@3l0A<0E`0002@0003k0@004`000?`1000F00000P000?`14`3k0AH0E`0002@0003k0@00
5P000?/1000H00000P000?/15P3k0AP0E`0002@0003k0@0060000?`1000K00000P000?/1603l0A/0
E`0002@0003l0@006`000?h1000M00000P000?`16`3n0Ad0E`0002@0003n0@007@000042000M0000
0P000?h17@010Qd0E`0002@000010P007@000082000M00000P0000427@020Qd0E`0002@000020P00
7@0000@2000M00000P0000827@040Qd0E`0002@000040P006`0000H2000M00000P0000@27@060Q/0
E`0002@0003o0@0040000042000A00000P000042403o0A40E`0002@0003m0@004@000?l1000C0000
0P000?l14@3m0A<0E`0002@0003l0@004`000?d1000F00000P000?d14`3l0AH0E`0002@0003l0@00
5P000?`1000H00000P000?`15P3l0AP0E`0002@0003l0@0060000?d1000K00000P000?`1603m0A/0
E`0002@0003m0@006`000?l1000M00000P000?d16`3o0Ad0E`0002@0003o0@007@000042000M0000
0P000?l17@010Qd0E`0002@000030P002P0000L2000:00000P0000<22P070PX0E`0002@000060P00
7@0000T2000M00000P0000H27@090Qd0E`0002@0000J0P00400001X2000T00000P0001X2400J0R@0
E`0002@0000K0P00400001/2000T00000P0001/2400K0R@0E`0002@0000K0P004@0001d2000C0000
0P0001/24`0M0Q40E`0002@0000M0P00400001h2000A00000P0001d24@0N0Q00E`0002@0000N0P00
40000202000@00000P0001h2400P0Q00E`0002@0000P0P00400002<2000A00000P000202400S0Q40
E`0002@0000S0P004@0002D2000C00000P0002<24@0U0Q<0E`0002@0000U0P004`0002H2000F0000
0P0002D24`0V0QH0E`0002@0000V0P005P0002H2000H00000P0002H25P0V0QP0E`0002@0000U0P00
600002H2000K00000P0002H2600U0Q/0E`0002@0000S0P006`0002D2000M00000P0002D26`0S0Qd0
E`0002@0000P0P007@0002<2000M00000P0002<27@0P0Qd0E`0002@0000N0P007@000202000M0000
0P0002027@0N0Qd0E`0002@0000M0P007@0001h2000M00000P0001h27@0M0Qd0E`0002@0000K0P00
6`0001d2000M00000P0001d27@0K0Q/0E`0002@0000P0P0040000282000A00000P000202400R0Q40
E`0002@0000R0P004@0002@2000C00000P0002824@0T0Q<0E`0002@0000T0P004`0002D2000F0000
0P0002@24`0U0QH0E`0002@0000U0P005P0002D2000H00000P0002D25P0U0QP0E`0002@0000T0P00
600002D2000K00000P0002D2600T0Q/0E`0002@0000R0P006`0002@2000M00000P0002@26`0R0Qd0
E`0002@0000P0P007@000282000M00000P0002827@0P0Qd0E`0002@0000G0P00400001/2000@0000
0P0001L2400K0Q00E`0002@0000G0P00900001h2000T00000P0001L2900N0R@0E`0002@0000]0P00
400002d2000M00000P0002d2400]0Qd0E`0002@0000^0P00400002h2000M00000P0002h2400^0Qd0
E`0002@0000^0P004`0002l2000F00000P0002h25P0_0Q<0E`0002@0000_0P004@000342000C0000
0P0002l24`0a0Q40E`0002@0000a0P0040000382000A00000P0003424@0b0Q00E`0002@0000b0P00
400003D2000@00000P000382400e0Q00E`0002@0000e0P00400003H2000A00000P0003D2400f0Q40
E`0002@0000f0P004@0003H2000B00000P0003H24@0f0Q80E`0002@0000e0P004P0003H2000C0000
0P0003H24P0e0Q<0E`0002@0000d0P004P0003D2000C00000P0003D24`0d0Q80E`0002@0000d0P00
4@0003D2000B00000P0003@24P0e0Q40E`0002@0000Z0P00400002h2000@00000P0002X2400^0Q00
E`0002@0000Z0P007@000342000M00000P0002X27@0a0Qd0E`0002@0000k0P002P0003`2000;0000
0P0003`22P0k0P/0E`0002@0000k0P002`0003`2000<00000P0003/22`0l0P`0E`0002@0000l0P00
2`0003d2000<00000P0003`2300m0P/0E`0002@0000l0P002P0003d2000;00000P0003d22`0l0PX0
E`0002@0000l0P00400003`2000M00000P0003`2400l0Qd0E`0002@0000m0P00400003d2000M0000
0P0003d2400m0Qd0E`0002@0000i0P00400003d2000@00000P0003T2400m0Q00E`0002@0000i0P00
7@000402000M00000P0003T27@100Qd0E`0002@0001>0P004`0004l2000D00000P0004l24`1>0Q@0
E`0002@0001>0P00500004l2000E00000P0004h2501?0QD0E`0002@0001?0P0050000502000E0000
0P0004l25@1@0Q@0E`0002@0001@0P004`000502000D00000P000502501@0Q<0E`0002@0001>0P00
4@000502000C00000P0005024`1>0Q40E`0002@0001<0P00400004h2000A00000P0004h24@1<0Q00
E`0002@0001:0P00400004`2000@00000P0004`2401:0Q00E`0002@000170P00400004X2000A0000
0P0004X240170Q40E`0002@000150P004@0004L2000C00000P0004L24@150Q<0E`0002@000140P00
4`0004D2000F00000P0004D24`140QH0E`0002@000140P005P0004@2000H00000P0004@25P140QP0
E`0002@000140P00600004D2000K00000P0004@260150Q/0E`0002@000150P006`0004L2000M0000
0P0004D26`170Qd0E`0002@000170P007@0004X2000M00000P0004L27@1:0Qd0E`0002@0001:0P00
7@0004/2000M00000P0004X27@1;0Qd0E`0002@0001;0P007@0004h2000M00000P0004/27@1>0Qd0
E`0002@0001>0P006`000502000M00000P0004h27@1@0Q/0E`0002@000180P00400004X2000A0000
0P0004X240180Q40E`0002@000160P004@0004P2000C00000P0004P24@160Q<0E`0002@000150P00
4`0004H2000F00000P0004H24`150QH0E`0002@000150P005P0004D2000H00000P0004D25P150QP0
E`0002@000150P00600004H2000K00000P0004D260160Q/0E`0002@000160P006`0004P2000M0000
0P0004H26`180Qd0E`0002@000180P007@0004X2000M00000P0004P27@1:0Qd0E`0002@0001E0P00
5P000602000F00000P0005D25P1P0QH0E`0002@0001P0P0050000602000F00000P0006025P1P0Q@0
E`0002@0001O0P004P000602000D00000P000602501O0Q80E`0002@0001N0P004@0005l2000B0000
0P0005l24P1N0Q40E`0002@0001L0P00400005h2000A00000P0005h24@1L0Q00E`0002@0001J0P00
400005`2000@00000P0005`2401J0Q00E`0002@0001G0P00400005X2000A00000P0005X2401G0Q40
E`0002@0001E0P004@0005L2000C00000P0005L24@1E0Q<0E`0002@0001D0P004`0005D2000F0000
0P0005D24`1D0QH0E`0002@0001D0P005P0005@2000H00000P0005@25P1D0QP0E`0002@0001D0P00
600005D2000K00000P0005@2601E0Q/0E`0002@0001E0P006`0005L2000M00000P0005D26`1G0Qd0
E`0002@0001G0P007@0005X2000M00000P0005L27@1J0Qd0E`0002@0001J0P007@0005/2000M0000
0P0005X27@1K0Qd0E`0002@0001K0P007@0005h2000M00000P0005/27@1N0Qd0E`0002@0001N0P00
6`000602000M00000P0005h27@1P0Q/0E`0002@0001O0P004`0005l2000F00000P0005l25P1O0Q<0
E`0002@0001N0P004@0005l2000C00000P0005l24`1N0Q40E`0002@0001H0P00400005X2000A0000
0P0005X2401H0Q40E`0002@0001F0P004@0005P2000C00000P0005P24@1F0Q<0E`0002@0001E0P00
4`0005H2000F00000P0005H24`1E0QH0E`0002@0001E0P005P0005D2000H00000P0005D25P1E0QP0
E`0002@0001E0P00600005H2000K00000P0005D2601F0Q/0E`0002@0001F0P006`0005P2000M0000
0P0005H26`1H0Qd0E`0002@0001H0P007@0005X2000M00000P0005P27@1J0Qd0E`0002@0001a0P00
40000742000T00000P000742401a0R@0E`0002@0001b0P0040000782000T00000P000782401b0R@0
E`0002@0001b0P004@0007@2000C00000P0007824`1d0Q40E`0002@0001d0P00400007D2000A0000
0P0007@24@1e0Q00E`0002@0001e0P00400007L2000@00000P0007D2401g0Q00E`0002@0001g0P00
400007X2000A00000P0007L2401j0Q40E`0002@0001j0P004@0007`2000C00000P0007X24@1l0Q<0
E`0002@0001l0P004`0007d2000F00000P0007`24`1m0QH0E`0002@0001m0P005P0007d2000H0000
0P0007d25P1m0QP0E`0002@0001l0P00600007d2000K00000P0007d2601l0Q/0E`0002@0001j0P00
6`0007`2000M00000P0007`26`1j0Qd0E`0002@0001g0P007@0007X2000M00000P0007X27@1g0Qd0
E`0002@0001e0P007@0007L2000M00000P0007L27@1e0Qd0E`0002@0001d0P007@0007D2000M0000
0P0007D27@1d0Qd0E`0002@0001b0P006`0007@2000M00000P0007@27@1b0Q/0E`0002@0001g0P00
400007T2000A00000P0007L2401i0Q40E`0002@0001i0P004@0007/2000C00000P0007T24@1k0Q<0
E`0002@0001k0P004`0007`2000F00000P0007/24`1l0QH0E`0002@0001l0P005P0007`2000H0000
0P0007`25P1l0QP0E`0002@0001k0P00600007`2000K00000P0007`2601k0Q/0E`0002@0001i0P00
6`0007/2000M00000P0007/26`1i0Qd0E`0002@0001g0P007@0007T2000M00000P0007T27@1g0Qd0
E`0002@0001^0P0040000782000@00000P0006h2401b0Q00E`0002@0001^0P00900007D2000T0000
0P0006h2901e0R@0E`0002@000240P004P0008@2000C00000P0008@24P240Q<0E`0002@000230P00
4`0008@2000C00000P0008@24`230Q<0E`0002@000230P004P0008<2000C00000P0008<24`230Q80
E`0002@000230P004@0008@2000B00000P0008<24P240Q40E`0002@000240P00400008H2000A0000
0P0008@24@260Q00E`0002@000260P00400008T2000@00000P0008H240290Q00E`0002@000290P00
400008/2000A00000P0008T2402;0Q40E`0002@0002;0P004@0008`2000B00000P0008/24@2<0Q80
E`0002@0002<0P004P0008d2000D00000P0008`24P2=0Q@0E`0002@0002=0P00500008d2000K0000
0P0008d2502=0Q/0E`0002@0002=0P006`0008h2000M00000P0008d26`2>0Qd0E`0002@0002>0P00
7@0008l2000M00000P0008h27@2?0Qd0E`0002@0002<0P004P0008`2000K00000P0008`24P2<0Q/0
E`0002@0002<0P006`0008d2000M00000P0008`26`2=0Qd0E`0002@0002=0P007@0008l2000M0000
0P0008d27@2?0Qd0E`0002@0002?0P007@000902000M00000P0008l27@2@0Qd0E`0002@0002;0P00
500008`2000E00000P0008`2502;0QD0E`0002@000260P005@0008/2000F00000P0008/25@260QH0
E`0002@000230P005P0008H2000G00000P0008H25P230QL0E`0002@000220P005`0008<2000I0000
0P0008<25`220QT0E`0002@000220P006@000882000K00000P0008826@220Q/0E`0002@000220P00
6`0008<2000M00000P0008826`230Qd0E`0002@000230P007@0008H2000M00000P0008<27@260Qd0
E`0002@000260P007@0008P2000M00000P0008H27@280Qd0E`0002@000280P007@0008X2000M0000
0P0008P27@2:0Qd0E`0002@0002:0P006`0008`2000M00000P0008X27@2<0Q/0E`0002@000240P00
5P0008H2000G00000P0008H25P240QL0E`0002@000230P005`0008@2000I00000P0008@25`230QT0
E`0002@000230P006@0008<2000K00000P0008<26@230Q/0E`0002@000230P006`0008@2000M0000
0P0008<26`240Qd0E`0002@000240P007@0008H2000M00000P0008@27@260Qd0E`0002@0002F0P00
2P0009H2000J00000P0009H22P2F0QX0E`0002@0002F0P006P0009L2000M00000P0009H26P2G0Qd0
E`0002@0002G0P007@0009T2000M00000P0009L27@2I0Qd0E`0002@0002I0P007@0009X2000M0000
0P0009T27@2J0Qd0E`0002@0002J0P007@0009`2000M00000P0009X27@2L0Qd0E`0002@0002L0P00
6`0009d2000M00000P0009`27@2M0Q/0E`0002@0002G0P002P0009L2000J00000P0009L22P2G0QX0
E`0002@0002G0P006P0009P2000M00000P0009L26P2H0Qd0E`0002@0002H0P007@0009T2000M0000
0P0009P27@2I0Qd0E`0002@0002C0P00400009X2000@00000P0009<2402J0Q00E`0002@0002S0P00
2P000:<2000M00000P000:<22P2S0Qd0E`0002@0002T0P002P000:@2000M00000P000:@22P2T0Qd0
E`0002@0002T0P004@000:H2000C00000P000:@24`2V0Q40E`0002@0002V0P0040000:P2000A0000
0P000:H24@2X0Q00E`0002@0002X0P0040000:X2000@00000P000:P2402Z0Q00E`0002@0002Z0P00
40000:d2000A00000P000:X2402]0Q40E`0002@0002]0P004@000:h2000C00000P000:d24@2^0Q<0
E`0002@0002^0P004`000:h2000M00000P000:h24`2^0Qd0E`0002@0002Z0P0040000:`2000A0000
0P000:X2402/0Q40E`0002@0002/0P004@000:d2000C00000P000:`24@2]0Q<0E`0002@0002]0P00
4`000:d2000M00000P000:d24`2]0Qd0E`0002@0002P0P002P000:@2000:00000P000:022P2T0PX0
E`0002@0002P0P007@000:L2000M00000P000:027@2W0Qd0E`0002@0002Z0P007@000;02000M0000
0P000:X27@2`0Qd0E`0002@000300P0040000<@2000M00000P000<0240340Qd0E`0002@000310P00
40000<@2000K00000P000<4240340Q/0E`0002@000340P0040000<L2000M00000P000<L240340Qd0
E`0002@000370P0040000</2000M00000P000<L2403;0Qd0E`0002@000380P0040000</2000K0000
0P000<P2403;0Q/0E`0002@0003;0P0040000<h2000M00000P000<h2403;0Qd0E`0002@0002m0P00
40000<@2000@00000P000;d240340Q00E`0002@0003<0P0040000=42000@00000P000<`2403A0Q00
E`0002@0003E0P002P000=H2000;00000P000=H22P3E0P/0E`0002@0003E0P002`000=H2000<0000
0P000=D22`3F0P`0E`0002@0003F0P002`000=L2000<00000P000=H2303G0P/0E`0002@0003F0P00
2P000=L2000;00000P000=L22`3F0PX0E`0002@0003F0P0040000=H2000M00000P000=H2403F0Qd0
E`0002@0003G0P0040000=L2000M00000P000=L2403G0Qd0E`0002@0003C0P0040000=L2000@0000
0P000=<2403G0Q00E`0002@0003C0P007@000=X2000M00000P000=<27@3J0Qd0E`0002@0003P0P00
2P000>02000J00000P000>022P3P0QX0E`0002@0003P0P006P000>42000M00000P000>026P3Q0Qd0
E`0002@0003Q0P007@000><2000M00000P000>427@3S0Qd0E`0002@0003S0P007@000>@2000M0000
0P000><27@3T0Qd0E`0002@0003T0P007@000>H2000M00000P000>@27@3V0Qd0E`0002@0003V0P00
6`000>L2000M00000P000>H27@3W0Q/0E`0002@0003Q0P002P000>42000J00000P000>422P3Q0QX0
E`0002@0003Q0P006P000>82000M00000P000>426P3R0Qd0E`0002@0003R0P007@000><2000M0000
0P000>827@3S0Qd0E`0002@0003M0P0040000>@2000@00000P000=d2403T0Q00E`0002@0003^0P00
2P000>h2000M00000P000>h22P3^0Qd0E`0002@0003_0P002P000>l2000M00000P000>l22P3_0Qd0
E`0002@0003_0P004@000?42000C00000P000>l24`3a0Q40E`0002@0003a0P0040000?<2000A0000
0P000?424@3c0Q00E`0002@0003c0P0040000?D2000@00000P000?<2403e0Q00E`0002@0003e0P00
40000?P2000A00000P000?D2403h0Q40E`0002@0003h0P004@000?T2000C00000P000?P24@3i0Q<0
E`0002@0003i0P004`000?T2000M00000P000?T24`3i0Qd0E`0002@0003e0P0040000?L2000A0000
0P000?D2403g0Q40E`0002@0003g0P004@000?P2000C00000P000?L24@3h0Q<0E`0002@0003h0P00
4`000?P2000M00000P000?P24`3h0Qd0E`0002@0003[0P002P000>l2000:00000P000>/22P3_0PX0
E`0002@0003[0P007@000?82000M00000P000>/27@3b0Qd0E`0002@0003e0P007@000?/2000M0000
0P000?D27@3k0Qd0E`0002@0000C0`004`0001@3000D00000P0001@34`0C0a@0E`0002@0000C0`00
500001@3000E00000P0001<3500D0aD0E`0002@0000D0`00500001D3000E00000P0001@35@0E0a@0
E`0002@0000E0`004`0001D3000D00000P0001D3500E0a<0E`0002@0000C0`004@0001D3000C0000
0P0001D34`0C0a40E`0002@0000A0`00400001<3000A00000P0001<34@0A0a00E`0002@0000?0`00
40000143000@00000P000143400?0a00E`0002@0000<0`00400000l3000A00000P0000l3400<0a40
E`0002@0000:0`004@0000`3000C00000P0000`34@0:0a<0E`0002@000090`004`0000X3000F0000
0P0000X34`090aH0E`0002@000090`005P0000T3000H00000P0000T35P090aP0E`0002@000090`00
600000X3000K00000P0000T3600:0a/0E`0002@0000:0`006`0000`3000M00000P0000X36`0<0ad0
E`0002@0000<0`007@0000l3000M00000P0000`37@0?0ad0E`0002@0000?0`007@000103000M0000
0P0000l37@0@0ad0E`0002@0000@0`007@0001<3000M00000P0001037@0C0ad0E`0002@0000C0`00
6`0001D3000M00000P0001<37@0E0a/0E`0002@0000=0`00400000l3000A00000P0000l3400=0a40
E`0002@0000;0`004@0000d3000C00000P0000d34@0;0a<0E`0002@0000:0`004`0000/3000F0000
0P0000/34`0:0aH0E`0002@0000:0`005P0000X3000H00000P0000X35P0:0aP0E`0002@0000:0`00
600000/3000K00000P0000X3600;0a/0E`0002@0000;0`006`0000d3000M00000P0000/36`0=0ad0
E`0002@0000=0`007@0000l3000M00000P0000d37@0?0ad0E`0002@0000K0`004P0002/3000B0000
0P0001/34P0[0a80E`0002@0000K0`00600002/3000H00000P0001/3600[0aP0E`0002@0000d0`00
2P0003L3000;00000P0003L32P0d0`/0E`0002@0000b0`002`0003@3000>00000P0003@32`0b0`h0
E`0002@0000a0`003P000383000B00000P0003833P0a0a80E`0002@0000a0`004P000343000E0000
0P0003434P0a0aD0E`0002@0000a0`005@000383000J00000P0003435@0b0aX0E`0002@0000b0`00
6P0003@3000M00000P0003836P0d0ad0E`0002@0000d0`007@0003L3000M00000P0003@37@0g0ad0
E`0002@0000g0`007@0003P3000M00000P0003L37@0h0ad0E`0002@0000h0`007@0003/3000M0000
0P0003P37@0k0ad0E`0002@0000k0`006P0003d3000M00000P0003/37@0m0aX0E`0002@0000m0`00
5@0003h3000J00000P0003d36P0n0aD0E`0002@0000n0`004P0003h3000E00000P0003h35@0n0a80
E`0002@0000m0`003P0003h3000B00000P0003h34P0m0`h0E`0002@0000k0`002`0003d3000>0000
0P0003d33P0k0`/0E`0002@0000h0`002P0003/3000;00000P0003/32`0h0`X0E`0002@0000g0`00
2P0003P3000:00000P0003P32P0g0`X0E`0002@0000e0`002P0003L3000;00000P0003L32P0e0`/0
E`0002@0000d0`002`0003D3000<00000P0003D32`0d0``0E`0002@0000c0`00300003@3000>0000
0P0003@3300c0`h0E`0002@0000b0`003P0003<3000B00000P0003<33P0b0a80E`0002@0000b0`00
4P000383000E00000P0003834P0b0aD0E`0002@0000b0`005@0003<3000J00000P0003835@0c0aX0
E`0002@0000c0`006P0003@3000L00000P0003<36P0d0a`0E`0002@0000d0`00700003D3000M0000
0P0003@3700e0ad0E`0002@0000e0`007@0003L3000M00000P0003D37@0g0ad0E`0002@0000h0`00
7@0003X3000M00000P0003P37@0j0ad0E`0002@0000j0`00700003/3000M00000P0003X37@0k0a`0
E`0002@0000k0`006P0003`3000L00000P0003/3700l0aX0E`0002@0000l0`005@0003d3000J0000
0P0003`36P0m0aD0E`0002@0000m0`004P0003d3000E00000P0003d35@0m0a80E`0002@0000l0`00
3P0003d3000B00000P0003d34P0l0`h0E`0002@0000k0`00300003`3000>00000P0003`33P0k0``0
E`0002@0000j0`002`0003/3000<00000P0003/3300j0`/0E`0002@0000h0`002P0003X3000;0000
0P0003X32`0h0`X0E`0002@000140`00700004D3000M00000P0004D370140ad0E`0002@000140`00
7@0004D3000M00000P0004@37@150ad0E`0002@000150`007@0004H3000M00000P0004D37@160ad0
E`0002@000150`00700004H3000M00000P0004H37@150a`0E`0002@0001?0`002P000583000;0000
0P0005832P1?0`/0E`0002@0001=0`002`0004l3000>00000P0004l32`1=0`h0E`0002@0001<0`00
3P0004d3000B00000P0004d33P1<0a80E`0002@0001<0`004P0004`3000E00000P0004`34P1<0aD0
E`0002@0001<0`005@0004d3000J00000P0004`35@1=0aX0E`0002@0001=0`006P0004l3000M0000
0P0004d36P1?0ad0E`0002@0001?0`007@000583000M00000P0004l37@1B0ad0E`0002@0001B0`00
7@0005<3000M00000P0005837@1C0ad0E`0002@0001C0`007@0005H3000M00000P0005<37@1F0ad0
E`0002@0001F0`006P0005P3000M00000P0005H37@1H0aX0E`0002@0001H0`005@0005T3000J0000
0P0005P36P1I0aD0E`0002@0001I0`004P0005T3000E00000P0005T35@1I0a80E`0002@0001H0`00
3P0005T3000B00000P0005T34P1H0`h0E`0002@0001F0`002`0005P3000>00000P0005P33P1F0`/0
E`0002@0001C0`002P0005H3000;00000P0005H32`1C0`X0E`0002@0001B0`002P0005<3000:0000
0P0005<32P1B0`X0E`0002@0001@0`002P000583000;00000P0005832P1@0`/0E`0002@0001?0`00
2`000503000<00000P0005032`1?0``0E`0002@0001>0`00300004l3000>00000P0004l3301>0`h0
E`0002@0001=0`003P0004h3000B00000P0004h33P1=0a80E`0002@0001=0`004P0004d3000E0000
0P0004d34P1=0aD0E`0002@0001=0`005@0004h3000J00000P0004d35@1>0aX0E`0002@0001>0`00
6P0004l3000L00000P0004h36P1?0a`0E`0002@0001?0`0070000503000M00000P0004l3701@0ad0
E`0002@0001@0`007@000583000M00000P0005037@1B0ad0E`0002@0001C0`007@0005D3000M0000
0P0005<37@1E0ad0E`0002@0001E0`00700005H3000M00000P0005D37@1F0a`0E`0002@0001F0`00
6P0005L3000L00000P0005H3701G0aX0E`0002@0001G0`005@0005P3000J00000P0005L36P1H0aD0
E`0002@0001H0`004P0005P3000E00000P0005P35@1H0a80E`0002@0001G0`003P0005P3000B0000
0P0005P34P1G0`h0E`0002@0001F0`00300005L3000>00000P0005L33P1F0``0E`0002@0001E0`00
2`0005H3000<00000P0005H3301E0`/0E`0002@0001C0`002P0005D3000;00000P0005D32`1C0`X0
E`0002@0001Q0`002P0006@3000;00000P0006@32P1Q0`/0E`0002@0001O0`002`000643000>0000
0P0006432`1O0`h0E`0002@0001N0`003P0005l3000B00000P0005l33P1N0a80E`0002@0001N0`00
4P0005h3000E00000P0005h34P1N0aD0E`0002@0001N0`005@0005l3000J00000P0005h35@1O0aX0
E`0002@0001O0`006P000643000M00000P0005l36P1Q0ad0E`0002@0001Q0`007@0006@3000M0000
0P0006437@1T0ad0E`0002@0001T0`007@0006D3000M00000P0006@37@1U0ad0E`0002@0001U0`00
7@0006P3000M00000P0006D37@1X0ad0E`0002@0001X0`006P0006X3000M00000P0006P37@1Z0aX0
E`0002@0001Z0`005@0006/3000J00000P0006X36P1[0aD0E`0002@0001[0`004P0006/3000E0000
0P0006/35@1[0a80E`0002@0001Z0`003P0006/3000B00000P0006/34P1Z0`h0E`0002@0001X0`00
2`0006X3000>00000P0006X33P1X0`/0E`0002@0001U0`002P0006P3000;00000P0006P32`1U0`X0
E`0002@0001T0`002P0006D3000:00000P0006D32P1T0`X0E`0002@0001R0`002P0006@3000;0000
0P0006@32P1R0`/0E`0002@0001Q0`002`000683000<00000P0006832`1Q0``0E`0002@0001P0`00
30000643000>00000P000643301P0`h0E`0002@0001O0`003P000603000B00000P0006033P1O0a80
E`0002@0001O0`004P0005l3000E00000P0005l34P1O0aD0E`0002@0001O0`005@000603000J0000
0P0005l35@1P0aX0E`0002@0001P0`006P000643000L00000P0006036P1Q0a`0E`0002@0001Q0`00
70000683000M00000P000643701R0ad0E`0002@0001R0`007@0006@3000M00000P0006837@1T0ad0
E`0002@0001U0`007@0006L3000M00000P0006D37@1W0ad0E`0002@0001W0`00700006P3000M0000
0P0006L37@1X0a`0E`0002@0001X0`006P0006T3000L00000P0006P3701Y0aX0E`0002@0001Y0`00
5@0006X3000J00000P0006T36P1Z0aD0E`0002@0001Z0`004P0006X3000E00000P0006X35@1Z0a80
E`0002@0001Y0`003P0006X3000B00000P0006X34P1Y0`h0E`0002@0001X0`00300006T3000>0000
0P0006T33P1X0``0E`0002@0001W0`002`0006P3000<00000P0006P3301W0`/0E`0002@0001U0`00
2P0006L3000;00000P0006L32`1U0`X0E`0002@0001c0`003@0007D3000>00000P0007<33P1e0`d0
E`0002@0001e0`002P0007L3000=00000P0007D33@1g0`X0E`0002@0001g0`002P0007L3000M0000
0P0007L32P1g0ad0E`0002@0001f0`002`0007H3000M00000P0007H32`1f0ad0E`0002@0001c0`00
7@0007/3000M00000P0007<37@1k0ad0E`0002@0002L0000O0<00:40001l0`000P000:40O0>L07`3
E`0002@0002L00005`<00:40000G0`000P000:405`>L01L3E`0002@0002L0000/@800:40002a0P00
0P000:40/@:L0;42E`0002@0002L0000C0800:40001<0P000P000:40C0:L04`2E`0002@0002L0000
iP400:40003V0@000P000:40iP6L0>H1E`0002@0002L0000P0400:4000200@000P000:40P06L0801
E`0002@0002L00006`400:40000K0@000P000:406`6L01/1E`0002@0002L0000]@000:40002e0000
0P000:40]@2L0;D0E`0002@0002L0000D0000:40001@00000P000:40D02L0500E`0002@0000@0000
2P8002<0000:0P000P0001002P8S00X2E`0002@0000@00002@8002<000090P000P0001002@8S00T2
E`0002@0000@00000P800100000=0P000P0001003@8@0082E`0002@0000@0000o`40014000020P00
0P0001000P8A0?l1E`0002@0000A0000oP400180003o0@000P000140o`4B0?h1E`0002@0000B0000
o@4001@0003n0@000P000180oP4D0?d1E`0002@0000D0000o@4001L0003m0@000P0001@0o@4G0?d1
E`0002@0000G0000o@4001P0003n0@000P0001L0o@4H0?h1E`0002@0000H0000oP4001T0003o0@00
0P0001P0oP4I0?l1E`0002@0000I0000o`4001X000020P000P0001T0o`4J0082E`0002@0000J0000
0P8001X000090P000P0001X00P8J00T2E`0002@0000@00000080014000020P000P0001000P8A0002
E`0002@0000A0000o`40018000000P000P000140008B0?l1E`0002@0000B0000oP4001@0003o0@00
0P000180o`4D0?h1E`0002@0000D0000oP4001L0003n0@000P0001@0oP4G0?h1E`0002@0000G0000
oP4001P0003o0@000P0001L0oP4H0?l1E`0002@0000H0000o`4001T000000P000P0001P0o`4I0002
E`0002@0000I0000008001X000020P000P0001T0008J0082E`0002@0000S00001`8002<0000=0P00
0P0002<03@8S00L2E`0002@0000G0000mP4002<0003f0@000P0001L0mP4S0?H1E`0002@0000G0000
m@4002<0003e0@000P0001L0m@4S0?D1E`0002@0000I0000m04001`0003e0@000P0001`0m@4I0?@1
E`0002@0000G0000l`4001T0003d0@000P0001T0m04G0?<1E`0002@0000G0000l@4001L0003c0@00
0P0001L0l`4G0?41E`0002@0000G0000kP4001L0003a0@000P0001L0l@4G0>h1E`0002@0000G0000
k@4001L0003^0@000P0001L0kP4G0>d1E`0002@0000G0000k@4001P0003]0@000P0001L0k@4H0>d1
E`0002@0000H0000k@4001T0003^0@000P0001P0k@4I0>h1E`0002@0000H0000kP4001T0003_0@00
0P0001T0kP4H0>l1E`0002@0000G0000kP4001P0003_0@000P0001P0k`4G0>h1E`0002@0000G0000
m@4001L0003i0@000P0001L0n@4G0?D1E`0002@0000S0000l`4002<0003i0@000P0002<0n@4S0?<1
E`0002@0000@0000iP400140003W0@000P000100iP4A0>L1E`0002@0000A0000iP400180003W0@00
0P000140i`4B0>H1E`0002@0000A0000i@400180003V0@000P000180iP4A0>D1E`0002@0000@0000
i@400140003V0@000P000140i@4@0>H1E`0002@0000G0000iP4002<0003V0@000P0001L0iP4S0>H1
E`0002@0000G0000i@4002<0003U0@000P0001L0i@4S0>D1E`0002@0000G0000i@4001L0003Y0@00
0P0001L0j@4G0>D1E`0002@0000S0000h`4002<0003Y0@000P0002<0j@4S0><1E`0002@0000I0000
dP4001X0003C0@000P0001T0dP4J0=<1E`0002@0000J0000dP4001/0003C0@000P0001X0d`4K0=81
E`0002@0000J0000d@4001/0003B0@000P0001/0dP4J0=41E`0002@0000I0000d@4001X0003A0@00
0P0001X0d@4I0=41E`0002@0000G0000d@4001T0003C0@000P0001T0d@4G0=<1E`0002@0000G0000
d`4001L0003E0@000P0001L0d`4G0=D1E`0002@0000G0000e@4001L0003H0@000P0001L0e@4G0=P1
E`0002@0000G0000f04001L0003J0@000P0001L0f04G0=X1E`0002@0000G0000fP4001T0003L0@00
0P0001L0fP4I0=`1E`0002@0000I0000g04001`0003M0@000P0001T0g04L0=d1E`0002@0000L0000
g@4001h0003M0@000P0001`0g@4N0=d1E`0002@0000N0000g0400200003M0@000P0001h0g@4P0=`1
E`0002@0000P0000fP400280003L0@000P000200g04R0=X1E`0002@0000R0000f04002<0003J0@00
0P000280fP4S0=P1E`0002@0000S0000eP4002<0003H0@000P0002<0f04S0=H1E`0002@0000R0000
d`4002<0003F0@000P0002<0eP4R0=<1E`0002@0000P0000d@400280003C0@000P000280d`4P0=41
E`0002@0000G0000f04001L0003I0@000P0001L0f04G0=T1E`0002@0000G0000f@4001T0003K0@00
0P0001L0f@4I0=/1E`0002@0000I0000f`4001`0003L0@000P0001T0f`4L0=`1E`0002@0000L0000
g04001h0003L0@000P0001`0g04N0=`1E`0002@0000N0000f`400200003L0@000P0001h0g04P0=/1
E`0002@0000P0000f@400280003K0@000P000200f`4R0=T1E`0002@0000R0000f04002<0003I0@00
0P000280f@4S0=P1E`0002@0000L0000`04001`0003;0@000P0001`0b`4L0<01E`0002@0000J0000
`04001`000300@000P0001`0`04J0<01E`0002@0000H0000`04001X000310@000P0001X0`04H0<41
E`0002@0000G0000`@4001P000320@000P0001P0`@4G0<81E`0002@0000G0000`P4001L000340@00
0P0001L0`P4G0<@1E`0002@0000G0000a04001L000370@000P0001L0a04G0<L1E`0002@0000G0000
a`4001L000390@000P0001L0a`4G0<T1E`0002@0000G0000b@4001T0003;0@000P0001L0b@4I0</1
E`0002@0000I0000b`4001`0003<0@000P0001T0b`4L0<`1E`0002@0000L0000c04001h0003<0@00
0P0001`0c04N0<`1E`0002@0000N0000b`400200003<0@000P0001h0c04P0</1E`0002@0000P0000
b@400280003;0@000P000200b`4R0<T1E`0002@0000R0000a`4002<000390@000P000280b@4S0<L1
E`0002@0000S0000a@4002<000370@000P0002<0a`4S0<D1E`0002@0000R0000`P4002<000350@00
0P0002<0a@4R0<81E`0002@0000P0000`040028000320@000P000280`P4P0<01E`0002@0000I0000
`@4001`000310@000P0001`0`@4I0<41E`0002@0000G0000`@4001T000320@000P0001T0`@4G0<81
E`0002@0000G0000a`4001L000380@000P0001L0a`4G0<P1E`0002@0000G0000b04001T0003:0@00
0P0001L0b04I0<X1E`0002@0000I0000bP4001`0003;0@000P0001T0bP4L0</1E`0002@0000L0000
b`4001h0003;0@000P0001`0b`4N0</1E`0002@0000N0000bP400200003;0@000P0001h0b`4P0<X1
E`0002@0000P0000b0400280003:0@000P000200bP4R0<P1E`0002@0000R0000a`4002<000380@00
0P000280b04S0<L1E`0002@000140000OP<005@0001n0`000P0004@0OP=D07h3E`0002@0001N0000
L`<00640001d0`000P000640L`=N07@3E`0002@0001L0000M0<005h0001g0`000P0005h0M0=L07L3
E`0002@0001K0000M`<005`0001k0`000P0005`0M`=K07/3E`0002@0001K0000N`<005/0001n0`00
0P0005/0N`=K07h3E`0002@0001K0000OP<005`000230`000P0005/0OP=L08<3E`0002@0001L0000
P`<005h000260`000P0005`0P`=N08H3E`0002@0001N0000QP<0064000260`000P0005h0QP=Q08H3
E`0002@0001Q0000QP<0068000260`000P000640QP=R08H3E`0002@0001R0000QP<006D000260`00
0P000680QP=U08H3E`0002@0001U0000P`<006L000260`000P0006D0QP=W08<3E`0002@0001W0000
OP<006P000230`000P0006L0P`=X07h3E`0002@0001X0000N`<006P0001n0`000P0006P0OP=X07/3
E`0002@0001W0000M`<006P0001k0`000P0006P0N`=W07L3E`0002@0001U0000M0<006L0001g0`00
0P0006L0M`=U07@3E`0002@0001R0000L`<006D0001d0`000P0006D0M0=R07<3E`0002@0001Q0000
L`<00680001c0`000P000680L`=Q07<3E`0002@0001O0000L`<00640001d0`000P000640L`=O07@3
E`0002@0001N0000M0<005l0001e0`000P0005l0M0=N07D3E`0002@0001M0000M@<005h0001g0`00
0P0005h0M@=M07L3E`0002@0001L0000M`<005d0001k0`000P0005d0M`=L07/3E`0002@0001L0000
N`<005`0001n0`000P0005`0N`=L07h3E`0002@0001L0000OP<005d000230`000P0005`0OP=M08<3
E`0002@0001M0000P`<005h000250`000P0005d0P`=N08D3E`0002@0001N0000Q@<005l000260`00
0P0005h0Q@=O08H3E`0002@0001O0000QP<0064000260`000P0005l0QP=Q08H3E`0002@0001R0000
QP<006@000260`000P000680QP=T08H3E`0002@0001T0000Q@<006D000260`000P0006@0QP=U08D3
E`0002@0001U0000P`<006H000250`000P0006D0Q@=V08<3E`0002@0001V0000OP<006L000230`00
0P0006H0P`=W07h3E`0002@0001W0000N`<006L0001n0`000P0006L0OP=W07/3E`0002@0001V0000
M`<006L0001k0`000P0006L0N`=V07L3E`0002@0001U0000M@<006H0001g0`000P0006H0M`=U07D3
E`0002@0001T0000M0<006D0001e0`000P0006D0M@=T07@3E`0002@0001R0000L`<006@0001d0`00
0P0006@0M0=R07<3E`0002@0001]0000Q@<006h000260`000P0006h0Q@=]08H3E`0002@0001]0000
QP<006h000260`000P0006d0QP=^08H3E`0002@0001^0000QP<006l000260`000P0006h0QP=_08H3
E`0002@0001^0000Q@<006l000260`000P0006l0QP=^08D3E`0002@0001h0000L`<007/0001d0`00
0P0007/0L`=h07@3E`0002@0001f0000M0<007P0001g0`000P0007P0M0=f07L3E`0002@0001e0000
M`<007H0001k0`000P0007H0M`=e07/3E`0002@0001e0000N`<007D0001n0`000P0007D0N`=e07h3
E`0002@0001e0000OP<007H000230`000P0007D0OP=f08<3E`0002@0001f0000P`<007P000260`00
0P0007H0P`=h08H3E`0002@0001h0000QP<007/000260`000P0007P0QP=k08H3E`0002@0001k0000
QP<007`000260`000P0007/0QP=l08H3E`0002@0001l0000QP<007l000260`000P0007`0QP=o08H3
E`0002@0001o0000P`<0084000260`000P0007l0QP>108<3E`0002@000210000OP<0088000230`00
0P000840P`>207h3E`0002@000220000N`<00880001n0`000P000880OP>207/3E`0002@000210000
M`<00880001k0`000P000880N`>107L3E`0002@0001o0000M0<00840001g0`000P000840M`=o07@3
E`0002@0001l0000L`<007l0001d0`000P0007l0M0=l07<3E`0002@0001k0000L`<007`0001c0`00
0P0007`0L`=k07<3E`0002@0001i0000L`<007/0001d0`000P0007/0L`=i07@3E`0002@0001h0000
M0<007T0001e0`000P0007T0M0=h07D3E`0002@0001g0000M@<007P0001g0`000P0007P0M@=g07L3
E`0002@0001f0000M`<007L0001k0`000P0007L0M`=f07/3E`0002@0001f0000N`<007H0001n0`00
0P0007H0N`=f07h3E`0002@0001f0000OP<007L000230`000P0007H0OP=g08<3E`0002@0001g0000
P`<007P000250`000P0007L0P`=h08D3E`0002@0001h0000Q@<007T000260`000P0007P0Q@=i08H3
E`0002@0001i0000QP<007/000260`000P0007T0QP=k08H3E`0002@0001l0000QP<007h000260`00
0P0007`0QP=n08H3E`0002@0001n0000Q@<007l000260`000P0007h0QP=o08D3E`0002@0001o0000
P`<0080000250`000P0007l0Q@>008<3E`0002@000200000OP<0084000230`000P000800P`>107h3
E`0002@000210000N`<00840001n0`000P000840OP>107/3E`0002@000200000M`<00840001k0`00
0P000840N`>007L3E`0002@0001o0000M@<00800001g0`000P000800M`=o07D3E`0002@0001n0000
M0<007l0001e0`000P0007l0M@=n07@3E`0002@0001l0000L`<007h0001d0`000P0007h0M0=l07<3
E`0002@000270000L`<008L0001h0`000P0008L0L`>707P3E`0002@000270000M@<008P0001g0`00
0P0008L0M`>807D3E`0002@000280000L`<008X0001e0`000P0008P0M@>:07<3E`0002@0002:0000
L`<008`0001c0`000P0008X0L`><07<3E`0002@0002<0000L`<00900001f0`000P0008`0L`>@07H3
E`0002@0002@0000MP<00980001f0`000P000900MP>B07H3E`0002@0002B0000M@<009<0001f0`00
0P000980MP>C07D3E`0002@0002C0000L`<009@0001e0`000P0009<0M@>D07<3E`0002@000280000
M0<008X0001e0`000P0008P0M@>:07@3E`0002@0002:0000M0<008`0001d0`000P0008X0M0><07@3
E`0002@0002<0000M0<00900001f0`000P0008`0M0>@07H3E`0002@0002D0000L`<009@0001f0`00
0P0009@0L`>D07H3E`0002@0002C0000MP<009@0001h0`000P0009@0MP>C07P3E`0002@0002?0000
N0<009<0001m0`000P0009<0N0>?07d3E`0002@0002>0000O@<008l0001o0`000P0008l0O@>>07l3
E`0002@0002=0000O`<008h000220`000P0008h0O`>=0883E`0002@0002=0000PP<008d000260`00
0P0008d0PP>=08H3E`0002@0002>0000N0<009<0001m0`000P0009<0N0>>07d3E`0002@0002=0000
O@<008h0001o0`000P0008h0O@>=07l3E`0002@0002=0000O`<008d000220`000P0008d0O`>=0883
E`0002@0002=0000PP<008d000260`000P0008d0PP>=08H3E`0002@00014000060<005@0000H0`00
0P0004@060=D01P3E`0002@0001N00003@<00640000>0`000P0006403@=N00h3E`0002@0001L0000
3P<005h0000A0`000P0005h03P=L0143E`0002@0001K00004@<005`0000E0`000P0005`04@=K01D3
E`0002@0001K00005@<005/0000H0`000P0005/05@=K01P3E`0002@0001K000060<005`0000M0`00
0P0005/060=L01d3E`0002@0001L00007@<005h0000P0`000P0005`07@=N0203E`0002@0001N0000
80<00640000P0`000P0005h080=Q0203E`0002@0001Q000080<00680000P0`000P00064080=R0203
E`0002@0001R000080<006D0000P0`000P00068080=U0203E`0002@0001U00007@<006L0000P0`00
0P0006D080=W01d3E`0002@0001W000060<006P0000M0`000P0006L07@=X01P3E`0002@0001X0000
5@<006P0000H0`000P0006P060=X01D3E`0002@0001W00004@<006P0000E0`000P0006P05@=W0143
E`0002@0001U00003P<006L0000A0`000P0006L04@=U00h3E`0002@0001R00003@<006D0000>0`00
0P0006D03P=R00d3E`0002@0001Q00003@<00680000=0`000P0006803@=Q00d3E`0002@0001O0000
3@<00640000>0`000P0006403@=O00h3E`0002@0001N00003P<005l0000?0`000P0005l03P=N00l3
E`0002@0001M00003`<005h0000A0`000P0005h03`=M0143E`0002@0001L00004@<005d0000E0`00
0P0005d04@=L01D3E`0002@0001L00005@<005`0000H0`000P0005`05@=L01P3E`0002@0001L0000
60<005d0000M0`000P0005`060=M01d3E`0002@0001M00007@<005h0000O0`000P0005d07@=N01l3
E`0002@0001N00007`<005l0000P0`000P0005h07`=O0203E`0002@0001O000080<00640000P0`00
0P0005l080=Q0203E`0002@0001R000080<006@0000P0`000P00068080=T0203E`0002@0001T0000
7`<006D0000P0`000P0006@080=U01l3E`0002@0001U00007@<006H0000O0`000P0006D07`=V01d3
E`0002@0001V000060<006L0000M0`000P0006H07@=W01P3E`0002@0001W00005@<006L0000H0`00
0P0006L060=W01D3E`0002@0001V00004@<006L0000E0`000P0006L05@=V0143E`0002@0001U0000
3`<006H0000A0`000P0006H04@=U00l3E`0002@0001T00003P<006D0000?0`000P0006D03`=T00h3
E`0002@0001R00003@<006@0000>0`000P0006@03P=R00d3E`0002@0001]00007`<006h0000P0`00
0P0006h07`=]0203E`0002@0001]000080<006h0000P0`000P0006d080=^0203E`0002@0001^0000
80<006l0000P0`000P0006h080=_0203E`0002@0001^00007`<006l0000P0`000P0006l080=^01l3
E`0002@0001h00003@<007/0000>0`000P0007/03@=h00h3E`0002@0001f00003P<007P0000A0`00
0P0007P03P=f0143E`0002@0001e00004@<007H0000E0`000P0007H04@=e01D3E`0002@0001e0000
5@<007D0000H0`000P0007D05@=e01P3E`0002@0001e000060<007H0000M0`000P0007D060=f01d3
E`0002@0001f00007@<007P0000P0`000P0007H07@=h0203E`0002@0001h000080<007/0000P0`00
0P0007P080=k0203E`0002@0001k000080<007`0000P0`000P0007/080=l0203E`0002@0001l0000
80<007l0000P0`000P0007`080=o0203E`0002@0001o00007@<00840000P0`000P0007l080>101d3
E`0002@00021000060<00880000M0`000P0008407@>201P3E`0002@0002200005@<00880000H0`00
0P00088060>201D3E`0002@0002100004@<00880000E0`000P0008805@>10143E`0002@0001o0000
3P<00840000A0`000P0008404@=o00h3E`0002@0001l00003@<007l0000>0`000P0007l03P=l00d3
E`0002@0001k00003@<007`0000=0`000P0007`03@=k00d3E`0002@0001i00003@<007/0000>0`00
0P0007/03@=i00h3E`0002@0001h00003P<007T0000?0`000P0007T03P=h00l3E`0002@0001g0000
3`<007P0000A0`000P0007P03`=g0143E`0002@0001f00004@<007L0000E0`000P0007L04@=f01D3
E`0002@0001f00005@<007H0000H0`000P0007H05@=f01P3E`0002@0001f000060<007L0000M0`00
0P0007H060=g01d3E`0002@0001g00007@<007P0000O0`000P0007L07@=h01l3E`0002@0001h0000
7`<007T0000P0`000P0007P07`=i0203E`0002@0001i000080<007/0000P0`000P0007T080=k0203
E`0002@0001l000080<007h0000P0`000P0007`080=n0203E`0002@0001n00007`<007l0000P0`00
0P0007h080=o01l3E`0002@0001o00007@<00800000O0`000P0007l07`>001d3E`0002@000200000
60<00840000M0`000P0008007@>101P3E`0002@0002100005@<00840000H0`000P00084060>101D3
E`0002@0002000004@<00840000E0`000P0008405@>00143E`0002@0001o00003`<00800000A0`00
0P0008004@=o00l3E`0002@0001n00003P<007l0000?0`000P0007l03`=n00h3E`0002@0001l0000
3@<007h0000>0`000P0007h03P=l00d3E`0002@0002A000040<00980000A0`000P00098040>A0143
E`0002@0002A00004@<00980000B0`000P0009404@>B0183E`0002@0002B00004@<009<0000B0`00
0P0009804P>C0143E`0002@0002C000040<009<0000A0`000P0009<04@>C0103E`0002@0002B0000
3P<009<0000@0`000P0009<040>B00h3E`0002@0002@00003@<00980000>0`000P0009803P>@00d3
E`0002@0002=00003@<00900000=0`000P0009003@>=00d3E`0002@0002;00003@<008d0000>0`00
0P0008d03@>;00h3E`0002@0002900003P<008/0000@0`000P0008/03P>90103E`0002@000280000
40<008T0000B0`000P0008T040>80183E`0002@0002700004P<008P0000E0`000P0008P04P>701D3
E`0002@0002700005@<008L0000K0`000P0008L05@>701/3E`0002@0002700006`<008P0000N0`00
0P0008L06`>801h3E`0002@0002800007P<008X0000P0`000P0008P07P>:0203E`0002@0002:0000
80<008d0000P0`000P0008X080>=0203E`0002@0002=000080<008h0000P0`000P0008d080>>0203
E`0002@0002>000080<00940000P0`000P0008h080>A0203E`0002@0002A00007P<009<0000P0`00
0P00094080>C01h3E`0002@0002C00006`<009@0000N0`000P0009<07P>D01/3E`0002@0002D0000
6P<009@0000K0`000P0009@06`>D01X3E`0002@0002C00005`<009@0000J0`000P0009@06P>C01L3
E`0002@0002A00005@<009<0000G0`000P0009<05`>A01D3E`0002@0002>000050<00940000E0`00
0P0009405@>>01@3E`0002@0002=000050<008h0000D0`000P0008h050>=01@3E`0002@0002;0000
50<008d0000E0`000P0008d050>;01D3E`0002@0002900005@<008/0000G0`000P0008/05@>901L3
E`0002@0002800005`<008T0000J0`000P0008T05`>801X3E`0002@0002<00003@<008d0000>0`00
0P0008d03@><00h3E`0002@0002:00003P<008`0000@0`000P0008`03P>:0103E`0002@000290000
40<008X0000B0`000P0008X040>90183E`0002@0002800004P<008T0000E0`000P0008T04P>801D3
E`0002@0002800005@<008P0000K0`000P0008P05@>801/3E`0002@0002800006`<008T0000N0`00
0P0008P06`>901h3E`0002@0002900007P<008/0000P0`000P0008T07P>;0203E`0002@0002;0000
80<008d0000P0`000P0008/080>=0203E`0002@0002>000080<00900000P0`000P0008h080>@0203
E`0002@0002@00007P<00980000P0`000P00090080>B01h3E`0002@0002B00006`<009<0000N0`00
0P0009807P>C01/3E`0002@0002C00006P<009<0000K0`000P0009<06`>C01X3E`0002@0002B0000
5`<009<0000J0`000P0009<06P>B01L3E`0002@0002@00005@<00980000G0`000P0009805`>@01D3
E`0002@0002>000050<00900000E0`000P0009005@>>01@3E`0002@000140000/`8005@0002c0P00
0P0004@0/`9D0;<2E`0002@0001N0000Z0800640002Y0P000P000640Z09N0:T2E`0002@0001L0000
Z@8005h0002/0P000P0005h0Z@9L0:`2E`0002@0001K0000[08005`0002`0P000P0005`0[09K0;02
E`0002@0001K0000/08005/0002c0P000P0005/0/09K0;<2E`0002@0001K0000/`8005`0002h0P00
0P0005/0/`9L0;P2E`0002@0001L0000^08005h0002k0P000P0005`0^09N0;/2E`0002@0001N0000
^`800640002k0P000P0005h0^`9Q0;/2E`0002@0001Q0000^`800680002k0P000P000640^`9R0;/2
E`0002@0001R0000^`8006D0002k0P000P000680^`9U0;/2E`0002@0001U0000^08006L0002k0P00
0P0006D0^`9W0;P2E`0002@0001W0000/`8006P0002h0P000P0006L0^09X0;<2E`0002@0001X0000
/08006P0002c0P000P0006P0/`9X0;02E`0002@0001W0000[08006P0002`0P000P0006P0/09W0:`2
E`0002@0001U0000Z@8006L0002/0P000P0006L0[09U0:T2E`0002@0001R0000Z08006D0002Y0P00
0P0006D0Z@9R0:P2E`0002@0001Q0000Z0800680002X0P000P000680Z09Q0:P2E`0002@0001O0000
Z0800640002Y0P000P000640Z09O0:T2E`0002@0001N0000Z@8005l0002Z0P000P0005l0Z@9N0:X2
E`0002@0001M0000ZP8005h0002/0P000P0005h0ZP9M0:`2E`0002@0001L0000[08005d0002`0P00
0P0005d0[09L0;02E`0002@0001L0000/08005`0002c0P000P0005`0/09L0;<2E`0002@0001L0000
/`8005d0002h0P000P0005`0/`9M0;P2E`0002@0001M0000^08005h0002j0P000P0005d0^09N0;X2
E`0002@0001N0000^P8005l0002k0P000P0005h0^P9O0;/2E`0002@0001O0000^`800640002k0P00
0P0005l0^`9Q0;/2E`0002@0001R0000^`8006@0002k0P000P000680^`9T0;/2E`0002@0001T0000
^P8006D0002k0P000P0006@0^`9U0;X2E`0002@0001U0000^08006H0002j0P000P0006D0^P9V0;P2
E`0002@0001V0000/`8006L0002h0P000P0006H0^09W0;<2E`0002@0001W0000/08006L0002c0P00
0P0006L0/`9W0;02E`0002@0001V0000[08006L0002`0P000P0006L0/09V0:`2E`0002@0001U0000
ZP8006H0002/0P000P0006H0[09U0:X2E`0002@0001T0000Z@8006D0002Z0P000P0006D0ZP9T0:T2
E`0002@0001R0000Z08006@0002Y0P000P0006@0Z@9R0:P2E`0002@0001]0000^P8006h0002k0P00
0P0006h0^P9]0;/2E`0002@0001]0000^`8006h0002k0P000P0006d0^`9^0;/2E`0002@0001^0000
^`8006l0002k0P000P0006h0^`9_0;/2E`0002@0001^0000^P8006l0002k0P000P0006l0^`9^0;X2
E`0002@0001h0000Z08007/0002Y0P000P0007/0Z09h0:T2E`0002@0001f0000Z@8007P0002/0P00
0P0007P0Z@9f0:`2E`0002@0001e0000[08007H0002`0P000P0007H0[09e0;02E`0002@0001e0000
/08007D0002c0P000P0007D0/09e0;<2E`0002@0001e0000/`8007H0002h0P000P0007D0/`9f0;P2
E`0002@0001f0000^08007P0002k0P000P0007H0^09h0;/2E`0002@0001h0000^`8007/0002k0P00
0P0007P0^`9k0;/2E`0002@0001k0000^`8007`0002k0P000P0007/0^`9l0;/2E`0002@0001l0000
^`8007l0002k0P000P0007`0^`9o0;/2E`0002@0001o0000^0800840002k0P000P0007l0^`:10;P2
E`0002@000210000/`800880002h0P000P000840^0:20;<2E`0002@000220000/0800880002c0P00
0P000880/`:20;02E`0002@000210000[0800880002`0P000P000880/0:10:`2E`0002@0001o0000
Z@800840002/0P000P000840[09o0:T2E`0002@0001l0000Z08007l0002Y0P000P0007l0Z@9l0:P2
E`0002@0001k0000Z08007`0002X0P000P0007`0Z09k0:P2E`0002@0001i0000Z08007/0002Y0P00
0P0007/0Z09i0:T2E`0002@0001h0000Z@8007T0002Z0P000P0007T0Z@9h0:X2E`0002@0001g0000
ZP8007P0002/0P000P0007P0ZP9g0:`2E`0002@0001f0000[08007L0002`0P000P0007L0[09f0;02
E`0002@0001f0000/08007H0002c0P000P0007H0/09f0;<2E`0002@0001f0000/`8007L0002h0P00
0P0007H0/`9g0;P2E`0002@0001g0000^08007P0002j0P000P0007L0^09h0;X2E`0002@0001h0000
^P8007T0002k0P000P0007P0^P9i0;/2E`0002@0001i0000^`8007/0002k0P000P0007T0^`9k0;/2
E`0002@0001l0000^`8007h0002k0P000P0007`0^`9n0;/2E`0002@0001n0000^P8007l0002k0P00
0P0007h0^`9o0;X2E`0002@0001o0000^0800800002j0P000P0007l0^P:00;P2E`0002@000200000
/`800840002h0P000P000800^0:10;<2E`0002@000210000/0800840002c0P000P000840/`:10;02
E`0002@000200000[0800840002`0P000P000840/0:00:`2E`0002@0001o0000ZP800800002/0P00
0P000800[09o0:X2E`0002@0001n0000Z@8007l0002Z0P000P0007l0ZP9n0:T2E`0002@0001l0000
Z08007h0002Y0P000P0007h0Z@9l0:P2E`0002@000270000Z08008T0002a0P000P0008T0Z0:70;42
E`0002@000270000[`8008T0002a0P000P0008L0/@:90:l2E`0002@000290000[P8008`0002_0P00
0P0008T0[`:<0:h2E`0002@0002<0000[P8008h0002^0P000P0008`0[P:>0:h2E`0002@0002>0000
[P800940002_0P000P0008h0[P:A0:l2E`0002@0002A0000[`8009<0002a0P000P000940[`:C0;42
E`0002@0002C0000/@8009@0002d0P000P0009<0/@:D0;@2E`0002@0002D0000]08009@0002f0P00
0P0009@0]0:D0;H2E`0002@0002C0000]P8009@0002i0P000P0009@0]P:C0;T2E`0002@0002A0000
^@8009<0002k0P000P0009<0^@:A0;/2E`0002@0002>0000^`800940002k0P000P000940^`:>0;/2
E`0002@0002<0000^`8008h0002k0P000P0008h0^`:<0;/2E`0002@000290000^`8008`0002k0P00
0P0008`0^`:90;/2E`0002@000280000^P8008T0002k0P000P0008T0^`:80;X2E`0002@000270000
^08008P0002j0P000P0008P0^P:70;P2E`0002@000270000]`8008L0002h0P000P0008L0^0:70;L2
E`0002@000270000]P8008P0002g0P000P0008L0]`:80;H2E`0002@000280000]P8008T0002g0P00
0P0008P0]P:90;L2E`0002@000280000]`8008T0002h0P000P0008T0]`:80;P2E`0002@0002>0000
[P800900002_0P000P0008h0[P:@0:l2E`0002@0002@0000[`800980002a0P000P000900[`:B0;42
E`0002@0002B0000/@8009<0002d0P000P000980/@:C0;@2E`0002@0002C0000]08009<0002f0P00
0P0009<0]0:C0;H2E`0002@0002B0000]P8009<0002i0P000P0009<0]P:B0;T2E`0002@0002@0000
^@800980002k0P000P000980^@:@0;/2E`0002@0002>0000^`800900002k0P000P000900^`:>0;/2
E`0002@000290000Z0800980002X0P000P0008T0Z0:B0:P2E`0002@000290000Z@8008d0002Y0P00
0P0008T0Z@:=0:T2E`0002@0002=0000Z0800980002Y0P000P0008d0Z@:B0:P2E`0002@000140000
C@8005@0001=0P000P0004@0C@9D04d2E`0002@0001N0000@P80064000130P000P000640@P9N04<2
E`0002@0001L0000@`8005h000160P000P0005h0@`9L04H2E`0002@0001K0000AP8005`0001:0P00
0P0005`0AP9K04X2E`0002@0001K0000BP8005/0001=0P000P0005/0BP9K04d2E`0002@0001K0000
C@8005`0001B0P000P0005/0C@9L0582E`0002@0001L0000DP8005h0001E0P000P0005`0DP9N05D2
E`0002@0001N0000E@800640001E0P000P0005h0E@9Q05D2E`0002@0001Q0000E@800680001E0P00
0P000640E@9R05D2E`0002@0001R0000E@8006D0001E0P000P000680E@9U05D2E`0002@0001U0000
DP8006L0001E0P000P0006D0E@9W0582E`0002@0001W0000C@8006P0001B0P000P0006L0DP9X04d2
E`0002@0001X0000BP8006P0001=0P000P0006P0C@9X04X2E`0002@0001W0000AP8006P0001:0P00
0P0006P0BP9W04H2E`0002@0001U0000@`8006L000160P000P0006L0AP9U04<2E`0002@0001R0000
@P8006D000130P000P0006D0@`9R0482E`0002@0001Q0000@P80068000120P000P000680@P9Q0482
E`0002@0001O0000@P80064000130P000P000640@P9O04<2E`0002@0001N0000@`8005l000140P00
0P0005l0@`9N04@2E`0002@0001M0000A08005h000160P000P0005h0A09M04H2E`0002@0001L0000
AP8005d0001:0P000P0005d0AP9L04X2E`0002@0001L0000BP8005`0001=0P000P0005`0BP9L04d2
E`0002@0001L0000C@8005d0001B0P000P0005`0C@9M0582E`0002@0001M0000DP8005h0001D0P00
0P0005d0DP9N05@2E`0002@0001N0000E08005l0001E0P000P0005h0E09O05D2E`0002@0001O0000
E@800640001E0P000P0005l0E@9Q05D2E`0002@0001R0000E@8006@0001E0P000P000680E@9T05D2
E`0002@0001T0000E08006D0001E0P000P0006@0E@9U05@2E`0002@0001U0000DP8006H0001D0P00
0P0006D0E09V0582E`0002@0001V0000C@8006L0001B0P000P0006H0DP9W04d2E`0002@0001W0000
BP8006L0001=0P000P0006L0C@9W04X2E`0002@0001V0000AP8006L0001:0P000P0006L0BP9V04H2
E`0002@0001U0000A08006H000160P000P0006H0AP9U04@2E`0002@0001T0000@`8006D000140P00
0P0006D0A09T04<2E`0002@0001R0000@P8006@000130P000P0006@0@`9R0482E`0002@0001]0000
E08006h0001E0P000P0006h0E09]05D2E`0002@0001]0000E@8006h0001E0P000P0006d0E@9^05D2
E`0002@0001^0000E@8006l0001E0P000P0006h0E@9_05D2E`0002@0001^0000E08006l0001E0P00
0P0006l0E@9^05@2E`0002@0001h0000@P8007/000130P000P0007/0@P9h04<2E`0002@0001f0000
@`8007P000160P000P0007P0@`9f04H2E`0002@0001e0000AP8007H0001:0P000P0007H0AP9e04X2
E`0002@0001e0000BP8007D0001=0P000P0007D0BP9e04d2E`0002@0001e0000C@8007H0001B0P00
0P0007D0C@9f0582E`0002@0001f0000DP8007P0001E0P000P0007H0DP9h05D2E`0002@0001h0000
E@8007/0001E0P000P0007P0E@9k05D2E`0002@0001k0000E@8007`0001E0P000P0007/0E@9l05D2
E`0002@0001l0000E@8007l0001E0P000P0007`0E@9o05D2E`0002@0001o0000DP800840001E0P00
0P0007l0E@:10582E`0002@000210000C@800880001B0P000P000840DP:204d2E`0002@000220000
BP800880001=0P000P000880C@:204X2E`0002@000210000AP800880001:0P000P000880BP:104H2
E`0002@0001o0000@`80084000160P000P000840AP9o04<2E`0002@0001l0000@P8007l000130P00
0P0007l0@`9l0482E`0002@0001k0000@P8007`000120P000P0007`0@P9k0482E`0002@0001i0000
@P8007/000130P000P0007/0@P9i04<2E`0002@0001h0000@`8007T000140P000P0007T0@`9h04@2
E`0002@0001g0000A08007P000160P000P0007P0A09g04H2E`0002@0001f0000AP8007L0001:0P00
0P0007L0AP9f04X2E`0002@0001f0000BP8007H0001=0P000P0007H0BP9f04d2E`0002@0001f0000
C@8007L0001B0P000P0007H0C@9g0582E`0002@0001g0000DP8007P0001D0P000P0007L0DP9h05@2
E`0002@0001h0000E08007T0001E0P000P0007P0E09i05D2E`0002@0001i0000E@8007/0001E0P00
0P0007T0E@9k05D2E`0002@0001l0000E@8007h0001E0P000P0007`0E@9n05D2E`0002@0001n0000
E08007l0001E0P000P0007h0E@9o05@2E`0002@0001o0000DP800800001D0P000P0007l0E0:00582
E`0002@000200000C@800840001B0P000P000800DP:104d2E`0002@000210000BP800840001=0P00
0P000840C@:104X2E`0002@000200000AP800840001:0P000P000840BP:004H2E`0002@0001o0000
A080080000160P000P000800AP9o04@2E`0002@0001n0000@`8007l000140P000P0007l0A09n04<2
E`0002@0001l0000@P8007h000130P000P0007h0@`9l0482E`0002@0002?0000A08008l0001E0P00
0P0008l0A0:?05D2E`0002@0002@0000@P800900001E0P000P000900@P:@05D2E`0002@000260000
@P800900001@0P000P000900@P:60502E`0002@000260000D08009D0001@0P000P0008H0D0:E0502
E`0002@0002=0000E@8009<0001E0P000P0008d0E@:C05D2E`0002@000140000i`4005@0003W0@00
0P0004@0i`5D0>L1E`0002@0001N0000g0400640003M0@000P000640g05N0=d1E`0002@0001L0000
g@4005h0003P0@000P0005h0g@5L0>01E`0002@0001K0000h04005`0003T0@000P0005`0h05K0>@1
E`0002@0001K0000i04005/0003W0@000P0005/0i05K0>L1E`0002@0001K0000i`4005`0003/0@00
0P0005/0i`5L0>`1E`0002@0001L0000k04005h0003_0@000P0005`0k05N0>l1E`0002@0001N0000
k`400640003_0@000P0005h0k`5Q0>l1E`0002@0001Q0000k`400680003_0@000P000640k`5R0>l1
E`0002@0001R0000k`4006D0003_0@000P000680k`5U0>l1E`0002@0001U0000k04006L0003_0@00
0P0006D0k`5W0>`1E`0002@0001W0000i`4006P0003/0@000P0006L0k05X0>L1E`0002@0001X0000
i04006P0003W0@000P0006P0i`5X0>@1E`0002@0001W0000h04006P0003T0@000P0006P0i05W0>01
E`0002@0001U0000g@4006L0003P0@000P0006L0h05U0=d1E`0002@0001R0000g04006D0003M0@00
0P0006D0g@5R0=`1E`0002@0001Q0000g0400680003L0@000P000680g05Q0=`1E`0002@0001O0000
g0400640003M0@000P000640g05O0=d1E`0002@0001N0000g@4005l0003N0@000P0005l0g@5N0=h1
E`0002@0001M0000gP4005h0003P0@000P0005h0gP5M0>01E`0002@0001L0000h04005d0003T0@00
0P0005d0h05L0>@1E`0002@0001L0000i04005`0003W0@000P0005`0i05L0>L1E`0002@0001L0000
i`4005d0003/0@000P0005`0i`5M0>`1E`0002@0001M0000k04005h0003^0@000P0005d0k05N0>h1
E`0002@0001N0000kP4005l0003_0@000P0005h0kP5O0>l1E`0002@0001O0000k`400640003_0@00
0P0005l0k`5Q0>l1E`0002@0001R0000k`4006@0003_0@000P000680k`5T0>l1E`0002@0001T0000
kP4006D0003_0@000P0006@0k`5U0>h1E`0002@0001U0000k04006H0003^0@000P0006D0kP5V0>`1
E`0002@0001V0000i`4006L0003/0@000P0006H0k05W0>L1E`0002@0001W0000i04006L0003W0@00
0P0006L0i`5W0>@1E`0002@0001V0000h04006L0003T0@000P0006L0i05V0>01E`0002@0001U0000
gP4006H0003P0@000P0006H0h05U0=h1E`0002@0001T0000g@4006D0003N0@000P0006D0gP5T0=d1
E`0002@0001R0000g04006@0003M0@000P0006@0g@5R0=`1E`0002@0001]0000kP4006h0003_0@00
0P0006h0kP5]0>l1E`0002@0001]0000k`4006h0003_0@000P0006d0k`5^0>l1E`0002@0001^0000
k`4006l0003_0@000P0006h0k`5_0>l1E`0002@0001^0000kP4006l0003_0@000P0006l0k`5^0>h1
E`0002@0001h0000g04007/0003M0@000P0007/0g05h0=d1E`0002@0001f0000g@4007P0003P0@00
0P0007P0g@5f0>01E`0002@0001e0000h04007H0003T0@000P0007H0h05e0>@1E`0002@0001e0000
i04007D0003W0@000P0007D0i05e0>L1E`0002@0001e0000i`4007H0003/0@000P0007D0i`5f0>`1
E`0002@0001f0000k04007P0003_0@000P0007H0k05h0>l1E`0002@0001h0000k`4007/0003_0@00
0P0007P0k`5k0>l1E`0002@0001k0000k`4007`0003_0@000P0007/0k`5l0>l1E`0002@0001l0000
k`4007l0003_0@000P0007`0k`5o0>l1E`0002@0001o0000k0400840003_0@000P0007l0k`610>`1
E`0002@000210000i`400880003/0@000P000840k0620>L1E`0002@000220000i0400880003W0@00
0P000880i`620>@1E`0002@000210000h0400880003T0@000P000880i0610>01E`0002@0001o0000
g@400840003P0@000P000840h05o0=d1E`0002@0001l0000g04007l0003M0@000P0007l0g@5l0=`1
E`0002@0001k0000g04007`0003L0@000P0007`0g05k0=`1E`0002@0001i0000g04007/0003M0@00
0P0007/0g05i0=d1E`0002@0001h0000g@4007T0003N0@000P0007T0g@5h0=h1E`0002@0001g0000
gP4007P0003P0@000P0007P0gP5g0>01E`0002@0001f0000h04007L0003T0@000P0007L0h05f0>@1
E`0002@0001f0000i04007H0003W0@000P0007H0i05f0>L1E`0002@0001f0000i`4007L0003/0@00
0P0007H0i`5g0>`1E`0002@0001g0000k04007P0003^0@000P0007L0k05h0>h1E`0002@0001h0000
kP4007T0003_0@000P0007P0kP5i0>l1E`0002@0001i0000k`4007/0003_0@000P0007T0k`5k0>l1
E`0002@0001l0000k`4007h0003_0@000P0007`0k`5n0>l1E`0002@0001n0000kP4007l0003_0@00
0P0007h0k`5o0>h1E`0002@0001o0000k0400800003^0@000P0007l0kP600>`1E`0002@000200000
i`400840003/0@000P000800k0610>L1E`0002@000210000i0400840003W0@000P000840i`610>@1
E`0002@000200000h0400840003T0@000P000840i0600>01E`0002@0001o0000gP400800003P0@00
0P000800h05o0=h1E`0002@0001n0000g@4007l0003N0@000P0007l0gP5n0=d1E`0002@0001l0000
g04007h0003M0@000P0007h0g@5l0=`1E`0002@000280000h04008T0003Q0@000P0008P0h0690>41
E`0002@000280000h@4008T0003Q0@000P0008T0h@680>41E`0002@000270000h@4008P0003Q0@00
0P0008P0h@670>41E`0002@000270000h04008L0003Q0@000P0008L0h@670>01E`0002@000270000
gP4008P0003P0@000P0008L0h0680=h1E`0002@000280000g@4008T0003N0@000P0008P0gP690=d1
E`0002@000290000g04008`0003M0@000P0008T0g@6<0=`1E`0002@0002<0000g04008l0003L0@00
0P0008`0g06?0=`1E`0002@0002?0000g0400980003M0@000P0008l0g06B0=d1E`0002@0002B0000
g@4009<0003O0@000P000980g@6C0=l1E`0002@0002C0000g`4009<0003Q0@000P0009<0g`6C0>41
E`0002@0002B0000h@4009<0003S0@000P0009<0h@6B0><1E`0002@0002?0000h`400980003T0@00
0P000980h`6?0>@1E`0002@0002=0000i04008l0003T0@000P0008l0i06=0>@1E`0002@0002?0000
g0400940003M0@000P0008l0g06A0=d1E`0002@0002A0000g@400980003O0@000P000940g@6B0=l1
E`0002@0002B0000g`400980003Q0@000P000980g`6B0>41E`0002@0002A0000h@400980003S0@00
0P000980h@6A0><1E`0002@0002?0000h`400940003T0@000P000940h`6?0>@1E`0002@0002?0000
i0400940003U0@000P0008l0i06A0>D1E`0002@0002A0000i@4009<0003W0@000P000940i@6C0>L1
E`0002@0002C0000i`4009@0003Y0@000P0009<0i`6D0>T1E`0002@0002D0000j@4009@0003/0@00
0P0009@0j@6D0>`1E`0002@0002C0000k04009@0003^0@000P0009@0k06C0>h1E`0002@0002B0000
kP4009<0003_0@000P0009<0kP6B0>l1E`0002@0002?0000k`400980003_0@000P000980k`6?0>l1
E`0002@0002<0000k`4008l0003_0@000P0008l0k`6<0>l1E`0002@000290000k`4008`0003_0@00
0P0008`0k`690>l1E`0002@000280000kP4008T0003_0@000P0008T0k`680>h1E`0002@000270000
k04008P0003^0@000P0008P0kP670>`1E`0002@000270000j`4008L0003/0@000P0008L0k0670>/1
E`0002@000270000jP4008P0003[0@000P0008L0j`680>X1E`0002@000280000jP4008T0003[0@00
0P0008P0jP690>/1E`0002@000280000j`4008T0003/0@000P0008T0j`680>`1E`0002@0002B0000
iP4009<0003Y0@000P000980iP6C0>T1E`0002@0002C0000j@4009<0003/0@000P0009<0j@6C0>`1
E`0002@0002B0000k04009<0003^0@000P0009<0k06B0>h1E`0002@0002A0000kP400980003_0@00
0P000980kP6A0>l1E`0002@0002?0000k`400940003_0@000P000940k`6?0>l1E`0002@000140000
PP4005@000220@000P0004@0PP5D0881E`0002@0001N0000M`400640001h0@000P000640M`5N07P1
E`0002@0001L0000N04005h0001k0@000P0005h0N05L07/1E`0002@0001K0000N`4005`0001o0@00
0P0005`0N`5K07l1E`0002@0001K0000O`4005/000220@000P0005/0O`5K0881E`0002@0001K0000
PP4005`000270@000P0005/0PP5L08L1E`0002@0001L0000Q`4005h0002:0@000P0005`0Q`5N08X1
E`0002@0001N0000RP400640002:0@000P0005h0RP5Q08X1E`0002@0001Q0000RP400680002:0@00
0P000640RP5R08X1E`0002@0001R0000RP4006D0002:0@000P000680RP5U08X1E`0002@0001U0000
Q`4006L0002:0@000P0006D0RP5W08L1E`0002@0001W0000PP4006P000270@000P0006L0Q`5X0881
E`0002@0001X0000O`4006P000220@000P0006P0PP5X07l1E`0002@0001W0000N`4006P0001o0@00
0P0006P0O`5W07/1E`0002@0001U0000N04006L0001k0@000P0006L0N`5U07P1E`0002@0001R0000
M`4006D0001h0@000P0006D0N05R07L1E`0002@0001Q0000M`400680001g0@000P000680M`5Q07L1
E`0002@0001O0000M`400640001h0@000P000640M`5O07P1E`0002@0001N0000N04005l0001i0@00
0P0005l0N05N07T1E`0002@0001M0000N@4005h0001k0@000P0005h0N@5M07/1E`0002@0001L0000
N`4005d0001o0@000P0005d0N`5L07l1E`0002@0001L0000O`4005`000220@000P0005`0O`5L0881
E`0002@0001L0000PP4005d000270@000P0005`0PP5M08L1E`0002@0001M0000Q`4005h000290@00
0P0005d0Q`5N08T1E`0002@0001N0000R@4005l0002:0@000P0005h0R@5O08X1E`0002@0001O0000
RP400640002:0@000P0005l0RP5Q08X1E`0002@0001R0000RP4006@0002:0@000P000680RP5T08X1
E`0002@0001T0000R@4006D0002:0@000P0006@0RP5U08T1E`0002@0001U0000Q`4006H000290@00
0P0006D0R@5V08L1E`0002@0001V0000PP4006L000270@000P0006H0Q`5W0881E`0002@0001W0000
O`4006L000220@000P0006L0PP5W07l1E`0002@0001V0000N`4006L0001o0@000P0006L0O`5V07/1
E`0002@0001U0000N@4006H0001k0@000P0006H0N`5U07T1E`0002@0001T0000N04006D0001i0@00
0P0006D0N@5T07P1E`0002@0001R0000M`4006@0001h0@000P0006@0N05R07L1E`0002@0001]0000
R@4006h0002:0@000P0006h0R@5]08X1E`0002@0001]0000RP4006h0002:0@000P0006d0RP5^08X1
E`0002@0001^0000RP4006l0002:0@000P0006h0RP5_08X1E`0002@0001^0000R@4006l0002:0@00
0P0006l0RP5^08T1E`0002@0001h0000M`4007/0001h0@000P0007/0M`5h07P1E`0002@0001f0000
N04007P0001k0@000P0007P0N05f07/1E`0002@0001e0000N`4007H0001o0@000P0007H0N`5e07l1
E`0002@0001e0000O`4007D000220@000P0007D0O`5e0881E`0002@0001e0000PP4007H000270@00
0P0007D0PP5f08L1E`0002@0001f0000Q`4007P0002:0@000P0007H0Q`5h08X1E`0002@0001h0000
RP4007/0002:0@000P0007P0RP5k08X1E`0002@0001k0000RP4007`0002:0@000P0007/0RP5l08X1
E`0002@0001l0000RP4007l0002:0@000P0007`0RP5o08X1E`0002@0001o0000Q`400840002:0@00
0P0007l0RP6108L1E`0002@000210000PP40088000270@000P000840Q`620881E`0002@000220000
O`40088000220@000P000880PP6207l1E`0002@000210000N`400880001o0@000P000880O`6107/1
E`0002@0001o0000N0400840001k0@000P000840N`5o07P1E`0002@0001l0000M`4007l0001h0@00
0P0007l0N05l07L1E`0002@0001k0000M`4007`0001g0@000P0007`0M`5k07L1E`0002@0001i0000
M`4007/0001h0@000P0007/0M`5i07P1E`0002@0001h0000N04007T0001i0@000P0007T0N05h07T1
E`0002@0001g0000N@4007P0001k0@000P0007P0N@5g07/1E`0002@0001f0000N`4007L0001o0@00
0P0007L0N`5f07l1E`0002@0001f0000O`4007H000220@000P0007H0O`5f0881E`0002@0001f0000
PP4007L000270@000P0007H0PP5g08L1E`0002@0001g0000Q`4007P000290@000P0007L0Q`5h08T1
E`0002@0001h0000R@4007T0002:0@000P0007P0R@5i08X1E`0002@0001i0000RP4007/0002:0@00
0P0007T0RP5k08X1E`0002@0001l0000RP4007h0002:0@000P0007`0RP5n08X1E`0002@0001n0000
R@4007l0002:0@000P0007h0RP5o08T1E`0002@0001o0000Q`40080000290@000P0007l0R@6008L1
E`0002@000200000PP40084000270@000P000800Q`610881E`0002@000210000O`40084000220@00
0P000840PP6107l1E`0002@000200000N`400840001o0@000P000840O`6007/1E`0002@0001o0000
N@400800001k0@000P000800N`5o07T1E`0002@0001n0000N04007l0001i0@000P0007l0N@5n07P1
E`0002@0001l0000M`4007h0001h0@000P0007h0N05l07L1E`0002@000280000N`4008T0001l0@00
0P0008P0N`6907`1E`0002@000280000O04008T0001l0@000P0008T0O06807`1E`0002@000270000
O04008P0001l0@000P0008P0O06707`1E`0002@000270000N`4008L0001l0@000P0008L0O06707/1
E`0002@000270000N@4008P0001k0@000P0008L0N`6807T1E`0002@000280000N04008T0001i0@00
0P0008P0N@6907P1E`0002@000290000M`4008`0001h0@000P0008T0N06<07L1E`0002@0002<0000
M`4008l0001g0@000P0008`0M`6?07L1E`0002@0002?0000M`400980001h0@000P0008l0M`6B07P1
E`0002@0002B0000N04009<0001i0@000P000980N06C07T1E`0002@0002C0000N@4009@0001k0@00
0P0009<0N@6D07/1E`0002@0002D0000N`4009@0001l0@000P0009@0N`6D07`1E`0002@0002C0000
O04009@0001n0@000P0009@0O06C07h1E`0002@0002@0000OP4009<000200@000P0009<0OP6@0801
E`0002@0002<0000P040090000220@000P000900P06<0881E`0002@0002:0000PP4008`000230@00
0P0008`0PP6:08<1E`0002@000280000P`4008X000250@000P0008X0P`6808D1E`0002@000270000
Q@4008P000280@000P0008P0Q@6708P1E`0002@000270000R04008L0002:0@000P0008L0R06708X1
E`0002@0002?0000M`400940001h0@000P0008l0M`6A07P1E`0002@0002A0000N0400980001i0@00
0P000940N06B07T1E`0002@0002B0000N@4009<0001k0@000P000980N@6C07/1E`0002@0002C0000
N`4009<0001l0@000P0009<0N`6C07`1E`0002@0002B0000O04009<0001n0@000P0009<0O06B07h1
E`0002@0002?0000OP40098000200@000P000980OP6?0801E`0002@0002<0000P04008l000220@00
0P0008l0P06<0881E`0002@000270000R04008P000290@000P0008L0R@6808P1E`0002@000280000
R04008X000280@000P0008P0R06:08P1E`0002@0002:0000R04008h0002:0@000P0008X0R06>08X1
E`0002@0002>0000RP400940002:0@000P0008h0RP6A08X1E`0002@0002A0000R@4009<0002:0@00
0P000940RP6C08T1E`0002@0002C0000R04009@000290@000P0009<0R@6D08P1E`0002@0002:0000
R04008h0002:0@000P0008X0R06>08X1E`0002@0002>0000RP400980002:0@000P0008h0RP6B08X1
E`0002@0002B0000RP4009<0002:0@000P000980RP6C08X1E`0002@0002C0000R04009@0002:0@00
0P0009<0RP6D08P1E`0002@0002D0000QP4009@000280@000P0009@0R06D08H1E`0002@000140000
704005@0000L0@000P0004@0705D01`1E`0002@0001N00004@400640000B0@000P0006404@5N0181
E`0002@0001L00004P4005h0000E0@000P0005h04P5L01D1E`0002@0001K00005@4005`0000I0@00
0P0005`05@5K01T1E`0002@0001K00006@4005/0000L0@000P0005/06@5K01`1E`0002@0001K0000
704005`0000Q0@000P0005/0705L0241E`0002@0001L00008@4005h0000T0@000P0005`08@5N02@1
E`0002@0001N000090400640000T0@000P0005h0905Q02@1E`0002@0001Q000090400680000T0@00
0P000640905R02@1E`0002@0001R0000904006D0000T0@000P000680905U02@1E`0002@0001U0000
8@4006L0000T0@000P0006D0905W0241E`0002@0001W0000704006P0000Q0@000P0006L08@5X01`1
E`0002@0001X00006@4006P0000L0@000P0006P0705X01T1E`0002@0001W00005@4006P0000I0@00
0P0006P06@5W01D1E`0002@0001U00004P4006L0000E0@000P0006L05@5U0181E`0002@0001R0000
4@4006D0000B0@000P0006D04P5R0141E`0002@0001Q00004@400680000A0@000P0006804@5Q0141
E`0002@0001O00004@400640000B0@000P0006404@5O0181E`0002@0001N00004P4005l0000C0@00
0P0005l04P5N01<1E`0002@0001M00004`4005h0000E0@000P0005h04`5M01D1E`0002@0001L0000
5@4005d0000I0@000P0005d05@5L01T1E`0002@0001L00006@4005`0000L0@000P0005`06@5L01`1
E`0002@0001L0000704005d0000Q0@000P0005`0705M0241E`0002@0001M00008@4005h0000S0@00
0P0005d08@5N02<1E`0002@0001N00008`4005l0000T0@000P0005h08`5O02@1E`0002@0001O0000
90400640000T0@000P0005l0905Q02@1E`0002@0001R0000904006@0000T0@000P000680905T02@1
E`0002@0001T00008`4006D0000T0@000P0006@0905U02<1E`0002@0001U00008@4006H0000S0@00
0P0006D08`5V0241E`0002@0001V0000704006L0000Q0@000P0006H08@5W01`1E`0002@0001W0000
6@4006L0000L0@000P0006L0705W01T1E`0002@0001V00005@4006L0000I0@000P0006L06@5V01D1
E`0002@0001U00004`4006H0000E0@000P0006H05@5U01<1E`0002@0001T00004P4006D0000C0@00
0P0006D04`5T0181E`0002@0001R00004@4006@0000B0@000P0006@04P5R0141E`0002@0001]0000
8`4006h0000T0@000P0006h08`5]02@1E`0002@0001]0000904006h0000T0@000P0006d0905^02@1
E`0002@0001^0000904006l0000T0@000P0006h0905_02@1E`0002@0001^00008`4006l0000T0@00
0P0006l0905^02<1E`0002@0001h00004@4007/0000B0@000P0007/04@5h0181E`0002@0001f0000
4P4007P0000E0@000P0007P04P5f01D1E`0002@0001e00005@4007H0000I0@000P0007H05@5e01T1
E`0002@0001e00006@4007D0000L0@000P0007D06@5e01`1E`0002@0001e0000704007H0000Q0@00
0P0007D0705f0241E`0002@0001f00008@4007P0000T0@000P0007H08@5h02@1E`0002@0001h0000
904007/0000T0@000P0007P0905k02@1E`0002@0001k0000904007`0000T0@000P0007/0905l02@1
E`0002@0001l0000904007l0000T0@000P0007`0905o02@1E`0002@0001o00008@400840000T0@00
0P0007l090610241E`0002@00021000070400880000Q0@000P0008408@6201`1E`0002@000220000
6@400880000L0@000P000880706201T1E`0002@0002100005@400880000I0@000P0008806@6101D1
E`0002@0001o00004P400840000E0@000P0008405@5o0181E`0002@0001l00004@4007l0000B0@00
0P0007l04P5l0141E`0002@0001k00004@4007`0000A0@000P0007`04@5k0141E`0002@0001i0000
4@4007/0000B0@000P0007/04@5i0181E`0002@0001h00004P4007T0000C0@000P0007T04P5h01<1
E`0002@0001g00004`4007P0000E0@000P0007P04`5g01D1E`0002@0001f00005@4007L0000I0@00
0P0007L05@5f01T1E`0002@0001f00006@4007H0000L0@000P0007H06@5f01`1E`0002@0001f0000
704007L0000Q0@000P0007H0705g0241E`0002@0001g00008@4007P0000S0@000P0007L08@5h02<1
E`0002@0001h00008`4007T0000T0@000P0007P08`5i02@1E`0002@0001i0000904007/0000T0@00
0P0007T0905k02@1E`0002@0001l0000904007h0000T0@000P0007`0905n02@1E`0002@0001n0000
8`4007l0000T0@000P0007h0905o02<1E`0002@0001o00008@400800000S0@000P0007l08`600241
E`0002@00020000070400840000Q0@000P0008008@6101`1E`0002@0002100006@400840000L0@00
0P000840706101T1E`0002@0002000005@400840000I0@000P0008406@6001D1E`0002@0001o0000
4`400800000E0@000P0008005@5o01<1E`0002@0001n00004P4007l0000C0@000P0007l04`5n0181
E`0002@0001l00004@4007h0000B0@000P0007h04P5l0141E`0002@0002:0000504008`0000E0@00
0P0008X05@6<01@1E`0002@0002<00004@4008h0000D0@000P0008`0506>0141E`0002@0002>0000
4@4008h0000T0@000P0008h04@6>02@1E`0002@0002=00004P4008d0000T0@000P0008d04P6=02@1
E`0002@0002:000090400980000T0@000P0008X0906B02@1E`0002@0001N0000[0000640002]0000
0P000640[01N0:d0E`0002@0001L0000[@0005h0002`00000P0005h0[@1L0;00E`0002@0001K0000
/00005`0002d00000P0005`0/01K0;@0E`0002@0001K0000]00005/0002g00000P0005/0]01K0;L0
E`0002@0001K0000]`0005`0002l00000P0005/0]`1L0;`0E`0002@0001L0000_00005h0002o0000
0P0005`0_01N0;l0E`0002@0001N0000_`000640002o00000P0005h0_`1Q0;l0E`0002@0001Q0000
_`000680002o00000P000640_`1R0;l0E`0002@0001R0000_`0006D0002o00000P000680_`1U0;l0
E`0002@0001U0000_00006L0002o00000P0006D0_`1W0;`0E`0002@0001W0000]`0006P0002l0000
0P0006L0_01X0;L0E`0002@0001X0000]00006P0002g00000P0006P0]`1X0;@0E`0002@0001W0000
/00006P0002d00000P0006P0]01W0;00E`0002@0001U0000[@0006L0002`00000P0006L0/01U0:d0
E`0002@0001R0000[00006D0002]00000P0006D0[@1R0:`0E`0002@0001Q0000[0000680002/0000
0P000680[01Q0:`0E`0002@0001O0000[0000640002]00000P000640[01O0:d0E`0002@0001N0000
[@0005l0002^00000P0005l0[@1N0:h0E`0002@0001M0000[P0005h0002`00000P0005h0[P1M0;00
E`0002@0001L0000/00005d0002d00000P0005d0/01L0;@0E`0002@0001L0000]00005`0002g0000
0P0005`0]01L0;L0E`0002@0001L0000]`0005d0002l00000P0005`0]`1M0;`0E`0002@0001M0000
_00005h0002n00000P0005d0_01N0;h0E`0002@0001N0000_P0005l0002o00000P0005h0_P1O0;l0
E`0002@0001O0000_`000640002o00000P0005l0_`1Q0;l0E`0002@0001R0000_`0006@0002o0000
0P000680_`1T0;l0E`0002@0001T0000_P0006D0002o00000P0006@0_`1U0;h0E`0002@0001U0000
_00006H0002n00000P0006D0_P1V0;`0E`0002@0001V0000]`0006L0002l00000P0006H0_01W0;L0
E`0002@0001W0000]00006L0002g00000P0006L0]`1W0;@0E`0002@0001V0000/00006L0002d0000
0P0006L0]01V0;00E`0002@0001U0000[P0006H0002`00000P0006H0/01U0:h0E`0002@0001T0000
[@0006D0002^00000P0006D0[P1T0:d0E`0002@0001R0000[00006@0002]00000P0006@0[@1R0:`0
E`0002@0001]0000_P0006h0002o00000P0006h0_P1]0;l0E`0002@0001]0000_`0006h0002o0000
0P0006d0_`1^0;l0E`0002@0001^0000_`0006l0002o00000P0006h0_`1_0;l0E`0002@0001^0000
_P0006l0002o00000P0006l0_`1^0;h0E`0002@0001h0000[00007/0002]00000P0007/0[01h0:d0
E`0002@0001f0000[@0007P0002`00000P0007P0[@1f0;00E`0002@0001e0000/00007H0002d0000
0P0007H0/01e0;@0E`0002@0001e0000]00007D0002g00000P0007D0]01e0;L0E`0002@0001e0000
]`0007H0002l00000P0007D0]`1f0;`0E`0002@0001f0000_00007P0002o00000P0007H0_01h0;l0
E`0002@0001h0000_`0007/0002o00000P0007P0_`1k0;l0E`0002@0001k0000_`0007`0002o0000
0P0007/0_`1l0;l0E`0002@0001l0000_`0007l0002o00000P0007`0_`1o0;l0E`0002@0001o0000
_0000840002o00000P0007l0_`210;`0E`0002@000210000]`000880002l00000P000840_0220;L0
E`0002@000220000]0000880002g00000P000880]`220;@0E`0002@000210000/0000880002d0000
0P000880]0210;00E`0002@0001o0000[@000840002`00000P000840/01o0:d0E`0002@0001l0000
[00007l0002]00000P0007l0[@1l0:`0E`0002@0001k0000[00007`0002/00000P0007`0[01k0:`0
E`0002@0001i0000[00007/0002]00000P0007/0[01i0:d0E`0002@0001h0000[@0007T0002^0000
0P0007T0[@1h0:h0E`0002@0001g0000[P0007P0002`00000P0007P0[P1g0;00E`0002@0001f0000
/00007L0002d00000P0007L0/01f0;@0E`0002@0001f0000]00007H0002g00000P0007H0]01f0;L0
E`0002@0001f0000]`0007L0002l00000P0007H0]`1g0;`0E`0002@0001g0000_00007P0002n0000
0P0007L0_01h0;h0E`0002@0001h0000_P0007T0002o00000P0007P0_P1i0;l0E`0002@0001i0000
_`0007/0002o00000P0007T0_`1k0;l0E`0002@0001l0000_`0007h0002o00000P0007`0_`1n0;l0
E`0002@0001n0000_P0007l0002o00000P0007h0_`1o0;h0E`0002@0001o0000_0000800002n0000
0P0007l0_P200;`0E`0002@000200000]`000840002l00000P000800_0210;L0E`0002@000210000
]0000840002g00000P000840]`210;@0E`0002@000200000/0000840002d00000P000840]0200;00
E`0002@0001o0000[P000800002`00000P000800/01o0:h0E`0002@0001n0000[@0007l0002^0000
0P0007l0[P1n0:d0E`0002@0001l0000[00007h0002]00000P0007h0[@1l0:`0E`0002@0002:0000
[00008d0002]00000P0008d0[02:0:d0E`0002@000280000[@0008X0002`00000P0008X0[@280;00
E`0002@000270000/00008P0002d00000P0008P0/0270;@0E`0002@000270000]00008L0002g0000
0P0008L0]0270;L0E`0002@000270000]`0008P0002l00000P0008L0]`280;`0E`0002@000280000
_00008X0002o00000P0008P0_02:0;l0E`0002@0002:0000_`0008d0002o00000P0008X0_`2=0;l0
E`0002@0002=0000_`0008h0002o00000P0008d0_`2>0;l0E`0002@0002>0000_`000940002o0000
0P0008h0_`2A0;l0E`0002@0002A0000_00009<0002o00000P000940_`2C0;`0E`0002@0002C0000
]`0009@0002l00000P0009<0_02D0;L0E`0002@0002D0000]00009@0002g00000P0009@0]`2D0;@0
E`0002@0002C0000/00009@0002d00000P0009@0]02C0;00E`0002@0002A0000[@0009<0002`0000
0P0009<0/02A0:d0E`0002@0002>0000[0000940002]00000P000940[@2>0:`0E`0002@0002=0000
[00008h0002/00000P0008h0[02=0:`0E`0002@0002;0000[00008d0002]00000P0008d0[02;0:d0
E`0002@0002:0000[@0008/0002^00000P0008/0[@2:0:h0E`0002@000290000[P0008X0002`0000
0P0008X0[P290;00E`0002@000280000/00008T0002d00000P0008T0/0280;@0E`0002@000280000
]00008P0002g00000P0008P0]0280;L0E`0002@000280000]`0008T0002l00000P0008P0]`290;`0
E`0002@000290000_00008X0002n00000P0008T0_02:0;h0E`0002@0002:0000_P0008/0002o0000
0P0008X0_P2;0;l0E`0002@0002;0000_`0008d0002o00000P0008/0_`2=0;l0E`0002@0002>0000
_`000900002o00000P0008h0_`2@0;l0E`0002@0002@0000_P000940002o00000P000900_`2A0;h0
E`0002@0002A0000_0000980002n00000P000940_P2B0;`0E`0002@0002B0000]`0009<0002l0000
0P000980_02C0;L0E`0002@0002C0000]00009<0002g00000P0009<0]`2C0;@0E`0002@0002B0000
/00009<0002d00000P0009<0]02B0;00E`0002@0002A0000[P000980002`00000P000980/02A0:h0
E`0002@0002@0000[@000940002^00000P000940[P2@0:d0E`0002@0002>0000[0000900002]0000
0P000900[@2>0:`0E`0002@0001N0000AP000640001700000P000640AP1N04L0E`0002@0001L0000
A`0005h0001:00000P0005h0A`1L04X0E`0002@0001K0000BP0005`0001>00000P0005`0BP1K04h0
E`0002@0001K0000CP0005/0001A00000P0005/0CP1K0540E`0002@0001K0000D@0005`0001F0000
0P0005/0D@1L05H0E`0002@0001L0000EP0005h0001I00000P0005`0EP1N05T0E`0002@0001N0000
F@000640001I00000P0005h0F@1Q05T0E`0002@0001Q0000F@000680001I00000P000640F@1R05T0
E`0002@0001R0000F@0006D0001I00000P000680F@1U05T0E`0002@0001U0000EP0006L0001I0000
0P0006D0F@1W05H0E`0002@0001W0000D@0006P0001F00000P0006L0EP1X0540E`0002@0001X0000
CP0006P0001A00000P0006P0D@1X04h0E`0002@0001W0000BP0006P0001>00000P0006P0CP1W04X0
E`0002@0001U0000A`0006L0001:00000P0006L0BP1U04L0E`0002@0001R0000AP0006D000170000
0P0006D0A`1R04H0E`0002@0001Q0000AP000680001600000P000680AP1Q04H0E`0002@0001O0000
AP000640001700000P000640AP1O04L0E`0002@0001N0000A`0005l0001800000P0005l0A`1N04P0
E`0002@0001M0000B00005h0001:00000P0005h0B01M04X0E`0002@0001L0000BP0005d0001>0000
0P0005d0BP1L04h0E`0002@0001L0000CP0005`0001A00000P0005`0CP1L0540E`0002@0001L0000
D@0005d0001F00000P0005`0D@1M05H0E`0002@0001M0000EP0005h0001H00000P0005d0EP1N05P0
E`0002@0001N0000F00005l0001I00000P0005h0F01O05T0E`0002@0001O0000F@000640001I0000
0P0005l0F@1Q05T0E`0002@0001R0000F@0006@0001I00000P000680F@1T05T0E`0002@0001T0000
F00006D0001I00000P0006@0F@1U05P0E`0002@0001U0000EP0006H0001H00000P0006D0F01V05H0
E`0002@0001V0000D@0006L0001F00000P0006H0EP1W0540E`0002@0001W0000CP0006L0001A0000
0P0006L0D@1W04h0E`0002@0001V0000BP0006L0001>00000P0006L0CP1V04X0E`0002@0001U0000
B00006H0001:00000P0006H0BP1U04P0E`0002@0001T0000A`0006D0001800000P0006D0B01T04L0
E`0002@0001R0000AP0006@0001700000P0006@0A`1R04H0E`0002@0001]0000F00006h0001I0000
0P0006h0F01]05T0E`0002@0001]0000F@0006h0001I00000P0006d0F@1^05T0E`0002@0001^0000
F@0006l0001I00000P0006h0F@1_05T0E`0002@0001^0000F00006l0001I00000P0006l0F@1^05P0
E`0002@0001h0000AP0007/0001700000P0007/0AP1h04L0E`0002@0001f0000A`0007P0001:0000
0P0007P0A`1f04X0E`0002@0001e0000BP0007H0001>00000P0007H0BP1e04h0E`0002@0001e0000
CP0007D0001A00000P0007D0CP1e0540E`0002@0001e0000D@0007H0001F00000P0007D0D@1f05H0
E`0002@0001f0000EP0007P0001I00000P0007H0EP1h05T0E`0002@0001h0000F@0007/0001I0000
0P0007P0F@1k05T0E`0002@0001k0000F@0007`0001I00000P0007/0F@1l05T0E`0002@0001l0000
F@0007l0001I00000P0007`0F@1o05T0E`0002@0001o0000EP000840001I00000P0007l0F@2105H0
E`0002@000210000D@000880001F00000P000840EP220540E`0002@000220000CP000880001A0000
0P000880D@2204h0E`0002@000210000BP000880001>00000P000880CP2104X0E`0002@0001o0000
A`000840001:00000P000840BP1o04L0E`0002@0001l0000AP0007l0001700000P0007l0A`1l04H0
E`0002@0001k0000AP0007`0001600000P0007`0AP1k04H0E`0002@0001i0000AP0007/000170000
0P0007/0AP1i04L0E`0002@0001h0000A`0007T0001800000P0007T0A`1h04P0E`0002@0001g0000
B00007P0001:00000P0007P0B01g04X0E`0002@0001f0000BP0007L0001>00000P0007L0BP1f04h0
E`0002@0001f0000CP0007H0001A00000P0007H0CP1f0540E`0002@0001f0000D@0007L0001F0000
0P0007H0D@1g05H0E`0002@0001g0000EP0007P0001H00000P0007L0EP1h05P0E`0002@0001h0000
F00007T0001I00000P0007P0F01i05T0E`0002@0001i0000F@0007/0001I00000P0007T0F@1k05T0
E`0002@0001l0000F@0007h0001I00000P0007`0F@1n05T0E`0002@0001n0000F00007l0001I0000
0P0007h0F@1o05P0E`0002@0001o0000EP000800001H00000P0007l0F02005H0E`0002@000200000
D@000840001F00000P000800EP210540E`0002@000210000CP000840001A00000P000840D@2104h0
E`0002@000200000BP000840001>00000P000840CP2004X0E`0002@0001o0000B0000800001:0000
0P000800BP1o04P0E`0002@0001n0000A`0007l0001800000P0007l0B01n04L0E`0002@0001l0000
AP0007h0001700000P0007h0A`1l04H0E`0002@0002:0000B@0008`0001:00000P0008X0BP2<04T0
E`0002@0002<0000AP0008h0001900000P0008`0B@2>04H0E`0002@0002>0000AP0008h0001I0000
0P0008h0AP2>05T0E`0002@0002=0000A`0008d0001I00000P0008d0A`2=05T0E`0002@0002:0000
F@000980001I00000P0008X0F@2B05T0E`0002@0002V0000Q@<00:H0002=0`000P000:H0Q@>V08d3
E`0002@0003N0000Q@<00=h0002=0`000P000=h0Q@?N08d3E`0002@0000G0@00Q@<001L1002=0`00
0P0001L1Q@<G0Hd3E`0002@0001?0@00Q@<004l1002=0`000P0004l1Q@=?0Hd3E`0002@000280@00
Q@<008P1002=0`000P0008P1Q@>80Hd3E`0002@000300@00Q@<00<01002=0`000P000<01Q@?00Hd3
E`0002@0003h0@00Q@<00?P1002=0`000P000?P1Q@?h0Hd3E`0002@0000a0P00Q@<00342002=0`00
0P000342Q@<a0Xd3E`0002@0001Y0P00Q@<006T2002=0`000P0006T2Q@=Y0Xd3E`0002@0002R0P00
Q@<00:82002=0`000P000:82Q@>R0Xd3E`0002@0003J0P00Q@<00=X2002=0`000P000=X2Q@?J0Xd3
E`0002@0000B0`00Q@<00183002=0`000P000183Q@<B0hd3E`0002@0001;0`00Q@<004/3002=0`00
0P0004/3Q@=;0hd3E`0002@000230`00Q@<008<3002=0`000P0008<3Q@>30hd3E`0002@0002k0`00
Q@<00;/3002=0`000P000;/3Q@>k0hd3E`0002@0003d0`00Q@<00?@3002=0`000P000?@3Q@?d0hd3
E`0002@0000/1000Q@<002`4002=0`000P0002`4Q@</18d3E`0002@0001U1000Q@<006D4002=0`00
0P0006D4Q@=U18d3E`0002@0002M1000Q@<009d4002=0`000P0009d4Q@>M18d3E`0002@0003E1000
Q@<00=D4002=0`000P000=D4Q@?E18d3E`0002@0002Q0P00b`<00:42003N0`000P000:42b`>Q0]h3
E`0002@0002Q0P00b`<00:42003N0`000P000:42b`>Q0]h3E`0002@0002J0P00b`<009/2003@0`00
0P0009/2b`>J0]03E`0002@0002J0P00b`<009X2003@0`000P0009X2d0>J0//3E`0002@0002J0P00
b`<00:P2003;0`000P0009X2b`>X0//3E`0002@0002X0P00b`<00:P2003@0`000P000:P2b`>X0]03
E`0002@0002W0P00b`<00:P2003@0`000P000:P2d0>W0//3E`0002@0002N0P00gP<00:@2003N0`00
0P0009h2gP>T0]h3E`0002@0002]0P00b`<00:h2003<0`000P000:h2b`>]0/`3E`0002@0002]0P00
c0<00:h2003=0`000P000:d2c0>^0/d3E`0002@0002^0P00c0<00:l2003=0`000P000:h2c@>_0/`3
E`0002@0002^0P00b`<00:l2003<0`000P000:l2c0>^0//3E`0002@0002^0P00d@<00:h2003N0`00
0P000:h2d@>^0]h3E`0002@0002_0P00d@<00:l2003N0`000P000:l2d@>_0]h3E`0002@0002[0P00
d@<00:l2003A0`000P000:/2d@>_0]43E`0002@0002[0P00gP<00;82003N0`000P000:/2gP>b0]h3
E`0002@0002h0P00d@<00;P2003N0`000P000;P2d@>h0]h3E`0002@0002i0P00d@<00;T2003N0`00
0P000;T2d@>i0]h3E`0002@0002i0P00dP<00;/2003D0`000P000;T2e0>k0]83E`0002@0002k0P00
d@<00;d2003B0`000P000;/2dP>m0]43E`0002@0002m0P00d@<00;l2003A0`000P000;d2d@>o0]43
E`0002@0002o0P00d@<00<82003B0`000P000;l2d@?20]83E`0002@000320P00dP<00<<2003D0`00
0P000<82dP?30]@3E`0002@000330P00e0<00<<2003N0`000P000<<2e0?30]h3E`0002@0002o0P00
d@<00<42003B0`000P000;l2d@?10]83E`0002@000310P00dP<00<82003D0`000P000<42dP?20]@3
E`0002@000320P00e0<00<82003N0`000P000<82e0?20]h3E`0002@000330P00dP<00<@2003D0`00
0P000<<2e0?40]83E`0002@000340P00d@<00<L2003B0`000P000<@2dP?70]43E`0002@000370P00
d@<00<T2003A0`000P000<L2d@?90]43E`0002@000390P00d@<00<`2003B0`000P000<T2d@?<0]83
E`0002@0003<0P00dP<00<d2003D0`000P000<`2dP?=0]@3E`0002@0003=0P00e0<00<d2003N0`00
0P000<d2e0?=0]h3E`0002@000390P00d@<00</2003B0`000P000<T2d@?;0]83E`0002@0003;0P00
dP<00<`2003D0`000P000</2dP?<0]@3E`0002@0003<0P00e0<00<`2003N0`000P000<`2e0?<0]h3
E`0002@0002e0P00d@<00;T2003A0`000P000;D2d@>i0]43E`0002@0002e0P00gP<00;`2003N0`00
0P000;D2gP>l0]h3E`0002@0002o0P00gP<00<D2003N0`000P000;l2gP?50]h3E`0002@000390P00
gP<00<l2003N0`000P000<T2gP??0]h3E`0002@0003D0P00e`<00=l2003G0`000P000=@2e`?O0]L3
E`0002@0003O0P00e@<00=l2003G0`000P000=l2e`?O0]D3E`0002@0003N0P00d`<00=l2003E0`00
0P000=l2e@?N0]<3E`0002@0003M0P00dP<00=h2003C0`000P000=h2d`?M0]83E`0002@0003K0P00
d@<00=d2003B0`000P000=d2dP?K0]43E`0002@0003I0P00d@<00=/2003A0`000P000=/2d@?I0]43
E`0002@0003F0P00d@<00=T2003B0`000P000=T2d@?F0]83E`0002@0003D0P00dP<00=H2003D0`00
0P000=H2dP?D0]@3E`0002@0003C0P00e0<00=@2003G0`000P000=@2e0?C0]L3E`0002@0003C0P00
e`<00=<2003I0`000P000=<2e`?C0]T3E`0002@0003C0P00f@<00=@2003L0`000P000=<2f@?D0]`3
E`0002@0003D0P00g0<00=H2003N0`000P000=@2g0?F0]h3E`0002@0003F0P00gP<00=T2003N0`00
0P000=H2gP?I0]h3E`0002@0003I0P00gP<00=X2003N0`000P000=T2gP?J0]h3E`0002@0003J0P00
gP<00=d2003N0`000P000=X2gP?M0]h3E`0002@0003M0P00g0<00=l2003N0`000P000=d2gP?O0]`3
E`0002@0003N0P00e0<00=h2003G0`000P000=h2e`?N0]@3E`0002@0003M0P00dP<00=h2003D0`00
0P000=h2e0?M0]83E`0002@0003G0P00d@<00=T2003B0`000P000=T2d@?G0]83E`0002@0003E0P00
dP<00=L2003D0`000P000=L2dP?E0]@3E`0002@0003D0P00e0<00=D2003G0`000P000=D2e0?D0]L3
E`0002@0003D0P00e`<00=@2003I0`000P000=@2e`?D0]T3E`0002@0003D0P00f@<00=D2003L0`00
0P000=@2f@?E0]`3E`0002@0003E0P00g0<00=L2003N0`000P000=D2g0?G0]h3E`0002@0003G0P00
gP<00=T2003N0`000P000=L2gP?I0]h3E`0002@0002R0000WP<00:@0002O0`000P000:80W`>T09h3
E`0002@0002T0000V`<00:H0002N0`000P000:@0WP>V09/3E`0002@0002V0000V`<00:H0002^0`00
0P000:H0V`>V0:h3E`0002@0002U0000W0<00:D0002^0`000P000:D0W0>U0:h3E`0002@0002R0000
[P<00:X0002^0`000P000:80[P>Z0:h3E`0002@0003I0000W`<00=X0002P0`000P000=T0W`?J0:03
E`0002@0003I0000X0<00=X0002P0`000P000=X0X0?I0:03E`0002@0003H0000X0<00=T0002P0`00
0P000=T0X0?H0:03E`0002@0003H0000W`<00=P0002P0`000P000=P0X0?H09l3E`0002@0003H0000
W@<00=T0002O0`000P000=P0W`?I09d3E`0002@0003I0000W0<00=X0002M0`000P000=T0W@?J09`3
E`0002@0003J0000V`<00=d0002L0`000P000=X0W0?M09/3E`0002@0003M0000V`<00>00002K0`00
0P000=d0V`?P09/3E`0002@0003P0000V`<00><0002L0`000P000>00V`?S09`3E`0002@0003S0000
W0<00>@0002M0`000P000><0W0?T09d3E`0002@0003T0000W@<00>D0002O0`000P000>@0W@?U09l3
E`0002@0003U0000W`<00>D0002P0`000P000>D0W`?U0:03E`0002@0003T0000X0<00>D0002R0`00
0P000>D0X0?T0:83E`0002@0003Q0000XP<00>@0002T0`000P000>@0XP?Q0:@3E`0002@0003M0000
Y0<00>40002V0`000P000>40Y0?M0:H3E`0002@0003K0000YP<00=d0002W0`000P000=d0YP?K0:L3
E`0002@0003I0000Y`<00=/0002Y0`000P000=/0Y`?I0:T3E`0002@0003H0000Z@<00=T0002/0`00
0P000=T0Z@?H0:`3E`0002@0003H0000[0<00=P0002^0`000P000=P0[0?H0:h3E`0002@0003P0000
V`<00>80002L0`000P000>00V`?R09`3E`0002@0003R0000W0<00><0002M0`000P000>80W0?S09d3
E`0002@0003S0000W@<00>@0002O0`000P000><0W@?T09l3E`0002@0003T0000W`<00>@0002P0`00
0P000>@0W`?T0:03E`0002@0003S0000X0<00>@0002R0`000P000>@0X0?S0:83E`0002@0003P0000
XP<00><0002T0`000P000><0XP?P0:@3E`0002@0003M0000Y0<00>00002V0`000P000>00Y0?M0:H3
E`0002@0003H0000[0<00=T0002]0`000P000=P0[@?I0:`3E`0002@0003I0000[0<00=/0002/0`00
0P000=T0[0?K0:`3E`0002@0003K0000[0<00=l0002^0`000P000=/0[0?O0:h3E`0002@0003O0000
[P<00>80002^0`000P000=l0[P?R0:h3E`0002@0003R0000[@<00>@0002^0`000P000>80[P?T0:d3
E`0002@0003T0000[0<00>D0002]0`000P000>@0[@?U0:`3E`0002@0003K0000[0<00=l0002^0`00
0P000=/0[0?O0:h3E`0002@0003O0000[P<00><0002^0`000P000=l0[P?S0:h3E`0002@0003S0000
[P<00>@0002^0`000P000><0[P?T0:h3E`0002@0003T0000[0<00>D0002^0`000P000>@0[P?U0:`3
E`0002@0003U0000ZP<00>D0002/0`000P000>D0[0?U0:X3E`0002@0000A0@00W`<00181002P0`00
0P000141W`<B0J03E`0002@0000A0@00X0<00181002P0`000P000181X0<A0J03E`0002@0000@0@00
X0<00141002P0`000P000141X0<@0J03E`0002@0000@0@00W`<00101002P0`000P000101X0<@0Il3
E`0002@0000@0@00W@<00141002O0`000P000101W`<A0Id3E`0002@0000A0@00W0<00181002M0`00
0P000141W@<B0I`3E`0002@0000B0@00V`<001D1002L0`000P000181W0<E0I/3E`0002@0000E0@00
V`<001P1002K0`000P0001D1V`<H0I/3E`0002@0000H0@00V`<001/1002L0`000P0001P1V`<K0I`3
E`0002@0000K0@00W0<001`1002N0`000P0001/1W0<L0Ih3E`0002@0000L0@00WP<001`1002P0`00
0P0001`1WP<L0J03E`0002@0000K0@00X0<001`1002R0`000P0001`1X0<K0J83E`0002@0000H0@00
XP<001/1002S0`000P0001/1XP<H0J<3E`0002@0000F0@00X`<001P1002S0`000P0001P1X`<F0J<3
E`0002@0000H0@00V`<001X1002L0`000P0001P1V`<J0I`3E`0002@0000J0@00W0<001/1002N0`00
0P0001X1W0<K0Ih3E`0002@0000K0@00WP<001/1002P0`000P0001/1WP<K0J03E`0002@0000J0@00
X0<001/1002R0`000P0001/1X0<J0J83E`0002@0000H0@00XP<001X1002S0`000P0001X1XP<H0J<3
E`0002@0000H0@00X`<001X1002T0`000P0001P1X`<J0J@3E`0002@0000J0@00Y0<001`1002V0`00
0P0001X1Y0<L0JH3E`0002@0000L0@00YP<001d1002X0`000P0001`1YP<M0JP3E`0002@0000M0@00
Z0<001d1002[0`000P0001d1Z0<M0J/3E`0002@0000L0@00Z`<001d1002]0`000P0001d1Z`<L0Jd3
E`0002@0000K0@00[@<001`1002^0`000P0001`1[@<K0Jh3E`0002@0000H0@00[P<001/1002^0`00
0P0001/1[P<H0Jh3E`0002@0000E0@00[P<001P1002^0`000P0001P1[P<E0Jh3E`0002@0000B0@00
[P<001D1002^0`000P0001D1[P<B0Jh3E`0002@0000A0@00[@<00181002^0`000P000181[P<A0Jd3
E`0002@0000@0@00Z`<00141002]0`000P000141[@<@0J/3E`0002@0000@0@00ZP<00101002[0`00
0P000101Z`<@0JX3E`0002@0000@0@00Z@<00141002Z0`000P000101ZP<A0JT3E`0002@0000A0@00
Z@<00181002Z0`000P000141Z@<B0JX3E`0002@0000A0@00ZP<00181002[0`000P000181ZP<A0J/3
E`0002@0000K0@00Y@<001`1002X0`000P0001/1Y@<L0JP3E`0002@0000L0@00Z0<001`1002[0`00
0P0001`1Z0<L0J/3E`0002@0000K0@00Z`<001`1002]0`000P0001`1Z`<K0Jd3E`0002@0000J0@00
[@<001/1002^0`000P0001/1[@<J0Jh3E`0002@0000H0@00[P<001X1002^0`000P0001X1[P<H0Jh3
E`0002@0001@0@00W@<00501002^0`000P000501W@=@0Jh3E`0002@0001A0@00V`<00541002^0`00
0P000541V`=A0Jh3E`0002@000170@00V`<00541002Y0`000P000541V`=70JT3E`0002@000170@00
Z@<005H1002Y0`000P0004L1Z@=F0JT3E`0002@0001>0@00[P<005@1002^0`000P0004h1[P=D0Jh3
E`0002@000210@00V`<008<1002T0`000P0008<1V`>10J@3E`0002@000210@00XP<008<1002T0`00
0P000841Y0>30J83E`0002@000230@00X@<008H1002R0`000P0008<1XP>60J43E`0002@000260@00
X@<008P1002Q0`000P0008H1X@>80J43E`0002@000280@00X@<008/1002R0`000P0008P1X@>;0J83
E`0002@0002;0@00XP<008d1002T0`000P0008/1XP>=0J@3E`0002@0002=0@00Y0<008h1002W0`00
0P0008d1Y0>>0JL3E`0002@0002>0@00Y`<008h1002Y0`000P0008h1Y`>>0JT3E`0002@0002=0@00
Z@<008h1002/0`000P0008h1Z@>=0J`3E`0002@0002;0@00[0<008d1002^0`000P0008d1[0>;0Jh3
E`0002@000280@00[P<008/1002^0`000P0008/1[P>80Jh3E`0002@000260@00[P<008P1002^0`00
0P0008P1[P>60Jh3E`0002@000230@00[P<008H1002^0`000P0008H1[P>30Jh3E`0002@000220@00
[@<008<1002^0`000P0008<1[P>20Jd3E`0002@000210@00Z`<00881002]0`000P000881[@>10J/3
E`0002@000210@00ZP<00841002[0`000P000841Z`>10JX3E`0002@000210@00Z@<00881002Z0`00
0P000841ZP>20JT3E`0002@000220@00Z@<008<1002Z0`000P000881Z@>30JX3E`0002@000220@00
ZP<008<1002[0`000P0008<1ZP>20J/3E`0002@000280@00X@<008X1002R0`000P0008P1X@>:0J83
E`0002@0002:0@00XP<008`1002T0`000P0008X1XP><0J@3E`0002@0002<0@00Y0<008d1002W0`00
0P0008`1Y0>=0JL3E`0002@0002=0@00Y`<008d1002Y0`000P0008d1Y`>=0JT3E`0002@0002<0@00
Z@<008d1002/0`000P0008d1Z@><0J`3E`0002@0002:0@00[0<008`1002^0`000P0008`1[0>:0Jh3
E`0002@000280@00[P<008X1002^0`000P0008X1[P>80Jh3E`0002@000230@00V`<008`1002K0`00
0P0008<1V`><0I/3E`0002@000230@00W0<008L1002L0`000P0008<1W0>70I`3E`0002@000270@00
V`<008`1002L0`000P0008L1W0><0I/3E`0002@000330@00WP<00<@1002O0`000P000<@1WP?30Il3
E`0002@000330@00W`<00<@1002P0`000P000<<1W`?40J03E`0002@000340@00W`<00<D1002P0`00
0P000<@1X0?50Il3E`0002@000350@00WP<00<D1002O0`000P000<D1W`?50Ih3E`0002@000340@00
W0<00<D1002N0`000P000<D1WP?40I`3E`0002@000320@00V`<00<@1002L0`000P000<@1W0?20I/3
E`0002@0002o0@00V`<00<81002K0`000P000<81V`>o0I/3E`0002@0002m0@00V`<00;l1002L0`00
0P000;l1V`>m0I`3E`0002@0002k0@00W0<00;d1002N0`000P000;d1W0>k0Ih3E`0002@0002j0@00
WP<00;/1002P0`000P000;/1WP>j0J03E`0002@0002i0@00X0<00;X1002S0`000P000;X1X0>i0J<3
E`0002@0002i0@00X`<00;T1002Y0`000P000;T1X`>i0JT3E`0002@0002i0@00Z@<00;X1002/0`00
0P000;T1Z@>j0J`3E`0002@0002j0@00[0<00;`1002^0`000P000;X1[0>l0Jh3E`0002@0002l0@00
[P<00;l1002^0`000P000;`1[P>o0Jh3E`0002@0002o0@00[P<00<01002^0`000P000;l1[P?00Jh3
E`0002@000300@00[P<00<<1002^0`000P000<01[P?30Jh3E`0002@000330@00[0<00<D1002^0`00
0P000<<1[P?50J`3E`0002@000350@00Z@<00<H1002/0`000P000<D1[0?60JT3E`0002@000360@00
Z0<00<H1002Y0`000P000<H1Z@?60JP3E`0002@000350@00Y@<00<H1002X0`000P000<H1Z0?50JD3
E`0002@000330@00X`<00<D1002U0`000P000<D1Y@?30J<3E`0002@000300@00XP<00<<1002S0`00
0P000<<1X`?00J83E`0002@0002o0@00XP<00<01002R0`000P000<01XP>o0J83E`0002@0002m0@00
XP<00;l1002S0`000P000;l1XP>m0J<3E`0002@0002k0@00X`<00;d1002U0`000P000;d1X`>k0JD3
E`0002@0002j0@00Y@<00;/1002X0`000P000;/1Y@>j0JP3E`0002@0002n0@00V`<00;l1002L0`00
0P000;l1V`>n0I`3E`0002@0002l0@00W0<00;h1002N0`000P000;h1W0>l0Ih3E`0002@0002k0@00
WP<00;`1002P0`000P000;`1WP>k0J03E`0002@0002j0@00X0<00;/1002S0`000P000;/1X0>j0J<3
E`0002@0002j0@00X`<00;X1002Y0`000P000;X1X`>j0JT3E`0002@0002j0@00Z@<00;/1002/0`00
0P000;X1Z@>k0J`3E`0002@0002k0@00[0<00;d1002^0`000P000;/1[0>m0Jh3E`0002@0002m0@00
[P<00;l1002^0`000P000;d1[P>o0Jh3E`0002@000300@00[P<00<81002^0`000P000<01[P?20Jh3
E`0002@000320@00[0<00<@1002^0`000P000<81[P?40J`3E`0002@000340@00Z@<00<D1002/0`00
0P000<@1[0?50JT3E`0002@000350@00Z0<00<D1002Y0`000P000<D1Z@?50JP3E`0002@000340@00
Y@<00<D1002X0`000P000<D1Z0?40JD3E`0002@000320@00X`<00<@1002U0`000P000<@1Y@?20J<3
E`0002@000300@00XP<00<81002S0`000P000<81X`?00J83E`0002@0003a0@00V`<00?41002P0`00
0P000?41V`?a0J03E`0002@0003a0@00W@<00?81002O0`000P000?41W`?b0Id3E`0002@0003b0@00
V`<00?@1002M0`000P000?81W@?d0I/3E`0002@0003d0@00V`<00?H1002K0`000P000?@1V`?f0I/3
E`0002@0003f0@00V`<00?X1002N0`000P000?H1V`?j0Ih3E`0002@0003j0@00WP<00?`1002N0`00
0P000?X1WP?l0Ih3E`0002@0003l0@00W@<00?d1002N0`000P000?`1WP?m0Id3E`0002@0003m0@00
V`<00?h1002M0`000P000?d1W@?n0I/3E`0002@0003b0@00W0<00?@1002M0`000P000?81W@?d0I`3
E`0002@0003d0@00W0<00?H1002L0`000P000?@1W0?f0I`3E`0002@0003f0@00W0<00?X1002N0`00
0P000?H1W0?j0Ih3E`0002@0003n0@00V`<00?h1002N0`000P000?h1V`?n0Ih3E`0002@0003m0@00
WP<00?h1002P0`000P000?h1WP?m0J03E`0002@0003i0@00X0<00?d1002U0`000P000?d1X0?i0JD3
E`0002@0003h0@00Y@<00?T1002W0`000P000?T1Y@?h0JL3E`0002@0003g0@00Y`<00?P1002Z0`00
0P000?P1Y`?g0JX3E`0002@0003g0@00ZP<00?L1002^0`000P000?L1ZP?g0Jh3E`0002@0003h0@00
X0<00?d1002U0`000P000?d1X0?h0JD3E`0002@0003g0@00Y@<00?P1002W0`000P000?P1Y@?g0JL3
E`0002@0003g0@00Y`<00?L1002Z0`000P000?L1Y`?g0JX3E`0002@0003g0@00ZP<00?L1002^0`00
0P000?L1ZP?g0Jh3E`0002@0000/0P00V`<002l2002L0`000P0002l2V`</0Y`3E`0002@0000[0P00
W0<002`2002N0`000P0002`2W0<[0Yh3E`0002@0000[0P00WP<002/2002P0`000P0002/2WP<[0Z03
E`0002@0000[0P00X0<002`2002R0`000P0002/2X0</0Z83E`0002@0000/0P00XP<002l2002S0`00
0P0002`2XP<_0Z<3E`0002@0000_0P00X`<00382002S0`000P0002l2X`<b0Z<3E`0002@0000b0P00
XP<003D2002S0`000P000382X`<e0Z83E`0002@0000e0P00X0<003H2002R0`000P0003D2XP<f0Z03
E`0002@0000f0P00WP<003H2002P0`000P0003H2X0<f0Yh3E`0002@0000e0P00W0<003H2002N0`00
0P0003H2WP<e0Y`3E`0002@0000b0P00V`<003D2002L0`000P0003D2W0<b0Y/3E`0002@0000_0P00
V`<00382002K0`000P000382V`<_0Y/3E`0002@0000]0P00V`<002l2002L0`000P0002l2V`<]0Y`3
E`0002@0000/0P00W0<002d2002N0`000P0002d2W0</0Yh3E`0002@0000/0P00WP<002`2002P0`00
0P0002`2WP</0Z03E`0002@0000/0P00X0<002d2002R0`000P0002`2X0<]0Z83E`0002@0000]0P00
XP<002l2002S0`000P0002d2XP<_0Z<3E`0002@0000b0P00XP<003@2002S0`000P000382X`<d0Z83
E`0002@0000d0P00X0<003D2002R0`000P0003@2XP<e0Z03E`0002@0000e0P00WP<003D2002P0`00
0P0003D2X0<e0Yh3E`0002@0000d0P00W0<003D2002N0`000P0003D2WP<d0Y`3E`0002@0000b0P00
V`<003@2002L0`000P0003@2W0<b0Y/3E`0002@0000/0P00X`<002l2002T0`000P0002l2X`</0Z@3
E`0002@0000[0P00Y0<002`2002U0`000P0002`2Y0<[0ZD3E`0002@0000Z0P00Y@<002/2002W0`00
0P0002/2Y@<Z0ZL3E`0002@0000Z0P00Y`<002X2002[0`000P0002X2Y`<Z0Z/3E`0002@0000Z0P00
Z`<002/2002]0`000P0002X2Z`<[0Zd3E`0002@0000[0P00[@<002`2002^0`000P0002/2[@</0Zh3
E`0002@0000/0P00[P<002l2002^0`000P0002`2[P<_0Zh3E`0002@0000_0P00[P<00382002^0`00
0P0002l2[P<b0Zh3E`0002@0000b0P00[P<003D2002^0`000P000382[P<e0Zh3E`0002@0000e0P00
[@<003H2002^0`000P0003D2[P<f0Zd3E`0002@0000f0P00Z`<003L2002]0`000P0003H2[@<g0Z/3
E`0002@0000g0P00Y`<003L2002[0`000P0003L2Z`<g0ZL3E`0002@0000f0P00Y@<003L2002W0`00
0P0003L2Y`<f0ZD3E`0002@0000e0P00Y0<003H2002U0`000P0003H2Y@<e0Z@3E`0002@0000b0P00
X`<003D2002T0`000P0003D2Y0<b0Z<3E`0002@0000]0P00X`<002l2002T0`000P0002l2X`<]0Z@3
E`0002@0000/0P00Y0<002d2002U0`000P0002d2Y0</0ZD3E`0002@0000[0P00Y@<002`2002W0`00
0P0002`2Y@<[0ZL3E`0002@0000[0P00Y`<002/2002[0`000P0002/2Y`<[0Z/3E`0002@0000[0P00
Z`<002`2002]0`000P0002/2Z`</0Zd3E`0002@0000/0P00[@<002d2002^0`000P0002`2[@<]0Zh3
E`0002@0000]0P00[P<002l2002^0`000P0002d2[P<_0Zh3E`0002@0000b0P00[P<003@2002^0`00
0P000382[P<d0Zh3E`0002@0000d0P00[@<003D2002^0`000P0003@2[P<e0Zd3E`0002@0000e0P00
Z`<003H2002]0`000P0003D2[@<f0Z/3E`0002@0000f0P00Y`<003H2002[0`000P0003H2Z`<f0ZL3
E`0002@0000e0P00Y@<003H2002W0`000P0003H2Y`<e0ZD3E`0002@0000d0P00Y0<003D2002U0`00
0P0003D2Y@<d0Z@3E`0002@0000b0P00X`<003@2002T0`000P0003@2Y0<b0Z<3E`0002@0001]0P00
X@<006h2002T0`000P0006h2X@=]0Z@3E`0002@0001[0P00Y0<006d2002V0`000P0006d2Y0=[0ZH3
E`0002@0001X0P00YP<006/2002W0`000P0006/2YP=X0ZL3E`0002@0001X0P00Y`<006P2002W0`00
0P0006P2Y`=X0ZL3E`0002@0001U0P00YP<006P2002W0`000P0006P2Y`=U0ZH3E`0002@0001S0P00
Y0<006D2002V0`000P0006D2YP=S0Z@3E`0002@0001R0P00X@<006<2002T0`000P0006<2Y0=R0Z43
E`0002@0001R0P00X0<00682002Q0`000P000682X@=R0Z03E`0002@0001R0P00WP<006<2002P0`00
0P000682X0=S0Yh3E`0002@0001S0P00W0<006D2002N0`000P0006<2WP=U0Y`3E`0002@0001U0P00
V`<006P2002L0`000P0006D2W0=X0Y/3E`0002@0001X0P00V`<006T2002K0`000P0006P2V`=Y0Y/3
E`0002@0001Y0P00V`<006`2002L0`000P0006T2V`=/0Y`3E`0002@0001/0P00W0<006h2002N0`00
0P0006`2W0=^0Yh3E`0002@0001^0P00WP<006l2002P0`000P0006h2WP=_0Z03E`0002@0001_0P00
X0<006l2002V0`000P0006l2X0=_0ZH3E`0002@0001^0P00YP<006l2002Z0`000P0006l2YP=^0ZX3
E`0002@0001]0P00ZP<006h2002/0`000P0006h2ZP=]0Z`3E`0002@0001[0P00[0<006d2002^0`00
0P0006d2[0=[0Zh3E`0002@0001X0P00[P<006/2002^0`000P0006/2[P=X0Zh3E`0002@0001V0P00
[P<006P2002^0`000P0006P2[P=V0Zh3E`0002@0001T0P00[P<006H2002^0`000P0006H2[P=T0Zh3
E`0002@0001S0P00[0<006@2002^0`000P0006@2[P=S0Z`3E`0002@0001S0P00Z`<006<2002/0`00
0P0006<2[0=S0Z/3E`0002@0001S0P00ZP<006@2002[0`000P0006<2Z`=T0ZX3E`0002@0001T0P00
ZP<006D2002[0`000P0006@2ZP=U0Z/3E`0002@0001T0P00Z`<006D2002/0`000P0006D2Z`=T0Z`3
E`0002@0001V0P00YP<006P2002W0`000P0006P2Y`=V0ZH3E`0002@0001T0P00Y0<006H2002V0`00
0P0006H2YP=T0Z@3E`0002@0001S0P00X@<006@2002T0`000P0006@2Y0=S0Z43E`0002@0001S0P00
X0<006<2002Q0`000P0006<2X@=S0Z03E`0002@0001S0P00WP<006@2002P0`000P0006<2X0=T0Yh3
E`0002@0001T0P00W0<006H2002N0`000P0006@2WP=V0Y`3E`0002@0001V0P00V`<006P2002L0`00
0P0006H2W0=X0Y/3E`0002@0001Y0P00V`<006/2002L0`000P0006T2V`=[0Y`3E`0002@0001[0P00
W0<006d2002N0`000P0006/2W0=]0Yh3E`0002@0001]0P00WP<006h2002P0`000P0006d2WP=^0Z03
E`0002@0001^0P00X0<006h2002V0`000P0006h2X0=^0ZH3E`0002@0001]0P00YP<006h2002Z0`00
0P0006h2YP=]0ZX3E`0002@0001/0P00ZP<006d2002/0`000P0006d2ZP=/0Z`3E`0002@0001Z0P00
[0<006`2002^0`000P0006`2[0=Z0Zh3E`0002@0001X0P00[P<006X2002^0`000P0006X2[P=X0Zh3
E`0002@0002E0P00WP<009L2002O0`000P0009D2W`>G0Yh3E`0002@0002G0P00V`<009T2002N0`00
0P0009L2WP>I0Y/3E`0002@0002I0P00V`<009T2002^0`000P0009T2V`>I0Zh3E`0002@0002H0P00
W0<009P2002^0`000P0009P2W0>H0Zh3E`0002@0002E0P00[P<009d2002^0`000P0009D2[P>M0Zh3
E`0002@0002W0P00V`<00:X2002L0`000P000:X2V`>W0Y`3E`0002@0002U0P00W0<00:L2002O0`00
0P000:L2W0>U0Yl3E`0002@0002T0P00W`<00:D2002S0`000P000:D2W`>T0Z<3E`0002@0002T0P00
X`<00:@2002V0`000P000:@2X`>T0ZH3E`0002@0002T0P00YP<00:D2002[0`000P000:@2YP>U0Z/3
E`0002@0002U0P00Z`<00:L2002^0`000P000:D2Z`>W0Zh3E`0002@0002W0P00[P<00:X2002^0`00
0P000:L2[P>Z0Zh3E`0002@0002Z0P00[P<00:/2002^0`000P000:X2[P>[0Zh3E`0002@0002[0P00
[P<00:h2002^0`000P000:/2[P>^0Zh3E`0002@0002^0P00Z`<00;02002^0`000P000:h2[P>`0Z/3
E`0002@0002`0P00YP<00;42002[0`000P000;02Z`>a0ZH3E`0002@0002a0P00X`<00;42002V0`00
0P000;42YP>a0Z<3E`0002@0002`0P00W`<00;42002S0`000P000;42X`>`0Yl3E`0002@0002^0P00
W0<00;02002O0`000P000;02W`>^0Y`3E`0002@0002[0P00V`<00:h2002L0`000P000:h2W0>[0Y/3
E`0002@0002Z0P00V`<00:/2002K0`000P000:/2V`>Z0Y/3E`0002@0002X0P00V`<00:X2002L0`00
0P000:X2V`>X0Y`3E`0002@0002W0P00W0<00:P2002M0`000P000:P2W0>W0Yd3E`0002@0002V0P00
W@<00:L2002O0`000P000:L2W@>V0Yl3E`0002@0002U0P00W`<00:H2002S0`000P000:H2W`>U0Z<3
E`0002@0002U0P00X`<00:D2002V0`000P000:D2X`>U0ZH3E`0002@0002U0P00YP<00:H2002[0`00
0P000:D2YP>V0Z/3E`0002@0002V0P00Z`<00:L2002]0`000P000:H2Z`>W0Zd3E`0002@0002W0P00
[@<00:P2002^0`000P000:L2[@>X0Zh3E`0002@0002X0P00[P<00:X2002^0`000P000:P2[P>Z0Zh3
E`0002@0002[0P00[P<00:d2002^0`000P000:/2[P>]0Zh3E`0002@0002]0P00[@<00:h2002^0`00
0P000:d2[P>^0Zd3E`0002@0002^0P00Z`<00:l2002]0`000P000:h2[@>_0Z/3E`0002@0002_0P00
YP<00;02002[0`000P000:l2Z`>`0ZH3E`0002@0002`0P00X`<00;02002V0`000P000;02YP>`0Z<3
E`0002@0002_0P00W`<00;02002S0`000P000;02X`>_0Yl3E`0002@0002^0P00W@<00:l2002O0`00
0P000:l2W`>^0Yd3E`0002@0002]0P00W0<00:h2002M0`000P000:h2W@>]0Y`3E`0002@0002[0P00
V`<00:d2002L0`000P000:d2W0>[0Y/3E`0002@0003=0P00WP<00<l2002O0`000P000<d2W`??0Yh3
E`0002@0003?0P00V`<00=42002N0`000P000<l2WP?A0Y/3E`0002@0003A0P00V`<00=42002^0`00
0P000=42V`?A0Zh3E`0002@0003@0P00W0<00=02002^0`000P000=02W0?@0Zh3E`0002@0003=0P00
[P<00=D2002^0`000P000<d2[P?E0Zh3E`0002@0003O0P00WP<00>42002O0`000P000=l2W`?Q0Yh3
E`0002@0003Q0P00V`<00><2002N0`000P000>42WP?S0Y/3E`0002@0003S0P00V`<00><2002^0`00
0P000><2V`?S0Zh3E`0002@0003R0P00W0<00>82002^0`000P000>82W0?R0Zh3E`0002@0003O0P00
[P<00>L2002^0`000P000=l2[P?W0Zh3E`0002@000050`00WP<000L3002O0`000P0000D3W`<70ih3
E`0002@000070`00V`<000T3002N0`000P0000L3WP<90i/3E`0002@000090`00V`<000T3002^0`00
0P0000T3V`<90jh3E`0002@000080`00W0<000P3002^0`000P0000P3W0<80jh3E`0002@000050`00
[P<000d3002^0`000P0000D3[P<=0jh3E`0002@0000E0`00W`<001H3002P0`000P0001D3W`<F0j03
E`0002@0000E0`00X0<001H3002P0`000P0001H3X0<E0j03E`0002@0000D0`00X0<001D3002P0`00
0P0001D3X0<D0j03E`0002@0000D0`00W`<001@3002P0`000P0001@3X0<D0il3E`0002@0000D0`00
W@<001D3002O0`000P0001@3W`<E0id3E`0002@0000E0`00W0<001H3002M0`000P0001D3W@<F0i`3
E`0002@0000F0`00V`<001T3002L0`000P0001H3W0<I0i/3E`0002@0000I0`00V`<001`3002K0`00
0P0001T3V`<L0i/3E`0002@0000L0`00V`<001l3002L0`000P0001`3V`<O0i`3E`0002@0000O0`00
W0<00203002M0`000P0001l3W0<P0id3E`0002@0000P0`00W@<00243002O0`000P000203W@<Q0il3
E`0002@0000Q0`00W`<00243002P0`000P000243W`<Q0j03E`0002@0000P0`00X0<00243002R0`00
0P000243X0<P0j83E`0002@0000M0`00XP<00203002T0`000P000203XP<M0j@3E`0002@0000I0`00
Y0<001d3002V0`000P0001d3Y0<I0jH3E`0002@0000G0`00YP<001T3002W0`000P0001T3YP<G0jL3
E`0002@0000E0`00Y`<001L3002Y0`000P0001L3Y`<E0jT3E`0002@0000D0`00Z@<001D3002/0`00
0P0001D3Z@<D0j`3E`0002@0000D0`00[0<001@3002^0`000P0001@3[0<D0jh3E`0002@0000L0`00
V`<001h3002L0`000P0001`3V`<N0i`3E`0002@0000N0`00W0<001l3002M0`000P0001h3W0<O0id3
E`0002@0000O0`00W@<00203002O0`000P0001l3W@<P0il3E`0002@0000P0`00W`<00203002P0`00
0P000203W`<P0j03E`0002@0000O0`00X0<00203002R0`000P000203X0<O0j83E`0002@0000L0`00
XP<001l3002T0`000P0001l3XP<L0j@3E`0002@0000I0`00Y0<001`3002V0`000P0001`3Y0<I0jH3
E`0002@0000D0`00[0<001D3002]0`000P0001@3[@<E0j`3E`0002@0000E0`00[0<001L3002/0`00
0P0001D3[0<G0j`3E`0002@0000G0`00[0<001/3002^0`000P0001L3[0<K0jh3E`0002@0000K0`00
[P<001h3002^0`000P0001/3[P<N0jh3E`0002@0000N0`00[@<00203002^0`000P0001h3[P<P0jd3
E`0002@0000P0`00[0<00243002]0`000P000203[@<Q0j`3E`0002@0000G0`00[0<001/3002^0`00
0P0001L3[0<K0jh3E`0002@0000K0`00[P<001l3002^0`000P0001/3[P<O0jh3E`0002@0000O0`00
[P<00203002^0`000P0001l3[P<P0jh3E`0002@0000P0`00[0<00243002^0`000P000203[P<Q0j`3
E`0002@0000Q0`00ZP<00243002/0`000P000243[0<Q0jX3E`0002@0000n0`00WP<00403002O0`00
0P0003h3W`=00ih3E`0002@000100`00V`<00483002N0`000P000403WP=20i/3E`0002@000120`00
V`<00483002^0`000P000483V`=20jh3E`0002@000110`00W0<00443002^0`000P000443W0=10jh3
E`0002@0000n0`00[P<004H3002^0`000P0003h3[P=60jh3E`0002@0001>0`00W`<004l3002P0`00
0P0004h3W`=?0j03E`0002@0001>0`00X0<004l3002P0`000P0004l3X0=>0j03E`0002@0001=0`00
X0<004h3002P0`000P0004h3X0==0j03E`0002@0001=0`00W`<004d3002P0`000P0004d3X0==0il3
E`0002@0001=0`00W@<004h3002O0`000P0004d3W`=>0id3E`0002@0001>0`00W0<004l3002M0`00
0P0004h3W@=?0i`3E`0002@0001?0`00V`<00583002L0`000P0004l3W0=B0i/3E`0002@0001B0`00
V`<005D3002K0`000P000583V`=E0i/3E`0002@0001E0`00V`<005P3002L0`000P0005D3V`=H0i`3
E`0002@0001H0`00W0<005T3002N0`000P0005P3W0=I0ih3E`0002@0001I0`00WP<005T3002P0`00
0P0005T3WP=I0j03E`0002@0001H0`00X0<005T3002R0`000P0005T3X0=H0j83E`0002@0001E0`00
XP<005P3002S0`000P0005P3XP=E0j<3E`0002@0001C0`00X`<005D3002S0`000P0005D3X`=C0j<3
E`0002@0001E0`00V`<005L3002L0`000P0005D3V`=G0i`3E`0002@0001G0`00W0<005P3002N0`00
0P0005L3W0=H0ih3E`0002@0001H0`00WP<005P3002P0`000P0005P3WP=H0j03E`0002@0001G0`00
X0<005P3002R0`000P0005P3X0=G0j83E`0002@0001E0`00XP<005L3002S0`000P0005L3XP=E0j<3
E`0002@0001E0`00X`<005L3002T0`000P0005D3X`=G0j@3E`0002@0001G0`00Y0<005T3002V0`00
0P0005L3Y0=I0jH3E`0002@0001I0`00YP<005X3002X0`000P0005T3YP=J0jP3E`0002@0001J0`00
Z0<005X3002[0`000P0005X3Z0=J0j/3E`0002@0001I0`00Z`<005X3002]0`000P0005X3Z`=I0jd3
E`0002@0001H0`00[@<005T3002^0`000P0005T3[@=H0jh3E`0002@0001E0`00[P<005P3002^0`00
0P0005P3[P=E0jh3E`0002@0001B0`00[P<005D3002^0`000P0005D3[P=B0jh3E`0002@0001?0`00
[P<00583002^0`000P000583[P=?0jh3E`0002@0001>0`00[@<004l3002^0`000P0004l3[P=>0jd3
E`0002@0001=0`00Z`<004h3002]0`000P0004h3[@==0j/3E`0002@0001=0`00ZP<004d3002[0`00
0P0004d3Z`==0jX3E`0002@0001=0`00Z@<004h3002Z0`000P0004d3ZP=>0jT3E`0002@0001>0`00
Z@<004l3002Z0`000P0004h3Z@=?0jX3E`0002@0001>0`00ZP<004l3002[0`000P0004l3ZP=>0j/3
E`0002@0001H0`00Y@<005T3002X0`000P0005P3Y@=I0jP3E`0002@0001I0`00Z0<005T3002[0`00
0P0005T3Z0=I0j/3E`0002@0001H0`00Z`<005T3002]0`000P0005T3Z`=H0jd3E`0002@0001G0`00
[@<005P3002^0`000P0005P3[@=G0jh3E`0002@0001E0`00[P<005L3002^0`000P0005L3[P=E0jh3
E`0002@0001f0`00WP<007P3002O0`000P0007H3W`=h0ih3E`0002@0001h0`00V`<007X3002N0`00
0P0007P3WP=j0i/3E`0002@0001j0`00V`<007X3002^0`000P0007X3V`=j0jh3E`0002@0001i0`00
W0<007T3002^0`000P0007T3W0=i0jh3E`0002@0001f0`00[P<007h3002^0`000P0007H3[P=n0jh3
E`0002@0002=0`00W@<008d3002^0`000P0008d3W@>=0jh3E`0002@0002>0`00V`<008h3002^0`00
0P0008h3V`>>0jh3E`0002@000240`00V`<008h3002Y0`000P0008h3V`>40jT3E`0002@000240`00
Z@<009<3002Y0`000P0008@3Z@>C0jT3E`0002@0002;0`00[P<00943002^0`000P0008/3[P>A0jh3
E`0002@0002_0`00WP<00;43002O0`000P000:l3W`>a0ih3E`0002@0002a0`00V`<00;<3002N0`00
0P000;43WP>c0i/3E`0002@0002c0`00V`<00;<3002^0`000P000;<3V`>c0jh3E`0002@0002b0`00
W0<00;83002^0`000P000;83W0>b0jh3E`0002@0002_0`00[P<00;L3002^0`000P000:l3[P>g0jh3
E`0002@0002m0`00V`<00;l3002T0`000P000;l3V`>m0j@3E`0002@0002m0`00XP<00;l3002T0`00
0P000;d3Y0>o0j83E`0002@0002o0`00X@<00<83002R0`000P000;l3XP?20j43E`0002@000320`00
X@<00<@3002Q0`000P000<83X@?40j43E`0002@000340`00X@<00<L3002R0`000P000<@3X@?70j83
E`0002@000370`00XP<00<T3002T0`000P000<L3XP?90j@3E`0002@000390`00Y0<00<X3002W0`00
0P000<T3Y0?:0jL3E`0002@0003:0`00Y`<00<X3002Y0`000P000<X3Y`?:0jT3E`0002@000390`00
Z@<00<X3002/0`000P000<X3Z@?90j`3E`0002@000370`00[0<00<T3002^0`000P000<T3[0?70jh3
E`0002@000340`00[P<00<L3002^0`000P000<L3[P?40jh3E`0002@000320`00[P<00<@3002^0`00
0P000<@3[P?20jh3E`0002@0002o0`00[P<00<83002^0`000P000<83[P>o0jh3E`0002@0002n0`00
[@<00;l3002^0`000P000;l3[P>n0jd3E`0002@0002m0`00Z`<00;h3002]0`000P000;h3[@>m0j/3
E`0002@0002m0`00ZP<00;d3002[0`000P000;d3Z`>m0jX3E`0002@0002m0`00Z@<00;h3002Z0`00
0P000;d3ZP>n0jT3E`0002@0002n0`00Z@<00;l3002Z0`000P000;h3Z@>o0jX3E`0002@0002n0`00
ZP<00;l3002[0`000P000;l3ZP>n0j/3E`0002@000340`00X@<00<H3002R0`000P000<@3X@?60j83
E`0002@000360`00XP<00<P3002T0`000P000<H3XP?80j@3E`0002@000380`00Y0<00<T3002W0`00
0P000<P3Y0?90jL3E`0002@000390`00Y`<00<T3002Y0`000P000<T3Y`?90jT3E`0002@000380`00
Z@<00<T3002/0`000P000<T3Z@?80j`3E`0002@000360`00[0<00<P3002^0`000P000<P3[0?60jh3
E`0002@000340`00[P<00<H3002^0`000P000<H3[P?40jh3E`0002@0002o0`00V`<00<P3002K0`00
0P000;l3V`?80i/3E`0002@0002o0`00W0<00<<3002L0`000P000;l3W0?30i`3E`0002@000330`00
V`<00<P3002L0`000P000<<3W0?80i/3E`0002@0003W0`00WP<00>T3002O0`000P000>L3W`?Y0ih3
E`0002@0003Y0`00V`<00>/3002N0`000P000>T3WP?[0i/3E`0002@0003[0`00V`<00>/3002^0`00
0P000>/3V`?[0jh3E`0002@0003Z0`00W0<00>X3002^0`000P000>X3W0?Z0jh3E`0002@0003W0`00
[P<00>l3002^0`000P000>L3[P?_0jh3E`0002@000001000WP<00044002O0`000P000044WP<019l3
E`0002@000001000W`<00044002P0`000P000004W`<11:03E`0002@000011000W`<00084002P0`00
0P000044X0<219l3E`0002@000021000WP<00084002O0`000P000084W`<219h3E`0002@000011000
W0<00084002N0`000P000084WP<119`3E`0002@0003o0`00V`<00044002L0`000P000044W0?o0i/3
E`0002@0003l0`00V`<00?l3002K0`000P000?l3V`?l0i/3E`0002@0003j0`00V`<00?`3002L0`00
0P000?`3V`?j0i`3E`0002@0003h0`00W0<00?X3002N0`000P000?X3W0?h0ih3E`0002@0003g0`00
WP<00?P3002P0`000P000?P3WP?g0j03E`0002@0003f0`00X0<00?L3002S0`000P000?L3X0?f0j<3
E`0002@0003f0`00X`<00?H3002Y0`000P000?H3X`?f0jT3E`0002@0003f0`00Z@<00?L3002/0`00
0P000?H3Z@?g0j`3E`0002@0003g0`00[0<00?T3002^0`000P000?L3[0?i0jh3E`0002@0003i0`00
[P<00?`3002^0`000P000?T3[P?l0jh3E`0002@0003l0`00[P<00?d3002^0`000P000?`3[P?m0jh3
E`0002@0003m0`00[P<00004002^0`000P000?d3[P<01:h3E`0002@000001000[0<00084002^0`00
0P000004[P<21:`3E`0002@000021000Z@<000<4002/0`000P000084[0<31:T3E`0002@000031000
Z0<000<4002Y0`000P0000<4Z@<31:P3E`0002@000021000Y@<000<4002X0`000P0000<4Z0<21:D3
E`0002@000001000X`<00084002U0`000P000084Y@<01:<3E`0002@0003m0`00XP<00004002S0`00
0P000004X`?m0j83E`0002@0003l0`00XP<00?d3002R0`000P000?d3XP?l0j83E`0002@0003j0`00
XP<00?`3002S0`000P000?`3XP?j0j<3E`0002@0003h0`00X`<00?X3002U0`000P000?X3X`?h0jD3
E`0002@0003g0`00Y@<00?P3002X0`000P000?P3Y@?g0jP3E`0002@0003k0`00V`<00?`3002L0`00
0P000?`3V`?k0i`3E`0002@0003i0`00W0<00?/3002N0`000P000?/3W0?i0ih3E`0002@0003h0`00
WP<00?T3002P0`000P000?T3WP?h0j03E`0002@0003g0`00X0<00?P3002S0`000P000?P3X0?g0j<3
E`0002@0003g0`00X`<00?L3002Y0`000P000?L3X`?g0jT3E`0002@0003g0`00Z@<00?P3002/0`00
0P000?L3Z@?h0j`3E`0002@0003h0`00[0<00?X3002^0`000P000?P3[0?j0jh3E`0002@0003j0`00
[P<00?`3002^0`000P000?X3[P?l0jh3E`0002@0003m0`00[P<00?l3002^0`000P000?d3[P?o0jh3
E`0002@0003o0`00[0<00044002^0`000P000?l3[P<11:`3E`0002@000011000Z@<00084002/0`00
0P000044[0<21:T3E`0002@000021000Z0<00084002Y0`000P000084Z@<21:P3E`0002@000011000
Y@<00084002X0`000P000084Z0<11:D3E`0002@0003o0`00X`<00044002U0`000P000044Y@?o0j<3
E`0002@0003m0`00XP<00?l3002S0`000P000?l3X`?m0j83E`0002@0000O1000WP<00244002O0`00
0P0001l4W`<Q19h3E`0002@0000Q1000V`<002<4002N0`000P000244WP<S19/3E`0002@0000S1000
V`<002<4002^0`000P0002<4V`<S1:h3E`0002@0000R1000W0<00284002^0`000P000284W0<R1:h3
E`0002@0000O1000[P<002L4002^0`000P0001l4[P<W1:h3E`0002@0000^1000V`<002h4002P0`00
0P0002h4V`<^1:03E`0002@0000^1000W@<002l4002O0`000P0002h4W`<_19d3E`0002@0000_1000
V`<00344002M0`000P0002l4W@<a19/3E`0002@0000a1000V`<003<4002K0`000P000344V`<c19/3
E`0002@0000c1000V`<003L4002N0`000P0003<4V`<g19h3E`0002@0000g1000WP<003T4002N0`00
0P0003L4WP<i19h3E`0002@0000i1000W@<003X4002N0`000P0003T4WP<j19d3E`0002@0000j1000
V`<003/4002M0`000P0003X4W@<k19/3E`0002@0000_1000W0<00344002M0`000P0002l4W@<a19`3
E`0002@0000a1000W0<003<4002L0`000P000344W0<c19`3E`0002@0000c1000W0<003L4002N0`00
0P0003<4W0<g19h3E`0002@0000k1000V`<003/4002N0`000P0003/4V`<k19h3E`0002@0000j1000
WP<003/4002P0`000P0003/4WP<j1:03E`0002@0000f1000X0<003X4002U0`000P0003X4X0<f1:D3
E`0002@0000e1000Y@<003H4002W0`000P0003H4Y@<e1:L3E`0002@0000d1000Y`<003D4002Z0`00
0P0003D4Y`<d1:X3E`0002@0000d1000ZP<003@4002^0`000P0003@4ZP<d1:h3E`0002@0000e1000
X0<003X4002U0`000P0003X4X0<e1:D3E`0002@0000d1000Y@<003D4002W0`000P0003D4Y@<d1:L3
E`0002@0000d1000Y`<003@4002Z0`000P0003@4Y`<d1:X3E`0002@0000d1000ZP<003@4002^0`00
0P0003@4ZP<d1:h3E`0002@0001H1000WP<005X4002O0`000P0005P4W`=J19h3E`0002@0001J1000
V`<005`4002N0`000P0005X4WP=L19/3E`0002@0001L1000V`<005`4002^0`000P0005`4V`=L1:h3
E`0002@0001K1000W0<005/4002^0`000P0005/4W0=K1:h3E`0002@0001H1000[P<00604002^0`00
0P0005P4[P=P1:h3E`0002@0001Y1000V`<006`4002L0`000P0006`4V`=Y19`3E`0002@0001X1000
W0<006T4002N0`000P0006T4W0=X19h3E`0002@0001X1000WP<006P4002P0`000P0006P4WP=X1:03
E`0002@0001X1000X0<006T4002R0`000P0006P4X0=Y1:83E`0002@0001Y1000XP<006`4002S0`00
0P0006T4XP=/1:<3E`0002@0001/1000X`<006l4002S0`000P0006`4X`=_1:<3E`0002@0001_1000
XP<00784002S0`000P0006l4X`=b1:83E`0002@0001b1000X0<007<4002R0`000P000784XP=c1:03
E`0002@0001c1000WP<007<4002P0`000P0007<4X0=c19h3E`0002@0001b1000W0<007<4002N0`00
0P0007<4WP=b19`3E`0002@0001_1000V`<00784002L0`000P000784W0=_19/3E`0002@0001/1000
V`<006l4002K0`000P0006l4V`=/19/3E`0002@0001Z1000V`<006`4002L0`000P0006`4V`=Z19`3
E`0002@0001Y1000W0<006X4002N0`000P0006X4W0=Y19h3E`0002@0001Y1000WP<006T4002P0`00
0P0006T4WP=Y1:03E`0002@0001Y1000X0<006X4002R0`000P0006T4X0=Z1:83E`0002@0001Z1000
XP<006`4002S0`000P0006X4XP=/1:<3E`0002@0001_1000XP<00744002S0`000P0006l4X`=a1:83
E`0002@0001a1000X0<00784002R0`000P000744XP=b1:03E`0002@0001b1000WP<00784002P0`00
0P000784X0=b19h3E`0002@0001a1000W0<00784002N0`000P000784WP=a19`3E`0002@0001_1000
V`<00744002L0`000P000744W0=_19/3E`0002@0001Y1000X`<006`4002T0`000P0006`4X`=Y1:@3
E`0002@0001X1000Y0<006T4002U0`000P0006T4Y0=X1:D3E`0002@0001W1000Y@<006P4002W0`00
0P0006P4Y@=W1:L3E`0002@0001W1000Y`<006L4002[0`000P0006L4Y`=W1:/3E`0002@0001W1000
Z`<006P4002]0`000P0006L4Z`=X1:d3E`0002@0001X1000[@<006T4002^0`000P0006P4[@=Y1:h3
E`0002@0001Y1000[P<006`4002^0`000P0006T4[P=/1:h3E`0002@0001/1000[P<006l4002^0`00
0P0006`4[P=_1:h3E`0002@0001_1000[P<00784002^0`000P0006l4[P=b1:h3E`0002@0001b1000
[@<007<4002^0`000P000784[P=c1:d3E`0002@0001c1000Z`<007@4002]0`000P0007<4[@=d1:/3
E`0002@0001d1000Y`<007@4002[0`000P0007@4Z`=d1:L3E`0002@0001c1000Y@<007@4002W0`00
0P0007@4Y`=c1:D3E`0002@0001b1000Y0<007<4002U0`000P0007<4Y@=b1:@3E`0002@0001_1000
X`<00784002T0`000P000784Y0=_1:<3E`0002@0001Z1000X`<006`4002T0`000P0006`4X`=Z1:@3
E`0002@0001Y1000Y0<006X4002U0`000P0006X4Y0=Y1:D3E`0002@0001X1000Y@<006T4002W0`00
0P0006T4Y@=X1:L3E`0002@0001X1000Y`<006P4002[0`000P0006P4Y`=X1:/3E`0002@0001X1000
Z`<006T4002]0`000P0006P4Z`=Y1:d3E`0002@0001Y1000[@<006X4002^0`000P0006T4[@=Z1:h3
E`0002@0001Z1000[P<006`4002^0`000P0006X4[P=/1:h3E`0002@0001_1000[P<00744002^0`00
0P0006l4[P=a1:h3E`0002@0001a1000[@<00784002^0`000P000744[P=b1:d3E`0002@0001b1000
Z`<007<4002]0`000P000784[@=c1:/3E`0002@0001c1000Y`<007<4002[0`000P0007<4Z`=c1:L3
E`0002@0001b1000Y@<007<4002W0`000P0007<4Y`=b1:D3E`0002@0001a1000Y0<00784002U0`00
0P000784Y@=a1:@3E`0002@0001_1000X`<00744002T0`000P000744Y0=_1:<3E`0002@0002@1000
WP<00984002O0`000P000904W`>B19h3E`0002@0002B1000V`<009@4002N0`000P000984WP>D19/3
E`0002@0002D1000V`<009@4002^0`000P0009@4V`>D1:h3E`0002@0002C1000W0<009<4002^0`00
0P0009<4W0>C1:h3E`0002@0002@1000[P<009P4002^0`000P000904[P>H1:h3E`0002@0002Z1000
X@<00:/4002T0`000P000:/4X@>Z1:@3E`0002@0002X1000Y0<00:X4002V0`000P000:X4Y0>X1:H3
E`0002@0002U1000YP<00:P4002W0`000P000:P4YP>U1:L3E`0002@0002U1000Y`<00:D4002W0`00
0P000:D4Y`>U1:L3E`0002@0002R1000YP<00:D4002W0`000P000:D4Y`>R1:H3E`0002@0002P1000
Y0<00:84002V0`000P000:84YP>P1:@3E`0002@0002O1000X@<00:04002T0`000P000:04Y0>O1:43
E`0002@0002O1000X0<009l4002Q0`000P0009l4X@>O1:03E`0002@0002O1000WP<00:04002P0`00
0P0009l4X0>P19h3E`0002@0002P1000W0<00:84002N0`000P000:04WP>R19`3E`0002@0002R1000
V`<00:D4002L0`000P000:84W0>U19/3E`0002@0002U1000V`<00:H4002K0`000P000:D4V`>V19/3
E`0002@0002V1000V`<00:T4002L0`000P000:H4V`>Y19`3E`0002@0002Y1000W0<00:/4002N0`00
0P000:T4W0>[19h3E`0002@0002[1000WP<00:`4002P0`000P000:/4WP>/1:03E`0002@0002/1000
X0<00:`4002V0`000P000:`4X0>/1:H3E`0002@0002[1000YP<00:`4002Z0`000P000:`4YP>[1:X3
E`0002@0002Z1000ZP<00:/4002/0`000P000:/4ZP>Z1:`3E`0002@0002X1000[0<00:X4002^0`00
0P000:X4[0>X1:h3E`0002@0002U1000[P<00:P4002^0`000P000:P4[P>U1:h3E`0002@0002S1000
[P<00:D4002^0`000P000:D4[P>S1:h3E`0002@0002Q1000[P<00:<4002^0`000P000:<4[P>Q1:h3
E`0002@0002P1000[0<00:44002^0`000P000:44[P>P1:`3E`0002@0002P1000Z`<00:04002/0`00
0P000:04[0>P1:/3E`0002@0002P1000ZP<00:44002[0`000P000:04Z`>Q1:X3E`0002@0002Q1000
ZP<00:84002[0`000P000:44ZP>R1:/3E`0002@0002Q1000Z`<00:84002/0`000P000:84Z`>Q1:`3
E`0002@0002S1000YP<00:D4002W0`000P000:D4Y`>S1:H3E`0002@0002Q1000Y0<00:<4002V0`00
0P000:<4YP>Q1:@3E`0002@0002P1000X@<00:44002T0`000P000:44Y0>P1:43E`0002@0002P1000
X0<00:04002Q0`000P000:04X@>P1:03E`0002@0002P1000WP<00:44002P0`000P000:04X0>Q19h3
E`0002@0002Q1000W0<00:<4002N0`000P000:44WP>S19`3E`0002@0002S1000V`<00:D4002L0`00
0P000:<4W0>U19/3E`0002@0002V1000V`<00:P4002L0`000P000:H4V`>X19`3E`0002@0002X1000
W0<00:X4002N0`000P000:P4W0>Z19h3E`0002@0002Z1000WP<00:/4002P0`000P000:X4WP>[1:03
E`0002@0002[1000X0<00:/4002V0`000P000:/4X0>[1:H3E`0002@0002Z1000YP<00:/4002Z0`00
0P000:/4YP>Z1:X3E`0002@0002Y1000ZP<00:X4002/0`000P000:X4ZP>Y1:`3E`0002@0002W1000
[0<00:T4002^0`000P000:T4[0>W1:h3E`0002@0002U1000[P<00:L4002^0`000P000:L4[P>U1:h3
E`0002@000361000W`<00<L4002P0`000P000<H4W`?71:03E`0002@000361000X0<00<L4002P0`00
0P000<L4X0?61:03E`0002@000351000X0<00<H4002P0`000P000<H4X0?51:03E`0002@000351000
W`<00<D4002P0`000P000<D4X0?519l3E`0002@000351000W@<00<H4002O0`000P000<D4W`?619d3
E`0002@000361000W0<00<L4002M0`000P000<H4W@?719`3E`0002@000371000V`<00<X4002L0`00
0P000<L4W0?:19/3E`0002@0003:1000V`<00<d4002K0`000P000<X4V`?=19/3E`0002@0003=1000
V`<00=04002L0`000P000<d4V`?@19`3E`0002@0003@1000W0<00=44002M0`000P000=04W0?A19d3
E`0002@0003A1000W@<00=84002O0`000P000=44W@?B19l3E`0002@0003B1000W`<00=84002P0`00
0P000=84W`?B1:03E`0002@0003A1000X0<00=84002R0`000P000=84X0?A1:83E`0002@0003>1000
XP<00=44002T0`000P000=44XP?>1:@3E`0002@0003:1000Y0<00<h4002V0`000P000<h4Y0?:1:H3
E`0002@000381000YP<00<X4002W0`000P000<X4YP?81:L3E`0002@000361000Y`<00<P4002Y0`00
0P000<P4Y`?61:T3E`0002@000351000Z@<00<H4002/0`000P000<H4Z@?51:`3E`0002@000351000
[0<00<D4002^0`000P000<D4[0?51:h3E`0002@0003=1000V`<00<l4002L0`000P000<d4V`??19`3
E`0002@0003?1000W0<00=04002M0`000P000<l4W0?@19d3E`0002@0003@1000W@<00=44002O0`00
0P000=04W@?A19l3E`0002@0003A1000W`<00=44002P0`000P000=44W`?A1:03E`0002@0003@1000
X0<00=44002R0`000P000=44X0?@1:83E`0002@0003=1000XP<00=04002T0`000P000=04XP?=1:@3
E`0002@0003:1000Y0<00<d4002V0`000P000<d4Y0?:1:H3E`0002@000351000[0<00<H4002]0`00
0P000<D4[@?61:`3E`0002@000361000[0<00<P4002/0`000P000<H4[0?81:`3E`0002@000381000
[0<00<`4002^0`000P000<P4[0?<1:h3E`0002@0003<1000[P<00<l4002^0`000P000<`4[P??1:h3
E`0002@0003?1000[@<00=44002^0`000P000<l4[P?A1:d3E`0002@0003A1000[0<00=84002]0`00
0P000=44[@?B1:`3E`0002@000381000[0<00<`4002^0`000P000<P4[0?<1:h3E`0002@0003<1000
[P<00=04002^0`000P000<`4[P?@1:h3E`0002@0003@1000[P<00=44002^0`000P000=04[P?A1:h3
E`0002@0003A1000[0<00=84002^0`000P000=44[P?B1:`3E`0002@0003B1000ZP<00=84002/0`00
0P000=84[0?B1:X3E`0002@0003J1000V`<00=d4002L0`000P000=d4V`?J19`3E`0002@0003H1000
W0<00=X4002O0`000P000=X4W0?H19l3E`0002@0003G1000W`<00=P4002S0`000P000=P4W`?G1:<3
E`0002@0003G1000X`<00=L4002V0`000P000=L4X`?G1:H3E`0002@0003G1000YP<00=P4002[0`00
0P000=L4YP?H1:/3E`0002@0003H1000Z`<00=X4002^0`000P000=P4Z`?J1:h3E`0002@0003J1000
[P<00=d4002^0`000P000=X4[P?M1:h3E`0002@0003M1000[P<00=h4002^0`000P000=d4[P?N1:h3
E`0002@0003N1000[P<00>44002^0`000P000=h4[P?Q1:h3E`0002@0003Q1000Z`<00><4002^0`00
0P000>44[P?S1:/3E`0002@0003S1000YP<00>@4002[0`000P000><4Z`?T1:H3E`0002@0003T1000
X`<00>@4002V0`000P000>@4YP?T1:<3E`0002@0003S1000W`<00>@4002S0`000P000>@4X`?S19l3
E`0002@0003Q1000W0<00><4002O0`000P000><4W`?Q19`3E`0002@0003N1000V`<00>44002L0`00
0P000>44W0?N19/3E`0002@0003M1000V`<00=h4002K0`000P000=h4V`?M19/3E`0002@0003K1000
V`<00=d4002L0`000P000=d4V`?K19`3E`0002@0003J1000W0<00=/4002M0`000P000=/4W0?J19d3
E`0002@0003I1000W@<00=X4002O0`000P000=X4W@?I19l3E`0002@0003H1000W`<00=T4002S0`00
0P000=T4W`?H1:<3E`0002@0003H1000X`<00=P4002V0`000P000=P4X`?H1:H3E`0002@0003H1000
YP<00=T4002[0`000P000=P4YP?I1:/3E`0002@0003I1000Z`<00=X4002]0`000P000=T4Z`?J1:d3
E`0002@0003J1000[@<00=/4002^0`000P000=X4[@?K1:h3E`0002@0003K1000[P<00=d4002^0`00
0P000=/4[P?M1:h3E`0002@0003N1000[P<00>04002^0`000P000=h4[P?P1:h3E`0002@0003P1000
[@<00>44002^0`000P000>04[P?Q1:d3E`0002@0003Q1000Z`<00>84002]0`000P000>44[@?R1:/3
E`0002@0003R1000YP<00><4002[0`000P000>84Z`?S1:H3E`0002@0003S1000X`<00><4002V0`00
0P000><4YP?S1:<3E`0002@0003R1000W`<00><4002S0`000P000><4X`?R19l3E`0002@0003Q1000
W@<00>84002O0`000P000>84W`?Q19d3E`0002@0003P1000W0<00>44002M0`000P000>44W@?P19`3
E`0002@0003N1000V`<00>04002L0`000P000>04W0?N19/39`0001P0000300000000000000000000
9@0000`000030000:00000`000010000F`000800002R0000j0000:X0003a00000@0001L0000G0000
ZP3/0:X0j`2Y0>X0Z@3Y0:L0j@2W0>P0Y@3X0:@0j@2S0>T0X`3Z0:80jP2R0>l0X`3_0:<0l02T0?00
Y03a0:L0l@2X0?00Z@3`0:T0k`2Z0>h0ZP3]0:X0k01K0000O0000=X0002S0000hP000:`000010000
5P0001H0003R0:L0hP2U0>40Y@3Q0:@0h02T0=l0X`3M0:<0g@2T0=`0Y03K0:D0fP2V0=X0Z@3K0:X0
g02[0=d0[03P0:`0h02[0>40Z`3Q0:X0hP2Z0>80Z@3R0:L0F`000700000C0@00hP0001/1003[0000
0@0001<0000C00006`7V01/1i@0J0ND06P7S01T1h`0H0N805@7R01@1h`0C0N<04`7Y01@1jP0F0NX0
5`7[01P1jP0I0NX06P7Y01X1j00K0NL06`7V05/0001P0000B`400>d0001C0@00m@000040000?0000
3`0005<1l@1C0Nl0DP7^0541k@1=0Nd0C07^04/1k`1;0O<0C07d04d1m@1A0OD0DP7d05<1l`1C0O80
D`7a05/0001`0000P`4001h1002;0@009P400040000C00004`0008/18P6;0Al1RP4O0HX17P650Ah1
Q@4O0H@17`640B01P`4Q0H<18`640B@1Q04U0HD19@650BH1RP4V0HX19@6;0BD1R`4S0H/18P5K0000
L0000;`1001e0@00a04007d1000100004`0001<000340GT1a05h0L<1M`730GH1`P5f0L41M@6n0GD1
_P5f0Kd1MP6l0GL1_05k0Kd1O06m0Gd1`P5m0L<1O0730G/1a05k0L@1NP740GT1F`000800003d0@00
bP400?`1003C0@000@0001L0000G0000o07>0O`1c07k0L`1n`7;0OX1b`7i0LX1m`7:0OL1b`7e0L/1
m@7<0O@1c@7d0M01m@7A0OD1dP7f0M81m`7C0OX1d`7j0M81n`7B0O/1d@7l0M41o07@0O`1cP5K0000
N00002`2002A0P00=@8009X2000100005@0001D0000e0YD2=0:D0S@2T`8c0Y82<@:B0S42T@8`0Y82
;P:B0Rd2T`8]0YD2;0:E0Rd2U`8]0YT2;P:I0Rl2VP8b0YX2<`:I0S@2V@8d0YL2=@:G0SD2U@9K0000
J00006D2001Q0P00K@8006T2000100004@000140001]0VD2K@9S0V`2H`9/0V82J`9Q0VL2H@9V0V82
I@9S0VD2I`9V0VP2I`9Y0V/2J@9/0VP2K09W0Vd2I`9]0VH2K@9U0U/0001l0000W@8001L2002U0P00
80800040000F00005P000:D26`:U0QT2Y08I0Z@260:S0QP2X`8G0Z025`:O0QP2WP8I0Yd26P:M0Qd2
WP8N0Yl27`:P0Ql2X08P0Z8280:S0Ql2Y08O0Z@27P:U0Qh2Y@8L0ZD26`9K0000N0000=H200170P00
gP800502000100005@0001D0003N0T/2gP9:0]d2BP;M0TT2g0980]/2A`;H0TL2f0980]L2B0;F0TT2
eP9>0]L2C`;H0Tl2f@9@0]/2D0;K0Tl2g09?0]d2CP;M0Td2gP9<0]h2B`9K0000P00000h3000B0P00
5P<001/2000100005`0001L0000F0aH25P<D0QD3508E0a<250<C0Q@34P8@0a8240<C0Pl34`8?0a@2
3P<D0Ph36@8?0aT23`<J0Q036P8A0a/24`<K0Q@36P8E0aX25@<I0QH36@8F0aL25P<F0U/0001`0000
AP<000@2001>0`0030800040000C00004`0004h3209>0`D2C@<50Td310980`@2B0<50TL31@970`H2
AP<70TH32@970`T2A`<;0TP32`980``2C@<<0Td32`9>0`/2CP<90Th3209K0000N00007l3003`0@00
Q`<00?T1000100005@0001D000270o@1Q`?c0HH3lP650o41Q0?a0H@3l0620o01P@?a0H03l@600o81
O`?b0Gl3m`600oL1P0?h0H43n0610oT1Q0?i0HD3n0660oL1Q`?f0HL3m05K0000J0000;L3000R0P00
_`<002X2000100004@000140002o0bH2_`<T0[h38`:m0b82^@<R0[P38`:h0b@2]`<T0[L39`:h0bP2
^0<Y0[T3:P:m0bX2_P<Y0[l3:0:o0bL2_`<V0U/0001T0000l0<00?/1003h0`000`800040000@0000
40000?P3o`7h0oh1m`?m0OL3o07f0o/1l@?k0O03o07`0`82l@<20_830`;e0`<2mP<20_L30P;g0`02
n0<00_P3o`5K0000J00002P4002U0P00<0@00:d2000100004@000140000`1:T2<0BX0Rl4Y`8_1:H2
;PBU0RX4Y@8Y1:H2:0BW0RP4Z`8Y1:`2:PB]0Rh4[@8_1:`2;`B[0S04Z`8`1:X2<0BY0U/0001l0000
H0@001P3001X10008@<00040000F00005P0006P470=X11X3I`@J0fL46@=V11T3IP@H0f<460=R11T3
H@@J0f046`=P11h3H@@O0f8480=S1203H`@Q0fD48@=V1203I`@P0fL47`=X11l3J0@M0fP470=K0000
L00009T4000o0`00X@@004L3000100004`0001<0002Q14<3X@A10j04@@>P1403W`A00il4?`>K13l3
VPA00iT4@@>I14D3VPA60i/4A`>O14L3W`A60j04AP>P14D3X@A50j44A0>Q14<3F`0006`0003A1000
H@<00=T4001Y0`000@000180000B0000f@AT0mT4H`?H1683f0AQ0m84H@?B16<3d@AS0m44IP?B16L3
dPAX0m<4J0?D16T3e`AY0mL4J0?H16P3f@AW0mT4IP?I16@3E`000700002V0000Y`000=D4001U0`00
5@000:H0k03N0:L05`7V04l1l@280B81`05i0OP1cP4a0YD2J@9U0Z826`;J0T/24P<F0T/320:30o@1
^`<V0_@3o`4/1:T2I@@L0id4@`?E16@3e@AU0eL0000d0000X@0004P0003J1000Q@<000H0002Q08@3
X@180=X4B03J18@3X@240j40Q@<U0000300000D0080X0000300000<0000U0000300000P0080X0000
30000080000U0000300000d0080>000050000000000@00005000
\>"], "Graphics",
 ImageSize->{253.75, 200.938},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}}],

Cell["And here are the actual and estimated trade directions.", "Text"],

Cell[GraphicsData["Metafile", "\<\
CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@000ALP0@0006`00000000000000?D4003/0`00
000000000037FP00idL00215CDH0004081L109H7000300000000000000000000@0H00;04000U0@00
g000000000000000000008Qh101PF`<02@000100003f1000k@<000X0000@000000000000000D0000
300000d0000U0000300000P0080W00006000004000000000oooo0000000U000030000040000[0000
6000000000000000mP@00>d3000[00006000000000000000mP@00>d3000V00007000008000000000
0@000000000000009@0000`000020000E`0002@0003d00002P000?X0000M00000P000?X02P3d01d0
E`0002@0003j00002P000041000M00000P000?X02P010Ad0E`0002@0003j00003@000001000M0000
0P000?X03@000Ad0E`0002@0003f000060000?h0000H00000P000?H0603n01P0E`0002@0003b0000
7@000?P0000M00000P000?807@3h01d0E`0002@0003m00007@000081000M00000P000?d07@020Ad0
E`0002@0000@0@004`000141000D00000P0001414`0@0A@0E`0002@0000@0@0050000141000E0000
0P000101500A0AD0E`0002@0000A0@0050000181000E00000P0001415@0B0A@0E`0002@0000B0@00
4`000181000D00000P000181500B0A<0E`0002@0000@0@004@000181000C00000P0001814`0@0A40
E`0002@0000>0@0040000101000A00000P0001014@0>0A00E`0002@0000<0@00400000h1000@0000
0P0000h1400<0A00E`0002@000090@00400000`1000A00000P0000`140090A40E`0002@000070@00
4@0000T1000C00000P0000T14@070A<0E`0002@000060@004`0000L1000F00000P0000L14`060AH0
E`0002@000060@005P0000H1000H00000P0000H15P060AP0E`0002@000060@00600000L1000K0000
0P0000H160070A/0E`0002@000070@006`0000T1000M00000P0000L16`090Ad0E`0002@000090@00
7@0000`1000M00000P0000T17@0<0Ad0E`0002@0000<0@007@0000d1000M00000P0000`17@0=0Ad0
E`0002@0000=0@007@000101000M00000P0000d17@0@0Ad0E`0002@0000@0@006`000181000M0000
0P0001017@0B0A/0E`0002@0000:0@00400000`1000A00000P0000`1400:0A40E`0002@000080@00
4@0000X1000C00000P0000X14@080A<0E`0002@000070@004`0000P1000F00000P0000P14`070AH0
E`0002@000070@005P0000L1000H00000P0000L15P070AP0E`0002@000070@00600000P1000K0000
0P0000L160080A/0E`0002@000080@006`0000X1000M00000P0000P16`0:0Ad0E`0002@0000:0@00
7@0000`1000M00000P0000X17@0<0Ad0E`0002@0000I0@002P0001T1000J00000P0001T12P0I0AX0
E`0002@0000I0@006P0001X1000M00000P0001T16P0J0Ad0E`0002@0000J0@007@0001`1000M0000
0P0001X17@0L0Ad0E`0002@0000L0@007@0001d1000M00000P0001`17@0M0Ad0E`0002@0000M0@00
7@0001l1000M00000P0001d17@0O0Ad0E`0002@0000O0@006`000201000M00000P0001l17@0P0A/0
E`0002@0000J0@002P0001X1000J00000P0001X12P0J0AX0E`0002@0000J0@006P0001/1000M0000
0P0001X16P0K0Ad0E`0002@0000K0@007@0001`1000M00000P0001/17@0L0Ad0E`0002@0000F0@00
400001d1000@00000P0001H1400M0A00E`0002@0000V0@00400002H1000K00000P0002H1400V0A/0
E`0002@0000V0@006`0002L1000M00000P0002H16`0W0Ad0E`0002@0000W0@007@0002X1000M0000
0P0002L17@0Z0Ad0E`0002@0000Z0@007@0002/1000M00000P0002X17@0[0Ad0E`0002@0000[0@00
7@0002h1000M00000P0002/17@0^0Ad0E`0002@0000^0@006`000301000M00000P0002h17@0`0A/0
E`0002@0000W0@00400002L1000K00000P0002L1400W0A/0E`0002@0000W0@006`0002P1000M0000
0P0002L16`0X0Ad0E`0002@0000X0@007@0002X1000M00000P0002P17@0Z0Ad0E`0002@0000`0@00
40000301000M00000P000301400`0Ad0E`0002@0000a0@0040000341000M00000P000341400a0Ad0
E`0002@0000S0@00400002L1000@00000P0002<1400W0A00E`0002@0000]0@0040000341000@0000
0P0002d1400a0A00E`0002@0000`0@007@0003<1000M00000P0003017@0c0Ad0E`0002@0000j0@00
4P0003X1000C00000P0003X14P0j0A<0E`0002@0000i0@004`0003X1000C00000P0003X14`0i0A<0
E`0002@0000i0@004P0003T1000C00000P0003T14`0i0A80E`0002@0000i0@004@0003X1000B0000
0P0003T14P0j0A40E`0002@0000j0@00400003`1000A00000P0003X14@0l0A00E`0002@0000l0@00
400003l1000@00000P0003`1400o0A00E`0002@0000o0@0040000441000A00000P0003l140110A40
E`0002@000110@004@000481000B00000P0004414@120A80E`0002@000120@004P0004<1000D0000
0P0004814P130A@0E`0002@000130@00500004<1000K00000P0004<150130A/0E`0002@000130@00
6`0004@1000M00000P0004<16`140Ad0E`0002@000140@007@0004D1000M00000P0004@17@150Ad0
E`0002@000120@004P000481000K00000P0004814P120A/0E`0002@000120@006`0004<1000M0000
0P0004816`130Ad0E`0002@000130@007@0004D1000M00000P0004<17@150Ad0E`0002@000150@00
7@0004H1000M00000P0004D17@160Ad0E`0002@000110@0050000481000E00000P00048150110AD0
E`0002@0000l0@005@000441000F00000P0004415@0l0AH0E`0002@0000i0@005P0003`1000G0000
0P0003`15P0i0AL0E`0002@0000h0@005`0003T1000I00000P0003T15`0h0AT0E`0002@0000h0@00
6@0003P1000K00000P0003P16@0h0A/0E`0002@0000h0@006`0003T1000M00000P0003P16`0i0Ad0
E`0002@0000i0@007@0003`1000M00000P0003T17@0l0Ad0E`0002@0000l0@007@0003h1000M0000
0P0003`17@0n0Ad0E`0002@0000n0@007@000401000M00000P0003h17@100Ad0E`0002@000100@00
6`000481000M00000P0004017@120A/0E`0002@0000j0@005P0003`1000G00000P0003`15P0j0AL0
E`0002@0000i0@005`0003X1000I00000P0003X15`0i0AT0E`0002@0000i0@006@0003T1000K0000
0P0003T16@0i0A/0E`0002@0000i0@006`0003X1000M00000P0003T16`0j0Ad0E`0002@0000j0@00
7@0003`1000M00000P0003X17@0l0Ad0E`0002@0001<0@002P0004`1000M00000P0004`12P1<0Ad0
E`0002@0001=0@002P0004d1000M00000P0004d12P1=0Ad0E`0002@000190@002P0004d1000:0000
0P0004T12P1=0@X0E`0002@000190@007@000501000M00000P0004T17@1@0Ad0E`0002@0001O0@00
4P0005l1000C00000P0005l14P1O0A<0E`0002@0001N0@004`0005l1000C00000P0005l14`1N0A<0
E`0002@0001N0@004P0005h1000C00000P0005h14`1N0A80E`0002@0001N0@004@0005l1000B0000
0P0005h14P1O0A40E`0002@0001O0@0040000641000A00000P0005l14@1Q0A00E`0002@0001Q0@00
400006@1000@00000P000641401T0A00E`0002@0001T0@00400006H1000A00000P0006@1401V0A40
E`0002@0001V0@004@0006L1000B00000P0006H14@1W0A80E`0002@0001W0@004P0006P1000D0000
0P0006L14P1X0A@0E`0002@0001X0@00500006P1000K00000P0006P1501X0A/0E`0002@0001X0@00
6`0006T1000M00000P0006P16`1Y0Ad0E`0002@0001Y0@007@0006X1000M00000P0006T17@1Z0Ad0
E`0002@0001W0@004P0006L1000K00000P0006L14P1W0A/0E`0002@0001W0@006`0006P1000M0000
0P0006L16`1X0Ad0E`0002@0001X0@007@0006X1000M00000P0006P17@1Z0Ad0E`0002@0001Z0@00
7@0006/1000M00000P0006X17@1[0Ad0E`0002@0001V0@00500006L1000E00000P0006L1501V0AD0
E`0002@0001Q0@005@0006H1000F00000P0006H15@1Q0AH0E`0002@0001N0@005P000641000G0000
0P0006415P1N0AL0E`0002@0001M0@005`0005h1000I00000P0005h15`1M0AT0E`0002@0001M0@00
6@0005d1000K00000P0005d16@1M0A/0E`0002@0001M0@006`0005h1000M00000P0005d16`1N0Ad0
E`0002@0001N0@007@000641000M00000P0005h17@1Q0Ad0E`0002@0001Q0@007@0006<1000M0000
0P0006417@1S0Ad0E`0002@0001S0@007@0006D1000M00000P0006<17@1U0Ad0E`0002@0001U0@00
6`0006L1000M00000P0006D17@1W0A/0E`0002@0001O0@005P000641000G00000P0006415P1O0AL0
E`0002@0001N0@005`0005l1000I00000P0005l15`1N0AT0E`0002@0001N0@006@0005h1000K0000
0P0005h16@1N0A/0E`0002@0001N0@006`0005l1000M00000P0005h16`1O0Ad0E`0002@0001O0@00
7@000641000M00000P0005l17@1Q0Ad0E`0002@0001a0@0040000741000M00000P000741401a0Ad0
E`0002@0001b0@0040000781000M00000P000781401b0Ad0E`0002@0001b0@004@0007@1000C0000
0P0007814`1d0A40E`0002@0001d0@00400007H1000A00000P0007@14@1f0A00E`0002@0001f0@00
400007P1000@00000P0007H1401h0A00E`0002@0001h0@00400007/1000A00000P0007P1401k0A40
E`0002@0001k0@004@0007`1000C00000P0007/14@1l0A<0E`0002@0001l0@004`0007`1000M0000
0P0007`14`1l0Ad0E`0002@0001h0@00400007X1000A00000P0007P1401j0A40E`0002@0001j0@00
4@0007/1000C00000P0007X14@1k0A<0E`0002@0001k0@004`0007/1000M00000P0007/14`1k0Ad0
E`0002@0001^0@0040000781000@00000P0006h1401b0A00E`0002@0001^0@007@0007D1000M0000
0P0006h17@1e0Ad0E`0002@0001h0@007@0007h1000M00000P0007P17@1n0Ad0E`0002@0002>0@00
2P0008h1000M00000P0008h12P2>0Ad0E`0002@0002?0@002P0008l1000M00000P0008l12P2?0Ad0
E`0002@0002<0@004@0008h1000C00000P0008h14`2<0A40E`0002@0002:0@00400008`1000A0000
0P0008`14@2:0A00E`0002@000290@00400008X1000@00000P0008X140290A00E`0002@000260@00
400008T1000A00000P0008T140260A40E`0002@000240@004@0008H1000C00000P0008H14@240A<0
E`0002@000230@004`0008@1000F00000P0008@14`230AH0E`0002@000230@005P0008<1000H0000
0P0008<15P230AP0E`0002@000230@00600008@1000K00000P0008<160240A/0E`0002@000240@00
6`0008H1000M00000P0008@16`260Ad0E`0002@000260@007@0008T1000M00000P0008H17@290Ad0
E`0002@000290@007@0008X1000M00000P0008T17@2:0Ad0E`0002@0002:0@007@0008`1000M0000
0P0008X17@2<0Ad0E`0002@0002<0@006`0008h1000M00000P0008`17@2>0A/0E`0002@000270@00
400008T1000A00000P0008T140270A40E`0002@000250@004@0008L1000C00000P0008L14@250A<0
E`0002@000240@004`0008D1000F00000P0008D14`240AH0E`0002@000240@005P0008@1000H0000
0P0008@15P240AP0E`0002@000240@00600008D1000K00000P0008@160250A/0E`0002@000250@00
6`0008L1000M00000P0008D16`270Ad0E`0002@000270@007@0008T1000M00000P0008L17@290Ad0
E`0002@0002;0@002P0008l1000:00000P0008/12P2?0@X0E`0002@0002>0@007@000941000M0000
0P0008h17@2A0Ad0E`0002@0002Y0@0040000:X1000B00000P000:T14P2Z0A00E`0002@0002Z0@00
40000:X1000D00000P000:X1402Z0A@0E`0002@0002Y0@004P000:X1000D00000P000:X1502Y0A80
E`0002@0002X0@004@000:T1000B00000P000:T14P2X0A40E`0002@0002V0@0040000:P1000A0000
0P000:P14@2V0A00E`0002@0002S0@0040000:H1000@00000P000:H1402S0A00E`0002@0002Q0@00
40000:<1000A00000P000:<1402Q0A40E`0002@0002P0@004@000:41000B00000P000:414@2P0A80
E`0002@0002P0@004P000:01000D00000P000:014P2P0A@0E`0002@0002P0@0050000:41000E0000
0P000:01502Q0AD0E`0002@0002Q0@005@000:<1000F00000P000:415@2S0AH0E`0002@0002S0@00
5P000:L1000H00000P000:<15P2W0AP0E`0002@0002W0@0060000:T1000I00000P000:L1602Y0AT0
E`0002@0002Y0@006@000:X1000J00000P000:T16@2Z0AX0E`0002@0002P0@004`000:41000D0000
0P000:014`2Q0A@0E`0002@0002Q0@0050000:<1000E00000P000:41502S0AD0E`0002@0002S0@00
5@000:L1000G00000P000:<15@2W0AL0E`0002@0002W0@005`000:T1000H00000P000:L15`2Y0AP0
E`0002@0002Y0@0060000:X1000I00000P000:T1602Z0AT0E`0002@0002Z0@006@000:X1000L0000
0P000:X16@2Z0A`0E`0002@0002Y0@0070000:X1000M00000P000:X1702Y0Ad0E`0002@0002W0@00
7@000:T1000M00000P000:T17@2W0Ad0E`0002@0002T0@007@000:L1000M00000P000:L17@2T0Ad0
E`0002@0002R0@007@000:@1000M00000P000:@17@2R0Ad0E`0002@0002Q0@0070000:81000M0000
0P000:817@2Q0A`0E`0002@0002P0@006P000:41000L00000P000:41702P0AX0E`0002@0002P0@00
6P000:01000M00000P000:016P2P0Ad0E`0002@0002P0@0070000:41000M00000P000:017@2Q0A`0
E`0002@0002`0@002P000;41000;00000P000;412P2`0@/0E`0002@0002`0@002`000;41000<0000
0P000;012`2a0@`0E`0002@0002a0@002`000;81000<00000P000;41302b0@/0E`0002@0002a0@00
2P000;81000;00000P000;812`2a0@X0E`0002@0002a0@0040000;41000M00000P000;41402a0Ad0
E`0002@0002b0@0040000;81000M00000P000;81402b0Ad0E`0002@0002^0@0040000;81000@0000
0P000:h1402b0A00E`0002@0002^0@007@000;D1000M00000P000:h17@2e0Ad0E`0002@0002k0@00
40000;/1000M00000P000;/1402k0Ad0E`0002@0002l0@0040000;`1000M00000P000;`1402l0Ad0
E`0002@0002l0@004@000;h1000C00000P000;`14`2n0A40E`0002@0002n0@0040000<01000A0000
0P000;h14@300A00E`0002@000300@0040000<81000@00000P000<0140320A00E`0002@000320@00
40000<D1000A00000P000<8140350A40E`0002@000350@004@000<H1000C00000P000<D14@360A<0
E`0002@000360@004`000<H1000M00000P000<H14`360Ad0E`0002@000320@0040000<@1000A0000
0P000<8140340A40E`0002@000340@004@000<D1000C00000P000<@14@350A<0E`0002@000350@00
4`000<D1000M00000P000<D14`350Ad0E`0002@000360@004@000<L1000C00000P000<H14`370A40
E`0002@000370@0040000<X1000A00000P000<L14@3:0A00E`0002@0003:0@0040000<`1000@0000
0P000<X1403<0A00E`0002@0003<0@0040000<l1000A00000P000<`1403?0A40E`0002@0003?0@00
4@000=01000C00000P000<l14@3@0A<0E`0002@0003@0@004`000=01000M00000P000=014`3@0Ad0
E`0002@0003<0@0040000<h1000A00000P000<`1403>0A40E`0002@0003>0@004@000<l1000C0000
0P000<h14@3?0A<0E`0002@0003?0@004`000<l1000M00000P000<l14`3?0Ad0E`0002@0002h0@00
40000;`1000@00000P000;P1402l0A00E`0002@0002h0@007@000;l1000M00000P000;P17@2o0Ad0
E`0002@000320@007@000<P1000M00000P000<817@380Ad0E`0002@0003<0@007@000=81000M0000
0P000<`17@3B0Ad0E`0002@0003H0@0040000=P1000K00000P000=P1403H0A/0E`0002@0003H0@00
6`000=T1000M00000P000=P16`3I0Ad0E`0002@0003I0@007@000=`1000M00000P000=T17@3L0Ad0
E`0002@0003L0@007@000=d1000M00000P000=`17@3M0Ad0E`0002@0003M0@007@000>01000M0000
0P000=d17@3P0Ad0E`0002@0003P0@006`000>81000M00000P000>017@3R0A/0E`0002@0003I0@00
40000=T1000K00000P000=T1403I0A/0E`0002@0003I0@006`000=X1000M00000P000=T16`3J0Ad0
E`0002@0003J0@007@000=`1000M00000P000=X17@3L0Ad0E`0002@0003R0@0040000>81000M0000
0P000>81403R0Ad0E`0002@0003S0@0040000><1000M00000P000><1403S0Ad0E`0002@0003E0@00
40000=T1000@00000P000=D1403I0A00E`0002@0003O0@0040000><1000@00000P000=l1403S0A00
E`0002@0003R0@007@000>D1000M00000P000>817@3U0Ad0E`0002@0003/0@002P000>`1000M0000
0P000>`12P3/0Ad0E`0002@0003]0@002P000>d1000M00000P000>d12P3]0Ad0E`0002@0003Y0@00
2P000>d1000:00000P000>T12P3]0@X0E`0002@0003Y0@007@000?01000M00000P000>T17@3`0Ad0
E`0002@0003e0@004P000?D1000C00000P000?D14P3e0A<0E`0002@0003d0@004`000?D1000C0000
0P000?D14`3d0A<0E`0002@0003d0@004P000?@1000C00000P000?@14`3d0A80E`0002@0003d0@00
4@000?D1000B00000P000?@14P3e0A40E`0002@0003e0@0040000?L1000A00000P000?D14@3g0A00
E`0002@0003g0@0040000?X1000@00000P000?L1403j0A00E`0002@0003j0@0040000?`1000A0000
0P000?X1403l0A40E`0002@0003l0@004@000?d1000B00000P000?`14@3m0A80E`0002@0003m0@00
4P000?h1000D00000P000?d14P3n0A@0E`0002@0003n0@0050000?h1000K00000P000?h1503n0A/0
E`0002@0003n0@006`000?l1000M00000P000?h16`3o0Ad0E`0002@0003o0@007@000002000M0000
0P000?l17@000Qd0E`0002@0003m0@004P000?d1000K00000P000?d14P3m0A/0E`0002@0003m0@00
6`000?h1000M00000P000?d16`3n0Ad0E`0002@0003n0@007@000002000M00000P000?h17@000Qd0
E`0002@000000P007@000042000M00000P0000027@010Qd0E`0002@0003l0@0050000?d1000E0000
0P000?d1503l0AD0E`0002@0003g0@005@000?`1000F00000P000?`15@3g0AH0E`0002@0003d0@00
5P000?L1000G00000P000?L15P3d0AL0E`0002@0003c0@005`000?@1000I00000P000?@15`3c0AT0
E`0002@0003c0@006@000?<1000K00000P000?<16@3c0A/0E`0002@0003c0@006`000?@1000M0000
0P000?<16`3d0Ad0E`0002@0003d0@007@000?L1000M00000P000?@17@3g0Ad0E`0002@0003g0@00
7@000?T1000M00000P000?L17@3i0Ad0E`0002@0003i0@007@000?/1000M00000P000?T17@3k0Ad0
E`0002@0003k0@006`000?d1000M00000P000?/17@3m0A/0E`0002@0003e0@005P000?L1000G0000
0P000?L15P3e0AL0E`0002@0003d0@005`000?D1000I00000P000?D15`3d0AT0E`0002@0003d0@00
6@000?@1000K00000P000?@16@3d0A/0E`0002@0003d0@006`000?D1000M00000P000?@16`3e0Ad0
E`0002@0003e0@007@000?L1000M00000P000?D17@3g0Ad0E`0002@000070P002P0000L2000J0000
0P0000L22P070QX0E`0002@000070P006P0000P2000M00000P0000L26P080Qd0E`0002@000080P00
7@0000X2000M00000P0000P27@0:0Qd0E`0002@0000:0P007@0000/2000M00000P0000X27@0;0Qd0
E`0002@0000;0P007@0000d2000M00000P0000/27@0=0Qd0E`0002@0000=0P006`0000h2000M0000
0P0000d27@0>0Q/0E`0002@000080P002P0000P2000J00000P0000P22P080QX0E`0002@000080P00
6P0000T2000M00000P0000P26P090Qd0E`0002@000090P007@0000X2000M00000P0000T27@0:0Qd0
E`0002@000040P00400000/2000@00000P0000@2400;0Q00E`0002@0000D0P005P0001l2000F0000
0P0001@25P0O0QH0E`0002@0000O0P00500001l2000F00000P0001l25P0O0Q@0E`0002@0000N0P00
4P0001l2000D00000P0001l2500N0Q80E`0002@0000M0P004@0001h2000B00000P0001h24P0M0Q40
E`0002@0000K0P00400001d2000A00000P0001d24@0K0Q00E`0002@0000I0P00400001/2000@0000
0P0001/2400I0Q00E`0002@0000F0P00400001T2000A00000P0001T2400F0Q40E`0002@0000D0P00
4@0001H2000C00000P0001H24@0D0Q<0E`0002@0000C0P004`0001@2000F00000P0001@24`0C0QH0
E`0002@0000C0P005P0001<2000H00000P0001<25P0C0QP0E`0002@0000C0P00600001@2000K0000
0P0001<2600D0Q/0E`0002@0000D0P006`0001H2000M00000P0001@26`0F0Qd0E`0002@0000F0P00
7@0001T2000M00000P0001H27@0I0Qd0E`0002@0000I0P007@0001X2000M00000P0001T27@0J0Qd0
E`0002@0000J0P007@0001d2000M00000P0001X27@0M0Qd0E`0002@0000M0P006`0001l2000M0000
0P0001d27@0O0Q/0E`0002@0000N0P004`0001h2000F00000P0001h25P0N0Q<0E`0002@0000M0P00
4@0001h2000C00000P0001h24`0M0Q40E`0002@0000G0P00400001T2000A00000P0001T2400G0Q40
E`0002@0000E0P004@0001L2000C00000P0001L24@0E0Q<0E`0002@0000D0P004`0001D2000F0000
0P0001D24`0D0QH0E`0002@0000D0P005P0001@2000H00000P0001@25P0D0QP0E`0002@0000D0P00
600001D2000K00000P0001@2600E0Q/0E`0002@0000E0P006`0001L2000M00000P0001D26`0G0Qd0
E`0002@0000G0P007@0001T2000M00000P0001L27@0I0Qd0E`0002@0000^0P002P0002h2000M0000
0P0002h22P0^0Qd0E`0002@0000_0P002P0002l2000M00000P0002l22P0_0Qd0E`0002@0000/0P00
4@0002h2000C00000P0002h24`0/0Q40E`0002@0000Z0P00400002`2000A00000P0002`24@0Z0Q00
E`0002@0000Y0P00400002X2000@00000P0002X2400Y0Q00E`0002@0000V0P00400002T2000A0000
0P0002T2400V0Q40E`0002@0000T0P004@0002H2000C00000P0002H24@0T0Q<0E`0002@0000S0P00
4`0002@2000F00000P0002@24`0S0QH0E`0002@0000S0P005P0002<2000H00000P0002<25P0S0QP0
E`0002@0000S0P00600002@2000K00000P0002<2600T0Q/0E`0002@0000T0P006`0002H2000M0000
0P0002@26`0V0Qd0E`0002@0000V0P007@0002T2000M00000P0002H27@0Y0Qd0E`0002@0000Y0P00
7@0002X2000M00000P0002T27@0Z0Qd0E`0002@0000Z0P007@0002`2000M00000P0002X27@0/0Qd0
E`0002@0000/0P006`0002h2000M00000P0002`27@0^0Q/0E`0002@0000W0P00400002T2000A0000
0P0002T2400W0Q40E`0002@0000U0P004@0002L2000C00000P0002L24@0U0Q<0E`0002@0000T0P00
4`0002D2000F00000P0002D24`0T0QH0E`0002@0000T0P005P0002@2000H00000P0002@25P0T0QP0
E`0002@0000T0P00600002D2000K00000P0002@2600U0Q/0E`0002@0000U0P006`0002L2000M0000
0P0002D26`0W0Qd0E`0002@0000W0P007@0002T2000M00000P0002L27@0Y0Qd0E`0002@0000[0P00
2P0002l2000:00000P0002/22P0_0PX0E`0002@0000^0P007@000342000M00000P0002h27@0a0Qd0
E`0002@000120P002P000482000J00000P0004822P120QX0E`0002@000120P006P0004<2000M0000
0P0004826P130Qd0E`0002@000130P007@0004D2000M00000P0004<27@150Qd0E`0002@000150P00
7@0004H2000M00000P0004D27@160Qd0E`0002@000160P007@0004P2000M00000P0004H27@180Qd0
E`0002@000180P006`0004T2000M00000P0004P27@190Q/0E`0002@000130P002P0004<2000J0000
0P0004<22P130QX0E`0002@000130P006P0004@2000M00000P0004<26P140Qd0E`0002@000140P00
7@0004D2000M00000P0004@27@150Qd0E`0002@0000o0P00400004H2000@00000P0003l240160Q00
E`0002@0001@0P0040000502000M00000P000502401@0Qd0E`0002@0001A0P0040000542000M0000
0P000542401A0Qd0E`0002@0001A0P004`000582000F00000P0005425P1B0Q<0E`0002@0001B0P00
4@0005@2000C00000P0005824`1D0Q40E`0002@0001D0P00400005D2000A00000P0005@24@1E0Q00
E`0002@0001E0P00400005P2000@00000P0005D2401H0Q00E`0002@0001H0P00400005T2000A0000
0P0005P2401I0Q40E`0002@0001I0P004@0005T2000B00000P0005T24@1I0Q80E`0002@0001H0P00
4P0005T2000C00000P0005T24P1H0Q<0E`0002@0001G0P004P0005P2000C00000P0005P24`1G0Q80
E`0002@0001G0P004@0005P2000B00000P0005L24P1H0Q40E`0002@0001=0P0040000542000@0000
0P0004d2401A0Q00E`0002@0001=0P007@0005@2000M00000P0004d27@1D0Qd0E`0002@0001O0P00
4P0005l2000C00000P0005l24P1O0Q<0E`0002@0001N0P004`0005l2000C00000P0005l24`1N0Q<0
E`0002@0001N0P004P0005h2000C00000P0005h24`1N0Q80E`0002@0001N0P004@0005l2000B0000
0P0005h24P1O0Q40E`0002@0001O0P0040000642000A00000P0005l24@1Q0Q00E`0002@0001Q0P00
400006@2000@00000P000642401T0Q00E`0002@0001T0P00400006H2000A00000P0006@2401V0Q40
E`0002@0001V0P004@0006L2000B00000P0006H24@1W0Q80E`0002@0001W0P004P0006P2000D0000
0P0006L24P1X0Q@0E`0002@0001X0P00500006P2000K00000P0006P2501X0Q/0E`0002@0001X0P00
6`0006T2000M00000P0006P26`1Y0Qd0E`0002@0001Y0P007@0006X2000M00000P0006T27@1Z0Qd0
E`0002@0001W0P004P0006L2000K00000P0006L24P1W0Q/0E`0002@0001W0P006`0006P2000M0000
0P0006L26`1X0Qd0E`0002@0001X0P007@0006X2000M00000P0006P27@1Z0Qd0E`0002@0001Z0P00
7@0006/2000M00000P0006X27@1[0Qd0E`0002@0001V0P00500006L2000E00000P0006L2501V0QD0
E`0002@0001Q0P005@0006H2000F00000P0006H25@1Q0QH0E`0002@0001N0P005P000642000G0000
0P0006425P1N0QL0E`0002@0001M0P005`0005h2000I00000P0005h25`1M0QT0E`0002@0001M0P00
6@0005d2000K00000P0005d26@1M0Q/0E`0002@0001M0P006`0005h2000M00000P0005d26`1N0Qd0
E`0002@0001N0P007@000642000M00000P0005h27@1Q0Qd0E`0002@0001Q0P007@0006<2000M0000
0P0006427@1S0Qd0E`0002@0001S0P007@0006D2000M00000P0006<27@1U0Qd0E`0002@0001U0P00
6`0006L2000M00000P0006D27@1W0Q/0E`0002@0001O0P005P000642000G00000P0006425P1O0QL0
E`0002@0001N0P005`0005l2000I00000P0005l25`1N0QT0E`0002@0001N0P006@0005h2000K0000
0P0005h26@1N0Q/0E`0002@0001N0P006`0005l2000M00000P0005h26`1O0Qd0E`0002@0001O0P00
7@000642000M00000P0005l27@1Q0Qd0E`0002@0001j0P002P0007X2000M00000P0007X22P1j0Qd0
E`0002@0001k0P002P0007/2000M00000P0007/22P1k0Qd0E`0002@0001h0P004@0007X2000C0000
0P0007X24`1h0Q40E`0002@0001f0P00400007P2000A00000P0007P24@1f0Q00E`0002@0001e0P00
400007H2000@00000P0007H2401e0Q00E`0002@0001b0P00400007D2000A00000P0007D2401b0Q40
E`0002@0001`0P004@000782000C00000P0007824@1`0Q<0E`0002@0001_0P004`000702000F0000
0P0007024`1_0QH0E`0002@0001_0P005P0006l2000H00000P0006l25P1_0QP0E`0002@0001_0P00
60000702000K00000P0006l2601`0Q/0E`0002@0001`0P006`000782000M00000P0007026`1b0Qd0
E`0002@0001b0P007@0007D2000M00000P0007827@1e0Qd0E`0002@0001e0P007@0007H2000M0000
0P0007D27@1f0Qd0E`0002@0001f0P007@0007P2000M00000P0007H27@1h0Qd0E`0002@0001h0P00
6`0007X2000M00000P0007P27@1j0Q/0E`0002@0001c0P00400007D2000A00000P0007D2401c0Q40
E`0002@0001a0P004@0007<2000C00000P0007<24@1a0Q<0E`0002@0001`0P004`000742000F0000
0P0007424`1`0QH0E`0002@0001`0P005P000702000H00000P0007025P1`0QP0E`0002@0001`0P00
60000742000K00000P000702601a0Q/0E`0002@0001a0P006`0007<2000M00000P0007426`1c0Qd0
E`0002@0001c0P007@0007D2000M00000P0007<27@1e0Qd0E`0002@0001g0P002P0007/2000:0000
0P0007L22P1k0PX0E`0002@0001j0P007@0007d2000M00000P0007X27@1m0Qd0E`0002@000220P00
5P0008d2000F00000P0008825P2=0QH0E`0002@0002=0P00500008d2000F00000P0008d25P2=0Q@0
E`0002@0002<0P004P0008d2000D00000P0008d2502<0Q80E`0002@0002;0P004@0008`2000B0000
0P0008`24P2;0Q40E`0002@000290P00400008/2000A00000P0008/24@290Q00E`0002@000270P00
400008T2000@00000P0008T240270Q00E`0002@000240P00400008L2000A00000P0008L240240Q40
E`0002@000220P004@0008@2000C00000P0008@24@220Q<0E`0002@000210P004`000882000F0000
0P0008824`210QH0E`0002@000210P005P000842000H00000P0008425P210QP0E`0002@000210P00
60000882000K00000P00084260220Q/0E`0002@000220P006`0008@2000M00000P0008826`240Qd0
E`0002@000240P007@0008L2000M00000P0008@27@270Qd0E`0002@000270P007@0008P2000M0000
0P0008L27@280Qd0E`0002@000280P007@0008/2000M00000P0008P27@2;0Qd0E`0002@0002;0P00
6`0008d2000M00000P0008/27@2=0Q/0E`0002@0002<0P004`0008`2000F00000P0008`25P2<0Q<0
E`0002@0002;0P004@0008`2000C00000P0008`24`2;0Q40E`0002@000250P00400008L2000A0000
0P0008L240250Q40E`0002@000230P004@0008D2000C00000P0008D24@230Q<0E`0002@000220P00
4`0008<2000F00000P0008<24`220QH0E`0002@000220P005P000882000H00000P0008825P220QP0
E`0002@000220P00600008<2000K00000P00088260230Q/0E`0002@000230P006`0008D2000M0000
0P0008<26`250Qd0E`0002@000250P007@0008L2000M00000P0008D27@270Qd0E`0002@0002W0P00
2P000:L2000M00000P000:L22P2W0Qd0E`0002@0002X0P002P000:P2000M00000P000:P22P2X0Qd0
E`0002@0002U0P004@000:L2000C00000P000:L24`2U0Q40E`0002@0002S0P0040000:D2000A0000
0P000:D24@2S0Q00E`0002@0002R0P0040000:<2000@00000P000:<2402R0Q00E`0002@0002O0P00
40000:82000A00000P000:82402O0Q40E`0002@0002M0P004@0009l2000C00000P0009l24@2M0Q<0
E`0002@0002L0P004`0009d2000F00000P0009d24`2L0QH0E`0002@0002L0P005P0009`2000H0000
0P0009`25P2L0QP0E`0002@0002L0P00600009d2000K00000P0009`2602M0Q/0E`0002@0002M0P00
6`0009l2000M00000P0009d26`2O0Qd0E`0002@0002O0P007@000:82000M00000P0009l27@2R0Qd0
E`0002@0002R0P007@000:<2000M00000P000:827@2S0Qd0E`0002@0002S0P007@000:D2000M0000
0P000:<27@2U0Qd0E`0002@0002U0P006`000:L2000M00000P000:D27@2W0Q/0E`0002@0002P0P00
40000:82000A00000P000:82402P0Q40E`0002@0002N0P004@000:02000C00000P000:024@2N0Q<0
E`0002@0002M0P004`0009h2000F00000P0009h24`2M0QH0E`0002@0002M0P005P0009d2000H0000
0P0009d25P2M0QP0E`0002@0002M0P00600009h2000K00000P0009d2602N0Q/0E`0002@0002N0P00
6`000:02000M00000P0009h26`2P0Qd0E`0002@0002P0P007@000:82000M00000P000:027@2R0Qd0
E`0002@0002T0P002P000:P2000:00000P000:@22P2X0PX0E`0002@0002W0P007@000:X2000M0000
0P000:L27@2Z0Qd0E`0002@0002`0P002P000;42000;00000P000;422P2`0P/0E`0002@0002`0P00
2`000;42000<00000P000;022`2a0P`0E`0002@0002a0P002`000;82000<00000P000;42302b0P/0
E`0002@0002a0P002P000;82000;00000P000;822`2a0PX0E`0002@0002a0P0040000;42000M0000
0P000;42402a0Qd0E`0002@0002b0P0040000;82000M00000P000;82402b0Qd0E`0002@0002^0P00
40000;82000@00000P000:h2402b0Q00E`0002@0002^0P007@000;D2000M00000P000:h27@2e0Qd0
E`0002@0002k0P0040000;/2000M00000P000;/2402k0Qd0E`0002@0002l0P0040000;`2000M0000
0P000;`2402l0Qd0E`0002@0002l0P004`000;d2000F00000P000;`25P2m0Q<0E`0002@0002m0P00
4@000;l2000C00000P000;d24`2o0Q40E`0002@0002o0P0040000<02000A00000P000;l24@300Q00
E`0002@000300P0040000<<2000@00000P000<0240330Q00E`0002@000330P0040000<@2000A0000
0P000<<240340Q40E`0002@000340P004@000<@2000B00000P000<@24@340Q80E`0002@000330P00
4P000<@2000C00000P000<@24P330Q<0E`0002@000320P004P000<<2000C00000P000<<24`320Q80
E`0002@000320P004@000<<2000B00000P000<824P330Q40E`0002@0002h0P0040000;`2000@0000
0P000;P2402l0Q00E`0002@0002h0P007@000;l2000M00000P000;P27@2o0Qd0E`0002@000390P00
5P000=@2000F00000P000<T25P3D0QH0E`0002@0003D0P0050000=@2000F00000P000=@25P3D0Q@0
E`0002@0003C0P004P000=@2000D00000P000=@2503C0Q80E`0002@0003B0P004@000=<2000B0000
0P000=<24P3B0Q40E`0002@0003@0P0040000=82000A00000P000=824@3@0Q00E`0002@0003>0P00
40000=02000@00000P000=02403>0Q00E`0002@0003;0P0040000<h2000A00000P000<h2403;0Q40
E`0002@000390P004@000</2000C00000P000</24@390Q<0E`0002@000380P004`000<T2000F0000
0P000<T24`380QH0E`0002@000380P005P000<P2000H00000P000<P25P380QP0E`0002@000380P00
60000<T2000K00000P000<P260390Q/0E`0002@000390P006`000</2000M00000P000<T26`3;0Qd0
E`0002@0003;0P007@000<h2000M00000P000</27@3>0Qd0E`0002@0003>0P007@000<l2000M0000
0P000<h27@3?0Qd0E`0002@0003?0P007@000=82000M00000P000<l27@3B0Qd0E`0002@0003B0P00
6`000=@2000M00000P000=827@3D0Q/0E`0002@0003C0P004`000=<2000F00000P000=<25P3C0Q<0
E`0002@0003B0P004@000=<2000C00000P000=<24`3B0Q40E`0002@0003<0P0040000<h2000A0000
0P000<h2403<0Q40E`0002@0003:0P004@000<`2000C00000P000<`24@3:0Q<0E`0002@000390P00
4`000<X2000F00000P000<X24`390QH0E`0002@000390P005P000<T2000H00000P000<T25P390QP0
E`0002@000390P0060000<X2000K00000P000<T2603:0Q/0E`0002@0003:0P006`000<`2000M0000
0P000<X26`3<0Qd0E`0002@0003<0P007@000<h2000M00000P000<`27@3>0Qd0E`0002@0003S0P00
4`000>@2000D00000P000>@24`3S0Q@0E`0002@0003S0P0050000>@2000E00000P000><2503T0QD0
E`0002@0003T0P0050000>D2000E00000P000>@25@3U0Q@0E`0002@0003U0P004`000>D2000D0000
0P000>D2503U0Q<0E`0002@0003S0P004@000>D2000C00000P000>D24`3S0Q40E`0002@0003Q0P00
40000><2000A00000P000><24@3Q0Q00E`0002@0003O0P0040000>42000@00000P000>42403O0Q00
E`0002@0003L0P0040000=l2000A00000P000=l2403L0Q40E`0002@0003J0P004@000=`2000C0000
0P000=`24@3J0Q<0E`0002@0003I0P004`000=X2000F00000P000=X24`3I0QH0E`0002@0003I0P00
5P000=T2000H00000P000=T25P3I0QP0E`0002@0003I0P0060000=X2000K00000P000=T2603J0Q/0
E`0002@0003J0P006`000=`2000M00000P000=X26`3L0Qd0E`0002@0003L0P007@000=l2000M0000
0P000=`27@3O0Qd0E`0002@0003O0P007@000>02000M00000P000=l27@3P0Qd0E`0002@0003P0P00
7@000><2000M00000P000>027@3S0Qd0E`0002@0003S0P006`000>D2000M00000P000><27@3U0Q/0
E`0002@0003M0P0040000=l2000A00000P000=l2403M0Q40E`0002@0003K0P004@000=d2000C0000
0P000=d24@3K0Q<0E`0002@0003J0P004`000=/2000F00000P000=/24`3J0QH0E`0002@0003J0P00
5P000=X2000H00000P000=X25P3J0QP0E`0002@0003J0P0060000=/2000K00000P000=X2603K0Q/0
E`0002@0003K0P006`000=d2000M00000P000=/26`3M0Qd0E`0002@0003M0P007@000=l2000M0000
0P000=d27@3O0Qd0E`0002@0003/0P002P000>`2000J00000P000>`22P3/0QX0E`0002@0003/0P00
6P000>d2000M00000P000>`26P3]0Qd0E`0002@0003]0P007@000>l2000M00000P000>d27@3_0Qd0
E`0002@0003_0P007@000?02000M00000P000>l27@3`0Qd0E`0002@0003`0P007@000?82000M0000
0P000?027@3b0Qd0E`0002@0003b0P006`000?<2000M00000P000?827@3c0Q/0E`0002@0003]0P00
2P000>d2000J00000P000>d22P3]0QX0E`0002@0003]0P006P000>h2000M00000P000>d26P3^0Qd0
E`0002@0003^0P007@000>l2000M00000P000>h27@3_0Qd0E`0002@0003Y0P0040000?02000@0000
0P000>T2403`0Q00E`0002@0003h0P002P000?T2000;00000P000?T22P3h0P/0E`0002@0003h0P00
2`000?T2000<00000P000?P22`3i0P`0E`0002@0003i0P002`000?X2000<00000P000?T2303j0P/0
E`0002@0003i0P002P000?X2000;00000P000?X22`3i0PX0E`0002@0003i0P0040000?T2000M0000
0P000?T2403i0Qd0E`0002@0003j0P0040000?X2000M00000P000?X2403j0Qd0E`0002@0003f0P00
40000?X2000@00000P000?H2403j0Q00E`0002@0003f0P007@000?d2000M00000P000?H27@3m0Qd0
E`0002@000040`00400000L3000A00000P0000L340040a40E`0002@000020`004@0000@3000C0000
0P0000@34@020a<0E`0002@000010`004`000083000F00000P0000834`010aH0E`0002@000010`00
5P000043000H00000P0000435P010aP0E`0002@000010`0060000083000K00000P00004360020a/0
E`0002@000020`006`0000@3000M00000P0000836`040ad0E`0002@000040`007@0000L3000M0000
0P0000@37@070ad0E`0002@000070`007@0000P3000M00000P0000L37@080ad0E`0002@000080`00
7@0000/3000M00000P0000P37@0;0ad0E`0002@0000;0`006`0000d3000M00000P0000/37@0=0a/0
E`0002@0000=0`00600000h3000K00000P0000d36`0>0aP0E`0002@0000>0`005P0000h3000H0000
0P0000h3600>0aH0E`0002@0000=0`004`0000h3000F00000P0000h35P0=0a<0E`0002@0000;0`00
4@0000d3000C00000P0000d34`0;0a40E`0002@000080`00400000/3000A00000P0000/34@080a00
E`0002@000070`00400000P3000@00000P0000P340070a00E`0002@000050`00400000L3000A0000
0P0000L340050a40E`0002@000030`004@0000D3000C00000P0000D34@030a<0E`0002@000020`00
4`0000<3000F00000P0000<34`020aH0E`0002@000020`005P000083000H00000P0000835P020aP0
E`0002@000020`00600000<3000K00000P00008360030a/0E`0002@000030`006`0000D3000M0000
0P0000<36`050ad0E`0002@000050`007@0000L3000M00000P0000D37@070ad0E`0002@000080`00
7@0000X3000M00000P0000P37@0:0ad0E`0002@0000:0`006`0000`3000M00000P0000X37@0<0a/0
E`0002@0000<0`00600000d3000K00000P0000`36`0=0aP0E`0002@0000=0`005P0000d3000H0000
0P0000d3600=0aH0E`0002@0000<0`004`0000d3000F00000P0000d35P0<0a<0E`0002@0000:0`00
4@0000`3000C00000P0000`34`0:0a40E`0002@000080`00400000X3000A00000P0000X34@080a00
E`0002@0000E0`00400001D3000M00000P0001D3400E0ad0E`0002@0000F0`00400001H3000M0000
0P0001H3400F0ad0E`0002@0000F0`004@0001P3000C00000P0001H34`0H0a40E`0002@0000H0`00
400001X3000A00000P0001P34@0J0a00E`0002@0000J0`00400001`3000@00000P0001X3400L0a00
E`0002@0000L0`00400001l3000A00000P0001`3400O0a40E`0002@0000O0`004@000203000C0000
0P0001l34@0P0a<0E`0002@0000P0`004`000203000M00000P0002034`0P0ad0E`0002@0000L0`00
400001h3000A00000P0001`3400N0a40E`0002@0000N0`004@0001l3000C00000P0001h34@0O0a<0
E`0002@0000O0`004`0001l3000M00000P0001l34`0O0ad0E`0002@0000B0`00400001H3000@0000
0P000183400F0a00E`0002@0000B0`007@0001T3000M00000P0001837@0I0ad0E`0002@0000L0`00
7@000283000M00000P0001`37@0R0ad0E`0002@0000_0`0040000303000B00000P0002l34P0`0a00
E`0002@0000`0`0040000303000D00000P000303400`0a@0E`0002@0000_0`004P000303000D0000
0P000303500_0a80E`0002@0000^0`004@0002l3000B00000P0002l34P0^0a40E`0002@0000/0`00
400002h3000A00000P0002h34@0/0a00E`0002@0000Y0`00400002`3000@00000P0002`3400Y0a00
E`0002@0000W0`00400002T3000A00000P0002T3400W0a40E`0002@0000V0`004@0002L3000B0000
0P0002L34@0V0a80E`0002@0000V0`004P0002H3000D00000P0002H34P0V0a@0E`0002@0000V0`00
500002L3000E00000P0002H3500W0aD0E`0002@0000W0`005@0002T3000F00000P0002L35@0Y0aH0
E`0002@0000Y0`005P0002d3000H00000P0002T35P0]0aP0E`0002@0000]0`00600002l3000I0000
0P0002d3600_0aT0E`0002@0000_0`006@000303000J00000P0002l36@0`0aX0E`0002@0000V0`00
4`0002L3000D00000P0002H34`0W0a@0E`0002@0000W0`00500002T3000E00000P0002L3500Y0aD0
E`0002@0000Y0`005@0002d3000G00000P0002T35@0]0aL0E`0002@0000]0`005`0002l3000H0000
0P0002d35`0_0aP0E`0002@0000_0`0060000303000I00000P0002l3600`0aT0E`0002@0000`0`00
6@000303000L00000P0003036@0`0a`0E`0002@0000_0`0070000303000M00000P000303700_0ad0
E`0002@0000]0`007@0002l3000M00000P0002l37@0]0ad0E`0002@0000Z0`007@0002d3000M0000
0P0002d37@0Z0ad0E`0002@0000X0`007@0002X3000M00000P0002X37@0X0ad0E`0002@0000W0`00
700002P3000M00000P0002P37@0W0a`0E`0002@0000V0`006P0002L3000L00000P0002L3700V0aX0
E`0002@0000V0`006P0002H3000M00000P0002H36P0V0ad0E`0002@0000V0`00700002L3000M0000
0P0002H37@0W0a`0E`0002@000100`00400004@3000M00000P00040340140ad0E`0002@000110`00
400004@3000K00000P00044340140a/0E`0002@000140`00400004L3000M00000P0004L340140ad0
E`0002@000170`00400004/3000M00000P0004L3401;0ad0E`0002@000180`00400004/3000K0000
0P0004P3401;0a/0E`0002@0001;0`00400004h3000M00000P0004h3401;0ad0E`0002@0000m0`00
400004@3000@00000P0003d340140a00E`0002@0001<0`0040000543000@00000P0004`3401A0a00
E`0002@0001F0`002P0005L3000;00000P0005L32P1F0`/0E`0002@0001F0`002`0005L3000<0000
0P0005H32`1G0``0E`0002@0001G0`002`0005P3000<00000P0005L3301H0`/0E`0002@0001G0`00
2P0005P3000;00000P0005P32`1G0`X0E`0002@0001G0`00400005L3000M00000P0005L3401G0ad0
E`0002@0001H0`00400005P3000M00000P0005P3401H0ad0E`0002@0001D0`00400005P3000@0000
0P0005@3401H0a00E`0002@0001D0`007@0005/3000M00000P0005@37@1K0ad0E`0002@0001Q0`00
2P000643000J00000P0006432P1Q0aX0E`0002@0001Q0`006P000683000M00000P0006436P1R0ad0
E`0002@0001R0`007@0006@3000M00000P0006837@1T0ad0E`0002@0001T0`007@0006D3000M0000
0P0006@37@1U0ad0E`0002@0001U0`007@0006L3000M00000P0006D37@1W0ad0E`0002@0001W0`00
6`0006P3000M00000P0006L37@1X0a/0E`0002@0001R0`002P000683000J00000P0006832P1R0aX0
E`0002@0001R0`006P0006<3000M00000P0006836P1S0ad0E`0002@0001S0`007@0006@3000M0000
0P0006<37@1T0ad0E`0002@0001N0`00400006D3000@00000P0005h3401U0a00E`0002@0001^0`00
2P0006h3000M00000P0006h32P1^0ad0E`0002@0001_0`002P0006l3000M00000P0006l32P1_0ad0
E`0002@0001_0`004@000743000C00000P0006l34`1a0a40E`0002@0001a0`00400007<3000A0000
0P0007434@1c0a00E`0002@0001c0`00400007D3000@00000P0007<3401e0a00E`0002@0001e0`00
400007P3000A00000P0007D3401h0a40E`0002@0001h0`004@0007T3000C00000P0007P34@1i0a<0
E`0002@0001i0`004`0007T3000M00000P0007T34`1i0ad0E`0002@0001e0`00400007L3000A0000
0P0007D3401g0a40E`0002@0001g0`004@0007P3000C00000P0007L34@1h0a<0E`0002@0001h0`00
4`0007P3000M00000P0007P34`1h0ad0E`0002@0001[0`002P0006l3000:00000P0006/32P1_0`X0
E`0002@0001[0`007@000783000M00000P0006/37@1b0ad0E`0002@0001e0`007@0007/3000M0000
0P0007D37@1k0ad0E`0002@0002D0`004`0009D3000D00000P0009D34`2D0a@0E`0002@0002D0`00
500009D3000E00000P0009@3502E0aD0E`0002@0002E0`00500009H3000E00000P0009D35@2F0a@0
E`0002@0002F0`004`0009H3000D00000P0009H3502F0a<0E`0002@0002D0`004@0009H3000C0000
0P0009H34`2D0a40E`0002@0002B0`00400009@3000A00000P0009@34@2B0a00E`0002@0002@0`00
40000983000@00000P000983402@0a00E`0002@0002=0`0040000903000A00000P000903402=0a40
E`0002@0002;0`004@0008d3000C00000P0008d34@2;0a<0E`0002@0002:0`004`0008/3000F0000
0P0008/34`2:0aH0E`0002@0002:0`005P0008X3000H00000P0008X35P2:0aP0E`0002@0002:0`00
600008/3000K00000P0008X3602;0a/0E`0002@0002;0`006`0008d3000M00000P0008/36`2=0ad0
E`0002@0002=0`007@000903000M00000P0008d37@2@0ad0E`0002@0002@0`007@000943000M0000
0P0009037@2A0ad0E`0002@0002A0`007@0009@3000M00000P0009437@2D0ad0E`0002@0002D0`00
6`0009H3000M00000P0009@37@2F0a/0E`0002@0002>0`0040000903000A00000P000903402>0a40
E`0002@0002<0`004@0008h3000C00000P0008h34@2<0a<0E`0002@0002;0`004`0008`3000F0000
0P0008`34`2;0aH0E`0002@0002;0`005P0008/3000H00000P0008/35P2;0aP0E`0002@0002;0`00
600008`3000K00000P0008/3602<0a/0E`0002@0002<0`006`0008h3000M00000P0008`36`2>0ad0
E`0002@0002>0`007@000903000M00000P0008h37@2@0ad0E`0002@0002L0`004P000:`3000B0000
0P0009`34P2/0a80E`0002@0002L0`0060000:`3000H00000P0009`3602/0aP0E`0002@0002e0`00
2P000;P3000;00000P000;P32P2e0`/0E`0002@0002c0`002`000;D3000>00000P000;D32`2c0`h0
E`0002@0002b0`003P000;<3000B00000P000;<33P2b0a80E`0002@0002b0`004P000;83000E0000
0P000;834P2b0aD0E`0002@0002b0`005@000;<3000J00000P000;835@2c0aX0E`0002@0002c0`00
6P000;D3000M00000P000;<36P2e0ad0E`0002@0002e0`007@000;P3000M00000P000;D37@2h0ad0
E`0002@0002h0`007@000;T3000M00000P000;P37@2i0ad0E`0002@0002i0`007@000;`3000M0000
0P000;T37@2l0ad0E`0002@0002l0`006P000;h3000M00000P000;`37@2n0aX0E`0002@0002n0`00
5@000;l3000J00000P000;h36P2o0aD0E`0002@0002o0`004P000;l3000E00000P000;l35@2o0a80
E`0002@0002n0`003P000;l3000B00000P000;l34P2n0`h0E`0002@0002l0`002`000;h3000>0000
0P000;h33P2l0`/0E`0002@0002i0`002P000;`3000;00000P000;`32`2i0`X0E`0002@0002h0`00
2P000;T3000:00000P000;T32P2h0`X0E`0002@0002f0`002P000;P3000;00000P000;P32P2f0`/0
E`0002@0002e0`002`000;H3000<00000P000;H32`2e0``0E`0002@0002d0`0030000;D3000>0000
0P000;D3302d0`h0E`0002@0002c0`003P000;@3000B00000P000;@33P2c0a80E`0002@0002c0`00
4P000;<3000E00000P000;<34P2c0aD0E`0002@0002c0`005@000;@3000J00000P000;<35@2d0aX0
E`0002@0002d0`006P000;D3000L00000P000;@36P2e0a`0E`0002@0002e0`0070000;H3000M0000
0P000;D3702f0ad0E`0002@0002f0`007@000;P3000M00000P000;H37@2h0ad0E`0002@0002i0`00
7@000;/3000M00000P000;T37@2k0ad0E`0002@0002k0`0070000;`3000M00000P000;/37@2l0a`0
E`0002@0002l0`006P000;d3000L00000P000;`3702m0aX0E`0002@0002m0`005@000;h3000J0000
0P000;d36P2n0aD0E`0002@0002n0`004P000;h3000E00000P000;h35@2n0a80E`0002@0002m0`00
3P000;h3000B00000P000;h34P2m0`h0E`0002@0002l0`0030000;d3000>00000P000;d33P2l0``0
E`0002@0002k0`002`000;`3000<00000P000;`3302k0`/0E`0002@0002i0`002P000;/3000;0000
0P000;/32`2i0`X0E`0002@000350`0070000<H3000M00000P000<H370350ad0E`0002@000350`00
7@000<H3000M00000P000<D37@360ad0E`0002@000360`007@000<L3000M00000P000<H37@370ad0
E`0002@000360`0070000<L3000M00000P000<L37@360a`0E`0002@0003@0`002P000=<3000;0000
0P000=<32P3@0`/0E`0002@0003>0`002`000=03000>00000P000=032`3>0`h0E`0002@0003=0`00
3P000<h3000B00000P000<h33P3=0a80E`0002@0003=0`004P000<d3000E00000P000<d34P3=0aD0
E`0002@0003=0`005@000<h3000J00000P000<d35@3>0aX0E`0002@0003>0`006P000=03000M0000
0P000<h36P3@0ad0E`0002@0003@0`007@000=<3000M00000P000=037@3C0ad0E`0002@0003C0`00
7@000=@3000M00000P000=<37@3D0ad0E`0002@0003D0`007@000=L3000M00000P000=@37@3G0ad0
E`0002@0003G0`006P000=T3000M00000P000=L37@3I0aX0E`0002@0003I0`005@000=X3000J0000
0P000=T36P3J0aD0E`0002@0003J0`004P000=X3000E00000P000=X35@3J0a80E`0002@0003I0`00
3P000=X3000B00000P000=X34P3I0`h0E`0002@0003G0`002`000=T3000>00000P000=T33P3G0`/0
E`0002@0003D0`002P000=L3000;00000P000=L32`3D0`X0E`0002@0003C0`002P000=@3000:0000
0P000=@32P3C0`X0E`0002@0003A0`002P000=<3000;00000P000=<32P3A0`/0E`0002@0003@0`00
2`000=43000<00000P000=432`3@0``0E`0002@0003?0`0030000=03000>00000P000=03303?0`h0
E`0002@0003>0`003P000<l3000B00000P000<l33P3>0a80E`0002@0003>0`004P000<h3000E0000
0P000<h34P3>0aD0E`0002@0003>0`005@000<l3000J00000P000<h35@3?0aX0E`0002@0003?0`00
6P000=03000L00000P000<l36P3@0a`0E`0002@0003@0`0070000=43000M00000P000=03703A0ad0
E`0002@0003A0`007@000=<3000M00000P000=437@3C0ad0E`0002@0003D0`007@000=H3000M0000
0P000=@37@3F0ad0E`0002@0003F0`0070000=L3000M00000P000=H37@3G0a`0E`0002@0003G0`00
6P000=P3000L00000P000=L3703H0aX0E`0002@0003H0`005@000=T3000J00000P000=P36P3I0aD0
E`0002@0003I0`004P000=T3000E00000P000=T35@3I0a80E`0002@0003H0`003P000=T3000B0000
0P000=T34P3H0`h0E`0002@0003G0`0030000=P3000>00000P000=P33P3G0``0E`0002@0003F0`00
2`000=L3000<00000P000=L3303F0`/0E`0002@0003D0`002P000=H3000;00000P000=H32`3D0`X0
E`0002@0003Q0`002P000>@3000;00000P000>@32P3Q0`/0E`0002@0003O0`002`000>43000>0000
0P000>432`3O0`h0E`0002@0003N0`003P000=l3000B00000P000=l33P3N0a80E`0002@0003N0`00
4P000=h3000E00000P000=h34P3N0aD0E`0002@0003N0`005@000=l3000J00000P000=h35@3O0aX0
E`0002@0003O0`006P000>43000M00000P000=l36P3Q0ad0E`0002@0003Q0`007@000>@3000M0000
0P000>437@3T0ad0E`0002@0003T0`007@000>D3000M00000P000>@37@3U0ad0E`0002@0003U0`00
7@000>P3000M00000P000>D37@3X0ad0E`0002@0003X0`006P000>X3000M00000P000>P37@3Z0aX0
E`0002@0003Z0`005@000>/3000J00000P000>X36P3[0aD0E`0002@0003[0`004P000>/3000E0000
0P000>/35@3[0a80E`0002@0003Z0`003P000>/3000B00000P000>/34P3Z0`h0E`0002@0003X0`00
2`000>X3000>00000P000>X33P3X0`/0E`0002@0003U0`002P000>P3000;00000P000>P32`3U0`X0
E`0002@0003T0`002P000>D3000:00000P000>D32P3T0`X0E`0002@0003R0`002P000>@3000;0000
0P000>@32P3R0`/0E`0002@0003Q0`002`000>83000<00000P000>832`3Q0``0E`0002@0003P0`00
30000>43000>00000P000>43303P0`h0E`0002@0003O0`003P000>03000B00000P000>033P3O0a80
E`0002@0003O0`004P000=l3000E00000P000=l34P3O0aD0E`0002@0003O0`005@000>03000J0000
0P000=l35@3P0aX0E`0002@0003P0`006P000>43000L00000P000>036P3Q0a`0E`0002@0003Q0`00
70000>83000M00000P000>43703R0ad0E`0002@0003R0`007@000>@3000M00000P000>837@3T0ad0
E`0002@0003U0`007@000>L3000M00000P000>D37@3W0ad0E`0002@0003W0`0070000>P3000M0000
0P000>L37@3X0a`0E`0002@0003X0`006P000>T3000L00000P000>P3703Y0aX0E`0002@0003Y0`00
5@000>X3000J00000P000>T36P3Z0aD0E`0002@0003Z0`004P000>X3000E00000P000>X35@3Z0a80
E`0002@0003Y0`003P000>X3000B00000P000>X34P3Y0`h0E`0002@0003X0`0030000>T3000>0000
0P000>T33P3X0``0E`0002@0003W0`002`000>P3000<00000P000>P3303W0`/0E`0002@0003U0`00
2P000>L3000;00000P000>L32`3U0`X0E`0002@0003c0`003@000?D3000>00000P000?<33P3e0`d0
E`0002@0003e0`002P000?L3000=00000P000?D33@3g0`X0E`0002@0003g0`002P000?L3000M0000
0P000?L32P3g0ad0E`0002@0003f0`002`000?H3000M00000P000?H32`3f0ad0E`0002@0003c0`00
7@000?/3000M00000P000?<37@3k0ad0E`0002@0003:0000O0<00<l0001l0`000P000<l0O0?:07`3
E`0002@0003:0000iP400<l0003V0@000P000<l0iP7:0>H1E`0002@0003:0000D0000<l0001@0000
0P000<l0D03:0500E`0002@0000@00005@8002<0000E0P000P0001005@8S01D2E`0002@0000@0000
508002<0000D0P000P000100508S01@2E`0002@0000F00003`8001d0000?0P000P0001H03`8M00l2
E`0002@0000@00002@800100000H0P000P000100608@00T2E`0002@0000@00002@8001H000090P00
0P0001002@8F00T2E`0002@0000@00002@8001H0000:0P000P0001H02@8@00X2E`0002@0000I0000
3`8001T0000D0P000P0001T0508I00l2E`0002@0000S00002@8002<0000H0P000P0002<0608S00T2
E`0002@0000N00002@8002<000090P000P0002<02@8N00T2E`0002@0000N00002@8002<0000:0P00
0P0001h02@8S00X2E`0002@0000=0000o`4000d000020P000P0000d0o`4=0082E`0002@0000=0000
0P8002T000020P000P0000d00P8Y0082E`0002@0000Y0000o`4002T000020P000P0002T00P8Y0?l1
E`0002@0000=00000@8002T000010P000P0000d00@8Y0042E`0002@0000G0000k04002T0003/0@00
0P0001L0k04Y0>`1E`0002@0000G0000j`4002T0003[0@000P0001L0j`4Y0>/1E`0002@0000G0000
k04001T0003^0@000P0001T0k04G0>h1E`0002@0000G0000kP4001L0003`0@000P0001L0kP4G0?01
E`0002@0000G0000l04001L0003b0@000P0001L0l04G0?81E`0002@0000G0000lP4001L0003d0@00
0P0001L0lP4G0?@1E`0002@0000G0000m04001T0003f0@000P0001L0m04I0?H1E`0002@0000I0000
mP4001`0003g0@000P0001T0mP4L0?L1E`0002@0000L0000m`4001h0003g0@000P0001`0m`4N0?L1
E`0002@0000N0000mP400200003g0@000P0001h0m`4P0?H1E`0002@0000P0000m0400280003f0@00
0P000200mP4R0?@1E`0002@0000R0000lP4002<0003d0@000P000280m04S0?81E`0002@0000S0000
l04002<0003b0@000P0002<0lP4S0?01E`0002@0000R0000kP4002<0003`0@000P0002<0l04R0>h1
E`0002@0000P0000k0400280003^0@000P000280kP4P0>`1E`0002@0000G0000lP4001L0003c0@00
0P0001L0lP4G0?<1E`0002@0000G0000l`4001T0003e0@000P0001L0l`4I0?D1E`0002@0000I0000
m@4001`0003f0@000P0001T0m@4L0?H1E`0002@0000L0000mP4001h0003f0@000P0001`0mP4N0?H1
E`0002@0000N0000m@400200003f0@000P0001h0mP4P0?D1E`0002@0000P0000l`400280003e0@00
0P000200m@4R0?<1E`0002@0000R0000lP4002<0003c0@000P000280l`4S0?81E`0002@0000Y0000
j04002T0003_0@000P0002T0k`4Y0>P1E`0002@0000=0000g@4000h0003O0@000P0000d0g@4>0=l1
E`0002@0000>0000g`400140003P0@000P0000h0g`4A0>01E`0002@0000A0000h04001D0003R0@00
0P000140h04E0>81E`0002@0000E0000hP4001T0003S0@000P0001D0hP4I0><1E`0002@0000I0000
h`4001d0003S0@000P0001T0h`4M0><1E`0002@0000M0000hP400240003S0@000P0001d0h`4Q0>81
E`0002@0000Q0000h04002D0003R0@000P000240hP4U0>01E`0002@0000U0000g`4002P0003P0@00
0P0002D0h04X0=l1E`0002@0000X0000g@4002T0003O0@000P0002P0g`4Y0=d1E`0002@0000>0000
g`400180003P0@000P0000h0g`4B0>01E`0002@0000B0000h04001D0003Q0@000P000180h04E0>41
E`0002@0000E0000h@4001T0003R0@000P0001D0h@4I0>81E`0002@0000I0000hP4001d0003R0@00
0P0001T0hP4M0>81E`0002@0000M0000h@400240003R0@000P0001d0hP4Q0>41E`0002@0000Q0000
h04002@0003Q0@000P000240h@4T0>01E`0002@0000T0000g`4002P0003P0@000P0002@0h04X0=l1
E`0002@0000@0000e@4001l0003E0@000P000100e@4O0=D1E`0002@0000O0000e0400280003E0@00
0P0001l0e@4R0=@1E`0002@0000R0000d`4002<0003D0@000P000280e04S0=<1E`0002@0000S0000
d@4002<0003C0@000P0002<0d`4S0=41E`0002@0000R0000c`4002<0003A0@000P0002<0d@4R0<l1
E`0002@0000P0000cP400280003?0@000P000280c`4P0<h1E`0002@0000@0000e04001l0003D0@00
0P000100e04O0=@1E`0002@0000O0000d`400280003D0@000P0001l0e04R0=<1E`0002@0000R0000
d`4002<0003C0@000P000280d`4S0=<1E`0002@0000G0000d@4001L0003H0@000P0001L0f04G0=41
E`0002@0000=0000a`4000h000390@000P0000d0b@4>0<L1E`0002@0000>0000a@40014000370@00
0P0000h0a`4A0<D1E`0002@0000A0000a04001D000350@000P000140a@4E0<@1E`0002@0000E0000
``4001T000340@000P0001D0a04I0<<1E`0002@0000I0000``4001d000330@000P0001T0``4M0<<1
E`0002@0000M0000``40024000340@000P0001d0``4Q0<@1E`0002@0000Q0000a04002D000350@00
0P000240a04U0<D1E`0002@0000U0000a@4002P000370@000P0002D0a@4X0<L1E`0002@0000X0000
a`4002T000390@000P0002P0a`4Y0<T1E`0002@0000>0000a@40018000370@000P0000h0a`4B0<D1
E`0002@0000B0000a@4001D000350@000P000180a@4E0<D1E`0002@0000E0000a04001T000350@00
0P0001D0a@4I0<@1E`0002@0000I0000a04001d000340@000P0001T0a04M0<@1E`0002@0000M0000
a040024000350@000P0001d0a04Q0<D1E`0002@0000Q0000a@4002@000350@000P000240a@4T0<D1
E`0002@0000T0000a@4002P000370@000P0002@0a@4X0<L1E`0002@0000=0000]`4000d0002j0@00
0P0000d0^P4=0;L1E`0002@0000=0000]`4002T0002g0@000P0000d0]`4Y0;L1E`0002@0000Y0000
]`4002T0002j0@000P0002T0]`4Y0;X1E`0002@0000=0000^04002T0002h0@000P0000d0^04Y0;P1
E`0002@000160000OP<005H0001n0`000P0004H0OP=F07h3E`0002@0001O0000MP<00640001g0`00
0P0005l0M`=Q07H3E`0002@0001Q0000L`<006<0001f0`000P000640MP=S07<3E`0002@0001S0000
L`<006<000260`000P0006<0L`=S08H3E`0002@0001R0000M0<0068000260`000P000680M0=R08H3
E`0002@0001O0000QP<006L000260`000P0005l0QP=W08H3E`0002@0001n0000K`<007l0001a0`00
0P0007l0K`=n0743E`0002@0001l0000L@<007h0001d0`000P0007h0L@=l07@3E`0002@0001j0000
M0<007`0001h0`000P0007`0M0=j07P3E`0002@0001i0000N0<007X0001l0`000P0007X0N0=i07`3
E`0002@0001i0000O0<007T000200`000P0007T0O0=i0803E`0002@0001i0000P0<007X000250`00
0P0007T0P0=j08D3E`0002@0001j0000Q@<007`000280`000P0007X0Q@=l08P3E`0002@0001l0000
R0<007h0002;0`000P0007`0R0=n08/3E`0002@0001n0000R`<007l0002=0`000P0007h0R`=o08d3
E`0002@0001l0000L@<007h0001e0`000P0007h0L@=l07D3E`0002@0001k0000M@<007`0001h0`00
0P0007`0M@=k07P3E`0002@0001j0000N0<007/0001l0`000P0007/0N0=j07`3E`0002@0001j0000
O0<007X000200`000P0007X0O0=j0803E`0002@0001j0000P0<007/000250`000P0007X0P0=k08D3
E`0002@0001k0000Q@<007`000270`000P0007/0Q@=l08L3E`0002@0001l0000Q`<007h0002;0`00
0P0007`0Q`=n08/3E`0002@0002A0000L`<00980001f0`000P000940MP>B07<3E`0002@0002B0000
L`<00980001h0`000P000980L`>B07P3E`0002@0002A0000MP<00980001h0`000P000980N0>A07H3
E`0002@0002?0000M0<00940001f0`000P000940MP>?07@3E`0002@0002<0000L`<008l0001d0`00
0P0008l0M0><07<3E`0002@0002:0000L`<008`0001c0`000P0008`0L`>:07<3E`0002@000270000
L`<008X0001d0`000P0008X0L`>707@3E`0002@000250000M0<008L0001f0`000P0008L0M0>507H3
E`0002@000250000MP<008D0001h0`000P0008D0MP>507P3E`0002@000250000N0<008H0001i0`00
0P0008D0N0>607T3E`0002@000260000N@<008L0001j0`000P0008H0N@>707X3E`0002@000270000
NP<008T0001k0`000P0008L0NP>907/3E`0002@000290000N`<008h0001m0`000P0008T0N`>>07d3
E`0002@0002>0000O@<00900001n0`000P0008h0O@>@07h3E`0002@0002@0000OP<0098000200`00
0P000900OP>B0803E`0002@000250000N0<008L0001i0`000P0008D0N0>707T3E`0002@000270000
N@<008T0001j0`000P0008L0N@>907X3E`0002@000290000NP<008h0001l0`000P0008T0NP>>07`3
E`0002@0002>0000O0<00900001m0`000P0008h0O0>@07d3E`0002@0002@0000O@<00940001n0`00
0P000900O@>A07h3E`0002@0002A0000OP<0098000200`000P000940OP>B0803E`0002@0002B0000
P0<0098000240`000P000980P0>B08@3E`0002@0002@0000Q0<0098000260`000P000980Q0>@08H3
E`0002@0002=0000QP<0090000260`000P000900QP>=08H3E`0002@0002;0000QP<008d000260`00
0P0008d0QP>;08H3E`0002@000280000QP<008/000260`000P0008/0QP>808H3E`0002@000260000
Q0<008P000260`000P0008P0QP>608@3E`0002@000250000P@<008H000240`000P0008H0Q0>50843
E`0002@000250000P@<008D000260`000P0008D0P@>508H3E`0002@000250000Q0<008H000260`00
0P0008D0QP>608@3E`0002@0002H0000O`<00:<0001o0`000P0009P0O`>S07l3E`0002@0002S0000
O@<00:<0001o0`000P000:<0O`>S07d3E`0002@0002R0000N`<00:<0001m0`000P000:<0O@>R07/3
E`0002@0002Q0000NP<00:80001k0`000P000:80N`>Q07X3E`0002@0002O0000N@<00:40001j0`00
0P000:40NP>O07T3E`0002@0002M0000N@<009l0001i0`000P0009l0N@>M07T3E`0002@0002J0000
N@<009d0001j0`000P0009d0N@>J07X3E`0002@0002H0000NP<009X0001l0`000P0009X0NP>H07`3
E`0002@0002G0000O0<009P0001o0`000P0009P0O0>G07l3E`0002@0002G0000O`<009L000210`00
0P0009L0O`>G0843E`0002@0002G0000P@<009P000240`000P0009L0P@>H08@3E`0002@0002H0000
Q0<009X000260`000P0009P0Q0>J08H3E`0002@0002J0000QP<009d000260`000P0009X0QP>M08H3
E`0002@0002M0000QP<009h000260`000P0009d0QP>N08H3E`0002@0002N0000QP<00:4000260`00
0P0009h0QP>Q08H3E`0002@0002Q0000Q0<00:<000260`000P000:40QP>S08@3E`0002@0002R0000
O0<00:80001o0`000P000:80O`>R07`3E`0002@0002Q0000NP<00:80001l0`000P000:80O0>Q07X3
E`0002@0002K0000N@<009d0001j0`000P0009d0N@>K07X3E`0002@0002I0000NP<009/0001l0`00
0P0009/0NP>I07`3E`0002@0002H0000O0<009T0001o0`000P0009T0O0>H07l3E`0002@0002H0000
O`<009P000210`000P0009P0O`>H0843E`0002@0002H0000P@<009T000240`000P0009P0P@>I08@3
E`0002@0002I0000Q0<009/000260`000P0009T0Q0>K08H3E`0002@0002K0000QP<009d000260`00
0P0009/0QP>M08H3E`0002@0002Y0000L`<00:T000260`000P000:T0L`>Y08H3E`0002@0002Z0000
L`<00:X000260`000P000:X0L`>Z08H3E`0002@0002V0000L`<00:X0001c0`000P000:H0L`>Z07<3
E`0002@0002V0000QP<00:d000260`000P000:H0QP>]08H3E`0002@0002c0000L`<00;<000260`00
0P000;<0L`>c08H3E`0002@0002d0000L`<00;@000260`000P000;@0L`>d08H3E`0002@0002`0000
L`<00;@0001c0`000P000;00L`>d07<3E`0002@0002`0000QP<00;L000260`000P000;00QP>g08H3
E`0002@0002k0000K`<00;d0001a0`000P000;/0K`>m0743E`0002@0002m0000L@<00;l0001d0`00
0P000;d0L@>o07@3E`0002@0002o0000M0<00<40001h0`000P000;l0M0?107P3E`0002@000310000
N0<00<40001l0`000P000<40N0?107`3E`0002@000310000O0<00<4000200`000P000<40O0?10803
E`0002@000310000P0<00<4000250`000P000<40P0?108D3E`0002@0002o0000Q@<00<4000280`00
0P000<40Q@>o08P3E`0002@0002m0000R0<00;l0002;0`000P000;l0R0>m08/3E`0002@0002k0000
R`<00;d0002=0`000P000;d0R`>k08d3E`0002@0002m0000L@<00;l0001e0`000P000;d0L@>o07D3
E`0002@0002o0000M@<00<00001h0`000P000;l0M@?007P3E`0002@000300000N0<00<40001l0`00
0P000<00N0?107`3E`0002@000310000O0<00<4000200`000P000<40O0?10803E`0002@000300000
P0<00<4000250`000P000<40P0?008D3E`0002@0002o0000Q@<00<0000270`000P000<00Q@>o08L3
E`0002@0002m0000Q`<00;l0002;0`000P000;l0Q`>m08/3E`0002@0001<0000B@0004`0001I0000
0P0004`0B@1<05T0E`0002@000140000D@0005@0001A00000P0004@0D@1D0540E`0002@0001N0000
B@000600001:00000P0005h0BP1P04T0E`0002@0001P0000AP000680001900000P000600B@1R04H0
E`0002@0001R0000AP000680001I00000P000680AP1R05T0E`0002@0001Q0000A`000640001I0000
0P000640A`1Q05T0E`0002@0001N0000F@0006H0001I00000P0005h0F@1V05T0E`0002@0001l0000
@P0007d0001400000P0007d0@P1l04@0E`0002@0001j0000A00007`0001700000P0007`0A01j04L0
E`0002@0001h0000A`0007X0001;00000P0007X0A`1h04/0E`0002@0001g0000B`0007P0001?0000
0P0007P0B`1g04l0E`0002@0001g0000C`0007L0001C00000P0007L0C`1g05<0E`0002@0001g0000
D`0007P0001H00000P0007L0D`1h05P0E`0002@0001h0000F00007X0001K00000P0007P0F01j05/0
E`0002@0001j0000F`0007`0001N00000P0007X0F`1l05h0E`0002@0001l0000GP0007d0001P0000
0P0007`0GP1m0600E`0002@0001j0000A00007`0001800000P0007`0A01j04P0E`0002@0001i0000
B00007X0001;00000P0007X0B01i04/0E`0002@0001h0000B`0007T0001?00000P0007T0B`1h04l0
E`0002@0001h0000C`0007P0001C00000P0007P0C`1h05<0E`0002@0001h0000D`0007T0001H0000
0P0007P0D`1i05P0E`0002@0001i0000F00007X0001J00000P0007T0F01j05X0E`0002@0001j0000
FP0007`0001N00000P0007X0FP1l05h0E`0002@000250000AP0008D0001I00000P0008D0AP2505T0
E`0002@000260000AP0008H0001I00000P0008H0AP2605T0E`0002@000220000AP0008d000160000
0P000880AP2=04H0E`0002@0002=0000AP000900001700000P0008d0AP2@04L0E`0002@0002@0000
A`000940001800000P000900A`2A04P0E`0002@0002A0000B0000940001:00000P000940B02A04X0
E`0002@0002A0000BP000940001;00000P000940BP2A04/0E`0002@0002A0000B`000940001=0000
0P000940B`2A04d0E`0002@0002@0000C@000940001>00000P000940C@2@04h0E`0002@0002=0000
CP000900001?00000P000900CP2=04l0E`0002@0002=0000AP0008l0001700000P0008d0AP2?04L0
E`0002@0002?0000A`000900001800000P0008l0A`2@04P0E`0002@0002@0000B0000940001:0000
0P000900B02A04X0E`0002@0002A0000BP000940001;00000P000940BP2A04/0E`0002@0002@0000
B`000940001=00000P000940B`2@04d0E`0002@0002?0000C@000900001>00000P000900C@2?04h0
E`0002@0002=0000CP0008l0001?00000P0008l0CP2=04l0E`0002@000260000C`0008d0001?0000
0P0008H0C`2=04l0E`0002@0002=0000C`000900001@00000P0008d0C`2@0500E`0002@0002@0000
D0000940001A00000P000900D02A0540E`0002@0002A0000D@000940001C00000P000940D@2A05<0
E`0002@0002A0000D`000940001F00000P000940D`2A05H0E`0002@0002A0000EP000940001H0000
0P000940EP2A05P0E`0002@0002@0000F0000940001I00000P000940F02@05T0E`0002@0002=0000
F@000900001I00000P000900F@2=05T0E`0002@000220000F@0008d0001I00000P0008d0F@2205T0
E`0002@0002=0000C`0008l0001@00000P0008d0C`2?0500E`0002@0002?0000D0000900001A0000
0P0008l0D02@0540E`0002@0002@0000D@000940001C00000P000900D@2A05<0E`0002@0002A0000
D`000940001F00000P000940D`2A05H0E`0002@0002@0000EP000940001H00000P000940EP2@05P0
E`0002@0002?0000F0000900001I00000P000900F02?05T0E`0002@0002=0000F@0008l0001I0000
0P0008l0F@2=05T0E`0002@0002I0000C00009T0001G00000P0009T0C02I05L0E`0002@0002I0000
E`0009X0001I00000P0009T0E`2J05T0E`0002@0002J0000F@0009d0001I00000P0009X0F@2M05T0
E`0002@0002M0000F@0009h0001I00000P0009d0F@2N05T0E`0002@0002N0000F@000:40001I0000
0P0009h0F@2Q05T0E`0002@0002Q0000E`000:<0001I00000P000:40F@2S05L0E`0002@0002J0000
C00009X0001G00000P0009X0C02J05L0E`0002@0002J0000E`0009/0001I00000P0009X0E`2K05T0
E`0002@0002K0000F@0009d0001I00000P0009/0F@2M05T0E`0002@0002S0000C0000:<0001I0000
0P000:<0C02S05T0E`0002@0002T0000C0000:@0001I00000P000:@0C02T05T0E`0002@0002F0000
C00009X0001<00000P0009H0C02J04`0E`0002@0002P0000C0000:@0001<00000P000:00C02T04`0
E`0002@0002S0000F@000:H0001I00000P000:<0F@2V05T0E`0002@0002[0000C0000;00001I0000
0P000:/0C02`05T0E`0002@0002/0000C0000;00001H00000P000:`0C02`05P0E`0002@0002`0000
C0000;H0001I00000P000;H0C02`05T0E`0002@0002_0000F@000;00001M00000P000;00F@2_05d0
E`0002@0002]0000G@000:l0001O00000P000:l0G@2]05l0E`0002@0002[0000G`000:d0001P0000
0P000:d0G`2[0600E`0002@0002Z0000H0000:/0001P00000P000:/0H02Z0600E`0002@0002Y0000
G`000:X0001P00000P000:X0H02Y05l0E`0002@0002Y0000GP000:X0001O00000P000:T0G`2Z05h0
E`0002@0002Z0000GP000:/0001O00000P000:X0GP2[05l0E`0002@0002Y0000C0000:l0001<0000
0P000:T0C02_04`0E`0002@0002b0000C0000;P0001<00000P000;80C02h04`0E`0002@0002k0000
@P000;d0001400000P000;/0@P2m04@0E`0002@0002m0000A0000;l0001700000P000;d0A02o04L0
E`0002@0002o0000A`000<40001;00000P000;l0A`3104/0E`0002@000310000B`000<40001?0000
0P000<40B`3104l0E`0002@000310000C`000<40001C00000P000<40C`3105<0E`0002@000310000
D`000<40001H00000P000<40D`3105P0E`0002@0002o0000F0000<40001K00000P000<40F02o05/0
E`0002@0002m0000F`000;l0001N00000P000;l0F`2m05h0E`0002@0002k0000GP000;d0001P0000
0P000;d0GP2k0600E`0002@0002m0000A0000;l0001800000P000;d0A02o04P0E`0002@0002o0000
B0000<00001;00000P000;l0B03004/0E`0002@000300000B`000<40001?00000P000<00B`3104l0
E`0002@000310000C`000<40001C00000P000<40C`3105<0E`0002@000300000D`000<40001H0000
0P000<40D`3005P0E`0002@0002o0000F0000<00001J00000P000<00F02o05X0E`0002@0002m0000
FP000;l0001N00000P000;l0FP2m05h0E`0002@0003E0000Q@<00=D0002=0`000P000=D0Q@?E08d3
E`0002@0000;0@00Q@<000/1002=0`000P0000/1Q@<;0Hd3E`0002@000100@00Q@<00401002=0`00
0P000401Q@=00Hd3E`0002@0001f0@00Q@<007H1002=0`000P0007H1Q@=f0Hd3E`0002@0002/0@00
Q@<00:`1002=0`000P000:`1Q@>/0Hd3E`0002@0003R0@00Q@<00>81002=0`000P000>81Q@?R0Hd3
E`0002@0000H0P00Q@<001P2002=0`000P0001P2Q@<H0Xd3E`0002@0001>0P00Q@<004h2002=0`00
0P0004h2Q@=>0Xd3E`0002@000240P00Q@<008@2002=0`000P0008@2Q@>40Xd3E`0002@0002j0P00
Q@<00;X2002=0`000P000;X2Q@>j0Xd3E`0002@0003`0P00Q@<00?02002=0`000P000?02Q@?`0Xd3
E`0002@0000V0`00Q@<002H3002=0`000P0002H3Q@<V0hd3E`0002@0001L0`00Q@<005`3002=0`00
0P0005`3Q@=L0hd3E`0002@0002B0`00Q@<00983002=0`000P000983Q@>B0hd3E`0002@000380`00
Q@<00<P3002=0`000P000<P3Q@?80hd3E`0002@0003n0`00Q@<00?h3002=0`000P000?h3Q@?n0hd3
E`0002@0000d1000Q@<003@4002=0`000P0003@4Q@<d18d3E`0002@0001Y1000Q@<006T4002=0`00
0P0006T4Q@=Y18d3E`0002@0002O1000Q@<009l4002=0`000P0009l4Q@>O18d3E`0002@0003E1000
Q@<00=D4002=0`000P000=D4Q@?E18d3E`0002@0002h0P00b`<00;P2003N0`000P000;P2b`>h0]h3
E`0002@0002h0P00b`<00;P2003N0`000P000;P2b`>h0]h3E`0002@0002a0P00b`<00;82003@0`00
0P000;82b`>a0]03E`0002@0002a0P00b`<00;42003@0`000P000;42d0>a0//3E`0002@0002a0P00
b`<00;l2003;0`000P000;42b`>o0//3E`0002@0002o0P00b`<00;l2003@0`000P000;l2b`>o0]03
E`0002@0002n0P00b`<00;l2003@0`000P000;l2d0>n0//3E`0002@0002e0P00gP<00;/2003N0`00
0P000;D2gP>k0]h3E`0002@000340P00b`<00<D2003<0`000P000<D2b`?40/`3E`0002@000340P00
c0<00<D2003=0`000P000<@2c0?50/d3E`0002@000350P00c0<00<H2003=0`000P000<D2c@?60/`3
E`0002@000350P00b`<00<H2003<0`000P000<H2c0?50//3E`0002@000350P00d@<00<D2003N0`00
0P000<D2d@?50]h3E`0002@000360P00d@<00<H2003N0`000P000<H2d@?60]h3E`0002@000320P00
d@<00<H2003A0`000P000<82d@?60]43E`0002@000320P00gP<00<T2003N0`000P000<82gP?90]h3
E`0002@0003?0P00d@<00<l2003N0`000P000<l2d@??0]h3E`0002@0003@0P00d@<00=02003N0`00
0P000=02d@?@0]h3E`0002@0003@0P00dP<00=82003D0`000P000=02e0?B0]83E`0002@0003B0P00
d@<00=@2003B0`000P000=82dP?D0]43E`0002@0003D0P00d@<00=H2003A0`000P000=@2d@?F0]43
E`0002@0003F0P00d@<00=T2003B0`000P000=H2d@?I0]83E`0002@0003I0P00dP<00=X2003D0`00
0P000=T2dP?J0]@3E`0002@0003J0P00e0<00=X2003N0`000P000=X2e0?J0]h3E`0002@0003F0P00
d@<00=P2003B0`000P000=H2d@?H0]83E`0002@0003H0P00dP<00=T2003D0`000P000=P2dP?I0]@3
E`0002@0003I0P00e0<00=T2003N0`000P000=T2e0?I0]h3E`0002@0003J0P00dP<00=/2003D0`00
0P000=X2e0?K0]83E`0002@0003K0P00d@<00=h2003B0`000P000=/2dP?N0]43E`0002@0003N0P00
d@<00>02003A0`000P000=h2d@?P0]43E`0002@0003P0P00d@<00><2003B0`000P000>02d@?S0]83
E`0002@0003S0P00dP<00>@2003D0`000P000><2dP?T0]@3E`0002@0003T0P00e0<00>@2003N0`00
0P000>@2e0?T0]h3E`0002@0003P0P00d@<00>82003B0`000P000>02d@?R0]83E`0002@0003R0P00
dP<00><2003D0`000P000>82dP?S0]@3E`0002@0003S0P00e0<00><2003N0`000P000><2e0?S0]h3
E`0002@0003<0P00d@<00=02003A0`000P000<`2d@?@0]43E`0002@0003<0P00gP<00=<2003N0`00
0P000<`2gP?C0]h3E`0002@0003F0P00gP<00=`2003N0`000P000=H2gP?L0]h3E`0002@0003P0P00
gP<00>H2003N0`000P000>02gP?V0]h3E`0002@0003/0P00e`<00?L2003G0`000P000>`2e`?g0]L3
E`0002@0003g0P00e@<00?L2003G0`000P000?L2e`?g0]D3E`0002@0003f0P00d`<00?L2003E0`00
0P000?L2e@?f0]<3E`0002@0003e0P00dP<00?H2003C0`000P000?H2d`?e0]83E`0002@0003c0P00
d@<00?D2003B0`000P000?D2dP?c0]43E`0002@0003a0P00d@<00?<2003A0`000P000?<2d@?a0]43
E`0002@0003^0P00d@<00?42003B0`000P000?42d@?^0]83E`0002@0003/0P00dP<00>h2003D0`00
0P000>h2dP?/0]@3E`0002@0003[0P00e0<00>`2003G0`000P000>`2e0?[0]L3E`0002@0003[0P00
e`<00>/2003I0`000P000>/2e`?[0]T3E`0002@0003[0P00f@<00>`2003L0`000P000>/2f@?/0]`3
E`0002@0003/0P00g0<00>h2003N0`000P000>`2g0?^0]h3E`0002@0003^0P00gP<00?42003N0`00
0P000>h2gP?a0]h3E`0002@0003a0P00gP<00?82003N0`000P000?42gP?b0]h3E`0002@0003b0P00
gP<00?D2003N0`000P000?82gP?e0]h3E`0002@0003e0P00g0<00?L2003N0`000P000?D2gP?g0]`3
E`0002@0003f0P00e0<00?H2003G0`000P000?H2e`?f0]@3E`0002@0003e0P00dP<00?H2003D0`00
0P000?H2e0?e0]83E`0002@0003_0P00d@<00?42003B0`000P000?42d@?_0]83E`0002@0003]0P00
dP<00>l2003D0`000P000>l2dP?]0]@3E`0002@0003/0P00e0<00>d2003G0`000P000>d2e0?/0]L3
E`0002@0003/0P00e`<00>`2003I0`000P000>`2e`?/0]T3E`0002@0003/0P00f@<00>d2003L0`00
0P000>`2f@?]0]`3E`0002@0003]0P00g0<00>l2003N0`000P000>d2g0?_0]h3E`0002@0003_0P00
gP<00?42003N0`000P000>l2gP?a0]h3E`0002@0003A0000WP<00=<0002O0`000P000=40W`?C09h3
E`0002@0003C0000V`<00=D0002N0`000P000=<0WP?E09/3E`0002@0003E0000V`<00=D0002^0`00
0P000=D0V`?E0:h3E`0002@0003D0000W0<00=@0002^0`000P000=@0W0?D0:h3E`0002@0003A0000
[P<00=T0002^0`000P000=40[P?I0:h3E`0002@000050@00W`<000H1002P0`000P0000D1W`<60J03
E`0002@000050@00X0<000H1002P0`000P0000H1X0<50J03E`0002@000040@00X0<000D1002P0`00
0P0000D1X0<40J03E`0002@000040@00W`<000@1002P0`000P0000@1X0<40Il3E`0002@000040@00
W@<000D1002O0`000P0000@1W`<50Id3E`0002@000050@00W0<000H1002M0`000P0000D1W@<60I`3
E`0002@000060@00V`<000T1002L0`000P0000H1W0<90I/3E`0002@000090@00V`<000`1002K0`00
0P0000T1V`<<0I/3E`0002@0000<0@00V`<000l1002L0`000P0000`1V`<?0I`3E`0002@0000?0@00
W0<00101002M0`000P0000l1W0<@0Id3E`0002@0000@0@00W@<00141002O0`000P000101W@<A0Il3
E`0002@0000A0@00W`<00141002P0`000P000141W`<A0J03E`0002@0000@0@00X0<00141002R0`00
0P000141X0<@0J83E`0002@0000=0@00XP<00101002T0`000P000101XP<=0J@3E`0002@000090@00
Y0<000d1002V0`000P0000d1Y0<90JH3E`0002@000070@00YP<000T1002W0`000P0000T1YP<70JL3
E`0002@000050@00Y`<000L1002Y0`000P0000L1Y`<50JT3E`0002@000040@00Z@<000D1002/0`00
0P0000D1Z@<40J`3E`0002@000040@00[0<000@1002^0`000P0000@1[0<40Jh3E`0002@0000<0@00
V`<000h1002L0`000P0000`1V`<>0I`3E`0002@0000>0@00W0<000l1002M0`000P0000h1W0<?0Id3
E`0002@0000?0@00W@<00101002O0`000P0000l1W@<@0Il3E`0002@0000@0@00W`<00101002P0`00
0P000101W`<@0J03E`0002@0000?0@00X0<00101002R0`000P000101X0<?0J83E`0002@0000<0@00
XP<000l1002T0`000P0000l1XP<<0J@3E`0002@000090@00Y0<000`1002V0`000P0000`1Y0<90JH3
E`0002@000040@00[0<000D1002]0`000P0000@1[@<50J`3E`0002@000050@00[0<000L1002/0`00
0P0000D1[0<70J`3E`0002@000070@00[0<000/1002^0`000P0000L1[0<;0Jh3E`0002@0000;0@00
[P<000h1002^0`000P0000/1[P<>0Jh3E`0002@0000>0@00[@<00101002^0`000P0000h1[P<@0Jd3
E`0002@0000@0@00[0<00141002]0`000P000101[@<A0J`3E`0002@000070@00[0<000/1002^0`00
0P0000L1[0<;0Jh3E`0002@0000;0@00[P<000l1002^0`000P0000/1[P<?0Jh3E`0002@0000?0@00
[P<00101002^0`000P0000l1[P<@0Jh3E`0002@0000@0@00[0<00141002^0`000P000101[P<A0J`3
E`0002@0000A0@00ZP<00141002/0`000P000141[0<A0JX3E`0002@0000k0@00W`<003`1002P0`00
0P0003/1W`<l0J03E`0002@0000k0@00X0<003`1002P0`000P0003`1X0<k0J03E`0002@0000j0@00
X0<003/1002P0`000P0003/1X0<j0J03E`0002@0000j0@00W`<003X1002P0`000P0003X1X0<j0Il3
E`0002@0000j0@00W@<003/1002O0`000P0003X1W`<k0Id3E`0002@0000k0@00W0<003`1002M0`00
0P0003/1W@<l0I`3E`0002@0000l0@00V`<003l1002L0`000P0003`1W0<o0I/3E`0002@0000o0@00
V`<00481002K0`000P0003l1V`=20I/3E`0002@000120@00V`<004D1002L0`000P000481V`=50I`3
E`0002@000150@00W0<004H1002N0`000P0004D1W0=60Ih3E`0002@000160@00WP<004H1002P0`00
0P0004H1WP=60J03E`0002@000150@00X0<004H1002R0`000P0004H1X0=50J83E`0002@000120@00
XP<004D1002S0`000P0004D1XP=20J<3E`0002@000100@00X`<00481002S0`000P000481X`=00J<3
E`0002@000120@00V`<004@1002L0`000P000481V`=40I`3E`0002@000140@00W0<004D1002N0`00
0P0004@1W0=50Ih3E`0002@000150@00WP<004D1002P0`000P0004D1WP=50J03E`0002@000140@00
X0<004D1002R0`000P0004D1X0=40J83E`0002@000120@00XP<004@1002S0`000P0004@1XP=20J<3
E`0002@000120@00X`<004@1002T0`000P000481X`=40J@3E`0002@000140@00Y0<004H1002V0`00
0P0004@1Y0=60JH3E`0002@000160@00YP<004L1002X0`000P0004H1YP=70JP3E`0002@000170@00
Z0<004L1002[0`000P0004L1Z0=70J/3E`0002@000160@00Z`<004L1002]0`000P0004L1Z`=60Jd3
E`0002@000150@00[@<004H1002^0`000P0004H1[@=50Jh3E`0002@000120@00[P<004D1002^0`00
0P0004D1[P=20Jh3E`0002@0000o0@00[P<00481002^0`000P000481[P<o0Jh3E`0002@0000l0@00
[P<003l1002^0`000P0003l1[P<l0Jh3E`0002@0000k0@00[@<003`1002^0`000P0003`1[P<k0Jd3
E`0002@0000j0@00Z`<003/1002]0`000P0003/1[@<j0J/3E`0002@0000j0@00ZP<003X1002[0`00
0P0003X1Z`<j0JX3E`0002@0000j0@00Z@<003/1002Z0`000P0003X1ZP<k0JT3E`0002@0000k0@00
Z@<003`1002Z0`000P0003/1Z@<l0JX3E`0002@0000k0@00ZP<003`1002[0`000P0003`1ZP<k0J/3
E`0002@000150@00Y@<004H1002X0`000P0004D1Y@=60JP3E`0002@000160@00Z0<004H1002[0`00
0P0004H1Z0=60J/3E`0002@000150@00Z`<004H1002]0`000P0004H1Z`=50Jd3E`0002@000140@00
[@<004D1002^0`000P0004D1[@=40Jh3E`0002@000120@00[P<004@1002^0`000P0004@1[P=20Jh3
E`0002@0001h0@00W@<007P1002^0`000P0007P1W@=h0Jh3E`0002@0001i0@00V`<007T1002^0`00
0P0007T1V`=i0Jh3E`0002@0001_0@00V`<007T1002Y0`000P0007T1V`=_0JT3E`0002@0001_0@00
Z@<007h1002Y0`000P0006l1Z@=n0JT3E`0002@0001f0@00[P<007`1002^0`000P0007H1[P=l0Jh3
E`0002@0002U0@00V`<00:L1002T0`000P000:L1V`>U0J@3E`0002@0002U0@00XP<00:L1002T0`00
0P000:D1Y0>W0J83E`0002@0002W0@00X@<00:X1002R0`000P000:L1XP>Z0J43E`0002@0002Z0@00
X@<00:`1002Q0`000P000:X1X@>/0J43E`0002@0002/0@00X@<00:l1002R0`000P000:`1X@>_0J83
E`0002@0002_0@00XP<00;41002T0`000P000:l1XP>a0J@3E`0002@0002a0@00Y0<00;81002W0`00
0P000;41Y0>b0JL3E`0002@0002b0@00Y`<00;81002Y0`000P000;81Y`>b0JT3E`0002@0002a0@00
Z@<00;81002/0`000P000;81Z@>a0J`3E`0002@0002_0@00[0<00;41002^0`000P000;41[0>_0Jh3
E`0002@0002/0@00[P<00:l1002^0`000P000:l1[P>/0Jh3E`0002@0002Z0@00[P<00:`1002^0`00
0P000:`1[P>Z0Jh3E`0002@0002W0@00[P<00:X1002^0`000P000:X1[P>W0Jh3E`0002@0002V0@00
[@<00:L1002^0`000P000:L1[P>V0Jd3E`0002@0002U0@00Z`<00:H1002]0`000P000:H1[@>U0J/3
E`0002@0002U0@00ZP<00:D1002[0`000P000:D1Z`>U0JX3E`0002@0002U0@00Z@<00:H1002Z0`00
0P000:D1ZP>V0JT3E`0002@0002V0@00Z@<00:L1002Z0`000P000:H1Z@>W0JX3E`0002@0002V0@00
ZP<00:L1002[0`000P000:L1ZP>V0J/3E`0002@0002/0@00X@<00:h1002R0`000P000:`1X@>^0J83
E`0002@0002^0@00XP<00;01002T0`000P000:h1XP>`0J@3E`0002@0002`0@00Y0<00;41002W0`00
0P000;01Y0>a0JL3E`0002@0002a0@00Y`<00;41002Y0`000P000;41Y`>a0JT3E`0002@0002`0@00
Z@<00;41002/0`000P000;41Z@>`0J`3E`0002@0002^0@00[0<00;01002^0`000P000;01[0>^0Jh3
E`0002@0002/0@00[P<00:h1002^0`000P000:h1[P>/0Jh3E`0002@0002W0@00V`<00;01002K0`00
0P000:L1V`>`0I/3E`0002@0002W0@00W0<00:/1002L0`000P000:L1W0>[0I`3E`0002@0002[0@00
V`<00;01002L0`000P000:/1W0>`0I/3E`0002@0003U0@00WP<00>H1002O0`000P000>H1WP?U0Il3
E`0002@0003U0@00W`<00>H1002P0`000P000>D1W`?V0J03E`0002@0003V0@00W`<00>L1002P0`00
0P000>H1X0?W0Il3E`0002@0003W0@00WP<00>L1002O0`000P000>L1W`?W0Ih3E`0002@0003V0@00
W0<00>L1002N0`000P000>L1WP?V0I`3E`0002@0003T0@00V`<00>H1002L0`000P000>H1W0?T0I/3
E`0002@0003Q0@00V`<00>@1002K0`000P000>@1V`?Q0I/3E`0002@0003O0@00V`<00>41002L0`00
0P000>41V`?O0I`3E`0002@0003M0@00W0<00=l1002N0`000P000=l1W0?M0Ih3E`0002@0003L0@00
WP<00=d1002P0`000P000=d1WP?L0J03E`0002@0003K0@00X0<00=`1002S0`000P000=`1X0?K0J<3
E`0002@0003K0@00X`<00=/1002Y0`000P000=/1X`?K0JT3E`0002@0003K0@00Z@<00=`1002/0`00
0P000=/1Z@?L0J`3E`0002@0003L0@00[0<00=h1002^0`000P000=`1[0?N0Jh3E`0002@0003N0@00
[P<00>41002^0`000P000=h1[P?Q0Jh3E`0002@0003Q0@00[P<00>81002^0`000P000>41[P?R0Jh3
E`0002@0003R0@00[P<00>D1002^0`000P000>81[P?U0Jh3E`0002@0003U0@00[0<00>L1002^0`00
0P000>D1[P?W0J`3E`0002@0003W0@00Z@<00>P1002/0`000P000>L1[0?X0JT3E`0002@0003X0@00
Z0<00>P1002Y0`000P000>P1Z@?X0JP3E`0002@0003W0@00Y@<00>P1002X0`000P000>P1Z0?W0JD3
E`0002@0003U0@00X`<00>L1002U0`000P000>L1Y@?U0J<3E`0002@0003R0@00XP<00>D1002S0`00
0P000>D1X`?R0J83E`0002@0003Q0@00XP<00>81002R0`000P000>81XP?Q0J83E`0002@0003O0@00
XP<00>41002S0`000P000>41XP?O0J<3E`0002@0003M0@00X`<00=l1002U0`000P000=l1X`?M0JD3
E`0002@0003L0@00Y@<00=d1002X0`000P000=d1Y@?L0JP3E`0002@0003P0@00V`<00>41002L0`00
0P000>41V`?P0I`3E`0002@0003N0@00W0<00>01002N0`000P000>01W0?N0Ih3E`0002@0003M0@00
WP<00=h1002P0`000P000=h1WP?M0J03E`0002@0003L0@00X0<00=d1002S0`000P000=d1X0?L0J<3
E`0002@0003L0@00X`<00=`1002Y0`000P000=`1X`?L0JT3E`0002@0003L0@00Z@<00=d1002/0`00
0P000=`1Z@?M0J`3E`0002@0003M0@00[0<00=l1002^0`000P000=d1[0?O0Jh3E`0002@0003O0@00
[P<00>41002^0`000P000=l1[P?Q0Jh3E`0002@0003R0@00[P<00>@1002^0`000P000>81[P?T0Jh3
E`0002@0003T0@00[0<00>H1002^0`000P000>@1[P?V0J`3E`0002@0003V0@00Z@<00>L1002/0`00
0P000>H1[0?W0JT3E`0002@0003W0@00Z0<00>L1002Y0`000P000>L1Z@?W0JP3E`0002@0003V0@00
Y@<00>L1002X0`000P000>L1Z0?V0JD3E`0002@0003T0@00X`<00>H1002U0`000P000>H1Y@?T0J<3
E`0002@0003R0@00XP<00>@1002S0`000P000>@1X`?R0J83E`0002@0000A0P00V`<00142002P0`00
0P000142V`<A0Z03E`0002@0000A0P00W@<00182002O0`000P000142W`<B0Yd3E`0002@0000B0P00
V`<001@2002M0`000P000182W@<D0Y/3E`0002@0000D0P00V`<001H2002K0`000P0001@2V`<F0Y/3
E`0002@0000F0P00V`<001X2002N0`000P0001H2V`<J0Yh3E`0002@0000J0P00WP<001`2002N0`00
0P0001X2WP<L0Yh3E`0002@0000L0P00W@<001d2002N0`000P0001`2WP<M0Yd3E`0002@0000M0P00
V`<001h2002M0`000P0001d2W@<N0Y/3E`0002@0000B0P00W0<001@2002M0`000P000182W@<D0Y`3
E`0002@0000D0P00W0<001H2002L0`000P0001@2W0<F0Y`3E`0002@0000F0P00W0<001X2002N0`00
0P0001H2W0<J0Yh3E`0002@0000N0P00V`<001h2002N0`000P0001h2V`<N0Yh3E`0002@0000M0P00
WP<001h2002P0`000P0001h2WP<M0Z03E`0002@0000I0P00X0<001d2002U0`000P0001d2X0<I0ZD3
E`0002@0000H0P00Y@<001T2002W0`000P0001T2Y@<H0ZL3E`0002@0000G0P00Y`<001P2002Z0`00
0P0001P2Y`<G0ZX3E`0002@0000G0P00ZP<001L2002^0`000P0001L2ZP<G0Zh3E`0002@0000H0P00
X0<001d2002U0`000P0001d2X0<H0ZD3E`0002@0000G0P00Y@<001P2002W0`000P0001P2Y@<G0ZL3
E`0002@0000G0P00Y`<001L2002Z0`000P0001L2Y`<G0ZX3E`0002@0000G0P00ZP<001L2002^0`00
0P0001L2ZP<G0Zh3E`0002@000190P00V`<004`2002L0`000P0004`2V`=90Y`3E`0002@000180P00
W0<004T2002N0`000P0004T2W0=80Yh3E`0002@000180P00WP<004P2002P0`000P0004P2WP=80Z03
E`0002@000180P00X0<004T2002R0`000P0004P2X0=90Z83E`0002@000190P00XP<004`2002S0`00
0P0004T2XP=<0Z<3E`0002@0001<0P00X`<004l2002S0`000P0004`2X`=?0Z<3E`0002@0001?0P00
XP<00582002S0`000P0004l2X`=B0Z83E`0002@0001B0P00X0<005<2002R0`000P000582XP=C0Z03
E`0002@0001C0P00WP<005<2002P0`000P0005<2X0=C0Yh3E`0002@0001B0P00W0<005<2002N0`00
0P0005<2WP=B0Y`3E`0002@0001?0P00V`<00582002L0`000P000582W0=?0Y/3E`0002@0001<0P00
V`<004l2002K0`000P0004l2V`=<0Y/3E`0002@0001:0P00V`<004`2002L0`000P0004`2V`=:0Y`3
E`0002@000190P00W0<004X2002N0`000P0004X2W0=90Yh3E`0002@000190P00WP<004T2002P0`00
0P0004T2WP=90Z03E`0002@000190P00X0<004X2002R0`000P0004T2X0=:0Z83E`0002@0001:0P00
XP<004`2002S0`000P0004X2XP=<0Z<3E`0002@0001?0P00XP<00542002S0`000P0004l2X`=A0Z83
E`0002@0001A0P00X0<00582002R0`000P000542XP=B0Z03E`0002@0001B0P00WP<00582002P0`00
0P000582X0=B0Yh3E`0002@0001A0P00W0<00582002N0`000P000582WP=A0Y`3E`0002@0001?0P00
V`<00542002L0`000P000542W0=?0Y/3E`0002@000190P00X`<004`2002T0`000P0004`2X`=90Z@3
E`0002@000180P00Y0<004T2002U0`000P0004T2Y0=80ZD3E`0002@000170P00Y@<004P2002W0`00
0P0004P2Y@=70ZL3E`0002@000170P00Y`<004L2002[0`000P0004L2Y`=70Z/3E`0002@000170P00
Z`<004P2002]0`000P0004L2Z`=80Zd3E`0002@000180P00[@<004T2002^0`000P0004P2[@=90Zh3
E`0002@000190P00[P<004`2002^0`000P0004T2[P=<0Zh3E`0002@0001<0P00[P<004l2002^0`00
0P0004`2[P=?0Zh3E`0002@0001?0P00[P<00582002^0`000P0004l2[P=B0Zh3E`0002@0001B0P00
[@<005<2002^0`000P000582[P=C0Zd3E`0002@0001C0P00Z`<005@2002]0`000P0005<2[@=D0Z/3
E`0002@0001D0P00Y`<005@2002[0`000P0005@2Z`=D0ZL3E`0002@0001C0P00Y@<005@2002W0`00
0P0005@2Y`=C0ZD3E`0002@0001B0P00Y0<005<2002U0`000P0005<2Y@=B0Z@3E`0002@0001?0P00
X`<00582002T0`000P000582Y0=?0Z<3E`0002@0001:0P00X`<004`2002T0`000P0004`2X`=:0Z@3
E`0002@000190P00Y0<004X2002U0`000P0004X2Y0=90ZD3E`0002@000180P00Y@<004T2002W0`00
0P0004T2Y@=80ZL3E`0002@000180P00Y`<004P2002[0`000P0004P2Y`=80Z/3E`0002@000180P00
Z`<004T2002]0`000P0004P2Z`=90Zd3E`0002@000190P00[@<004X2002^0`000P0004T2[@=:0Zh3
E`0002@0001:0P00[P<004`2002^0`000P0004X2[P=<0Zh3E`0002@0001?0P00[P<00542002^0`00
0P0004l2[P=A0Zh3E`0002@0001A0P00[@<00582002^0`000P000542[P=B0Zd3E`0002@0001B0P00
Z`<005<2002]0`000P000582[@=C0Z/3E`0002@0001C0P00Y`<005<2002[0`000P0005<2Z`=C0ZL3
E`0002@0001B0P00Y@<005<2002W0`000P0005<2Y`=B0ZD3E`0002@0001A0P00Y0<00582002U0`00
0P000582Y@=A0Z@3E`0002@0001?0P00X`<00542002T0`000P000542Y0=?0Z<3E`0002@000280P00
X@<008T2002T0`000P0008T2X@>80Z@3E`0002@000260P00Y0<008P2002V0`000P0008P2Y0>60ZH3
E`0002@000230P00YP<008H2002W0`000P0008H2YP>30ZL3E`0002@000230P00Y`<008<2002W0`00
0P0008<2Y`>30ZL3E`0002@000200P00YP<008<2002W0`000P0008<2Y`>00ZH3E`0002@0001n0P00
Y0<00802002V0`000P000802YP=n0Z@3E`0002@0001m0P00X@<007h2002T0`000P0007h2Y0=m0Z43
E`0002@0001m0P00X0<007d2002Q0`000P0007d2X@=m0Z03E`0002@0001m0P00WP<007h2002P0`00
0P0007d2X0=n0Yh3E`0002@0001n0P00W0<00802002N0`000P0007h2WP>00Y`3E`0002@000200P00
V`<008<2002L0`000P000802W0>30Y/3E`0002@000230P00V`<008@2002K0`000P0008<2V`>40Y/3
E`0002@000240P00V`<008L2002L0`000P0008@2V`>70Y`3E`0002@000270P00W0<008T2002N0`00
0P0008L2W0>90Yh3E`0002@000290P00WP<008X2002P0`000P0008T2WP>:0Z03E`0002@0002:0P00
X0<008X2002V0`000P0008X2X0>:0ZH3E`0002@000290P00YP<008X2002Z0`000P0008X2YP>90ZX3
E`0002@000280P00ZP<008T2002/0`000P0008T2ZP>80Z`3E`0002@000260P00[0<008P2002^0`00
0P0008P2[0>60Zh3E`0002@000230P00[P<008H2002^0`000P0008H2[P>30Zh3E`0002@000210P00
[P<008<2002^0`000P0008<2[P>10Zh3E`0002@0001o0P00[P<00842002^0`000P000842[P=o0Zh3
E`0002@0001n0P00[0<007l2002^0`000P0007l2[P=n0Z`3E`0002@0001n0P00Z`<007h2002/0`00
0P0007h2[0=n0Z/3E`0002@0001n0P00ZP<007l2002[0`000P0007h2Z`=o0ZX3E`0002@0001o0P00
ZP<00802002[0`000P0007l2ZP>00Z/3E`0002@0001o0P00Z`<00802002/0`000P000802Z`=o0Z`3
E`0002@000210P00YP<008<2002W0`000P0008<2Y`>10ZH3E`0002@0001o0P00Y0<00842002V0`00
0P000842YP=o0Z@3E`0002@0001n0P00X@<007l2002T0`000P0007l2Y0=n0Z43E`0002@0001n0P00
X0<007h2002Q0`000P0007h2X@=n0Z03E`0002@0001n0P00WP<007l2002P0`000P0007h2X0=o0Yh3
E`0002@0001o0P00W0<00842002N0`000P0007l2WP>10Y`3E`0002@000210P00V`<008<2002L0`00
0P000842W0>30Y/3E`0002@000240P00V`<008H2002L0`000P0008@2V`>60Y`3E`0002@000260P00
W0<008P2002N0`000P0008H2W0>80Yh3E`0002@000280P00WP<008T2002P0`000P0008P2WP>90Z03
E`0002@000290P00X0<008T2002V0`000P0008T2X0>90ZH3E`0002@000280P00YP<008T2002Z0`00
0P0008T2YP>80ZX3E`0002@000270P00ZP<008P2002/0`000P0008P2ZP>70Z`3E`0002@000250P00
[0<008L2002^0`000P0008L2[0>50Zh3E`0002@000230P00[P<008D2002^0`000P0008D2[P>30Zh3
E`0002@0002]0P00WP<00:l2002O0`000P000:d2W`>_0Yh3E`0002@0002_0P00V`<00;42002N0`00
0P000:l2WP>a0Y/3E`0002@0002a0P00V`<00;42002^0`000P000;42V`>a0Zh3E`0002@0002`0P00
W0<00;02002^0`000P000;02W0>`0Zh3E`0002@0002]0P00[P<00;D2002^0`000P000:d2[P>e0Zh3
E`0002@0002o0P00V`<00<82002L0`000P000<82V`>o0Y`3E`0002@0002m0P00W0<00;l2002O0`00
0P000;l2W0>m0Yl3E`0002@0002l0P00W`<00;d2002S0`000P000;d2W`>l0Z<3E`0002@0002l0P00
X`<00;`2002V0`000P000;`2X`>l0ZH3E`0002@0002l0P00YP<00;d2002[0`000P000;`2YP>m0Z/3
E`0002@0002m0P00Z`<00;l2002^0`000P000;d2Z`>o0Zh3E`0002@0002o0P00[P<00<82002^0`00
0P000;l2[P?20Zh3E`0002@000320P00[P<00<<2002^0`000P000<82[P?30Zh3E`0002@000330P00
[P<00<H2002^0`000P000<<2[P?60Zh3E`0002@000360P00Z`<00<P2002^0`000P000<H2[P?80Z/3
E`0002@000380P00YP<00<T2002[0`000P000<P2Z`?90ZH3E`0002@000390P00X`<00<T2002V0`00
0P000<T2YP?90Z<3E`0002@000380P00W`<00<T2002S0`000P000<T2X`?80Yl3E`0002@000360P00
W0<00<P2002O0`000P000<P2W`?60Y`3E`0002@000330P00V`<00<H2002L0`000P000<H2W0?30Y/3
E`0002@000320P00V`<00<<2002K0`000P000<<2V`?20Y/3E`0002@000300P00V`<00<82002L0`00
0P000<82V`?00Y`3E`0002@0002o0P00W0<00<02002M0`000P000<02W0>o0Yd3E`0002@0002n0P00
W@<00;l2002O0`000P000;l2W@>n0Yl3E`0002@0002m0P00W`<00;h2002S0`000P000;h2W`>m0Z<3
E`0002@0002m0P00X`<00;d2002V0`000P000;d2X`>m0ZH3E`0002@0002m0P00YP<00;h2002[0`00
0P000;d2YP>n0Z/3E`0002@0002n0P00Z`<00;l2002]0`000P000;h2Z`>o0Zd3E`0002@0002o0P00
[@<00<02002^0`000P000;l2[@?00Zh3E`0002@000300P00[P<00<82002^0`000P000<02[P?20Zh3
E`0002@000330P00[P<00<D2002^0`000P000<<2[P?50Zh3E`0002@000350P00[@<00<H2002^0`00
0P000<D2[P?60Zd3E`0002@000360P00Z`<00<L2002]0`000P000<H2[@?70Z/3E`0002@000370P00
YP<00<P2002[0`000P000<L2Z`?80ZH3E`0002@000380P00X`<00<P2002V0`000P000<P2YP?80Z<3
E`0002@000370P00W`<00<P2002S0`000P000<P2X`?70Yl3E`0002@000360P00W@<00<L2002O0`00
0P000<L2W`?60Yd3E`0002@000350P00W0<00<H2002M0`000P000<H2W@?50Y`3E`0002@000330P00
V`<00<D2002L0`000P000<D2W0?30Y/3E`0002@0003S0P00WP<00>D2002O0`000P000><2W`?U0Yh3
E`0002@0003U0P00V`<00>L2002N0`000P000>D2WP?W0Y/3E`0002@0003W0P00V`<00>L2002^0`00
0P000>L2V`?W0Zh3E`0002@0003V0P00W0<00>H2002^0`000P000>H2W0?V0Zh3E`0002@0003S0P00
[P<00>/2002^0`000P000><2[P?[0Zh3E`0002@0003e0P00WP<00?L2002O0`000P000?D2W`?g0Yh3
E`0002@0003g0P00V`<00?T2002N0`000P000?L2WP?i0Y/3E`0002@0003i0P00V`<00?T2002^0`00
0P000?T2V`?i0Zh3E`0002@0003h0P00W0<00?P2002^0`000P000?P2W0?h0Zh3E`0002@0003e0P00
[P<00?d2002^0`000P000?D2[P?m0Zh3E`0002@0000I0`00WP<001/3002O0`000P0001T3W`<K0ih3
E`0002@0000K0`00V`<001d3002N0`000P0001/3WP<M0i/3E`0002@0000M0`00V`<001d3002^0`00
0P0001d3V`<M0jh3E`0002@0000L0`00W0<001`3002^0`000P0001`3W0<L0jh3E`0002@0000I0`00
[P<00243002^0`000P0001T3[P<Q0jh3E`0002@0000Y0`00W`<002X3002P0`000P0002T3W`<Z0j03
E`0002@0000Y0`00X0<002X3002P0`000P0002X3X0<Y0j03E`0002@0000X0`00X0<002T3002P0`00
0P0002T3X0<X0j03E`0002@0000X0`00W`<002P3002P0`000P0002P3X0<X0il3E`0002@0000X0`00
W@<002T3002O0`000P0002P3W`<Y0id3E`0002@0000Y0`00W0<002X3002M0`000P0002T3W@<Z0i`3
E`0002@0000Z0`00V`<002d3002L0`000P0002X3W0<]0i/3E`0002@0000]0`00V`<00303002K0`00
0P0002d3V`<`0i/3E`0002@0000`0`00V`<003<3002L0`000P000303V`<c0i`3E`0002@0000c0`00
W0<003@3002M0`000P0003<3W0<d0id3E`0002@0000d0`00W@<003D3002O0`000P0003@3W@<e0il3
E`0002@0000e0`00W`<003D3002P0`000P0003D3W`<e0j03E`0002@0000d0`00X0<003D3002R0`00
0P0003D3X0<d0j83E`0002@0000a0`00XP<003@3002T0`000P0003@3XP<a0j@3E`0002@0000]0`00
Y0<00343002V0`000P000343Y0<]0jH3E`0002@0000[0`00YP<002d3002W0`000P0002d3YP<[0jL3
E`0002@0000Y0`00Y`<002/3002Y0`000P0002/3Y`<Y0jT3E`0002@0000X0`00Z@<002T3002/0`00
0P0002T3Z@<X0j`3E`0002@0000X0`00[0<002P3002^0`000P0002P3[0<X0jh3E`0002@0000`0`00
V`<00383002L0`000P000303V`<b0i`3E`0002@0000b0`00W0<003<3002M0`000P000383W0<c0id3
E`0002@0000c0`00W@<003@3002O0`000P0003<3W@<d0il3E`0002@0000d0`00W`<003@3002P0`00
0P0003@3W`<d0j03E`0002@0000c0`00X0<003@3002R0`000P0003@3X0<c0j83E`0002@0000`0`00
XP<003<3002T0`000P0003<3XP<`0j@3E`0002@0000]0`00Y0<00303002V0`000P000303Y0<]0jH3
E`0002@0000X0`00[0<002T3002]0`000P0002P3[@<Y0j`3E`0002@0000Y0`00[0<002/3002/0`00
0P0002T3[0<[0j`3E`0002@0000[0`00[0<002l3002^0`000P0002/3[0<_0jh3E`0002@0000_0`00
[P<00383002^0`000P0002l3[P<b0jh3E`0002@0000b0`00[@<003@3002^0`000P000383[P<d0jd3
E`0002@0000d0`00[0<003D3002]0`000P0003@3[@<e0j`3E`0002@0000[0`00[0<002l3002^0`00
0P0002/3[0<_0jh3E`0002@0000_0`00[P<003<3002^0`000P0002l3[P<c0jh3E`0002@0000c0`00
[P<003@3002^0`000P0003<3[P<d0jh3E`0002@0000d0`00[0<003D3002^0`000P0003@3[P<e0j`3
E`0002@0000e0`00ZP<003D3002/0`000P0003D3[0<e0jX3E`0002@0001?0`00WP<00543002O0`00
0P0004l3W`=A0ih3E`0002@0001A0`00V`<005<3002N0`000P000543WP=C0i/3E`0002@0001C0`00
V`<005<3002^0`000P0005<3V`=C0jh3E`0002@0001B0`00W0<00583002^0`000P000583W0=B0jh3
E`0002@0001?0`00[P<005L3002^0`000P0004l3[P=G0jh3E`0002@0001O0`00W`<00603002P0`00
0P0005l3W`=P0j03E`0002@0001O0`00X0<00603002P0`000P000603X0=O0j03E`0002@0001N0`00
X0<005l3002P0`000P0005l3X0=N0j03E`0002@0001N0`00W`<005h3002P0`000P0005h3X0=N0il3
E`0002@0001N0`00W@<005l3002O0`000P0005h3W`=O0id3E`0002@0001O0`00W0<00603002M0`00
0P0005l3W@=P0i`3E`0002@0001P0`00V`<006<3002L0`000P000603W0=S0i/3E`0002@0001S0`00
V`<006H3002K0`000P0006<3V`=V0i/3E`0002@0001V0`00V`<006T3002L0`000P0006H3V`=Y0i`3
E`0002@0001Y0`00W0<006X3002N0`000P0006T3W0=Z0ih3E`0002@0001Z0`00WP<006X3002P0`00
0P0006X3WP=Z0j03E`0002@0001Y0`00X0<006X3002R0`000P0006X3X0=Y0j83E`0002@0001V0`00
XP<006T3002S0`000P0006T3XP=V0j<3E`0002@0001T0`00X`<006H3002S0`000P0006H3X`=T0j<3
E`0002@0001V0`00V`<006P3002L0`000P0006H3V`=X0i`3E`0002@0001X0`00W0<006T3002N0`00
0P0006P3W0=Y0ih3E`0002@0001Y0`00WP<006T3002P0`000P0006T3WP=Y0j03E`0002@0001X0`00
X0<006T3002R0`000P0006T3X0=X0j83E`0002@0001V0`00XP<006P3002S0`000P0006P3XP=V0j<3
E`0002@0001V0`00X`<006P3002T0`000P0006H3X`=X0j@3E`0002@0001X0`00Y0<006X3002V0`00
0P0006P3Y0=Z0jH3E`0002@0001Z0`00YP<006/3002X0`000P0006X3YP=[0jP3E`0002@0001[0`00
Z0<006/3002[0`000P0006/3Z0=[0j/3E`0002@0001Z0`00Z`<006/3002]0`000P0006/3Z`=Z0jd3
E`0002@0001Y0`00[@<006X3002^0`000P0006X3[@=Y0jh3E`0002@0001V0`00[P<006T3002^0`00
0P0006T3[P=V0jh3E`0002@0001S0`00[P<006H3002^0`000P0006H3[P=S0jh3E`0002@0001P0`00
[P<006<3002^0`000P0006<3[P=P0jh3E`0002@0001O0`00[@<00603002^0`000P000603[P=O0jd3
E`0002@0001N0`00Z`<005l3002]0`000P0005l3[@=N0j/3E`0002@0001N0`00ZP<005h3002[0`00
0P0005h3Z`=N0jX3E`0002@0001N0`00Z@<005l3002Z0`000P0005h3ZP=O0jT3E`0002@0001O0`00
Z@<00603002Z0`000P0005l3Z@=P0jX3E`0002@0001O0`00ZP<00603002[0`000P000603ZP=O0j/3
E`0002@0001Y0`00Y@<006X3002X0`000P0006T3Y@=Z0jP3E`0002@0001Z0`00Z0<006X3002[0`00
0P0006X3Z0=Z0j/3E`0002@0001Y0`00Z`<006X3002]0`000P0006X3Z`=Y0jd3E`0002@0001X0`00
[@<006T3002^0`000P0006T3[@=X0jh3E`0002@0001V0`00[P<006P3002^0`000P0006P3[P=V0jh3
E`0002@000250`00WP<008L3002O0`000P0008D3W`>70ih3E`0002@000270`00V`<008T3002N0`00
0P0008L3WP>90i/3E`0002@000290`00V`<008T3002^0`000P0008T3V`>90jh3E`0002@000280`00
W0<008P3002^0`000P0008P3W0>80jh3E`0002@000250`00[P<008d3002^0`000P0008D3[P>=0jh3
E`0002@0002L0`00W@<009`3002^0`000P0009`3W@>L0jh3E`0002@0002M0`00V`<009d3002^0`00
0P0009d3V`>M0jh3E`0002@0002C0`00V`<009d3002Y0`000P0009d3V`>C0jT3E`0002@0002C0`00
Z@<00:83002Y0`000P0009<3Z@>R0jT3E`0002@0002J0`00[P<00:03002^0`000P0009X3[P>P0jh3
E`0002@0002k0`00WP<00;d3002O0`000P000;/3W`>m0ih3E`0002@0002m0`00V`<00;l3002N0`00
0P000;d3WP>o0i/3E`0002@0002o0`00V`<00;l3002^0`000P000;l3V`>o0jh3E`0002@0002n0`00
W0<00;h3002^0`000P000;h3W0>n0jh3E`0002@0002k0`00[P<00<<3002^0`000P000;/3[P?30jh3
E`0002@0003:0`00V`<00<`3002T0`000P000<`3V`?:0j@3E`0002@0003:0`00XP<00<`3002T0`00
0P000<X3Y0?<0j83E`0002@0003<0`00X@<00<l3002R0`000P000<`3XP??0j43E`0002@0003?0`00
X@<00=43002Q0`000P000<l3X@?A0j43E`0002@0003A0`00X@<00=@3002R0`000P000=43X@?D0j83
E`0002@0003D0`00XP<00=H3002T0`000P000=@3XP?F0j@3E`0002@0003F0`00Y0<00=L3002W0`00
0P000=H3Y0?G0jL3E`0002@0003G0`00Y`<00=L3002Y0`000P000=L3Y`?G0jT3E`0002@0003F0`00
Z@<00=L3002/0`000P000=L3Z@?F0j`3E`0002@0003D0`00[0<00=H3002^0`000P000=H3[0?D0jh3
E`0002@0003A0`00[P<00=@3002^0`000P000=@3[P?A0jh3E`0002@0003?0`00[P<00=43002^0`00
0P000=43[P??0jh3E`0002@0003<0`00[P<00<l3002^0`000P000<l3[P?<0jh3E`0002@0003;0`00
[@<00<`3002^0`000P000<`3[P?;0jd3E`0002@0003:0`00Z`<00</3002]0`000P000</3[@?:0j/3
E`0002@0003:0`00ZP<00<X3002[0`000P000<X3Z`?:0jX3E`0002@0003:0`00Z@<00</3002Z0`00
0P000<X3ZP?;0jT3E`0002@0003;0`00Z@<00<`3002Z0`000P000</3Z@?<0jX3E`0002@0003;0`00
ZP<00<`3002[0`000P000<`3ZP?;0j/3E`0002@0003A0`00X@<00=<3002R0`000P000=43X@?C0j83
E`0002@0003C0`00XP<00=D3002T0`000P000=<3XP?E0j@3E`0002@0003E0`00Y0<00=H3002W0`00
0P000=D3Y0?F0jL3E`0002@0003F0`00Y`<00=H3002Y0`000P000=H3Y`?F0jT3E`0002@0003E0`00
Z@<00=H3002/0`000P000=H3Z@?E0j`3E`0002@0003C0`00[0<00=D3002^0`000P000=D3[0?C0jh3
E`0002@0003A0`00[P<00=<3002^0`000P000=<3[P?A0jh3E`0002@0003<0`00V`<00=D3002K0`00
0P000<`3V`?E0i/3E`0002@0003<0`00W0<00=03002L0`000P000<`3W0?@0i`3E`0002@0003@0`00
V`<00=D3002L0`000P000=03W0?E0i/3E`0002@0003a0`00WP<00?<3002O0`000P000?43W`?c0ih3
E`0002@0003c0`00V`<00?D3002N0`000P000?<3WP?e0i/3E`0002@0003e0`00V`<00?D3002^0`00
0P000?D3V`?e0jh3E`0002@0003d0`00W0<00?@3002^0`000P000?@3W0?d0jh3E`0002@0003a0`00
[P<00?T3002^0`000P000?43[P?i0jh3E`0002@0000:1000WP<000/4002O0`000P0000/4WP<:19l3
E`0002@0000:1000W`<000/4002P0`000P0000X4W`<;1:03E`0002@0000;1000W`<000`4002P0`00
0P0000/4X0<<19l3E`0002@0000<1000WP<000`4002O0`000P0000`4W`<<19h3E`0002@0000;1000
W0<000`4002N0`000P0000`4WP<;19`3E`0002@000091000V`<000/4002L0`000P0000/4W0<919/3
E`0002@000061000V`<000T4002K0`000P0000T4V`<619/3E`0002@000041000V`<000H4002L0`00
0P0000H4V`<419`3E`0002@000021000W0<000@4002N0`000P0000@4W0<219h3E`0002@000011000
WP<00084002P0`000P000084WP<11:03E`0002@000001000X0<00044002S0`000P000044X0<01:<3
E`0002@000001000X`<00004002Y0`000P000004X`<01:T3E`0002@000001000Z@<00044002/0`00
0P000004Z@<11:`3E`0002@000011000[0<000<4002^0`000P000044[0<31:h3E`0002@000031000
[P<000H4002^0`000P0000<4[P<61:h3E`0002@000061000[P<000L4002^0`000P0000H4[P<71:h3
E`0002@000071000[P<000X4002^0`000P0000L4[P<:1:h3E`0002@0000:1000[0<000`4002^0`00
0P0000X4[P<<1:`3E`0002@0000<1000Z@<000d4002/0`000P0000`4[0<=1:T3E`0002@0000=1000
Z0<000d4002Y0`000P0000d4Z@<=1:P3E`0002@0000<1000Y@<000d4002X0`000P0000d4Z0<<1:D3
E`0002@0000:1000X`<000`4002U0`000P0000`4Y@<:1:<3E`0002@000071000XP<000X4002S0`00
0P0000X4X`<71:83E`0002@000061000XP<000L4002R0`000P0000L4XP<61:83E`0002@000041000
XP<000H4002S0`000P0000H4XP<41:<3E`0002@000021000X`<000@4002U0`000P0000@4X`<21:D3
E`0002@000011000Y@<00084002X0`000P000084Y@<11:P3E`0002@000051000V`<000H4002L0`00
0P0000H4V`<519`3E`0002@000031000W0<000D4002N0`000P0000D4W0<319h3E`0002@000021000
WP<000<4002P0`000P0000<4WP<21:03E`0002@000011000X0<00084002S0`000P000084X0<11:<3
E`0002@000011000X`<00044002Y0`000P000044X`<11:T3E`0002@000011000Z@<00084002/0`00
0P000044Z@<21:`3E`0002@000021000[0<000@4002^0`000P000084[0<41:h3E`0002@000041000
[P<000H4002^0`000P0000@4[P<61:h3E`0002@000071000[P<000T4002^0`000P0000L4[P<91:h3
E`0002@000091000[0<000/4002^0`000P0000T4[P<;1:`3E`0002@0000;1000Z@<000`4002/0`00
0P0000/4[0<<1:T3E`0002@0000<1000Z0<000`4002Y0`000P0000`4Z@<<1:P3E`0002@0000;1000
Y@<000`4002X0`000P0000`4Z0<;1:D3E`0002@000091000X`<000/4002U0`000P0000/4Y@<91:<3
E`0002@000071000XP<000T4002S0`000P0000T4X`<71:83E`0002@0000W1000WP<002T4002O0`00
0P0002L4W`<Y19h3E`0002@0000Y1000V`<002/4002N0`000P0002T4WP<[19/3E`0002@0000[1000
V`<002/4002^0`000P0002/4V`<[1:h3E`0002@0000Z1000W0<002X4002^0`000P0002X4W0<Z1:h3
E`0002@0000W1000[P<002l4002^0`000P0002L4[P<_1:h3E`0002@0000f1000V`<003H4002P0`00
0P0003H4V`<f1:03E`0002@0000f1000W@<003L4002O0`000P0003H4W`<g19d3E`0002@0000g1000
V`<003T4002M0`000P0003L4W@<i19/3E`0002@0000i1000V`<003/4002K0`000P0003T4V`<k19/3
E`0002@0000k1000V`<003l4002N0`000P0003/4V`<o19h3E`0002@0000o1000WP<00444002N0`00
0P0003l4WP=119h3E`0002@000111000W@<00484002N0`000P000444WP=219d3E`0002@000121000
V`<004<4002M0`000P000484W@=319/3E`0002@0000g1000W0<003T4002M0`000P0003L4W@<i19`3
E`0002@0000i1000W0<003/4002L0`000P0003T4W0<k19`3E`0002@0000k1000W0<003l4002N0`00
0P0003/4W0<o19h3E`0002@000131000V`<004<4002N0`000P0004<4V`=319h3E`0002@000121000
WP<004<4002P0`000P0004<4WP=21:03E`0002@0000n1000X0<00484002U0`000P000484X0<n1:D3
E`0002@0000m1000Y@<003h4002W0`000P0003h4Y@<m1:L3E`0002@0000l1000Y`<003d4002Z0`00
0P0003d4Y`<l1:X3E`0002@0000l1000ZP<003`4002^0`000P0003`4ZP<l1:h3E`0002@0000m1000
X0<00484002U0`000P000484X0<m1:D3E`0002@0000l1000Y@<003d4002W0`000P0003d4Y@<l1:L3
E`0002@0000l1000Y`<003`4002Z0`000P0003`4Y`<l1:X3E`0002@0000l1000ZP<003`4002^0`00
0P0003`4ZP<l1:h3E`0002@0001M1000WP<005l4002O0`000P0005d4W`=O19h3E`0002@0001O1000
V`<00644002N0`000P0005l4WP=Q19/3E`0002@0001Q1000V`<00644002^0`000P000644V`=Q1:h3
E`0002@0001P1000W0<00604002^0`000P000604W0=P1:h3E`0002@0001M1000[P<006D4002^0`00
0P0005d4[P=U1:h3E`0002@0001]1000V`<00704002L0`000P000704V`=]19`3E`0002@0001/1000
W0<006d4002N0`000P0006d4W0=/19h3E`0002@0001/1000WP<006`4002P0`000P0006`4WP=/1:03
E`0002@0001/1000X0<006d4002R0`000P0006`4X0=]1:83E`0002@0001]1000XP<00704002S0`00
0P0006d4XP=`1:<3E`0002@0001`1000X`<007<4002S0`000P000704X`=c1:<3E`0002@0001c1000
XP<007H4002S0`000P0007<4X`=f1:83E`0002@0001f1000X0<007L4002R0`000P0007H4XP=g1:03
E`0002@0001g1000WP<007L4002P0`000P0007L4X0=g19h3E`0002@0001f1000W0<007L4002N0`00
0P0007L4WP=f19`3E`0002@0001c1000V`<007H4002L0`000P0007H4W0=c19/3E`0002@0001`1000
V`<007<4002K0`000P0007<4V`=`19/3E`0002@0001^1000V`<00704002L0`000P000704V`=^19`3
E`0002@0001]1000W0<006h4002N0`000P0006h4W0=]19h3E`0002@0001]1000WP<006d4002P0`00
0P0006d4WP=]1:03E`0002@0001]1000X0<006h4002R0`000P0006d4X0=^1:83E`0002@0001^1000
XP<00704002S0`000P0006h4XP=`1:<3E`0002@0001c1000XP<007D4002S0`000P0007<4X`=e1:83
E`0002@0001e1000X0<007H4002R0`000P0007D4XP=f1:03E`0002@0001f1000WP<007H4002P0`00
0P0007H4X0=f19h3E`0002@0001e1000W0<007H4002N0`000P0007H4WP=e19`3E`0002@0001c1000
V`<007D4002L0`000P0007D4W0=c19/3E`0002@0001]1000X`<00704002T0`000P000704X`=]1:@3
E`0002@0001/1000Y0<006d4002U0`000P0006d4Y0=/1:D3E`0002@0001[1000Y@<006`4002W0`00
0P0006`4Y@=[1:L3E`0002@0001[1000Y`<006/4002[0`000P0006/4Y`=[1:/3E`0002@0001[1000
Z`<006`4002]0`000P0006/4Z`=/1:d3E`0002@0001/1000[@<006d4002^0`000P0006`4[@=]1:h3
E`0002@0001]1000[P<00704002^0`000P0006d4[P=`1:h3E`0002@0001`1000[P<007<4002^0`00
0P000704[P=c1:h3E`0002@0001c1000[P<007H4002^0`000P0007<4[P=f1:h3E`0002@0001f1000
[@<007L4002^0`000P0007H4[P=g1:d3E`0002@0001g1000Z`<007P4002]0`000P0007L4[@=h1:/3
E`0002@0001h1000Y`<007P4002[0`000P0007P4Z`=h1:L3E`0002@0001g1000Y@<007P4002W0`00
0P0007P4Y`=g1:D3E`0002@0001f1000Y0<007L4002U0`000P0007L4Y@=f1:@3E`0002@0001c1000
X`<007H4002T0`000P0007H4Y0=c1:<3E`0002@0001^1000X`<00704002T0`000P000704X`=^1:@3
E`0002@0001]1000Y0<006h4002U0`000P0006h4Y0=]1:D3E`0002@0001/1000Y@<006d4002W0`00
0P0006d4Y@=/1:L3E`0002@0001/1000Y`<006`4002[0`000P0006`4Y`=/1:/3E`0002@0001/1000
Z`<006d4002]0`000P0006`4Z`=]1:d3E`0002@0001]1000[@<006h4002^0`000P0006d4[@=^1:h3
E`0002@0001^1000[P<00704002^0`000P0006h4[P=`1:h3E`0002@0001c1000[P<007D4002^0`00
0P0007<4[P=e1:h3E`0002@0001e1000[@<007H4002^0`000P0007D4[P=f1:d3E`0002@0001f1000
Z`<007L4002]0`000P0007H4[@=g1:/3E`0002@0001g1000Y`<007L4002[0`000P0007L4Z`=g1:L3
E`0002@0001f1000Y@<007L4002W0`000P0007L4Y`=f1:D3E`0002@0001e1000Y0<007H4002U0`00
0P0007H4Y@=e1:@3E`0002@0001c1000X`<007D4002T0`000P0007D4Y0=c1:<3E`0002@0002C1000
WP<009D4002O0`000P0009<4W`>E19h3E`0002@0002E1000V`<009L4002N0`000P0009D4WP>G19/3
E`0002@0002G1000V`<009L4002^0`000P0009L4V`>G1:h3E`0002@0002F1000W0<009H4002^0`00
0P0009H4W0>F1:h3E`0002@0002C1000[P<009/4002^0`000P0009<4[P>K1:h3E`0002@0002/1000
X@<00:d4002T0`000P000:d4X@>/1:@3E`0002@0002Z1000Y0<00:`4002V0`000P000:`4Y0>Z1:H3
E`0002@0002W1000YP<00:X4002W0`000P000:X4YP>W1:L3E`0002@0002W1000Y`<00:L4002W0`00
0P000:L4Y`>W1:L3E`0002@0002T1000YP<00:L4002W0`000P000:L4Y`>T1:H3E`0002@0002R1000
Y0<00:@4002V0`000P000:@4YP>R1:@3E`0002@0002Q1000X@<00:84002T0`000P000:84Y0>Q1:43
E`0002@0002Q1000X0<00:44002Q0`000P000:44X@>Q1:03E`0002@0002Q1000WP<00:84002P0`00
0P000:44X0>R19h3E`0002@0002R1000W0<00:@4002N0`000P000:84WP>T19`3E`0002@0002T1000
V`<00:L4002L0`000P000:@4W0>W19/3E`0002@0002W1000V`<00:P4002K0`000P000:L4V`>X19/3
E`0002@0002X1000V`<00:/4002L0`000P000:P4V`>[19`3E`0002@0002[1000W0<00:d4002N0`00
0P000:/4W0>]19h3E`0002@0002]1000WP<00:h4002P0`000P000:d4WP>^1:03E`0002@0002^1000
X0<00:h4002V0`000P000:h4X0>^1:H3E`0002@0002]1000YP<00:h4002Z0`000P000:h4YP>]1:X3
E`0002@0002/1000ZP<00:d4002/0`000P000:d4ZP>/1:`3E`0002@0002Z1000[0<00:`4002^0`00
0P000:`4[0>Z1:h3E`0002@0002W1000[P<00:X4002^0`000P000:X4[P>W1:h3E`0002@0002U1000
[P<00:L4002^0`000P000:L4[P>U1:h3E`0002@0002S1000[P<00:D4002^0`000P000:D4[P>S1:h3
E`0002@0002R1000[0<00:<4002^0`000P000:<4[P>R1:`3E`0002@0002R1000Z`<00:84002/0`00
0P000:84[0>R1:/3E`0002@0002R1000ZP<00:<4002[0`000P000:84Z`>S1:X3E`0002@0002S1000
ZP<00:@4002[0`000P000:<4ZP>T1:/3E`0002@0002S1000Z`<00:@4002/0`000P000:@4Z`>S1:`3
E`0002@0002U1000YP<00:L4002W0`000P000:L4Y`>U1:H3E`0002@0002S1000Y0<00:D4002V0`00
0P000:D4YP>S1:@3E`0002@0002R1000X@<00:<4002T0`000P000:<4Y0>R1:43E`0002@0002R1000
X0<00:84002Q0`000P000:84X@>R1:03E`0002@0002R1000WP<00:<4002P0`000P000:84X0>S19h3
E`0002@0002S1000W0<00:D4002N0`000P000:<4WP>U19`3E`0002@0002U1000V`<00:L4002L0`00
0P000:D4W0>W19/3E`0002@0002X1000V`<00:X4002L0`000P000:P4V`>Z19`3E`0002@0002Z1000
W0<00:`4002N0`000P000:X4W0>/19h3E`0002@0002/1000WP<00:d4002P0`000P000:`4WP>]1:03
E`0002@0002]1000X0<00:d4002V0`000P000:d4X0>]1:H3E`0002@0002/1000YP<00:d4002Z0`00
0P000:d4YP>/1:X3E`0002@0002[1000ZP<00:`4002/0`000P000:`4ZP>[1:`3E`0002@0002Y1000
[0<00:/4002^0`000P000:/4[0>Y1:h3E`0002@0002W1000[P<00:T4002^0`000P000:T4[P>W1:h3
E`0002@000361000W`<00<L4002P0`000P000<H4W`?71:03E`0002@000361000X0<00<L4002P0`00
0P000<L4X0?61:03E`0002@000351000X0<00<H4002P0`000P000<H4X0?51:03E`0002@000351000
W`<00<D4002P0`000P000<D4X0?519l3E`0002@000351000W@<00<H4002O0`000P000<D4W`?619d3
E`0002@000361000W0<00<L4002M0`000P000<H4W@?719`3E`0002@000371000V`<00<X4002L0`00
0P000<L4W0?:19/3E`0002@0003:1000V`<00<d4002K0`000P000<X4V`?=19/3E`0002@0003=1000
V`<00=04002L0`000P000<d4V`?@19`3E`0002@0003@1000W0<00=44002M0`000P000=04W0?A19d3
E`0002@0003A1000W@<00=84002O0`000P000=44W@?B19l3E`0002@0003B1000W`<00=84002P0`00
0P000=84W`?B1:03E`0002@0003A1000X0<00=84002R0`000P000=84X0?A1:83E`0002@0003>1000
XP<00=44002T0`000P000=44XP?>1:@3E`0002@0003:1000Y0<00<h4002V0`000P000<h4Y0?:1:H3
E`0002@000381000YP<00<X4002W0`000P000<X4YP?81:L3E`0002@000361000Y`<00<P4002Y0`00
0P000<P4Y`?61:T3E`0002@000351000Z@<00<H4002/0`000P000<H4Z@?51:`3E`0002@000351000
[0<00<D4002^0`000P000<D4[0?51:h3E`0002@0003=1000V`<00<l4002L0`000P000<d4V`??19`3
E`0002@0003?1000W0<00=04002M0`000P000<l4W0?@19d3E`0002@0003@1000W@<00=44002O0`00
0P000=04W@?A19l3E`0002@0003A1000W`<00=44002P0`000P000=44W`?A1:03E`0002@0003@1000
X0<00=44002R0`000P000=44X0?@1:83E`0002@0003=1000XP<00=04002T0`000P000=04XP?=1:@3
E`0002@0003:1000Y0<00<d4002V0`000P000<d4Y0?:1:H3E`0002@000351000[0<00<H4002]0`00
0P000<D4[@?61:`3E`0002@000361000[0<00<P4002/0`000P000<H4[0?81:`3E`0002@000381000
[0<00<`4002^0`000P000<P4[0?<1:h3E`0002@0003<1000[P<00<l4002^0`000P000<`4[P??1:h3
E`0002@0003?1000[@<00=44002^0`000P000<l4[P?A1:d3E`0002@0003A1000[0<00=84002]0`00
0P000=44[@?B1:`3E`0002@000381000[0<00<`4002^0`000P000<P4[0?<1:h3E`0002@0003<1000
[P<00=04002^0`000P000<`4[P?@1:h3E`0002@0003@1000[P<00=44002^0`000P000=04[P?A1:h3
E`0002@0003A1000[0<00=84002^0`000P000=44[P?B1:`3E`0002@0003B1000ZP<00=84002/0`00
0P000=84[0?B1:X3E`0002@0003J1000V`<00=d4002L0`000P000=d4V`?J19`3E`0002@0003H1000
W0<00=X4002O0`000P000=X4W0?H19l3E`0002@0003G1000W`<00=P4002S0`000P000=P4W`?G1:<3
E`0002@0003G1000X`<00=L4002V0`000P000=L4X`?G1:H3E`0002@0003G1000YP<00=P4002[0`00
0P000=L4YP?H1:/3E`0002@0003H1000Z`<00=X4002^0`000P000=P4Z`?J1:h3E`0002@0003J1000
[P<00=d4002^0`000P000=X4[P?M1:h3E`0002@0003M1000[P<00=h4002^0`000P000=d4[P?N1:h3
E`0002@0003N1000[P<00>44002^0`000P000=h4[P?Q1:h3E`0002@0003Q1000Z`<00><4002^0`00
0P000>44[P?S1:/3E`0002@0003S1000YP<00>@4002[0`000P000><4Z`?T1:H3E`0002@0003T1000
X`<00>@4002V0`000P000>@4YP?T1:<3E`0002@0003S1000W`<00>@4002S0`000P000>@4X`?S19l3
E`0002@0003Q1000W0<00><4002O0`000P000><4W`?Q19`3E`0002@0003N1000V`<00>44002L0`00
0P000>44W0?N19/3E`0002@0003M1000V`<00=h4002K0`000P000=h4V`?M19/3E`0002@0003K1000
V`<00=d4002L0`000P000=d4V`?K19`3E`0002@0003J1000W0<00=/4002M0`000P000=/4W0?J19d3
E`0002@0003I1000W@<00=X4002O0`000P000=X4W@?I19l3E`0002@0003H1000W`<00=T4002S0`00
0P000=T4W`?H1:<3E`0002@0003H1000X`<00=P4002V0`000P000=P4X`?H1:H3E`0002@0003H1000
YP<00=T4002[0`000P000=P4YP?I1:/3E`0002@0003I1000Z`<00=X4002]0`000P000=T4Z`?J1:d3
E`0002@0003J1000[@<00=/4002^0`000P000=X4[@?K1:h3E`0002@0003K1000[P<00=d4002^0`00
0P000=/4[P?M1:h3E`0002@0003N1000[P<00>04002^0`000P000=h4[P?P1:h3E`0002@0003P1000
[@<00>44002^0`000P000>04[P?Q1:d3E`0002@0003Q1000Z`<00>84002]0`000P000>44[@?R1:/3
E`0002@0003R1000YP<00><4002[0`000P000>84Z`?S1:H3E`0002@0003S1000X`<00><4002V0`00
0P000><4YP?S1:<3E`0002@0003R1000W`<00><4002S0`000P000><4X`?R19l3E`0002@0003Q1000
W@<00>84002O0`000P000>84W`?Q19d3E`0002@0003P1000W0<00>44002M0`000P000>44W@?P19`3
E`0002@0003N1000V`<00>04002L0`000P000>04W0?N19/3E`0002@0003E0000iP400=D4003V0@00
0P000=D0iP7E1>H19@0000`000080020:00000`0000200009`0001P0000200000000000000000000
9@0000`000020000:00000`000010000EP000300003C0000iP400=P0003m0@001@000=<0iP7H0>H1
f03m0M<0o@7C0>H19@0000`000080020EP00030000090@00[04000h1003V0@001@0000h1iP490NH1
2@6/0@h1[04>0NH19@0000`000080020EP000300000n0@00iP4004<1003m0@001@0003h1iP530NH1
@`7m0Ch1o@4n0NH19@0000`000080020EP000300001d0@00e`4007T1003V0@001@0007T1iP5d0NH1
M07G0GT1e`5i0NH19@0000`000080020EP000300002Z0@00g0400:l1003V0@001@000:l1iP6Z0NH1
ZP7L0Jl1g06_0NH19@0000`000080020EP000300003P0@00iP400>D1003X0@001@000>01iP7U0NH1
i@7X0N01j07P0NH19@0000`000080020EP000300000F0P00`04001/2003V0@001@0001/2iP4F0^H1
5P;00A/2`04K0^H19@0000`000080020EP000300001<0P00iP400542001=0P001@0004`2iP5A0^H1
D@9=0T`2C@9<0^H19@0000`000080020EP00030000220P00iP4008L2003[0@001@000882iP670^H1
Q`;[0H82j`620^H19@0000`000080020EP000300002h0P00Y`400;d2003V0@001@000;d2iP6h0^H1
^0:W0Kd2Y`6m0^H19@0000`000080020EP000300003^0P00iP400?<2000@0P001@000>h2iP7c0^H1
l`8@0^h240;^0^H19@0000`000080020EP000300000T0`00c04002T3003V0@001@0002T3iP4T0nH1
90?<0BT3c04Y0nH19@0000`000080020EP000300001J0`00h04005l3003V0@001@0005l3iP5J0nH1
FP?P0El3h05O0nH19@0000`000080020EP000300002@0`00c@4009D3003V0@001@0009D3iP6@0nH1
T0?=0ID3c@6E0nH19@0000`000080020EP00030000360`00iP400</300090P001@000<H3iP7;0nH1
b`<90/H32@;60nH19@0000`000080020EP000300003l0`00V0400044003V0@001@000044iP7l0nH1
o0>H0@44V0411>H19@0000`000080020EP000300000b1000iP4003L4003l0@001@000384iP4g1>H1
=`Cl0C84o04b1>H19@0000`000080020EP000300001W1000iP4006`4000;0P001@0006L4iP5/1>H1
K0@;0VL42`9W1>H19@0000`000080020EP000300002M1000iP400:84003W0@001@0009d4iP6R1>H1
XPCW0Id4i`6M1>H19@0000`000080020EP000300003C1000iP400=P4003j0@001@000=<4iP7H1>H1
f0Cj0M<4nP7C1>H19P0001`0000100000000004000000000000002D0000<00000@0005/0002@0000
cP0007H3003J0000P`<00040000K00006`000=X0O0?J07T3f@1i0mP0N0?G07L3eP1g0mD0MP?C07L3
d@1g0m00N0?@07T3c`1i0ll0N`?>07`3c`1n0ll0P0?@0843d0220m40PP?B08<3e`230mL0PP?H0883
f@210mX0P0?J07h3fP1l0e/0002@0000104004X0000@0@00EP000040000K00006`000101D00@0Dd0
3`5<00h1B`0=0D/03@5:00P1BP070D/01P5;00H1C0050Dd01@5>00@1C`040E401@5B00D1D`060E@0
1P5E00L1E@080EH03@5F00d1E@0>0ED03`5D0101D`0@0E40405@05/0002H0000>P4007H300160@00
P`<00040000M00007@0004H1O0=60GX3A@5i0d@1N0=30GL3@@5g0d01MP<o0GL3?@5g0c`1N0<l0GT3
>`5i0c/1NP<j0G/3>P5n0c/1O`<k0H03?0610c`1PP<m0H83?P630d81P`=30H83A0620dD1P@=50H03
AP5o0dH1OP=60G`3F`0009P0001`0@00MP<007`100230`000@0001d0000M0000O05l0g`1NP=k0GT3
NP5h0gT1M`=g0GL3MP5f0gD1M`=c0GL3LP5h0g81N@=a0GT3L@5j0g01N`=`0Gh3L@5o0g41P0=b0H43
LP620g<1PP=d0H<3N0630gT1PP=j0H83N`610g/1P0=l0Gl3O05n0g`1O0=K0000T0000:H1001:0000
/P4005H0000100006`0001/0002b0E00/P5>0;41C@2a0D`0/05;0:l1B`2^0DX0ZP5:0:T1B`2X0D/0
Y`5<0:L1CP2V0Dl0YP5A0:L1DP2W0E@0Z05E0:T1E@2Z0EH0[P5F0:l1E@2`0ED0/@5D0;41D`2b0E80
/P5A0;81D01K0000S0000=`1001f0`00j04008<3000100006P0001X0003X0G`3j05j0nL1N@?V0GP3
i@5g0n<1M`?R0GH3h@5g0ml1M`?N0GP3g@5i0m`1NP?L0Gl3g@600md1P@?N0H83g`620n01P`?T0H<3
i@620nH1PP?W0H43i`600nP1O`?X0Gh3j05l0e/0002<00004P8007H3000N0P00P`<00040000J0000
6P0001h2O0<N0WX37@9i0a`2N0<K0WL36@9g0aP2MP<G0WL35@9g0a@2N0<C0WT34P9j0a82O`<C0X03
4`:10a@2PP<E0X835P:30aX2P`<K0X8370:20ad2P@<M0X037P9o0ah2OP<N0W`3F`0008`000180P00
MP<005@200230`000@0001X0000J0000E09l0e@2NP=C0WT3DP9h0e42M`=?0WL3CP9f0dd2M`=;0WL3
BP9h0dT2N@=80WX3B09o0dT2P0=90X43BP:20d/2PP=<0X<3D0:30e42PP=B0X83D`:10e<2P0=D0Wl3
E09n0e@2O0=K0000U00007h2001:0000RP8005H000010000700001`0002:0U00RP9?08T2CP290Td0
R09<08P2B`270T/0QP9:0882BP210T/0P09;07l2C01o0Td0OP9>07h2DP1o0U<0O`9D0802E@210UD0
PP9F08H2EP270UD0R09E08P2E0290U<0R@9B08X2D@2:0U00F`0009@0002d0P00BP000<02001F0000
0@0001`0000L0000`09@0<02C`2o0Th0_`9=0;h2C02n0T/0_@9;0;`2BP2h0TX0]`9;0;H2B`2e0T`0
]@9=0;@2CP2d0U80]@9C0;D2E02f0UD0]`9E0;P2EP2l0UH0_@9E0;h2E@2n0U@0_`9C0;l2DP300U40
`09@05/0002D0000jP8007H3003f0P00P`<00040000L000070000?H2O0?f0W/3m@9j0oD2N@?d0WT3
m09h0o<2M`?a0WL3l09f0nh2M`?]0WL3k09h0n/2N@?Z0WX3jP9o0n/2P0?[0X43k0:20nd2PP?]0X<3
lP:30o<2PP?d0X83m0:10oD2P0?e0Wl3mP9n0oH2O0=K0000R0000203001f0`00;0<008<300010000
6@0001T0000/0g`3:`=k0b/3N@<Z0gT3:P=h0bT3M`<W0gL39P=f0b@3M`<S0gL38P=h0b43N@<P0gT3
80>00b43P@<R0h838`>20b<3P`<X0h<3:@>20bX3PP<Z0h43:`>00b/3OP</0g`3F`0007@0001F0`00
BP000683001F00000@0001@0000D0000HP=@0643C`1Q0dd0H0=<05l3B`1N0dX0F@=:05P3B`1G0d`0
EP==05H3D`1G0e@0F0=E05T3EP1N0eH0G`=E0603E01Q0e<0H@=A0683D01K0000L00008`3001:0000
U`<005H0000100004`0001<0002G0e00U`==09H3C02E0d/0U0=:08l3BP2>0d/0S@=<08`3C@2<0e<0
S@=D08h3E@2?0eH0U0=F09D3E@2F0e@0U`=C09L3D@2G0e00F`0008P000310`00MP<00<d300230`00
0@0001T0000I0000c@=l0ld3N@?<0gT3b`=h0l/3M`?90gL3b0=f0lH3M`?40gL3a0=h0l<3N@?20gT3
`P=k0l43O0?20gh3`P>00l<3P@?40h83a@>30lX3P`?;0h83c0>10ld3P0?=0gh3c@=l0e/0002D0000
m`<007H300031000P`<00040000L0000700000<4O0<317T30PAi0`44N0<017L3o`=g0oh3MP?l0gL3
nP=g0oT3N0?i0gT3n0=i0oP3NP?g0g/3m`=n0oP3O`?h0h03n@>10oT3PP?j0h83n`>30`04P`<01883
0@B20`84P@<318030`An0`<4O0=K0000V00002d4001f0`00>@@008<3000100007@0001d0000i17`3
>@Aj0cP4N@<g17P3=PAg0cD4M`<d17H3<PAg0c04M`<_17P3;`Ai0bh4N@<^17X3;@Ak0bd4OP<^17l3
;PB00bl4P@<_1883<0B20c44P`<e18<3=PB20cL4PP<h1843>0B00cT4O`<i17h3>@Al0e/0002H0000
H`@004X0001_1000EP000040000M00007@0006l4D01_14h0KPA=06h4C01]14/0K0A;06/4BP1W14X0
IPA;06D4B`1U14`0I0A=06@4CP1S14l0H`AA06@4DP1T15<0I@AD06D4E@1V15D0I`AF06/4EP1/15D0
K@AE06h4E01^15<0K`AB06l4D@1_1500F`0009P0002I1000MP<00:D400230`000@0001d0000M0000
Y@Al0jD4NP>T17T3X`Ah0j84M`>P17L3W`Af0ih4M`>L17L3V`Ah0i/4N@>J17T3VPAj0iT4N`>I17h3
VPAo0iX4P0>K1843V`B20i`4PP>M18<3X@B30j84PP>S1883Y0B10j@4P0>U17l3Y@An0jD4O0=K0000
T0000<l4001:0000f`@005H0000100006`0001/0003K1500f`A>0=X4C@3J14`0f@A;0=P4B`3G14X0
d`A:0=84B`3A14/0d0A<0=04C@3?14h0c`AB0=04D`3@15@0d@AE0=84E@3C15H0e`AF0=P4E@3I15D0
fPAD0=X4D`3K1580f`AA0=/4D01G0000=0000=0000180000fP@008D300060000d0240m00B03J14P0
fPB40m00Q0?@08D39@0000`000050020:00000`0000200009@0000`000080020:00000`000010000
9@0000`0000=00203P0001@000000000400001@0
\>"], "Graphics",
 ImageSize->{250.75, 198.563},
 ImageMargins->{{0, 0}, {0, 1.5}},
 ImageRegion->{{0, 1}, {0, 1}}],

Cell[TextData[{
 "The intuition is as follows. Intuitively, the Gibbs sampler tries to figure \
out how much of an oberserved price change is transient (due to bid ask \
effects) and how much is permanent (the efficient price innovation). When ",
 Cell[BoxData["c"]],
 " is large relative to ",
 Cell[BoxData[
  SubscriptBox["\[Sigma]", "u"]]],
 ", bid-ask bounce generates reversals that are easy to pick out visually, \
and using the sampler.  When ",
 Cell[BoxData["c"]],
 " is small, though, bid-ask effects are swamped by the innovations in the \
efficient price, and are not easily discerned."
}], "Text"],

Cell["\<\
We'll see that this extends to the parameter estimates as well.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Modification when some of the q's are known.", "Subsection"],

Cell[TextData[{
 "In some samples, it might happen that the trade directions are known for \
some subset of the ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 ". For these ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 ", we don't simulate; we simply leave them at their known values."
}], "Text"],

Cell[TextData[{
 "This might seem to violate the assumed probability structure of the model \
in a fundamental way. After all, if the data generating process and our \
priors are that ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", 
   RowBox[{"\[PlusMinus]", "1"}]}]]],
 ", with equal probability, how can a definite realization be accomodated? \
The answer is that we're conditioning on the observed data, and the only \
thing that matters is the prior distribution of the ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " that we don't observe."
}], "Text"],

Cell["\<\
By way of a more formal justification, we could assume that the data \
generating process involves two steps:\
\>", "Text"],

Cell[TextData[{
 "First ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " is drawn, ",
 Cell[BoxData[
  RowBox[{"\[PlusMinus]", "1"}]]],
 ", each with probability ",
 Cell[BoxData[
  FractionBox["1", "2"]]],
 "."
}], "BulletedList"],

Cell[TextData[{
 "Next, an indicator variable ",
 Cell[BoxData[
  SubscriptBox["O", "t"]]],
 " is drawn. With probability ",
 Cell[BoxData["\[Pi]"]],
 ", ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["O", "t"], "=", "1"}]]],
 ", and the actual ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " is observed. With probability ",
 Cell[BoxData[
  RowBox[{"1", "-", "\[Pi]"}]]],
 ", ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["O", "t"], "=", "0"}]]],
 ", and the actual ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " is unobserved."
}], "BulletedList"],

Cell[TextData[{
 "As part of the sample, we \"observe\" the realizations of ",
 Cell[BoxData[
  SubscriptBox["O", "t"]]],
 ". That is, we know which ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " are known for sure. If we don't care about modeling the ",
 Cell[BoxData[
  SubscriptBox["O", "t"]]],
 " process, letting the observed ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " remain at their known values and simulating the rest corresponds to \
estimation conditional on the realized ",
 Cell[BoxData[
  SubscriptBox["O", "t"]]],
 ". This is a sensible way to proceed."
}], "Text"],

Cell[TextData[{
 "In doing this, we are implicitly assuming that the ",
 Cell[BoxData[
  SubscriptBox["O", "t"]]],
 " process is independent of ",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " and ",
 Cell[BoxData[
  SubscriptBox["u", "t"]]],
 ". If buys are more likely to be observed than sells, or if the realization \
of ",
 Cell[BoxData[
  SubscriptBox["O", "t"]]],
 " depends on the magnitude of ",
 Cell[BoxData[
  SubscriptBox["u", "t"]]],
 " (\"Trades are more likely to be observed for large efficient price \
movements\"), then the ",
 Cell[BoxData[
  SubscriptBox["O", "t"]]],
 " process is informative, and we may wish to model it explicitly."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Modification when some of the q's are zero.", "Subsection"],

Cell[TextData[{
 "The U.S. daily CRSP dataset reports closing prices. But if there is no \
trade on day ",
 Cell[BoxData["t"]],
 ", then the reported price is the midpoint of the closing bid and ask. This \
event is indicated in the data (using a negative price). Essentially, ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["p", "t"], "=", 
   SubscriptBox["m", "t"]}]]],
 "."
}], "Text"],

Cell[TextData[{
 "Formally, we can incorporate this into the data generating process by \
noting that ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["p", "t"], "=", 
   RowBox[{
    RowBox[{
     SubscriptBox["m", "t"], "+", 
     RowBox[{"c", " ", 
      SubscriptBox["q", "t"]}]}], " ", "=", " ", 
    SubscriptBox["m", "t"]}]}]]],
 " when ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", "0"}]]],
 ". "
}], "Text"],

Cell[TextData[{
 "We can handle this situation in a fashion similar to the known-",
 Cell[BoxData[
  SubscriptBox["q", "t"]]],
 " case discussed above. If the price reported at time ",
 Cell[BoxData["t"]],
 " is a quote midpoint, we set ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox["q", "t"], "=", "0"}]]],
 ", and don't resample it during the sweeps. "
}], "Text"],

Cell[TextData[{
 "Formally, this can be justified by letting ",
 Cell[BoxData[
  SubscriptBox["O", "t"]]],
 " denote an indicator variable of whether or not there was a trade. \
Estimation can then proceed condtional on the ",
 Cell[BoxData[
  SubscriptBox["O", "t"]]],
 " realizations.  Here as well, we're implicitly assuming idependence. We're \
ruling out (or at least not modeling), for example, the possibility that \
trades are more likely to occur on days when there are large efficient price \
changes."
}], "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
 
 CounterBox["Section"],
 ". Full estimation of the Roll model"
}], "Section"],

Cell[CellGroupData[{

Cell[TextData[{
 "Sample runs from ",
 StyleBox["Roll Gibbs 01.sas",
  FontSlant->"Italic"]
}], "Subsection"],

Cell[TextData[{
 "In all cases, the prior on ",
 Cell[BoxData["c"]],
 " is ",
 Cell[BoxData[
  RowBox[{"N", 
   RowBox[{"(", 
    RowBox[{
     SubsuperscriptBox["\[Mu]", "c", "Prior"], ",", 
     SubsuperscriptBox["\[CapitalOmega]", "c", "Prior"]}], ")"}]}]]],
 ", restricted to ",
 Cell[BoxData[
  RowBox[{"c", ">", "0"}]]],
 ", with ",
 Cell[BoxData[
  RowBox[{
   SubsuperscriptBox["\[Mu]", "c", "Prior"], "=", "0"}]]],
 " and ",
 Cell[BoxData[
  RowBox[{
   SubsuperscriptBox["\[CapitalOmega]", "c", "Prior"], "=", "1"}]]],
 ". The prior on ",
 Cell[BoxData[
  SubsuperscriptBox["\[Sigma]", "u", "2"]]],
 " is inverted gamma with ",
 Cell[BoxData[
  RowBox[{"a", "=", 
   RowBox[{"b", "=", 
    RowBox[{"1", "\[Times]", 
     SuperscriptBox["10", 
      RowBox[{"-", "6"}]]}]}]}]]],
 "."
}], "Text"],

Cell[CellGroupData[{

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   RowBox[{
    SubscriptBox["\[Sigma]", "u"], "=", ".01"}], ";", " ", 
   RowBox[{"c", "=", "0.01"}]}]]],
 ", 100 observations, 20,000 sweeps (first 20% dropped)"
}], "Subsubsection"],

Cell["Posteriors:", "Text"],

Cell["\<\
Variable        N      Mean       Dev       Min       Max
---------------------------------------------------------
SWEEP       16000     12001      4619      4001     20000
SDU         16000    0.0129    0.0016    0.0088    0.0211
C           16000    0.0091    0.0014    0.0001    0.0132
---------------------------------------------------------\
\>", "Text",
 FontFamily->"Courier New",
 FontSize->10],

Cell[CellGroupData[{

Cell[GraphicsData["Metafile", "\<\
CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@000M6/0@0006`00000000000000?D4003/0`00
000000000037FP00idL00215CDH00040[=41018;000300000000000000000000@0H00;04000U0@00
g000000000000000000008Qh101PF`<02@000100003f1000k@<000X0000@000000000000000D0000
300000d0000U0000300000P0080W00006000004000000000oooo0000000U000030000040000[0000
6000000000000000mP@00>d3000[00006000000000000000mP@00>d3000V00007000008000000000
0@000000000000009@0000`000020000E`0002@0003k00002P000?`0000=00000P000?/03@3l00X0
E`0002@0003l00002P000?`0000?00000P000?`02P3l00l0E`0002@0003k00003@000?`0000?0000
0P000?`03`3k00d0E`0002@0003i00002`000?/0000=00000P000?/03@3i00/0E`0002@0003f0000
2P000?T0000;00000P000?T02`3f00X0E`0002@0003d00002P000?H0000:00000P000?H02P3d00X0
E`0002@0003a00002P000?@0000;00000P000?@02P3a00/0E`0002@0003_00002`000?40000=0000
0P000?402`3_00d0E`0002@0003_00003@000>l0000?00000P000>l03@3_00l0E`0002@0003_0000
3`000?00000@00000P000>l03`3`0100E`0002@0003`000040000?40000A00000P000?00403a0140
E`0002@0003a00004@000?<0000B00000P000?404@3c0180E`0002@0003c00004P000?P0000D0000
0P000?<04P3h01@0E`0002@0003h000050000?X0000E00000P000?P0503j01D0E`0002@0003j0000
5@000?`0000G00000P000?X05@3l01L0E`0002@0003_00003`000?40000@00000P000>l03`3a0100
E`0002@0003a000040000?<0000A00000P000?40403c0140E`0002@0003c00004@000?P0000C0000
0P000?<04@3h01<0E`0002@0003h00004`000?X0000D00000P000?P04`3j01@0E`0002@0003j0000
50000?/0000E00000P000?X0503k01D0E`0002@0003k00005@000?`0000G00000P000?/05@3l01L0
E`0002@0003l00005`000?`0000K00000P000?`05`3l01/0E`0002@0003j00006`000?`0000M0000
0P000?`06`3j01d0E`0002@0003g00007@000?X0000M00000P000?X07@3g01d0E`0002@0003e0000
7@000?L0000M00000P000?L07@3e01d0E`0002@0003b00007@000?D0000M00000P000?D07@3b01d0
E`0002@0003`00006`000?80000M00000P000?807@3`01/0E`0002@0003_000060000?00000K0000
0P000?006`3_01P0E`0002@0003_000060000>l0000M00000P000>l0603_01d0E`0002@0003_0000
6`000?00000M00000P000>l07@3`01/0E`0002@000020@002P0000<1000;00000P0000<12P020@/0
E`0002@000020@002`0000<1000<00000P0000812`030@`0E`0002@000030@002`0000@1000<0000
0P0000<130040@/0E`0002@000030@002P0000@1000;00000P0000@12`030@X0E`0002@000030@00
400000<1000M00000P0000<140030Ad0E`0002@000040@00400000@1000M00000P0000@140040Ad0
E`0002@000000@00400000@1000@00000P00000140040A00E`0002@000000@007@0000L1000M0000
0P0000017@070Ad0E`0002@0000=0@00400000d1000M00000P0000d1400=0Ad0E`0002@0000>0@00
400000h1000M00000P0000h1400>0Ad0E`0002@0000>0@004@000101000C00000P0000h14`0@0A40
E`0002@0000@0@0040000181000A00000P0001014@0B0A00E`0002@0000B0@00400001@1000@0000
0P000181400D0A00E`0002@0000D0@00400001L1000A00000P0001@1400G0A40E`0002@0000G0@00
4@0001P1000C00000P0001L14@0H0A<0E`0002@0000H0@004`0001P1000M00000P0001P14`0H0Ad0
E`0002@0000D0@00400001H1000A00000P0001@1400F0A40E`0002@0000F0@004@0001L1000C0000
0P0001H14@0G0A<0E`0002@0000G0@004`0001L1000M00000P0001L14`0G0Ad0E`0002@0000H0@00
4@0001T1000C00000P0001P14`0I0A40E`0002@0000I0@00400001`1000A00000P0001T14@0L0A00
E`0002@0000L0@00400001h1000@00000P0001`1400N0A00E`0002@0000N0@0040000241000A0000
0P0001h1400Q0A40E`0002@0000Q0@004@000281000C00000P0002414@0R0A<0E`0002@0000R0@00
4`000281000M00000P0002814`0R0Ad0E`0002@0000N0@0040000201000A00000P0001h1400P0A40
E`0002@0000P0@004@000241000C00000P0002014@0Q0A<0E`0002@0000Q0@004`000241000M0000
0P0002414`0Q0Ad0E`0002@0000:0@00400000h1000@00000P0000X1400>0A00E`0002@0000:0@00
7@000141000M00000P0000X17@0A0Ad0E`0002@0000D0@007@0001X1000M00000P0001@17@0J0Ad0
E`0002@0000N0@007@0002@1000M00000P0001h17@0T0Ad0E`0002@0000Z0@00400002X1000K0000
0P0002X1400Z0A/0E`0002@0000Z0@006`0002/1000M00000P0002X16`0[0Ad0E`0002@0000[0@00
7@0002h1000M00000P0002/17@0^0Ad0E`0002@0000^0@007@0002l1000M00000P0002h17@0_0Ad0
E`0002@0000_0@007@000381000M00000P0002l17@0b0Ad0E`0002@0000b0@006`0003@1000M0000
0P0003817@0d0A/0E`0002@0000[0@00400002/1000K00000P0002/1400[0A/0E`0002@0000[0@00
6`0002`1000M00000P0002/16`0/0Ad0E`0002@0000/0@007@0002h1000M00000P0002`17@0^0Ad0
E`0002@0000d0@00400003@1000M00000P0003@1400d0Ad0E`0002@0000e0@00400003D1000M0000
0P0003D1400e0Ad0E`0002@0000W0@00400002/1000@00000P0002L1400[0A00E`0002@0000a0@00
400003D1000@00000P000341400e0A00E`0002@0000d0@007@0003L1000M00000P0003@17@0g0Ad0
E`0002@0000n0@002P0003h1000M00000P0003h12P0n0Ad0E`0002@0000o0@002P0003l1000M0000
0P0003l12P0o0Ad0E`0002@0000k0@002P0003l1000:00000P0003/12P0o0@X0E`0002@0000k0@00
7@000481000M00000P0003/17@120Ad0E`0002@000170@004P0004L1000C00000P0004L14P170A<0
E`0002@000160@004`0004L1000C00000P0004L14`160A<0E`0002@000160@004P0004H1000C0000
0P0004H14`160A80E`0002@000160@004@0004L1000B00000P0004H14P170A40E`0002@000170@00
400004T1000A00000P0004L14@190A00E`0002@000190@00400004`1000@00000P0004T1401<0A00
E`0002@0001<0@00400004h1000A00000P0004`1401>0A40E`0002@0001>0@004@0004l1000B0000
0P0004h14@1?0A80E`0002@0001?0@004P000501000D00000P0004l14P1@0A@0E`0002@0001@0@00
50000501000K00000P000501501@0A/0E`0002@0001@0@006`000541000M00000P0005016`1A0Ad0
E`0002@0001A0@007@000581000M00000P0005417@1B0Ad0E`0002@0001?0@004P0004l1000K0000
0P0004l14P1?0A/0E`0002@0001?0@006`000501000M00000P0004l16`1@0Ad0E`0002@0001@0@00
7@000581000M00000P0005017@1B0Ad0E`0002@0001B0@007@0005<1000M00000P0005817@1C0Ad0
E`0002@0001>0@00500004l1000E00000P0004l1501>0AD0E`0002@000190@005@0004h1000F0000
0P0004h15@190AH0E`0002@000160@005P0004T1000G00000P0004T15P160AL0E`0002@000150@00
5`0004H1000I00000P0004H15`150AT0E`0002@000150@006@0004D1000K00000P0004D16@150A/0
E`0002@000150@006`0004H1000M00000P0004D16`160Ad0E`0002@000160@007@0004T1000M0000
0P0004H17@190Ad0E`0002@000190@007@0004/1000M00000P0004T17@1;0Ad0E`0002@0001;0@00
7@0004d1000M00000P0004/17@1=0Ad0E`0002@0001=0@006`0004l1000M00000P0004d17@1?0A/0
E`0002@000170@005P0004T1000G00000P0004T15P170AL0E`0002@000160@005`0004L1000I0000
0P0004L15`160AT0E`0002@000160@006@0004H1000K00000P0004H16@160A/0E`0002@000160@00
6`0004L1000M00000P0004H16`170Ad0E`0002@000170@007@0004T1000M00000P0004L17@190Ad0
E`0002@0001I0@002P0005T1000J00000P0005T12P1I0AX0E`0002@0001I0@006P0005X1000M0000
0P0005T16P1J0Ad0E`0002@0001J0@007@0005`1000M00000P0005X17@1L0Ad0E`0002@0001L0@00
7@0005d1000M00000P0005`17@1M0Ad0E`0002@0001M0@007@0005l1000M00000P0005d17@1O0Ad0
E`0002@0001O0@006`000601000M00000P0005l17@1P0A/0E`0002@0001J0@002P0005X1000J0000
0P0005X12P1J0AX0E`0002@0001J0@006P0005/1000M00000P0005X16P1K0Ad0E`0002@0001K0@00
7@0005`1000M00000P0005/17@1L0Ad0E`0002@0001F0@00400005d1000@00000P0005H1401M0A00
E`0002@0001V0@002P0006L1000;00000P0006L12P1V0@/0E`0002@0001V0@002`0006L1000<0000
0P0006H12`1W0@`0E`0002@0001W0@002`0006P1000<00000P0006L1301X0@/0E`0002@0001W0@00
2P0006P1000;00000P0006P12`1W0@X0E`0002@0001W0@00400006L1000M00000P0006L1401W0Ad0
E`0002@0001X0@00400006P1000M00000P0006P1401X0Ad0E`0002@0001T0@00400006P1000@0000
0P0006@1401X0A00E`0002@0001T0@007@0006/1000M00000P0006@17@1[0Ad0E`0002@0001a0@00
400007@1000A00000P0007@1401a0A40E`0002@0001_0@004@000741000C00000P0007414@1_0A<0
E`0002@0001^0@004`0006l1000F00000P0006l14`1^0AH0E`0002@0001^0@005P0006h1000H0000
0P0006h15P1^0AP0E`0002@0001^0@00600006l1000K00000P0006h1601_0A/0E`0002@0001_0@00
6`000741000M00000P0006l16`1a0Ad0E`0002@0001a0@007@0007@1000M00000P0007417@1d0Ad0
E`0002@0001d0@007@0007D1000M00000P0007@17@1e0Ad0E`0002@0001e0@007@0007P1000M0000
0P0007D17@1h0Ad0E`0002@0001h0@006`0007X1000M00000P0007P17@1j0A/0E`0002@0001j0@00
600007/1000K00000P0007X16`1k0AP0E`0002@0001k0@005P0007/1000H00000P0007/1601k0AH0
E`0002@0001j0@004`0007/1000F00000P0007/15P1j0A<0E`0002@0001h0@004@0007X1000C0000
0P0007X14`1h0A40E`0002@0001e0@00400007P1000A00000P0007P14@1e0A00E`0002@0001d0@00
400007D1000@00000P0007D1401d0A00E`0002@0001b0@00400007@1000A00000P0007@1401b0A40
E`0002@0001`0@004@000781000C00000P0007814@1`0A<0E`0002@0001_0@004`000701000F0000
0P0007014`1_0AH0E`0002@0001_0@005P0006l1000H00000P0006l15P1_0AP0E`0002@0001_0@00
60000701000K00000P0006l1601`0A/0E`0002@0001`0@006`000781000M00000P0007016`1b0Ad0
E`0002@0001b0@007@0007@1000M00000P0007817@1d0Ad0E`0002@0001e0@007@0007L1000M0000
0P0007D17@1g0Ad0E`0002@0001g0@006`0007T1000M00000P0007L17@1i0A/0E`0002@0001i0@00
600007X1000K00000P0007T16`1j0AP0E`0002@0001j0@005P0007X1000H00000P0007X1601j0AH0
E`0002@0001i0@004`0007X1000F00000P0007X15P1i0A<0E`0002@0001g0@004@0007T1000C0000
0P0007T14`1g0A40E`0002@0001e0@00400007L1000A00000P0007L14@1e0A00E`0002@000220@00
40000881000M00000P00088140220Ad0E`0002@000230@00400008<1000M00000P0008<140230Ad0
E`0002@000230@004@0008D1000C00000P0008<14`250A40E`0002@000250@00400008L1000A0000
0P0008D14@270A00E`0002@000270@00400008T1000@00000P0008L140290A00E`0002@000290@00
400008`1000A00000P0008T1402<0A40E`0002@0002<0@004@0008d1000C00000P0008`14@2=0A<0
E`0002@0002=0@004`0008d1000M00000P0008d14`2=0Ad0E`0002@000290@00400008/1000A0000
0P0008T1402;0A40E`0002@0002;0@004@0008`1000C00000P0008/14@2<0A<0E`0002@0002<0@00
4`0008`1000M00000P0008`14`2<0Ad0E`0002@0001o0@00400008<1000@00000P0007l140230A00
E`0002@0001o0@007@0008H1000M00000P0007l17@260Ad0E`0002@000290@007@0008l1000M0000
0P0008T17@2?0Ad0E`0002@0002O0@0040000:<1000M00000P0009l1402S0Ad0E`0002@0002P0@00
40000:<1000K00000P000:01402S0A/0E`0002@0002S0@0040000:H1000M00000P000:H1402S0Ad0
E`0002@0002V0@0040000:X1000M00000P000:H1402Z0Ad0E`0002@0002W0@0040000:X1000K0000
0P000:L1402Z0A/0E`0002@0002Z0@0040000:d1000M00000P000:d1402Z0Ad0E`0002@0002L0@00
40000:<1000@00000P0009`1402S0A00E`0002@0002[0@0040000;01000@00000P000:/1402`0A00
E`0002@0002d0@002P000;D1000;00000P000;D12P2d0@/0E`0002@0002d0@002`000;D1000<0000
0P000;@12`2e0@`0E`0002@0002e0@002`000;H1000<00000P000;D1302f0@/0E`0002@0002e0@00
2P000;H1000;00000P000;H12`2e0@X0E`0002@0002e0@0040000;D1000M00000P000;D1402e0Ad0
E`0002@0002f0@0040000;H1000M00000P000;H1402f0Ad0E`0002@0002b0@0040000;H1000@0000
0P000;81402f0A00E`0002@0002b0@007@000;T1000M00000P000;817@2i0Ad0E`0002@0002o0@00
2P000;l1000J00000P000;l12P2o0AX0E`0002@0002o0@006P000<01000M00000P000;l16P300Ad0
E`0002@000300@007@000<81000M00000P000<017@320Ad0E`0002@000320@007@000<<1000M0000
0P000<817@330Ad0E`0002@000330@007@000<D1000M00000P000<<17@350Ad0E`0002@000350@00
6`000<H1000M00000P000<D17@360A/0E`0002@000300@002P000<01000J00000P000<012P300AX0
E`0002@000300@006P000<41000M00000P000<016P310Ad0E`0002@000310@007@000<81000M0000
0P000<417@320Ad0E`0002@0002l0@0040000<<1000@00000P000;`140330A00E`0002@0003<0@00
2P000<`1000M00000P000<`12P3<0Ad0E`0002@0003=0@002P000<d1000M00000P000<d12P3=0Ad0
E`0002@0003=0@004@000<l1000C00000P000<d14`3?0A40E`0002@0003?0@0040000=41000A0000
0P000<l14@3A0A00E`0002@0003A0@0040000=<1000@00000P000=41403C0A00E`0002@0003C0@00
40000=H1000A00000P000=<1403F0A40E`0002@0003F0@004@000=L1000C00000P000=H14@3G0A<0
E`0002@0003G0@004`000=L1000M00000P000=L14`3G0Ad0E`0002@0003C0@0040000=D1000A0000
0P000=<1403E0A40E`0002@0003E0@004@000=H1000C00000P000=D14@3F0A<0E`0002@0003F0@00
4`000=H1000M00000P000=H14`3F0Ad0E`0002@000390@002P000<d1000:00000P000<T12P3=0@X0
E`0002@000390@007@000=01000M00000P000<T17@3@0Ad0E`0002@0003C0@007@000=T1000M0000
0P000=<17@3I0Ad0E`0002@0003a0@0040000?81000B00000P000?414P3b0A00E`0002@0003b0@00
40000?81000D00000P000?81403b0A@0E`0002@0003a0@004P000?81000D00000P000?81503a0A80
E`0002@0003`0@004@000?41000B00000P000?414P3`0A40E`0002@0003^0@0040000?01000A0000
0P000?014@3^0A00E`0002@0003[0@0040000>h1000@00000P000>h1403[0A00E`0002@0003Y0@00
40000>/1000A00000P000>/1403Y0A40E`0002@0003X0@004@000>T1000B00000P000>T14@3X0A80
E`0002@0003X0@004P000>P1000D00000P000>P14P3X0A@0E`0002@0003X0@0050000>T1000E0000
0P000>P1503Y0AD0E`0002@0003Y0@005@000>/1000F00000P000>T15@3[0AH0E`0002@0003[0@00
5P000>l1000H00000P000>/15P3_0AP0E`0002@0003_0@0060000?41000I00000P000>l1603a0AT0
E`0002@0003a0@006@000?81000J00000P000?416@3b0AX0E`0002@0003X0@004`000>T1000D0000
0P000>P14`3Y0A@0E`0002@0003Y0@0050000>/1000E00000P000>T1503[0AD0E`0002@0003[0@00
5@000>l1000G00000P000>/15@3_0AL0E`0002@0003_0@005`000?41000H00000P000>l15`3a0AP0
E`0002@0003a0@0060000?81000I00000P000?41603b0AT0E`0002@0003b0@006@000?81000L0000
0P000?816@3b0A`0E`0002@0003a0@0070000?81000M00000P000?81703a0Ad0E`0002@0003_0@00
7@000?41000M00000P000?417@3_0Ad0E`0002@0003/0@007@000>l1000M00000P000>l17@3/0Ad0
E`0002@0003Z0@007@000>`1000M00000P000>`17@3Z0Ad0E`0002@0003Y0@0070000>X1000M0000
0P000>X17@3Y0A`0E`0002@0003X0@006P000>T1000L00000P000>T1703X0AX0E`0002@0003X0@00
6P000>P1000M00000P000>P16P3X0Ad0E`0002@0003X0@0070000>T1000M00000P000>P17@3Y0A`0
E`0002@000020P002P000082000M00000P0000822P020Qd0E`0002@000030P002P0000<2000M0000
0P0000<22P030Qd0E`0002@000000P004@000082000C00000P0000824`000Q40E`0002@0003n0@00
40000002000A00000P0000024@3n0A00E`0002@0003m0@0040000?h1000@00000P000?h1403m0A00
E`0002@0003j0@0040000?d1000A00000P000?d1403j0A40E`0002@0003h0@004@000?X1000C0000
0P000?X14@3h0A<0E`0002@0003g0@004`000?P1000F00000P000?P14`3g0AH0E`0002@0003g0@00
5P000?L1000H00000P000?L15P3g0AP0E`0002@0003g0@0060000?P1000K00000P000?L1603h0A/0
E`0002@0003h0@006`000?X1000M00000P000?P16`3j0Ad0E`0002@0003j0@007@000?d1000M0000
0P000?X17@3m0Ad0E`0002@0003m0@007@000?h1000M00000P000?d17@3n0Ad0E`0002@0003n0@00
7@000002000M00000P000?h17@000Qd0E`0002@000000P006`000082000M00000P0000027@020Q/0
E`0002@0003k0@0040000?d1000A00000P000?d1403k0A40E`0002@0003i0@004@000?/1000C0000
0P000?/14@3i0A<0E`0002@0003h0@004`000?T1000F00000P000?T14`3h0AH0E`0002@0003h0@00
5P000?P1000H00000P000?P15P3h0AP0E`0002@0003h0@0060000?T1000K00000P000?P1603i0A/0
E`0002@0003i0@006`000?/1000M00000P000?T16`3k0Ad0E`0002@0003k0@007@000?d1000M0000
0P000?/17@3m0Ad0E`0002@0003o0@002P0000<2000:00000P000?l12P030PX0E`0002@000020P00
7@0000D2000M00000P0000827@050Qd0E`0002@0000<0P00400000`2000K00000P0000`2400<0Q/0
E`0002@0000<0P006`0000d2000M00000P0000`26`0=0Qd0E`0002@0000=0P007@000102000M0000
0P0000d27@0@0Qd0E`0002@0000@0P007@000142000M00000P0001027@0A0Qd0E`0002@0000A0P00
7@0001@2000M00000P0001427@0D0Qd0E`0002@0000D0P006`0001H2000M00000P0001@27@0F0Q/0
E`0002@0000=0P00400000d2000K00000P0000d2400=0Q/0E`0002@0000=0P006`0000h2000M0000
0P0000d26`0>0Qd0E`0002@0000>0P007@000102000M00000P0000h27@0@0Qd0E`0002@0000F0P00
400001H2000M00000P0001H2400F0Qd0E`0002@0000G0P00400001L2000M00000P0001L2400G0Qd0
E`0002@000090P00400000d2000@00000P0000T2400=0Q00E`0002@0000C0P00400001L2000@0000
0P0001<2400G0Q00E`0002@0000F0P007@0001T2000M00000P0001H27@0I0Qd0E`0002@0000O0P00
4P0002l2000B00000P0001l24P0_0Q80E`0002@0000O0P00600002l2000H00000P0001l2600_0QP0
E`0002@0000f0P00700003L2000M00000P0003L2700f0Qd0E`0002@0000f0P007@0003L2000M0000
0P0003H27@0g0Qd0E`0002@0000g0P007@0003P2000M00000P0003L27@0h0Qd0E`0002@0000g0P00
700003P2000M00000P0003P27@0g0Q`0E`0002@000110P002P0004@2000;00000P0004@22P110P/0
E`0002@0000o0P002`000442000>00000P0004422`0o0Ph0E`0002@0000n0P003P0003l2000B0000
0P0003l23P0n0Q80E`0002@0000n0P004P0003h2000E00000P0003h24P0n0QD0E`0002@0000n0P00
5@0003l2000J00000P0003h25@0o0QX0E`0002@0000o0P006P000442000M00000P0003l26P110Qd0
E`0002@000110P007@0004@2000M00000P0004427@140Qd0E`0002@000140P007@0004D2000M0000
0P0004@27@150Qd0E`0002@000150P007@0004P2000M00000P0004D27@180Qd0E`0002@000180P00
6P0004X2000M00000P0004P27@1:0QX0E`0002@0001:0P005@0004/2000J00000P0004X26P1;0QD0
E`0002@0001;0P004P0004/2000E00000P0004/25@1;0Q80E`0002@0001:0P003P0004/2000B0000
0P0004/24P1:0Ph0E`0002@000180P002`0004X2000>00000P0004X23P180P/0E`0002@000150P00
2P0004P2000;00000P0004P22`150PX0E`0002@000140P002P0004D2000:00000P0004D22P140PX0
E`0002@000120P002P0004@2000;00000P0004@22P120P/0E`0002@000110P002`000482000<0000
0P0004822`110P`0E`0002@000100P0030000442000>00000P00044230100Ph0E`0002@0000o0P00
3P000402000B00000P0004023P0o0Q80E`0002@0000o0P004P0003l2000E00000P0003l24P0o0QD0
E`0002@0000o0P005@000402000J00000P0003l25@100QX0E`0002@000100P006P000442000L0000
0P0004026P110Q`0E`0002@000110P0070000482000M00000P00044270120Qd0E`0002@000120P00
7@0004@2000M00000P0004827@140Qd0E`0002@000150P007@0004L2000M00000P0004D27@170Qd0
E`0002@000170P00700004P2000M00000P0004L27@180Q`0E`0002@000180P006P0004T2000L0000
0P0004P270190QX0E`0002@000190P005@0004X2000J00000P0004T26P1:0QD0E`0002@0001:0P00
4P0004X2000E00000P0004X25@1:0Q80E`0002@000190P003P0004X2000B00000P0004X24P190Ph0
E`0002@000180P00300004T2000>00000P0004T23P180P`0E`0002@000170P002`0004P2000<0000
0P0004P230170P/0E`0002@000150P002P0004L2000;00000P0004L22`150PX0E`0002@0001C0P00
3@0005D2000>00000P0005<23P1E0Pd0E`0002@0001E0P002P0005L2000=00000P0005D23@1G0PX0
E`0002@0001G0P002P0005L2000M00000P0005L22P1G0Qd0E`0002@0001F0P002`0005H2000M0000
0P0005H22`1F0Qd0E`0002@0001C0P007@0005/2000M00000P0005<27@1K0Qd0E`0002@0001]0P00
4P0006d2000C00000P0006d24P1]0Q<0E`0002@0001/0P004`0006d2000C00000P0006d24`1/0Q<0
E`0002@0001/0P004P0006`2000C00000P0006`24`1/0Q80E`0002@0001/0P004@0006d2000B0000
0P0006`24P1]0Q40E`0002@0001]0P00400006l2000A00000P0006d24@1_0Q00E`0002@0001_0P00
40000782000@00000P0006l2401b0Q00E`0002@0001b0P00400007@2000A00000P000782401d0Q40
E`0002@0001d0P004@0007D2000B00000P0007@24@1e0Q80E`0002@0001e0P004P0007H2000D0000
0P0007D24P1f0Q@0E`0002@0001f0P00500007H2000K00000P0007H2501f0Q/0E`0002@0001f0P00
6`0007L2000M00000P0007H26`1g0Qd0E`0002@0001g0P007@0007P2000M00000P0007L27@1h0Qd0
E`0002@0001e0P004P0007D2000K00000P0007D24P1e0Q/0E`0002@0001e0P006`0007H2000M0000
0P0007D26`1f0Qd0E`0002@0001f0P007@0007P2000M00000P0007H27@1h0Qd0E`0002@0001h0P00
7@0007T2000M00000P0007P27@1i0Qd0E`0002@0001d0P00500007D2000E00000P0007D2501d0QD0
E`0002@0001_0P005@0007@2000F00000P0007@25@1_0QH0E`0002@0001/0P005P0006l2000G0000
0P0006l25P1/0QL0E`0002@0001[0P005`0006`2000I00000P0006`25`1[0QT0E`0002@0001[0P00
6@0006/2000K00000P0006/26@1[0Q/0E`0002@0001[0P006`0006`2000M00000P0006/26`1/0Qd0
E`0002@0001/0P007@0006l2000M00000P0006`27@1_0Qd0E`0002@0001_0P007@000742000M0000
0P0006l27@1a0Qd0E`0002@0001a0P007@0007<2000M00000P0007427@1c0Qd0E`0002@0001c0P00
6`0007D2000M00000P0007<27@1e0Q/0E`0002@0001]0P005P0006l2000G00000P0006l25P1]0QL0
E`0002@0001/0P005`0006d2000I00000P0006d25`1/0QT0E`0002@0001/0P006@0006`2000K0000
0P0006`26@1/0Q/0E`0002@0001/0P006`0006d2000M00000P0006`26`1]0Qd0E`0002@0001]0P00
7@0006l2000M00000P0006d27@1_0Qd0E`0002@0001o0P00400007l2000M00000P0007l2401o0Qd0
E`0002@000200P0040000802000M00000P00080240200Qd0E`0002@000200P004@000882000C0000
0P0008024`220Q40E`0002@000220P00400008@2000A00000P0008824@240Q00E`0002@000240P00
400008H2000@00000P0008@240260Q00E`0002@000260P00400008T2000A00000P0008H240290Q40
E`0002@000290P004@0008X2000C00000P0008T24@2:0Q<0E`0002@0002:0P004`0008X2000M0000
0P0008X24`2:0Qd0E`0002@000260P00400008P2000A00000P0008H240280Q40E`0002@000280P00
4@0008T2000C00000P0008P24@290Q<0E`0002@000290P004`0008T2000M00000P0008T24`290Qd0
E`0002@0001l0P0040000802000@00000P0007`240200Q00E`0002@0001l0P007@0008<2000M0000
0P0007`27@230Qd0E`0002@000260P007@0008`2000M00000P0008H27@2<0Qd0E`0002@0002L0P00
2P0009`2000M00000P0009`22P2L0Qd0E`0002@0002M0P002P0009d2000M00000P0009d22P2M0Qd0
E`0002@0002J0P004@0009`2000C00000P0009`24`2J0Q40E`0002@0002H0P00400009X2000A0000
0P0009X24@2H0Q00E`0002@0002G0P00400009P2000@00000P0009P2402G0Q00E`0002@0002D0P00
400009L2000A00000P0009L2402D0Q40E`0002@0002B0P004@0009@2000C00000P0009@24@2B0Q<0
E`0002@0002A0P004`000982000F00000P0009824`2A0QH0E`0002@0002A0P005P000942000H0000
0P0009425P2A0QP0E`0002@0002A0P0060000982000K00000P000942602B0Q/0E`0002@0002B0P00
6`0009@2000M00000P0009826`2D0Qd0E`0002@0002D0P007@0009L2000M00000P0009@27@2G0Qd0
E`0002@0002G0P007@0009P2000M00000P0009L27@2H0Qd0E`0002@0002H0P007@0009X2000M0000
0P0009P27@2J0Qd0E`0002@0002J0P006`0009`2000M00000P0009X27@2L0Q/0E`0002@0002E0P00
400009L2000A00000P0009L2402E0Q40E`0002@0002C0P004@0009D2000C00000P0009D24@2C0Q<0
E`0002@0002B0P004`0009<2000F00000P0009<24`2B0QH0E`0002@0002B0P005P000982000H0000
0P0009825P2B0QP0E`0002@0002B0P00600009<2000K00000P000982602C0Q/0E`0002@0002C0P00
6`0009D2000M00000P0009<26`2E0Qd0E`0002@0002E0P007@0009L2000M00000P0009D27@2G0Qd0
E`0002@0002I0P002P0009d2000:00000P0009T22P2M0PX0E`0002@0002L0P007@0009l2000M0000
0P0009`27@2O0Qd0E`0002@0002h0P004`000;T2000D00000P000;T24`2h0Q@0E`0002@0002h0P00
50000;T2000E00000P000;P2502i0QD0E`0002@0002i0P0050000;X2000E00000P000;T25@2j0Q@0
E`0002@0002j0P004`000;X2000D00000P000;X2502j0Q<0E`0002@0002h0P004@000;X2000C0000
0P000;X24`2h0Q40E`0002@0002f0P0040000;P2000A00000P000;P24@2f0Q00E`0002@0002d0P00
40000;H2000@00000P000;H2402d0Q00E`0002@0002a0P0040000;@2000A00000P000;@2402a0Q40
E`0002@0002_0P004@000;42000C00000P000;424@2_0Q<0E`0002@0002^0P004`000:l2000F0000
0P000:l24`2^0QH0E`0002@0002^0P005P000:h2000H00000P000:h25P2^0QP0E`0002@0002^0P00
60000:l2000K00000P000:h2602_0Q/0E`0002@0002_0P006`000;42000M00000P000:l26`2a0Qd0
E`0002@0002a0P007@000;@2000M00000P000;427@2d0Qd0E`0002@0002d0P007@000;D2000M0000
0P000;@27@2e0Qd0E`0002@0002e0P007@000;P2000M00000P000;D27@2h0Qd0E`0002@0002h0P00
6`000;X2000M00000P000;P27@2j0Q/0E`0002@0002b0P0040000;@2000A00000P000;@2402b0Q40
E`0002@0002`0P004@000;82000C00000P000;824@2`0Q<0E`0002@0002_0P004`000;02000F0000
0P000;024`2_0QH0E`0002@0002_0P005P000:l2000H00000P000:l25P2_0QP0E`0002@0002_0P00
60000;02000K00000P000:l2602`0Q/0E`0002@0002`0P006`000;82000M00000P000;026`2b0Qd0
E`0002@0002b0P007@000;@2000M00000P000;827@2d0Qd0E`0002@0002o0P004P000<l2000B0000
0P000;l24P3?0Q80E`0002@0002o0P0060000<l2000H00000P000;l2603?0QP0E`0002@0003I0P00
2P000=`2000;00000P000=`22P3I0P/0E`0002@0003G0P002`000=T2000>00000P000=T22`3G0Ph0
E`0002@0003F0P003P000=L2000B00000P000=L23P3F0Q80E`0002@0003F0P004P000=H2000E0000
0P000=H24P3F0QD0E`0002@0003F0P005@000=L2000J00000P000=H25@3G0QX0E`0002@0003G0P00
6P000=T2000M00000P000=L26P3I0Qd0E`0002@0003I0P007@000=`2000M00000P000=T27@3L0Qd0
E`0002@0003L0P007@000=d2000M00000P000=`27@3M0Qd0E`0002@0003M0P007@000>02000M0000
0P000=d27@3P0Qd0E`0002@0003P0P006P000>82000M00000P000>027@3R0QX0E`0002@0003R0P00
5@000><2000J00000P000>826P3S0QD0E`0002@0003S0P004P000><2000E00000P000><25@3S0Q80
E`0002@0003R0P003P000><2000B00000P000><24P3R0Ph0E`0002@0003P0P002`000>82000>0000
0P000>823P3P0P/0E`0002@0003M0P002P000>02000;00000P000>022`3M0PX0E`0002@0003L0P00
2P000=d2000:00000P000=d22P3L0PX0E`0002@0003J0P002P000=`2000;00000P000=`22P3J0P/0
E`0002@0003I0P002`000=X2000<00000P000=X22`3I0P`0E`0002@0003H0P0030000=T2000>0000
0P000=T2303H0Ph0E`0002@0003G0P003P000=P2000B00000P000=P23P3G0Q80E`0002@0003G0P00
4P000=L2000E00000P000=L24P3G0QD0E`0002@0003G0P005@000=P2000J00000P000=L25@3H0QX0
E`0002@0003H0P006P000=T2000L00000P000=P26P3I0Q`0E`0002@0003I0P0070000=X2000M0000
0P000=T2703J0Qd0E`0002@0003J0P007@000=`2000M00000P000=X27@3L0Qd0E`0002@0003M0P00
7@000=l2000M00000P000=d27@3O0Qd0E`0002@0003O0P0070000>02000M00000P000=l27@3P0Q`0
E`0002@0003P0P006P000>42000L00000P000>02703Q0QX0E`0002@0003Q0P005@000>82000J0000
0P000>426P3R0QD0E`0002@0003R0P004P000>82000E00000P000>825@3R0Q80E`0002@0003Q0P00
3P000>82000B00000P000>824P3Q0Ph0E`0002@0003P0P0030000>42000>00000P000>423P3P0P`0
E`0002@0003O0P002`000>02000<00000P000>02303O0P/0E`0002@0003M0P002P000=l2000;0000
0P000=l22`3M0PX0E`0002@0003X0P0070000>T2000M00000P000>T2703X0Qd0E`0002@0003X0P00
7@000>T2000M00000P000>P27@3Y0Qd0E`0002@0003Y0P007@000>X2000M00000P000>T27@3Z0Qd0
E`0002@0003Y0P0070000>X2000M00000P000>X27@3Y0Q`0E`0002@0003c0P002P000?H2000;0000
0P000?H22P3c0P/0E`0002@0003a0P002`000?<2000>00000P000?<22`3a0Ph0E`0002@0003`0P00
3P000?42000B00000P000?423P3`0Q80E`0002@0003`0P004P000?02000E00000P000?024P3`0QD0
E`0002@0003`0P005@000?42000J00000P000?025@3a0QX0E`0002@0003a0P006P000?<2000M0000
0P000?426P3c0Qd0E`0002@0003c0P007@000?H2000M00000P000?<27@3f0Qd0E`0002@0003f0P00
7@000?L2000M00000P000?H27@3g0Qd0E`0002@0003g0P007@000?X2000M00000P000?L27@3j0Qd0
E`0002@0003j0P006P000?`2000M00000P000?X27@3l0QX0E`0002@0003l0P005@000?d2000J0000
0P000?`26P3m0QD0E`0002@0003m0P004P000?d2000E00000P000?d25@3m0Q80E`0002@0003l0P00
3P000?d2000B00000P000?d24P3l0Ph0E`0002@0003j0P002`000?`2000>00000P000?`23P3j0P/0
E`0002@0003g0P002P000?X2000;00000P000?X22`3g0PX0E`0002@0003f0P002P000?L2000:0000
0P000?L22P3f0PX0E`0002@0003d0P002P000?H2000;00000P000?H22P3d0P/0E`0002@0003c0P00
2`000?@2000<00000P000?@22`3c0P`0E`0002@0003b0P0030000?<2000>00000P000?<2303b0Ph0
E`0002@0003a0P003P000?82000B00000P000?823P3a0Q80E`0002@0003a0P004P000?42000E0000
0P000?424P3a0QD0E`0002@0003a0P005@000?82000J00000P000?425@3b0QX0E`0002@0003b0P00
6P000?<2000L00000P000?826P3c0Q`0E`0002@0003c0P0070000?@2000M00000P000?<2703d0Qd0
E`0002@0003d0P007@000?H2000M00000P000?@27@3f0Qd0E`0002@0003g0P007@000?T2000M0000
0P000?L27@3i0Qd0E`0002@0003i0P0070000?X2000M00000P000?T27@3j0Q`0E`0002@0003j0P00
6P000?/2000L00000P000?X2703k0QX0E`0002@0003k0P005@000?`2000J00000P000?/26P3l0QD0
E`0002@0003l0P004P000?`2000E00000P000?`25@3l0Q80E`0002@0003k0P003P000?`2000B0000
0P000?`24P3k0Ph0E`0002@0003j0P0030000?/2000>00000P000?/23P3j0P`0E`0002@0003i0P00
2`000?X2000<00000P000?X2303i0P/0E`0002@0003g0P002P000?T2000;00000P000?T22`3g0PX0
E`0002@000050`003@0000L3000>00000P0000D33P070`d0E`0002@000070`002P0000T3000=0000
0P0000L33@090`X0E`0002@000090`002P0000T3000M00000P0000T32P090ad0E`0002@000080`00
2`0000P3000M00000P0000P32`080ad0E`0002@000050`007@0000d3000M00000P0000D37@0=0ad0
E`0002@0000E0`00400001H3000A00000P0001H3400E0a40E`0002@0000E0`004@0001H3000B0000
0P0001D34@0F0a80E`0002@0000F0`004@0001L3000B00000P0001H34P0G0a40E`0002@0000F0`00
400001L3000A00000P0001L34@0F0a00E`0002@0000E0`00700001H3000M00000P0001H3700E0ad0
E`0002@0000E0`007@0001H3000M00000P0001D37@0F0ad0E`0002@0000F0`007@0001L3000M0000
0P0001H37@0G0ad0E`0002@0000F0`00700001L3000M00000P0001L37@0F0a`0E`0002@0000Y0`00
2P0002T3000M00000P0002T32P0Y0ad0E`0002@0000Z0`002P0002X3000M00000P0002X32P0Z0ad0
E`0002@0000V0`002P000343000:00000P0002H32P0a0`X0E`0002@0000a0`002P0003@3000;0000
0P0003432P0d0`/0E`0002@0000d0`002`0003D3000<00000P0003@32`0e0``0E`0002@0000e0`00
300003D3000>00000P0003D3300e0`h0E`0002@0000e0`003P0003D3000@00000P0003D33P0e0a00
E`0002@0000e0`00400003D3000B00000P0003D3400e0a80E`0002@0000d0`004P0003D3000C0000
0P0003D34P0d0a<0E`0002@0000a0`004`0003@3000D00000P0003@34`0a0a@0E`0002@0000Z0`00
50000343000D00000P000343500Z0a@0E`0002@0000a0`002P0003<3000;00000P0003432P0c0`/0
E`0002@0000c0`002`0003@3000<00000P0003<32`0d0``0E`0002@0000d0`00300003D3000>0000
0P0003@3300e0`h0E`0002@0000e0`003P0003D3000@00000P0003D33P0e0a00E`0002@0000d0`00
400003D3000B00000P0003D3400d0a80E`0002@0000c0`004P0003@3000C00000P0003@34P0c0a<0
E`0002@0000a0`004`0003<3000D00000P0003<34`0a0a@0E`0002@0000V0`007@0002d3000M0000
0P0002H37@0]0ad0E`0002@0000m0`0040000403000A00000P000403400m0a40E`0002@0000k0`00
4@0003d3000C00000P0003d34@0k0a<0E`0002@0000j0`004`0003/3000F00000P0003/34`0j0aH0
E`0002@0000j0`005P0003X3000H00000P0003X35P0j0aP0E`0002@0000j0`00600003/3000K0000
0P0003X3600k0a/0E`0002@0000k0`006`0003d3000M00000P0003/36`0m0ad0E`0002@0000m0`00
7@000403000M00000P0003d37@100ad0E`0002@000100`007@000443000M00000P0004037@110ad0
E`0002@000110`007@0004@3000M00000P0004437@140ad0E`0002@000140`006`0004H3000M0000
0P0004@37@160a/0E`0002@000160`00600004L3000K00000P0004H36`170aP0E`0002@000170`00
5P0004L3000H00000P0004L360170aH0E`0002@000160`004`0004L3000F00000P0004L35P160a<0
E`0002@000140`004@0004H3000C00000P0004H34`140a40E`0002@000110`00400004@3000A0000
0P0004@34@110a00E`0002@000100`0040000443000@00000P00044340100a00E`0002@0000n0`00
40000403000A00000P000403400n0a40E`0002@0000l0`004@0003h3000C00000P0003h34@0l0a<0
E`0002@0000k0`004`0003`3000F00000P0003`34`0k0aH0E`0002@0000k0`005P0003/3000H0000
0P0003/35P0k0aP0E`0002@0000k0`00600003`3000K00000P0003/3600l0a/0E`0002@0000l0`00
6`0003h3000M00000P0003`36`0n0ad0E`0002@0000n0`007@000403000M00000P0003h37@100ad0
E`0002@000110`007@0004<3000M00000P0004437@130ad0E`0002@000130`006`0004D3000M0000
0P0004<37@150a/0E`0002@000150`00600004H3000K00000P0004D36`160aP0E`0002@000160`00
5P0004H3000H00000P0004H360160aH0E`0002@000150`004`0004H3000F00000P0004H35P150a<0
E`0002@000130`004@0004D3000C00000P0004D34`130a40E`0002@000110`00400004<3000A0000
0P0004<34@110a00E`0002@0001E0`00400005H3000B00000P0005D34P1F0a00E`0002@0001F0`00
400005H3000D00000P0005H3401F0a@0E`0002@0001E0`004P0005H3000D00000P0005H3501E0a80
E`0002@0001D0`004@0005D3000B00000P0005D34P1D0a40E`0002@0001B0`00400005@3000A0000
0P0005@34@1B0a00E`0002@0001?0`0040000583000@00000P000583401?0a00E`0002@0001=0`00
400004l3000A00000P0004l3401=0a40E`0002@0001<0`004@0004d3000B00000P0004d34@1<0a80
E`0002@0001<0`004P0004`3000D00000P0004`34P1<0a@0E`0002@0001<0`00500004d3000E0000
0P0004`3501=0aD0E`0002@0001=0`005@0004l3000F00000P0004d35@1?0aH0E`0002@0001?0`00
5P0005<3000H00000P0004l35P1C0aP0E`0002@0001C0`00600005D3000I00000P0005<3601E0aT0
E`0002@0001E0`006@0005H3000J00000P0005D36@1F0aX0E`0002@0001<0`004`0004d3000D0000
0P0004`34`1=0a@0E`0002@0001=0`00500004l3000E00000P0004d3501?0aD0E`0002@0001?0`00
5@0005<3000G00000P0004l35@1C0aL0E`0002@0001C0`005`0005D3000H00000P0005<35`1E0aP0
E`0002@0001E0`00600005H3000I00000P0005D3601F0aT0E`0002@0001F0`006@0005H3000L0000
0P0005H36@1F0a`0E`0002@0001E0`00700005H3000M00000P0005H3701E0ad0E`0002@0001C0`00
7@0005D3000M00000P0005D37@1C0ad0E`0002@0001@0`007@0005<3000M00000P0005<37@1@0ad0
E`0002@0001>0`007@000503000M00000P0005037@1>0ad0E`0002@0001=0`00700004h3000M0000
0P0004h37@1=0a`0E`0002@0001<0`006P0004d3000L00000P0004d3701<0aX0E`0002@0001<0`00
6P0004`3000M00000P0004`36P1<0ad0E`0002@0001<0`00700004d3000M00000P0004`37@1=0a`0
E`0002@0001M0`002P0005d3000J00000P0005d32P1M0aX0E`0002@0001M0`006P0005h3000M0000
0P0005d36P1N0ad0E`0002@0001N0`007@000603000M00000P0005h37@1P0ad0E`0002@0001P0`00
7@000643000M00000P0006037@1Q0ad0E`0002@0001Q0`007@0006<3000M00000P0006437@1S0ad0
E`0002@0001S0`006`0006@3000M00000P0006<37@1T0a/0E`0002@0001N0`002P0005h3000J0000
0P0005h32P1N0aX0E`0002@0001N0`006P0005l3000M00000P0005h36P1O0ad0E`0002@0001O0`00
7@000603000M00000P0005l37@1P0ad0E`0002@0001J0`0040000643000@00000P0005X3401Q0a00
E`0002@0001Z0`005P0007D3000F00000P0006X35P1e0aH0E`0002@0001e0`00500007D3000F0000
0P0007D35P1e0a@0E`0002@0001d0`004P0007D3000D00000P0007D3501d0a80E`0002@0001c0`00
4@0007@3000B00000P0007@34P1c0a40E`0002@0001a0`00400007<3000A00000P0007<34@1a0a00
E`0002@0001_0`0040000743000@00000P000743401_0a00E`0002@0001/0`00400006l3000A0000
0P0006l3401/0a40E`0002@0001Z0`004@0006`3000C00000P0006`34@1Z0a<0E`0002@0001Y0`00
4`0006X3000F00000P0006X34`1Y0aH0E`0002@0001Y0`005P0006T3000H00000P0006T35P1Y0aP0
E`0002@0001Y0`00600006X3000K00000P0006T3601Z0a/0E`0002@0001Z0`006`0006`3000M0000
0P0006X36`1/0ad0E`0002@0001/0`007@0006l3000M00000P0006`37@1_0ad0E`0002@0001_0`00
7@000703000M00000P0006l37@1`0ad0E`0002@0001`0`007@0007<3000M00000P0007037@1c0ad0
E`0002@0001c0`006`0007D3000M00000P0007<37@1e0a/0E`0002@0001d0`004`0007@3000F0000
0P0007@35P1d0a<0E`0002@0001c0`004@0007@3000C00000P0007@34`1c0a40E`0002@0001]0`00
400006l3000A00000P0006l3401]0a40E`0002@0001[0`004@0006d3000C00000P0006d34@1[0a<0
E`0002@0001Z0`004`0006/3000F00000P0006/34`1Z0aH0E`0002@0001Z0`005P0006X3000H0000
0P0006X35P1Z0aP0E`0002@0001Z0`00600006/3000K00000P0006X3601[0a/0E`0002@0001[0`00
6`0006d3000M00000P0006/36`1]0ad0E`0002@0001]0`007@0006l3000M00000P0006d37@1_0ad0
E`0002@0001l0`00400007`3000M00000P0007`3401l0ad0E`0002@0001m0`00400007d3000M0000
0P0007d3401m0ad0E`0002@0001m0`004`0007h3000F00000P0007d35P1n0a<0E`0002@0001n0`00
4@000803000C00000P0007h34`200a40E`0002@000200`0040000843000A00000P0008034@210a00
E`0002@000210`00400008@3000@00000P00084340240a00E`0002@000240`00400008D3000A0000
0P0008@340250a40E`0002@000250`004@0008D3000B00000P0008D34@250a80E`0002@000240`00
4P0008D3000C00000P0008D34P240a<0E`0002@000230`004P0008@3000C00000P0008@34`230a80
E`0002@000230`004@0008@3000B00000P0008<34P240a40E`0002@0001i0`00400007d3000@0000
0P0007T3401m0a00E`0002@0001i0`007@000803000M00000P0007T37@200ad0E`0002@0002:0`00
2P0008/3000;00000P0008/32P2:0`/0E`0002@0002:0`002`0008/3000<00000P0008X32`2;0``0
E`0002@0002;0`002`0008`3000<00000P0008/3302<0`/0E`0002@0002;0`002P0008`3000;0000
0P0008`32`2;0`X0E`0002@0002;0`00400008/3000M00000P0008/3402;0ad0E`0002@0002<0`00
400008`3000M00000P0008`3402<0ad0E`0002@000280`00400008`3000@00000P0008P3402<0a00
E`0002@000280`007@0008l3000M00000P0008P37@2?0ad0E`0002@0002F0`00400009T3000A0000
0P0009T3402F0a40E`0002@0002D0`004@0009H3000C00000P0009H34@2D0a<0E`0002@0002C0`00
4`0009@3000F00000P0009@34`2C0aH0E`0002@0002C0`005P0009<3000H00000P0009<35P2C0aP0
E`0002@0002C0`00600009@3000K00000P0009<3602D0a/0E`0002@0002D0`006`0009H3000M0000
0P0009@36`2F0ad0E`0002@0002F0`007@0009T3000M00000P0009H37@2I0ad0E`0002@0002I0`00
7@0009X3000M00000P0009T37@2J0ad0E`0002@0002J0`007@0009d3000M00000P0009X37@2M0ad0
E`0002@0002M0`006`0009l3000M00000P0009d37@2O0a/0E`0002@0002O0`0060000:03000K0000
0P0009l36`2P0aP0E`0002@0002P0`005P000:03000H00000P000:03602P0aH0E`0002@0002O0`00
4`000:03000F00000P000:035P2O0a<0E`0002@0002M0`004@0009l3000C00000P0009l34`2M0a40
E`0002@0002J0`00400009d3000A00000P0009d34@2J0a00E`0002@0002I0`00400009X3000@0000
0P0009X3402I0a00E`0002@0002G0`00400009T3000A00000P0009T3402G0a40E`0002@0002E0`00
4@0009L3000C00000P0009L34@2E0a<0E`0002@0002D0`004`0009D3000F00000P0009D34`2D0aH0
E`0002@0002D0`005P0009@3000H00000P0009@35P2D0aP0E`0002@0002D0`00600009D3000K0000
0P0009@3602E0a/0E`0002@0002E0`006`0009L3000M00000P0009D36`2G0ad0E`0002@0002G0`00
7@0009T3000M00000P0009L37@2I0ad0E`0002@0002J0`007@0009`3000M00000P0009X37@2L0ad0
E`0002@0002L0`006`0009h3000M00000P0009`37@2N0a/0E`0002@0002N0`00600009l3000K0000
0P0009h36`2O0aP0E`0002@0002O0`005P0009l3000H00000P0009l3602O0aH0E`0002@0002N0`00
4`0009l3000F00000P0009l35P2N0a<0E`0002@0002L0`004@0009h3000C00000P0009h34`2L0a40
E`0002@0002J0`00400009`3000A00000P0009`34@2J0a00E`0002@0002W0`0040000:L3000M0000
0P000:L3402W0ad0E`0002@0002X0`0040000:P3000M00000P000:P3402X0ad0E`0002@0002X0`00
4`000:T3000F00000P000:P35P2Y0a<0E`0002@0002Y0`004@000:/3000C00000P000:T34`2[0a40
E`0002@0002[0`0040000:`3000A00000P000:/34@2/0a00E`0002@0002/0`0040000:l3000@0000
0P000:`3402_0a00E`0002@0002_0`0040000;03000A00000P000:l3402`0a40E`0002@0002`0`00
4@000;03000B00000P000;034@2`0a80E`0002@0002_0`004P000;03000C00000P000;034P2_0a<0
E`0002@0002^0`004P000:l3000C00000P000:l34`2^0a80E`0002@0002^0`004@000:l3000B0000
0P000:h34P2_0a40E`0002@0002T0`0040000:P3000@00000P000:@3402X0a00E`0002@0002T0`00
7@000:/3000M00000P000:@37@2[0ad0E`0002@000340`002`000<@3000<00000P000<@32`340``0
E`0002@000340`0030000<@3000=00000P000<@330340`d0E`0002@000340`0030000<D3000=0000
0P000<@33@350``0E`0002@000350`002`000<D3000<00000P000<D330350`/0E`0002@000340`00
2P000<D3000;00000P000<D32`340`X0E`0002@000330`002P000<@3000:00000P000<@32P330`X0
E`0002@000310`002P000<<3000;00000P000<<32P310`/0E`0002@000300`002`000<43000=0000
0P000<432`300`d0E`0002@000300`003@000<03000M00000P000<033@300ad0E`0002@000320`00
2P000<<3000;00000P000<<32P320`/0E`0002@000310`002`000<83000=00000P000<832`310`d0
E`0002@000310`003@000<43000M00000P000<433@310ad0E`0002@0002m0`0040000<@3000@0000
0P000;d340340a00E`0002@0002m0`007@000<@3000M00000P000;d37@340ad0E`0002@0003<0`00
40000<l3000A00000P000<l3403<0a40E`0002@0003:0`004@000<`3000C00000P000<`34@3:0a<0
E`0002@000390`004`000<X3000F00000P000<X34`390aH0E`0002@000390`005P000<T3000H0000
0P000<T35P390aP0E`0002@000390`0060000<X3000K00000P000<T3603:0a/0E`0002@0003:0`00
6`000<`3000M00000P000<X36`3<0ad0E`0002@0003<0`007@000<l3000M00000P000<`37@3?0ad0
E`0002@0003?0`007@000=03000M00000P000<l37@3@0ad0E`0002@0003@0`007@000=<3000M0000
0P000=037@3C0ad0E`0002@0003C0`006`000=D3000M00000P000=<37@3E0a/0E`0002@0003E0`00
60000=H3000K00000P000=D36`3F0aP0E`0002@0003F0`005P000=H3000H00000P000=H3603F0aH0
E`0002@0003E0`004`000=H3000F00000P000=H35P3E0a<0E`0002@0003C0`004@000=D3000C0000
0P000=D34`3C0a40E`0002@0003@0`0040000=<3000A00000P000=<34@3@0a00E`0002@0003?0`00
40000=03000@00000P000=03403?0a00E`0002@0003=0`0040000<l3000A00000P000<l3403=0a40
E`0002@0003;0`004@000<d3000C00000P000<d34@3;0a<0E`0002@0003:0`004`000</3000F0000
0P000</34`3:0aH0E`0002@0003:0`005P000<X3000H00000P000<X35P3:0aP0E`0002@0003:0`00
60000</3000K00000P000<X3603;0a/0E`0002@0003;0`006`000<d3000M00000P000</36`3=0ad0
E`0002@0003=0`007@000<l3000M00000P000<d37@3?0ad0E`0002@0003@0`007@000=83000M0000
0P000=037@3B0ad0E`0002@0003B0`006`000=@3000M00000P000=837@3D0a/0E`0002@0003D0`00
60000=D3000K00000P000=@36`3E0aP0E`0002@0003E0`005P000=D3000H00000P000=D3603E0aH0
E`0002@0003D0`004`000=D3000F00000P000=D35P3D0a<0E`0002@0003B0`004@000=@3000C0000
0P000=@34`3B0a40E`0002@0003@0`0040000=83000A00000P000=834@3@0a00E`0002@0003M0`00
40000=d3000M00000P000=d3403M0ad0E`0002@0003N0`0040000=h3000M00000P000=h3403N0ad0
E`0002@0003N0`004`000=l3000F00000P000=h35P3O0a<0E`0002@0003O0`004@000>43000C0000
0P000=l34`3Q0a40E`0002@0003Q0`0040000>83000A00000P000>434@3R0a00E`0002@0003R0`00
40000>D3000@00000P000>83403U0a00E`0002@0003U0`0040000>H3000A00000P000>D3403V0a40
E`0002@0003V0`004@000>H3000B00000P000>H34@3V0a80E`0002@0003U0`004P000>H3000C0000
0P000>H34P3U0a<0E`0002@0003T0`004P000>D3000C00000P000>D34`3T0a80E`0002@0003T0`00
4@000>D3000B00000P000>@34P3U0a40E`0002@0003J0`0040000=h3000@00000P000=X3403N0a00
E`0002@0003J0`007@000>43000M00000P000=X37@3Q0ad0E`0002@0000110003@000084000?0000
0P0000443@0210l0E`0002@0000210002P000084000?00000P0000843`0210X0E`0002@000011000
2P000084000=00000P0000842P0110d0E`0002@0003o0`002`000044000=00000P0000443@3o0`/0
E`0002@0003l0`002P000?l3000;00000P000?l32`3l0`X0E`0002@0003j0`002P000?`3000:0000
0P000?`32P3j0`X0E`0002@0003h0`002P000?X3000;00000P000?X32P3h0`/0E`0002@0003f0`00
2`000?P3000=00000P000?P32`3f0`d0E`0002@0003e0`003@000?H3000?00000P000?H33@3e0`l0
E`0002@0003d0`003`000?D3000A00000P000?D33`3d0a40E`0002@0003d0`004@000?@3000F0000
0P000?@34@3d0aH0E`0002@0003d0`005P000?D3000I00000P000?@35P3e0aT0E`0002@0003e0`00
6@000?H3000K00000P000?D36@3f0a/0E`0002@0003f0`006`000?P3000M00000P000?H36`3h0ad0
E`0002@0003h0`007@000?X3000M00000P000?P37@3j0ad0E`0002@0003j0`007@000?`3000M0000
0P000?X37@3l0ad0E`0002@0003l0`007@000?l3000M00000P000?`37@3o0ad0E`0002@0003o0`00
6`000044000M00000P000?l37@0111/0E`0002@0000110006@000084000K00000P0000446`0211T0
E`0002@0003i0`002P000?X3000;00000P000?X32P3i0`/0E`0002@0003g0`002`000?T3000=0000
0P000?T32`3g0`d0E`0002@0003f0`003@000?L3000?00000P000?L33@3f0`l0E`0002@0003e0`00
3`000?H3000A00000P000?H33`3e0a40E`0002@0003e0`004@000?D3000F00000P000?D34@3e0aH0
E`0002@0003e0`005P000?H3000I00000P000?D35P3f0aT0E`0002@0003f0`006@000?L3000K0000
0P000?H36@3g0a/0E`0002@0003g0`006`000?T3000M00000P000?L36`3i0ad0E`0002@0003i0`00
7@000?X3000M00000P000?T37@3j0ad0E`0002@000250000O0<008X0001l0`000P0008X0O0>507`3
E`0002@000270000J@<008X0001Y0`000P0008L0J@>:06T3E`0002@000270000E@<008X0001E0`00
0P0008L0E@>:05D3E`0002@000270000@P<008X000120`000P0008L0@P>:0483E`0002@000270000
;P<008X0000^0`000P0008L0;P>:02h3E`0002@0002700006`<008X0000K0`000P0008L06`>:01/3
E`0002@0002700001`<008X000070`000P0008L01`>:00L3E`0002@000270000m08008X0003d0P00
0P0008L0m0::0?@2E`0002@000270000h08008X0003P0P000P0008L0h0::0>02E`0002@000270000
c@8008X0003=0P000P0008L0c@::0<d2E`0002@000250000^@8008X0002i0P000P0008X0^@:50;T2
E`0002@000270000YP8008X0002V0P000P0008L0YP::0:H2E`0002@000270000TP8008X0002B0P00
0P0008L0TP::0982E`0002@000270000O`8008X0001o0P000P0008L0O`::07l2E`0002@000270000
J`8008X0001[0P000P0008L0J`::06/2E`0002@000270000F08008X0001H0P000P0008L0F0::05P2
E`0002@000270000A08008X000140P000P0008L0A0::04@2E`0002@000270000<@8008X0000a0P00
0P0008L0<@::0342E`0002@0002700007@8008X0000M0P000P0008L07@::01d2E`0002@000270000
2P8008X0000:0P000P0008L02P::00X2E`0002@000250000mP4008X0003f0@000P0008X0mP650?H1
E`0002@000270000hP4008X0003R0@000P0008L0hP6:0>81E`0002@000270000c`4008X0003?0@00
0P0008L0c`6:0<l1E`0002@000270000^`4008X0002k0@000P0008L0^`6:0;/1E`0002@000270000
Z04008X0002X0@000P0008L0Z06:0:P1E`0002@000270000U04008X0002D0@000P0008L0U06:09@1
E`0002@000270000P@4008X000210@000P0008L0P@6:0841E`0002@000270000K@4008X0001]0@00
0P0008L0K@6:06d1E`0002@000270000FP4008X0001J0@000P0008L0FP6:05X1E`0002@000270000
AP4008X000160@000P0008L0AP6:04H1E`0002@000250000<`4008X0000c0@000P0008X0<`6503<1
E`0002@0002700007`4008X0000O0@000P0008L07`6:01l1E`0002@000270000304008X0000<0@00
0P0008L0306:00`1E`0002@000270000n00008X0003h00000P0008L0n02:0?P0E`0002@000270000
i@0008X0003U00000P0008L0i@2:0>D0E`0002@000270000d@0008X0003A00000P0008L0d@2:0=40
E`0002@000270000_P0008X0002n00000P0008L0_P2:0;h0E`0002@000270000ZP0008X0002Z0000
0P0008L0ZP2:0:X0E`0002@000270000U`0008X0002G00000P0008L0U`2:09L0E`0002@000270000
P`0008X0002300000P0008L0P`2:08<0E`0002@000250000L00008X0001`00000P0008X0L0250700
E`0002@0000E0000AP0001D0001I00000P0001D0AP0E05T0E`0002@0000F0000AP0001H0001I0000
0P0001H0AP0F05T0E`0002@0000B0000AP0001/0001600000P000180AP0K04H0E`0002@0000K0000
AP0001h0001700000P0001/0AP0N04L0E`0002@0000N0000A`000200001900000P0001h0A`0P04T0
E`0002@0000P0000B@000240001;00000P000200B@0Q04/0E`0002@0000Q0000B`000240001=0000
0P000240B`0Q04d0E`0002@0000Q0000C@000240001B00000P000240C@0Q0580E`0002@0000Q0000
DP000240001E00000P000240DP0Q05D0E`0002@0000P0000E@000240001G00000P000240E@0P05L0
E`0002@0000N0000E`000200001I00000P000200E`0N05T0E`0002@0000K0000F@0001h0001I0000
0P0001h0F@0K05T0E`0002@0000B0000F@0001/0001I00000P0001/0F@0B05T0E`0002@0000K0000
AP0001d0001700000P0001/0AP0M04L0E`0002@0000M0000A`0001l0001900000P0001d0A`0O04T0
E`0002@0000O0000B@000200001;00000P0001l0B@0P04/0E`0002@0000P0000B`000240001=0000
0P000200B`0Q04d0E`0002@0000Q0000C@000240001B00000P000240C@0Q0580E`0002@0000P0000
DP000240001E00000P000240DP0P05D0E`0002@0000O0000E@000200001G00000P000200E@0O05L0
E`0002@0000M0000E`0001l0001I00000P0001l0E`0M05T0E`0002@0000K0000F@0001d0001I0000
0P0001d0F@0K05T0E`0002@0000W0000DP000380001B00000P0002L0DP0b0580E`0002@0000b0000
D0000380001B00000P000380DP0b0500E`0002@0000a0000CP000380001@00000P000380D00a04h0
E`0002@0000`0000C@000340001>00000P000340CP0`04d0E`0002@0000^0000C0000300001=0000
0P000300C@0^04`0E`0002@0000/0000C00002h0001<00000P0002h0C00/04`0E`0002@0000Y0000
C00002`0001=00000P0002`0C00Y04d0E`0002@0000W0000C@0002T0001?00000P0002T0C@0W04l0
E`0002@0000V0000C`0002L0001B00000P0002L0C`0V0580E`0002@0000V0000DP0002H0001D0000
0P0002H0DP0V05@0E`0002@0000V0000E00002L0001G00000P0002H0E00W05L0E`0002@0000W0000
E`0002T0001I00000P0002L0E`0Y05T0E`0002@0000Y0000F@0002`0001I00000P0002T0F@0/05T0
E`0002@0000/0000F@0002d0001I00000P0002`0F@0]05T0E`0002@0000]0000F@000300001I0000
0P0002d0F@0`05T0E`0002@0000`0000E`000380001I00000P000300F@0b05L0E`0002@0000a0000
C`000340001B00000P000340DP0a04l0E`0002@0000`0000C@000340001?00000P000340C`0`04d0
E`0002@0000Z0000C00002`0001=00000P0002`0C00Z04d0E`0002@0000X0000C@0002X0001?0000
0P0002X0C@0X04l0E`0002@0000W0000C`0002P0001B00000P0002P0C`0W0580E`0002@0000W0000
DP0002L0001D00000P0002L0DP0W05@0E`0002@0000W0000E00002P0001G00000P0002L0E00X05L0
E`0002@0000X0000E`0002X0001I00000P0002P0E`0Z05T0E`0002@0000Z0000F@0002`0001I0000
0P0002X0F@0/05T0E`0002@0000i0000C00003T0001I00000P0003T0C00i05T0E`0002@0000j0000
C00003X0001I00000P0003X0C00j05T0E`0002@0000j0000C@0003`0001?00000P0003X0C`0l04d0
E`0002@0000l0000C00003h0001=00000P0003`0C@0n04`0E`0002@0000n0000C0000400001<0000
0P0003h0C01004`0E`0002@000100000C00004<0001=00000P000400C01304d0E`0002@000130000
C@0004@0001?00000P0004<0C@1404l0E`0002@000140000C`0004@0001I00000P0004@0C`1405T0
E`0002@000100000C0000480001=00000P000400C01204d0E`0002@000120000C@0004<0001?0000
0P000480C@1304l0E`0002@000130000C`0004<0001I00000P0004<0C`1305T0E`0002@0000f0000
C00003X0001<00000P0003H0C00j04`0E`0002@0000f0000F@0003d0001I00000P0003H0F@0m05T0
E`0002@000100000F@0004H0001I00000P000400F@1605T0E`0002@0001D0000C00005D0001>0000
0P0005@0CP1E04`0E`0002@0001E0000C00005D0001@00000P0005D0C01E0500E`0002@0001D0000
CP0005D0001@00000P0005D0D01D04h0E`0002@0001C0000C@0005@0001>00000P0005@0CP1C04d0
E`0002@0001A0000C00005<0001=00000P0005<0C@1A04`0E`0002@0001>0000C0000540001<0000
0P000540C01>04`0E`0002@0001<0000C00004h0001=00000P0004h0C01<04d0E`0002@0001;0000
C@0004`0001>00000P0004`0C@1;04h0E`0002@0001;0000CP0004/0001@00000P0004/0CP1;0500
E`0002@0001;0000D00004`0001A00000P0004/0D01<0540E`0002@0001<0000D@0004h0001B0000
0P0004`0D@1>0580E`0002@0001>0000DP000580001D00000P0004h0DP1B05@0E`0002@0001B0000
E00005@0001E00000P000580E01D05D0E`0002@0001D0000E@0005D0001F00000P0005@0E@1E05H0
E`0002@0001;0000C`0004`0001@00000P0004/0C`1<0500E`0002@0001<0000D00004h0001A0000
0P0004`0D01>0540E`0002@0001>0000D@000580001C00000P0004h0D@1B05<0E`0002@0001B0000
D`0005@0001D00000P000580D`1D05@0E`0002@0001D0000E00005D0001E00000P0005@0E01E05D0
E`0002@0001E0000E@0005D0001H00000P0005D0E@1E05P0E`0002@0001D0000F00005D0001I0000
0P0005D0F01D05T0E`0002@0001B0000F@0005@0001I00000P0005@0F@1B05T0E`0002@0001?0000
F@000580001I00000P000580F@1?05T0E`0002@0001=0000F@0004l0001I00000P0004l0F@1=05T0
E`0002@0001<0000F00004d0001I00000P0004d0F@1<05P0E`0002@0001;0000EP0004`0001H0000
0P0004`0F01;05H0E`0002@0001;0000EP0004/0001I00000P0004/0EP1;05T0E`0002@0001;0000
F00004`0001I00000P0004/0F@1<05P0E`0002@0001K0000AP0005`0001700000P0005`0AP1K04L0
E`0002@0001K0000A`0005`0001800000P0005/0A`1L04P0E`0002@0001L0000A`0005d000180000
0P0005`0B01M04L0E`0002@0001L0000AP0005d0001700000P0005d0A`1L04H0E`0002@0001L0000
C00005`0001I00000P0005`0C01L05T0E`0002@0001M0000C00005d0001I00000P0005d0C01M05T0
E`0002@0001I0000C00005d0001<00000P0005T0C01M04`0E`0002@0001I0000F@000600001I0000
0P0005T0F@1P05T0E`0002@0001V0000AP0006H0001F00000P0006H0AP1V05H0E`0002@0001V0000
EP0006L0001I00000P0006H0EP1W05T0E`0002@0001W0000F@0006T0001I00000P0006L0F@1Y05T0
E`0002@0001Y0000F@0006X0001I00000P0006T0F@1Z05T0E`0002@0001Z0000F@0006`0001I0000
0P0006X0F@1/05T0E`0002@0001/0000E`0006d0001I00000P0006`0F@1]05L0E`0002@0001W0000
AP0006L0001F00000P0006L0AP1W05H0E`0002@0001W0000EP0006P0001I00000P0006L0EP1X05T0
E`0002@0001X0000F@0006T0001I00000P0006P0F@1Y05T0E`0002@0001S0000C00006X0001<0000
0P0006<0C01Z04`0E`0002@0001b0000C00007L0001I00000P000780C01g05T0E`0002@0001c0000
C00007L0001H00000P0007<0C01g05P0E`0002@0001g0000C00007d0001I00000P0007d0C01g05T0
E`0002@0001f0000F@0007L0001M00000P0007L0F@1f05d0E`0002@0001d0000G@0007H0001O0000
0P0007H0G@1d05l0E`0002@0001b0000G`0007@0001P00000P0007@0G`1b0600E`0002@0001a0000
H0000780001P00000P000780H01a0600E`0002@0001`0000G`000740001P00000P000740H01`05l0
E`0002@0001`0000GP000740001O00000P000700G`1a05h0E`0002@0001a0000GP000780001O0000
0P000740GP1b05l0E`0002@0001`0000C00007H0001<00000P000700C01f04`0E`0002@0001i0000
C00007l0001<00000P0007T0C01o04`0E`0002@0001c0000L`<007H0001d0`000P0007H0L`=c07@3
E`0002@0001a0000M0<007<0001g0`000P0007<0M0=a07L3E`0002@0001`0000M`<00740001k0`00
0P000740M`=`07/3E`0002@0001`0000N`<00700001n0`000P000700N`=`07h3E`0002@0001`0000
OP<0074000230`000P000700OP=a08<3E`0002@0001a0000P`<007<000260`000P000740P`=c08H3
E`0002@0001c0000QP<007H000260`000P0007<0QP=f08H3E`0002@0001f0000QP<007L000260`00
0P0007H0QP=g08H3E`0002@0001g0000QP<007X000260`000P0007L0QP=j08H3E`0002@0001j0000
P`<007`000260`000P0007X0QP=l08<3E`0002@0001l0000OP<007d000230`000P0007`0P`=m07h3
E`0002@0001m0000N`<007d0001n0`000P0007d0OP=m07/3E`0002@0001l0000M`<007d0001k0`00
0P0007d0N`=l07L3E`0002@0001j0000M0<007`0001g0`000P0007`0M`=j07@3E`0002@0001g0000
L`<007X0001d0`000P0007X0M0=g07<3E`0002@0001f0000L`<007L0001c0`000P0007L0L`=f07<3
E`0002@0001d0000L`<007H0001d0`000P0007H0L`=d07@3E`0002@0001c0000M0<007@0001e0`00
0P0007@0M0=c07D3E`0002@0001b0000M@<007<0001g0`000P0007<0M@=b07L3E`0002@0001a0000
M`<00780001k0`000P000780M`=a07/3E`0002@0001a0000N`<00740001n0`000P000740N`=a07h3
E`0002@0001a0000OP<0078000230`000P000740OP=b08<3E`0002@0001b0000P`<007<000250`00
0P000780P`=c08D3E`0002@0001c0000Q@<007@000260`000P0007<0Q@=d08H3E`0002@0001d0000
QP<007H000260`000P0007@0QP=f08H3E`0002@0001g0000QP<007T000260`000P0007L0QP=i08H3
E`0002@0001i0000Q@<007X000260`000P0007T0QP=j08D3E`0002@0001j0000P`<007/000250`00
0P0007X0Q@=k08<3E`0002@0001k0000OP<007`000230`000P0007/0P`=l07h3E`0002@0001l0000
N`<007`0001n0`000P0007`0OP=l07/3E`0002@0001k0000M`<007`0001k0`000P0007`0N`=k07L3
E`0002@0001j0000M@<007/0001g0`000P0007/0M`=j07D3E`0002@0001i0000M0<007X0001e0`00
0P0007X0M@=i07@3E`0002@0001g0000L`<007T0001d0`000P0007T0M0=g07<3E`0002@0001@0000
/`800580002d0P000P000500]09B0;<2E`0002@0001B0000/08005@0002c0P000P000580/`9D0;02
E`0002@0001D0000/08005@000330P000P0005@0/09D0<<2E`0002@0001C0000/@8005<000330P00
0P0005<0/@9C0<<2E`0002@0001@0000``8005P000330P000P000500``9H0<<2E`0002@0001Q0000
/08006@0002a0P000P0006@0/09Q0;42E`0002@0001O0000/@800640002d0P000P000640/@9O0;@2
E`0002@0001N0000]08005l0002h0P000P0005l0]09N0;P2E`0002@0001N0000^08005h0002k0P00
0P0005h0^09N0;/2E`0002@0001N0000^`8005l000300P000P0005h0^`9O0<02E`0002@0001O0000
`080064000330P000P0005l0`09Q0<<2E`0002@0001Q0000``8006@000330P000P000640``9T0<<2
E`0002@0001T0000``8006D000330P000P0006@0``9U0<<2E`0002@0001U0000``8006P000330P00
0P0006D0``9X0<<2E`0002@0001X0000`08006X000330P000P0006P0``9Z0<02E`0002@0001Z0000
^`8006/000300P000P0006X0`09[0;/2E`0002@0001[0000^08006/0002k0P000P0006/0^`9[0;P2
E`0002@0001Z0000]08006/0002h0P000P0006/0^09Z0;@2E`0002@0001X0000/@8006X0002d0P00
0P0006X0]09X0;42E`0002@0001U0000/08006P0002a0P000P0006P0/@9U0;02E`0002@0001T0000
/08006D0002`0P000P0006D0/09T0;02E`0002@0001R0000/08006@0002a0P000P0006@0/09R0;42
E`0002@0001Q0000/@800680002b0P000P000680/@9Q0;82E`0002@0001P0000/P800640002d0P00
0P000640/P9P0;@2E`0002@0001O0000]0800600002h0P000P000600]09O0;P2E`0002@0001O0000
^08005l0002k0P000P0005l0^09O0;/2E`0002@0001O0000^`80060000300P000P0005l0^`9P0<02
E`0002@0001P0000`080064000320P000P000600`09Q0<82E`0002@0001Q0000`P80068000330P00
0P000640`P9R0<<2E`0002@0001R0000``8006@000330P000P000680``9T0<<2E`0002@0001U0000
``8006L000330P000P0006D0``9W0<<2E`0002@0001W0000`P8006P000330P000P0006L0``9X0<82
E`0002@0001X0000`08006T000320P000P0006P0`P9Y0<02E`0002@0001Y0000^`8006X000300P00
0P0006T0`09Z0;/2E`0002@0001Z0000^08006X0002k0P000P0006X0^`9Z0;P2E`0002@0001Y0000
]08006X0002h0P000P0006X0^09Y0;@2E`0002@0001X0000/P8006T0002d0P000P0006T0]09X0;82
E`0002@0001W0000/@8006P0002b0P000P0006P0/P9W0;42E`0002@0001U0000/08006L0002a0P00
0P0006L0/@9U0;02E`0002@0001c0000/08007H0002a0P000P0007H0/09c0;42E`0002@0001a0000
/@8007<0002d0P000P0007<0/@9a0;@2E`0002@0001`0000]0800740002h0P000P000740]09`0;P2
E`0002@0001`0000^0800700002k0P000P000700^09`0;/2E`0002@0001`0000^`80074000300P00
0P000700^`9a0<02E`0002@0001a0000`08007<000330P000P000740`09c0<<2E`0002@0001c0000
``8007H000330P000P0007<0``9f0<<2E`0002@0001f0000``8007L000330P000P0007H0``9g0<<2
E`0002@0001g0000``8007X000330P000P0007L0``9j0<<2E`0002@0001j0000`08007`000330P00
0P0007X0``9l0<02E`0002@0001l0000^`8007d000300P000P0007`0`09m0;/2E`0002@0001m0000
^08007d0002k0P000P0007d0^`9m0;P2E`0002@0001l0000]08007d0002h0P000P0007d0^09l0;@2
E`0002@0001j0000/@8007`0002d0P000P0007`0]09j0;42E`0002@0001g0000/08007X0002a0P00
0P0007X0/@9g0;02E`0002@0001f0000/08007L0002`0P000P0007L0/09f0;02E`0002@0001d0000
/08007H0002a0P000P0007H0/09d0;42E`0002@0001c0000/@8007@0002b0P000P0007@0/@9c0;82
E`0002@0001b0000/P8007<0002d0P000P0007<0/P9b0;@2E`0002@0001a0000]0800780002h0P00
0P000780]09a0;P2E`0002@0001a0000^0800740002k0P000P000740^09a0;/2E`0002@0001a0000
^`80078000300P000P000740^`9b0<02E`0002@0001b0000`08007<000320P000P000780`09c0<82
E`0002@0001c0000`P8007@000330P000P0007<0`P9d0<<2E`0002@0001d0000``8007H000330P00
0P0007@0``9f0<<2E`0002@0001g0000``8007T000330P000P0007L0``9i0<<2E`0002@0001i0000
`P8007X000330P000P0007T0``9j0<82E`0002@0001j0000`08007/000320P000P0007X0`P9k0<02
E`0002@0001k0000^`8007`000300P000P0007/0`09l0;/2E`0002@0001l0000^08007`0002k0P00
0P0007`0^`9l0;P2E`0002@0001k0000]08007`0002h0P000P0007`0^09k0;@2E`0002@0001j0000
/P8007/0002d0P000P0007/0]09j0;82E`0002@0001i0000/@8007X0002b0P000P0007X0/P9i0;42
E`0002@0001g0000/08007T0002a0P000P0007T0/@9g0;02E`0002@0001>0000l04004l0003a0@00
0P0004h0l05?0?41E`0002@0001>0000l@4004l0003a0@000P0004l0l@5>0?41E`0002@0001=0000
l@4004h0003a0@000P0004h0l@5=0?41E`0002@0001=0000l04004d0003a0@000P0004d0l@5=0?01
E`0002@0001=0000kP4004h0003`0@000P0004d0l05>0>h1E`0002@0001>0000k@4004l0003^0@00
0P0004h0kP5?0>d1E`0002@0001?0000k0400580003]0@000P0004l0k@5B0>`1E`0002@0001B0000
k04005D0003/0@000P000580k05E0>`1E`0002@0001E0000k04005P0003]0@000P0005D0k05H0>d1
E`0002@0001H0000k@4005T0003^0@000P0005P0k@5I0>h1E`0002@0001I0000kP4005X0003`0@00
0P0005T0kP5J0?01E`0002@0001J0000l04005X0003a0@000P0005X0l05J0?41E`0002@0001I0000
l@4005X0003c0@000P0005X0l@5I0?<1E`0002@0001F0000l`4005T0003e0@000P0005T0l`5F0?D1
E`0002@0001B0000m@4005H0003g0@000P0005H0m@5B0?L1E`0002@0001@0000m`400580003h0@00
0P000580m`5@0?P1E`0002@0001>0000n0400500003j0@000P000500n05>0?X1E`0002@0001=0000
nP4004h0003m0@000P0004h0nP5=0?d1E`0002@0001=0000o@4004d0003o0@000P0004d0o@5=0?l1
E`0002@0001E0000k04005L0003]0@000P0005D0k05G0>d1E`0002@0001G0000k@4005P0003^0@00
0P0005L0k@5H0>h1E`0002@0001H0000kP4005T0003`0@000P0005P0kP5I0?01E`0002@0001I0000
l04005T0003a0@000P0005T0l05I0?41E`0002@0001H0000l@4005T0003c0@000P0005T0l@5H0?<1
E`0002@0001E0000l`4005P0003e0@000P0005P0l`5E0?D1E`0002@0001B0000m@4005D0003g0@00
0P0005D0m@5B0?L1E`0002@0001=0000o@4004h0003n0@000P0004d0oP5>0?d1E`0002@0001>0000
o@400500003m0@000P0004h0o@5@0?d1E`0002@0001@0000o@4005@0003o0@000P000500o@5D0?l1
E`0002@0001D0000o`4005L0003o0@000P0005@0o`5G0?l1E`0002@0001G0000oP4005T0003o0@00
0P0005L0o`5I0?h1E`0002@0001I0000o@4005X0003n0@000P0005T0oP5J0?d1E`0002@0001@0000
o@4005@0003o0@000P000500o@5D0?l1E`0002@0001D0000o`4005P0003o0@000P0005@0o`5H0?l1
E`0002@0001H0000o`4005T0003o0@000P0005P0o`5I0?l1E`0002@0001I0000o@4005X0003o0@00
0P0005T0o`5J0?d1E`0002@0001J0000n`4005X0003m0@000P0005X0o@5J0?/1E`0002@0001Q0000
k04006@0003]0@000P0006@0k05Q0>d1E`0002@0001O0000k@400640003`0@000P000640k@5O0?01
E`0002@0001N0000l04005l0003d0@000P0005l0l05N0?@1E`0002@0001N0000m04005h0003g0@00
0P0005h0m05N0?L1E`0002@0001N0000m`4005l0003l0@000P0005h0m`5O0?`1E`0002@0001O0000
o0400640003o0@000P0005l0o05Q0?l1E`0002@0001Q0000o`4006@0003o0@000P000640o`5T0?l1
E`0002@0001T0000o`4006D0003o0@000P0006@0o`5U0?l1E`0002@0001U0000o`4006P0003o0@00
0P0006D0o`5X0?l1E`0002@0001X0000o04006X0003o0@000P0006P0o`5Z0?`1E`0002@0001Z0000
m`4006/0003l0@000P0006X0o05[0?L1E`0002@0001[0000m04006/0003g0@000P0006/0m`5[0?@1
E`0002@0001Z0000l04006/0003d0@000P0006/0m05Z0?01E`0002@0001X0000k@4006X0003`0@00
0P0006X0l05X0>d1E`0002@0001U0000k04006P0003]0@000P0006P0k@5U0>`1E`0002@0001T0000
k04006D0003/0@000P0006D0k05T0>`1E`0002@0001R0000k04006@0003]0@000P0006@0k05R0>d1
E`0002@0001Q0000k@400680003^0@000P000680k@5Q0>h1E`0002@0001P0000kP400640003`0@00
0P000640kP5P0?01E`0002@0001O0000l0400600003d0@000P000600l05O0?@1E`0002@0001O0000
m04005l0003g0@000P0005l0m05O0?L1E`0002@0001O0000m`400600003l0@000P0005l0m`5P0?`1
E`0002@0001P0000o0400640003n0@000P000600o05Q0?h1E`0002@0001Q0000oP400680003o0@00
0P000640oP5R0?l1E`0002@0001R0000o`4006@0003o0@000P000680o`5T0?l1E`0002@0001U0000
o`4006L0003o0@000P0006D0o`5W0?l1E`0002@0001W0000oP4006P0003o0@000P0006L0o`5X0?h1
E`0002@0001X0000o04006T0003n0@000P0006P0oP5Y0?`1E`0002@0001Y0000m`4006X0003l0@00
0P0006T0o05Z0?L1E`0002@0001Z0000m04006X0003g0@000P0006X0m`5Z0?@1E`0002@0001Y0000
l04006X0003d0@000P0006X0m05Y0?01E`0002@0001X0000kP4006T0003`0@000P0006T0l05X0>h1
E`0002@0001W0000k@4006P0003^0@000P0006P0kP5W0>d1E`0002@0001U0000k04006L0003]0@00
0P0006L0k@5U0>`1E`0002@0001c0000k04007H0003]0@000P0007H0k05c0>d1E`0002@0001a0000
k@4007<0003`0@000P0007<0k@5a0?01E`0002@0001`0000l0400740003d0@000P000740l05`0?@1
E`0002@0001`0000m0400700003g0@000P000700m05`0?L1E`0002@0001`0000m`400740003l0@00
0P000700m`5a0?`1E`0002@0001a0000o04007<0003o0@000P000740o05c0?l1E`0002@0001c0000
o`4007H0003o0@000P0007<0o`5f0?l1E`0002@0001f0000o`4007L0003o0@000P0007H0o`5g0?l1
E`0002@0001g0000o`4007X0003o0@000P0007L0o`5j0?l1E`0002@0001j0000o04007`0003o0@00
0P0007X0o`5l0?`1E`0002@0001l0000m`4007d0003l0@000P0007`0o05m0?L1E`0002@0001m0000
m04007d0003g0@000P0007d0m`5m0?@1E`0002@0001l0000l04007d0003d0@000P0007d0m05l0?01
E`0002@0001j0000k@4007`0003`0@000P0007`0l05j0>d1E`0002@0001g0000k04007X0003]0@00
0P0007X0k@5g0>`1E`0002@0001f0000k04007L0003/0@000P0007L0k05f0>`1E`0002@0001d0000
k04007H0003]0@000P0007H0k05d0>d1E`0002@0001c0000k@4007@0003^0@000P0007@0k@5c0>h1
E`0002@0001b0000kP4007<0003`0@000P0007<0kP5b0?01E`0002@0001a0000l0400780003d0@00
0P000780l05a0?@1E`0002@0001a0000m0400740003g0@000P000740m05a0?L1E`0002@0001a0000
m`400780003l0@000P000740m`5b0?`1E`0002@0001b0000o04007<0003n0@000P000780o05c0?h1
E`0002@0001c0000oP4007@0003o0@000P0007<0oP5d0?l1E`0002@0001d0000o`4007H0003o0@00
0P0007@0o`5f0?l1E`0002@0001g0000o`4007T0003o0@000P0007L0o`5i0?l1E`0002@0001i0000
oP4007X0003o0@000P0007T0o`5j0?h1E`0002@0001j0000o04007/0003n0@000P0007X0oP5k0?`1
E`0002@0001k0000m`4007`0003l0@000P0007/0o05l0?L1E`0002@0001l0000m04007`0003g0@00
0P0007`0m`5l0?@1E`0002@0001k0000l04007`0003d0@000P0007`0m05k0?01E`0002@0001j0000
kP4007/0003`0@000P0007/0l05j0>h1E`0002@0001i0000k@4007X0003^0@000P0007X0kP5i0>d1
E`0002@0001g0000k04007T0003]0@000P0007T0k@5g0>`1E`0002@0001>0000;@4004l0000^0@00
0P0004h0;@5?02h1E`0002@0001>0000;P4004l0000^0@000P0004l0;P5>02h1E`0002@0001=0000
;P4004h0000^0@000P0004h0;P5=02h1E`0002@0001=0000;@4004d0000^0@000P0004d0;P5=02d1
E`0002@0001=0000:`4004h0000]0@000P0004d0;@5>02/1E`0002@0001>0000:P4004l0000[0@00
0P0004h0:`5?02X1E`0002@0001?0000:@400580000Z0@000P0004l0:P5B02T1E`0002@0001B0000
:@4005D0000Y0@000P000580:@5E02T1E`0002@0001E0000:@4005P0000Z0@000P0005D0:@5H02X1
E`0002@0001H0000:P4005T0000/0@000P0005P0:P5I02`1E`0002@0001I0000;04005T0000^0@00
0P0005T0;05I02h1E`0002@0001H0000;P4005T0000`0@000P0005T0;P5H0301E`0002@0001E0000
<04005P0000a0@000P0005P0<05E0341E`0002@0001C0000<@4005D0000a0@000P0005D0<@5C0341
E`0002@0001E0000:@4005L0000Z0@000P0005D0:@5G02X1E`0002@0001G0000:P4005P0000/0@00
0P0005L0:P5H02`1E`0002@0001H0000;04005P0000^0@000P0005P0;05H02h1E`0002@0001G0000
;P4005P0000`0@000P0005P0;P5G0301E`0002@0001E0000<04005L0000a0@000P0005L0<05E0341
E`0002@0001E0000<@4005L0000b0@000P0005D0<@5G0381E`0002@0001G0000<P4005T0000d0@00
0P0005L0<P5I03@1E`0002@0001I0000=04005X0000f0@000P0005T0=05J03H1E`0002@0001J0000
=P4005X0000i0@000P0005X0=P5J03T1E`0002@0001I0000>@4005X0000k0@000P0005X0>@5I03/1
E`0002@0001H0000>`4005T0000l0@000P0005T0>`5H03`1E`0002@0001E0000?04005P0000l0@00
0P0005P0?05E03`1E`0002@0001B0000?04005D0000l0@000P0005D0?05B03`1E`0002@0001?0000
?0400580000l0@000P000580?05?03`1E`0002@0001>0000>`4004l0000l0@000P0004l0?05>03/1
E`0002@0001=0000>@4004h0000k0@000P0004h0>`5=03T1E`0002@0001=0000>04004d0000i0@00
0P0004d0>@5=03P1E`0002@0001=0000=`4004h0000h0@000P0004d0>05>03L1E`0002@0001>0000
=`4004l0000h0@000P0004h0=`5?03P1E`0002@0001>0000>04004l0000i0@000P0004l0>05>03T1
E`0002@0001H0000<`4005T0000f0@000P0005P0<`5I03H1E`0002@0001I0000=P4005T0000i0@00
0P0005T0=P5I03T1E`0002@0001H0000>@4005T0000k0@000P0005T0>@5H03/1E`0002@0001G0000
>`4005P0000l0@000P0005P0>`5G03`1E`0002@0001E0000?04005L0000l0@000P0005L0?05E03`1
E`0002@0001Q0000:@4006@0000Z0@000P0006@0:@5Q02X1E`0002@0001O0000:P400640000]0@00
0P000640:P5O02d1E`0002@0001N0000;@4005l0000a0@000P0005l0;@5N0341E`0002@0001N0000
<@4005h0000d0@000P0005h0<@5N03@1E`0002@0001N0000=04005l0000i0@000P0005h0=05O03T1
E`0002@0001O0000>@400640000l0@000P0005l0>@5Q03`1E`0002@0001Q0000?04006@0000l0@00
0P000640?05T03`1E`0002@0001T0000?04006D0000l0@000P0006@0?05U03`1E`0002@0001U0000
?04006P0000l0@000P0006D0?05X03`1E`0002@0001X0000>@4006X0000l0@000P0006P0?05Z03T1
E`0002@0001Z0000=04006/0000i0@000P0006X0>@5[03@1E`0002@0001[0000<@4006/0000d0@00
0P0006/0=05[0341E`0002@0001Z0000;@4006/0000a0@000P0006/0<@5Z02d1E`0002@0001X0000
:P4006X0000]0@000P0006X0;@5X02X1E`0002@0001U0000:@4006P0000Z0@000P0006P0:P5U02T1
E`0002@0001T0000:@4006D0000Y0@000P0006D0:@5T02T1E`0002@0001R0000:@4006@0000Z0@00
0P0006@0:@5R02X1E`0002@0001Q0000:P400680000[0@000P000680:P5Q02/1E`0002@0001P0000
:`400640000]0@000P000640:`5P02d1E`0002@0001O0000;@400600000a0@000P000600;@5O0341
E`0002@0001O0000<@4005l0000d0@000P0005l0<@5O03@1E`0002@0001O0000=0400600000i0@00
0P0005l0=05P03T1E`0002@0001P0000>@400640000k0@000P000600>@5Q03/1E`0002@0001Q0000
>`400680000l0@000P000640>`5R03`1E`0002@0001R0000?04006@0000l0@000P000680?05T03`1
E`0002@0001U0000?04006L0000l0@000P0006D0?05W03`1E`0002@0001W0000>`4006P0000l0@00
0P0006L0?05X03/1E`0002@0001X0000>@4006T0000k0@000P0006P0>`5Y03T1E`0002@0001Y0000
=04006X0000i0@000P0006T0>@5Z03@1E`0002@0001Z0000<@4006X0000d0@000P0006X0=05Z0341
E`0002@0001Y0000;@4006X0000a0@000P0006X0<@5Y02d1E`0002@0001X0000:`4006T0000]0@00
0P0006T0;@5X02/1E`0002@0001W0000:P4006P0000[0@000P0006P0:`5W02X1E`0002@0001U0000
:@4006L0000Z0@000P0006L0:P5U02T1E`0002@0001c0000:@4007H0000Z0@000P0007H0:@5c02X1
E`0002@0001a0000:P4007<0000]0@000P0007<0:P5a02d1E`0002@0001`0000;@400740000a0@00
0P000740;@5`0341E`0002@0001`0000<@400700000d0@000P000700<@5`03@1E`0002@0001`0000
=0400740000i0@000P000700=05a03T1E`0002@0001a0000>@4007<0000l0@000P000740>@5c03`1
E`0002@0001c0000?04007H0000l0@000P0007<0?05f03`1E`0002@0001f0000?04007L0000l0@00
0P0007H0?05g03`1E`0002@0001g0000?04007X0000l0@000P0007L0?05j03`1E`0002@0001j0000
>@4007`0000l0@000P0007X0?05l03T1E`0002@0001l0000=04007d0000i0@000P0007`0>@5m03@1
E`0002@0001m0000<@4007d0000d0@000P0007d0=05m0341E`0002@0001l0000;@4007d0000a0@00
0P0007d0<@5l02d1E`0002@0001j0000:P4007`0000]0@000P0007`0;@5j02X1E`0002@0001g0000
:@4007X0000Z0@000P0007X0:P5g02T1E`0002@0001f0000:@4007L0000Y0@000P0007L0:@5f02T1
E`0002@0001d0000:@4007H0000Z0@000P0007H0:@5d02X1E`0002@0001c0000:P4007@0000[0@00
0P0007@0:P5c02/1E`0002@0001b0000:`4007<0000]0@000P0007<0:`5b02d1E`0002@0001a0000
;@400780000a0@000P000780;@5a0341E`0002@0001a0000<@400740000d0@000P000740<@5a03@1
E`0002@0001a0000=0400780000i0@000P000740=05b03T1E`0002@0001b0000>@4007<0000k0@00
0P000780>@5c03/1E`0002@0001c0000>`4007@0000l0@000P0007<0>`5d03`1E`0002@0001d0000
?04007H0000l0@000P0007@0?05f03`1E`0002@0001g0000?04007T0000l0@000P0007L0?05i03`1
E`0002@0001i0000>`4007X0000l0@000P0007T0?05j03/1E`0002@0001j0000>@4007/0000k0@00
0P0007X0>`5k03T1E`0002@0001k0000=04007`0000i0@000P0007/0>@5l03@1E`0002@0001l0000
<@4007`0000d0@000P0007`0=05l0341E`0002@0001k0000;@4007`0000a0@000P0007`0<@5k02d1
E`0002@0001j0000:`4007/0000]0@000P0007/0;@5j02/1E`0002@0001i0000:P4007X0000[0@00
0P0007X0:`5i02X1E`0002@0001g0000:@4007T0000Z0@000P0007T0:P5g02T1E`0002@0001E0000
J00005D0001i00000P0005D0J01E07T0E`0002@0001F0000IP0005H0001i00000P0005H0IP1F07T0
E`0002@0001<0000IP0005H0001d00000P0005H0IP1<07@0E`0002@0001<0000M00005/0001d0000
0P0004`0M01K07@0E`0002@0001C0000N@0005T0001i00000P0005<0N@1I07T0E`0002@0001Q0000
IP0006@0001W00000P0006@0IP1Q06L0E`0002@0001O0000I`000640001Z00000P000640I`1O06X0
E`0002@0001N0000JP0005l0001^00000P0005l0JP1N06h0E`0002@0001N0000KP0005h0001a0000
0P0005h0KP1N0740E`0002@0001N0000L@0005l0001f00000P0005h0L@1O07H0E`0002@0001O0000
MP000640001i00000P0005l0MP1Q07T0E`0002@0001Q0000N@0006@0001i00000P000640N@1T07T0
E`0002@0001T0000N@0006D0001i00000P0006@0N@1U07T0E`0002@0001U0000N@0006P0001i0000
0P0006D0N@1X07T0E`0002@0001X0000MP0006X0001i00000P0006P0N@1Z07H0E`0002@0001Z0000
L@0006/0001f00000P0006X0MP1[0740E`0002@0001[0000KP0006/0001a00000P0006/0L@1[06h0
E`0002@0001Z0000JP0006/0001^00000P0006/0KP1Z06X0E`0002@0001X0000I`0006X0001Z0000
0P0006X0JP1X06L0E`0002@0001U0000IP0006P0001W00000P0006P0I`1U06H0E`0002@0001T0000
IP0006D0001V00000P0006D0IP1T06H0E`0002@0001R0000IP0006@0001W00000P0006@0IP1R06L0
E`0002@0001Q0000I`000680001X00000P000680I`1Q06P0E`0002@0001P0000J0000640001Z0000
0P000640J01P06X0E`0002@0001O0000JP000600001^00000P000600JP1O06h0E`0002@0001O0000
KP0005l0001a00000P0005l0KP1O0740E`0002@0001O0000L@000600001f00000P0005l0L@1P07H0
E`0002@0001P0000MP000640001h00000P000600MP1Q07P0E`0002@0001Q0000N0000680001i0000
0P000640N01R07T0E`0002@0001R0000N@0006@0001i00000P000680N@1T07T0E`0002@0001U0000
N@0006L0001i00000P0006D0N@1W07T0E`0002@0001W0000N00006P0001i00000P0006L0N@1X07P0
E`0002@0001X0000MP0006T0001h00000P0006P0N01Y07H0E`0002@0001Y0000L@0006X0001f0000
0P0006T0MP1Z0740E`0002@0001Z0000KP0006X0001a00000P0006X0L@1Z06h0E`0002@0001Y0000
JP0006X0001^00000P0006X0KP1Y06X0E`0002@0001X0000J00006T0001Z00000P0006T0JP1X06P0
E`0002@0001W0000I`0006P0001X00000P0006P0J01W06L0E`0002@0001U0000IP0006L0001W0000
0P0006L0I`1U06H0E`0002@0001c0000IP0007H0001W00000P0007H0IP1c06L0E`0002@0001a0000
I`0007<0001Z00000P0007<0I`1a06X0E`0002@0001`0000JP000740001^00000P000740JP1`06h0
E`0002@0001`0000KP000700001a00000P000700KP1`0740E`0002@0001`0000L@000740001f0000
0P000700L@1a07H0E`0002@0001a0000MP0007<0001i00000P000740MP1c07T0E`0002@0001c0000
N@0007H0001i00000P0007<0N@1f07T0E`0002@0001f0000N@0007L0001i00000P0007H0N@1g07T0
E`0002@0001g0000N@0007X0001i00000P0007L0N@1j07T0E`0002@0001j0000MP0007`0001i0000
0P0007X0N@1l07H0E`0002@0001l0000L@0007d0001f00000P0007`0MP1m0740E`0002@0001m0000
KP0007d0001a00000P0007d0L@1m06h0E`0002@0001l0000JP0007d0001^00000P0007d0KP1l06X0
E`0002@0001j0000I`0007`0001Z00000P0007`0JP1j06L0E`0002@0001g0000IP0007X0001W0000
0P0007X0I`1g06H0E`0002@0001f0000IP0007L0001V00000P0007L0IP1f06H0E`0002@0001d0000
IP0007H0001W00000P0007H0IP1d06L0E`0002@0001c0000I`0007@0001X00000P0007@0I`1c06P0
E`0002@0001b0000J00007<0001Z00000P0007<0J01b06X0E`0002@0001a0000JP000780001^0000
0P000780JP1a06h0E`0002@0001a0000KP000740001a00000P000740KP1a0740E`0002@0001a0000
L@000780001f00000P000740L@1b07H0E`0002@0001b0000MP0007<0001h00000P000780MP1c07P0
E`0002@0001c0000N00007@0001i00000P0007<0N01d07T0E`0002@0001d0000N@0007H0001i0000
0P0007@0N@1f07T0E`0002@0001g0000N@0007T0001i00000P0007L0N@1i07T0E`0002@0001i0000
N00007X0001i00000P0007T0N@1j07P0E`0002@0001j0000MP0007/0001h00000P0007X0N01k07H0
E`0002@0001k0000L@0007`0001f00000P0007/0MP1l0740E`0002@0001l0000KP0007`0001a0000
0P0007`0L@1l06h0E`0002@0001k0000JP0007`0001^00000P0007`0KP1k06X0E`0002@0001j0000
J00007/0001Z00000P0007/0JP1j06P0E`0002@0001i0000I`0007X0001X00000P0007X0J01i06L0
E`0002@0001g0000IP0007T0001W00000P0007T0I`1g06H0E`0002@0002?0000Q@<008l0002=0`00
0P0008l0Q@>?08d3E`0002@0000X0@00Q@<002P1002=0`000P0002P1Q@<X0Hd3E`0002@000310@00
Q@<00<41002=0`000P000<41Q@?10Hd3E`0002@0001K0P00Q@<005/2002=0`000P0005/2Q@=K0Xd3
E`0002@0003d0P00Q@<00?@2002=0`000P000?@2Q@?d0Xd3E`0002@0002=0`00Q@<008d3002=0`00
0P0008d3Q@>=0hd3E`0002@0000V1000Q@<002H4002=0`000P0002H4Q@<V18d3E`0002@0002o1000
Q@<00;l4002=0`000P000;l4Q@>o18d3E`0002@000200P00b`<008H2003N0`000P000802b`>60]h3
E`0002@000210P00b`<008H2003L0`000P000842b`>60]`3E`0002@000260P00b`<008d2003N0`00
0P0008d2b`>60]h3E`0002@0001n0P00b`<008@2003;0`000P0007h2b`>40//3E`0002@000290P00
b`<008h2003;0`000P0008T2b`>>0//3E`0002@0002D0P00d`<009@2003D0`000P0009@2d`>D0]@3
E`0002@0002C0P00e0<009@2003D0`000P0009@2e0>C0]@3E`0002@0002C0P00d`<009<2003D0`00
0P0009<2e0>C0]<3E`0002@0002C0P00dP<009@2003C0`000P0009<2d`>D0]83E`0002@0002D0P00
d@<009H2003B0`000P0009@2dP>F0]43E`0002@0002F0P00d@<009T2003A0`000P0009H2d@>I0]43
E`0002@0002I0P00d@<009/2003B0`000P0009T2d@>K0]83E`0002@0002K0P00dP<009`2003C0`00
0P0009/2dP>L0]<3E`0002@0002L0P00d`<009d2003E0`000P0009`2d`>M0]D3E`0002@0002M0P00
e@<009d2003L0`000P0009d2e@>M0]`3E`0002@0002M0P00g0<009h2003N0`000P0009d2g0>N0]h3
E`0002@0002N0P00gP<009l2003N0`000P0009h2gP>O0]h3E`0002@0002L0P00d`<009`2003L0`00
0P0009`2d`>L0]`3E`0002@0002L0P00g0<009d2003N0`000P0009`2g0>M0]h3E`0002@0002M0P00
gP<009l2003N0`000P0009d2gP>O0]h3E`0002@0002O0P00gP<00:02003N0`000P0009l2gP>P0]h3
E`0002@0002K0P00e@<009`2003F0`000P0009`2e@>K0]H3E`0002@0002F0P00eP<009/2003G0`00
0P0009/2eP>F0]L3E`0002@0002C0P00e`<009H2003H0`000P0009H2e`>C0]P3E`0002@0002B0P00
f0<009<2003J0`000P0009<2f0>B0]X3E`0002@0002B0P00fP<00982003L0`000P000982fP>B0]`3
E`0002@0002B0P00g0<009<2003N0`000P000982g0>C0]h3E`0002@0002C0P00gP<009H2003N0`00
0P0009<2gP>F0]h3E`0002@0002F0P00gP<009P2003N0`000P0009H2gP>H0]h3E`0002@0002H0P00
gP<009X2003N0`000P0009P2gP>J0]h3E`0002@0002J0P00g0<009`2003N0`000P0009X2gP>L0]`3
E`0002@0002D0P00e`<009H2003H0`000P0009H2e`>D0]P3E`0002@0002C0P00f0<009@2003J0`00
0P0009@2f0>C0]X3E`0002@0002C0P00fP<009<2003L0`000P0009<2fP>C0]`3E`0002@0002C0P00
g0<009@2003N0`000P0009<2g0>D0]h3E`0002@0002D0P00gP<009H2003N0`000P0009@2gP>F0]h3
E`0002@0002V0P00b`<00:H2003N0`000P000:H2b`>V0]h3E`0002@0002W0P00b`<00:L2003N0`00
0P000:L2b`>W0]h3E`0002@0002S0P00b`<00:L2003;0`000P000:<2b`>W0//3E`0002@0002S0P00
gP<00:X2003N0`000P000:<2gP>Z0]h3E`0002@0002`0P00d@<00;02003L0`000P000;02d@>`0]`3
E`0002@0002`0P00g0<00;42003N0`000P000;02g0>a0]h3E`0002@0002a0P00gP<00;@2003N0`00
0P000;42gP>d0]h3E`0002@0002d0P00gP<00;D2003N0`000P000;@2gP>e0]h3E`0002@0002e0P00
gP<00;P2003N0`000P000;D2gP>h0]h3E`0002@0002h0P00g0<00;X2003N0`000P000;P2gP>j0]`3
E`0002@0002a0P00d@<00;42003L0`000P000;42d@>a0]`3E`0002@0002a0P00g0<00;82003N0`00
0P000;42g0>b0]h3E`0002@0002b0P00gP<00;@2003N0`000P000;82gP>d0]h3E`0002@0002j0P00
d@<00;X2003N0`000P000;X2d@>j0]h3E`0002@0002k0P00d@<00;/2003N0`000P000;/2d@>k0]h3
E`0002@0002]0P00d@<00;42003A0`000P000:d2d@>a0]43E`0002@0002g0P00d@<00;/2003A0`00
0P000;L2d@>k0]43E`0002@0002j0P00gP<00;d2003N0`000P000;X2gP>m0]h3E`0002@000320P00
e`<00<d2003G0`000P000<82e`?=0]L3E`0002@0003=0P00e@<00<d2003G0`000P000<d2e`?=0]D3
E`0002@0003<0P00d`<00<d2003E0`000P000<d2e@?<0]<3E`0002@0003;0P00dP<00<`2003C0`00
0P000<`2d`?;0]83E`0002@000390P00d@<00</2003B0`000P000</2dP?90]43E`0002@000370P00
d@<00<T2003A0`000P000<T2d@?70]43E`0002@000340P00d@<00<L2003B0`000P000<L2d@?40]83
E`0002@000320P00dP<00<@2003D0`000P000<@2dP?20]@3E`0002@000310P00e0<00<82003G0`00
0P000<82e0?10]L3E`0002@000310P00e`<00<42003I0`000P000<42e`?10]T3E`0002@000310P00
f@<00<82003L0`000P000<42f@?20]`3E`0002@000320P00g0<00<@2003N0`000P000<82g0?40]h3
E`0002@000340P00gP<00<L2003N0`000P000<@2gP?70]h3E`0002@000370P00gP<00<P2003N0`00
0P000<L2gP?80]h3E`0002@000380P00gP<00</2003N0`000P000<P2gP?;0]h3E`0002@0003;0P00
g0<00<d2003N0`000P000</2gP?=0]`3E`0002@0003<0P00e0<00<`2003G0`000P000<`2e`?<0]@3
E`0002@0003;0P00dP<00<`2003D0`000P000<`2e0?;0]83E`0002@000350P00d@<00<L2003B0`00
0P000<L2d@?50]83E`0002@000330P00dP<00<D2003D0`000P000<D2dP?30]@3E`0002@000320P00
e0<00<<2003G0`000P000<<2e0?20]L3E`0002@000320P00e`<00<82003I0`000P000<82e`?20]T3
E`0002@000320P00f@<00<<2003L0`000P000<82f@?30]`3E`0002@000330P00g0<00<D2003N0`00
0P000<<2g0?50]h3E`0002@000350P00gP<00<L2003N0`000P000<D2gP?70]h3E`0002@0001/0000
V`<006l0002L0`000P0006l0V`=/09`3E`0002@0001Z0000W0<006`0002O0`000P0006`0W0=Z09l3
E`0002@0001Y0000W`<006X0002S0`000P0006X0W`=Y0:<3E`0002@0001Y0000X`<006T0002V0`00
0P0006T0X`=Y0:H3E`0002@0001Y0000YP<006X0002[0`000P0006T0YP=Z0:/3E`0002@0001Z0000
Z`<006`0002^0`000P0006X0Z`=/0:h3E`0002@0001/0000[P<006l0002^0`000P0006`0[P=_0:h3
E`0002@0001_0000[P<00700002^0`000P0006l0[P=`0:h3E`0002@0001`0000[P<007<0002^0`00
0P000700[P=c0:h3E`0002@0001c0000Z`<007D0002^0`000P0007<0[P=e0:/3E`0002@0001e0000
YP<007H0002[0`000P0007D0Z`=f0:H3E`0002@0001f0000X`<007H0002V0`000P0007H0YP=f0:<3
E`0002@0001e0000W`<007H0002S0`000P0007H0X`=e09l3E`0002@0001c0000W0<007D0002O0`00
0P0007D0W`=c09`3E`0002@0001`0000V`<007<0002L0`000P0007<0W0=`09/3E`0002@0001_0000
V`<00700002K0`000P000700V`=_09/3E`0002@0001]0000V`<006l0002L0`000P0006l0V`=]09`3
E`0002@0001/0000W0<006d0002M0`000P0006d0W0=/09d3E`0002@0001[0000W@<006`0002O0`00
0P0006`0W@=[09l3E`0002@0001Z0000W`<006/0002S0`000P0006/0W`=Z0:<3E`0002@0001Z0000
X`<006X0002V0`000P0006X0X`=Z0:H3E`0002@0001Z0000YP<006/0002[0`000P0006X0YP=[0:/3
E`0002@0001[0000Z`<006`0002]0`000P0006/0Z`=/0:d3E`0002@0001/0000[@<006d0002^0`00
0P0006`0[@=]0:h3E`0002@0001]0000[P<006l0002^0`000P0006d0[P=_0:h3E`0002@0001`0000
[P<00780002^0`000P000700[P=b0:h3E`0002@0001b0000[@<007<0002^0`000P000780[P=c0:d3
E`0002@0001c0000Z`<007@0002]0`000P0007<0[@=d0:/3E`0002@0001d0000YP<007D0002[0`00
0P0007@0Z`=e0:H3E`0002@0001e0000X`<007D0002V0`000P0007D0YP=e0:<3E`0002@0001d0000
W`<007D0002S0`000P0007D0X`=d09l3E`0002@0001c0000W@<007@0002O0`000P0007@0W`=c09d3
E`0002@0001b0000W0<007<0002M0`000P0007<0W@=b09`3E`0002@0001`0000V`<00780002L0`00
0P000780W0=`09/3E`0002@0001l0000[@<007d0002^0`000P0007d0[@=l0:h3E`0002@0001l0000
[P<007d0002^0`000P0007`0[P=m0:h3E`0002@0001m0000[P<007h0002^0`000P0007d0[P=n0:h3
E`0002@0001m0000[@<007h0002^0`000P0007h0[P=m0:d3E`0002@000270000V`<008X0002L0`00
0P0008X0V`>709`3E`0002@000250000W0<008L0002O0`000P0008L0W0>509l3E`0002@000240000
W`<008D0002S0`000P0008D0W`>40:<3E`0002@000240000X`<008@0002V0`000P0008@0X`>40:H3
E`0002@000240000YP<008D0002[0`000P0008@0YP>50:/3E`0002@000250000Z`<008L0002^0`00
0P0008D0Z`>70:h3E`0002@000270000[P<008X0002^0`000P0008L0[P>:0:h3E`0002@0002:0000
[P<008/0002^0`000P0008X0[P>;0:h3E`0002@0002;0000[P<008h0002^0`000P0008/0[P>>0:h3
E`0002@0002>0000Z`<00900002^0`000P0008h0[P>@0:/3E`0002@0002@0000YP<00940002[0`00
0P000900Z`>A0:H3E`0002@0002A0000X`<00940002V0`000P000940YP>A0:<3E`0002@0002@0000
W`<00940002S0`000P000940X`>@09l3E`0002@0002>0000W0<00900002O0`000P000900W`>>09`3
E`0002@0002;0000V`<008h0002L0`000P0008h0W0>;09/3E`0002@0002:0000V`<008/0002K0`00
0P0008/0V`>:09/3E`0002@000280000V`<008X0002L0`000P0008X0V`>809`3E`0002@000270000
W0<008P0002M0`000P0008P0W0>709d3E`0002@000260000W@<008L0002O0`000P0008L0W@>609l3
E`0002@000250000W`<008H0002S0`000P0008H0W`>50:<3E`0002@000250000X`<008D0002V0`00
0P0008D0X`>50:H3E`0002@000250000YP<008H0002[0`000P0008D0YP>60:/3E`0002@000260000
Z`<008L0002]0`000P0008H0Z`>70:d3E`0002@000270000[@<008P0002^0`000P0008L0[@>80:h3
E`0002@000280000[P<008X0002^0`000P0008P0[P>:0:h3E`0002@0002;0000[P<008d0002^0`00
0P0008/0[P>=0:h3E`0002@0002=0000[@<008h0002^0`000P0008d0[P>>0:d3E`0002@0002>0000
Z`<008l0002]0`000P0008h0[@>?0:/3E`0002@0002?0000YP<00900002[0`000P0008l0Z`>@0:H3
E`0002@0002@0000X`<00900002V0`000P000900YP>@0:<3E`0002@0002?0000W`<00900002S0`00
0P000900X`>?09l3E`0002@0002>0000W@<008l0002O0`000P0008l0W`>>09d3E`0002@0002=0000
W0<008h0002M0`000P0008h0W@>=09`3E`0002@0002;0000V`<008d0002L0`000P0008d0W0>;09/3
E`0002@0002I0000V`<009`0002L0`000P0009`0V`>I09`3E`0002@0002G0000W0<009T0002O0`00
0P0009T0W0>G09l3E`0002@0002F0000W`<009L0002S0`000P0009L0W`>F0:<3E`0002@0002F0000
X`<009H0002V0`000P0009H0X`>F0:H3E`0002@0002F0000YP<009L0002[0`000P0009H0YP>G0:/3
E`0002@0002G0000Z`<009T0002^0`000P0009L0Z`>I0:h3E`0002@0002I0000[P<009`0002^0`00
0P0009T0[P>L0:h3E`0002@0002L0000[P<009d0002^0`000P0009`0[P>M0:h3E`0002@0002M0000
[P<00:00002^0`000P0009d0[P>P0:h3E`0002@0002P0000Z`<00:80002^0`000P000:00[P>R0:/3
E`0002@0002R0000YP<00:<0002[0`000P000:80Z`>S0:H3E`0002@0002S0000X`<00:<0002V0`00
0P000:<0YP>S0:<3E`0002@0002R0000W`<00:<0002S0`000P000:<0X`>R09l3E`0002@0002P0000
W0<00:80002O0`000P000:80W`>P09`3E`0002@0002M0000V`<00:00002L0`000P000:00W0>M09/3
E`0002@0002L0000V`<009d0002K0`000P0009d0V`>L09/3E`0002@0002J0000V`<009`0002L0`00
0P0009`0V`>J09`3E`0002@0002I0000W0<009X0002M0`000P0009X0W0>I09d3E`0002@0002H0000
W@<009T0002O0`000P0009T0W@>H09l3E`0002@0002G0000W`<009P0002S0`000P0009P0W`>G0:<3
E`0002@0002G0000X`<009L0002V0`000P0009L0X`>G0:H3E`0002@0002G0000YP<009P0002[0`00
0P0009L0YP>H0:/3E`0002@0002H0000Z`<009T0002]0`000P0009P0Z`>I0:d3E`0002@0002I0000
[@<009X0002^0`000P0009T0[@>J0:h3E`0002@0002J0000[P<009`0002^0`000P0009X0[P>L0:h3
E`0002@0002M0000[P<009l0002^0`000P0009d0[P>O0:h3E`0002@0002O0000[@<00:00002^0`00
0P0009l0[P>P0:d3E`0002@0002P0000Z`<00:40002]0`000P000:00[@>Q0:/3E`0002@0002Q0000
YP<00:80002[0`000P000:40Z`>R0:H3E`0002@0002R0000X`<00:80002V0`000P000:80YP>R0:<3
E`0002@0002Q0000W`<00:80002S0`000P000:80X`>Q09l3E`0002@0002P0000W@<00:40002O0`00
0P000:40W`>P09d3E`0002@0002O0000W0<00:00002M0`000P000:00W@>O09`3E`0002@0002M0000
V`<009l0002L0`000P0009l0W0>M09/3E`0002@0002Z0000V`<00:d0002L0`000P000:d0V`>Z09`3
E`0002@0002X0000W0<00:X0002O0`000P000:X0W0>X09l3E`0002@0002W0000W`<00:P0002S0`00
0P000:P0W`>W0:<3E`0002@0002W0000X`<00:L0002V0`000P000:L0X`>W0:H3E`0002@0002W0000
YP<00:P0002[0`000P000:L0YP>X0:/3E`0002@0002X0000Z`<00:X0002^0`000P000:P0Z`>Z0:h3
E`0002@0002Z0000[P<00:d0002^0`000P000:X0[P>]0:h3E`0002@0002]0000[P<00:h0002^0`00
0P000:d0[P>^0:h3E`0002@0002^0000[P<00;40002^0`000P000:h0[P>a0:h3E`0002@0002a0000
Z`<00;<0002^0`000P000;40[P>c0:/3E`0002@0002c0000YP<00;@0002[0`000P000;<0Z`>d0:H3
E`0002@0002d0000X`<00;@0002V0`000P000;@0YP>d0:<3E`0002@0002c0000W`<00;@0002S0`00
0P000;@0X`>c09l3E`0002@0002a0000W0<00;<0002O0`000P000;<0W`>a09`3E`0002@0002^0000
V`<00;40002L0`000P000;40W0>^09/3E`0002@0002]0000V`<00:h0002K0`000P000:h0V`>]09/3
E`0002@0002[0000V`<00:d0002L0`000P000:d0V`>[09`3E`0002@0002Z0000W0<00:/0002M0`00
0P000:/0W0>Z09d3E`0002@0002Y0000W@<00:X0002O0`000P000:X0W@>Y09l3E`0002@0002X0000
W`<00:T0002S0`000P000:T0W`>X0:<3E`0002@0002X0000X`<00:P0002V0`000P000:P0X`>X0:H3
E`0002@0002X0000YP<00:T0002[0`000P000:P0YP>Y0:/3E`0002@0002Y0000Z`<00:X0002]0`00
0P000:T0Z`>Z0:d3E`0002@0002Z0000[@<00:/0002^0`000P000:X0[@>[0:h3E`0002@0002[0000
[P<00:d0002^0`000P000:/0[P>]0:h3E`0002@0002^0000[P<00;00002^0`000P000:h0[P>`0:h3
E`0002@0002`0000[@<00;40002^0`000P000;00[P>a0:d3E`0002@0002a0000Z`<00;80002]0`00
0P000;40[@>b0:/3E`0002@0002b0000YP<00;<0002[0`000P000;80Z`>c0:H3E`0002@0002c0000
X`<00;<0002V0`000P000;<0YP>c0:<3E`0002@0002b0000W`<00;<0002S0`000P000;<0X`>b09l3
E`0002@0002a0000W@<00;80002O0`000P000;80W`>a09d3E`0002@0002`0000W0<00;40002M0`00
0P000;40W@>`09`3E`0002@0002^0000V`<00;00002L0`000P000;00W0>^09/3E`0002@000050@00
V`<000P1002L0`000P0000P1V`<50I`3E`0002@000030@00W0<000D1002O0`000P0000D1W0<30Il3
E`0002@000020@00W`<000<1002S0`000P0000<1W`<20J<3E`0002@000020@00X`<00081002V0`00
0P000081X`<20JH3E`0002@000020@00YP<000<1002[0`000P000081YP<30J/3E`0002@000030@00
Z`<000D1002^0`000P0000<1Z`<50Jh3E`0002@000050@00[P<000P1002^0`000P0000D1[P<80Jh3
E`0002@000080@00[P<000T1002^0`000P0000P1[P<90Jh3E`0002@000090@00[P<000`1002^0`00
0P0000T1[P<<0Jh3E`0002@0000<0@00Z`<000h1002^0`000P0000`1[P<>0J/3E`0002@0000>0@00
YP<000l1002[0`000P0000h1Z`<?0JH3E`0002@0000?0@00X`<000l1002V0`000P0000l1YP<?0J<3
E`0002@0000>0@00W`<000l1002S0`000P0000l1X`<>0Il3E`0002@0000<0@00W0<000h1002O0`00
0P0000h1W`<<0I`3E`0002@000090@00V`<000`1002L0`000P0000`1W0<90I/3E`0002@000080@00
V`<000T1002K0`000P0000T1V`<80I/3E`0002@000060@00V`<000P1002L0`000P0000P1V`<60I`3
E`0002@000050@00W0<000H1002M0`000P0000H1W0<50Id3E`0002@000040@00W@<000D1002O0`00
0P0000D1W@<40Il3E`0002@000030@00W`<000@1002S0`000P0000@1W`<30J<3E`0002@000030@00
X`<000<1002V0`000P0000<1X`<30JH3E`0002@000030@00YP<000@1002[0`000P0000<1YP<40J/3
E`0002@000040@00Z`<000D1002]0`000P0000@1Z`<50Jd3E`0002@000050@00[@<000H1002^0`00
0P0000D1[@<60Jh3E`0002@000060@00[P<000P1002^0`000P0000H1[P<80Jh3E`0002@000090@00
[P<000/1002^0`000P0000T1[P<;0Jh3E`0002@0000;0@00[@<000`1002^0`000P0000/1[P<<0Jd3
E`0002@0000<0@00Z`<000d1002]0`000P0000`1[@<=0J/3E`0002@0000=0@00YP<000h1002[0`00
0P0000d1Z`<>0JH3E`0002@0000>0@00X`<000h1002V0`000P0000h1YP<>0J<3E`0002@0000=0@00
W`<000h1002S0`000P0000h1X`<=0Il3E`0002@0000<0@00W@<000d1002O0`000P0000d1W`<<0Id3
E`0002@0000;0@00W0<000`1002M0`000P0000`1W@<;0I`3E`0002@000090@00V`<000/1002L0`00
0P0000/1W0<90I/3E`0002@0000E0@00[@<001H1002^0`000P0001H1[@<E0Jh3E`0002@0000E0@00
[P<001H1002^0`000P0001D1[P<F0Jh3E`0002@0000F0@00[P<001L1002^0`000P0001H1[P<G0Jh3
E`0002@0000F0@00[@<001L1002^0`000P0001L1[P<F0Jd3E`0002@0000P0@00V`<002<1002L0`00
0P0002<1V`<P0I`3E`0002@0000N0@00W0<00201002O0`000P000201W0<N0Il3E`0002@0000M0@00
W`<001h1002S0`000P0001h1W`<M0J<3E`0002@0000M0@00X`<001d1002V0`000P0001d1X`<M0JH3
E`0002@0000M0@00YP<001h1002[0`000P0001d1YP<N0J/3E`0002@0000N0@00Z`<00201002^0`00
0P0001h1Z`<P0Jh3E`0002@0000P0@00[P<002<1002^0`000P000201[P<S0Jh3E`0002@0000S0@00
[P<002@1002^0`000P0002<1[P<T0Jh3E`0002@0000T0@00[P<002L1002^0`000P0002@1[P<W0Jh3
E`0002@0000W0@00Z`<002T1002^0`000P0002L1[P<Y0J/3E`0002@0000Y0@00YP<002X1002[0`00
0P0002T1Z`<Z0JH3E`0002@0000Z0@00X`<002X1002V0`000P0002X1YP<Z0J<3E`0002@0000Y0@00
W`<002X1002S0`000P0002X1X`<Y0Il3E`0002@0000W0@00W0<002T1002O0`000P0002T1W`<W0I`3
E`0002@0000T0@00V`<002L1002L0`000P0002L1W0<T0I/3E`0002@0000S0@00V`<002@1002K0`00
0P0002@1V`<S0I/3E`0002@0000Q0@00V`<002<1002L0`000P0002<1V`<Q0I`3E`0002@0000P0@00
W0<00241002M0`000P000241W0<P0Id3E`0002@0000O0@00W@<00201002O0`000P000201W@<O0Il3
E`0002@0000N0@00W`<001l1002S0`000P0001l1W`<N0J<3E`0002@0000N0@00X`<001h1002V0`00
0P0001h1X`<N0JH3E`0002@0000N0@00YP<001l1002[0`000P0001h1YP<O0J/3E`0002@0000O0@00
Z`<00201002]0`000P0001l1Z`<P0Jd3E`0002@0000P0@00[@<00241002^0`000P000201[@<Q0Jh3
E`0002@0000Q0@00[P<002<1002^0`000P000241[P<S0Jh3E`0002@0000T0@00[P<002H1002^0`00
0P0002@1[P<V0Jh3E`0002@0000V0@00[@<002L1002^0`000P0002H1[P<W0Jd3E`0002@0000W0@00
Z`<002P1002]0`000P0002L1[@<X0J/3E`0002@0000X0@00YP<002T1002[0`000P0002P1Z`<Y0JH3
E`0002@0000Y0@00X`<002T1002V0`000P0002T1YP<Y0J<3E`0002@0000X0@00W`<002T1002S0`00
0P0002T1X`<X0Il3E`0002@0000W0@00W@<002P1002O0`000P0002P1W`<W0Id3E`0002@0000V0@00
W0<002L1002M0`000P0002L1W@<V0I`3E`0002@0000T0@00V`<002H1002L0`000P0002H1W0<T0I/3
E`0002@0000b0@00V`<003D1002L0`000P0003D1V`<b0I`3E`0002@0000`0@00W0<00381002O0`00
0P000381W0<`0Il3E`0002@0000_0@00W`<00301002S0`000P000301W`<_0J<3E`0002@0000_0@00
X`<002l1002V0`000P0002l1X`<_0JH3E`0002@0000_0@00YP<00301002[0`000P0002l1YP<`0J/3
E`0002@0000`0@00Z`<00381002^0`000P000301Z`<b0Jh3E`0002@0000b0@00[P<003D1002^0`00
0P000381[P<e0Jh3E`0002@0000e0@00[P<003H1002^0`000P0003D1[P<f0Jh3E`0002@0000f0@00
[P<003T1002^0`000P0003H1[P<i0Jh3E`0002@0000i0@00Z`<003/1002^0`000P0003T1[P<k0J/3
E`0002@0000k0@00YP<003`1002[0`000P0003/1Z`<l0JH3E`0002@0000l0@00X`<003`1002V0`00
0P0003`1YP<l0J<3E`0002@0000k0@00W`<003`1002S0`000P0003`1X`<k0Il3E`0002@0000i0@00
W0<003/1002O0`000P0003/1W`<i0I`3E`0002@0000f0@00V`<003T1002L0`000P0003T1W0<f0I/3
E`0002@0000e0@00V`<003H1002K0`000P0003H1V`<e0I/3E`0002@0000c0@00V`<003D1002L0`00
0P0003D1V`<c0I`3E`0002@0000b0@00W0<003<1002M0`000P0003<1W0<b0Id3E`0002@0000a0@00
W@<00381002O0`000P000381W@<a0Il3E`0002@0000`0@00W`<00341002S0`000P000341W`<`0J<3
E`0002@0000`0@00X`<00301002V0`000P000301X`<`0JH3E`0002@0000`0@00YP<00341002[0`00
0P000301YP<a0J/3E`0002@0000a0@00Z`<00381002]0`000P000341Z`<b0Jd3E`0002@0000b0@00
[@<003<1002^0`000P000381[@<c0Jh3E`0002@0000c0@00[P<003D1002^0`000P0003<1[P<e0Jh3
E`0002@0000f0@00[P<003P1002^0`000P0003H1[P<h0Jh3E`0002@0000h0@00[@<003T1002^0`00
0P0003P1[P<i0Jd3E`0002@0000i0@00Z`<003X1002]0`000P0003T1[@<j0J/3E`0002@0000j0@00
YP<003/1002[0`000P0003X1Z`<k0JH3E`0002@0000k0@00X`<003/1002V0`000P0003/1YP<k0J<3
E`0002@0000j0@00W`<003/1002S0`000P0003/1X`<j0Il3E`0002@0000i0@00W@<003X1002O0`00
0P0003X1W`<i0Id3E`0002@0000h0@00W0<003T1002M0`000P0003T1W@<h0I`3E`0002@0000f0@00
V`<003P1002L0`000P0003P1W0<f0I/3E`0002@000120@00W`<004<1002P0`000P000481W`=30J03
E`0002@000120@00X0<004<1002P0`000P0004<1X0=20J03E`0002@000110@00X0<00481002P0`00
0P000481X0=10J03E`0002@000110@00W`<00441002P0`000P000441X0=10Il3E`0002@000110@00
W@<00481002O0`000P000441W`=20Id3E`0002@000120@00W0<004<1002M0`000P000481W@=30I`3
E`0002@000130@00V`<004H1002L0`000P0004<1W0=60I/3E`0002@000160@00V`<004T1002K0`00
0P0004H1V`=90I/3E`0002@000190@00V`<004`1002L0`000P0004T1V`=<0I`3E`0002@0001<0@00
W0<004d1002M0`000P0004`1W0==0Id3E`0002@0001=0@00W@<004h1002O0`000P0004d1W@=>0Il3
E`0002@0001>0@00W`<004h1002P0`000P0004h1W`=>0J03E`0002@0001=0@00X0<004h1002R0`00
0P0004h1X0==0J83E`0002@0001:0@00XP<004d1002T0`000P0004d1XP=:0J@3E`0002@000160@00
Y0<004X1002V0`000P0004X1Y0=60JH3E`0002@000140@00YP<004H1002W0`000P0004H1YP=40JL3
E`0002@000120@00Y`<004@1002Y0`000P0004@1Y`=20JT3E`0002@000110@00Z@<00481002/0`00
0P000481Z@=10J`3E`0002@000110@00[0<00441002^0`000P000441[0=10Jh3E`0002@000190@00
V`<004/1002L0`000P0004T1V`=;0I`3E`0002@0001;0@00W0<004`1002M0`000P0004/1W0=<0Id3
E`0002@0001<0@00W@<004d1002O0`000P0004`1W@==0Il3E`0002@0001=0@00W`<004d1002P0`00
0P0004d1W`==0J03E`0002@0001<0@00X0<004d1002R0`000P0004d1X0=<0J83E`0002@000190@00
XP<004`1002T0`000P0004`1XP=90J@3E`0002@000160@00Y0<004T1002V0`000P0004T1Y0=60JH3
E`0002@000110@00[0<00481002]0`000P000441[@=20J`3E`0002@000120@00[0<004@1002/0`00
0P000481[0=40J`3E`0002@000140@00[0<004P1002^0`000P0004@1[0=80Jh3E`0002@000180@00
[P<004/1002^0`000P0004P1[P=;0Jh3E`0002@0001;0@00[@<004d1002^0`000P0004/1[P==0Jd3
E`0002@0001=0@00[0<004h1002]0`000P0004d1[@=>0J`3E`0002@000140@00[0<004P1002^0`00
0P0004@1[0=80Jh3E`0002@000180@00[P<004`1002^0`000P0004P1[P=<0Jh3E`0002@0001<0@00
[P<004d1002^0`000P0004`1[P==0Jh3E`0002@0001=0@00[0<004h1002^0`000P0004d1[P=>0J`3
E`0002@0001>0@00ZP<004h1002/0`000P0004h1[0=>0JX3E`0002@0002N0@00V`<00:41002L0`00
0P000:41V`>N0I`3E`0002@0002L0@00W0<009h1002O0`000P0009h1W0>L0Il3E`0002@0002K0@00
W`<009`1002S0`000P0009`1W`>K0J<3E`0002@0002K0@00X`<009/1002V0`000P0009/1X`>K0JH3
E`0002@0002K0@00YP<009`1002[0`000P0009/1YP>L0J/3E`0002@0002L0@00Z`<009h1002^0`00
0P0009`1Z`>N0Jh3E`0002@0002N0@00[P<00:41002^0`000P0009h1[P>Q0Jh3E`0002@0002Q0@00
[P<00:81002^0`000P000:41[P>R0Jh3E`0002@0002R0@00[P<00:D1002^0`000P000:81[P>U0Jh3
E`0002@0002U0@00Z`<00:L1002^0`000P000:D1[P>W0J/3E`0002@0002W0@00YP<00:P1002[0`00
0P000:L1Z`>X0JH3E`0002@0002X0@00X`<00:P1002V0`000P000:P1YP>X0J<3E`0002@0002W0@00
W`<00:P1002S0`000P000:P1X`>W0Il3E`0002@0002U0@00W0<00:L1002O0`000P000:L1W`>U0I`3
E`0002@0002R0@00V`<00:D1002L0`000P000:D1W0>R0I/3E`0002@0002Q0@00V`<00:81002K0`00
0P000:81V`>Q0I/3E`0002@0002O0@00V`<00:41002L0`000P000:41V`>O0I`3E`0002@0002N0@00
W0<009l1002M0`000P0009l1W0>N0Id3E`0002@0002M0@00W@<009h1002O0`000P0009h1W@>M0Il3
E`0002@0002L0@00W`<009d1002S0`000P0009d1W`>L0J<3E`0002@0002L0@00X`<009`1002V0`00
0P0009`1X`>L0JH3E`0002@0002L0@00YP<009d1002[0`000P0009`1YP>M0J/3E`0002@0002M0@00
Z`<009h1002]0`000P0009d1Z`>N0Jd3E`0002@0002N0@00[@<009l1002^0`000P0009h1[@>O0Jh3
E`0002@0002O0@00[P<00:41002^0`000P0009l1[P>Q0Jh3E`0002@0002R0@00[P<00:@1002^0`00
0P000:81[P>T0Jh3E`0002@0002T0@00[@<00:D1002^0`000P000:@1[P>U0Jd3E`0002@0002U0@00
Z`<00:H1002]0`000P000:D1[@>V0J/3E`0002@0002V0@00YP<00:L1002[0`000P000:H1Z`>W0JH3
E`0002@0002W0@00X`<00:L1002V0`000P000:L1YP>W0J<3E`0002@0002V0@00W`<00:L1002S0`00
0P000:L1X`>V0Il3E`0002@0002U0@00W@<00:H1002O0`000P000:H1W`>U0Id3E`0002@0002T0@00
W0<00:D1002M0`000P000:D1W@>T0I`3E`0002@0002R0@00V`<00:@1002L0`000P000:@1W0>R0I/3
E`0002@0002^0@00[@<00:l1002^0`000P000:l1[@>^0Jh3E`0002@0002^0@00[P<00:l1002^0`00
0P000:h1[P>_0Jh3E`0002@0002_0@00[P<00;01002^0`000P000:l1[P>`0Jh3E`0002@0002_0@00
[@<00;01002^0`000P000;01[P>_0Jd3E`0002@0002i0@00V`<00;`1002L0`000P000;`1V`>i0I`3
E`0002@0002g0@00W0<00;T1002O0`000P000;T1W0>g0Il3E`0002@0002f0@00W`<00;L1002S0`00
0P000;L1W`>f0J<3E`0002@0002f0@00X`<00;H1002V0`000P000;H1X`>f0JH3E`0002@0002f0@00
YP<00;L1002[0`000P000;H1YP>g0J/3E`0002@0002g0@00Z`<00;T1002^0`000P000;L1Z`>i0Jh3
E`0002@0002i0@00[P<00;`1002^0`000P000;T1[P>l0Jh3E`0002@0002l0@00[P<00;d1002^0`00
0P000;`1[P>m0Jh3E`0002@0002m0@00[P<00<01002^0`000P000;d1[P?00Jh3E`0002@000300@00
Z`<00<81002^0`000P000<01[P?20J/3E`0002@000320@00YP<00<<1002[0`000P000<81Z`?30JH3
E`0002@000330@00X`<00<<1002V0`000P000<<1YP?30J<3E`0002@000320@00W`<00<<1002S0`00
0P000<<1X`?20Il3E`0002@000300@00W0<00<81002O0`000P000<81W`?00I`3E`0002@0002m0@00
V`<00<01002L0`000P000<01W0>m0I/3E`0002@0002l0@00V`<00;d1002K0`000P000;d1V`>l0I/3
E`0002@0002j0@00V`<00;`1002L0`000P000;`1V`>j0I`3E`0002@0002i0@00W0<00;X1002M0`00
0P000;X1W0>i0Id3E`0002@0002h0@00W@<00;T1002O0`000P000;T1W@>h0Il3E`0002@0002g0@00
W`<00;P1002S0`000P000;P1W`>g0J<3E`0002@0002g0@00X`<00;L1002V0`000P000;L1X`>g0JH3
E`0002@0002g0@00YP<00;P1002[0`000P000;L1YP>h0J/3E`0002@0002h0@00Z`<00;T1002]0`00
0P000;P1Z`>i0Jd3E`0002@0002i0@00[@<00;X1002^0`000P000;T1[@>j0Jh3E`0002@0002j0@00
[P<00;`1002^0`000P000;X1[P>l0Jh3E`0002@0002m0@00[P<00;l1002^0`000P000;d1[P>o0Jh3
E`0002@0002o0@00[@<00<01002^0`000P000;l1[P?00Jd3E`0002@000300@00Z`<00<41002]0`00
0P000<01[@?10J/3E`0002@000310@00YP<00<81002[0`000P000<41Z`?20JH3E`0002@000320@00
X`<00<81002V0`000P000<81YP?20J<3E`0002@000310@00W`<00<81002S0`000P000<81X`?10Il3
E`0002@000300@00W@<00<41002O0`000P000<41W`?00Id3E`0002@0002o0@00W0<00<01002M0`00
0P000<01W@>o0I`3E`0002@0002m0@00V`<00;l1002L0`000P000;l1W0>m0I/3E`0002@0003;0@00
V`<00<h1002L0`000P000<h1V`?;0I`3E`0002@000390@00W0<00</1002O0`000P000</1W0?90Il3
E`0002@000380@00W`<00<T1002S0`000P000<T1W`?80J<3E`0002@000380@00X`<00<P1002V0`00
0P000<P1X`?80JH3E`0002@000380@00YP<00<T1002[0`000P000<P1YP?90J/3E`0002@000390@00
Z`<00</1002^0`000P000<T1Z`?;0Jh3E`0002@0003;0@00[P<00<h1002^0`000P000</1[P?>0Jh3
E`0002@0003>0@00[P<00<l1002^0`000P000<h1[P??0Jh3E`0002@0003?0@00[P<00=81002^0`00
0P000<l1[P?B0Jh3E`0002@0003B0@00Z`<00=@1002^0`000P000=81[P?D0J/3E`0002@0003D0@00
YP<00=D1002[0`000P000=@1Z`?E0JH3E`0002@0003E0@00X`<00=D1002V0`000P000=D1YP?E0J<3
E`0002@0003D0@00W`<00=D1002S0`000P000=D1X`?D0Il3E`0002@0003B0@00W0<00=@1002O0`00
0P000=@1W`?B0I`3E`0002@0003?0@00V`<00=81002L0`000P000=81W0??0I/3E`0002@0003>0@00
V`<00<l1002K0`000P000<l1V`?>0I/3E`0002@0003<0@00V`<00<h1002L0`000P000<h1V`?<0I`3
E`0002@0003;0@00W0<00<`1002M0`000P000<`1W0?;0Id3E`0002@0003:0@00W@<00</1002O0`00
0P000</1W@?:0Il3E`0002@000390@00W`<00<X1002S0`000P000<X1W`?90J<3E`0002@000390@00
X`<00<T1002V0`000P000<T1X`?90JH3E`0002@000390@00YP<00<X1002[0`000P000<T1YP?:0J/3
E`0002@0003:0@00Z`<00</1002]0`000P000<X1Z`?;0Jd3E`0002@0003;0@00[@<00<`1002^0`00
0P000</1[@?<0Jh3E`0002@0003<0@00[P<00<h1002^0`000P000<`1[P?>0Jh3E`0002@0003?0@00
[P<00=41002^0`000P000<l1[P?A0Jh3E`0002@0003A0@00[@<00=81002^0`000P000=41[P?B0Jd3
E`0002@0003B0@00Z`<00=<1002]0`000P000=81[@?C0J/3E`0002@0003C0@00YP<00=@1002[0`00
0P000=<1Z`?D0JH3E`0002@0003D0@00X`<00=@1002V0`000P000=@1YP?D0J<3E`0002@0003C0@00
W`<00=@1002S0`000P000=@1X`?C0Il3E`0002@0003B0@00W@<00=<1002O0`000P000=<1W`?B0Id3
E`0002@0003A0@00W0<00=81002M0`000P000=81W@?A0I`3E`0002@0003?0@00V`<00=41002L0`00
0P000=41W0??0I/3E`0002@0003R0@00W@<00>81002^0`000P000>81W@?R0Jh3E`0002@0003S0@00
V`<00><1002^0`000P000><1V`?S0Jh3E`0002@0003I0@00V`<00><1002Y0`000P000><1V`?I0JT3
E`0002@0003I0@00Z@<00>P1002Y0`000P000=T1Z@?X0JT3E`0002@0003P0@00[P<00>H1002^0`00
0P000>01[P?V0Jh3E`0002@0000h0P00V`<003/2002L0`000P0003/2V`<h0Y`3E`0002@0000f0P00
W0<003P2002O0`000P0003P2W0<f0Yl3E`0002@0000e0P00W`<003H2002S0`000P0003H2W`<e0Z<3
E`0002@0000e0P00X`<003D2002V0`000P0003D2X`<e0ZH3E`0002@0000e0P00YP<003H2002[0`00
0P0003D2YP<f0Z/3E`0002@0000f0P00Z`<003P2002^0`000P0003H2Z`<h0Zh3E`0002@0000h0P00
[P<003/2002^0`000P0003P2[P<k0Zh3E`0002@0000k0P00[P<003`2002^0`000P0003/2[P<l0Zh3
E`0002@0000l0P00[P<003l2002^0`000P0003`2[P<o0Zh3E`0002@0000o0P00Z`<00442002^0`00
0P0003l2[P=10Z/3E`0002@000110P00YP<00482002[0`000P000442Z`=20ZH3E`0002@000120P00
X`<00482002V0`000P000482YP=20Z<3E`0002@000110P00W`<00482002S0`000P000482X`=10Yl3
E`0002@0000o0P00W0<00442002O0`000P000442W`<o0Y`3E`0002@0000l0P00V`<003l2002L0`00
0P0003l2W0<l0Y/3E`0002@0000k0P00V`<003`2002K0`000P0003`2V`<k0Y/3E`0002@0000i0P00
V`<003/2002L0`000P0003/2V`<i0Y`3E`0002@0000h0P00W0<003T2002M0`000P0003T2W0<h0Yd3
E`0002@0000g0P00W@<003P2002O0`000P0003P2W@<g0Yl3E`0002@0000f0P00W`<003L2002S0`00
0P0003L2W`<f0Z<3E`0002@0000f0P00X`<003H2002V0`000P0003H2X`<f0ZH3E`0002@0000f0P00
YP<003L2002[0`000P0003H2YP<g0Z/3E`0002@0000g0P00Z`<003P2002]0`000P0003L2Z`<h0Zd3
E`0002@0000h0P00[@<003T2002^0`000P0003P2[@<i0Zh3E`0002@0000i0P00[P<003/2002^0`00
0P0003T2[P<k0Zh3E`0002@0000l0P00[P<003h2002^0`000P0003`2[P<n0Zh3E`0002@0000n0P00
[@<003l2002^0`000P0003h2[P<o0Zd3E`0002@0000o0P00Z`<00402002]0`000P0003l2[@=00Z/3
E`0002@000100P00YP<00442002[0`000P000402Z`=10ZH3E`0002@000110P00X`<00442002V0`00
0P000442YP=10Z<3E`0002@000100P00W`<00442002S0`000P000442X`=00Yl3E`0002@0000o0P00
W@<00402002O0`000P000402W`<o0Yd3E`0002@0000n0P00W0<003l2002M0`000P0003l2W@<n0Y`3
E`0002@0000l0P00V`<003h2002L0`000P0003h2W0<l0Y/3E`0002@000170P00[@<004P2002^0`00
0P0004P2[@=70Zh3E`0002@000170P00[P<004P2002^0`000P0004L2[P=80Zh3E`0002@000180P00
[P<004T2002^0`000P0004P2[P=90Zh3E`0002@000180P00[@<004T2002^0`000P0004T2[P=80Zd3
E`0002@0001B0P00V`<005D2002L0`000P0005D2V`=B0Y`3E`0002@0001@0P00W0<00582002O0`00
0P000582W0=@0Yl3E`0002@0001?0P00W`<00502002S0`000P000502W`=?0Z<3E`0002@0001?0P00
X`<004l2002V0`000P0004l2X`=?0ZH3E`0002@0001?0P00YP<00502002[0`000P0004l2YP=@0Z/3
E`0002@0001@0P00Z`<00582002^0`000P000502Z`=B0Zh3E`0002@0001B0P00[P<005D2002^0`00
0P000582[P=E0Zh3E`0002@0001E0P00[P<005H2002^0`000P0005D2[P=F0Zh3E`0002@0001F0P00
[P<005T2002^0`000P0005H2[P=I0Zh3E`0002@0001I0P00Z`<005/2002^0`000P0005T2[P=K0Z/3
E`0002@0001K0P00YP<005`2002[0`000P0005/2Z`=L0ZH3E`0002@0001L0P00X`<005`2002V0`00
0P0005`2YP=L0Z<3E`0002@0001K0P00W`<005`2002S0`000P0005`2X`=K0Yl3E`0002@0001I0P00
W0<005/2002O0`000P0005/2W`=I0Y`3E`0002@0001F0P00V`<005T2002L0`000P0005T2W0=F0Y/3
E`0002@0001E0P00V`<005H2002K0`000P0005H2V`=E0Y/3E`0002@0001C0P00V`<005D2002L0`00
0P0005D2V`=C0Y`3E`0002@0001B0P00W0<005<2002M0`000P0005<2W0=B0Yd3E`0002@0001A0P00
W@<00582002O0`000P000582W@=A0Yl3E`0002@0001@0P00W`<00542002S0`000P000542W`=@0Z<3
E`0002@0001@0P00X`<00502002V0`000P000502X`=@0ZH3E`0002@0001@0P00YP<00542002[0`00
0P000502YP=A0Z/3E`0002@0001A0P00Z`<00582002]0`000P000542Z`=B0Zd3E`0002@0001B0P00
[@<005<2002^0`000P000582[@=C0Zh3E`0002@0001C0P00[P<005D2002^0`000P0005<2[P=E0Zh3
E`0002@0001F0P00[P<005P2002^0`000P0005H2[P=H0Zh3E`0002@0001H0P00[@<005T2002^0`00
0P0005P2[P=I0Zd3E`0002@0001I0P00Z`<005X2002]0`000P0005T2[@=J0Z/3E`0002@0001J0P00
YP<005/2002[0`000P0005X2Z`=K0ZH3E`0002@0001K0P00X`<005/2002V0`000P0005/2YP=K0Z<3
E`0002@0001J0P00W`<005/2002S0`000P0005/2X`=J0Yl3E`0002@0001I0P00W@<005X2002O0`00
0P0005X2W`=I0Yd3E`0002@0001H0P00W0<005T2002M0`000P0005T2W@=H0Y`3E`0002@0001F0P00
V`<005P2002L0`000P0005P2W0=F0Y/3E`0002@0001T0P00V`<006L2002L0`000P0006L2V`=T0Y`3
E`0002@0001R0P00W0<006@2002O0`000P0006@2W0=R0Yl3E`0002@0001Q0P00W`<00682002S0`00
0P000682W`=Q0Z<3E`0002@0001Q0P00X`<00642002V0`000P000642X`=Q0ZH3E`0002@0001Q0P00
YP<00682002[0`000P000642YP=R0Z/3E`0002@0001R0P00Z`<006@2002^0`000P000682Z`=T0Zh3
E`0002@0001T0P00[P<006L2002^0`000P0006@2[P=W0Zh3E`0002@0001W0P00[P<006P2002^0`00
0P0006L2[P=X0Zh3E`0002@0001X0P00[P<006/2002^0`000P0006P2[P=[0Zh3E`0002@0001[0P00
Z`<006d2002^0`000P0006/2[P=]0Z/3E`0002@0001]0P00YP<006h2002[0`000P0006d2Z`=^0ZH3
E`0002@0001^0P00X`<006h2002V0`000P0006h2YP=^0Z<3E`0002@0001]0P00W`<006h2002S0`00
0P0006h2X`=]0Yl3E`0002@0001[0P00W0<006d2002O0`000P0006d2W`=[0Y`3E`0002@0001X0P00
V`<006/2002L0`000P0006/2W0=X0Y/3E`0002@0001W0P00V`<006P2002K0`000P0006P2V`=W0Y/3
E`0002@0001U0P00V`<006L2002L0`000P0006L2V`=U0Y`3E`0002@0001T0P00W0<006D2002M0`00
0P0006D2W0=T0Yd3E`0002@0001S0P00W@<006@2002O0`000P0006@2W@=S0Yl3E`0002@0001R0P00
W`<006<2002S0`000P0006<2W`=R0Z<3E`0002@0001R0P00X`<00682002V0`000P000682X`=R0ZH3
E`0002@0001R0P00YP<006<2002[0`000P000682YP=S0Z/3E`0002@0001S0P00Z`<006@2002]0`00
0P0006<2Z`=T0Zd3E`0002@0001T0P00[@<006D2002^0`000P0006@2[@=U0Zh3E`0002@0001U0P00
[P<006L2002^0`000P0006D2[P=W0Zh3E`0002@0001X0P00[P<006X2002^0`000P0006P2[P=Z0Zh3
E`0002@0001Z0P00[@<006/2002^0`000P0006X2[P=[0Zd3E`0002@0001[0P00Z`<006`2002]0`00
0P0006/2[@=/0Z/3E`0002@0001/0P00YP<006d2002[0`000P0006`2Z`=]0ZH3E`0002@0001]0P00
X`<006d2002V0`000P0006d2YP=]0Z<3E`0002@0001/0P00W`<006d2002S0`000P0006d2X`=/0Yl3
E`0002@0001[0P00W@<006`2002O0`000P0006`2W`=[0Yd3E`0002@0001Z0P00W0<006/2002M0`00
0P0006/2W@=Z0Y`3E`0002@0001X0P00V`<006X2002L0`000P0006X2W0=X0Y/3E`0002@0001m0P00
WP<007h2002O0`000P0007h2WP=m0Yl3E`0002@0001m0P00W`<007h2002P0`000P0007d2W`=n0Z03
E`0002@0001n0P00W`<007l2002P0`000P0007h2X0=o0Yl3E`0002@0001o0P00WP<007l2002O0`00
0P0007l2W`=o0Yh3E`0002@0001n0P00W0<007l2002N0`000P0007l2WP=n0Y`3E`0002@0001l0P00
V`<007h2002L0`000P0007h2W0=l0Y/3E`0002@0001i0P00V`<007`2002K0`000P0007`2V`=i0Y/3
E`0002@0001g0P00V`<007T2002L0`000P0007T2V`=g0Y`3E`0002@0001e0P00W0<007L2002N0`00
0P0007L2W0=e0Yh3E`0002@0001d0P00WP<007D2002P0`000P0007D2WP=d0Z03E`0002@0001c0P00
X0<007@2002S0`000P0007@2X0=c0Z<3E`0002@0001c0P00X`<007<2002Y0`000P0007<2X`=c0ZT3
E`0002@0001c0P00Z@<007@2002/0`000P0007<2Z@=d0Z`3E`0002@0001d0P00[0<007H2002^0`00
0P0007@2[0=f0Zh3E`0002@0001f0P00[P<007T2002^0`000P0007H2[P=i0Zh3E`0002@0001i0P00
[P<007X2002^0`000P0007T2[P=j0Zh3E`0002@0001j0P00[P<007d2002^0`000P0007X2[P=m0Zh3
E`0002@0001m0P00[0<007l2002^0`000P0007d2[P=o0Z`3E`0002@0001o0P00Z@<00802002/0`00
0P0007l2[0>00ZT3E`0002@000200P00Z0<00802002Y0`000P000802Z@>00ZP3E`0002@0001o0P00
Y@<00802002X0`000P000802Z0=o0ZD3E`0002@0001m0P00X`<007l2002U0`000P0007l2Y@=m0Z<3
E`0002@0001j0P00XP<007d2002S0`000P0007d2X`=j0Z83E`0002@0001i0P00XP<007X2002R0`00
0P0007X2XP=i0Z83E`0002@0001g0P00XP<007T2002S0`000P0007T2XP=g0Z<3E`0002@0001e0P00
X`<007L2002U0`000P0007L2X`=e0ZD3E`0002@0001d0P00Y@<007D2002X0`000P0007D2Y@=d0ZP3
E`0002@0001h0P00V`<007T2002L0`000P0007T2V`=h0Y`3E`0002@0001f0P00W0<007P2002N0`00
0P0007P2W0=f0Yh3E`0002@0001e0P00WP<007H2002P0`000P0007H2WP=e0Z03E`0002@0001d0P00
X0<007D2002S0`000P0007D2X0=d0Z<3E`0002@0001d0P00X`<007@2002Y0`000P0007@2X`=d0ZT3
E`0002@0001d0P00Z@<007D2002/0`000P0007@2Z@=e0Z`3E`0002@0001e0P00[0<007L2002^0`00
0P0007D2[0=g0Zh3E`0002@0001g0P00[P<007T2002^0`000P0007L2[P=i0Zh3E`0002@0001j0P00
[P<007`2002^0`000P0007X2[P=l0Zh3E`0002@0001l0P00[0<007h2002^0`000P0007`2[P=n0Z`3
E`0002@0001n0P00Z@<007l2002/0`000P0007h2[0=o0ZT3E`0002@0001o0P00Z0<007l2002Y0`00
0P0007l2Z@=o0ZP3E`0002@0001n0P00Y@<007l2002X0`000P0007l2Z0=n0ZD3E`0002@0001l0P00
X`<007h2002U0`000P0007h2Y@=l0Z<3E`0002@0001j0P00XP<007`2002S0`000P0007`2X`=j0Z83
E`0002@0003A0P00V`<00=@2002L0`000P000=@2V`?A0Y`3E`0002@0003?0P00W0<00=42002O0`00
0P000=42W0??0Yl3E`0002@0003>0P00W`<00<l2002S0`000P000<l2W`?>0Z<3E`0002@0003>0P00
X`<00<h2002V0`000P000<h2X`?>0ZH3E`0002@0003>0P00YP<00<l2002[0`000P000<h2YP??0Z/3
E`0002@0003?0P00Z`<00=42002^0`000P000<l2Z`?A0Zh3E`0002@0003A0P00[P<00=@2002^0`00
0P000=42[P?D0Zh3E`0002@0003D0P00[P<00=D2002^0`000P000=@2[P?E0Zh3E`0002@0003E0P00
[P<00=P2002^0`000P000=D2[P?H0Zh3E`0002@0003H0P00Z`<00=X2002^0`000P000=P2[P?J0Z/3
E`0002@0003J0P00YP<00=/2002[0`000P000=X2Z`?K0ZH3E`0002@0003K0P00X`<00=/2002V0`00
0P000=/2YP?K0Z<3E`0002@0003J0P00W`<00=/2002S0`000P000=/2X`?J0Yl3E`0002@0003H0P00
W0<00=X2002O0`000P000=X2W`?H0Y`3E`0002@0003E0P00V`<00=P2002L0`000P000=P2W0?E0Y/3
E`0002@0003D0P00V`<00=D2002K0`000P000=D2V`?D0Y/3E`0002@0003B0P00V`<00=@2002L0`00
0P000=@2V`?B0Y`3E`0002@0003A0P00W0<00=82002M0`000P000=82W0?A0Yd3E`0002@0003@0P00
W@<00=42002O0`000P000=42W@?@0Yl3E`0002@0003?0P00W`<00=02002S0`000P000=02W`??0Z<3
E`0002@0003?0P00X`<00<l2002V0`000P000<l2X`??0ZH3E`0002@0003?0P00YP<00=02002[0`00
0P000<l2YP?@0Z/3E`0002@0003@0P00Z`<00=42002]0`000P000=02Z`?A0Zd3E`0002@0003A0P00
[@<00=82002^0`000P000=42[@?B0Zh3E`0002@0003B0P00[P<00=@2002^0`000P000=82[P?D0Zh3
E`0002@0003E0P00[P<00=L2002^0`000P000=D2[P?G0Zh3E`0002@0003G0P00[@<00=P2002^0`00
0P000=L2[P?H0Zd3E`0002@0003H0P00Z`<00=T2002]0`000P000=P2[@?I0Z/3E`0002@0003I0P00
YP<00=X2002[0`000P000=T2Z`?J0ZH3E`0002@0003J0P00X`<00=X2002V0`000P000=X2YP?J0Z<3
E`0002@0003I0P00W`<00=X2002S0`000P000=X2X`?I0Yl3E`0002@0003H0P00W@<00=T2002O0`00
0P000=T2W`?H0Yd3E`0002@0003G0P00W0<00=P2002M0`000P000=P2W@?G0Y`3E`0002@0003E0P00
V`<00=L2002L0`000P000=L2W0?E0Y/3E`0002@0003P0P00[@<00>42002^0`000P000>42[@?P0Zh3
E`0002@0003P0P00[P<00>42002^0`000P000>02[P?Q0Zh3E`0002@0003Q0P00[P<00>82002^0`00
0P000>42[P?R0Zh3E`0002@0003Q0P00[@<00>82002^0`000P000>82[P?Q0Zd3E`0002@0003[0P00
V`<00>h2002L0`000P000>h2V`?[0Y`3E`0002@0003Y0P00W0<00>/2002O0`000P000>/2W0?Y0Yl3
E`0002@0003X0P00W`<00>T2002S0`000P000>T2W`?X0Z<3E`0002@0003X0P00X`<00>P2002V0`00
0P000>P2X`?X0ZH3E`0002@0003X0P00YP<00>T2002[0`000P000>P2YP?Y0Z/3E`0002@0003Y0P00
Z`<00>/2002^0`000P000>T2Z`?[0Zh3E`0002@0003[0P00[P<00>h2002^0`000P000>/2[P?^0Zh3
E`0002@0003^0P00[P<00>l2002^0`000P000>h2[P?_0Zh3E`0002@0003_0P00[P<00?82002^0`00
0P000>l2[P?b0Zh3E`0002@0003b0P00Z`<00?@2002^0`000P000?82[P?d0Z/3E`0002@0003d0P00
YP<00?D2002[0`000P000?@2Z`?e0ZH3E`0002@0003e0P00X`<00?D2002V0`000P000?D2YP?e0Z<3
E`0002@0003d0P00W`<00?D2002S0`000P000?D2X`?d0Yl3E`0002@0003b0P00W0<00?@2002O0`00
0P000?@2W`?b0Y`3E`0002@0003_0P00V`<00?82002L0`000P000?82W0?_0Y/3E`0002@0003^0P00
V`<00>l2002K0`000P000>l2V`?^0Y/3E`0002@0003/0P00V`<00>h2002L0`000P000>h2V`?/0Y`3
E`0002@0003[0P00W0<00>`2002M0`000P000>`2W0?[0Yd3E`0002@0003Z0P00W@<00>/2002O0`00
0P000>/2W@?Z0Yl3E`0002@0003Y0P00W`<00>X2002S0`000P000>X2W`?Y0Z<3E`0002@0003Y0P00
X`<00>T2002V0`000P000>T2X`?Y0ZH3E`0002@0003Y0P00YP<00>X2002[0`000P000>T2YP?Z0Z/3
E`0002@0003Z0P00Z`<00>/2002]0`000P000>X2Z`?[0Zd3E`0002@0003[0P00[@<00>`2002^0`00
0P000>/2[@?/0Zh3E`0002@0003/0P00[P<00>h2002^0`000P000>`2[P?^0Zh3E`0002@0003_0P00
[P<00?42002^0`000P000>l2[P?a0Zh3E`0002@0003a0P00[@<00?82002^0`000P000?42[P?b0Zd3
E`0002@0003b0P00Z`<00?<2002]0`000P000?82[@?c0Z/3E`0002@0003c0P00YP<00?@2002[0`00
0P000?<2Z`?d0ZH3E`0002@0003d0P00X`<00?@2002V0`000P000?@2YP?d0Z<3E`0002@0003c0P00
W`<00?@2002S0`000P000?@2X`?c0Yl3E`0002@0003b0P00W@<00?<2002O0`000P000?<2W`?b0Yd3
E`0002@0003a0P00W0<00?82002M0`000P000?82W@?a0Y`3E`0002@0003_0P00V`<00?42002L0`00
0P000?42W0?_0Y/3E`0002@0003m0P00V`<00003002L0`000P000003V`?m0Y`3E`0002@0003k0P00
W0<00?d2002O0`000P000?d2W0?k0Yl3E`0002@0003j0P00W`<00?/2002S0`000P000?/2W`?j0Z<3
E`0002@0003j0P00X`<00?X2002V0`000P000?X2X`?j0ZH3E`0002@0003j0P00YP<00?/2002[0`00
0P000?X2YP?k0Z/3E`0002@0003k0P00Z`<00?d2002^0`000P000?/2Z`?m0Zh3E`0002@0003m0P00
[P<00003002^0`000P000?d2[P<00jh3E`0002@000000`00[P<00043002^0`000P000003[P<10jh3
E`0002@000010`00[P<000@3002^0`000P000043[P<40jh3E`0002@000040`00Z`<000H3002^0`00
0P0000@3[P<60j/3E`0002@000060`00YP<000L3002[0`000P0000H3Z`<70jH3E`0002@000070`00
X`<000L3002V0`000P0000L3YP<70j<3E`0002@000060`00W`<000L3002S0`000P0000L3X`<60il3
E`0002@000040`00W0<000H3002O0`000P0000H3W`<40i`3E`0002@000010`00V`<000@3002L0`00
0P0000@3W0<10i/3E`0002@000000`00V`<00043002K0`000P000043V`<00i/3E`0002@0003n0P00
V`<00003002L0`000P000003V`?n0Y`3E`0002@0003m0P00W0<00?h2002M0`000P000?h2W0?m0Yd3
E`0002@0003l0P00W@<00?d2002O0`000P000?d2W@?l0Yl3E`0002@0003k0P00W`<00?`2002S0`00
0P000?`2W`?k0Z<3E`0002@0003k0P00X`<00?/2002V0`000P000?/2X`?k0ZH3E`0002@0003k0P00
YP<00?`2002[0`000P000?/2YP?l0Z/3E`0002@0003l0P00Z`<00?d2002]0`000P000?`2Z`?m0Zd3
E`0002@0003m0P00[@<00?h2002^0`000P000?d2[@?n0Zh3E`0002@0003n0P00[P<00003002^0`00
0P000?h2[P<00jh3E`0002@000010`00[P<000<3002^0`000P000043[P<30jh3E`0002@000030`00
[@<000@3002^0`000P0000<3[P<40jd3E`0002@000040`00Z`<000D3002]0`000P0000@3[@<50j/3
E`0002@000050`00YP<000H3002[0`000P0000D3Z`<60jH3E`0002@000060`00X`<000H3002V0`00
0P0000H3YP<60j<3E`0002@000050`00W`<000H3002S0`000P0000H3X`<50il3E`0002@000040`00
W@<000D3002O0`000P0000D3W`<40id3E`0002@000030`00W0<000@3002M0`000P0000@3W@<30i`3
E`0002@000010`00V`<000<3002L0`000P0000<3W0<10i/3E`0002@0000>0`00V`<00143002L0`00
0P000143V`<>0i`3E`0002@0000=0`00W0<000h3002N0`000P0000h3W0<=0ih3E`0002@0000=0`00
WP<000d3002P0`000P0000d3WP<=0j03E`0002@0000=0`00X0<000h3002R0`000P0000d3X0<>0j83
E`0002@0000>0`00XP<00143002S0`000P0000h3XP<A0j<3E`0002@0000A0`00X`<001@3002S0`00
0P000143X`<D0j<3E`0002@0000D0`00XP<001L3002S0`000P0001@3X`<G0j83E`0002@0000G0`00
X0<001P3002R0`000P0001L3XP<H0j03E`0002@0000H0`00WP<001P3002P0`000P0001P3X0<H0ih3
E`0002@0000G0`00W0<001P3002N0`000P0001P3WP<G0i`3E`0002@0000D0`00V`<001L3002L0`00
0P0001L3W0<D0i/3E`0002@0000A0`00V`<001@3002K0`000P0001@3V`<A0i/3E`0002@0000?0`00
V`<00143002L0`000P000143V`<?0i`3E`0002@0000>0`00W0<000l3002N0`000P0000l3W0<>0ih3
E`0002@0000>0`00WP<000h3002P0`000P0000h3WP<>0j03E`0002@0000>0`00X0<000l3002R0`00
0P0000h3X0<?0j83E`0002@0000?0`00XP<00143002S0`000P0000l3XP<A0j<3E`0002@0000D0`00
XP<001H3002S0`000P0001@3X`<F0j83E`0002@0000F0`00X0<001L3002R0`000P0001H3XP<G0j03
E`0002@0000G0`00WP<001L3002P0`000P0001L3X0<G0ih3E`0002@0000F0`00W0<001L3002N0`00
0P0001L3WP<F0i`3E`0002@0000D0`00V`<001H3002L0`000P0001H3W0<D0i/3E`0002@0000>0`00
X`<00143002T0`000P000143X`<>0j@3E`0002@0000=0`00Y0<000h3002U0`000P0000h3Y0<=0jD3
E`0002@0000<0`00Y@<000d3002W0`000P0000d3Y@<<0jL3E`0002@0000<0`00Y`<000`3002[0`00
0P0000`3Y`<<0j/3E`0002@0000<0`00Z`<000d3002]0`000P0000`3Z`<=0jd3E`0002@0000=0`00
[@<000h3002^0`000P0000d3[@<>0jh3E`0002@0000>0`00[P<00143002^0`000P0000h3[P<A0jh3
E`0002@0000A0`00[P<001@3002^0`000P000143[P<D0jh3E`0002@0000D0`00[P<001L3002^0`00
0P0001@3[P<G0jh3E`0002@0000G0`00[@<001P3002^0`000P0001L3[P<H0jd3E`0002@0000H0`00
Z`<001T3002]0`000P0001P3[@<I0j/3E`0002@0000I0`00Y`<001T3002[0`000P0001T3Z`<I0jL3
E`0002@0000H0`00Y@<001T3002W0`000P0001T3Y`<H0jD3E`0002@0000G0`00Y0<001P3002U0`00
0P0001P3Y@<G0j@3E`0002@0000D0`00X`<001L3002T0`000P0001L3Y0<D0j<3E`0002@0000?0`00
X`<00143002T0`000P000143X`<?0j@3E`0002@0000>0`00Y0<000l3002U0`000P0000l3Y0<>0jD3
E`0002@0000=0`00Y@<000h3002W0`000P0000h3Y@<=0jL3E`0002@0000=0`00Y`<000d3002[0`00
0P0000d3Y`<=0j/3E`0002@0000=0`00Z`<000h3002]0`000P0000d3Z`<>0jd3E`0002@0000>0`00
[@<000l3002^0`000P0000h3[@<?0jh3E`0002@0000?0`00[P<00143002^0`000P0000l3[P<A0jh3
E`0002@0000D0`00[P<001H3002^0`000P0001@3[P<F0jh3E`0002@0000F0`00[@<001L3002^0`00
0P0001H3[P<G0jd3E`0002@0000G0`00Z`<001P3002]0`000P0001L3[@<H0j/3E`0002@0000H0`00
Y`<001P3002[0`000P0001P3Z`<H0jL3E`0002@0000G0`00Y@<001P3002W0`000P0001P3Y`<G0jD3
E`0002@0000F0`00Y0<001L3002U0`000P0001L3Y@<F0j@3E`0002@0000D0`00X`<001H3002T0`00
0P0001H3Y0<D0j<3E`0002@0001Z0`00V`<006d3002L0`000P0006d3V`=Z0i`3E`0002@0001X0`00
W0<006X3002O0`000P0006X3W0=X0il3E`0002@0001W0`00W`<006P3002S0`000P0006P3W`=W0j<3
E`0002@0001W0`00X`<006L3002V0`000P0006L3X`=W0jH3E`0002@0001W0`00YP<006P3002[0`00
0P0006L3YP=X0j/3E`0002@0001X0`00Z`<006X3002^0`000P0006P3Z`=Z0jh3E`0002@0001Z0`00
[P<006d3002^0`000P0006X3[P=]0jh3E`0002@0001]0`00[P<006h3002^0`000P0006d3[P=^0jh3
E`0002@0001^0`00[P<00743002^0`000P0006h3[P=a0jh3E`0002@0001a0`00Z`<007<3002^0`00
0P000743[P=c0j/3E`0002@0001c0`00YP<007@3002[0`000P0007<3Z`=d0jH3E`0002@0001d0`00
X`<007@3002V0`000P0007@3YP=d0j<3E`0002@0001c0`00W`<007@3002S0`000P0007@3X`=c0il3
E`0002@0001a0`00W0<007<3002O0`000P0007<3W`=a0i`3E`0002@0001^0`00V`<00743002L0`00
0P000743W0=^0i/3E`0002@0001]0`00V`<006h3002K0`000P0006h3V`=]0i/3E`0002@0001[0`00
V`<006d3002L0`000P0006d3V`=[0i`3E`0002@0001Z0`00W0<006/3002M0`000P0006/3W0=Z0id3
E`0002@0001Y0`00W@<006X3002O0`000P0006X3W@=Y0il3E`0002@0001X0`00W`<006T3002S0`00
0P0006T3W`=X0j<3E`0002@0001X0`00X`<006P3002V0`000P0006P3X`=X0jH3E`0002@0001X0`00
YP<006T3002[0`000P0006P3YP=Y0j/3E`0002@0001Y0`00Z`<006X3002]0`000P0006T3Z`=Z0jd3
E`0002@0001Z0`00[@<006/3002^0`000P0006X3[@=[0jh3E`0002@0001[0`00[P<006d3002^0`00
0P0006/3[P=]0jh3E`0002@0001^0`00[P<00703002^0`000P0006h3[P=`0jh3E`0002@0001`0`00
[@<00743002^0`000P000703[P=a0jd3E`0002@0001a0`00Z`<00783002]0`000P000743[@=b0j/3
E`0002@0001b0`00YP<007<3002[0`000P000783Z`=c0jH3E`0002@0001c0`00X`<007<3002V0`00
0P0007<3YP=c0j<3E`0002@0001b0`00W`<007<3002S0`000P0007<3X`=b0il3E`0002@0001a0`00
W@<00783002O0`000P000783W`=a0id3E`0002@0001`0`00W0<00743002M0`000P000743W@=`0i`3
E`0002@0001^0`00V`<00703002L0`000P000703W0=^0i/3E`0002@0001j0`00[@<007/3002^0`00
0P0007/3[@=j0jh3E`0002@0001j0`00[P<007/3002^0`000P0007X3[P=k0jh3E`0002@0001k0`00
[P<007`3002^0`000P0007/3[P=l0jh3E`0002@0001k0`00[@<007`3002^0`000P0007`3[P=k0jd3
E`0002@000240`00V`<008L3002L0`000P0008L3V`>40i`3E`0002@000220`00W0<008@3002O0`00
0P0008@3W0>20il3E`0002@000210`00W`<00883002S0`000P000883W`>10j<3E`0002@000210`00
X`<00843002V0`000P000843X`>10jH3E`0002@000210`00YP<00883002[0`000P000843YP>20j/3
E`0002@000220`00Z`<008@3002^0`000P000883Z`>40jh3E`0002@000240`00[P<008L3002^0`00
0P0008@3[P>70jh3E`0002@000270`00[P<008P3002^0`000P0008L3[P>80jh3E`0002@000280`00
[P<008/3002^0`000P0008P3[P>;0jh3E`0002@0002;0`00Z`<008d3002^0`000P0008/3[P>=0j/3
E`0002@0002=0`00YP<008h3002[0`000P0008d3Z`>>0jH3E`0002@0002>0`00X`<008h3002V0`00
0P0008h3YP>>0j<3E`0002@0002=0`00W`<008h3002S0`000P0008h3X`>=0il3E`0002@0002;0`00
W0<008d3002O0`000P0008d3W`>;0i`3E`0002@000280`00V`<008/3002L0`000P0008/3W0>80i/3
E`0002@000270`00V`<008P3002K0`000P0008P3V`>70i/3E`0002@000250`00V`<008L3002L0`00
0P0008L3V`>50i`3E`0002@000240`00W0<008D3002M0`000P0008D3W0>40id3E`0002@000230`00
W@<008@3002O0`000P0008@3W@>30il3E`0002@000220`00W`<008<3002S0`000P0008<3W`>20j<3
E`0002@000220`00X`<00883002V0`000P000883X`>20jH3E`0002@000220`00YP<008<3002[0`00
0P000883YP>30j/3E`0002@000230`00Z`<008@3002]0`000P0008<3Z`>40jd3E`0002@000240`00
[@<008D3002^0`000P0008@3[@>50jh3E`0002@000250`00[P<008L3002^0`000P0008D3[P>70jh3
E`0002@000280`00[P<008X3002^0`000P0008P3[P>:0jh3E`0002@0002:0`00[@<008/3002^0`00
0P0008X3[P>;0jd3E`0002@0002;0`00Z`<008`3002]0`000P0008/3[@><0j/3E`0002@0002<0`00
YP<008d3002[0`000P0008`3Z`>=0jH3E`0002@0002=0`00X`<008d3002V0`000P0008d3YP>=0j<3
E`0002@0002<0`00W`<008d3002S0`000P0008d3X`><0il3E`0002@0002;0`00W@<008`3002O0`00
0P0008`3W`>;0id3E`0002@0002:0`00W0<008/3002M0`000P0008/3W@>:0i`3E`0002@000280`00
V`<008X3002L0`000P0008X3W0>80i/3E`0002@0002F0`00WP<009P3002O0`000P0009H3W`>H0ih3
E`0002@0002H0`00V`<009X3002N0`000P0009P3WP>J0i/3E`0002@0002J0`00V`<009X3002^0`00
0P0009X3V`>J0jh3E`0002@0002I0`00W0<009T3002^0`000P0009T3W0>I0jh3E`0002@0002F0`00
[P<009h3002^0`000P0009H3[P>N0jh3E`0002@0002X0`00V`<00:/3002L0`000P000:/3V`>X0i`3
E`0002@0002V0`00W0<00:P3002O0`000P000:P3W0>V0il3E`0002@0002U0`00W`<00:H3002S0`00
0P000:H3W`>U0j<3E`0002@0002U0`00X`<00:D3002V0`000P000:D3X`>U0jH3E`0002@0002U0`00
YP<00:H3002[0`000P000:D3YP>V0j/3E`0002@0002V0`00Z`<00:P3002^0`000P000:H3Z`>X0jh3
E`0002@0002X0`00[P<00:/3002^0`000P000:P3[P>[0jh3E`0002@0002[0`00[P<00:`3002^0`00
0P000:/3[P>/0jh3E`0002@0002/0`00[P<00:l3002^0`000P000:`3[P>_0jh3E`0002@0002_0`00
Z`<00;43002^0`000P000:l3[P>a0j/3E`0002@0002a0`00YP<00;83002[0`000P000;43Z`>b0jH3
E`0002@0002b0`00X`<00;83002V0`000P000;83YP>b0j<3E`0002@0002a0`00W`<00;83002S0`00
0P000;83X`>a0il3E`0002@0002_0`00W0<00;43002O0`000P000;43W`>_0i`3E`0002@0002/0`00
V`<00:l3002L0`000P000:l3W0>/0i/3E`0002@0002[0`00V`<00:`3002K0`000P000:`3V`>[0i/3
E`0002@0002Y0`00V`<00:/3002L0`000P000:/3V`>Y0i`3E`0002@0002X0`00W0<00:T3002M0`00
0P000:T3W0>X0id3E`0002@0002W0`00W@<00:P3002O0`000P000:P3W@>W0il3E`0002@0002V0`00
W`<00:L3002S0`000P000:L3W`>V0j<3E`0002@0002V0`00X`<00:H3002V0`000P000:H3X`>V0jH3
E`0002@0002V0`00YP<00:L3002[0`000P000:H3YP>W0j/3E`0002@0002W0`00Z`<00:P3002]0`00
0P000:L3Z`>X0jd3E`0002@0002X0`00[@<00:T3002^0`000P000:P3[@>Y0jh3E`0002@0002Y0`00
[P<00:/3002^0`000P000:T3[P>[0jh3E`0002@0002/0`00[P<00:h3002^0`000P000:`3[P>^0jh3
E`0002@0002^0`00[@<00:l3002^0`000P000:h3[P>_0jd3E`0002@0002_0`00Z`<00;03002]0`00
0P000:l3[@>`0j/3E`0002@0002`0`00YP<00;43002[0`000P000;03Z`>a0jH3E`0002@0002a0`00
X`<00;43002V0`000P000;43YP>a0j<3E`0002@0002`0`00W`<00;43002S0`000P000;43X`>`0il3
E`0002@0002_0`00W@<00;03002O0`000P000;03W`>_0id3E`0002@0002^0`00W0<00:l3002M0`00
0P000:l3W@>^0i`3E`0002@0002/0`00V`<00:h3002L0`000P000:h3W0>/0i/3E`0002@000031000
V`<000H4002L0`000P0000H4V`<319`3E`0002@000011000W0<000<4002O0`000P0000<4W0<119l3
E`0002@000001000W`<00044002S0`000P000044W`<01:<3E`0002@000001000X`<00004002V0`00
0P000004X`<01:H3E`0002@000001000YP<00044002[0`000P000004YP<11:/3E`0002@000011000
Z`<000<4002^0`000P000044Z`<31:h3E`0002@000031000[P<000H4002^0`000P0000<4[P<61:h3
E`0002@000061000[P<000L4002^0`000P0000H4[P<71:h3E`0002@000071000[P<000X4002^0`00
0P0000L4[P<:1:h3E`0002@0000:1000Z`<000`4002^0`000P0000X4[P<<1:/3E`0002@0000<1000
YP<000d4002[0`000P0000`4Z`<=1:H3E`0002@0000=1000X`<000d4002V0`000P0000d4YP<=1:<3
E`0002@0000<1000W`<000d4002S0`000P0000d4X`<<19l3E`0002@0000:1000W0<000`4002O0`00
0P0000`4W`<:19`3E`0002@000071000V`<000X4002L0`000P0000X4W0<719/3E`0002@000061000
V`<000L4002K0`000P0000L4V`<619/3E`0002@000041000V`<000H4002L0`000P0000H4V`<419`3
E`0002@000031000W0<000@4002M0`000P0000@4W0<319d3E`0002@000021000W@<000<4002O0`00
0P0000<4W@<219l3E`0002@000011000W`<00084002S0`000P000084W`<11:<3E`0002@000011000
X`<00044002V0`000P000044X`<11:H3E`0002@000011000YP<00084002[0`000P000044YP<21:/3
E`0002@000021000Z`<000<4002]0`000P000084Z`<31:d3E`0002@000031000[@<000@4002^0`00
0P0000<4[@<41:h3E`0002@000041000[P<000H4002^0`000P0000@4[P<61:h3E`0002@000071000
[P<000T4002^0`000P0000L4[P<91:h3E`0002@000091000[@<000X4002^0`000P0000T4[P<:1:d3
E`0002@0000:1000Z`<000/4002]0`000P0000X4[@<;1:/3E`0002@0000;1000YP<000`4002[0`00
0P0000/4Z`<<1:H3E`0002@0000<1000X`<000`4002V0`000P0000`4YP<<1:<3E`0002@0000;1000
W`<000`4002S0`000P0000`4X`<;19l3E`0002@0000:1000W@<000/4002O0`000P0000/4W`<:19d3
E`0002@000091000W0<000X4002M0`000P0000X4W@<919`3E`0002@000071000V`<000T4002L0`00
0P0000T4W0<719/3E`0002@0000C1000[@<001@4002^0`000P0001@4[@<C1:h3E`0002@0000C1000
[P<001@4002^0`000P0001<4[P<D1:h3E`0002@0000D1000[P<001D4002^0`000P0001@4[P<E1:h3
E`0002@0000D1000[@<001D4002^0`000P0001D4[P<D1:d3E`0002@0000N1000V`<00244002L0`00
0P000244V`<N19`3E`0002@0000L1000W0<001h4002O0`000P0001h4W0<L19l3E`0002@0000K1000
W`<001`4002S0`000P0001`4W`<K1:<3E`0002@0000K1000X`<001/4002V0`000P0001/4X`<K1:H3
E`0002@0000K1000YP<001`4002[0`000P0001/4YP<L1:/3E`0002@0000L1000Z`<001h4002^0`00
0P0001`4Z`<N1:h3E`0002@0000N1000[P<00244002^0`000P0001h4[P<Q1:h3E`0002@0000Q1000
[P<00284002^0`000P000244[P<R1:h3E`0002@0000R1000[P<002D4002^0`000P000284[P<U1:h3
E`0002@0000U1000Z`<002L4002^0`000P0002D4[P<W1:/3E`0002@0000W1000YP<002P4002[0`00
0P0002L4Z`<X1:H3E`0002@0000X1000X`<002P4002V0`000P0002P4YP<X1:<3E`0002@0000W1000
W`<002P4002S0`000P0002P4X`<W19l3E`0002@0000U1000W0<002L4002O0`000P0002L4W`<U19`3
E`0002@0000R1000V`<002D4002L0`000P0002D4W0<R19/3E`0002@0000Q1000V`<00284002K0`00
0P000284V`<Q19/3E`0002@0000O1000V`<00244002L0`000P000244V`<O19`3E`0002@0000N1000
W0<001l4002M0`000P0001l4W0<N19d3E`0002@0000M1000W@<001h4002O0`000P0001h4W@<M19l3
E`0002@0000L1000W`<001d4002S0`000P0001d4W`<L1:<3E`0002@0000L1000X`<001`4002V0`00
0P0001`4X`<L1:H3E`0002@0000L1000YP<001d4002[0`000P0001`4YP<M1:/3E`0002@0000M1000
Z`<001h4002]0`000P0001d4Z`<N1:d3E`0002@0000N1000[@<001l4002^0`000P0001h4[@<O1:h3
E`0002@0000O1000[P<00244002^0`000P0001l4[P<Q1:h3E`0002@0000R1000[P<002@4002^0`00
0P000284[P<T1:h3E`0002@0000T1000[@<002D4002^0`000P0002@4[P<U1:d3E`0002@0000U1000
Z`<002H4002]0`000P0002D4[@<V1:/3E`0002@0000V1000YP<002L4002[0`000P0002H4Z`<W1:H3
E`0002@0000W1000X`<002L4002V0`000P0002L4YP<W1:<3E`0002@0000V1000W`<002L4002S0`00
0P0002L4X`<V19l3E`0002@0000U1000W@<002H4002O0`000P0002H4W`<U19d3E`0002@0000T1000
W0<002D4002M0`000P0002D4W@<T19`3E`0002@0000R1000V`<002@4002L0`000P0002@4W0<R19/3
E`0002@0000_1000WP<00344002O0`000P0002l4W`<a19h3E`0002@0000a1000V`<003<4002N0`00
0P000344WP<c19/3E`0002@0000c1000V`<003<4002^0`000P0003<4V`<c1:h3E`0002@0000b1000
W0<00384002^0`000P000384W0<b1:h3E`0002@0000_1000[P<003L4002^0`000P0002l4[P<g1:h3
E`0002@0000o1000W`<00404002P0`000P0003l4W`=01:03E`0002@0000o1000X0<00404002P0`00
0P000404X0<o1:03E`0002@0000n1000X0<003l4002P0`000P0003l4X0<n1:03E`0002@0000n1000
W`<003h4002P0`000P0003h4X0<n19l3E`0002@0000n1000W@<003l4002O0`000P0003h4W`<o19d3
E`0002@0000o1000W0<00404002M0`000P0003l4W@=019`3E`0002@000101000V`<004<4002L0`00
0P000404W0=319/3E`0002@000131000V`<004H4002K0`000P0004<4V`=619/3E`0002@000161000
V`<004T4002L0`000P0004H4V`=919`3E`0002@000191000W0<004X4002M0`000P0004T4W0=:19d3
E`0002@0001:1000W@<004/4002O0`000P0004X4W@=;19l3E`0002@0001;1000W`<004/4002P0`00
0P0004/4W`=;1:03E`0002@0001:1000X0<004/4002R0`000P0004/4X0=:1:83E`0002@000171000
XP<004X4002T0`000P0004X4XP=71:@3E`0002@000131000Y0<004L4002V0`000P0004L4Y0=31:H3
E`0002@000111000YP<004<4002W0`000P0004<4YP=11:L3E`0002@0000o1000Y`<00444002Y0`00
0P000444Y`<o1:T3E`0002@0000n1000Z@<003l4002/0`000P0003l4Z@<n1:`3E`0002@0000n1000
[0<003h4002^0`000P0003h4[0<n1:h3E`0002@000161000V`<004P4002L0`000P0004H4V`=819`3
E`0002@000181000W0<004T4002M0`000P0004P4W0=919d3E`0002@000191000W@<004X4002O0`00
0P0004T4W@=:19l3E`0002@0001:1000W`<004X4002P0`000P0004X4W`=:1:03E`0002@000191000
X0<004X4002R0`000P0004X4X0=91:83E`0002@000161000XP<004T4002T0`000P0004T4XP=61:@3
E`0002@000131000Y0<004H4002V0`000P0004H4Y0=31:H3E`0002@0000n1000[0<003l4002]0`00
0P0003h4[@<o1:`3E`0002@0000o1000[0<00444002/0`000P0003l4[0=11:`3E`0002@000111000
[0<004D4002^0`000P000444[0=51:h3E`0002@000151000[P<004P4002^0`000P0004D4[P=81:h3
E`0002@000181000[@<004X4002^0`000P0004P4[P=:1:d3E`0002@0001:1000[0<004/4002]0`00
0P0004X4[@=;1:`3E`0002@000111000[0<004D4002^0`000P000444[0=51:h3E`0002@000151000
[P<004T4002^0`000P0004D4[P=91:h3E`0002@000191000[P<004X4002^0`000P0004T4[P=:1:h3
E`0002@0001:1000[0<004/4002^0`000P0004X4[P=;1:`3E`0002@0001;1000ZP<004/4002/0`00
0P0004/4[0=;1:X3E`0002@0002L1000V`<009l4002L0`000P0009l4V`>L19`3E`0002@0002J1000
W0<009`4002O0`000P0009`4W0>J19l3E`0002@0002I1000W`<009X4002S0`000P0009X4W`>I1:<3
E`0002@0002I1000X`<009T4002V0`000P0009T4X`>I1:H3E`0002@0002I1000YP<009X4002[0`00
0P0009T4YP>J1:/3E`0002@0002J1000Z`<009`4002^0`000P0009X4Z`>L1:h3E`0002@0002L1000
[P<009l4002^0`000P0009`4[P>O1:h3E`0002@0002O1000[P<00:04002^0`000P0009l4[P>P1:h3
E`0002@0002P1000[P<00:<4002^0`000P000:04[P>S1:h3E`0002@0002S1000Z`<00:D4002^0`00
0P000:<4[P>U1:/3E`0002@0002U1000YP<00:H4002[0`000P000:D4Z`>V1:H3E`0002@0002V1000
X`<00:H4002V0`000P000:H4YP>V1:<3E`0002@0002U1000W`<00:H4002S0`000P000:H4X`>U19l3
E`0002@0002S1000W0<00:D4002O0`000P000:D4W`>S19`3E`0002@0002P1000V`<00:<4002L0`00
0P000:<4W0>P19/3E`0002@0002O1000V`<00:04002K0`000P000:04V`>O19/3E`0002@0002M1000
V`<009l4002L0`000P0009l4V`>M19`3E`0002@0002L1000W0<009d4002M0`000P0009d4W0>L19d3
E`0002@0002K1000W@<009`4002O0`000P0009`4W@>K19l3E`0002@0002J1000W`<009/4002S0`00
0P0009/4W`>J1:<3E`0002@0002J1000X`<009X4002V0`000P0009X4X`>J1:H3E`0002@0002J1000
YP<009/4002[0`000P0009X4YP>K1:/3E`0002@0002K1000Z`<009`4002]0`000P0009/4Z`>L1:d3
E`0002@0002L1000[@<009d4002^0`000P0009`4[@>M1:h3E`0002@0002M1000[P<009l4002^0`00
0P0009d4[P>O1:h3E`0002@0002P1000[P<00:84002^0`000P000:04[P>R1:h3E`0002@0002R1000
[@<00:<4002^0`000P000:84[P>S1:d3E`0002@0002S1000Z`<00:@4002]0`000P000:<4[@>T1:/3
E`0002@0002T1000YP<00:D4002[0`000P000:@4Z`>U1:H3E`0002@0002U1000X`<00:D4002V0`00
0P000:D4YP>U1:<3E`0002@0002T1000W`<00:D4002S0`000P000:D4X`>T19l3E`0002@0002S1000
W@<00:@4002O0`000P000:@4W`>S19d3E`0002@0002R1000W0<00:<4002M0`000P000:<4W@>R19`3
E`0002@0002P1000V`<00:84002L0`000P000:84W0>P19/3E`0002@0002/1000[@<00:d4002^0`00
0P000:d4[@>/1:h3E`0002@0002/1000[P<00:d4002^0`000P000:`4[P>]1:h3E`0002@0002]1000
[P<00:h4002^0`000P000:d4[P>^1:h3E`0002@0002]1000[@<00:h4002^0`000P000:h4[P>]1:d3
E`0002@0002g1000V`<00;X4002L0`000P000;X4V`>g19`3E`0002@0002e1000W0<00;L4002O0`00
0P000;L4W0>e19l3E`0002@0002d1000W`<00;D4002S0`000P000;D4W`>d1:<3E`0002@0002d1000
X`<00;@4002V0`000P000;@4X`>d1:H3E`0002@0002d1000YP<00;D4002[0`000P000;@4YP>e1:/3
E`0002@0002e1000Z`<00;L4002^0`000P000;D4Z`>g1:h3E`0002@0002g1000[P<00;X4002^0`00
0P000;L4[P>j1:h3E`0002@0002j1000[P<00;/4002^0`000P000;X4[P>k1:h3E`0002@0002k1000
[P<00;h4002^0`000P000;/4[P>n1:h3E`0002@0002n1000Z`<00<04002^0`000P000;h4[P?01:/3
E`0002@000301000YP<00<44002[0`000P000<04Z`?11:H3E`0002@000311000X`<00<44002V0`00
0P000<44YP?11:<3E`0002@000301000W`<00<44002S0`000P000<44X`?019l3E`0002@0002n1000
W0<00<04002O0`000P000<04W`>n19`3E`0002@0002k1000V`<00;h4002L0`000P000;h4W0>k19/3
E`0002@0002j1000V`<00;/4002K0`000P000;/4V`>j19/3E`0002@0002h1000V`<00;X4002L0`00
0P000;X4V`>h19`3E`0002@0002g1000W0<00;P4002M0`000P000;P4W0>g19d3E`0002@0002f1000
W@<00;L4002O0`000P000;L4W@>f19l3E`0002@0002e1000W`<00;H4002S0`000P000;H4W`>e1:<3
E`0002@0002e1000X`<00;D4002V0`000P000;D4X`>e1:H3E`0002@0002e1000YP<00;H4002[0`00
0P000;D4YP>f1:/3E`0002@0002f1000Z`<00;L4002]0`000P000;H4Z`>g1:d3E`0002@0002g1000
[@<00;P4002^0`000P000;L4[@>h1:h3E`0002@0002h1000[P<00;X4002^0`000P000;P4[P>j1:h3
E`0002@0002k1000[P<00;d4002^0`000P000;/4[P>m1:h3E`0002@0002m1000[@<00;h4002^0`00
0P000;d4[P>n1:d3E`0002@0002n1000Z`<00;l4002]0`000P000;h4[@>o1:/3E`0002@0002o1000
YP<00<04002[0`000P000;l4Z`?01:H3E`0002@000301000X`<00<04002V0`000P000<04YP?01:<3
E`0002@0002o1000W`<00<04002S0`000P000<04X`>o19l3E`0002@0002n1000W@<00;l4002O0`00
0P000;l4W`>n19d3E`0002@0002m1000W0<00;h4002M0`000P000;h4W@>m19`3E`0002@0002k1000
V`<00;d4002L0`000P000;d4W0>k19/3E`0002@000381000WP<00<X4002O0`000P000<P4W`?:19h3
E`0002@0003:1000V`<00<`4002N0`000P000<X4WP?<19/3E`0002@0003<1000V`<00<`4002^0`00
0P000<`4V`?<1:h3E`0002@0003;1000W0<00</4002^0`000P000</4W0?;1:h3E`0002@000381000
[P<00=04002^0`000P000<P4[P?@1:h3E`0002@0003O1000W@<00=l4002^0`000P000=l4W@?O1:h3
E`0002@0003P1000V`<00>04002^0`000P000>04V`?P1:h3E`0002@0003F1000V`<00>04002Y0`00
0P000>04V`?F1:T3E`0002@0003F1000Z@<00>D4002Y0`000P000=H4Z@?U1:T3E`0002@0003M1000
[P<00><4002^0`000P000=d4[P?S1:h39@0000`000080020:00000`0000200009`0001P000020000
00000000000000009@0000`000020000:00000`000010000EP000500002G0000N`<009`0001m0`00
3@0009P0O@>G07d3U`1l0iL0O0>G07/3V01k0i/0N`>L07/3W01l0i`0O0>L07d3V`1m0iP0O@=F0000
D00009X0001k0`00WP0007d3000=0000V`1m0iX0O@>J07`3VP1l0iX0N`>K07/3W@1k0ih0N`>N07`3
WP1l0ih0O@>M07d3V`1m0eH0001@0000W00007/3002Q0000O@<000d0002M07d3W01m0i`0O0>L07`3
W01k0id0N`>P07/3X@1k0j40O0>Q07`3X@1m0j00O@>M07d3EP000500002O0000N`<00:H0001m0`00
3@000:00O@>O07d3W`1l0il0O0>O07/3X01k0jD0N`>V07/3YP1l0jH0O0>V07d3Y@1m0j00O@=F0000
D0000:@0001k0`00Z00007d3000=0000Y@1m0j@0O@>T07`3Y01l0j@0N`>U07/3Y`1k0jP0N`>X07`3
Z01l0jP0O@>W07d3Y@1m0eH0001@0000YP0007/3002[0000O@<000d0002W07d3YP1m0jH0O0>V07`3
YP1k0jL0N`>Z07/3Z`1k0j/0O0>[07`3Z`1m0jX0O@>W07d3EP000500002Y0000N`<00:d0001m0`00
3@000:X0O@>Y07d3Z@1l0jT0O0>Y07/3ZP1k0j`0N`>]07/3[@1l0jd0O0>]07d3[01m0jX0O@=F0000
D0000:/0001k0`00/00007d3000=0000[01m0j/0O@>[07`3Z`1l0j/0N`>/07/3[`1k0k00N`>`07`3
/01l0k00O@>_07d3[01m0eH0001@0000[P0007/3002b0000O@<000d0002_07d3[P1m0jh0O0>^07`3
[P1k0jl0N`>a07/3/P1k0k80O0>b07`3/P1m0k40O@>_07d3EP000500002`0000N`<00;D0001m0`00
3@000;40O@>`07d3/01l0k00O0>`07/3/@1k0k@0N`>e07/3]@1l0kD0O0>e07d3]01m0k40O@=F0000
D0000;<0001k0`00]`0007d3000=0000]01m0k<0O@>c07`3/`1l0k<0N`>d07/3]P1k0kL0N`>g07`3
]`1l0kL0O@>f07d3]01m0eH0001@0000]@0007/3002j0000O@<000d0002f07d3]@1m0kD0O0>e07`3
]@1k0kH0N`>i07/3^P1k0kX0O0>j07`3^P1m0kT0O@>f07d3EP000500002h0000N`<00;`0001m0`00
3@000;T0O@>h07d3^01l0kP0O0>h07/3^@1k0k/0N`>l07/3_01l0k`0O0>l07d3^`1m0kT0O@=F0000
D0000;X0001k0`00_`0007d3000=0000^`1m0kX0O@>j07`3^P1l0kX0N`>k07/3_P1k0kl0N`>o07`3
_`1l0kl0O@>n07d3^`1m0eH0001@0000_@0007/300310000O@<000d0002n07d3_@1m0kd0O0>m07`3
_@1k0kh0N`?007/3`@1k0l40O0?107`3`@1m0l00O@>n07d3EP000500002o0000N`<00<@0001m0`00
3@000<00O@>o07d3_`1l0kl0O0>o07/3`01k0l<0N`?407/3a01l0l@0O0?407d3``1m0l00O@=F0000
D0000<80001k0`00a`0007d3000=0000``1m0l80O@?207`3`P1l0l80N`?307/3aP1k0lL0N`?707`3
a`1l0lL0O@?607d3``1m0eH0001@0000a@0007/300390000O@<000d0003607d3a@1m0lD0O0?507`3
a@1k0lH0N`?807/3b@1k0lT0O0?907`3b@1m0lP0O@?607d3EP00050000370000N`<00<`0001m0`00
3@000<P0O@?707d3a`1l0lL0O0?707/3b01k0l/0N`?<07/3c01l0l`0O0?<07d3b`1m0lP0O@=F0000
D0000<X0001k0`00cP0007d3000=0000b`1m0lX0O@?:07`3bP1l0lX0N`?;07/3c@1k0lh0N`?>07`3
cP1l0lh0O@?=07d3b`1m0eH0001@0000c00007/3003A0000O@<000d0003=07d3c01m0l`0O0?<07`3
c01k0ld0N`?@07/3d@1k0m40O0?A07`3d@1m0m00O@?=07d3EP000500003?0000N`<00=<0001m0`00
3@000=00O@??07d3c`1l0ll0O0??07/3d01k0m80N`?C07/3d`1l0m<0O0?C07d3dP1m0m00O@=F0000
D0000=40001k0`00eP0007d3000=0000dP1m0m40O@?A07`3d@1l0m40N`?B07/3e@1k0mH0N`?F07`3
eP1l0mH0O@?E07d3dP1m0eH0001@0000e00007/3003H0000O@<000d0003E07d3e01m0m@0O0?D07`3
e01k0mD0N`?G07/3f01k0mP0O0?H07`3f01m0mL0O@?E07d3EP000500003F0000N`<00=/0001m0`00
3@000=L0O@?F07d3eP1l0mH0O0?F07/3e`1k0mX0N`?K07/3f`1l0m/0O0?K07d3fP1m0mL0O@=F0000
D0000=T0001k0`00g@0007d3000=0000fP1m0mT0O@?I07`3f@1l0mT0N`?J07/3g01k0md0N`?M07`3
g@1l0md0O@?L07d3fP1m0eH0001@0000f`0007/3003P0000O@<000d0003L07d3f`1m0m/0O0?K07`3
f`1k0m`0N`?O07/3h01k0n00O0?P07`3h01m0ml0O@?L07d3EP000500003N0000N`<00>80001m0`00
3@000=l0O@?N07d3gP1l0mh0O0?N07/3g`1k0n40N`?R07/3hP1l0n80O0?R07d3h@1m0ml0O@=F0000
D0000>00001k0`00i@0007d3000=0000h@1m0n00O@?P07`3h01l0n00N`?Q07/3i01k0nD0N`?U07`3
i@1l0nD0O@?T07d3h@1m0eH0001@0000h`0007/3003W0000O@<000d0003T07d3h`1m0n<0O0?S07`3
h`1k0n@0N`?V07/3i`1k0nL0O0?W07`3i`1m0nH0O@?T07d3EP000500003U0000N`<00>X0001m0`00
3@000>H0O@?U07d3i@1l0nD0O0?U07/3iP1k0nT0N`?Z07/3jP1l0nX0O0?Z07d3j@1m0nH0O@=F0000
D0000>P0001j0`00k00007d3000=0000j@1m0nT0O@?X07d3j01l0nP0O0?Y07/3j`1j0n/0NP?/07X3
k01k0n`0N`?[07`3j@1m0eH0001@0000jP0007X3003_0000O0<000d0003[07`3jP1l0nX0N`?Z07/3
jP1j0n/0NP?^07X3k`1j0nl0N`?_07/3k`1l0nh0O0?[07`3EP000500003]0000NP<00?40001l0`00
3@000>h0O0?]07`3k@1k0nd0N`?]07X3kP1j0o00NP?a07X3l@1k0o40N`?a07`3l01l0nh0O0=F0000
D0000>l0001j0`00m00007`3000=0000l01l0nl0O0?_07/3k`1k0nl0NP?`07X3l`1j0o@0NP?d07/3
m01k0o@0O0?c07`3l01l0eH0001@0000lP0007X3003f0000O0<000d0003c07`3lP1l0o80N`?b07/3
lP1j0o<0NP?e07X3mP1j0oH0N`?f07/3mP1l0oD0O0?c07`3EP000500003d0000NP<00?T0001l0`00
3@000?D0O0?d07`3m01k0o@0N`?d07X3m@1j0oP0NP?i07X3n@1k0oT0N`?i07`3n01l0oD0O0=F0000
D0000?L0001j0`00n`0007`3000=0000n01l0oL0O0?g07/3m`1k0oL0NP?h07X3nP1j0o/0NP?k07/3
n`1k0o/0O0?j07`3n01l0eH0001@0000n@0007X3003n0000O0<000d0003j07`3n@1l0oT0N`?i07/3
n@1j0oX0NP?m07X3oP1j0oh0N`?n07/3oP1l0od0O0?j07`3EP000500003l0000NP<00001001l0`00
3@000?d0O0?l07`3o01k0o`0N`?l07X3o@1j0ol0NP<00GX3005k0`01N`<00G`3o`1l0od0O0=F0000
D0000?h0001j0`000`4007`3000=0000o`1l0oh0O0?n07/3oP1k0oh0NP?o07X30P5j0`<1NP<30G/3
0`5k0`<1O0<20G`3o`1l0eH0001@00000@4007X300050@00O0<000d000020G`30@5l0`41N`<10G/3
0@5j0`81NP<40GX31@5j0`D1N`<50G/31@5l0`@1O0<20G`3EP00050000030@00NP<000P1001l0`00
3@0000@1O0<30G`30`5k0`<1N`<30GX3105j0`L1NP<80GX3205k0`P1N`<80G`31`5l0`@1O0=F0000
D00000H1001j0`002P4007`3000=00001`5l0`H1O0<60G/31P5k0`H1NP<70GX32@5j0`X1NP<:0G/3
2P5k0`X1O0<90G`31`5l0eH0001@0000204007X3000=0@00O0<000d000090G`3205l0`P1N`<80G/3
205j0`T1NP<<0GX33@5j0`d1N`<=0G/33@5l0``1O0<90G`3EP000500000;0@00NP<000l1001l0`00
3@0000`1O0<;0G`32`5k0`/1N`<;0GX3305j0`h1NP<?0GX33`5k0`l1N`<?0G`33P5l0``1O0=F0000
D00000d1001j0`004P4007`3000=00003P5l0`d1O0<=0G/33@5k0`d1NP<>0GX34@5j0a81NP<B0G/3
4P5k0a81O0<A0G`33P5l0eH0001@0000404007X3000D0@00O0<000d0000A0G`3405l0a01N`<@0G/3
405j0a41NP<C0GX3505j0a@1N`<D0G/3505l0a<1O0<A0G`3EP000500000B0@00NP<001L1001l0`00
3@0001<1O0<B0G`34P5k0a81N`<B0GX34`5j0aH1NP<G0GX35`5k0aL1N`<G0G`35P5l0a<1O0=F0000
D00001D1001j0`006@4007`3000=00005P5l0aD1O0<E0G/35@5k0aD1NP<F0GX3605j0aT1NP<I0G/3
6@5k0aT1O0<H0G`35P5l0eH0001@00005`4007X3000L0@00O0<000d0000H0G`35`5l0aL1N`<G0G/3
5`5j0aP1NP<K0GX3705j0a`1N`<L0G/3705l0a/1O0<H0G`3EP000500000J0@00NP<001h1001m0`00
3@0001/1O0<J0G/36P5k0aX1NP<K0GX36`5j0ad1N`<N0G`37P5l0ah1O@<M0Gd37@5m0a/1O0=F0000
D00001`1001k0`008@4007d3000=00007@5m0a`1O@<L0G`3705l0a`1N`<M0G/3805k0b41N`<Q0G`3
8@5l0b41O@<P0Gd37@5m0eH0001@00007`4007/3000S0@00O@<000d0000P0Gd37`5m0al1O0<O0G`3
7`5k0b01N`<R0G/38`5k0b<1O0<S0G`38`5m0b81O@<P0Gd3EP000500000Q0@00N`<002H1001m0`00
3@000281O@<Q0Gd38@5l0b41O0<Q0G/38P5k0bD1N`<V0G/39P5l0bH1O0<V0Gd39@5m0b81O@=F0000
D00002@1001k0`00:04007d3000=00009@5m0b@1O@<T0G`3905l0b@1N`<U0G/39`5k0bP1N`<X0G`3
:05l0bP1O@<W0Gd39@5m0eH0001@00009P4007/3000[0@00O@<000d0000W0Gd39P5m0bH1O0<V0G`3
9P5k0bL1N`<Z0G/3:`5k0b/1O0<[0G`3:`5m0bX1O@<W0Gd3EP000500000Y0@00N`<002d1001m0`00
3@0002X1O@<Y0Gd3:@5l0bT1O0<Y0G/3:P5k0b`1N`<]0G/3;@5l0bd1O0<]0Gd3;05m0bX1O@=F0000
D00002/1001j0`00<04007d3000=0000;05m0b`1O@<[0Gd3:`5l0b/1N`</0G/3;`5j0bl1NP<`0GX3
<05k0c01O0<_0G`3;05m0eH0001@0000;P4007X3000b0@00O0<000d0000_0G`3;P5l0bh1N`<^0G/3
;P5j0bl1NP<a0GX3<P5j0c81N`<b0G/3<P5l0c41O0<_0G`3EP000500000`0@00NP<003D1001l0`00
3@000341O0<`0G`3<05k0c01N`<`0GX3<@5j0c@1NP<e0GX3=@5k0cD1N`<e0G`3=05l0c41O0=F0000
D00003<1001j0`00=`4007`3000=0000=05l0c<1O0<c0G/3<`5k0c<1NP<d0GX3=P5j0cL1NP<g0G/3
=`5k0cL1O0<f0G`3=05l0eH0001@0000=@4007X3000j0@00O0<000d0000f0G`3=@5l0cD1N`<e0G/3
=@5j0cH1NP<i0GX3>P5j0cX1N`<j0G/3>P5l0cT1O0<f0G`3EP000500000h0@00NP<003`1001l0`00
3@0003T1O0<h0G`3>05k0cP1N`<h0GX3>@5j0c/1NP<l0GX3?05k0c`1N`<l0G`3>`5l0cT1O0=F0000
D00003X1001i0`00?`4007`3000=0000>`5l0c/1O0<j0G`3>P5k0cX1NP<k0GX3?P5i0ch1N@<o0GT3
?`5j0cl1N`<n0G/3>`5l0eH0001@0000?@4007T300110@00N`<000d0000n0G/3?@5k0cd1NP<m0GX3
?@5i0ch1N@=00GT3@@5i0d41NP=10GX3@@5k0d01N`<n0G/3EP000500000o0@00N@<004@1001k0`00
3@000401N`<o0G/3?`5j0cl1NP<o0GT3@05i0d<1N@=40GT3A05j0d@1NP=40G/3@`5k0d01N`=F0000
D0000481001i0`00AP4007/3000=0000@`5k0d81N`=20GX3@P5j0d81N@=30GT3A@5i0dH1N@=60GX3
AP5j0dH1N`=50G/3@`5k0eH0001@0000A04007T300190@00N`<000d000150G/3A05k0d@1NP=40GX3
A05i0dD1N@=80GT3B@5i0dT1NP=90GX3B@5k0dP1N`=50G/3EP00050000170@00N@<004`1001k0`00
3@0004P1N`=70G/3A`5j0dL1NP=70GT3B05i0d/1N@=<0GT3C05j0d`1NP=<0G/3B`5k0dP1N`=F0000
D00004X1001i0`00CP4007/3000=0000B`5k0dX1N`=:0GX3BP5j0dX1N@=;0GT3C@5i0dh1N@=>0GX3
CP5j0dh1N`==0G/3B`5k0eH0001@0000C04007T3001A0@00O0<000d0001=0G/3C05k0d`1NP=<0GT3
C@5i0dd1N@=@0GX3D@5j0e41N`=A0G`3D05l0e01O0==0G/3EP000500001?0@00NP<005<1001l0`00
3@000501O0=?0G`3C`5k0dl1N`=?0GX3D05j0e81NP=C0GX3D`5k0e<1N`=C0G`3DP5l0e01O0=F0000
D0000541001j0`00EP4007`3000=0000DP5l0e41O0=A0G/3D@5k0e41NP=B0GX3E@5j0eH1NP=F0G/3
EP5k0eH1O0=E0G`3DP5l0eH0001@0000E04007X3001H0@00O0<000d0001E0G`3E05l0e@1N`=D0G/3
E05j0eD1NP=G0GX3F05j0eP1N`=H0G/3F05l0eL1O0=E0G`3EP000500001F0@00NP<005/1001l0`00
3@0005L1O0=F0G`3EP5k0eH1N`=F0GX3E`5j0eX1NP=K0GX3F`5k0e/1N`=K0G`3FP5l0eL1O0=F0000
D00005T1001j0`00G@4007`3000=0000FP5l0eT1O0=I0G/3F@5k0eT1NP=J0GX3G05j0ed1NP=M0G/3
G@5k0ed1O0=L0G`3FP5l0eH0001@0000F`4007X3001P0@00O0<000d0001L0G`3F`5l0e/1N`=K0G/3
F`5j0e`1NP=O0GX3H05j0f01N`=P0G/3H05l0el1O0=L0G`3EP000500001N0@00NP<00681001l0`00
3@0005l1O0=N0G`3GP5k0eh1N`=N0GX3G`5j0f41NP=R0GX3HP5k0f81N`=R0G`3H@5l0el1O0=F0000
D0000601001j0`00I@4007`3000=0000H@5l0f01O0=P0G/3H05k0f01NP=Q0GX3I05j0fD1NP=U0G/3
I@5k0fD1O0=T0G`3H@5l0eH0001@0000H`4007X3001W0@00O0<000d0001T0G`3H`5l0f<1N`=S0G/3
H`5j0f@1NP=V0GX3I`5j0fL1N`=W0G/3I`5l0fH1O0=T0G`3EP000500001U0@00NP<006X1001l0`00
3@0006H1O0=U0G`3I@5k0fD1N`=U0GX3IP5j0fT1NP=Z0GX3JP5k0fX1N`=Z0G`3J@5l0fH1O0=F0000
D00006P1001j0`00K04007`3000=0000J@5l0fP1O0=X0G/3J05k0fP1NP=Y0GX3J`5j0f`1NP=/0G/3
K05k0f`1O0=[0G`3J@5l0eH0001@0000JP4007T3001_0@00O0<000d0001[0G`3J`5l0fX1O0=Z0G/3
JP5j0f/1NP=^0GT3KP5i0fl1N@=_0GX3K`5k0fh1N`=[0G`3EP000500001]0@00N@<00741001k0`00
3@0006h1N`=]0G/3K@5j0fd1NP=]0GT3KP5i0g01N@=a0GT3L@5j0g41NP=a0G/3L05k0fh1N`=F0000
D00006l1001i0`00M04007/3000=0000L05k0fl1N`=_0GX3K`5j0fl1N@=`0GT3L`5i0g@1N@=d0GX3
M05j0g@1N`=c0G/3L05k0eH0001@0000LP4007T3001f0@00N`<000d0001c0G/3LP5k0g81NP=b0GX3
LP5i0g<1N@=e0GT3MP5i0gH1NP=f0GX3MP5k0gD1N`=c0G/3EP000500001d0@00N@<007T1001k0`00
3@0007D1N`=d0G/3M05j0g@1NP=d0GT3M@5i0gP1N@=i0GT3N@5j0gT1NP=i0G/3N05k0gD1N`=F0000
D00007L1001i0`00N`4007/3000=0000N05k0gL1N`=g0GX3M`5j0gL1N@=h0GT3NP5i0g/1N@=k0GX3
N`5j0g/1N`=j0G/3N05k0eH0001@0000N@4007T3001n0@00N`<000d0001j0G/3N@5k0gT1NP=i0GX3
N@5i0gX1N@=m0GT3OP5i0gh1NP=n0GX3OP5k0gd1N`=j0G/3EP000500001l0@00N@<00801001k0`00
3@0007d1N`=l0G/3O05j0g`1NP=l0GT3O@5i0gl1N@>00GT3P05j0h01NP>00G/3O`5k0gd1N`=F0000
D00007h1001i0`00P`4007/3000=0000O`5k0gh1N`=n0GX3OP5j0gh1N@=o0GT3PP5i0h<1N@>30GX3
P`5j0h<1N`>20G/3O`5k0eH0001@0000P@4007P300250@00N`<000d000220G/3PP5k0h41N`>10GX3
P@5j0h81N@>40GP3Q05h0hD1N0>50GT3Q@5i0h@1NP>20G/3EP00050000230@00N0<008P1001j0`00
3@0008@1NP>30GX3P`5i0h<1N@>30GP3Q05h0hL1N0>80GP3R05i0hP1N@>80GX3Q`5j0h@1NP=F0000
D00008H1001h0`00RP4007X3000=0000Q`5j0hH1NP>60GT3QP5i0hH1N0>70GP3R@5h0hX1N0>:0GT3
RP5i0hX1NP>90GX3Q`5j0eH0001@0000R04007P3002=0@00NP<000d000290GX3R05j0hP1N@>80GT3
R05h0hT1N0><0GP3S@5h0hd1N@>=0GT3S@5j0h`1NP>90GX3EP000500002;0@00N0<008l1001j0`00
3@0008`1NP>;0GX3R`5i0h/1N@>;0GP3S05h0hh1N0>?0GP3S`5i0hl1N@>?0GX3SP5j0h`1NP=F0000
D00008d1001h0`00TP4007X3000=0000SP5j0hd1NP>=0GT3S@5i0hd1N0>>0GP3T@5h0i81N0>B0GT3
TP5i0i81NP>A0GX3SP5j0eH0001@0000T04007P3002D0@00NP<000d0002A0GX3T05j0i01N@>@0GT3
T05h0i41N0>C0GP3U05h0i@1N@>D0GT3U05j0i<1NP>A0GX3EP000500002B0@00N0<009L1001j0`00
3@0009<1NP>B0GX3TP5i0i81N@>B0GP3T`5h0iH1N0>G0GP3U`5i0iL1N@>G0GX3UP5j0i<1NP=F0000
D00009D1001h0`00V@4007X3000=0000UP5j0iD1NP>E0GT3U@5i0iD1N0>F0GP3V05h0iT1N0>I0GT3
V@5i0iT1NP>H0GX3UP5j0eH0001@0000U`4007P3002L0@00NP<000d0002H0GX3U`5j0iL1N@>G0GT3
U`5h0iP1N0>K0GP3W05h0i`1N@>L0GT3W05j0i/1NP>H0GX3EP000500002J0@00N0<009h1001j0`00
3@0009/1NP>J0GX3VP5i0iX1N@>J0GP3V`5h0id1N0>N0GP3WP5i0ih1N@>N0GX3W@5j0i/1NP=F0000
D00009`1001g0`00X@4007X3000=0000W@5j0id1NP>L0GX3W05i0i`1N0>M0GP3X05g0j01M`>Q0GL3
X@5h0j41N@>P0GT3W@5j0eH0001@0000W`4007L3002S0@00N@<000d0002P0GT3W`5i0il1N0>O0GP3
W`5g0j01M`>R0GL3X`5g0j<1N0>S0GP3X`5i0j81N@>P0GT3EP000500002Q0@00M`<00:H1001i0`00
3@000:81N@>Q0GT3X@5h0j41N0>Q0GL3XP5g0jD1M`>V0GL3YP5h0jH1N0>V0GT3Y@5i0j81N@=F0000
D0000:@1001g0`00Z04007T3000=0000Y@5i0j@1N@>T0GP3Y05h0j@1M`>U0GL3Y`5g0jP1M`>X0GP3
Z05h0jP1N@>W0GT3Y@5i0eH0001@0000YP4007L3002[0@00N@<000d0002W0GT3YP5i0jH1N0>V0GP3
YP5g0jL1M`>Z0GL3Z`5g0j/1N0>[0GP3Z`5i0jX1N@>W0GT3EP000500002Y0@00MP<00:d1001i0`00
3@000:X1N@>Z0GT3Z@5i0jT1N0>Y0GP3ZP5g0j`1MP>/0GH3[@5f0jd1M`>]0GL3[05h0jX1N@=F0000
D0000:/1001f0`00/04007P3000=0000[05h0j/1N0>[0GL3Z`5g0j/1MP>/0GH3[`5f0k01MP>`0GL3
/05g0k01N0>_0GP3[05h0eH0001@0000[P4007H3002b0@00N0<000d0002_0GP3[P5h0jh1M`>^0GL3
[P5f0jl1MP>a0GH3/P5f0k81M`>b0GL3/P5h0k41N0>_0GP3EP000500002`0@00MP<00;D1001h0`00
3@000;41N0>`0GP3/05g0k01M`>`0GH3/@5f0k@1MP>e0GH3]@5g0kD1M`>e0GP3]05h0k41N0=F0000
D0000;<1001f0`00]`4007P3000=0000]05h0k<1N0>c0GL3/`5g0k<1MP>d0GH3]P5f0kL1MP>g0GL3
]`5g0kL1N0>f0GP3]05h0eH0001@0000]@4007D3002j0@00N0<000d0002f0GP3]P5h0kD1N0>e0GL3
]@5f0kH1MP>i0GD3^@5e0kX1M@>j0GH3^P5g0kT1M`>f0GP3EP000500002h0@00M@<00;`1001g0`00
3@000;T1M`>h0GL3^05f0kP1MP>h0GD3^@5e0k/1M@>l0GD3_05f0k`1MP>l0GL3^`5g0kT1M`=F0000
D0000;X1001e0`00_`4007L3000=0000^`5g0kX1M`>j0GH3^P5f0kX1M@>k0GD3_P5e0kl1M@>o0GH3
_`5f0kl1M`>n0GL3^`5g0eH0001@0000_@4007D300310@00M`<000d0002n0GL3_@5g0kd1MP>m0GH3
_@5e0kh1M@?00GD3`@5e0l41MP?10GH3`@5g0l01M`>n0GL3EP000500002o0@00M0<00<@1001g0`00
3@000<01M`?00GL3_`5g0kl1MP>o0GD3`05e0l<1M0?30G@3a05d0l@1M@?40GH3``5f0l01M`=F0000
D0000<81001d0`00aP4007H3000=0000``5f0l81MP?20GD3`P5e0l81M0?30G@3a@5d0lH1M0?60GD3
aP5e0lH1MP?50GH3``5f0eH0001@0000a04007@300390@00MP<000d000350GH3a05f0l@1M@?40GD3
a05d0lD1M0?80G@3b@5d0lT1M@?90GD3b@5f0lP1MP?50GH3EP00050000370@00M0<00</1001f0`00
3@000<P1MP?70GH3a`5e0lL1M@?70G@3b05d0lX1M0?;0G@3b`5e0l/1M@?;0GH3bP5f0lP1MP=F0000
D0000<T1001c0`00cP4007H3000=0000bP5f0lX1MP?90GH3b@5e0lT1M0?:0G@3c@5c0ld1L`?>0G<3
cP5d0lh1M@?=0GD3bP5f0eH0001@0000c04007<3003@0@00M@<000d0003=0GD3c05e0l`1M0?<0G@3
c05c0ld1L`??0G<3d05c0m01M0?@0G@3d05e0ll1M@?=0GD3EP000500003>0@00L`<00=<1001e0`00
3@000<l1M@?>0GD3cP5d0lh1M0?>0G<3c`5c0m81L`?C0G<3d`5d0m<1M0?C0GD3dP5e0ll1M@=F0000
D0000=41001c0`00eP4007D3000=0000dP5e0m41M@?A0G@3d@5d0m41L`?B0G<3e@5c0mH1L`?F0G@3
eP5d0mH1M@?E0GD3dP5e0eH0001@0000e04007<3003H0@00M@<000d0003E0GD3e05e0m@1M0?D0G@3
e05c0mD1L`?G0G<3f05c0mP1M0?H0G@3f05e0mL1M@?E0GD3EP000500003F0@00L`<00=/1001e0`00
3@000=L1M@?F0GD3eP5d0mH1M0?F0G<3e`5c0mX1L`?K0G<3f`5d0m/1M0?K0GD3fP5e0mL1M@=F0000
D0000=T1001c0`00g@4007D3000=0000fP5e0mT1M@?I0G@3f@5d0mT1L`?J0G<3g05c0md1L`?M0G@3
g@5d0md1M@?L0GD3fP5e0eH0001@0000f`4007<3003P0@00M@<000d0003L0GD3f`5e0m/1M0?K0G@3
f`5c0m`1L`?O0G<3h05c0n01M0?P0G@3h05e0ml1M@?L0GD3EP000500003N0@00L`<00>81001e0`00
3@000=l1M@?N0GD3gP5d0mh1M0?N0G<3g`5c0n41L`?R0G<3hP5d0n81M0?R0GD3h@5e0ml1M@=F0000
D0000>01001c0`00i@4007D3000=0000h@5e0n01M@?P0G@3h05d0n01L`?Q0G<3i05c0nD1L`?U0G@3
i@5d0nD1M@?T0GD3h@5e0eH0001@0000h`4007<3003W0@00M@<000d0003T0GD3h`5e0n<1M0?S0G@3
h`5c0n@1L`?V0G<3i`5c0nL1M0?W0G@3i`5e0nH1M@?T0GD3EP000500003U0@00L`<00>X1001e0`00
3@000>H1M@?U0GD3i@5d0nD1M0?U0G<3iP5c0nT1L`?Z0G<3jP5d0nX1M0?Z0GD3j@5e0nH1M@=F0000
D0000>P1001b0`00k04007D3000=0000j@5e0nT1M@?X0GD3j05d0nP1M0?Y0G<3j`5b0n/1LP?/0G83
k05c0n`1L`?[0G@3j@5e0eH0001@0000jP400783003_0@00M0<000d0003[0G@3jP5d0nX1L`?Z0G<3
jP5b0n/1LP?^0G83k`5b0nl1L`?_0G<3k`5d0nh1M0?[0G@3EP000500003]0@00L@<00?41001d0`00
3@000>h1M0?^0G@3k@5d0nd1L`?]0G<3kP5b0o01L@?`0G43l@5a0o41LP?a0G83l05c0nh1M0=F0000
D0000>l1001`0`00m04007<3000=0000l05c0o01L`?_0G<3k`5b0nl1L@?`0G43l`5`0o<1L0?d0G03
m05a0o@1LP?c0G83l05c0eH0001@0000lP4006l3003f0@00LP<000d0003c0G83l`5b0o81LP?b0G43
lP5a0o<1L0?e0Fl3m@5_0oH1K`?f0G03mP5`0oD1L@?c0G83EP000500003d0@00KP<00?T1001a0`00
3@000?D1L@?e0G43m05a0o@1L0?d0Fl3m@5_0oP1KP?h0Fh3n@5^0oT1K`?i0G03n05`0oD1L@=F0000
D0000?L1001^0`00n`400703000=0000n05`0oL1L0?g0Fl3m`5_0oL1KP?h0Fh3nP5^0o/1KP?k0Fl3
n`5_0o/1L0?j0G03n05`0eH0001@0000n@4006d3003n0@00L0<000d0003j0G03nP5`0oT1L0?i0Fl3
n@5^0oX1KP?m0Fd3o@5]0oh1K@?n0Fh3oP5_0od1K`?j0G03EP000500003l0@00K0<00002001_0`00
3@000?d1K`?m0Fl3o05_0o`1KP?l0Fh3o@5]0ol1K0?o0F`3009/0`02K@<00Vd3o`5^0od1K`=F0000
D0000?h1001[0`000`8006h3000=0000o`5^0ol1KP?n0Fh3oP5]0oh1K0?o0F`30P9[0`82J`<30V/3
0`9/0`<2K@<20Vd3o`5^0eH0001@00000@8006X300050P00K@<000d000020Vd30P9]0`42K@<10V`3
0@9/0`82J`<40VX3109Z0`D2JP<50V/31@9[0`@2K0<20Vd3EP00050000030P00J@<000P2001/0`00
3@0000@2K0<40V`30`9/0`<2J`<30VX3109Z0`L2J@<70VT3209Y0`P2JP<80V/31`9[0`@2K0=F0000
D00000H2001Y0`002P8006/3000=00001`9[0`H2J`<60VX31P9Z0`H2J@<70VT32@9Y0`X2J@<:0VX3
2P9Z0`X2J`<90V/31`9[0eH0001@0000208006P3000=0P00J`<000d000090V/32@9[0`P2J`<80VX3
209Y0`T2J@<<0VP3309X0`d2J0<=0VT33@9Z0``2JP<90V/3EP000500000;0P00I`<000l2001Z0`00
3@0000`2JP<<0VX32`9Z0`/2J@<;0VT3309X0`h2I`<>0VL33`9W0`l2J0<?0VP33P9Y0``2JP=F0000
D00000d2001W0`004P8006T3000=00003P9Y0`d2J@<=0VP33@9X0`d2I`<>0VL34@9W0a82I`<B0VP3
4P9X0a82J@<A0VT33P9Y0eH0001@0000408006H3000D0P00J@<000d0000A0VT34@9Y0a02J@<@0VP3
409X0a42I`<C0VH34`9V0a@2IP<D0VL3509W0a<2J0<A0VT3EP000500000B0P00I@<001L2001X0`00
3@0001<2J0<C0VP34P9X0a82I`<B0VH34`9V0aH2I@<F0VD35`9U0aL2IP<G0VL35P9W0a<2J0=F0000
D00001D2001U0`006@8006L3000=00005P9W0aD2I`<E0VH35@9V0aD2I@<F0VD3609U0aT2I@<I0VH3
6@9V0aT2I`<H0VL35P9W0eH0001@00005`8006@3000L0P00I`<000d0000H0VL3609W0aL2I`<G0VH3
5`9U0aP2I@<K0V@36`9T0a`2I0<L0VD3709V0a/2IP<H0VL3EP000500000J0P00I0<001h2001V0`00
3@0001/2IP<J0VH36P9U0aX2I@<J0V@36`9T0ad2I0<N0V@37P9U0ah2I@<N0VH37@9V0a/2IP=F0000
D00001`2001T0`008@8006H3000=00007@9V0a`2IP<L0VD3709U0a`2I0<M0V@3809T0b42I0<Q0VD3
8@9U0b42IP<P0VH37@9V0eH0001@00007`8006@3000S0P00IP<000d0000P0VH37`9V0al2I@<O0VD3
7`9T0b02I0<R0V@38`9T0b<2I@<S0VD38`9V0b82IP<P0VH3EP000500000Q0P00I0<002H2001V0`00
3@000282IP<Q0VH38@9U0b42I@<Q0V@38P9T0bD2I0<V0V@39P9U0bH2I@<V0VH39@9V0b82IP=F0000
D00002@2001T0`00:08006H3000=00009@9V0b@2IP<T0VD3909U0b@2I0<U0V@39`9T0bP2I0<X0VD3
:09U0bP2IP<W0VH39@9V0eH0001@00009P8006@3000[0P00IP<000d0000W0VH39P9V0bH2I@<V0VD3
9P9T0bL2I0<Z0V@3:`9T0b/2I@<[0VD3:`9V0bX2IP<W0VH3EP000500000Y0P00I0<002d2001V0`00
3@0002X2IP<Y0VH3:@9U0bT2I@<Y0V@3:P9T0b`2I0<]0V@3;@9U0bd2I@<]0VH3;09V0bX2IP=F0000
D00002/2001S0`00<08006H3000=0000;09V0b`2IP<[0VH3:`9U0b/2I0</0V@3;`9S0bl2H`<`0V<3
<09T0c02I@<_0VD3;09V0eH0001@0000;P8006<3000b0P00I@<000d0000_0VD3;P9U0bh2I0<^0V@3
;P9S0bl2H`<a0V<3<P9S0c82I0<b0V@3<P9U0c42I@<_0VD3EP000500000`0P00H`<003D2001U0`00
3@000342I@<`0VD3<09T0c02I0<`0V<3<@9S0c@2H`<e0V<3=@9T0cD2I0<e0VD3=09U0c42I@=F0000
D00003<2001R0`00=`8006D3000=0000=09U0c@2I@<c0VD3<`9T0c<2I0<d0V<3=P9R0cH2HP<g0V83
=`9S0cL2H`<f0V@3=09U0eH0001@0000=@800643000j0P00I0<000d0000f0V@3=P9T0cD2I0<e0V<3
=@9R0cH2HP<i0V43>@9Q0cX2H@<j0V83>P9S0cT2H`<f0V@3EP000500000h0P00H0<003`2001S0`00
3@0003T2H`<i0V<3>09S0cP2HP<h0V83>@9Q0c/2H0<k0V03?09P0c`2H@<l0V43>`9R0cT2H`=F0000
D00003X2001O0`00?`800683000=0000>`9R0c/2HP<j0V83>P9Q0cX2H0<k0V03?P9O0ch2G`<o0Ul3
?`9P0cl2H@<n0V43>`9R0eH0001@0000?@8005d300110P00H@<000d0000o0V43?P9Q0ch2H@<m0V03
?@9P0cd2G`<o0Ud3@09M0d02G@=10Uh3@@9N0d42G`<o0V43EP000500000o0P00G0<004@2001O0`00
3@000402G`=00Ul3?`9O0cl2GP<o0Ud3@09M0d<2G0=30U`3A09L0d@2G@=40Uh3@`9N0d02G`=F0000
D0000482001J0`00AP8005h3000=0000A09N0d<2GP=30Uh3@P9M0d82G@=20U`3A09J0dD2FP=50UX3
AP9K0dH2F`=60U`3A09N0eH0001@0000A08005P300190P00G0<000d000160U`3A@9L0d@2G0=40U/3
A09K0d@2FP=70UP3B09H0dT2F0=90UT3B@9I0dT2FP=60U`3EP00050000170P00EP<004/2001J0`00
3@0004T2FP=80UX3B09J0dL2F@=70UT3A`9H0dT2EP=:0UH3BP9F0d/2E`=;0UL3B`9H0dT2FP=F0000
D00004T2001D0`00CP8005P3000=0000B`9H0dX2F0=90UP3B@9G0dT2E`=90UH3C09D0dd2E0=>0U@3
CP9E0dh2E@=>0UH3B`9H0eH0001@0000C0800543001@0P00EP<000d0001>0UH3C@9F0dd2EP=<0UH3
C09E0d`2E0=>0U43C`9A0dl2D@=@0U43D09B0e02D`=>0UH3EP000500001>0P00C`<005<2001C0`00
3@000502D`=?0U<3CP9C0dh2DP=>0U83CP9A0e42C`=B0Tl3D`9?0e<2D0=C0U03D`9A0e02D`=F0000
D0000542001=0`00E@800543000=0000D`9A0e82D@=B0U43D@9@0e42D0=A0Tl3D`9=0e@2C@=D0Td3
E@9>0eD2CP=E0Tl3D`9A0eH0001@0000D`8004/3001H0P00C`<000d0001E0Tl3E09?0e<2C`=C0Th3
D`9>0e<2C@=F0T/3E`9;0eP2B`=H0T`3F09<0eP2C@=E0Tl3EP000500001F0P00BP<005X2001=0`00
3@0005L2C@=G0Td3EP9=0eH2C0=F0T`3E`9;0eT2BP=I0TX3FP9:0eX2B`=J0T/3F@9<0eL2C@=F0000
D00005P200180`00G@8004`3000=0000FP9<0eT2C0=H0T`3F09;0eP2B`=H0TX3F`980e`2B0=M0TP3
G@990ed2B@=M0TX3FP9<0eH0001@0000F`8004H3001P0P00BP<000d0001M0TX3G09:0e/2BP=K0TT3
F`990e/2B0=N0TH3G`960f02AP=P0TL3H0970f02B0=M0TX3EP000500001N0P00A@<0068200180`00
3@0005l2B0=O0TP3GP980eh2A`=N0TL3G`960f42A@=Q0TD3HP950f82AP=R0TH3H@970el2B0=F0000
D000060200130`00I@8004L3000=0000HP970f42A`=P0TL3H0960f02AP=P0TD3H`930f@2@`=U0T<3
I@940fD2A0=U0TD3HP970eH0001@0000H`800483001W0P00A@<000d0001T0TD3I0950f<2A@=S0T@3
H`940f@2@`=V0T83IP920fL2@P=W0T<3I`930fH2A0=T0TD3EP000500001U0P00@0<006X200140`00
3@0006L2A0=V0T@3I@940fD2@`=U0T<3I@920fP2@0=Y0T03JP900fX2@@=Z0T43JP920fL2A0=F0000
D00006P2000o0`00K0800483000=0000J@920fT2@P=X0T83J0910fP2@@=Y0T03J`8o0f/2?`=/0Sl3
K0900f`2@0=[0T43J@920eH0001@0000JP8003d3001_0P00@@<000d0001/0T43J`910fX2@@=Z0T03
JP900fX2?`=]0Sd3KP8m0fl2?@=_0Sh3K`8n0fl2?`=/0T43EP000500001]0P00>`<00742000o0`00
3@0006l2?`=^0Sl3KP8o0fd2?P=]0Sh3K@8m0fl2>`=`0S/3L08k0g42?0=a0S`3L@8m0fl2?`=F0000
D00006l2000j0`00M08003d3000=0000L08m0g02?@=_0Sd3K`8l0fl2>`=`0S/3L`8j0g<2>P=d0SX3
M08k0g@2?0=c0S`3L08m0eH0001@0000LP8003P3001f0P00?0<000d0001d0S`3L`8l0g<2?0=b0S/3
LP8k0g82>P=d0SP3M@8h0gD2>0=f0ST3MP8i0gH2>P=d0S`3EP000500001d0P00=P<007T2000j0`00
3@0007H2>P=e0SX3M08j0g@2>@=d0ST3M08h0gL2=P=h0SH3N@8f0gT2=`=i0SL3N@8h0gH2>P=F0000
D00007L2000c0`00N`8003P3000=0000N@8h0gP2>0=h0SP3M`8h0gL2=`=g0SH3N@8c0gX2<`=j0S<3
N`8c0g/2=0=k0SD3N@8h0eH0001@0000N@800343001n0P00=@<000d0001k0SD3NP8e0gT2=@=i0S@3
N@8d0gT2<`=l0S43O@8a0gh2<@=n0S83OP8b0gh2<`=k0SD3EP000500001l0P00;`<00802000c0`00
3@0007h2<`=m0S<3O@8c0g`2<P=l0S83O08a0gh2;`=o0Rl3O`8_0h02<0>00S03P08a0gh2<`=F0000
D00007h2000]0`00P`800343000=0000P08a0gl2<@=n0S43OP8`0gh2<0=n0Rl3P@8]0h82;@>30Rd3
P`8^0h<2;P>30Rl3P08a0eH0001@0000P@8002/300250P00;`<000d000230Rl3PP8_0h82;`>10Rh3
P@8^0h42;@>30R/3Q08[0h@2:`>50R`3Q@8/0hD2;@>30Rl3EP00050000230P00:@<008P2000]0`00
3@0008D2;@>40Rd3P`8]0h<2;0>30R`3P`8[0hH2:@>70RT3R08Y0hP2:P>80RX3R08[0hD2;@=F0000
D00008H2000V0`00RP8002/3000=0000R08[0hL2:`>70R/3QP8[0hH2:P>60RT3R08V0hT29P>90RH3
RP8V0hX29`>:0RP3R08[0eH0001@0000R08002<3002=0P00:0<000d0002:0RP3R@8X0hT2:0>80RL3
R08W0hP29P>;0R<3S08S0h`28`>=0R@3S@8T0hd29@>:0RP3EP000500002;0P0080<008l2000U0`00
3@0008d29@><0RD3S08U0h/29@>;0R@3R`8S0hd280>>0R03SP8P0hl280>?0R43S`8R0hd29@=F0000
D00008d2000M0`00TP800283000=0000S`8R0hh28P>>0R83S@8Q0hd28@>=0R03T08M0i427@>A0Qd3
TP8N0i827P>B0Ql3S`8R0eH0001@0000T08001X3002D0P007`<000d0002B0Ql3T@8O0i427`>@0Ql3
T08N0i027@>B0QX3T`8J0i<26P>D0QX3U08K0i@270>B0Ql3EP000500002B0P005`<009L2000L0`00
3@0009@270>C0Q`3T`8L0i826`>B0Q/3TP8J0iD25`>F0QL3UP8G0iL260>G0QP3U`8I0i@270=F0000
D00009D2000C0`00V@8001T3000=0000U`8H0iH26@>F0QT3U@8I0iD260>E0QP3U`8D0iP24`>H0Q<3
V@8C0iT250>I0Q@3U`8H0eH0001@0000U`8000l3002L0P005@<000d0002I0QD3V08E0iP25@>G0QD3
U`8D0iL24`>J0Pl3V`8?0i/23`>L0Pl3W08@0i`24@>I0QD3EP000500002J0P002`<009h2000A0`00
3@0009`240>K0Q43V`8A0iX24@>J0Q03VP8@0i`230>M0P/3W@8;0ih22`>N0P`3WP8<0i`240=F0000
D00009`200070`00X@8000d3000=0000WP8=0id23@>M0Pd3W08=0i`230>L0P/3W`870j021`>P0PL3
X@870j4220>Q0PT3WP8=0eH0001@0000W`8000<3002S0P002@<000d0002Q0PP3X0890j022@>O0PT3
W`880il220>Q0P@3XP830j820`>S0P<3X`840j<210>Q0PP3EP000500002Q0P00o`800:H200050`00
3@000:<21@>R0PD3XP850j421@>Q0P@3X@830j@2o`:U0_l2Y@;o0ZH2o`:V0P03YP810j<21@=F0000
D0000:@2003k0P00Z0800043000=0000YP800jD20@>U0P43Y0810j@200>T0P03YP;l0ZL2n`:W0_/2
Z0;k0ZP2o0:X0_`2YP800eH0001@0000YP800?H2002[0P00o@8000d0002X0_d2Y`;m0ZL2o@:V0_d2
YP;l0ZH2n`:Y0_H2ZP;f0ZX2mP:[0_H2Z`;g0Z/2n0:X0_d2EP000500002Y0P00lP800:d2003h0P00
3@000:/2m`:Z0_P2ZP;h0ZT2n0:Y0_L2Z@;g0Z/2l`:/0_82[0;b0Zd2lP:]0_<2[@;c0Z/2m`9F0000
D0000:/2003]0P00/0800?@2000=0000[@;d0Z`2m0:/0_@2Z`;d0Z/2l`:[0_82[P;]0Zl2k@:_0^d2
/0;]0[02kP:`0^l2[@;d0UH0001@0000[P800>T2002b0P00k`8000d0002`0^h2[`;_0Zl2k`:^0^l2
[P;^0Zh2kP:`0^X2/@;Y0[42j@:b0^T2/P;Z0[82jP:`0^h2EP000500002`0P00i0800;D2003[0P00
3@000;82j`:a0^/2/@;[0[02j`:`0^X2/0;Y0[<2i0:d0^@2]0;T0[D2i0:e0^D2]@;V0[82j`9F0000
D0000;<2003O0P00]`800>H2000=0000]@;U0[D2iP:d0^H2/`;V0[<2i@:c0^D2]@;P0[D2g`:f0]l2
]`;O0[L2h0:g0^02]@;U0UH0001@0000]@800=/2002j0P00h@8000d0002g0^42]P;Q0[H2h@:e0^42
]@;P0[D2g`:h0]/2^@;K0[T2f`:j0]/2^P;L0[X2g@:g0^42EP000500002h0P00eP800;`2003M0P00
3@000;X2g0:j0]d2^@;M0[P2g@:h0]`2^0;L0[X2e`:j0]H2^`;F0[`2eP:l0]L2_0;G0[X2g09F0000
D0000;X2003A0P00_`800=P2000=0000_0;H0[/2f0:k0]P2^P;H0[X2e`:j0]H2_@;A0[h2d@:n0]42
_`;A0[l2dP:o0]<2_0;H0UH0001@0000_@800</200310P00d`8000d0002o0]82_`;C0[h2d`:m0]<2
_@;B0[d2dP:o0/`2_`;;0/02b`;10//2`@;<0/42c0:o0]82EP000500002o0P00aP800<@2003=0P00
3@000<42c@;00/d2`0;=0[l2c@:o0/`2_`;;0/82aP;30/H2``;60/@2aP;40/L2a0;80/42c@9F0000
D0000<8200300P00aP800<P2000=0000a0;70/@2b0;30/P2`P;80/82a`;20/L2a0;10/@2`0;50/02
aP;00/H2`@;60/42a0;70UH0001@0000a0800;T200390P00`P8000d000360/42aP;20/D2`P;40/82
a0;10/@2`@;70[X2a`:i0/P2^@;90[T2b@:j0/T2^P;60/42EP00050000370P00/`800</2002k0P00
3@000<T2^P;90[/2b0:k0/L2^`;70[X2a`:j0/T2]0;90[<2bP:c0//2/`;;0[@2b`:d0/T2^P9F0000
D0000<T2002/0P00cP800;D2000=0000b`:d0//2]@;:0[D2b@:e0/T2]0;90[@2c0:]0/`2[0;=0Z`2
cP:/0/h2[@;>0Zd2b`:d0UH0001@0000c0800:@2003@0P00[P8000d0003>0Zd2cP:^0/d2[P;<0Zh2
c0:]0/`2[@;>0ZD2cP:T0/l2Y0;@0Z@2d0:U0]02Y@;>0Zd2EP000500003>0P00W@800=<2002V0P00
3@000=02Y@;@0ZH2c`:V0/h2YP;>0ZD2cP:U0]42WP;A0Yd2dP:M0]<2W@;C0Yh2d`:N0]02Y@9F0000
D0000=42002E0P00e@8009l2000=0000d`:N0]<2W`;B0Yl2d@:O0]42WP;A0Yh2d`:F0]<2U@;D0YD2
e@:E0]D2UP;E0YH2d`:N0UH0001@0000d`8008`2003H0P00U`8000d0003E0YH2e@:G0]@2U`;C0YL2
d`:F0]<2UP;F0Xd2eP:<0]L2S0;H0X`2f0:=0]P2S@;E0YH2EP000500003F0P00Q0800=X2002>0P00
3@000=P2S@;H0Xh2e`:>0]H2SP;F0Xd2eP:=0]P2Q@;H0X@2f@:40]X2Q0;J0XD2fP:50]P2S@9F0000
D0000=P2001l0P00g@8008H2000=0000fP:50]X2QP;I0XH2f0:60]P2Q@;H0XD2f`9m0]/2O0;L0W`2
g@9l0]d2O@;M0Wd2fP:50UH0001@0000f`8007@2003O0P00OP8000d0003M0Wd2g@9n0]`2OP;K0Wh2
f`9m0]/2O@;M0WD2g@9d0]h2M0;O0W@2g`9e0]l2M@;M0Wd2EP000500003M0P00K0800>82001f0P00
3@000=l2M@;O0WH2gP9f0]d2MP;M0WD2g@9e0^02K@;P0V`2h@9/0^82K0;R0Vd2hP9]0]l2M@9F0000
D0000>02001T0P00i08006h2000=0000hP9]0^82KP;Q0Vh2h09^0^02K@;P0Vd2hP9U0^82I0;S0V@2
i09T0^@2I@;T0VD2hP9]0UH0001@0000hP8005`2003W0P00IP8000d0003T0VD2i09V0^<2IP;R0VH2
hP9U0^82I@;U0Ud2i@9L0^H2G0;W0U`2i`9M0^L2G@;T0VD2EP000500003U0P00E0800>X2001N0P00
3@000>L2G@;W0Uh2iP9N0^D2GP;U0Ud2i@9M0^P2E@;X0U@2j@9D0^X2E0;Z0UD2jP9E0^L2G@9F0000
D0000>P2001;0P00k08005H2000=0000jP9E0^X2EP;Y0UH2j09F0^P2E@;X0UD2jP9<0^X2B`;[0T/2
k09;0^`2C0;/0T`2jP9E0UH0001@0000jP8004<2003_0P00C@8000d0003/0T`2k09=0^/2C@;Z0Td2
jP9<0^X2C0;]0T@2k@930^h2@`;_0T<2k`940^l2A0;/0T`2EP000500003]0P00>`800?4200150P00
3@000>l2A0;_0TD2kP950^d2A@;]0T@2k@940^l2?0;_0S/2l08k0_42>`;a0S`2l@8l0^l2A09F0000
D0000>l2000b0P00m08003d2000=0000l@8l0_42?@;`0Sd2k`8m0^l2?0;_0S`2lP8c0_82<P;c0S82
m08b0_@2<`;d0S<2l@8l0UH0001@0000lP8002X2003f0P00=08000d0003d0S<2m08d0_<2=0;b0S@2
lP8c0_82<`;d0R/2m08Z0_D2:P;f0RX2mP8[0_H2:`;d0S<2EP000500003d0P008@800?T2000/0P00
3@000?H2:`;f0R`2m@8/0_@2;0;d0R/2m08[0_L28P;g0R42n08Q0_T28@;i0R82n@8R0_H2:`9F0000
D0000?L2000G0P00n`8002<2000=0000n@8R0_T28`;h0R<2n08S0_L28P;g0R82n@8H0_T25`;j0QL2
nP8G0_/260;k0QP2n@8R0UH0001@0000n@8000h2003n0P006@8000d0003k0QP2n`8I0_X26@;i0QT2
n@8H0_T260;l0Pl2o08>0_d23P;n0Ph2oP8?0_h23`;k0QP2EP000500003l0P0010800003000@0P00
3@000?h23`;n0Q02o@8@0_d240;l0Pl2o08?0_h21@;n0P@2o`840_l210800`D200<50_h23`9F0000
D0000?h2003k0@000`<000H2000=000000<50P031P;o0PH2oP860_h21@;n0PD20@?l0@43n`420o/1
0`?k0@<3o0430o`100<50UH0001@00000@<00?4100050`00o@4000d000030o`10`?m0@83o@420od1
0@?l0@43o0430o810`?a0@@3l@440o411@?b0@D3lP430o`1EP00050000030`00iP4000P3003c0@00
3@0000D3lP450o<110?c0@<3l`430o810`?b0@H3i`460nH11`?V0@P3iP480nL120?W0@D3lP5F0000
D00000H3003L0@002P<00>P1000=000020?W0@P3j0470nP11`?X0@H3i`460nL120?M0@P3g0490m`1
2@?L0@X3g@4:0md120?W0EH0001@000020<00=81000=0`00gP4000d0000:0md12P?N0@T3gP480mh1
20?M0@P3g@4;0m<12`?B0@`3dP4=0m813@?C0@d3d`4:0md1EP000500000;0`00b04000l3003D0@00
3@0000d3d`4=0m@130?D0@`3e04;0m<12`?C0@d3b@4=0lP13P?80@h3b04?0lT13`?90@d3d`5F0000
D00000d3002n0@004P<00<X1000=00003`?90@l3bP4>0lX13@?:0@d3b@4=0lT140>o0A03_P4A0kh1
4P>n0A83_`4B0kl13`?90EH0001@000040<00;@1000D0`00`04000d0000B0kl14P?00A43`04A0l01
40>o0A03_`4B0kD14P>d0A<3]04C0k@150>e0A@3]@4B0kl1EP000500000B0`00Z`4001L3002f0@00
3@0001@3]@4D0kH14`>f0A83]P4B0kD14P>e0AD3[04E0j/15P>[0AL3Z`4G0j`15`>/0A@3]@5F0000
D00001D3002R0@006@<00:d1000=00005`>/0AL3[@4F0jd15@>]0AD3[04E0j`15`>S0AL3XP4H0j81
6@>R0AT3X`4I0j<15`>/0EH0001@00005`<009X1000L0`00Y04000d0000I0j<16@>T0AP3Y04G0j@1
5`>S0AL3X`4J0i/16P>J0A/3VP4L0iX170>K0A`3V`4I0j<1EP000500000J0`00T@4001h3002L0@00
3@0001`3V`4L0i`16`>L0AX3W04J0i/16P>K0A`3TP4L0i417@>A0Ah3T@4N0i817P>B0A`3V`5F0000
D00001`300290@008@<009<1000=00007P>B0Ah3T`4M0i<170>C0A`3TP4L0i817`>:0Al3R@4P0hT1
8@>90B43RP4Q0hX17P>B0EH0001@00007`<00801000S0`00R`4000d0000Q0hX18@>;0B03R`4O0h/1
7`>:0Al3RP4Q0h418@>00B83P04S0h018`>10B<3P@4Q0hX1EP000500000Q0`00M`4002H300220@00
3@0002<3P@4S0h818P>20B43PP4Q0h418@>10B@3N04T0gL19@=g0BH3M`4V0gP19P=h0B<3P@5F0000
D00002@3001^0@00:0<007T1000=00009P=h0BH3N@4U0gT190=i0B@3N04T0gP19P=_0BH3KP4W0fh1
:0=^0BP3K`4X0fl19P=h0EH0001@00009P<006D1000[0`00L04000d0000X0fl1:0=`0BL3L04V0g01
9P=_0BH3K`4Y0fH1:@=U0BX3I@4[0fD1:`=V0B/3IP4X0fl1EP000500000Y0`00G04002d3001W0@00
3@0002/3IP4[0fL1:P=W0BT3I`4Y0fH1:@=V0B/3G@4[0e`1;0=L0Bd3G04]0ed1;@=M0B/3IP5F0000
D00002/3001C0@00<0<005h1000=0000;@=M0Bd3GP4/0eh1:`=N0B/3G@4[0ed1;P=D0Bh3D`4_0e<1
<0=C0C03E04`0e@1;@=M0EH0001@0000;P<004X1000b0`00E@4000d0000`0e@1<0=E0Bl3E@4^0eD1
;P=D0Bh3E04`0d/1<0=:0C43BP4b0dX1<P=;0C83B`4`0e@1EP000500000`0`00@@4003D3001<0@00
3@000383B`4b0d`1<@=<0C03C04`0d/1<0=;0C<3@P4c0d41=0=10CD3@@4e0d81=@=20C83B`5F0000
D00003<3000h0@00=`<004<1000=0000=@=20CD3@`4d0d<1<`=30C<3@P4c0d81=@<i0CD3>04f0cP1
=`<h0CL3>@4g0cT1=@=20EH0001@0000=@<00301000j0`00>P4000d0000g0cT1=`<j0CH3>P4e0cX1
=@<i0CD3>@4h0c41>0<`0CT3<04j0c01>P<a0CX3<@4g0cT1EP000500000h0`00:@4003`3000b0@00
3@0003X3<@4j0c81>@<b0CP3<P4h0c41>0<a0CX3:P4j0bT1>`<Y0C`3:@4l0bX1?0<Z0CX3<@5F0000
D00003X3000R0@00?`<002/1000=0000?0<Z0C`3:`4k0b/1>P<[0CX3:P4j0bX1?@<S0Cd38P4n0b81
?`<R0Cl38`4o0b<1?0<Z0EH0001@0000?@<001/100110`00904000d0000o0b<1?`<T0Ch3904m0b@1
?@<S0Cd38`4o0a`1?`<K0D036`510a/1@@<L0D43704o0b<1EP000500000o0`005@4004@3000M0@00
3@00044370500ad1@0<M0Cl37@4o0a`1?`<L0D835P530aD1@`<E0D@35@540aH1A0<F0D43705F0000
D0000483000?0@00AP<001L1000=0000A0<F0D@35`530aL1@P<G0D835P520aH1A0<@0D@33`550`l1
AP<?0DH340560a01A0<F0EH0001@0000A0<000X100190`004@4000d000160a41A@<A0DD34@540a41
A0<@0D@33`570`X1B0<:0DP32P590`X1B@<;0DT330560a41EP00050000170`001P4004/3000<0@00
3@0004T32`580``1B0<<0DL330570`/1A`<;0DT31`5:0`H1BP<60D/31P5;0`L1B`<70DT32`5F0000
D00004T300010@00CP<000P1000=0000B`<80DX3205:0`P1B@<80DT31`590`H1C0<10Dd30@5=0`41
CP<10Dh30P5>0`<1B`<80EH0001@0000C0<00?h0001@0`000`4000d0001>0`<1C@<30Dd30`5<0`<1
C0<20D`30@5>0oh0C`?n04l3oP1@0oh0D0?o0503005>0`<1EP000500001>0`00n`0005<300000@00
3@000503005?0`01C`<00Dh3o`1>0ol0CP?n0543n`1B0o/0DP?k05<3o01C0o`0D`?m0503005F0000
D0000543003i0000E@<00?d0000=0000D`?m0583o@1B0od0D@?l0543o01A0o/0D`?i05@3n@1D0oT0
E@?j05D3nP1E0o/0D`?m05H0001@0000D`<00?P0001H0`00n`0000d0001D0o/0E0?k05<3n`1C0oX0
D`?i05@3n@1G0oP0E`?h05P3n01H0oT0F0?j05L3nP1D0o/0EP000500001F0`00m`0005X3003j0000
3@0005L3nP1G0oX0EP?j05H3n@1F0oT0E`?h05T3m`1I0oL0FP?g05X3n01J0oP0F@?i05L3nP1F0000
D00005P3003g0000G@<00?X0000=0000F@?i05P3n@1H0oP0F0?g05T3m`1I0oL0G0?h05d3n01M0oT0
G@?j05`3nP1L0oX0F@?i05H0001@0000F`<00?P0001O0`00o00000d0001K0oX0F`?i05/3n@1L0oP0
G0?h05d3n01O0oX0G`?k05l3n`1N0o`0GP?l05d3o01K0oX0EP000500001M0`00nP000683003o0000
3@0005d3o01M0o/0G@?k05h3nP1N0oX0G`?j0683o@1R0oh0HP?n0643o`1Q0ol0H0?o05d3o01F0000
D0000603003m0000I0<000<1000=0000H0?n0603oP1P0od0H@?m0643o@1R0oh0I0<20F@30P5T0`<1
H`<30F<30`5R0`81H0?n05H0001@0000HP<00041001W0`001`4000d0001R0`<1HP<20F830@5S0`41
H`<10F@30@5W0`D1I`<60FL31`5V0`L1IP<70FD31`5R0`<1EP000500001U0`001@4006T3000;0@00
3@0006D31P5U0`H1I@<50FH31@5V0`D1I`<60FT32P5Y0`X1J@<;0FP32`5X0`/1I`<:0FD31P5F0000
D00006L300090@00K0<000h1000=0000I`<;0FL32P5W0`X1J0<90FP32@5Y0`T1K0<<0F`33@5/0`d1
J`<>0F/33P5Z0`h1I`<;0EH0001@0000JP<000`1001^0`004P4000d0001Z0`d1JP<=0FX3305[0``1
J`<<0F`33@5^0a41KP<A0Fh34P5]0a81K@<B0F`34@5Z0`d1EP000500001/0`0040400743000E0@00
3@0006`34P5/0a41K0<A0Fd3405]0a01KP<@0G434`5a0a@1L@<D0G035@5`0aD1K`<E0F`34P5F0000
D00006l3000C0@00M0<001T1000=0000K`<E0Fl3505_0a<1L0<C0G034`5a0a<1M0<G0G@3605d0aT1
L`<I0G<36@5b0aT1K`<E0EH0001@0000LP<001L1001f0`006`4000d0001b0aT1LP<H0G83605c0aL1
L`<G0G@35`5f0aT1MP<J0GH36P5e0a/1M@<K0G@36`5b0aT1EP000500001d0`006@4007T3000N0@00
3@0007@36`5d0aX1M0<J0GD36@5e0aT1MP<I0GT3705i0ad1N@<M0GP37P5h0ah1M`<N0G@36`5F0000
D00007L3000L0@00N`<00241000=0000M`<N0GL37@5g0a`1N0<L0GP3705i0a`1N`<O0G/3805k0b41
NP<Q0GX38@5i0b41M`<N0EH0001@0000N@<001l1001n0`00904000d0001i0b41N@<P0GT3805j0al1
NP<O0G/37`5n0b81OP<S0Gh38`5m0b@1O@<T0G`3905i0b41EP000500001l0`008P400803000W0@00
3@0007`3905l0b<1O0<R0Gd38P5m0b81OP<R0H039@600bH1P0<W0Gl39`5o0bL1OP<W0G`3905F0000
D00007h3000U0@00P`<002/1000=0000OP<W0Gh39P5n0bD1O`<U0Gl39@600bD1P`<Y0H<3:P630b/1
PP<[0H83:`610b/1OP<W0EH0001@0000P@<002T100250`00<04000d000210bX1P@<Z0H43:@620bT1
P`<Y0H<3:P650bl1Q@<_0HD3<0640c01P`<`0H<3;`610bX1EP00050000230`00;P4008P3000f0@00
3@0008<3;`630bl1P`<^0H@3;P640bh1Q@<_0HP3=@680cD1R0<f0HL3=P670cH1QP<e0H<3;`5F0000
D00008H3000d0@00RP<003d1000=0000QP<e0HH3=@660c@1Q`<d0HP3=0680cD1RP<l0HX3?06:0cd1
R@<m0HP3?@680c`1QP<e0EH0001@0000R0<003/1002=0`00A04000d000280c`1R0<l0HP3>`690c/1
RP<k0HX3?06=0d<1S@=30Hd3A06<0d@1R`=40H/3@`680c`1EP000500002;0`00@P4008l3001=0@00
3@0008/3@`6;0d<1R`=20H`3@P6=0d81S@=30Hl3C06?0d`1S`==0Hh3C@6=0dd1S@=<0H/3@`5F0000
D00008d3001;0@00TP<005L1000=0000S@=<0Hd3C06=0d/1SP=;0Hl3B`6?0d`1TP=F0I83EP6B0eL1
T@=G0I03E`6@0eH1S@=<0EH0001@0000T0<005D1002D0`00H@4000d0002@0eH1T0=F0I43E@6A0eD1
TP=E0I83EP6D0f01U0=P0I<3H@6C0f41TP=Q0I83H06@0eH1EP000500002B0`00G`4009L3001[0@00
3@000983H06B0f01TP=O0I<3G`6D0el1U0=P0IL3JP6G0fX1U`=[0IH3J`6E0f/1U@=Z0I83H05F0000
D00009D3001Y0@00V@<007L1000=0000U@=Z0ID3JP6F0fT1UP=Y0IL3J@6G0fX1V@=f0IT3MP6H0gL1
V0=g0IL3M`6G0gH1U@=Z0EH0001@0000U`<007D1002L0`00PP4000d0002G0gH1U`=f0IL3M@6H0gD1
V@=e0IT3MP6L0h41W0>10I`3PP6K0h81VP>20IX3P@6G0gH1EP000500002J0`00P04009h3002>0@00
3@0009X3P@6J0h41V`>00I/3P06L0h01W0>10Ih3S@6N0hd1W@>>0Id3SP6L0hh1W0>=0IX3P@5F0000
D00009`3002<0@00X@<009X1000=0000W0>=0I`3S@6L0h`1W@><0Ih3S06N0hd1X@>I0J43V@6Q0iX1
X0>J0Il3VP6O0iT1W0>=0EH0001@0000W`<009P1002S0`00YP4000d0002O0iT1W`>I0J03V06P0iP1
X@>H0J43V@6S0jD1X`>U0J83YP6R0jH1X@>V0J43Y@6O0iT1EP000500002Q0`00Y0400:H3002c0@00
3@000:43Y@6Q0jD1X@>T0J83Y06S0j@1X`>U0JH3/P6V0k81YP>c0JD3/`6T0k<1Y0>b0J43Y@5F0000
D0000:@3002a0@00Z0<00;l1000=0000Y0>b0J@3/P6U0k41Y@>a0JH3/@6V0k81Z0>n0JP3_P6W0kl1
Y`>o0JH3_`6V0kh1Y0>b0EH0001@0000YP<00;d1002[0`00c04000d0002V0kh1YP>n0JH3_@6W0kd1
Z0>m0JP3_P6[0l/1Z`?;0J/3c06Z0l`1Z@?<0JT3b`6V0kh1EP000500002Y0`00bP400:d3003I0@00
3@000:T3b`6Y0l/1ZP?:0JX3bP6[0lX1Z`?;0Jd3f06]0mP1[0?I0J`3f@6[0mT1Z`?H0JT3b`5F0000
D0000:/3003G0@00/0<00>D1000=0000Z`?H0J/3f06[0mL1[0?G0Jd3e`6]0mP1/0?T0K03i06`0nD1
[`?U0Jh3i@6^0n@1Z`?H0EH0001@0000[P<00><1002b0`00lP4000d0002^0n@1[P?T0Jl3h`6_0n<1
/0?S0K03i06b0o41/P?a0K43lP6a0o81/0?b0K03l@6^0n@1EP000500002`0`00l0400;D3003m0@00
3@000;03l@6`0o41/0?`0K43l06b0o01/P?a0KD3o06e0o`1]@?m0K@3o@6c0od1/`?l0K03l@5F0000
D0000;<3003k0@00]`<000T2000=0000/`?l0K<3o06d0o/1]0?k0KD3n`6e0o`1]`<80[L320:f0`T2
]P<90[D32@:e0`P2/`?l0EH0001@0000]@<000L2002j0`00508000d0002e0`P2]@<80[D31`:f0`L2
]`<70[L320:j0a<2^P<C0[X350:i0a@2^0<D0[P34`:e0`P2EP000500002h0`004P800;`3000O0P00
3@000;P34`:h0a<2^@<B0[T34P:j0a82^P<C0[`37P:l0ah2^`<O0[/37`:j0al2^P<N0[P34`9F0000
D0000;X3000M0P00_`<002T2000=0000^P<N0[X37P:j0ad2^`<M0[`37@:l0ah2_`<X0[l3:0:o0bT2
_P<Y0[d3:@:m0bP2^P<N0UH0001@0000_@<002L200310`00<`8000d0002m0bP2_@<X0[h39`:n0bL2
_`<W0[l3:0;10c82`@<b0/03<`;00c<2_`<c0[l3<P:m0bP2EP000500002o0`00<@800<@3000n0P00
3@000;l3<P:o0c82_`<a0/03<@;10c42`@<b0/@3?@;40cd2a0<n0/<3?P;20ch2`P<m0[l3<P9F0000
D0000<83000l0P00aP<004P2000=0000`P<m0/83?@;30c`2``<l0/@3?0;40cd2aP=70/H3A`;50dP2
a@=80/@3B0;40dL2`P<m0UH0001@0000a0<004H200390`00D`8000d000340dL2a0=70/@3AP;50dH2
aP=60/H3A`;90e82b@=B0/T3D`;80e<2a`=C0/L3DP;40dL2EP00050000370`00D@800</3001N0P00
3@000<L3DP;70e82b0=A0/P3D@;90e42b@=B0//3G@;;0ed2bP=N0/X3GP;90eh2b@=M0/L3DP9F0000
D0000<T3001L0P00cP<006T2000=0000b@=M0/T3G@;90e`2bP=L0//3G0;;0ed2cP=X0/h3J0;>0fT2
c@=Y0/`3J@;<0fP2b@=M0UH0001@0000c0<006L2003@0`00M08000d0003<0fP2c0=X0/d3I`;=0fL2
cP=W0/h3J0;@0g<2d0=c0/l3M0;?0g@2cP=d0/h3L`;<0fP2EP000500003>0`00LP800=<3001o0P00
3@000<h3L`;>0g<2cP=b0/l3LP;@0g82d0=c0]<3OP;C0gh2d`=o0]83O`;A0gl2d@=n0/h3L`9F0000
D0000=43001m0P00e@<008X2000=0000d@=n0]43OP;B0gd2dP=m0]<3O@;C0gh2e@>90]D3R@;D0hX2
e0>:0]<3RP;C0hT2d@=n0UH0001@0000d`<008P2003H0`00U@8000d0003C0hT2d`>90]<3R0;D0hP2
e@>80]D3R@;H0i@2f0>D0]P3U@;G0iD2eP>E0]H3U0;C0hT2EP000500003F0`00T`800=X3002P0P00
3@000=H3U0;F0i@2e`>C0]L3T`;H0i<2f0>D0]X3W`;J0il2f@>P0]T3X0;H0j02f0>O0]H3U09F0000
D0000=P3002N0P00g@<00:/2000=0000f0>O0]P3W`;H0ih2f@>N0]X3WP;J0il2g@>Z0]d3ZP;M0j/2
g0>[0]/3Z`;K0jX2f0>O0UH0001@0000f`<00:T2003O0`00]@8000d0003K0jX2f`>Z0]`3Z@;L0jT2
g@>Y0]d3ZP;O0k@2g`>d0]h3]@;N0kD2g@>e0]d3]0;K0jX2EP000500003M0`00/`800>8300300P00
3@000=d3]0;M0k@2g@>c0]h3/`;O0k<2g`>d0^83_`;R0kl2hP?00^43`0;P0l02h0>o0]d3]09F0000
D0000>03002n0P00i0<00<X2000=0000h0>o0^03_`;Q0kh2h@>n0^83_P;R0kl2i0?90^@3b@;S0lX2
h`?:0^83bP;R0lT2h0>o0UH0001@0000hP<00<P2003W0`00e08000d0003R0lT2hP?90^83b0;S0lP2
i0?80^@3b@;W0m<2i`?C0^L3e0;V0m@2i@?D0^D3d`;R0lT2EP000500003U0`00dP800>T3003M0P00
3@000>D3d`;U0m<2i@?B0^H3dP;W0m82i`?C0^T3g0;Y0m`2j@?M0^P3g@;W0md2i`?L0^D3d`9F0000
D0000>L3003K0P00k0<00>L2000=0000i`?L0^L3g0;W0m/2j0?K0^T3f`;Y0m`2k0?V0^`3iP;/0nL2
j`?W0^X3i`;Z0nH2i`?L0UH0001@0000jP<00>D2003^0`00k`8000d0003Z0nH2jP?V0^X3i@;[0nD2
k0?U0^`3iP;^0nh2kP?^0^h3k`;]0nl2k0?_0^`3kP;Z0nH2EP000500003/0`00k@800?43003g0P00
3@000>`3kP;/0nh2k0?]0^d3k@;^0nd2kP?^0_43mP;a0oH2l@?g0_03m`;_0oL2k`?f0^`3kP9F0000
D0000>l3003e0P00l`<00?l2000=0000k`?f0^l3mP;_0oD2l0?e0_43m@;a0oH2l`?n0_<3oP;c0ol2
lP?o0_43o`;a0oh2k`?f0UH0001@0000l@<00?d2003f0`001P<000d0003a0oh2l@?n0_43o@;b0od2
l`?m0_<3oP;f0`D3mP<50oH31P?e0`H3m0<60o@31@?a0oh2EP000500003d0`0010<00?T3000=0`00
3@000?@31@?d0`D3m0<40oD310?f0`@3mP<50oT330?i0``3n@<=0oP33@?g0`d3m`<<0o@31@=F0000
D0000?L3000;0`00n`<001@3000=0000m`<<0oL330?g0`/3n0<;0oT32`?i0``3n`<C0o/34`?k0a@3
nP<D0oT350?i0a<3m`<<0eH0001@0000n@<00183003n0`006P<000d0003i0a<3n@<C0oT34P?j0a83
nP<B0o/34`?n0aT3oP<I0oh36P?m0aX3o@<J0o`36@?i0a<3EP000500003l0`0060<00004000P0`00
3@000?`36@?l0aT3o0<H0od360?n0aP3oP<I0`047`<011l300@P0ol380?n0b03oP<O0o`36@=F0000
D0000?h3000N0`000`@002D3000=0000oP<P0oh37`?n0ah3o`<N0ol37P<011h30`@S0`<490<312D3
0P@U0`849@<112D3oP<P0eH0001@00000@@002<300051000:P<000d0000112@30@@T0`448`<212<3
0`@S0`<490<512T31@@Y0`D4:P<412X30`@Z0`<4:@<112@3EP00050000031000:0<000P4000`0`00
3@0000<4:@<312T30`@X0`@4:0<412P31@@Y0`P4;`<812l320@`0`L4<0<713031P@_0`<4:@=F0000
D00000H4000^0`002P@003@3000=00001P@_0`H4;`<612h31`@^0`L4;P<812l32P@c0`X4<`<:13@3
2@@d0`T4=0<813<31P@_0eH0001@000020@00383000=1000>@<000d0000813@320@c0`P4<P<91383
2@@b0`X4<P<=13L33@@h0`d4>@<<13T330@i0`/4>@<813@3EP000500000;1000=`<000l4000m0`00
3@0000/4>0<;13P32`@g0``4=`<<13L33@@h0`l4?0<?13`33`@m0`h4?@<>13d33@@l0`/4>0=F0000
D00000d4000k0`004P@00443000=00003@@m0`d4?0<=13/33P@k0`h4>`<?13/34P@o0a84@0<B1443
4@A10a44@@<@14433@@m0eH0001@000040@003l3000D1000A@<000d0000@140340A00a04?`<A13l3
4@@o0a84@0<D14@350A40a@4A@<C14D34`A50a84A0<@1403EP000500000B1000@`<001L400180`00
3@000184A@<B14@34PA40a<4@`<C14<350A30aL4AP<G14L35`A70aH4B0<F14P35@A80a84A@=F0000
D00001D400160`006@@004/3000=00005@A80aD4A`<E14H35PA60aH4AP<G14H36@A90aT4BP<I14/3
60A;0aP4B`<G14/35@A80eH0001@00005`@004T3000L1000CP<000d0000G14/35`A:0aL4BP<H14T3
60A90aT4B@<L14`370A=0a`4C@<K14h36`A>0aX4CP<G14/3EP000500000J1000C0<001h4001A0`00
3@0001X4CP<J14d36PA<0a/4C0<K14`370A<0ah4C`<N15037PAA0ad4D@<M154370AA0aX4CP=F0000
D00001`4001?0`008@@005@3000=000070AA0a`4D0<L15037@A?0ad4C`<N14l38@AB0b44D`<Q15<3
80AD0b04E0<O15@370AA0eH0001@00007`@00583000S1000E`<000d0000O15@37`AC0al4DP<P1583
80AB0b44DP<S15D38`AF0b<4E`<R15L38PAG0b44E`<O15@3EP000500000Q1000E@<002H4001I0`00
3@000244E`<Q15H38@AF0b44E@<R15D38`AE0bH4E`<V15P39PAH0bH4F@<U15T390AI0b44E`=F0000
D00002@4001G0`00:0@005/3000=000090AI0b@4F0<T15P39@AG0bD4E`<V15L3:0AI0bP4FP<X15X3
9`AK0bL4F`<V15/390AI0eH0001@00009P@005T3000[1000G@<000d0000V15/39PAJ0bH4FP<V15T3
9`AI0bP4F@<[15/3:`AL0b/4G0<[15d3:PAM0bT4G@<V15/3EP000500000Y1000F`<002d4001O0`00
3@0002T4G@<Y15`3:@AL0bX4F`<Z15/3:`AK0bd4G@<]15h3;@AN0b`4G`</15l3:`AO0bT4G@=F0000
D00002/4001M0`00<0@00643000=0000:`AO0b/4GP<[15h3:`AM0b`4G@<]15d3<0AO0c04H0<`1603
<0AQ0bl4H@<^1643:`AO0eH0001@0000;P@005l3000b1000H`<000d0000^1643;PAP0bh4H0<_15l3
;`AO0c04G`<b1643<PAR0c84HP<a16<3<@AS0c04H`<^1643EP000500000`1000H@<003D4001U0`00
3@000304H`<`1683<0AR0c04H@<a1643<PAQ0cD4H`<e16@3=@AT0cD4I@<d16D3<`AU0c04H`=F0000
D00003<4001S0`00=`@006L3000=0000<`AU0c<4I0<c16@3=0AS0c@4H`<e16<3=`AU0cL4IP<g16H3
=PAW0cH4I`<e16L3<`AU0eH0001@0000=@@006D3000j1000J0<000d0000f16L3=@AW0cD4IP<e16D3
=PAU0cH4I@<i16H3>PAV0cX4I`<j16P3>@AX0cT4J0<f16L3EP000500000h1000IP<003`4001Z0`00
3@0003P4J0<h16L3>0AW0cT4IP<i16H3>PAV0c`4J0<l16T3?0AY0c/4JP<k16X3>PAZ0cP4J0=F0000
D00003X4001X0`00?`@006/3000=0000>`AZ0cX4JP<j16T3>PAX0c/4J0<k16P3?PAY0cl4J@<o16X3
?`A[0ch4J`<n16/3>`AZ0eH0001@0000?@@006T300111000K@<000d0000m16/3?@AZ0cd4JP<n16T3
?PAY0cl4J@=116/3@@A/0d44K0=016d3@0A]0cl4K@<m16/3EP000500000o1000J`<004@4001^0`00
3@000404K@<o16d3?`A/0cl4J`=016/3@0A[0d<4K0=416`3A0A]0d@4KP=316h3@`A^0d04K@=F0000
D0000484001/0`00AP@00703000=0000@PA^0d84K@=216d3@`A/0d<4K0=416`3APA^0dH4K`=616l3
A@A`0dD4L0=41703@PA^0eH0001@0000A0@006h300191000L@<000d000151703A0A`0d@4K`=416h3
A@A^0dD4KP=816l3B@A_0dT4L0=91743B0Aa0dP4L@=51703EP00050000171000K`<004/4001b0`00
3@0004P4L@=71703A`A`0dL4K`=816l3B0A_0dX4L0=;1743B`Aa0d/4LP=:1783BPAb0dP4L@=F0000
D00004T4001`0`00CP@007<3000=0000BPAb0dT4LP=91743B@A`0dX4L0=:1703C@Aa0dh4L@=>1783
CPAc0dd4L`==17<3BPAb0eH0001@0000C0@00743001@1000M0<000d0001=17<3C0Ab0d`4LP=<1743
C@Aa0dd4L@=?1783D0Ac0e04L`=@17@3C`Ad0dl4M0==17<3EP000500001>1000LP<005<4001e0`00
3@0004l4M0=>17@3CPAc0dh4LP=?1783C`Ab0e84L`=C17<3D`Ad0e<4M@=B17D3DPAe0dl4M0=F0000
D0000544001c0`00E@@007H3000=0000DPAe0e44M0=A17@3D@Ac0e84L`=B17<3E0Ad0eD4M@=E17D3
E@Af0e@4MP=D17H3DPAe0eH0001@0000D`@007@3001H1000M`<000d0001D17H3D`Af0e<4M@=C17@3
E0Ad0e@4M0=G17D3F0Ae0eP4MP=H17L3E`Ag0eL4M`=D17H3EP000500001F1000M@<005X4001h0`00
3@0005L4M`=F17H3EPAf0eH4M@=G17D3E`Ae0eT4MP=J17L3FPAg0eX4N0=I17P3F@Ah0eL4M`=F0000
D00005P4001f0`00G@@007T3000=0000F@Ah0eP4N0=H17L3F0Af0eT4MP=I17H3G0Ag0ed4M`=M17P3
G@Ai0e`4N@=L17T3F@Ah0eH0001@0000F`@007L3001O1000N@<000d0001L17T3F`Ai0e/4N0=K17P3
F`Ag0e`4M`=N17L3G`Ag0el4N0=O17P3G`Ai0eh4N@=L17T3EP000500001M1000M`<00684001j0`00
3@0005h4N@=M17T3G@Ah0ed4M`=N17L3GPAg0f44N0=R17P3HPAi0f84NP=Q17X3H@Aj0eh4N@=F0000
D0000604001h0`00I0@007X3000=0000H@Aj0f04NP=P17T3H0Ai0f04N0=Q17P3H`Ah0f@4N0=T17T3
I0Ai0f@4NP=S17X3H@Aj0eH0001@0000HP@007P3001W1000N`<000d0001S17X3HPAj0f84N@=R17P3
H`Ah0f<4N0=V17T3I`Ai0fL4NP=W17/3IPAk0fH4N`=S17X3EP000500001U1000N@<006T4001k0`00
3@0006H4N`=U17/3I@Aj0fD4NP=U17T3IPAi0fP4N@=Y17T3J@Aj0fT4NP=Y17/3J0Ak0fH4N`=F0000
D00006L4001i0`00K0@007/3000=0000J0Ak0fL4N`=W17X3I`Aj0fL4N@=X17T3J`Ai0f`4N@=/17X3
K0Aj0f`4N`=[17/3J0Ak0eH0001@0000JP@007T3001^1000O0<000d0001[17/3JPAj0fX4NP=Z17T3
J`Ai0f/4N@=]17X3KPAk0fh4N`=^17`3K@Al0fd4O0=[17/3EP000500001/1000NP<00744001l0`00
3@0006d4O0=/17`3K0Ak0f`4N`=/17X3K@Aj0g04NP=a17X3L@Ak0g44N`=a17`3L0Al0fd4O0=F0000
D00006l4001j0`00L`@007`3000=0000L0Al0fl4O0=_17/3K`Ak0fl4NP=`17X3LPAj0g<4NP=c17/3
L`Ak0g<4O0=b17`3L0Al0eH0001@0000L@@007X3001f1000O0<000d0001b17`3L@Al0g44N`=a17/3
L@Aj0g84NP=e17X3MPAj0gH4N`=f17/3MPAl0gD4O0=b17`3EP000500001d1000NP<007P4001l0`00
3@0007D4O0=d17`3M0Ak0g@4N`=d17X3M@Aj0gL4NP=h17X3N0Ak0gP4N`=h17`3M`Al0gD4O0=F0000
D00007H4001j0`00N`@007`3000=0000M`Al0gH4O0=f17/3MPAk0gH4NP=g17X3NPAj0g/4NP=k17/3
N`Ak0g/4O0=j17`3M`Al0eH0001@0000N@@007X3001m1000O0<000d0001j17`3N@Al0gT4N`=i17/3
N@Aj0gX4NP=l17X3O@Aj0gd4N`=m17/3O@Al0g`4O0=j17`3EP000500001k1000NP<00804001l0`00
3@0007`4O0=k17`3N`Ak0g/4N`=k17X3O0Aj0gl4NP>017X3P0Ak0h04N`>017`3O`Al0g`4O0=F0000
D00007h4001j0`00P`@007`3000=0000O`Al0gh4O0=n17/3OPAk0gh4NP=o17X3PPAj0h<4NP>317/3
P`Ak0h<4O0>217`3O`Al0eH0001@0000P@@007X300251000O0<000d0002217`3P@Al0h44N`>117/3
P@Aj0h84NP>417X3Q@Aj0hD4N`>517/3Q@Al0h@4O0>217`39P0001`0000100000000004000000000
000002D0000<00000@0005L0000d0000RP0006P000341000Q@<000H0002:08@3RP1X0<@4J03418@3
RP240hX0Q@<U0000300000D0080X000030000080000U0000300000P0080X000030000040000U0000
300000d0080>000050000000000@00005000
\>"], "Graphics",
 ImageSize->{300.75, 238.188},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}}],

Cell[GraphicsData["Metafile", "\<\
CF5dJ6E]HGAYHf4PEfU^I6mgLb15CDHPAVmbKF5d0@000L[d0@0006`00000000000000?D4003/0`00
000000000037FP00idL00215CDH00040m<X10>4:000300000000000000000000@0H00;04000U0@00
g000000000000000000008Qh101PF`<02@000100003f1000k@<000X0000@000000000000000D0000
300000d0000U0000300000P0080W00006000004000000000oooo0000000U000030000040000[0000
6000000000000000mP@00>d3000[00006000000000000000mP@00>d3000V00007000008000000000
0@000000000000009@0000`000020000E`0002@0003W00002P000>P0000=00000P000>L03@3X00X0
E`0002@0003X00002P000>P0000?00000P000>P02P3X00l0E`0002@0003W00003@000>P0000?0000
0P000>P03`3W00d0E`0002@0003U00002`000>L0000=00000P000>L03@3U00/0E`0002@0003R0000
2P000>D0000;00000P000>D02`3R00X0E`0002@0003P00002P000>80000:00000P000>802P3P00X0
E`0002@0003M00002P000>00000;00000P000>002P3M00/0E`0002@0003K00002`000=d0000=0000
0P000=d02`3K00d0E`0002@0003K00003@000=/0000?00000P000=/03@3K00l0E`0002@0003K0000
3`000=`0000@00000P000=/03`3L0100E`0002@0003L000040000=d0000A00000P000=`0403M0140
E`0002@0003M00004@000=l0000B00000P000=d04@3O0180E`0002@0003O00004P000>@0000D0000
0P000=l04P3T01@0E`0002@0003T000050000>H0000E00000P000>@0503V01D0E`0002@0003V0000
5@000>P0000G00000P000>H05@3X01L0E`0002@0003K00003`000=d0000@00000P000=/03`3M0100
E`0002@0003M000040000=l0000A00000P000=d0403O0140E`0002@0003O00004@000>@0000C0000
0P000=l04@3T01<0E`0002@0003T00004`000>H0000D00000P000>@04`3V01@0E`0002@0003V0000
50000>L0000E00000P000>H0503W01D0E`0002@0003W00005@000>P0000G00000P000>L05@3X01L0
E`0002@0003X00005`000>P0000K00000P000>P05`3X01/0E`0002@0003V00006`000>P0000M0000
0P000>P06`3V01d0E`0002@0003S00007@000>H0000M00000P000>H07@3S01d0E`0002@0003Q0000
7@000><0000M00000P000><07@3Q01d0E`0002@0003N00007@000>40000M00000P000>407@3N01d0
E`0002@0003L00006`000=h0000M00000P000=h07@3L01/0E`0002@0003K000060000=`0000K0000
0P000=`06`3K01P0E`0002@0003K000060000=/0000M00000P000=/0603K01d0E`0002@0003K0000
6`000=`0000M00000P000=/07@3L01/0E`0002@0003^00002P000>l0000;00000P000>l02P3^00/0
E`0002@0003^00002`000>l0000<00000P000>h02`3_00`0E`0002@0003_00002`000?00000<0000
0P000>l0303`00/0E`0002@0003_00002P000?00000;00000P000?002`3_00X0E`0002@0003_0000
40000>l0000M00000P000>l0403_01d0E`0002@0003`000040000?00000M00000P000?00403`01d0
E`0002@0003/000040000?00000@00000P000>`0403`0100E`0002@0003/00007@000?<0000M0000
0P000>`07@3c01d0E`0002@0003h000040000?P0000M00000P000?P0403h01d0E`0002@0003i0000
40000?T0000M00000P000?T0403i01d0E`0002@0003i00004@000?/0000C00000P000?T04`3k0140
E`0002@0003k000040000?d0000A00000P000?/04@3m0100E`0002@0003m000040000?l0000@0000
0P000?d0403o0100E`0002@0003o000040000081000A00000P000?l040020A40E`0002@000020@00
4@0000<1000C00000P0000814@030A<0E`0002@000030@004`0000<1000M00000P0000<14`030Ad0
E`0002@0003o000040000041000A00000P000?l040010A40E`0002@000010@004@000081000C0000
0P0000414@020A<0E`0002@000020@004`000081000M00000P0000814`020Ad0E`0002@000030@00
4@0000@1000C00000P0000<14`040A40E`0002@000040@00400000L1000A00000P0000@14@070A00
E`0002@000070@00400000T1000@00000P0000L140090A00E`0002@000090@00400000`1000A0000
0P0000T1400<0A40E`0002@0000<0@004@0000d1000C00000P0000`14@0=0A<0E`0002@0000=0@00
4`0000d1000M00000P0000d14`0=0Ad0E`0002@000090@00400000/1000A00000P0000T1400;0A40
E`0002@0000;0@004@0000`1000C00000P0000/14@0<0A<0E`0002@0000<0@004`0000`1000M0000
0P0000`14`0<0Ad0E`0002@0003e000040000?T0000@00000P000?D0403i0100E`0002@0003e0000
7@000?`0000M00000P000?D07@3l01d0E`0002@0003o00007@0000D1000M00000P000?l07@050Ad0
E`0002@000090@007@0000l1000M00000P0000T17@0?0Ad0E`0002@0000F0@00400001H1000K0000
0P0001H1400F0A/0E`0002@0000F0@006`0001L1000M00000P0001H16`0G0Ad0E`0002@0000G0@00
7@0001X1000M00000P0001L17@0J0Ad0E`0002@0000J0@007@0001/1000M00000P0001X17@0K0Ad0
E`0002@0000K0@007@0001h1000M00000P0001/17@0N0Ad0E`0002@0000N0@006`000201000M0000
0P0001h17@0P0A/0E`0002@0000G0@00400001L1000K00000P0001L1400G0A/0E`0002@0000G0@00
6`0001P1000M00000P0001L16`0H0Ad0E`0002@0000H0@007@0001X1000M00000P0001P17@0J0Ad0
E`0002@0000P0@0040000201000M00000P000201400P0Ad0E`0002@0000Q0@0040000241000M0000
0P000241400Q0Ad0E`0002@0000C0@00400001L1000@00000P0001<1400G0A00E`0002@0000M0@00
40000241000@00000P0001d1400Q0A00E`0002@0000P0@007@0002<1000M00000P0002017@0S0Ad0
E`0002@0000Y0@002P0002T1000M00000P0002T12P0Y0Ad0E`0002@0000Z0@002P0002X1000M0000
0P0002X12P0Z0Ad0E`0002@0000V0@002P0002X1000:00000P0002H12P0Z0@X0E`0002@0000V0@00
7@0002d1000M00000P0002H17@0]0Ad0E`0002@0000c0@004P0003<1000C00000P0003<14P0c0A<0
E`0002@0000b0@004`0003<1000C00000P0003<14`0b0A<0E`0002@0000b0@004P000381000C0000
0P0003814`0b0A80E`0002@0000b0@004@0003<1000B00000P0003814P0c0A40E`0002@0000c0@00
400003D1000A00000P0003<14@0e0A00E`0002@0000e0@00400003P1000@00000P0003D1400h0A00
E`0002@0000h0@00400003X1000A00000P0003P1400j0A40E`0002@0000j0@004@0003/1000B0000
0P0003X14@0k0A80E`0002@0000k0@004P0003`1000D00000P0003/14P0l0A@0E`0002@0000l0@00
500003`1000K00000P0003`1500l0A/0E`0002@0000l0@006`0003d1000M00000P0003`16`0m0Ad0
E`0002@0000m0@007@0003h1000M00000P0003d17@0n0Ad0E`0002@0000k0@004P0003/1000K0000
0P0003/14P0k0A/0E`0002@0000k0@006`0003`1000M00000P0003/16`0l0Ad0E`0002@0000l0@00
7@0003h1000M00000P0003`17@0n0Ad0E`0002@0000n0@007@0003l1000M00000P0003h17@0o0Ad0
E`0002@0000j0@00500003/1000E00000P0003/1500j0AD0E`0002@0000e0@005@0003X1000F0000
0P0003X15@0e0AH0E`0002@0000b0@005P0003D1000G00000P0003D15P0b0AL0E`0002@0000a0@00
5`000381000I00000P0003815`0a0AT0E`0002@0000a0@006@000341000K00000P0003416@0a0A/0
E`0002@0000a0@006`000381000M00000P0003416`0b0Ad0E`0002@0000b0@007@0003D1000M0000
0P0003817@0e0Ad0E`0002@0000e0@007@0003L1000M00000P0003D17@0g0Ad0E`0002@0000g0@00
7@0003T1000M00000P0003L17@0i0Ad0E`0002@0000i0@006`0003/1000M00000P0003T17@0k0A/0
E`0002@0000c0@005P0003D1000G00000P0003D15P0c0AL0E`0002@0000b0@005`0003<1000I0000
0P0003<15`0b0AT0E`0002@0000b0@006@000381000K00000P0003816@0b0A/0E`0002@0000b0@00
6`0003<1000M00000P0003816`0c0Ad0E`0002@0000c0@007@0003D1000M00000P0003<17@0e0Ad0
E`0002@000150@002P0004D1000J00000P0004D12P150AX0E`0002@000150@006P0004H1000M0000
0P0004D16P160Ad0E`0002@000160@007@0004P1000M00000P0004H17@180Ad0E`0002@000180@00
7@0004T1000M00000P0004P17@190Ad0E`0002@000190@007@0004/1000M00000P0004T17@1;0Ad0
E`0002@0001;0@006`0004`1000M00000P0004/17@1<0A/0E`0002@000160@002P0004H1000J0000
0P0004H12P160AX0E`0002@000160@006P0004L1000M00000P0004H16P170Ad0E`0002@000170@00
7@0004P1000M00000P0004L17@180Ad0E`0002@000120@00400004T1000@00000P00048140190A00
E`0002@0001A0@002P000581000;00000P0005812P1A0@/0E`0002@0001A0@002`000581000<0000
0P0005412`1B0@`0E`0002@0001B0@002`0005<1000<00000P000581301C0@/0E`0002@0001B0@00
2P0005<1000;00000P0005<12`1B0@X0E`0002@0001B0@0040000581000M00000P000581401B0Ad0
E`0002@0001C0@00400005<1000M00000P0005<1401C0Ad0E`0002@0001?0@00400005<1000@0000
0P0004l1401C0A00E`0002@0001?0@007@0005H1000M00000P0004l17@1F0Ad0E`0002@0001M0@00
40000601000A00000P000601401M0A40E`0002@0001K0@004@0005d1000C00000P0005d14@1K0A<0
E`0002@0001J0@004`0005/1000F00000P0005/14`1J0AH0E`0002@0001J0@005P0005X1000H0000
0P0005X15P1J0AP0E`0002@0001J0@00600005/1000K00000P0005X1601K0A/0E`0002@0001K0@00
6`0005d1000M00000P0005/16`1M0Ad0E`0002@0001M0@007@000601000M00000P0005d17@1P0Ad0
E`0002@0001P0@007@000641000M00000P0006017@1Q0Ad0E`0002@0001Q0@007@0006@1000M0000
0P0006417@1T0Ad0E`0002@0001T0@006`0006H1000M00000P0006@17@1V0A/0E`0002@0001V0@00
600006L1000K00000P0006H16`1W0AP0E`0002@0001W0@005P0006L1000H00000P0006L1601W0AH0
E`0002@0001V0@004`0006L1000F00000P0006L15P1V0A<0E`0002@0001T0@004@0006H1000C0000
0P0006H14`1T0A40E`0002@0001Q0@00400006@1000A00000P0006@14@1Q0A00E`0002@0001P0@00
40000641000@00000P000641401P0A00E`0002@0001N0@0040000601000A00000P000601401N0A40
E`0002@0001L0@004@0005h1000C00000P0005h14@1L0A<0E`0002@0001K0@004`0005`1000F0000
0P0005`14`1K0AH0E`0002@0001K0@005P0005/1000H00000P0005/15P1K0AP0E`0002@0001K0@00
600005`1000K00000P0005/1601L0A/0E`0002@0001L0@006`0005h1000M00000P0005`16`1N0Ad0
E`0002@0001N0@007@000601000M00000P0005h17@1P0Ad0E`0002@0001Q0@007@0006<1000M0000
0P0006417@1S0Ad0E`0002@0001S0@006`0006D1000M00000P0006<17@1U0A/0E`0002@0001U0@00
600006H1000K00000P0006D16`1V0AP0E`0002@0001V0@005P0006H1000H00000P0006H1601V0AH0
E`0002@0001U0@004`0006H1000F00000P0006H15P1U0A<0E`0002@0001S0@004@0006D1000C0000
0P0006D14`1S0A40E`0002@0001Q0@00400006<1000A00000P0006<14@1Q0A00E`0002@0001^0@00
400006h1000M00000P0006h1401^0Ad0E`0002@0001_0@00400006l1000M00000P0006l1401_0Ad0
E`0002@0001_0@004@000741000C00000P0006l14`1a0A40E`0002@0001a0@00400007<1000A0000
0P0007414@1c0A00E`0002@0001c0@00400007D1000@00000P0007<1401e0A00E`0002@0001e0@00
400007P1000A00000P0007D1401h0A40E`0002@0001h0@004@0007T1000C00000P0007P14@1i0A<0
E`0002@0001i0@004`0007T1000M00000P0007T14`1i0Ad0E`0002@0001e0@00400007L1000A0000
0P0007D1401g0A40E`0002@0001g0@004@0007P1000C00000P0007L14@1h0A<0E`0002@0001h0@00
4`0007P1000M00000P0007P14`1h0Ad0E`0002@0001[0@00400006l1000@00000P0006/1401_0A00
E`0002@0001[0@007@000781000M00000P0006/17@1b0Ad0E`0002@0001e0@007@0007/1000M0000
0P0007D17@1k0Ad0E`0002@0002;0@00400008l1000M00000P0008/1402?0Ad0E`0002@0002<0@00
400008l1000K00000P0008`1402?0A/0E`0002@0002?0@0040000981000M00000P000981402?0Ad0
E`0002@0002B0@00400009H1000M00000P000981402F0Ad0E`0002@0002C0@00400009H1000K0000
0P0009<1402F0A/0E`0002@0002F0@00400009T1000M00000P0009T1402F0Ad0E`0002@000280@00
400008l1000@00000P0008P1402?0A00E`0002@0002G0@00400009`1000@00000P0009L1402L0A00
E`0002@0002P0@002P000:41000;00000P000:412P2P0@/0E`0002@0002P0@002`000:41000<0000
0P000:012`2Q0@`0E`0002@0002Q0@002`000:81000<00000P000:41302R0@/0E`0002@0002Q0@00
2P000:81000;00000P000:812`2Q0@X0E`0002@0002Q0@0040000:41000M00000P000:41402Q0Ad0
E`0002@0002R0@0040000:81000M00000P000:81402R0Ad0E`0002@0002N0@0040000:81000@0000
0P0009h1402R0A00E`0002@0002N0@007@000:D1000M00000P0009h17@2U0Ad0E`0002@0002[0@00
2P000:/1000J00000P000:/12P2[0AX0E`0002@0002[0@006P000:`1000M00000P000:/16P2/0Ad0
E`0002@0002/0@007@000:h1000M00000P000:`17@2^0Ad0E`0002@0002^0@007@000:l1000M0000
0P000:h17@2_0Ad0E`0002@0002_0@007@000;41000M00000P000:l17@2a0Ad0E`0002@0002a0@00
6`000;81000M00000P000;417@2b0A/0E`0002@0002/0@002P000:`1000J00000P000:`12P2/0AX0
E`0002@0002/0@006P000:d1000M00000P000:`16P2]0Ad0E`0002@0002]0@007@000:h1000M0000
0P000:d17@2^0Ad0E`0002@0002X0@0040000:l1000@00000P000:P1402_0A00E`0002@0002h0@00
2P000;P1000M00000P000;P12P2h0Ad0E`0002@0002i0@002P000;T1000M00000P000;T12P2i0Ad0
E`0002@0002i0@004@000;/1000C00000P000;T14`2k0A40E`0002@0002k0@0040000;d1000A0000
0P000;/14@2m0A00E`0002@0002m0@0040000;l1000@00000P000;d1402o0A00E`0002@0002o0@00
40000<81000A00000P000;l140320A40E`0002@000320@004@000<<1000C00000P000<814@330A<0
E`0002@000330@004`000<<1000M00000P000<<14`330Ad0E`0002@0002o0@0040000<41000A0000
0P000;l140310A40E`0002@000310@004@000<81000C00000P000<414@320A<0E`0002@000320@00
4`000<81000M00000P000<814`320Ad0E`0002@0002e0@002P000;T1000:00000P000;D12P2i0@X0
E`0002@0002e0@007@000;`1000M00000P000;D17@2l0Ad0E`0002@0002o0@007@000<D1000M0000
0P000;l17@350Ad0E`0002@0003M0@0040000=h1000B00000P000=d14P3N0A00E`0002@0003N0@00
40000=h1000D00000P000=h1403N0A@0E`0002@0003M0@004P000=h1000D00000P000=h1503M0A80
E`0002@0003L0@004@000=d1000B00000P000=d14P3L0A40E`0002@0003J0@0040000=`1000A0000
0P000=`14@3J0A00E`0002@0003G0@0040000=X1000@00000P000=X1403G0A00E`0002@0003E0@00
40000=L1000A00000P000=L1403E0A40E`0002@0003D0@004@000=D1000B00000P000=D14@3D0A80
E`0002@0003D0@004P000=@1000D00000P000=@14P3D0A@0E`0002@0003D0@0050000=D1000E0000
0P000=@1503E0AD0E`0002@0003E0@005@000=L1000F00000P000=D15@3G0AH0E`0002@0003G0@00
5P000=/1000H00000P000=L15P3K0AP0E`0002@0003K0@0060000=d1000I00000P000=/1603M0AT0
E`0002@0003M0@006@000=h1000J00000P000=d16@3N0AX0E`0002@0003D0@004`000=D1000D0000
0P000=@14`3E0A@0E`0002@0003E0@0050000=L1000E00000P000=D1503G0AD0E`0002@0003G0@00
5@000=/1000G00000P000=L15@3K0AL0E`0002@0003K0@005`000=d1000H00000P000=/15`3M0AP0
E`0002@0003M0@0060000=h1000I00000P000=d1603N0AT0E`0002@0003N0@006@000=h1000L0000
0P000=h16@3N0A`0E`0002@0003M0@0070000=h1000M00000P000=h1703M0Ad0E`0002@0003K0@00
7@000=d1000M00000P000=d17@3K0Ad0E`0002@0003H0@007@000=/1000M00000P000=/17@3H0Ad0
E`0002@0003F0@007@000=P1000M00000P000=P17@3F0Ad0E`0002@0003E0@0070000=H1000M0000
0P000=H17@3E0A`0E`0002@0003D0@006P000=D1000L00000P000=D1703D0AX0E`0002@0003D0@00
6P000=@1000M00000P000=@16P3D0Ad0E`0002@0003D0@0070000=D1000M00000P000=@17@3E0A`0
E`0002@0003^0@002P000>h1000M00000P000>h12P3^0Ad0E`0002@0003_0@002P000>l1000M0000
0P000>l12P3_0Ad0E`0002@0003/0@004@000>h1000C00000P000>h14`3/0A40E`0002@0003Z0@00
40000>`1000A00000P000>`14@3Z0A00E`0002@0003Y0@0040000>X1000@00000P000>X1403Y0A00
E`0002@0003V0@0040000>T1000A00000P000>T1403V0A40E`0002@0003T0@004@000>H1000C0000
0P000>H14@3T0A<0E`0002@0003S0@004`000>@1000F00000P000>@14`3S0AH0E`0002@0003S0@00
5P000><1000H00000P000><15P3S0AP0E`0002@0003S0@0060000>@1000K00000P000><1603T0A/0
E`0002@0003T0@006`000>H1000M00000P000>@16`3V0Ad0E`0002@0003V0@007@000>T1000M0000
0P000>H17@3Y0Ad0E`0002@0003Y0@007@000>X1000M00000P000>T17@3Z0Ad0E`0002@0003Z0@00
7@000>`1000M00000P000>X17@3/0Ad0E`0002@0003/0@006`000>h1000M00000P000>`17@3^0A/0
E`0002@0003W0@0040000>T1000A00000P000>T1403W0A40E`0002@0003U0@004@000>L1000C0000
0P000>L14@3U0A<0E`0002@0003T0@004`000>D1000F00000P000>D14`3T0AH0E`0002@0003T0@00
5P000>@1000H00000P000>@15P3T0AP0E`0002@0003T0@0060000>D1000K00000P000>@1603U0A/0
E`0002@0003U0@006`000>L1000M00000P000>D16`3W0Ad0E`0002@0003W0@007@000>T1000M0000
0P000>L17@3Y0Ad0E`0002@0003[0@002P000>l1000:00000P000>/12P3_0@X0E`0002@0003^0@00
7@000?41000M00000P000>h17@3a0Ad0E`0002@0003h0@0040000?P1000K00000P000?P1403h0A/0
E`0002@0003h0@006`000?T1000M00000P000?P16`3i0Ad0E`0002@0003i0@007@000?`1000M0000
0P000?T17@3l0Ad0E`0002@0003l0@007@000?d1000M00000P000?`17@3m0Ad0E`0002@0003m0@00
7@000002000M00000P000?d17@000Qd0E`0002@000000P006`000082000M00000P0000027@020Q/0
E`0002@0003i0@0040000?T1000K00000P000?T1403i0A/0E`0002@0003i0@006`000?X1000M0000
0P000?T16`3j0Ad0E`0002@0003j0@007@000?`1000M00000P000?X17@3l0Ad0E`0002@000020P00
40000082000M00000P00008240020Qd0E`0002@000030P00400000<2000M00000P0000<240030Qd0
E`0002@0003e0@0040000?T1000@00000P000?D1403i0A00E`0002@0003o0@00400000<2000@0000
0P000?l140030Q00E`0002@000020P007@0000D2000M00000P0000827@050Qd0E`0002@0000;0P00
4P0001/2000B00000P0000/24P0K0Q80E`0002@0000;0P00600001/2000H00000P0000/2600K0QP0
E`0002@0000R0P00700002<2000M00000P0002<2700R0Qd0E`0002@0000R0P007@0002<2000M0000
0P0002827@0S0Qd0E`0002@0000S0P007@0002@2000M00000P0002<27@0T0Qd0E`0002@0000S0P00
700002@2000M00000P0002@27@0S0Q`0E`0002@0000]0P002P000302000;00000P0003022P0]0P/0
E`0002@0000[0P002`0002d2000>00000P0002d22`0[0Ph0E`0002@0000Z0P003P0002/2000B0000
0P0002/23P0Z0Q80E`0002@0000Z0P004P0002X2000E00000P0002X24P0Z0QD0E`0002@0000Z0P00
5@0002/2000J00000P0002X25@0[0QX0E`0002@0000[0P006P0002d2000M00000P0002/26P0]0Qd0
E`0002@0000]0P007@000302000M00000P0002d27@0`0Qd0E`0002@0000`0P007@000342000M0000
0P0003027@0a0Qd0E`0002@0000a0P007@0003@2000M00000P0003427@0d0Qd0E`0002@0000d0P00
6P0003H2000M00000P0003@27@0f0QX0E`0002@0000f0P005@0003L2000J00000P0003H26P0g0QD0
E`0002@0000g0P004P0003L2000E00000P0003L25@0g0Q80E`0002@0000f0P003P0003L2000B0000
0P0003L24P0f0Ph0E`0002@0000d0P002`0003H2000>00000P0003H23P0d0P/0E`0002@0000a0P00
2P0003@2000;00000P0003@22`0a0PX0E`0002@0000`0P002P000342000:00000P0003422P0`0PX0
E`0002@0000^0P002P000302000;00000P0003022P0^0P/0E`0002@0000]0P002`0002h2000<0000
0P0002h22`0]0P`0E`0002@0000/0P00300002d2000>00000P0002d2300/0Ph0E`0002@0000[0P00
3P0002`2000B00000P0002`23P0[0Q80E`0002@0000[0P004P0002/2000E00000P0002/24P0[0QD0
E`0002@0000[0P005@0002`2000J00000P0002/25@0/0QX0E`0002@0000/0P006P0002d2000L0000
0P0002`26P0]0Q`0E`0002@0000]0P00700002h2000M00000P0002d2700^0Qd0E`0002@0000^0P00
7@000302000M00000P0002h27@0`0Qd0E`0002@0000a0P007@0003<2000M00000P0003427@0c0Qd0
E`0002@0000c0P00700003@2000M00000P0003<27@0d0Q`0E`0002@0000d0P006P0003D2000L0000
0P0003@2700e0QX0E`0002@0000e0P005@0003H2000J00000P0003D26P0f0QD0E`0002@0000f0P00
4P0003H2000E00000P0003H25@0f0Q80E`0002@0000e0P003P0003H2000B00000P0003H24P0e0Ph0
E`0002@0000d0P00300003D2000>00000P0003D23P0d0P`0E`0002@0000c0P002`0003@2000<0000
0P0003@2300c0P/0E`0002@0000a0P002P0003<2000;00000P0003<22`0a0PX0E`0002@0000n0P00
3@000402000>00000P0003h23P100Pd0E`0002@000100P002P000482000=00000P0004023@120PX0
E`0002@000120P002P000482000M00000P0004822P120Qd0E`0002@000110P002`000442000M0000
0P0004422`110Qd0E`0002@0000n0P007@0004H2000M00000P0003h27@160Qd0E`0002@0001I0P00
4P0005T2000C00000P0005T24P1I0Q<0E`0002@0001H0P004`0005T2000C00000P0005T24`1H0Q<0
E`0002@0001H0P004P0005P2000C00000P0005P24`1H0Q80E`0002@0001H0P004@0005T2000B0000
0P0005P24P1I0Q40E`0002@0001I0P00400005/2000A00000P0005T24@1K0Q00E`0002@0001K0P00
400005h2000@00000P0005/2401N0Q00E`0002@0001N0P0040000602000A00000P0005h2401P0Q40
E`0002@0001P0P004@000642000B00000P0006024@1Q0Q80E`0002@0001Q0P004P000682000D0000
0P0006424P1R0Q@0E`0002@0001R0P0050000682000K00000P000682501R0Q/0E`0002@0001R0P00
6`0006<2000M00000P0006826`1S0Qd0E`0002@0001S0P007@0006@2000M00000P0006<27@1T0Qd0
E`0002@0001Q0P004P000642000K00000P0006424P1Q0Q/0E`0002@0001Q0P006`000682000M0000
0P0006426`1R0Qd0E`0002@0001R0P007@0006@2000M00000P0006827@1T0Qd0E`0002@0001T0P00
7@0006D2000M00000P0006@27@1U0Qd0E`0002@0001P0P0050000642000E00000P000642501P0QD0
E`0002@0001K0P005@000602000F00000P0006025@1K0QH0E`0002@0001H0P005P0005/2000G0000
0P0005/25P1H0QL0E`0002@0001G0P005`0005P2000I00000P0005P25`1G0QT0E`0002@0001G0P00
6@0005L2000K00000P0005L26@1G0Q/0E`0002@0001G0P006`0005P2000M00000P0005L26`1H0Qd0
E`0002@0001H0P007@0005/2000M00000P0005P27@1K0Qd0E`0002@0001K0P007@0005d2000M0000
0P0005/27@1M0Qd0E`0002@0001M0P007@0005l2000M00000P0005d27@1O0Qd0E`0002@0001O0P00
6`000642000M00000P0005l27@1Q0Q/0E`0002@0001I0P005P0005/2000G00000P0005/25P1I0QL0
E`0002@0001H0P005`0005T2000I00000P0005T25`1H0QT0E`0002@0001H0P006@0005P2000K0000
0P0005P26@1H0Q/0E`0002@0001H0P006`0005T2000M00000P0005P26`1I0Qd0E`0002@0001I0P00
7@0005/2000M00000P0005T27@1K0Qd0E`0002@0001[0P00400006/2000M00000P0006/2401[0Qd0
E`0002@0001/0P00400006`2000M00000P0006`2401/0Qd0E`0002@0001/0P004@0006h2000C0000
0P0006`24`1^0Q40E`0002@0001^0P0040000702000A00000P0006h24@1`0Q00E`0002@0001`0P00
40000782000@00000P000702401b0Q00E`0002@0001b0P00400007D2000A00000P000782401e0Q40
E`0002@0001e0P004@0007H2000C00000P0007D24@1f0Q<0E`0002@0001f0P004`0007H2000M0000
0P0007H24`1f0Qd0E`0002@0001b0P00400007@2000A00000P000782401d0Q40E`0002@0001d0P00
4@0007D2000C00000P0007@24@1e0Q<0E`0002@0001e0P004`0007D2000M00000P0007D24`1e0Qd0
E`0002@0001X0P00400006`2000@00000P0006P2401/0Q00E`0002@0001X0P007@0006l2000M0000
0P0006P27@1_0Qd0E`0002@0001b0P007@0007P2000M00000P0007827@1h0Qd0E`0002@000280P00
2P0008P2000M00000P0008P22P280Qd0E`0002@000290P002P0008T2000M00000P0008T22P290Qd0
E`0002@000260P004@0008P2000C00000P0008P24`260Q40E`0002@000240P00400008H2000A0000
0P0008H24@240Q00E`0002@000230P00400008@2000@00000P0008@240230Q00E`0002@000200P00
400008<2000A00000P0008<240200Q40E`0002@0001n0P004@000802000C00000P0008024@1n0Q<0
E`0002@0001m0P004`0007h2000F00000P0007h24`1m0QH0E`0002@0001m0P005P0007d2000H0000
0P0007d25P1m0QP0E`0002@0001m0P00600007h2000K00000P0007d2601n0Q/0E`0002@0001n0P00
6`000802000M00000P0007h26`200Qd0E`0002@000200P007@0008<2000M00000P0008027@230Qd0
E`0002@000230P007@0008@2000M00000P0008<27@240Qd0E`0002@000240P007@0008H2000M0000
0P0008@27@260Qd0E`0002@000260P006`0008P2000M00000P0008H27@280Q/0E`0002@000210P00
400008<2000A00000P0008<240210Q40E`0002@0001o0P004@000842000C00000P0008424@1o0Q<0
E`0002@0001n0P004`0007l2000F00000P0007l24`1n0QH0E`0002@0001n0P005P0007h2000H0000
0P0007h25P1n0QP0E`0002@0001n0P00600007l2000K00000P0007h2601o0Q/0E`0002@0001o0P00
6`000842000M00000P0007l26`210Qd0E`0002@000210P007@0008<2000M00000P0008427@230Qd0
E`0002@000250P002P0008T2000:00000P0008D22P290PX0E`0002@000280P007@0008/2000M0000
0P0008P27@2;0Qd0E`0002@0002S0P004`000:@2000D00000P000:@24`2S0Q@0E`0002@0002S0P00
50000:@2000E00000P000:<2502T0QD0E`0002@0002T0P0050000:D2000E00000P000:@25@2U0Q@0
E`0002@0002U0P004`000:D2000D00000P000:D2502U0Q<0E`0002@0002S0P004@000:D2000C0000
0P000:D24`2S0Q40E`0002@0002Q0P0040000:<2000A00000P000:<24@2Q0Q00E`0002@0002O0P00
40000:42000@00000P000:42402O0Q00E`0002@0002L0P00400009l2000A00000P0009l2402L0Q40
E`0002@0002J0P004@0009`2000C00000P0009`24@2J0Q<0E`0002@0002I0P004`0009X2000F0000
0P0009X24`2I0QH0E`0002@0002I0P005P0009T2000H00000P0009T25P2I0QP0E`0002@0002I0P00
600009X2000K00000P0009T2602J0Q/0E`0002@0002J0P006`0009`2000M00000P0009X26`2L0Qd0
E`0002@0002L0P007@0009l2000M00000P0009`27@2O0Qd0E`0002@0002O0P007@000:02000M0000
0P0009l27@2P0Qd0E`0002@0002P0P007@000:<2000M00000P000:027@2S0Qd0E`0002@0002S0P00
6`000:D2000M00000P000:<27@2U0Q/0E`0002@0002M0P00400009l2000A00000P0009l2402M0Q40
E`0002@0002K0P004@0009d2000C00000P0009d24@2K0Q<0E`0002@0002J0P004`0009/2000F0000
0P0009/24`2J0QH0E`0002@0002J0P005P0009X2000H00000P0009X25P2J0QP0E`0002@0002J0P00
600009/2000K00000P0009X2602K0Q/0E`0002@0002K0P006`0009d2000M00000P0009/26`2M0Qd0
E`0002@0002M0P007@0009l2000M00000P0009d27@2O0Qd0E`0002@0002[0P004P000;/2000B0000
0P000:/24P2k0Q80E`0002@0002[0P0060000;/2000H00000P000:/2602k0QP0E`0002@000350P00
2P000<P2000;00000P000<P22P350P/0E`0002@000330P002`000<D2000>00000P000<D22`330Ph0
E`0002@000320P003P000<<2000B00000P000<<23P320Q80E`0002@000320P004P000<82000E0000
0P000<824P320QD0E`0002@000320P005@000<<2000J00000P000<825@330QX0E`0002@000330P00
6P000<D2000M00000P000<<26P350Qd0E`0002@000350P007@000<P2000M00000P000<D27@380Qd0
E`0002@000380P007@000<T2000M00000P000<P27@390Qd0E`0002@000390P007@000<`2000M0000
0P000<T27@3<0Qd0E`0002@0003<0P006P000<h2000M00000P000<`27@3>0QX0E`0002@0003>0P00
5@000<l2000J00000P000<h26P3?0QD0E`0002@0003?0P004P000<l2000E00000P000<l25@3?0Q80
E`0002@0003>0P003P000<l2000B00000P000<l24P3>0Ph0E`0002@0003<0P002`000<h2000>0000
0P000<h23P3<0P/0E`0002@000390P002P000<`2000;00000P000<`22`390PX0E`0002@000380P00
2P000<T2000:00000P000<T22P380PX0E`0002@000360P002P000<P2000;00000P000<P22P360P/0
E`0002@000350P002`000<H2000<00000P000<H22`350P`0E`0002@000340P0030000<D2000>0000
0P000<D230340Ph0E`0002@000330P003P000<@2000B00000P000<@23P330Q80E`0002@000330P00
4P000<<2000E00000P000<<24P330QD0E`0002@000330P005@000<@2000J00000P000<<25@340QX0
E`0002@000340P006P000<D2000L00000P000<@26P350Q`0E`0002@000350P0070000<H2000M0000
0P000<D270360Qd0E`0002@000360P007@000<P2000M00000P000<H27@380Qd0E`0002@000390P00
7@000</2000M00000P000<T27@3;0Qd0E`0002@0003;0P0070000<`2000M00000P000</27@3<0Q`0
E`0002@0003<0P006P000<d2000L00000P000<`2703=0QX0E`0002@0003=0P005@000<h2000J0000
0P000<d26P3>0QD0E`0002@0003>0P004P000<h2000E00000P000<h25@3>0Q80E`0002@0003=0P00
3P000<h2000B00000P000<h24P3=0Ph0E`0002@0003<0P0030000<d2000>00000P000<d23P3<0P`0
E`0002@0003;0P002`000<`2000<00000P000<`2303;0P/0E`0002@000390P002P000</2000;0000
0P000</22`390PX0E`0002@0003D0P0070000=D2000M00000P000=D2703D0Qd0E`0002@0003D0P00
7@000=D2000M00000P000=@27@3E0Qd0E`0002@0003E0P007@000=H2000M00000P000=D27@3F0Qd0
E`0002@0003E0P0070000=H2000M00000P000=H27@3E0Q`0E`0002@0003O0P002P000>82000;0000
0P000>822P3O0P/0E`0002@0003M0P002`000=l2000>00000P000=l22`3M0Ph0E`0002@0003L0P00
3P000=d2000B00000P000=d23P3L0Q80E`0002@0003L0P004P000=`2000E00000P000=`24P3L0QD0
E`0002@0003L0P005@000=d2000J00000P000=`25@3M0QX0E`0002@0003M0P006P000=l2000M0000
0P000=d26P3O0Qd0E`0002@0003O0P007@000>82000M00000P000=l27@3R0Qd0E`0002@0003R0P00
7@000><2000M00000P000>827@3S0Qd0E`0002@0003S0P007@000>H2000M00000P000><27@3V0Qd0
E`0002@0003V0P006P000>P2000M00000P000>H27@3X0QX0E`0002@0003X0P005@000>T2000J0000
0P000>P26P3Y0QD0E`0002@0003Y0P004P000>T2000E00000P000>T25@3Y0Q80E`0002@0003X0P00
3P000>T2000B00000P000>T24P3X0Ph0E`0002@0003V0P002`000>P2000>00000P000>P23P3V0P/0
E`0002@0003S0P002P000>H2000;00000P000>H22`3S0PX0E`0002@0003R0P002P000><2000:0000
0P000><22P3R0PX0E`0002@0003P0P002P000>82000;00000P000>822P3P0P/0E`0002@0003O0P00
2`000>02000<00000P000>022`3O0P`0E`0002@0003N0P0030000=l2000>00000P000=l2303N0Ph0
E`0002@0003M0P003P000=h2000B00000P000=h23P3M0Q80E`0002@0003M0P004P000=d2000E0000
0P000=d24P3M0QD0E`0002@0003M0P005@000=h2000J00000P000=d25@3N0QX0E`0002@0003N0P00
6P000=l2000L00000P000=h26P3O0Q`0E`0002@0003O0P0070000>02000M00000P000=l2703P0Qd0
E`0002@0003P0P007@000>82000M00000P000>027@3R0Qd0E`0002@0003S0P007@000>D2000M0000
0P000><27@3U0Qd0E`0002@0003U0P0070000>H2000M00000P000>D27@3V0Q`0E`0002@0003V0P00
6P000>L2000L00000P000>H2703W0QX0E`0002@0003W0P005@000>P2000J00000P000>L26P3X0QD0
E`0002@0003X0P004P000>P2000E00000P000>P25@3X0Q80E`0002@0003W0P003P000>P2000B0000
0P000>P24P3W0Ph0E`0002@0003V0P0030000>L2000>00000P000>L23P3V0P`0E`0002@0003U0P00
2`000>H2000<00000P000>H2303U0P/0E`0002@0003S0P002P000>D2000;00000P000>D22`3S0PX0
E`0002@0003a0P003@000?<2000>00000P000?423P3c0Pd0E`0002@0003c0P002P000?D2000=0000
0P000?<23@3e0PX0E`0002@0003e0P002P000?D2000M00000P000?D22P3e0Qd0E`0002@0003d0P00
2`000?@2000M00000P000?@22`3d0Qd0E`0002@0003a0P007@000?T2000M00000P000?427@3i0Qd0
E`0002@000010`0040000083000A00000P00008340010a40E`0002@000010`004@000083000B0000
0P0000434@020a80E`0002@000020`004@0000<3000B00000P0000834P030a40E`0002@000020`00
400000<3000A00000P0000<34@020a00E`0002@000010`0070000083000M00000P00008370010ad0
E`0002@000010`007@000083000M00000P0000437@020ad0E`0002@000020`007@0000<3000M0000
0P0000837@030ad0E`0002@000020`00700000<3000M00000P0000<37@020a`0E`0002@0000E0`00
2P0001D3000M00000P0001D32P0E0ad0E`0002@0000F0`002P0001H3000M00000P0001H32P0F0ad0
E`0002@0000B0`002P0001d3000:00000P0001832P0M0`X0E`0002@0000M0`002P000203000;0000
0P0001d32P0P0`/0E`0002@0000P0`002`000243000<00000P0002032`0Q0``0E`0002@0000Q0`00
30000243000>00000P000243300Q0`h0E`0002@0000Q0`003P000243000@00000P0002433P0Q0a00
E`0002@0000Q0`0040000243000B00000P000243400Q0a80E`0002@0000P0`004P000243000C0000
0P0002434P0P0a<0E`0002@0000M0`004`000203000D00000P0002034`0M0a@0E`0002@0000F0`00
500001d3000D00000P0001d3500F0a@0E`0002@0000M0`002P0001l3000;00000P0001d32P0O0`/0
E`0002@0000O0`002`000203000<00000P0001l32`0P0``0E`0002@0000P0`0030000243000>0000
0P000203300Q0`h0E`0002@0000Q0`003P000243000@00000P0002433P0Q0a00E`0002@0000P0`00
40000243000B00000P000243400P0a80E`0002@0000O0`004P000203000C00000P0002034P0O0a<0
E`0002@0000M0`004`0001l3000D00000P0001l34`0M0a@0E`0002@0000B0`007@0001T3000M0000
0P0001837@0I0ad0E`0002@0000Y0`00400002`3000A00000P0002`3400Y0a40E`0002@0000W0`00
4@0002T3000C00000P0002T34@0W0a<0E`0002@0000V0`004`0002L3000F00000P0002L34`0V0aH0
E`0002@0000V0`005P0002H3000H00000P0002H35P0V0aP0E`0002@0000V0`00600002L3000K0000
0P0002H3600W0a/0E`0002@0000W0`006`0002T3000M00000P0002L36`0Y0ad0E`0002@0000Y0`00
7@0002`3000M00000P0002T37@0/0ad0E`0002@0000/0`007@0002d3000M00000P0002`37@0]0ad0
E`0002@0000]0`007@000303000M00000P0002d37@0`0ad0E`0002@0000`0`006`000383000M0000
0P0003037@0b0a/0E`0002@0000b0`00600003<3000K00000P0003836`0c0aP0E`0002@0000c0`00
5P0003<3000H00000P0003<3600c0aH0E`0002@0000b0`004`0003<3000F00000P0003<35P0b0a<0
E`0002@0000`0`004@000383000C00000P0003834`0`0a40E`0002@0000]0`0040000303000A0000
0P0003034@0]0a00E`0002@0000/0`00400002d3000@00000P0002d3400/0a00E`0002@0000Z0`00
400002`3000A00000P0002`3400Z0a40E`0002@0000X0`004@0002X3000C00000P0002X34@0X0a<0
E`0002@0000W0`004`0002P3000F00000P0002P34`0W0aH0E`0002@0000W0`005P0002L3000H0000
0P0002L35P0W0aP0E`0002@0000W0`00600002P3000K00000P0002L3600X0a/0E`0002@0000X0`00
6`0002X3000M00000P0002P36`0Z0ad0E`0002@0000Z0`007@0002`3000M00000P0002X37@0/0ad0
E`0002@0000]0`007@0002l3000M00000P0002d37@0_0ad0E`0002@0000_0`006`000343000M0000
0P0002l37@0a0a/0E`0002@0000a0`0060000383000K00000P0003436`0b0aP0E`0002@0000b0`00
5P000383000H00000P000383600b0aH0E`0002@0000a0`004`000383000F00000P0003835P0a0a<0
E`0002@0000_0`004@000343000C00000P0003434`0_0a40E`0002@0000]0`00400002l3000A0000
0P0002l34@0]0a00E`0002@000110`0040000483000B00000P0004434P120a00E`0002@000120`00
40000483000D00000P00048340120a@0E`0002@000110`004P000483000D00000P00048350110a80
E`0002@000100`004@000443000B00000P0004434P100a40E`0002@0000n0`0040000403000A0000
0P0004034@0n0a00E`0002@0000k0`00400003h3000@00000P0003h3400k0a00E`0002@0000i0`00
400003/3000A00000P0003/3400i0a40E`0002@0000h0`004@0003T3000B00000P0003T34@0h0a80
E`0002@0000h0`004P0003P3000D00000P0003P34P0h0a@0E`0002@0000h0`00500003T3000E0000
0P0003P3500i0aD0E`0002@0000i0`005@0003/3000F00000P0003T35@0k0aH0E`0002@0000k0`00
5P0003l3000H00000P0003/35P0o0aP0E`0002@0000o0`0060000443000I00000P0003l360110aT0
E`0002@000110`006@000483000J00000P0004436@120aX0E`0002@0000h0`004`0003T3000D0000
0P0003P34`0i0a@0E`0002@0000i0`00500003/3000E00000P0003T3500k0aD0E`0002@0000k0`00
5@0003l3000G00000P0003/35@0o0aL0E`0002@0000o0`005`000443000H00000P0003l35`110aP0
E`0002@000110`0060000483000I00000P00044360120aT0E`0002@000120`006@000483000L0000
0P0004836@120a`0E`0002@000110`0070000483000M00000P00048370110ad0E`0002@0000o0`00
7@000443000M00000P0004437@0o0ad0E`0002@0000l0`007@0003l3000M00000P0003l37@0l0ad0
E`0002@0000j0`007@0003`3000M00000P0003`37@0j0ad0E`0002@0000i0`00700003X3000M0000
0P0003X37@0i0a`0E`0002@0000h0`006P0003T3000L00000P0003T3700h0aX0E`0002@0000h0`00
6P0003P3000M00000P0003P36P0h0ad0E`0002@0000h0`00700003T3000M00000P0003P37@0i0a`0
E`0002@000190`002P0004T3000J00000P0004T32P190aX0E`0002@000190`006P0004X3000M0000
0P0004T36P1:0ad0E`0002@0001:0`007@0004`3000M00000P0004X37@1<0ad0E`0002@0001<0`00
7@0004d3000M00000P0004`37@1=0ad0E`0002@0001=0`007@0004l3000M00000P0004d37@1?0ad0
E`0002@0001?0`006`000503000M00000P0004l37@1@0a/0E`0002@0001:0`002P0004X3000J0000
0P0004X32P1:0aX0E`0002@0001:0`006P0004/3000M00000P0004X36P1;0ad0E`0002@0001;0`00
7@0004`3000M00000P0004/37@1<0ad0E`0002@000160`00400004d3000@00000P0004H3401=0a00
E`0002@0001F0`005P000643000F00000P0005H35P1Q0aH0E`0002@0001Q0`0050000643000F0000
0P0006435P1Q0a@0E`0002@0001P0`004P000643000D00000P000643501P0a80E`0002@0001O0`00
4@000603000B00000P0006034P1O0a40E`0002@0001M0`00400005l3000A00000P0005l34@1M0a00
E`0002@0001K0`00400005d3000@00000P0005d3401K0a00E`0002@0001H0`00400005/3000A0000
0P0005/3401H0a40E`0002@0001F0`004@0005P3000C00000P0005P34@1F0a<0E`0002@0001E0`00
4`0005H3000F00000P0005H34`1E0aH0E`0002@0001E0`005P0005D3000H00000P0005D35P1E0aP0
E`0002@0001E0`00600005H3000K00000P0005D3601F0a/0E`0002@0001F0`006`0005P3000M0000
0P0005H36`1H0ad0E`0002@0001H0`007@0005/3000M00000P0005P37@1K0ad0E`0002@0001K0`00
7@0005`3000M00000P0005/37@1L0ad0E`0002@0001L0`007@0005l3000M00000P0005`37@1O0ad0
E`0002@0001O0`006`000643000M00000P0005l37@1Q0a/0E`0002@0001P0`004`000603000F0000
0P0006035P1P0a<0E`0002@0001O0`004@000603000C00000P0006034`1O0a40E`0002@0001I0`00
400005/3000A00000P0005/3401I0a40E`0002@0001G0`004@0005T3000C00000P0005T34@1G0a<0
E`0002@0001F0`004`0005L3000F00000P0005L34`1F0aH0E`0002@0001F0`005P0005H3000H0000
0P0005H35P1F0aP0E`0002@0001F0`00600005L3000K00000P0005H3601G0a/0E`0002@0001G0`00
6`0005T3000M00000P0005L36`1I0ad0E`0002@0001I0`007@0005/3000M00000P0005T37@1K0ad0
E`0002@0001X0`00400006P3000M00000P0006P3401X0ad0E`0002@0001Y0`00400006T3000M0000
0P0006T3401Y0ad0E`0002@0001Y0`004`0006X3000F00000P0006T35P1Z0a<0E`0002@0001Z0`00
4@0006`3000C00000P0006X34`1/0a40E`0002@0001/0`00400006d3000A00000P0006`34@1]0a00
E`0002@0001]0`0040000703000@00000P0006d3401`0a00E`0002@0001`0`0040000743000A0000
0P000703401a0a40E`0002@0001a0`004@000743000B00000P0007434@1a0a80E`0002@0001`0`00
4P000743000C00000P0007434P1`0a<0E`0002@0001_0`004P000703000C00000P0007034`1_0a80
E`0002@0001_0`004@000703000B00000P0006l34P1`0a40E`0002@0001U0`00400006T3000@0000
0P0006D3401Y0a00E`0002@0001U0`007@0006`3000M00000P0006D37@1/0ad0E`0002@0001f0`00
2P0007L3000;00000P0007L32P1f0`/0E`0002@0001f0`002`0007L3000<00000P0007H32`1g0``0
E`0002@0001g0`002`0007P3000<00000P0007L3301h0`/0E`0002@0001g0`002P0007P3000;0000
0P0007P32`1g0`X0E`0002@0001g0`00400007L3000M00000P0007L3401g0ad0E`0002@0001h0`00
400007P3000M00000P0007P3401h0ad0E`0002@0001d0`00400007P3000@00000P0007@3401h0a00
E`0002@0001d0`007@0007/3000M00000P0007@37@1k0ad0E`0002@000220`00400008D3000A0000
0P0008D340220a40E`0002@000200`004@000883000C00000P0008834@200a<0E`0002@0001o0`00
4`000803000F00000P0008034`1o0aH0E`0002@0001o0`005P0007l3000H00000P0007l35P1o0aP0
E`0002@0001o0`0060000803000K00000P0007l360200a/0E`0002@000200`006`000883000M0000
0P0008036`220ad0E`0002@000220`007@0008D3000M00000P0008837@250ad0E`0002@000250`00
7@0008H3000M00000P0008D37@260ad0E`0002@000260`007@0008T3000M00000P0008H37@290ad0
E`0002@000290`006`0008/3000M00000P0008T37@2;0a/0E`0002@0002;0`00600008`3000K0000
0P0008/36`2<0aP0E`0002@0002<0`005P0008`3000H00000P0008`3602<0aH0E`0002@0002;0`00
4`0008`3000F00000P0008`35P2;0a<0E`0002@000290`004@0008/3000C00000P0008/34`290a40
E`0002@000260`00400008T3000A00000P0008T34@260a00E`0002@000250`00400008H3000@0000
0P0008H340250a00E`0002@000230`00400008D3000A00000P0008D340230a40E`0002@000210`00
4@0008<3000C00000P0008<34@210a<0E`0002@000200`004`000843000F00000P0008434`200aH0
E`0002@000200`005P000803000H00000P0008035P200aP0E`0002@000200`0060000843000K0000
0P00080360210a/0E`0002@000210`006`0008<3000M00000P0008436`230ad0E`0002@000230`00
7@0008D3000M00000P0008<37@250ad0E`0002@000260`007@0008P3000M00000P0008H37@280ad0
E`0002@000280`006`0008X3000M00000P0008P37@2:0a/0E`0002@0002:0`00600008/3000K0000
0P0008X36`2;0aP0E`0002@0002;0`005P0008/3000H00000P0008/3602;0aH0E`0002@0002:0`00
4`0008/3000F00000P0008/35P2:0a<0E`0002@000280`004@0008X3000C00000P0008X34`280a40
E`0002@000260`00400008P3000A00000P0008P34@260a00E`0002@0002B0`0040000983000M0000
0P000983402B0ad0E`0002@0002C0`00400009<3000M00000P0009<3402C0ad0E`0002@0002C0`00
4`0009@3000F00000P0009<35P2D0a<0E`0002@0002D0`004@0009H3000C00000P0009@34`2F0a40
E`0002@0002F0`00400009L3000A00000P0009H34@2G0a00E`0002@0002G0`00400009X3000@0000
0P0009L3402J0a00E`0002@0002J0`00400009/3000A00000P0009X3402K0a40E`0002@0002K0`00
4@0009/3000B00000P0009/34@2K0a80E`0002@0002J0`004P0009/3000C00000P0009/34P2J0a<0
E`0002@0002I0`004P0009X3000C00000P0009X34`2I0a80E`0002@0002I0`004@0009X3000B0000
0P0009T34P2J0a40E`0002@0002?0`00400009<3000@00000P0008l3402C0a00E`0002@0002?0`00
7@0009H3000M00000P0008l37@2F0ad0E`0002@0002`0`002`000;03000<00000P000;032`2`0``0
E`0002@0002`0`0030000;03000=00000P000;03302`0`d0E`0002@0002`0`0030000;43000=0000
0P000;033@2a0``0E`0002@0002a0`002`000;43000<00000P000;43302a0`/0E`0002@0002`0`00
2P000;43000;00000P000;432`2`0`X0E`0002@0002_0`002P000;03000:00000P000;032P2_0`X0
E`0002@0002]0`002P000:l3000;00000P000:l32P2]0`/0E`0002@0002/0`002`000:d3000=0000
0P000:d32`2/0`d0E`0002@0002/0`003@000:`3000M00000P000:`33@2/0ad0E`0002@0002^0`00
2P000:l3000;00000P000:l32P2^0`/0E`0002@0002]0`002`000:h3000=00000P000:h32`2]0`d0
E`0002@0002]0`003@000:d3000M00000P000:d33@2]0ad0E`0002@0002Y0`0040000;03000@0000
0P000:T3402`0a00E`0002@0002Y0`007@000;03000M00000P000:T37@2`0ad0E`0002@0002h0`00
40000;/3000A00000P000;/3402h0a40E`0002@0002f0`004@000;P3000C00000P000;P34@2f0a<0
E`0002@0002e0`004`000;H3000F00000P000;H34`2e0aH0E`0002@0002e0`005P000;D3000H0000
0P000;D35P2e0aP0E`0002@0002e0`0060000;H3000K00000P000;D3602f0a/0E`0002@0002f0`00
6`000;P3000M00000P000;H36`2h0ad0E`0002@0002h0`007@000;/3000M00000P000;P37@2k0ad0
E`0002@0002k0`007@000;`3000M00000P000;/37@2l0ad0E`0002@0002l0`007@000;l3000M0000
0P000;`37@2o0ad0E`0002@0002o0`006`000<43000M00000P000;l37@310a/0E`0002@000310`00
60000<83000K00000P000<436`320aP0E`0002@000320`005P000<83000H00000P000<8360320aH0
E`0002@000310`004`000<83000F00000P000<835P310a<0E`0002@0002o0`004@000<43000C0000
0P000<434`2o0a40E`0002@0002l0`0040000;l3000A00000P000;l34@2l0a00E`0002@0002k0`00
40000;`3000@00000P000;`3402k0a00E`0002@0002i0`0040000;/3000A00000P000;/3402i0a40
E`0002@0002g0`004@000;T3000C00000P000;T34@2g0a<0E`0002@0002f0`004`000;L3000F0000
0P000;L34`2f0aH0E`0002@0002f0`005P000;H3000H00000P000;H35P2f0aP0E`0002@0002f0`00
60000;L3000K00000P000;H3602g0a/0E`0002@0002g0`006`000;T3000M00000P000;L36`2i0ad0
E`0002@0002i0`007@000;/3000M00000P000;T37@2k0ad0E`0002@0002l0`007@000;h3000M0000
0P000;`37@2n0ad0E`0002@0002n0`006`000<03000M00000P000;h37@300a/0E`0002@000300`00
60000<43000K00000P000<036`310aP0E`0002@000310`005P000<43000H00000P000<4360310aH0
E`0002@000300`004`000<43000F00000P000<435P300a<0E`0002@0002n0`004@000<03000C0000
0P000<034`2n0a40E`0002@0002l0`0040000;h3000A00000P000;h34@2l0a00E`0002@000390`00
40000<T3000M00000P000<T340390ad0E`0002@0003:0`0040000<X3000M00000P000<X3403:0ad0
E`0002@0003:0`004`000</3000F00000P000<X35P3;0a<0E`0002@0003;0`004@000<d3000C0000
0P000</34`3=0a40E`0002@0003=0`0040000<h3000A00000P000<d34@3>0a00E`0002@0003>0`00
40000=43000@00000P000<h3403A0a00E`0002@0003A0`0040000=83000A00000P000=43403B0a40
E`0002@0003B0`004@000=83000B00000P000=834@3B0a80E`0002@0003A0`004P000=83000C0000
0P000=834P3A0a<0E`0002@0003@0`004P000=43000C00000P000=434`3@0a80E`0002@0003@0`00
4@000=43000B00000P000=034P3A0a40E`0002@000360`0040000<X3000@00000P000<H3403:0a00
E`0002@000360`007@000<d3000M00000P000<H37@3=0ad0E`0002@0003/0`002P000>d3000=0000
0P000>`33@3]0`X0E`0002@0003]0`002P000>d3000?00000P000>d32P3]0`l0E`0002@0003/0`00
3@000>d3000?00000P000>d33`3/0`d0E`0002@0003Z0`002`000>`3000=00000P000>`33@3Z0`/0
E`0002@0003W0`002P000>X3000;00000P000>X32`3W0`X0E`0002@0003U0`002P000>L3000:0000
0P000>L32P3U0`X0E`0002@0003R0`002P000>D3000;00000P000>D32P3R0`/0E`0002@0003P0`00
2`000>83000=00000P000>832`3P0`d0E`0002@0003P0`003@000>03000?00000P000>033@3P0`l0
E`0002@0003P0`003`000>43000@00000P000>033`3Q0a00E`0002@0003Q0`0040000>83000A0000
0P000>43403R0a40E`0002@0003R0`004@000>@3000B00000P000>834@3T0a80E`0002@0003T0`00
4P000>T3000D00000P000>@34P3Y0a@0E`0002@0003Y0`0050000>/3000E00000P000>T3503[0aD0
E`0002@0003[0`005@000>d3000G00000P000>/35@3]0aL0E`0002@0003P0`003`000>83000@0000
0P000>033`3R0a00E`0002@0003R0`0040000>@3000A00000P000>83403T0a40E`0002@0003T0`00
4@000>T3000C00000P000>@34@3Y0a<0E`0002@0003Y0`004`000>/3000D00000P000>T34`3[0a@0
E`0002@0003[0`0050000>`3000E00000P000>/3503/0aD0E`0002@0003/0`005@000>d3000G0000
0P000>`35@3]0aL0E`0002@0003]0`005`000>d3000K00000P000>d35`3]0a/0E`0002@0003[0`00
6`000>d3000M00000P000>d36`3[0ad0E`0002@0003X0`007@000>/3000M00000P000>/37@3X0ad0
E`0002@0003V0`007@000>P3000M00000P000>P37@3V0ad0E`0002@0003S0`007@000>H3000M0000
0P000>H37@3S0ad0E`0002@0003Q0`006`000><3000M00000P000><37@3Q0a/0E`0002@0003P0`00
60000>43000K00000P000>436`3P0aP0E`0002@0003P0`0060000>03000M00000P000>03603P0ad0
E`0002@0003P0`006`000>43000M00000P000>037@3Q0a/0E`0002@0003d0`002P000?@3000M0000
0P000?@32P3d0ad0E`0002@0003e0`002P000?D3000M00000P000?D32P3e0ad0E`0002@0003a0`00
2P000?X3000:00000P000?432P3j0`X0E`0002@0003j0`002P000?d3000;00000P000?X32P3m0`/0
E`0002@0003m0`002`000?l3000=00000P000?d32`3o0`d0E`0002@0003o0`003@000004000?0000
0P000?l33@0010l0E`0002@0000010003`000004000A00000P0000043`001140E`0002@000001000
4@000004000F00000P0000044@0011H0E`0002@0000010005P000004000I00000P0000045P0011T0
E`0002@0003o0`006@000004000K00000P0000046@3o0a/0E`0002@0003m0`006`000?l3000M0000
0P000?l36`3m0ad0E`0002@0003j0`007@000?d3000M00000P000?d37@3j0ad0E`0002@0003a0`00
7@000?X3000M00000P000?X37@3a0ad0E`0002@0003j0`002P000?`3000;00000P000?X32P3l0`/0
E`0002@0003l0`002`000?h3000=00000P000?`32`3n0`d0E`0002@0003n0`003@000?l3000?0000
0P000?h33@3o0`l0E`0002@0003o0`003`000004000A00000P000?l33`001140E`0002@000001000
4@000004000F00000P0000044@0011H0E`0002@0003o0`005P000004000I00000P0000045P3o0aT0
E`0002@0003n0`006@000?l3000K00000P000?l36@3n0a/0E`0002@0003l0`006`000?h3000M0000
0P000?h36`3l0ad0E`0002@0003j0`007@000?`3000M00000P000?`37@3j0ad0E`0002@000081000
2P0000P4000H00000P0000P42P0811P0E`0002@000081000600000T4000K00000P0000P4600911/0
E`0002@0000910006`0000/4000M00000P0000T46`0;11d0E`0002@0000;10007@0000d4000M0000
0P0000/47@0=11d0E`0002@0000=10007@0000l4000M00000P0000d47@0?11d0E`0002@0000?1000
7@000184000M00000P0000l47@0B11d0E`0002@0000B10006`0001@4000M00000P0001847@0D11/0
E`0002@0000D1000600001@4000K00000P0001@46`0D11P0E`0002@0000D10002P0001@4000H0000
0P0001@4600D10X0E`0002@0000910002P0000T4000H00000P0000T42P0911P0E`0002@000091000
600000X4000K00000P0000T4600:11/0E`0002@0000:10006`0000`4000M00000P0000X46`0<11d0
E`0002@0000<10007@0000d4000M00000P0000`47@0=11d0E`0002@0000510002P0000`4000:0000
0P0000D42P0<10X0E`0002@0000B10002P0001L4000:00000P0001842P0G10X0E`0002@000250000
O0<008X0001l0`000P0008X0O0>507`3E`0002@000270000HP<008X0001R0`000P0008L0HP>:0683
E`0002@000270000B0<008X000180`000P0008L0B0>:04P3E`0002@000270000;P<008X0000^0`00
0P0008L0;P>:02h3E`0002@00027000050<008X0000D0`000P0008L050>:01@3E`0002@000270000
nP8008X0003j0P000P0008L0nP::0?X2E`0002@000270000h08008X0003P0P000P0008L0h0::0>02
E`0002@000270000aP8008X000360P000P0008L0aP::0<H2E`0002@000270000[08008X0002/0P00
0P0008L0[0::0:`2E`0002@000270000TP8008X0002B0P000P0008L0TP::0982E`0002@000250000
N08008X0001h0P000P0008X0N0:507P2E`0002@000270000GP8008X0001N0P000P0008L0GP::05h2
E`0002@000270000A08008X000140P000P0008L0A0::04@2E`0002@000270000:P8008X0000Z0P00
0P0008L0:P::02X2E`0002@000270000408008X0000@0P000P0008L040::0102E`0002@000270000
mP4008X0003f0@000P0008L0mP6:0?H1E`0002@000270000g04008X0003L0@000P0008L0g06:0=`1
E`0002@000270000`P4008X000320@000P0008L0`P6:0<81E`0002@000270000Z04008X0002X0@00
0P0008L0Z06:0:P1E`0002@000270000SP4008X0002>0@000P0008L0SP6:08h1E`0002@000250000
M04008X0001d0@000P0008X0M06507@1E`0002@000270000FP4008X0001J0@000P0008L0FP6:05X1
E`0002@000270000@04008X000100@000P0008L0@06:0401E`0002@0002700009P4008X0000V0@00
0P0008L09P6:02H1E`0002@000270000304008X0000<0@000P0008L0306:00`1E`0002@000270000
lP0008X0003b00000P0008L0lP2:0?80E`0002@000270000f00008X0003H00000P0008L0f02:0=P0
E`0002@000270000_P0008X0002n00000P0008L0_P2:0;h0E`0002@000270000Y00008X0002T0000
0P0008L0Y02:0:@0E`0002@000270000RP0008X0002:00000P0008L0RP2:08X0E`0002@000250000
L00008X0001`00000P0008X0L0250700E`0002@0000E0000AP0001D0001I00000P0001D0AP0E05T0
E`0002@0000F0000AP0001H0001I00000P0001H0AP0F05T0E`0002@0000B0000AP0001/000160000
0P000180AP0K04H0E`0002@0000K0000AP0001h0001700000P0001/0AP0N04L0E`0002@0000N0000
A`000200001900000P0001h0A`0P04T0E`0002@0000P0000B@000240001;00000P000200B@0Q04/0
E`0002@0000Q0000B`000240001=00000P000240B`0Q04d0E`0002@0000Q0000C@000240001B0000
0P000240C@0Q0580E`0002@0000Q0000DP000240001E00000P000240DP0Q05D0E`0002@0000P0000
E@000240001G00000P000240E@0P05L0E`0002@0000N0000E`000200001I00000P000200E`0N05T0
E`0002@0000K0000F@0001h0001I00000P0001h0F@0K05T0E`0002@0000B0000F@0001/0001I0000
0P0001/0F@0B05T0E`0002@0000K0000AP0001d0001700000P0001/0AP0M04L0E`0002@0000M0000
A`0001l0001900000P0001d0A`0O04T0E`0002@0000O0000B@000200001;00000P0001l0B@0P04/0
E`0002@0000P0000B`000240001=00000P000200B`0Q04d0E`0002@0000Q0000C@000240001B0000
0P000240C@0Q0580E`0002@0000P0000DP000240001E00000P000240DP0P05D0E`0002@0000O0000
E@000200001G00000P000200E@0O05L0E`0002@0000M0000E`0001l0001I00000P0001l0E`0M05T0
E`0002@0000K0000F@0001d0001I00000P0001d0F@0K05T0E`0002@0000W0000DP000380001B0000
0P0002L0DP0b0580E`0002@0000b0000D0000380001B00000P000380DP0b0500E`0002@0000a0000
CP000380001@00000P000380D00a04h0E`0002@0000`0000C@000340001>00000P000340CP0`04d0
E`0002@0000^0000C0000300001=00000P000300C@0^04`0E`0002@0000/0000C00002h0001<0000
0P0002h0C00/04`0E`0002@0000Y0000C00002`0001=00000P0002`0C00Y04d0E`0002@0000W0000
C@0002T0001?00000P0002T0C@0W04l0E`0002@0000V0000C`0002L0001B00000P0002L0C`0V0580
E`0002@0000V0000DP0002H0001D00000P0002H0DP0V05@0E`0002@0000V0000E00002L0001G0000
0P0002H0E00W05L0E`0002@0000W0000E`0002T0001I00000P0002L0E`0Y05T0E`0002@0000Y0000
F@0002`0001I00000P0002T0F@0/05T0E`0002@0000/0000F@0002d0001I00000P0002`0F@0]05T0
E`0002@0000]0000F@000300001I00000P0002d0F@0`05T0E`0002@0000`0000E`000380001I0000
0P000300F@0b05L0E`0002@0000a0000C`000340001B00000P000340DP0a04l0E`0002@0000`0000
C@000340001?00000P000340C`0`04d0E`0002@0000Z0000C00002`0001=00000P0002`0C00Z04d0
E`0002@0000X0000C@0002X0001?00000P0002X0C@0X04l0E`0002@0000W0000C`0002P0001B0000
0P0002P0C`0W0580E`0002@0000W0000DP0002L0001D00000P0002L0DP0W05@0E`0002@0000W0000
E00002P0001G00000P0002L0E00X05L0E`0002@0000X0000E`0002X0001I00000P0002P0E`0Z05T0
E`0002@0000Z0000F@0002`0001I00000P0002X0F@0/05T0E`0002@0000i0000C00003T0001I0000
0P0003T0C00i05T0E`0002@0000j0000C00003X0001I00000P0003X0C00j05T0E`0002@0000j0000
C@0003`0001?00000P0003X0C`0l04d0E`0002@0000l0000C00003h0001=00000P0003`0C@0n04`0
E`0002@0000n0000C0000400001<00000P0003h0C01004`0E`0002@000100000C00004<0001=0000
0P000400C01304d0E`0002@000130000C@0004@0001?00000P0004<0C@1404l0E`0002@000140000
C`0004@0001I00000P0004@0C`1405T0E`0002@000100000C0000480001=00000P000400C01204d0
E`0002@000120000C@0004<0001?00000P000480C@1304l0E`0002@000130000C`0004<0001I0000
0P0004<0C`1305T0E`0002@0000f0000C00003X0001<00000P0003H0C00j04`0E`0002@0000f0000
F@0003d0001I00000P0003H0F@0m05T0E`0002@000100000F@0004H0001I00000P000400F@1605T0
E`0002@0001D0000C00005D0001>00000P0005@0CP1E04`0E`0002@0001E0000C00005D0001@0000
0P0005D0C01E0500E`0002@0001D0000CP0005D0001@00000P0005D0D01D04h0E`0002@0001C0000
C@0005@0001>00000P0005@0CP1C04d0E`0002@0001A0000C00005<0001=00000P0005<0C@1A04`0
E`0002@0001>0000C0000540001<00000P000540C01>04`0E`0002@0001<0000C00004h0001=0000
0P0004h0C01<04d0E`0002@0001;0000C@0004`0001>00000P0004`0C@1;04h0E`0002@0001;0000
CP0004/0001@00000P0004/0CP1;0500E`0002@0001;0000D00004`0001A00000P0004/0D01<0540
E`0002@0001<0000D@0004h0001B00000P0004`0D@1>0580E`0002@0001>0000DP000580001D0000
0P0004h0DP1B05@0E`0002@0001B0000E00005@0001E00000P000580E01D05D0E`0002@0001D0000
E@0005D0001F00000P0005@0E@1E05H0E`0002@0001;0000C`0004`0001@00000P0004/0C`1<0500
E`0002@0001<0000D00004h0001A00000P0004`0D01>0540E`0002@0001>0000D@000580001C0000
0P0004h0D@1B05<0E`0002@0001B0000D`0005@0001D00000P000580D`1D05@0E`0002@0001D0000
E00005D0001E00000P0005@0E01E05D0E`0002@0001E0000E@0005D0001H00000P0005D0E@1E05P0
E`0002@0001D0000F00005D0001I00000P0005D0F01D05T0E`0002@0001B0000F@0005@0001I0000
0P0005@0F@1B05T0E`0002@0001?0000F@000580001I00000P000580F@1?05T0E`0002@0001=0000
F@0004l0001I00000P0004l0F@1=05T0E`0002@0001<0000F00004d0001I00000P0004d0F@1<05P0
E`0002@0001;0000EP0004`0001H00000P0004`0F01;05H0E`0002@0001;0000EP0004/0001I0000
0P0004/0EP1;05T0E`0002@0001;0000F00004`0001I00000P0004/0F@1<05P0E`0002@0001K0000
AP0005`0001700000P0005`0AP1K04L0E`0002@0001K0000A`0005`0001800000P0005/0A`1L04P0
E`0002@0001L0000A`0005d0001800000P0005`0B01M04L0E`0002@0001L0000AP0005d000170000
0P0005d0A`1L04H0E`0002@0001L0000C00005`0001I00000P0005`0C01L05T0E`0002@0001M0000
C00005d0001I00000P0005d0C01M05T0E`0002@0001I0000C00005d0001<00000P0005T0C01M04`0
E`0002@0001I0000F@000600001I00000P0005T0F@1P05T0E`0002@0001V0000AP0006H0001F0000
0P0006H0AP1V05H0E`0002@0001V0000EP0006L0001I00000P0006H0EP1W05T0E`0002@0001W0000
F@0006T0001I00000P0006L0F@1Y05T0E`0002@0001Y0000F@0006X0001I00000P0006T0F@1Z05T0
E`0002@0001Z0000F@0006`0001I00000P0006X0F@1/05T0E`0002@0001/0000E`0006d0001I0000
0P0006`0F@1]05L0E`0002@0001W0000AP0006L0001F00000P0006L0AP1W05H0E`0002@0001W0000
EP0006P0001I00000P0006L0EP1X05T0E`0002@0001X0000F@0006T0001I00000P0006P0F@1Y05T0
E`0002@0001S0000C00006X0001<00000P0006<0C01Z04`0E`0002@0001b0000C00007L0001I0000
0P000780C01g05T0E`0002@0001c0000C00007L0001H00000P0007<0C01g05P0E`0002@0001g0000
C00007d0001I00000P0007d0C01g05T0E`0002@0001f0000F@0007L0001M00000P0007L0F@1f05d0
E`0002@0001d0000G@0007H0001O00000P0007H0G@1d05l0E`0002@0001b0000G`0007@0001P0000
0P0007@0G`1b0600E`0002@0001a0000H0000780001P00000P000780H01a0600E`0002@0001`0000
G`000740001P00000P000740H01`05l0E`0002@0001`0000GP000740001O00000P000700G`1a05h0
E`0002@0001a0000GP000780001O00000P000740GP1b05l0E`0002@0001`0000C00007H0001<0000
0P000700C01f04`0E`0002@0001i0000C00007l0001<00000P0007T0C01o04`0E`0002@0001c0000
L`<007H0001d0`000P0007H0L`=c07@3E`0002@0001a0000M0<007<0001g0`000P0007<0M0=a07L3
E`0002@0001`0000M`<00740001k0`000P000740M`=`07/3E`0002@0001`0000N`<00700001n0`00
0P000700N`=`07h3E`0002@0001`0000OP<0074000230`000P000700OP=a08<3E`0002@0001a0000
P`<007<000260`000P000740P`=c08H3E`0002@0001c0000QP<007H000260`000P0007<0QP=f08H3
E`0002@0001f0000QP<007L000260`000P0007H0QP=g08H3E`0002@0001g0000QP<007X000260`00
0P0007L0QP=j08H3E`0002@0001j0000P`<007`000260`000P0007X0QP=l08<3E`0002@0001l0000
OP<007d000230`000P0007`0P`=m07h3E`0002@0001m0000N`<007d0001n0`000P0007d0OP=m07/3
E`0002@0001l0000M`<007d0001k0`000P0007d0N`=l07L3E`0002@0001j0000M0<007`0001g0`00
0P0007`0M`=j07@3E`0002@0001g0000L`<007X0001d0`000P0007X0M0=g07<3E`0002@0001f0000
L`<007L0001c0`000P0007L0L`=f07<3E`0002@0001d0000L`<007H0001d0`000P0007H0L`=d07@3
E`0002@0001c0000M0<007@0001e0`000P0007@0M0=c07D3E`0002@0001b0000M@<007<0001g0`00
0P0007<0M@=b07L3E`0002@0001a0000M`<00780001k0`000P000780M`=a07/3E`0002@0001a0000
N`<00740001n0`000P000740N`=a07h3E`0002@0001a0000OP<0078000230`000P000740OP=b08<3
E`0002@0001b0000P`<007<000250`000P000780P`=c08D3E`0002@0001c0000Q@<007@000260`00
0P0007<0Q@=d08H3E`0002@0001d0000QP<007H000260`000P0007@0QP=f08H3E`0002@0001g0000
QP<007T000260`000P0007L0QP=i08H3E`0002@0001i0000Q@<007X000260`000P0007T0QP=j08D3
E`0002@0001j0000P`<007/000250`000P0007X0Q@=k08<3E`0002@0001k0000OP<007`000230`00
0P0007/0P`=l07h3E`0002@0001l0000N`<007`0001n0`000P0007`0OP=l07/3E`0002@0001k0000
M`<007`0001k0`000P0007`0N`=k07L3E`0002@0001j0000M@<007/0001g0`000P0007/0M`=j07D3
E`0002@0001i0000M0<007X0001e0`000P0007X0M@=i07@3E`0002@0001g0000L`<007T0001d0`00
0P0007T0M0=g07<3E`0002@0001@0000LP800580001c0P000P000500L`9B0782E`0002@0001B0000
K`8005@0001b0P000P000580LP9D06l2E`0002@0001D0000K`8005@000220P000P0005@0K`9D0882
E`0002@0001C0000L08005<000220P000P0005<0L09C0882E`0002@0001@0000PP8005P000220P00
0P000500PP9H0882E`0002@0001Q0000K`8006@0001`0P000P0006@0K`9Q0702E`0002@0001O0000
L0800640001c0P000P000640L09O07<2E`0002@0001N0000L`8005l0001g0P000P0005l0L`9N07L2
E`0002@0001N0000M`8005h0001j0P000P0005h0M`9N07X2E`0002@0001N0000NP8005l0001o0P00
0P0005h0NP9O07l2E`0002@0001O0000O`80064000220P000P0005l0O`9Q0882E`0002@0001Q0000
PP8006@000220P000P000640PP9T0882E`0002@0001T0000PP8006D000220P000P0006@0PP9U0882
E`0002@0001U0000PP8006P000220P000P0006D0PP9X0882E`0002@0001X0000O`8006X000220P00
0P0006P0PP9Z07l2E`0002@0001Z0000NP8006/0001o0P000P0006X0O`9[07X2E`0002@0001[0000
M`8006/0001j0P000P0006/0NP9[07L2E`0002@0001Z0000L`8006/0001g0P000P0006/0M`9Z07<2
E`0002@0001X0000L08006X0001c0P000P0006X0L`9X0702E`0002@0001U0000K`8006P0001`0P00
0P0006P0L09U06l2E`0002@0001T0000K`8006D0001_0P000P0006D0K`9T06l2E`0002@0001R0000
K`8006@0001`0P000P0006@0K`9R0702E`0002@0001Q0000L0800680001a0P000P000680L09Q0742
E`0002@0001P0000L@800640001c0P000P000640L@9P07<2E`0002@0001O0000L`800600001g0P00
0P000600L`9O07L2E`0002@0001O0000M`8005l0001j0P000P0005l0M`9O07X2E`0002@0001O0000
NP800600001o0P000P0005l0NP9P07l2E`0002@0001P0000O`80064000210P000P000600O`9Q0842
E`0002@0001Q0000P@80068000220P000P000640P@9R0882E`0002@0001R0000PP8006@000220P00
0P000680PP9T0882E`0002@0001U0000PP8006L000220P000P0006D0PP9W0882E`0002@0001W0000
P@8006P000220P000P0006L0PP9X0842E`0002@0001X0000O`8006T000210P000P0006P0P@9Y07l2
E`0002@0001Y0000NP8006X0001o0P000P0006T0O`9Z07X2E`0002@0001Z0000M`8006X0001j0P00
0P0006X0NP9Z07L2E`0002@0001Y0000L`8006X0001g0P000P0006X0M`9Y07<2E`0002@0001X0000
L@8006T0001c0P000P0006T0L`9X0742E`0002@0001W0000L08006P0001a0P000P0006P0L@9W0702
E`0002@0001U0000K`8006L0001`0P000P0006L0L09U06l2E`0002@0001c0000K`8007H0001`0P00
0P0007H0K`9c0702E`0002@0001a0000L08007<0001c0P000P0007<0L09a07<2E`0002@0001`0000
L`800740001g0P000P000740L`9`07L2E`0002@0001`0000M`800700001j0P000P000700M`9`07X2
E`0002@0001`0000NP800740001o0P000P000700NP9a07l2E`0002@0001a0000O`8007<000220P00
0P000740O`9c0882E`0002@0001c0000PP8007H000220P000P0007<0PP9f0882E`0002@0001f0000
PP8007L000220P000P0007H0PP9g0882E`0002@0001g0000PP8007X000220P000P0007L0PP9j0882
E`0002@0001j0000O`8007`000220P000P0007X0PP9l07l2E`0002@0001l0000NP8007d0001o0P00
0P0007`0O`9m07X2E`0002@0001m0000M`8007d0001j0P000P0007d0NP9m07L2E`0002@0001l0000
L`8007d0001g0P000P0007d0M`9l07<2E`0002@0001j0000L08007`0001c0P000P0007`0L`9j0702
E`0002@0001g0000K`8007X0001`0P000P0007X0L09g06l2E`0002@0001f0000K`8007L0001_0P00
0P0007L0K`9f06l2E`0002@0001d0000K`8007H0001`0P000P0007H0K`9d0702E`0002@0001c0000
L08007@0001a0P000P0007@0L09c0742E`0002@0001b0000L@8007<0001c0P000P0007<0L@9b07<2
E`0002@0001a0000L`800780001g0P000P000780L`9a07L2E`0002@0001a0000M`800740001j0P00
0P000740M`9a07X2E`0002@0001a0000NP800780001o0P000P000740NP9b07l2E`0002@0001b0000
O`8007<000210P000P000780O`9c0842E`0002@0001c0000P@8007@000220P000P0007<0P@9d0882
E`0002@0001d0000PP8007H000220P000P0007@0PP9f0882E`0002@0001g0000PP8007T000220P00
0P0007L0PP9i0882E`0002@0001i0000P@8007X000220P000P0007T0PP9j0842E`0002@0001j0000
O`8007/000210P000P0007X0P@9k07l2E`0002@0001k0000NP8007`0001o0P000P0007/0O`9l07X2
E`0002@0001l0000M`8007`0001j0P000P0007`0NP9l07L2E`0002@0001k0000L`8007`0001g0P00
0P0007`0M`9k07<2E`0002@0001j0000L@8007/0001c0P000P0007/0L`9j0742E`0002@0001i0000
L08007X0001a0P000P0007X0L@9i0702E`0002@0001g0000K`8007T0001`0P000P0007T0L09g06l2
E`0002@0001>0000KP4004l0001_0@000P0004h0KP5?06l1E`0002@0001>0000K`4004l0001_0@00
0P0004l0K`5>06l1E`0002@0001=0000K`4004h0001_0@000P0004h0K`5=06l1E`0002@0001=0000
KP4004d0001_0@000P0004d0K`5=06h1E`0002@0001=0000K04004h0001^0@000P0004d0KP5>06`1
E`0002@0001>0000J`4004l0001/0@000P0004h0K05?06/1E`0002@0001?0000JP400580001[0@00
0P0004l0J`5B06X1E`0002@0001B0000JP4005D0001Z0@000P000580JP5E06X1E`0002@0001E0000
JP4005P0001[0@000P0005D0JP5H06/1E`0002@0001H0000J`4005T0001/0@000P0005P0J`5I06`1
E`0002@0001I0000K04005X0001^0@000P0005T0K05J06h1E`0002@0001J0000KP4005X0001_0@00
0P0005X0KP5J06l1E`0002@0001I0000K`4005X0001a0@000P0005X0K`5I0741E`0002@0001F0000
L@4005T0001c0@000P0005T0L@5F07<1E`0002@0001B0000L`4005H0001e0@000P0005H0L`5B07D1
E`0002@0001@0000M@400580001f0@000P000580M@5@07H1E`0002@0001>0000MP400500001h0@00
0P000500MP5>07P1E`0002@0001=0000N04004h0001k0@000P0004h0N05=07/1E`0002@0001=0000
N`4004d0001m0@000P0004d0N`5=07d1E`0002@0001E0000JP4005L0001[0@000P0005D0JP5G06/1
E`0002@0001G0000J`4005P0001/0@000P0005L0J`5H06`1E`0002@0001H0000K04005T0001^0@00
0P0005P0K05I06h1E`0002@0001I0000KP4005T0001_0@000P0005T0KP5I06l1E`0002@0001H0000
K`4005T0001a0@000P0005T0K`5H0741E`0002@0001E0000L@4005P0001c0@000P0005P0L@5E07<1
E`0002@0001B0000L`4005D0001e0@000P0005D0L`5B07D1E`0002@0001=0000N`4004h0001l0@00
0P0004d0O05>07/1E`0002@0001>0000N`400500001k0@000P0004h0N`5@07/1E`0002@0001@0000
N`4005@0001m0@000P000500N`5D07d1E`0002@0001D0000O@4005L0001m0@000P0005@0O@5G07d1
E`0002@0001G0000O04005T0001m0@000P0005L0O@5I07`1E`0002@0001I0000N`4005X0001l0@00
0P0005T0O05J07/1E`0002@0001@0000N`4005@0001m0@000P000500N`5D07d1E`0002@0001D0000
O@4005P0001m0@000P0005@0O@5H07d1E`0002@0001H0000O@4005T0001m0@000P0005P0O@5I07d1
E`0002@0001I0000N`4005X0001m0@000P0005T0O@5J07/1E`0002@0001J0000N@4005X0001k0@00
0P0005X0N`5J07T1E`0002@0001Q0000JP4006@0001[0@000P0006@0JP5Q06/1E`0002@0001O0000
J`400640001^0@000P000640J`5O06h1E`0002@0001N0000KP4005l0001b0@000P0005l0KP5N0781
E`0002@0001N0000LP4005h0001e0@000P0005h0LP5N07D1E`0002@0001N0000M@4005l0001j0@00
0P0005h0M@5O07X1E`0002@0001O0000NP400640001m0@000P0005l0NP5Q07d1E`0002@0001Q0000
O@4006@0001m0@000P000640O@5T07d1E`0002@0001T0000O@4006D0001m0@000P0006@0O@5U07d1
E`0002@0001U0000O@4006P0001m0@000P0006D0O@5X07d1E`0002@0001X0000NP4006X0001m0@00
0P0006P0O@5Z07X1E`0002@0001Z0000M@4006/0001j0@000P0006X0NP5[07D1E`0002@0001[0000
LP4006/0001e0@000P0006/0M@5[0781E`0002@0001Z0000KP4006/0001b0@000P0006/0LP5Z06h1
E`0002@0001X0000J`4006X0001^0@000P0006X0KP5X06/1E`0002@0001U0000JP4006P0001[0@00
0P0006P0J`5U06X1E`0002@0001T0000JP4006D0001Z0@000P0006D0JP5T06X1E`0002@0001R0000
JP4006@0001[0@000P0006@0JP5R06/1E`0002@0001Q0000J`400680001/0@000P000680J`5Q06`1
E`0002@0001P0000K0400640001^0@000P000640K05P06h1E`0002@0001O0000KP400600001b0@00
0P000600KP5O0781E`0002@0001O0000LP4005l0001e0@000P0005l0LP5O07D1E`0002@0001O0000
M@400600001j0@000P0005l0M@5P07X1E`0002@0001P0000NP400640001l0@000P000600NP5Q07`1
E`0002@0001Q0000O0400680001m0@000P000640O05R07d1E`0002@0001R0000O@4006@0001m0@00
0P000680O@5T07d1E`0002@0001U0000O@4006L0001m0@000P0006D0O@5W07d1E`0002@0001W0000
O04006P0001m0@000P0006L0O@5X07`1E`0002@0001X0000NP4006T0001l0@000P0006P0O05Y07X1
E`0002@0001Y0000M@4006X0001j0@000P0006T0NP5Z07D1E`0002@0001Z0000LP4006X0001e0@00
0P0006X0M@5Z0781E`0002@0001Y0000KP4006X0001b0@000P0006X0LP5Y06h1E`0002@0001X0000
K04006T0001^0@000P0006T0KP5X06`1E`0002@0001W0000J`4006P0001/0@000P0006P0K05W06/1
E`0002@0001U0000JP4006L0001[0@000P0006L0J`5U06X1E`0002@0001c0000JP4007H0001[0@00
0P0007H0JP5c06/1E`0002@0001a0000J`4007<0001^0@000P0007<0J`5a06h1E`0002@0001`0000
KP400740001b0@000P000740KP5`0781E`0002@0001`0000LP400700001e0@000P000700LP5`07D1
E`0002@0001`0000M@400740001j0@000P000700M@5a07X1E`0002@0001a0000NP4007<0001m0@00
0P000740NP5c07d1E`0002@0001c0000O@4007H0001m0@000P0007<0O@5f07d1E`0002@0001f0000
O@4007L0001m0@000P0007H0O@5g07d1E`0002@0001g0000O@4007X0001m0@000P0007L0O@5j07d1
E`0002@0001j0000NP4007`0001m0@000P0007X0O@5l07X1E`0002@0001l0000M@4007d0001j0@00
0P0007`0NP5m07D1E`0002@0001m0000LP4007d0001e0@000P0007d0M@5m0781E`0002@0001l0000
KP4007d0001b0@000P0007d0LP5l06h1E`0002@0001j0000J`4007`0001^0@000P0007`0KP5j06/1
E`0002@0001g0000JP4007X0001[0@000P0007X0J`5g06X1E`0002@0001f0000JP4007L0001Z0@00
0P0007L0JP5f06X1E`0002@0001d0000JP4007H0001[0@000P0007H0JP5d06/1E`0002@0001c0000
J`4007@0001/0@000P0007@0J`5c06`1E`0002@0001b0000K04007<0001^0@000P0007<0K05b06h1
E`0002@0001a0000KP400780001b0@000P000780KP5a0781E`0002@0001a0000LP400740001e0@00
0P000740LP5a07D1E`0002@0001a0000M@400780001j0@000P000740M@5b07X1E`0002@0001b0000
NP4007<0001l0@000P000780NP5c07`1E`0002@0001c0000O04007@0001m0@000P0007<0O05d07d1
E`0002@0001d0000O@4007H0001m0@000P0007@0O@5f07d1E`0002@0001g0000O@4007T0001m0@00
0P0007L0O@5i07d1E`0002@0001i0000O04007X0001m0@000P0007T0O@5j07`1E`0002@0001j0000
NP4007/0001l0@000P0007X0O05k07X1E`0002@0001k0000M@4007`0001j0@000P0007/0NP5l07D1
E`0002@0001l0000LP4007`0001e0@000P0007`0M@5l0781E`0002@0001k0000KP4007`0001b0@00
0P0007`0LP5k06h1E`0002@0001j0000K04007/0001^0@000P0007/0KP5j06`1E`0002@0001i0000
J`4007X0001/0@000P0007X0K05i06/1E`0002@0001g0000JP4007T0001[0@000P0007T0J`5g06X1
E`0002@0001>0000JP0004l0001[00000P0004h0JP1?06/0E`0002@0001>0000J`0004l0001[0000
0P0004l0J`1>06/0E`0002@0001=0000J`0004h0001[00000P0004h0J`1=06/0E`0002@0001=0000
JP0004d0001[00000P0004d0J`1=06X0E`0002@0001=0000J00004h0001Z00000P0004d0JP1>06P0
E`0002@0001>0000I`0004l0001X00000P0004h0J01?06L0E`0002@0001?0000IP000580001W0000
0P0004l0I`1B06H0E`0002@0001B0000IP0005D0001V00000P000580IP1E06H0E`0002@0001E0000
IP0005P0001W00000P0005D0IP1H06L0E`0002@0001H0000I`0005T0001Y00000P0005P0I`1I06T0
E`0002@0001I0000J@0005T0001[00000P0005T0J@1I06/0E`0002@0001H0000J`0005T0001]0000
0P0005T0J`1H06d0E`0002@0001E0000K@0005P0001^00000P0005P0K@1E06h0E`0002@0001C0000
KP0005D0001^00000P0005D0KP1C06h0E`0002@0001E0000IP0005L0001W00000P0005D0IP1G06L0
E`0002@0001G0000I`0005P0001Y00000P0005L0I`1H06T0E`0002@0001H0000J@0005P0001[0000
0P0005P0J@1H06/0E`0002@0001G0000J`0005P0001]00000P0005P0J`1G06d0E`0002@0001E0000
K@0005L0001^00000P0005L0K@1E06h0E`0002@0001E0000KP0005L0001_00000P0005D0KP1G06l0
E`0002@0001G0000K`0005T0001a00000P0005L0K`1I0740E`0002@0001I0000L@0005X0001c0000
0P0005T0L@1J07<0E`0002@0001J0000L`0005X0001f00000P0005X0L`1J07H0E`0002@0001I0000
MP0005X0001h00000P0005X0MP1I07P0E`0002@0001H0000N00005T0001i00000P0005T0N01H07T0
E`0002@0001E0000N@0005P0001i00000P0005P0N@1E07T0E`0002@0001B0000N@0005D0001i0000
0P0005D0N@1B07T0E`0002@0001?0000N@000580001i00000P000580N@1?07T0E`0002@0001>0000
N00004l0001i00000P0004l0N@1>07P0E`0002@0001=0000MP0004h0001h00000P0004h0N01=07H0
E`0002@0001=0000M@0004d0001f00000P0004d0MP1=07D0E`0002@0001=0000M00004h0001e0000
0P0004d0M@1>07@0E`0002@0001>0000M00004l0001e00000P0004h0M01?07D0E`0002@0001>0000
M@0004l0001f00000P0004l0M@1>07H0E`0002@0001H0000L00005T0001c00000P0005P0L01I07<0
E`0002@0001I0000L`0005T0001f00000P0005T0L`1I07H0E`0002@0001H0000MP0005T0001h0000
0P0005T0MP1H07P0E`0002@0001G0000N00005P0001i00000P0005P0N01G07T0E`0002@0001E0000
N@0005L0001i00000P0005L0N@1E07T0E`0002@0001Q0000IP0006@0001W00000P0006@0IP1Q06L0
E`0002@0001O0000I`000640001Z00000P000640I`1O06X0E`0002@0001N0000JP0005l0001^0000
0P0005l0JP1N06h0E`0002@0001N0000KP0005h0001a00000P0005h0KP1N0740E`0002@0001N0000
L@0005l0001f00000P0005h0L@1O07H0E`0002@0001O0000MP000640001i00000P0005l0MP1Q07T0
E`0002@0001Q0000N@0006@0001i00000P000640N@1T07T0E`0002@0001T0000N@0006D0001i0000
0P0006@0N@1U07T0E`0002@0001U0000N@0006P0001i00000P0006D0N@1X07T0E`0002@0001X0000
MP0006X0001i00000P0006P0N@1Z07H0E`0002@0001Z0000L@0006/0001f00000P0006X0MP1[0740
E`0002@0001[0000KP0006/0001a00000P0006/0L@1[06h0E`0002@0001Z0000JP0006/0001^0000
0P0006/0KP1Z06X0E`0002@0001X0000I`0006X0001Z00000P0006X0JP1X06L0E`0002@0001U0000
IP0006P0001W00000P0006P0I`1U06H0E`0002@0001T0000IP0006D0001V00000P0006D0IP1T06H0
E`0002@0001R0000IP0006@0001W00000P0006@0IP1R06L0E`0002@0001Q0000I`000680001X0000
0P000680I`1Q06P0E`0002@0001P0000J0000640001Z00000P000640J01P06X0E`0002@0001O0000
JP000600001^00000P000600JP1O06h0E`0002@0001O0000KP0005l0001a00000P0005l0KP1O0740
E`0002@0001O0000L@000600001f00000P0005l0L@1P07H0E`0002@0001P0000MP000640001h0000
0P000600MP1Q07P0E`0002@0001Q0000N0000680001i00000P000640N01R07T0E`0002@0001R0000
N@0006@0001i00000P000680N@1T07T0E`0002@0001U0000N@0006L0001i00000P0006D0N@1W07T0
E`0002@0001W0000N00006P0001i00000P0006L0N@1X07P0E`0002@0001X0000MP0006T0001h0000
0P0006P0N01Y07H0E`0002@0001Y0000L@0006X0001f00000P0006T0MP1Z0740E`0002@0001Z0000
KP0006X0001a00000P0006X0L@1Z06h0E`0002@0001Y0000JP0006X0001^00000P0006X0KP1Y06X0
E`0002@0001X0000J00006T0001Z00000P0006T0JP1X06P0E`0002@0001W0000I`0006P0001X0000
0P0006P0J01W06L0E`0002@0001U0000IP0006L0001W00000P0006L0I`1U06H0E`0002@0001c0000
IP0007H0001W00000P0007H0IP1c06L0E`0002@0001a0000I`0007<0001Z00000P0007<0I`1a06X0
E`0002@0001`0000JP000740001^00000P000740JP1`06h0E`0002@0001`0000KP000700001a0000
0P000700KP1`0740E`0002@0001`0000L@000740001f00000P000700L@1a07H0E`0002@0001a0000
MP0007<0001i00000P000740MP1c07T0E`0002@0001c0000N@0007H0001i00000P0007<0N@1f07T0
E`0002@0001f0000N@0007L0001i00000P0007H0N@1g07T0E`0002@0001g0000N@0007X0001i0000
0P0007L0N@1j07T0E`0002@0001j0000MP0007`0001i00000P0007X0N@1l07H0E`0002@0001l0000
L@0007d0001f00000P0007`0MP1m0740E`0002@0001m0000KP0007d0001a00000P0007d0L@1m06h0
E`0002@0001l0000JP0007d0001^00000P0007d0KP1l06X0E`0002@0001j0000I`0007`0001Z0000
0P0007`0JP1j06L0E`0002@0001g0000IP0007X0001W00000P0007X0I`1g06H0E`0002@0001f0000
IP0007L0001V00000P0007L0IP1f06H0E`0002@0001d0000IP0007H0001W00000P0007H0IP1d06L0
E`0002@0001c0000I`0007@0001X00000P0007@0I`1c06P0E`0002@0001b0000J00007<0001Z0000
0P0007<0J01b06X0E`0002@0001a0000JP000780001^00000P000780JP1a06h0E`0002@0001a0000
KP000740001a00000P000740KP1a0740E`0002@0001a0000L@000780001f00000P000740L@1b07H0
E`0002@0001b0000MP0007<0001h00000P000780MP1c07P0E`0002@0001c0000N00007@0001i0000
0P0007<0N01d07T0E`0002@0001d0000N@0007H0001i00000P0007@0N@1f07T0E`0002@0001g0000
N@0007T0001i00000P0007L0N@1i07T0E`0002@0001i0000N00007X0001i00000P0007T0N@1j07P0
E`0002@0001j0000MP0007/0001h00000P0007X0N01k07H0E`0002@0001k0000L@0007`0001f0000
0P0007/0MP1l0740E`0002@0001l0000KP0007`0001a00000P0007`0L@1l06h0E`0002@0001k0000
JP0007`0001^00000P0007`0KP1k06X0E`0002@0001j0000J00007/0001Z00000P0007/0JP1j06P0
E`0002@0001i0000I`0007X0001X00000P0007X0J01i06L0E`0002@0001g0000IP0007T0001W0000
0P0007T0I`1g06H0E`0002@0002?0000Q@<008l0002=0`000P0008l0Q@>?08d3E`0002@0000X0@00
Q@<002P1002=0`000P0002P1Q@<X0Hd3E`0002@000310@00Q@<00<41002=0`000P000<41Q@?10Hd3
E`0002@0001K0P00Q@<005/2002=0`000P0005/2Q@=K0Xd3E`0002@0003d0P00Q@<00?@2002=0`00
0P000?@2Q@?d0Xd3E`0002@0002=0`00Q@<008d3002=0`000P0008d3Q@>=0hd3E`0002@0000V1000
Q@<002H4002=0`000P0002H4Q@<V18d3E`0002@0002o1000Q@<00;l4002=0`000P000;l4Q@>o18d3
E`0002@000200P00b`<008H2003N0`000P000802b`>60]h3E`0002@000210P00b`<008H2003L0`00
0P000842b`>60]`3E`0002@000260P00b`<008d2003N0`000P0008d2b`>60]h3E`0002@0001n0P00
b`<008@2003;0`000P0007h2b`>40//3E`0002@000290P00b`<008h2003;0`000P0008T2b`>>0//3
E`0002@0002D0P00d`<009@2003D0`000P0009@2d`>D0]@3E`0002@0002C0P00e0<009@2003D0`00
0P0009@2e0>C0]@3E`0002@0002C0P00d`<009<2003D0`000P0009<2e0>C0]<3E`0002@0002C0P00
dP<009@2003C0`000P0009<2d`>D0]83E`0002@0002D0P00d@<009H2003B0`000P0009@2dP>F0]43
E`0002@0002F0P00d@<009T2003A0`000P0009H2d@>I0]43E`0002@0002I0P00d@<009/2003B0`00
0P0009T2d@>K0]83E`0002@0002K0P00dP<009`2003C0`000P0009/2dP>L0]<3E`0002@0002L0P00
d`<009d2003E0`000P0009`2d`>M0]D3E`0002@0002M0P00e@<009d2003L0`000P0009d2e@>M0]`3
E`0002@0002M0P00g0<009h2003N0`000P0009d2g0>N0]h3E`0002@0002N0P00gP<009l2003N0`00
0P0009h2gP>O0]h3E`0002@0002L0P00d`<009`2003L0`000P0009`2d`>L0]`3E`0002@0002L0P00
g0<009d2003N0`000P0009`2g0>M0]h3E`0002@0002M0P00gP<009l2003N0`000P0009d2gP>O0]h3
E`0002@0002O0P00gP<00:02003N0`000P0009l2gP>P0]h3E`0002@0002K0P00e@<009`2003F0`00
0P0009`2e@>K0]H3E`0002@0002F0P00eP<009/2003G0`000P0009/2eP>F0]L3E`0002@0002C0P00
e`<009H2003H0`000P0009H2e`>C0]P3E`0002@0002B0P00f0<009<2003J0`000P0009<2f0>B0]X3
E`0002@0002B0P00fP<00982003L0`000P000982fP>B0]`3E`0002@0002B0P00g0<009<2003N0`00
0P000982g0>C0]h3E`0002@0002C0P00gP<009H2003N0`000P0009<2gP>F0]h3E`0002@0002F0P00
gP<009P2003N0`000P0009H2gP>H0]h3E`0002@0002H0P00gP<009X2003N0`000P0009P2gP>J0]h3
E`0002@0002J0P00g0<009`2003N0`000P0009X2gP>L0]`3E`0002@0002D0P00e`<009H2003H0`00
0P0009H2e`>D0]P3E`0002@0002C0P00f0<009@2003J0`000P0009@2f0>C0]X3E`0002@0002C0P00
fP<009<2003L0`000P0009<2fP>C0]`3E`0002@0002C0P00g0<009@2003N0`000P0009<2g0>D0]h3
E`0002@0002D0P00gP<009H2003N0`000P0009@2gP>F0]h3E`0002@0002V0P00b`<00:H2003N0`00
0P000:H2b`>V0]h3E`0002@0002W0P00b`<00:L2003N0`000P000:L2b`>W0]h3E`0002@0002S0P00
b`<00:L2003;0`000P000:<2b`>W0//3E`0002@0002S0P00gP<00:X2003N0`000P000:<2gP>Z0]h3
E`0002@0002`0P00d@<00;02003L0`000P000;02d@>`0]`3E`0002@0002`0P00g0<00;42003N0`00
0P000;02g0>a0]h3E`0002@0002a0P00gP<00;@2003N0`000P000;42gP>d0]h3E`0002@0002d0P00
gP<00;D2003N0`000P000;@2gP>e0]h3E`0002@0002e0P00gP<00;P2003N0`000P000;D2gP>h0]h3
E`0002@0002h0P00g0<00;X2003N0`000P000;P2gP>j0]`3E`0002@0002a0P00d@<00;42003L0`00
0P000;42d@>a0]`3E`0002@0002a0P00g0<00;82003N0`000P000;42g0>b0]h3E`0002@0002b0P00
gP<00;@2003N0`000P000;82gP>d0]h3E`0002@0002j0P00d@<00;X2003N0`000P000;X2d@>j0]h3
E`0002@0002k0P00d@<00;/2003N0`000P000;/2d@>k0]h3E`0002@0002]0P00d@<00;42003A0`00
0P000:d2d@>a0]43E`0002@0002g0P00d@<00;/2003A0`000P000;L2d@>k0]43E`0002@0002j0P00
gP<00;d2003N0`000P000;X2gP>m0]h3E`0002@000320P00e`<00<d2003G0`000P000<82e`?=0]L3
E`0002@0003=0P00e@<00<d2003G0`000P000<d2e`?=0]D3E`0002@0003<0P00d`<00<d2003E0`00
0P000<d2e@?<0]<3E`0002@0003;0P00dP<00<`2003C0`000P000<`2d`?;0]83E`0002@000390P00
d@<00</2003B0`000P000</2dP?90]43E`0002@000370P00d@<00<T2003A0`000P000<T2d@?70]43
E`0002@000340P00d@<00<L2003B0`000P000<L2d@?40]83E`0002@000320P00dP<00<@2003D0`00
0P000<@2dP?20]@3E`0002@000310P00e0<00<82003G0`000P000<82e0?10]L3E`0002@000310P00
e`<00<42003I0`000P000<42e`?10]T3E`0002@000310P00f@<00<82003L0`000P000<42f@?20]`3
E`0002@000320P00g0<00<@2003N0`000P000<82g0?40]h3E`0002@000340P00gP<00<L2003N0`00
0P000<@2gP?70]h3E`0002@000370P00gP<00<P2003N0`000P000<L2gP?80]h3E`0002@000380P00
gP<00</2003N0`000P000<P2gP?;0]h3E`0002@0003;0P00g0<00<d2003N0`000P000</2gP?=0]`3
E`0002@0003<0P00e0<00<`2003G0`000P000<`2e`?<0]@3E`0002@0003;0P00dP<00<`2003D0`00
0P000<`2e0?;0]83E`0002@000350P00d@<00<L2003B0`000P000<L2d@?50]83E`0002@000330P00
dP<00<D2003D0`000P000<D2dP?30]@3E`0002@000320P00e0<00<<2003G0`000P000<<2e0?20]L3
E`0002@000320P00e`<00<82003I0`000P000<82e`?20]T3E`0002@000320P00f@<00<<2003L0`00
0P000<82f@?30]`3E`0002@000330P00g0<00<D2003N0`000P000<<2g0?50]h3E`0002@000350P00
gP<00<L2003N0`000P000<D2gP?70]h3E`0002@0001/0000V`<006l0002L0`000P0006l0V`=/09`3
E`0002@0001Z0000W0<006`0002O0`000P0006`0W0=Z09l3E`0002@0001Y0000W`<006X0002S0`00
0P0006X0W`=Y0:<3E`0002@0001Y0000X`<006T0002V0`000P0006T0X`=Y0:H3E`0002@0001Y0000
YP<006X0002[0`000P0006T0YP=Z0:/3E`0002@0001Z0000Z`<006`0002^0`000P0006X0Z`=/0:h3
E`0002@0001/0000[P<006l0002^0`000P0006`0[P=_0:h3E`0002@0001_0000[P<00700002^0`00
0P0006l0[P=`0:h3E`0002@0001`0000[P<007<0002^0`000P000700[P=c0:h3E`0002@0001c0000
Z`<007D0002^0`000P0007<0[P=e0:/3E`0002@0001e0000YP<007H0002[0`000P0007D0Z`=f0:H3
E`0002@0001f0000X`<007H0002V0`000P0007H0YP=f0:<3E`0002@0001e0000W`<007H0002S0`00
0P0007H0X`=e09l3E`0002@0001c0000W0<007D0002O0`000P0007D0W`=c09`3E`0002@0001`0000
V`<007<0002L0`000P0007<0W0=`09/3E`0002@0001_0000V`<00700002K0`000P000700V`=_09/3
E`0002@0001]0000V`<006l0002L0`000P0006l0V`=]09`3E`0002@0001/0000W0<006d0002M0`00
0P0006d0W0=/09d3E`0002@0001[0000W@<006`0002O0`000P0006`0W@=[09l3E`0002@0001Z0000
W`<006/0002S0`000P0006/0W`=Z0:<3E`0002@0001Z0000X`<006X0002V0`000P0006X0X`=Z0:H3
E`0002@0001Z0000YP<006/0002[0`000P0006X0YP=[0:/3E`0002@0001[0000Z`<006`0002]0`00
0P0006/0Z`=/0:d3E`0002@0001/0000[@<006d0002^0`000P0006`0[@=]0:h3E`0002@0001]0000
[P<006l0002^0`000P0006d0[P=_0:h3E`0002@0001`0000[P<00780002^0`000P000700[P=b0:h3
E`0002@0001b0000[@<007<0002^0`000P000780[P=c0:d3E`0002@0001c0000Z`<007@0002]0`00
0P0007<0[@=d0:/3E`0002@0001d0000YP<007D0002[0`000P0007@0Z`=e0:H3E`0002@0001e0000
X`<007D0002V0`000P0007D0YP=e0:<3E`0002@0001d0000W`<007D0002S0`000P0007D0X`=d09l3
E`0002@0001c0000W@<007@0002O0`000P0007@0W`=c09d3E`0002@0001b0000W0<007<0002M0`00
0P0007<0W@=b09`3E`0002@0001`0000V`<00780002L0`000P000780W0=`09/3E`0002@0001l0000
[@<007d0002^0`000P0007d0[@=l0:h3E`0002@0001l0000[P<007d0002^0`000P0007`0[P=m0:h3
E`0002@0001m0000[P<007h0002^0`000P0007d0[P=n0:h3E`0002@0001m0000[@<007h0002^0`00
0P0007h0[P=m0:d3E`0002@000270000V`<008X0002L0`000P0008X0V`>709`3E`0002@000250000
W0<008L0002O0`000P0008L0W0>509l3E`0002@000240000W`<008D0002S0`000P0008D0W`>40:<3
E`0002@000240000X`<008@0002V0`000P0008@0X`>40:H3E`0002@000240000YP<008D0002[0`00
0P0008@0YP>50:/3E`0002@000250000Z`<008L0002^0`000P0008D0Z`>70:h3E`0002@000270000
[P<008X0002^0`000P0008L0[P>:0:h3E`0002@0002:0000[P<008/0002^0`000P0008X0[P>;0:h3
E`0002@0002;0000[P<008h0002^0`000P0008/0[P>>0:h3E`0002@0002>0000Z`<00900002^0`00
0P0008h0[P>@0:/3E`0002@0002@0000YP<00940002[0`000P000900Z`>A0:H3E`0002@0002A0000
X`<00940002V0`000P000940YP>A0:<3E`0002@0002@0000W`<00940002S0`000P000940X`>@09l3
E`0002@0002>0000W0<00900002O0`000P000900W`>>09`3E`0002@0002;0000V`<008h0002L0`00
0P0008h0W0>;09/3E`0002@0002:0000V`<008/0002K0`000P0008/0V`>:09/3E`0002@000280000
V`<008X0002L0`000P0008X0V`>809`3E`0002@000270000W0<008P0002M0`000P0008P0W0>709d3
E`0002@000260000W@<008L0002O0`000P0008L0W@>609l3E`0002@000250000W`<008H0002S0`00
0P0008H0W`>50:<3E`0002@000250000X`<008D0002V0`000P0008D0X`>50:H3E`0002@000250000
YP<008H0002[0`000P0008D0YP>60:/3E`0002@000260000Z`<008L0002]0`000P0008H0Z`>70:d3
E`0002@000270000[@<008P0002^0`000P0008L0[@>80:h3E`0002@000280000[P<008X0002^0`00
0P0008P0[P>:0:h3E`0002@0002;0000[P<008d0002^0`000P0008/0[P>=0:h3E`0002@0002=0000
[@<008h0002^0`000P0008d0[P>>0:d3E`0002@0002>0000Z`<008l0002]0`000P0008h0[@>?0:/3
E`0002@0002?0000YP<00900002[0`000P0008l0Z`>@0:H3E`0002@0002@0000X`<00900002V0`00
0P000900YP>@0:<3E`0002@0002?0000W`<00900002S0`000P000900X`>?09l3E`0002@0002>0000
W@<008l0002O0`000P0008l0W`>>09d3E`0002@0002=0000W0<008h0002M0`000P0008h0W@>=09`3
E`0002@0002;0000V`<008d0002L0`000P0008d0W0>;09/3E`0002@0002I0000V`<009`0002L0`00
0P0009`0V`>I09`3E`0002@0002G0000W0<009T0002O0`000P0009T0W0>G09l3E`0002@0002F0000
W`<009L0002S0`000P0009L0W`>F0:<3E`0002@0002F0000X`<009H0002V0`000P0009H0X`>F0:H3
E`0002@0002F0000YP<009L0002[0`000P0009H0YP>G0:/3E`0002@0002G0000Z`<009T0002^0`00
0P0009L0Z`>I0:h3E`0002@0002I0000[P<009`0002^0`000P0009T0[P>L0:h3E`0002@0002L0000
[P<009d0002^0`000P0009`0[P>M0:h3E`0002@0002M0000[P<00:00002^0`000P0009d0[P>P0:h3
E`0002@0002P0000Z`<00:80002^0`000P000:00[P>R0:/3E`0002@0002R0000YP<00:<0002[0`00
0P000:80Z`>S0:H3E`0002@0002S0000X`<00:<0002V0`000P000:<0YP>S0:<3E`0002@0002R0000
W`<00:<0002S0`000P000:<0X`>R09l3E`0002@0002P0000W0<00:80002O0`000P000:80W`>P09`3
E`0002@0002M0000V`<00:00002L0`000P000:00W0>M09/3E`0002@0002L0000V`<009d0002K0`00
0P0009d0V`>L09/3E`0002@0002J0000V`<009`0002L0`000P0009`0V`>J09`3E`0002@0002I0000
W0<009X0002M0`000P0009X0W0>I09d3E`0002@0002H0000W@<009T0002O0`000P0009T0W@>H09l3
E`0002@0002G0000W`<009P0002S0`000P0009P0W`>G0:<3E`0002@0002G0000X`<009L0002V0`00
0P0009L0X`>G0:H3E`0002@0002G0000YP<009P0002[0`000P0009L0YP>H0:/3E`0002@0002H0000
Z`<009T0002]0`000P0009P0Z`>I0:d3E`0002@0002I0000[@<009X0002^0`000P0009T0[@>J0:h3
E`0002@0002J0000[P<009`0002^0`000P0009X0[P>L0:h3E`0002@0002M0000[P<009l0002^0`00
0P0009d0[P>O0:h3E`0002@0002O0000[@<00:00002^0`000P0009l0[P>P0:d3E`0002@0002P0000
Z`<00:40002]0`000P000:00[@>Q0:/3E`0002@0002Q0000YP<00:80002[0`000P000:40Z`>R0:H3
E`0002@0002R0000X`<00:80002V0`000P000:80YP>R0:<3E`0002@0002Q0000W`<00:80002S0`00
0P000:80X`>Q09l3E`0002@0002P0000W@<00:40002O0`000P000:40W`>P09d3E`0002@0002O0000
W0<00:00002M0`000P000:00W@>O09`3E`0002@0002M0000V`<009l0002L0`000P0009l0W0>M09/3
E`0002@0002Y0000V`<00:`0002L0`000P000:`0V`>Y09`3E`0002@0002X0000W0<00:T0002N0`00
0P000:T0W0>X09h3E`0002@0002X0000WP<00:P0002P0`000P000:P0WP>X0:03E`0002@0002X0000
X0<00:T0002R0`000P000:P0X0>Y0:83E`0002@0002Y0000XP<00:`0002S0`000P000:T0XP>/0:<3
E`0002@0002/0000X`<00:l0002S0`000P000:`0X`>_0:<3E`0002@0002_0000XP<00;80002S0`00
0P000:l0X`>b0:83E`0002@0002b0000X0<00;<0002R0`000P000;80XP>c0:03E`0002@0002c0000
WP<00;<0002P0`000P000;<0X0>c09h3E`0002@0002b0000W0<00;<0002N0`000P000;<0WP>b09`3
E`0002@0002_0000V`<00;80002L0`000P000;80W0>_09/3E`0002@0002/0000V`<00:l0002K0`00
0P000:l0V`>/09/3E`0002@0002Z0000V`<00:`0002L0`000P000:`0V`>Z09`3E`0002@0002Y0000
W0<00:X0002N0`000P000:X0W0>Y09h3E`0002@0002Y0000WP<00:T0002P0`000P000:T0WP>Y0:03
E`0002@0002Y0000X0<00:X0002R0`000P000:T0X0>Z0:83E`0002@0002Z0000XP<00:`0002S0`00
0P000:X0XP>/0:<3E`0002@0002_0000XP<00;40002S0`000P000:l0X`>a0:83E`0002@0002a0000
X0<00;80002R0`000P000;40XP>b0:03E`0002@0002b0000WP<00;80002P0`000P000;80X0>b09h3
E`0002@0002a0000W0<00;80002N0`000P000;80WP>a09`3E`0002@0002_0000V`<00;40002L0`00
0P000;40W0>_09/3E`0002@0002Y0000X`<00:`0002T0`000P000:`0X`>Y0:@3E`0002@0002X0000
Y0<00:T0002U0`000P000:T0Y0>X0:D3E`0002@0002W0000Y@<00:P0002W0`000P000:P0Y@>W0:L3
E`0002@0002W0000Y`<00:L0002[0`000P000:L0Y`>W0:/3E`0002@0002W0000Z`<00:P0002]0`00
0P000:L0Z`>X0:d3E`0002@0002X0000[@<00:T0002^0`000P000:P0[@>Y0:h3E`0002@0002Y0000
[P<00:`0002^0`000P000:T0[P>/0:h3E`0002@0002/0000[P<00:l0002^0`000P000:`0[P>_0:h3
E`0002@0002_0000[P<00;80002^0`000P000:l0[P>b0:h3E`0002@0002b0000[@<00;<0002^0`00
0P000;80[P>c0:d3E`0002@0002c0000Z`<00;@0002]0`000P000;<0[@>d0:/3E`0002@0002d0000
Y`<00;@0002[0`000P000;@0Z`>d0:L3E`0002@0002c0000Y@<00;@0002W0`000P000;@0Y`>c0:D3
E`0002@0002b0000Y0<00;<0002U0`000P000;<0Y@>b0:@3E`0002@0002_0000X`<00;80002T0`00
0P000;80Y0>_0:<3E`0002@0002Z0000X`<00:`0002T0`000P000:`0X`>Z0:@3E`0002@0002Y0000
Y0<00:X0002U0`000P000:X0Y0>Y0:D3E`0002@0002X0000Y@<00:T0002W0`000P000:T0Y@>X0:L3
E`0002@0002X0000Y`<00:P0002[0`000P000:P0Y`>X0:/3E`0002@0002X0000Z`<00:T0002]0`00
0P000:P0Z`>Y0:d3E`0002@0002Y0000[@<00:X0002^0`000P000:T0[@>Z0:h3E`0002@0002Z0000
[P<00:`0002^0`000P000:X0[P>/0:h3E`0002@0002_0000[P<00;40002^0`000P000:l0[P>a0:h3
E`0002@0002a0000[@<00;80002^0`000P000;40[P>b0:d3E`0002@0002b0000Z`<00;<0002]0`00
0P000;80[@>c0:/3E`0002@0002c0000Y`<00;<0002[0`000P000;<0Z`>c0:L3E`0002@0002b0000
Y@<00;<0002W0`000P000;<0Y`>b0:D3E`0002@0002a0000Y0<00;80002U0`000P000;80Y@>a0:@3
E`0002@0002_0000X`<00;40002T0`000P000;40Y0>_0:<3E`0002@000050@00V`<000P1002L0`00
0P0000P1V`<50I`3E`0002@000030@00W0<000D1002O0`000P0000D1W0<30Il3E`0002@000020@00
W`<000<1002S0`000P0000<1W`<20J<3E`0002@000020@00X`<00081002V0`000P000081X`<20JH3
E`0002@000020@00YP<000<1002[0`000P000081YP<30J/3E`0002@000030@00Z`<000D1002^0`00
0P0000<1Z`<50Jh3E`0002@000050@00[P<000P1002^0`000P0000D1[P<80Jh3E`0002@000080@00
[P<000T1002^0`000P0000P1[P<90Jh3E`0002@000090@00[P<000`1002^0`000P0000T1[P<<0Jh3
E`0002@0000<0@00Z`<000h1002^0`000P0000`1[P<>0J/3E`0002@0000>0@00YP<000l1002[0`00
0P0000h1Z`<?0JH3E`0002@0000?0@00X`<000l1002V0`000P0000l1YP<?0J<3E`0002@0000>0@00
W`<000l1002S0`000P0000l1X`<>0Il3E`0002@0000<0@00W0<000h1002O0`000P0000h1W`<<0I`3
E`0002@000090@00V`<000`1002L0`000P0000`1W0<90I/3E`0002@000080@00V`<000T1002K0`00
0P0000T1V`<80I/3E`0002@000060@00V`<000P1002L0`000P0000P1V`<60I`3E`0002@000050@00
W0<000H1002M0`000P0000H1W0<50Id3E`0002@000040@00W@<000D1002O0`000P0000D1W@<40Il3
E`0002@000030@00W`<000@1002S0`000P0000@1W`<30J<3E`0002@000030@00X`<000<1002V0`00
0P0000<1X`<30JH3E`0002@000030@00YP<000@1002[0`000P0000<1YP<40J/3E`0002@000040@00
Z`<000D1002]0`000P0000@1Z`<50Jd3E`0002@000050@00[@<000H1002^0`000P0000D1[@<60Jh3
E`0002@000060@00[P<000P1002^0`000P0000H1[P<80Jh3E`0002@000090@00[P<000/1002^0`00
0P0000T1[P<;0Jh3E`0002@0000;0@00[@<000`1002^0`000P0000/1[P<<0Jd3E`0002@0000<0@00
Z`<000d1002]0`000P0000`1[@<=0J/3E`0002@0000=0@00YP<000h1002[0`000P0000d1Z`<>0JH3
E`0002@0000>0@00X`<000h1002V0`000P0000h1YP<>0J<3E`0002@0000=0@00W`<000h1002S0`00
0P0000h1X`<=0Il3E`0002@0000<0@00W@<000d1002O0`000P0000d1W`<<0Id3E`0002@0000;0@00
W0<000`1002M0`000P0000`1W@<;0I`3E`0002@000090@00V`<000/1002L0`000P0000/1W0<90I/3
E`0002@0000E0@00[@<001H1002^0`000P0001H1[@<E0Jh3E`0002@0000E0@00[P<001H1002^0`00
0P0001D1[P<F0Jh3E`0002@0000F0@00[P<001L1002^0`000P0001H1[P<G0Jh3E`0002@0000F0@00
[@<001L1002^0`000P0001L1[P<F0Jd3E`0002@0000P0@00V`<002<1002L0`000P0002<1V`<P0I`3
E`0002@0000N0@00W0<00201002O0`000P000201W0<N0Il3E`0002@0000M0@00W`<001h1002S0`00
0P0001h1W`<M0J<3E`0002@0000M0@00X`<001d1002V0`000P0001d1X`<M0JH3E`0002@0000M0@00
YP<001h1002[0`000P0001d1YP<N0J/3E`0002@0000N0@00Z`<00201002^0`000P0001h1Z`<P0Jh3
E`0002@0000P0@00[P<002<1002^0`000P000201[P<S0Jh3E`0002@0000S0@00[P<002@1002^0`00
0P0002<1[P<T0Jh3E`0002@0000T0@00[P<002L1002^0`000P0002@1[P<W0Jh3E`0002@0000W0@00
Z`<002T1002^0`000P0002L1[P<Y0J/3E`0002@0000Y0@00YP<002X1002[0`000P0002T1Z`<Z0JH3
E`0002@0000Z0@00X`<002X1002V0`000P0002X1YP<Z0J<3E`0002@0000Y0@00W`<002X1002S0`00
0P0002X1X`<Y0Il3E`0002@0000W0@00W0<002T1002O0`000P0002T1W`<W0I`3E`0002@0000T0@00
V`<002L1002L0`000P0002L1W0<T0I/3E`0002@0000S0@00V`<002@1002K0`000P0002@1V`<S0I/3
E`0002@0000Q0@00V`<002<1002L0`000P0002<1V`<Q0I`3E`0002@0000P0@00W0<00241002M0`00
0P000241W0<P0Id3E`0002@0000O0@00W@<00201002O0`000P000201W@<O0Il3E`0002@0000N0@00
W`<001l1002S0`000P0001l1W`<N0J<3E`0002@0000N0@00X`<001h1002V0`000P0001h1X`<N0JH3
E`0002@0000N0@00YP<001l1002[0`000P0001h1YP<O0J/3E`0002@0000O0@00Z`<00201002]0`00
0P0001l1Z`<P0Jd3E`0002@0000P0@00[@<00241002^0`000P000201[@<Q0Jh3E`0002@0000Q0@00
[P<002<1002^0`000P000241[P<S0Jh3E`0002@0000T0@00[P<002H1002^0`000P0002@1[P<V0Jh3
E`0002@0000V0@00[@<002L1002^0`000P0002H1[P<W0Jd3E`0002@0000W0@00Z`<002P1002]0`00
0P0002L1[@<X0J/3E`0002@0000X0@00YP<002T1002[0`000P0002P1Z`<Y0JH3E`0002@0000Y0@00
X`<002T1002V0`000P0002T1YP<Y0J<3E`0002@0000X0@00W`<002T1002S0`000P0002T1X`<X0Il3
E`0002@0000W0@00W@<002P1002O0`000P0002P1W`<W0Id3E`0002@0000V0@00W0<002L1002M0`00
0P0002L1W@<V0I`3E`0002@0000T0@00V`<002H1002L0`000P0002H1W0<T0I/3E`0002@0000b0@00
WP<003@1002O0`000P000381W`<d0Ih3E`0002@0000d0@00V`<003H1002N0`000P0003@1WP<f0I/3
E`0002@0000f0@00V`<003H1002^0`000P0003H1V`<f0Jh3E`0002@0000e0@00W0<003D1002^0`00
0P0003D1W0<e0Jh3E`0002@0000b0@00[P<003X1002^0`000P000381[P<j0Jh3E`0002@000140@00
V`<004L1002L0`000P0004L1V`=40I`3E`0002@000120@00W0<004@1002O0`000P0004@1W0=20Il3
E`0002@000110@00W`<00481002S0`000P000481W`=10J<3E`0002@000110@00X`<00441002V0`00
0P000441X`=10JH3E`0002@000110@00YP<00481002[0`000P000441YP=20J/3E`0002@000120@00
Z`<004@1002^0`000P000481Z`=40Jh3E`0002@000140@00[P<004L1002^0`000P0004@1[P=70Jh3
E`0002@000170@00[P<004P1002^0`000P0004L1[P=80Jh3E`0002@000180@00[P<004/1002^0`00
0P0004P1[P=;0Jh3E`0002@0001;0@00Z`<004d1002^0`000P0004/1[P==0J/3E`0002@0001=0@00
YP<004h1002[0`000P0004d1Z`=>0JH3E`0002@0001>0@00X`<004h1002V0`000P0004h1YP=>0J<3
E`0002@0001=0@00W`<004h1002S0`000P0004h1X`==0Il3E`0002@0001;0@00W0<004d1002O0`00
0P0004d1W`=;0I`3E`0002@000180@00V`<004/1002L0`000P0004/1W0=80I/3E`0002@000170@00
V`<004P1002K0`000P0004P1V`=70I/3E`0002@000150@00V`<004L1002L0`000P0004L1V`=50I`3
E`0002@000140@00W0<004D1002M0`000P0004D1W0=40Id3E`0002@000130@00W@<004@1002O0`00
0P0004@1W@=30Il3E`0002@000120@00W`<004<1002S0`000P0004<1W`=20J<3E`0002@000120@00
X`<00481002V0`000P000481X`=20JH3E`0002@000120@00YP<004<1002[0`000P000481YP=30J/3
E`0002@000130@00Z`<004@1002]0`000P0004<1Z`=40Jd3E`0002@000140@00[@<004D1002^0`00
0P0004@1[@=50Jh3E`0002@000150@00[P<004L1002^0`000P0004D1[P=70Jh3E`0002@000180@00
[P<004X1002^0`000P0004P1[P=:0Jh3E`0002@0001:0@00[@<004/1002^0`000P0004X1[P=;0Jd3
E`0002@0001;0@00Z`<004`1002]0`000P0004/1[@=<0J/3E`0002@0001<0@00YP<004d1002[0`00
0P0004`1Z`==0JH3E`0002@0001=0@00X`<004d1002V0`000P0004d1YP==0J<3E`0002@0001<0@00
W`<004d1002S0`000P0004d1X`=<0Il3E`0002@0001;0@00W@<004`1002O0`000P0004`1W`=;0Id3
E`0002@0001:0@00W0<004/1002M0`000P0004/1W@=:0I`3E`0002@000180@00V`<004X1002L0`00
0P0004X1W0=80I/3E`0002@0002N0@00V`<00:41002L0`000P000:41V`>N0I`3E`0002@0002L0@00
W0<009h1002O0`000P0009h1W0>L0Il3E`0002@0002K0@00W`<009`1002S0`000P0009`1W`>K0J<3
E`0002@0002K0@00X`<009/1002V0`000P0009/1X`>K0JH3E`0002@0002K0@00YP<009`1002[0`00
0P0009/1YP>L0J/3E`0002@0002L0@00Z`<009h1002^0`000P0009`1Z`>N0Jh3E`0002@0002N0@00
[P<00:41002^0`000P0009h1[P>Q0Jh3E`0002@0002Q0@00[P<00:81002^0`000P000:41[P>R0Jh3
E`0002@0002R0@00[P<00:D1002^0`000P000:81[P>U0Jh3E`0002@0002U0@00Z`<00:L1002^0`00
0P000:D1[P>W0J/3E`0002@0002W0@00YP<00:P1002[0`000P000:L1Z`>X0JH3E`0002@0002X0@00
X`<00:P1002V0`000P000:P1YP>X0J<3E`0002@0002W0@00W`<00:P1002S0`000P000:P1X`>W0Il3
E`0002@0002U0@00W0<00:L1002O0`000P000:L1W`>U0I`3E`0002@0002R0@00V`<00:D1002L0`00
0P000:D1W0>R0I/3E`0002@0002Q0@00V`<00:81002K0`000P000:81V`>Q0I/3E`0002@0002O0@00
V`<00:41002L0`000P000:41V`>O0I`3E`0002@0002N0@00W0<009l1002M0`000P0009l1W0>N0Id3
E`0002@0002M0@00W@<009h1002O0`000P0009h1W@>M0Il3E`0002@0002L0@00W`<009d1002S0`00
0P0009d1W`>L0J<3E`0002@0002L0@00X`<009`1002V0`000P0009`1X`>L0JH3E`0002@0002L0@00
YP<009d1002[0`000P0009`1YP>M0J/3E`0002@0002M0@00Z`<009h1002]0`000P0009d1Z`>N0Jd3
E`0002@0002N0@00[@<009l1002^0`000P0009h1[@>O0Jh3E`0002@0002O0@00[P<00:41002^0`00
0P0009l1[P>Q0Jh3E`0002@0002R0@00[P<00:@1002^0`000P000:81[P>T0Jh3E`0002@0002T0@00
[@<00:D1002^0`000P000:@1[P>U0Jd3E`0002@0002U0@00Z`<00:H1002]0`000P000:D1[@>V0J/3
E`0002@0002V0@00YP<00:L1002[0`000P000:H1Z`>W0JH3E`0002@0002W0@00X`<00:L1002V0`00
0P000:L1YP>W0J<3E`0002@0002V0@00W`<00:L1002S0`000P000:L1X`>V0Il3E`0002@0002U0@00
W@<00:H1002O0`000P000:H1W`>U0Id3E`0002@0002T0@00W0<00:D1002M0`000P000:D1W@>T0I`3
E`0002@0002R0@00V`<00:@1002L0`000P000:@1W0>R0I/3E`0002@0002^0@00[@<00:l1002^0`00
0P000:l1[@>^0Jh3E`0002@0002^0@00[P<00:l1002^0`000P000:h1[P>_0Jh3E`0002@0002_0@00
[P<00;01002^0`000P000:l1[P>`0Jh3E`0002@0002_0@00[@<00;01002^0`000P000;01[P>_0Jd3
E`0002@0002i0@00V`<00;`1002L0`000P000;`1V`>i0I`3E`0002@0002g0@00W0<00;T1002O0`00
0P000;T1W0>g0Il3E`0002@0002f0@00W`<00;L1002S0`000P000;L1W`>f0J<3E`0002@0002f0@00
X`<00;H1002V0`000P000;H1X`>f0JH3E`0002@0002f0@00YP<00;L1002[0`000P000;H1YP>g0J/3
E`0002@0002g0@00Z`<00;T1002^0`000P000;L1Z`>i0Jh3E`0002@0002i0@00[P<00;`1002^0`00
0P000;T1[P>l0Jh3E`0002@0002l0@00[P<00;d1002^0`000P000;`1[P>m0Jh3E`0002@0002m0@00
[P<00<01002^0`000P000;d1[P?00Jh3E`0002@000300@00Z`<00<81002^0`000P000<01[P?20J/3
E`0002@000320@00YP<00<<1002[0`000P000<81Z`?30JH3E`0002@000330@00X`<00<<1002V0`00
0P000<<1YP?30J<3E`0002@000320@00W`<00<<1002S0`000P000<<1X`?20Il3E`0002@000300@00
W0<00<81002O0`000P000<81W`?00I`3E`0002@0002m0@00V`<00<01002L0`000P000<01W0>m0I/3
E`0002@0002l0@00V`<00;d1002K0`000P000;d1V`>l0I/3E`0002@0002j0@00V`<00;`1002L0`00
0P000;`1V`>j0I`3E`0002@0002i0@00W0<00;X1002M0`000P000;X1W0>i0Id3E`0002@0002h0@00
W@<00;T1002O0`000P000;T1W@>h0Il3E`0002@0002g0@00W`<00;P1002S0`000P000;P1W`>g0J<3
E`0002@0002g0@00X`<00;L1002V0`000P000;L1X`>g0JH3E`0002@0002g0@00YP<00;P1002[0`00
0P000;L1YP>h0J/3E`0002@0002h0@00Z`<00;T1002]0`000P000;P1Z`>i0Jd3E`0002@0002i0@00
[@<00;X1002^0`000P000;T1[@>j0Jh3E`0002@0002j0@00[P<00;`1002^0`000P000;X1[P>l0Jh3
E`0002@0002m0@00[P<00;l1002^0`000P000;d1[P>o0Jh3E`0002@0002o0@00[@<00<01002^0`00
0P000;l1[P?00Jd3E`0002@000300@00Z`<00<41002]0`000P000<01[@?10J/3E`0002@000310@00
YP<00<81002[0`000P000<41Z`?20JH3E`0002@000320@00X`<00<81002V0`000P000<81YP?20J<3
E`0002@000310@00W`<00<81002S0`000P000<81X`?10Il3E`0002@000300@00W@<00<41002O0`00
0P000<41W`?00Id3E`0002@0002o0@00W0<00<01002M0`000P000<01W@>o0I`3E`0002@0002m0@00
V`<00;l1002L0`000P000;l1W0>m0I/3E`0002@0003;0@00WP<00<d1002O0`000P000</1W`?=0Ih3
E`0002@0003=0@00V`<00<l1002N0`000P000<d1WP??0I/3E`0002@0003?0@00V`<00<l1002^0`00
0P000<l1V`??0Jh3E`0002@0003>0@00W0<00<h1002^0`000P000<h1W0?>0Jh3E`0002@0003;0@00
[P<00=<1002^0`000P000</1[P?C0Jh3E`0002@0003K0@00W`<00=`1002P0`000P000=/1W`?L0J03
E`0002@0003K0@00X0<00=`1002P0`000P000=`1X0?K0J03E`0002@0003J0@00X0<00=/1002P0`00
0P000=/1X0?J0J03E`0002@0003J0@00W`<00=X1002P0`000P000=X1X0?J0Il3E`0002@0003J0@00
W@<00=/1002O0`000P000=X1W`?K0Id3E`0002@0003K0@00W0<00=`1002M0`000P000=/1W@?L0I`3
E`0002@0003L0@00V`<00=l1002L0`000P000=`1W0?O0I/3E`0002@0003O0@00V`<00>81002K0`00
0P000=l1V`?R0I/3E`0002@0003R0@00V`<00>D1002L0`000P000>81V`?U0I`3E`0002@0003U0@00
W0<00>H1002M0`000P000>D1W0?V0Id3E`0002@0003V0@00W@<00>L1002O0`000P000>H1W@?W0Il3
E`0002@0003W0@00W`<00>L1002P0`000P000>L1W`?W0J03E`0002@0003V0@00X0<00>L1002R0`00
0P000>L1X0?V0J83E`0002@0003S0@00XP<00>H1002T0`000P000>H1XP?S0J@3E`0002@0003O0@00
Y0<00><1002V0`000P000><1Y0?O0JH3E`0002@0003M0@00YP<00=l1002W0`000P000=l1YP?M0JL3
E`0002@0003K0@00Y`<00=d1002Y0`000P000=d1Y`?K0JT3E`0002@0003J0@00Z@<00=/1002/0`00
0P000=/1Z@?J0J`3E`0002@0003J0@00[0<00=X1002^0`000P000=X1[0?J0Jh3E`0002@0003R0@00
V`<00>@1002L0`000P000>81V`?T0I`3E`0002@0003T0@00W0<00>D1002M0`000P000>@1W0?U0Id3
E`0002@0003U0@00W@<00>H1002O0`000P000>D1W@?V0Il3E`0002@0003V0@00W`<00>H1002P0`00
0P000>H1W`?V0J03E`0002@0003U0@00X0<00>H1002R0`000P000>H1X0?U0J83E`0002@0003R0@00
XP<00>D1002T0`000P000>D1XP?R0J@3E`0002@0003O0@00Y0<00>81002V0`000P000>81Y0?O0JH3
E`0002@0003J0@00[0<00=/1002]0`000P000=X1[@?K0J`3E`0002@0003K0@00[0<00=d1002/0`00
0P000=/1[0?M0J`3E`0002@0003M0@00[0<00>41002^0`000P000=d1[0?Q0Jh3E`0002@0003Q0@00
[P<00>@1002^0`000P000>41[P?T0Jh3E`0002@0003T0@00[@<00>H1002^0`000P000>@1[P?V0Jd3
E`0002@0003V0@00[0<00>L1002]0`000P000>H1[@?W0J`3E`0002@0003M0@00[0<00>41002^0`00
0P000=d1[0?Q0Jh3E`0002@0003Q0@00[P<00>D1002^0`000P000>41[P?U0Jh3E`0002@0003U0@00
[P<00>H1002^0`000P000>D1[P?V0Jh3E`0002@0003V0@00[0<00>L1002^0`000P000>H1[P?W0J`3
E`0002@0003W0@00ZP<00>L1002/0`000P000>L1[0?W0JX3E`0002@0000h0P00V`<003/2002L0`00
0P0003/2V`<h0Y`3E`0002@0000f0P00W0<003P2002O0`000P0003P2W0<f0Yl3E`0002@0000e0P00
W`<003H2002S0`000P0003H2W`<e0Z<3E`0002@0000e0P00X`<003D2002V0`000P0003D2X`<e0ZH3
E`0002@0000e0P00YP<003H2002[0`000P0003D2YP<f0Z/3E`0002@0000f0P00Z`<003P2002^0`00
0P0003H2Z`<h0Zh3E`0002@0000h0P00[P<003/2002^0`000P0003P2[P<k0Zh3E`0002@0000k0P00
[P<003`2002^0`000P0003/2[P<l0Zh3E`0002@0000l0P00[P<003l2002^0`000P0003`2[P<o0Zh3
E`0002@0000o0P00Z`<00442002^0`000P0003l2[P=10Z/3E`0002@000110P00YP<00482002[0`00
0P000442Z`=20ZH3E`0002@000120P00X`<00482002V0`000P000482YP=20Z<3E`0002@000110P00
W`<00482002S0`000P000482X`=10Yl3E`0002@0000o0P00W0<00442002O0`000P000442W`<o0Y`3
E`0002@0000l0P00V`<003l2002L0`000P0003l2W0<l0Y/3E`0002@0000k0P00V`<003`2002K0`00
0P0003`2V`<k0Y/3E`0002@0000i0P00V`<003/2002L0`000P0003/2V`<i0Y`3E`0002@0000h0P00
W0<003T2002M0`000P0003T2W0<h0Yd3E`0002@0000g0P00W@<003P2002O0`000P0003P2W@<g0Yl3
E`0002@0000f0P00W`<003L2002S0`000P0003L2W`<f0Z<3E`0002@0000f0P00X`<003H2002V0`00
0P0003H2X`<f0ZH3E`0002@0000f0P00YP<003L2002[0`000P0003H2YP<g0Z/3E`0002@0000g0P00
Z`<003P2002]0`000P0003L2Z`<h0Zd3E`0002@0000h0P00[@<003T2002^0`000P0003P2[@<i0Zh3
E`0002@0000i0P00[P<003/2002^0`000P0003T2[P<k0Zh3E`0002@0000l0P00[P<003h2002^0`00
0P0003`2[P<n0Zh3E`0002@0000n0P00[@<003l2002^0`000P0003h2[P<o0Zd3E`0002@0000o0P00
Z`<00402002]0`000P0003l2[@=00Z/3E`0002@000100P00YP<00442002[0`000P000402Z`=10ZH3
E`0002@000110P00X`<00442002V0`000P000442YP=10Z<3E`0002@000100P00W`<00442002S0`00
0P000442X`=00Yl3E`0002@0000o0P00W@<00402002O0`000P000402W`<o0Yd3E`0002@0000n0P00
W0<003l2002M0`000P0003l2W@<n0Y`3E`0002@0000l0P00V`<003h2002L0`000P0003h2W0<l0Y/3
E`0002@000170P00[@<004P2002^0`000P0004P2[@=70Zh3E`0002@000170P00[P<004P2002^0`00
0P0004L2[P=80Zh3E`0002@000180P00[P<004T2002^0`000P0004P2[P=90Zh3E`0002@000180P00
[@<004T2002^0`000P0004T2[P=80Zd3E`0002@0001B0P00V`<005D2002L0`000P0005D2V`=B0Y`3
E`0002@0001@0P00W0<00582002O0`000P000582W0=@0Yl3E`0002@0001?0P00W`<00502002S0`00
0P000502W`=?0Z<3E`0002@0001?0P00X`<004l2002V0`000P0004l2X`=?0ZH3E`0002@0001?0P00
YP<00502002[0`000P0004l2YP=@0Z/3E`0002@0001@0P00Z`<00582002^0`000P000502Z`=B0Zh3
E`0002@0001B0P00[P<005D2002^0`000P000582[P=E0Zh3E`0002@0001E0P00[P<005H2002^0`00
0P0005D2[P=F0Zh3E`0002@0001F0P00[P<005T2002^0`000P0005H2[P=I0Zh3E`0002@0001I0P00
Z`<005/2002^0`000P0005T2[P=K0Z/3E`0002@0001K0P00YP<005`2002[0`000P0005/2Z`=L0ZH3
E`0002@0001L0P00X`<005`2002V0`000P0005`2YP=L0Z<3E`0002@0001K0P00W`<005`2002S0`00
0P0005`2X`=K0Yl3E`0002@0001I0P00W0<005/2002O0`000P0005/2W`=I0Y`3E`0002@0001F0P00
V`<005T2002L0`000P0005T2W0=F0Y/3E`0002@0001E0P00V`<005H2002K0`000P0005H2V`=E0Y/3
E`0002@0001C0P00V`<005D2002L0`000P0005D2V`=C0Y`3E`0002@0001B0P00W0<005<2002M0`00
0P0005<2W0=B0Yd3E`0002@0001A0P00W@<00582002O0`000P000582W@=A0Yl3E`0002@0001@0P00
W`<00542002S0`000P000542W`=@0Z<3E`0002@0001@0P00X`<00502002V0`000P000502X`=@0ZH3
E`0002@0001@0P00YP<00542002[0`000P000502YP=A0Z/3E`0002@0001A0P00Z`<00582002]0`00
0P000542Z`=B0Zd3E`0002@0001B0P00[@<005<2002^0`000P000582[@=C0Zh3E`0002@0001C0P00
[P<005D2002^0`000P0005<2[P=E0Zh3E`0002@0001F0P00[P<005P2002^0`000P0005H2[P=H0Zh3
E`0002@0001H0P00[@<005T2002^0`000P0005P2[P=I0Zd3E`0002@0001I0P00Z`<005X2002]0`00
0P0005T2[@=J0Z/3E`0002@0001J0P00YP<005/2002[0`000P0005X2Z`=K0ZH3E`0002@0001K0P00
X`<005/2002V0`000P0005/2YP=K0Z<3E`0002@0001J0P00W`<005/2002S0`000P0005/2X`=J0Yl3
E`0002@0001I0P00W@<005X2002O0`000P0005X2W`=I0Yd3E`0002@0001H0P00W0<005T2002M0`00
0P0005T2W@=H0Y`3E`0002@0001F0P00V`<005P2002L0`000P0005P2W0=F0Y/3E`0002@0001T0P00
WP<006H2002O0`000P0006@2W`=V0Yh3E`0002@0001V0P00V`<006P2002N0`000P0006H2WP=X0Y/3
E`0002@0001X0P00V`<006P2002^0`000P0006P2V`=X0Zh3E`0002@0001W0P00W0<006L2002^0`00
0P0006L2W0=W0Zh3E`0002@0001T0P00[P<006`2002^0`000P0006@2[P=/0Zh3E`0002@0001k0P00
W@<007/2002^0`000P0007/2W@=k0Zh3E`0002@0001l0P00V`<007`2002^0`000P0007`2V`=l0Zh3
E`0002@0001b0P00V`<007`2002Y0`000P0007`2V`=b0ZT3E`0002@0001b0P00Z@<00842002Y0`00
0P000782Z@>10ZT3E`0002@0001i0P00[P<007l2002^0`000P0007T2[P=o0Zh3E`0002@0003A0P00
V`<00=@2002L0`000P000=@2V`?A0Y`3E`0002@0003?0P00W0<00=42002O0`000P000=42W0??0Yl3
E`0002@0003>0P00W`<00<l2002S0`000P000<l2W`?>0Z<3E`0002@0003>0P00X`<00<h2002V0`00
0P000<h2X`?>0ZH3E`0002@0003>0P00YP<00<l2002[0`000P000<h2YP??0Z/3E`0002@0003?0P00
Z`<00=42002^0`000P000<l2Z`?A0Zh3E`0002@0003A0P00[P<00=@2002^0`000P000=42[P?D0Zh3
E`0002@0003D0P00[P<00=D2002^0`000P000=@2[P?E0Zh3E`0002@0003E0P00[P<00=P2002^0`00
0P000=D2[P?H0Zh3E`0002@0003H0P00Z`<00=X2002^0`000P000=P2[P?J0Z/3E`0002@0003J0P00
YP<00=/2002[0`000P000=X2Z`?K0ZH3E`0002@0003K0P00X`<00=/2002V0`000P000=/2YP?K0Z<3
E`0002@0003J0P00W`<00=/2002S0`000P000=/2X`?J0Yl3E`0002@0003H0P00W0<00=X2002O0`00
0P000=X2W`?H0Y`3E`0002@0003E0P00V`<00=P2002L0`000P000=P2W0?E0Y/3E`0